/*
 * Driver for Broadcom RPC GPIO unit (pinctrl + GPIO)
 *
 * Copyright (C) 2015-2019 Broadcom
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation version 2.
 *
 * This program is distributed "as is" WITHOUT ANY WARRANTY of any
 * kind, whether express or implied; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#ifndef __PINCTRL_BCM_RPC_H
#define __PINCTRL_BCM_RPC_H

#include <linux/pinctrl/consumer.h>
#include <linux/pinctrl/machine.h>
#include <linux/pinctrl/pinconf.h>
#include <linux/pinctrl/pinctrl.h>
#include <linux/pinctrl/pinmux.h>

/**
 * enum gio_reg_index - GIO block GPIO register index.
 * @GIO_REG_ODEN: Open drain enable register.
 * @GIO_REG_DATA: Data register.
 * @GIO_REG_IODIR: Input/output direction register.
 * @GIO_REG_EC: Edge insensitive register.
 * @GIO_REG_MASK: Interrupt mask register.
 * @GIO_REG_LEVEL: Interrupt level type register.
 * @GIO_REG_STAT: Interrupt status register.
 * @GIO_REG_COUNT: Total GIO GPIO register count.
 */
enum gio_reg_index {
	GIO_REG_ODEN = 0,
	GIO_REG_DATA,
	GIO_REG_IODIR,
	GIO_REG_EC,
	GIO_REG_EI,
	GIO_REG_MASK,
	GIO_REG_LEVEL,
	GIO_REG_STAT,
	GIO_REG_COUNT
};

#define GIO_BANK_SIZE           (GIO_REG_COUNT * sizeof(u32))
#define GIO_BANK_OFF(bank, off)	(((bank) * GIO_BANK_SIZE) + (off * sizeof(u32)))
#define GIO_ODEN(bank)          GIO_BANK_OFF(bank, GIO_REG_ODEN)
#define GIO_DATA(bank)          GIO_BANK_OFF(bank, GIO_REG_DATA)
#define GIO_IODIR(bank)         GIO_BANK_OFF(bank, GIO_REG_IODIR)
#define GIO_EC(bank)            GIO_BANK_OFF(bank, GIO_REG_EC)
#define GIO_EI(bank)            GIO_BANK_OFF(bank, GIO_REG_EI)
#define GIO_MASK(bank)          GIO_BANK_OFF(bank, GIO_REG_MASK)
#define GIO_LEVEL(bank)         GIO_BANK_OFF(bank, GIO_REG_LEVEL)
#define GIO_STAT(bank)          GIO_BANK_OFF(bank, GIO_REG_STAT)

#define MAX_GPIO_PER_BANK		32
#define GPIO_BANK(gpio)			((gpio) >> 5)
#define BITS_PER_GPO			0x1

#define PIN_NUMBER_MASK         0x0000FFFF
#define PIN_FUNC_MASK           0xFFFF0000
#define PIN_FUNC_SHIFT			16
/**
 * enum tm_gpo_reg_index - TM block GPIO register index
 * @TM_GPO_REG_EN: Output enable control register.
 * @TM_GPO_REG_OD: Open drain control register.
 * @TM_GPO_REG_DATA: Data control register.
 * @TM_GPO_REG_COUNT: Total TM GPIO register count.
 */
enum tm_gpo_reg_index {
	TM_GPO_REG_EN = 0,
	TM_GPO_REG_OD,
	TM_GPO_REG_DATA,
	TM_GPO_REG_COUNT
};

#define TM_GPO_BANK_SIZE        (TM_GPO_REG_COUNT * sizeof(u32))
#define TM_GPO_BANK_OFF(bank, off)	\
		(((bank) * TM_GPO_BANK_SIZE) + (off * sizeof(u32)))
#define TM_GPO_EN(bank)          TM_GPO_BANK_OFF(bank, TM_GPO_REG_EN)
#define TM_GPO_OD(bank)          TM_GPO_BANK_OFF(bank, TM_GPO_REG_OD)
#define TM_GPO_DATA(bank)        TM_GPO_BANK_OFF(bank, TM_GPO_REG_DATA)


#define MODULE_NAME                 "bcmrpc-pinctrl-gpio"
#define COMPATIBLE_GPIO             "brcm,bcmrpc-pinctrl-gpio"
#define COMPATIBLE_GPO              "brcm,bcmrpc-pinctrl-gpo"
#define COMPATIBLE_PROPERTY         "compatible"
#define GPIO_PINS_INFO_PROPERTY     "gpio-pins-info"
#define PINS_PROPERTY               "brcm,pins"
#define PIN_NAMES_PROPERTY          "brcm,pin-names"
#define PMUX_FUNCTION_PROPERTY      "brcm,function"
#define GPIO_NAMES_PROPERTY         "gpio-names"
#define RPC_CHANNEL_NODE            "rpc-channel"
#define DEVICE_NAME_PROPERTY        "dev-name"
#define GPIO_BANK_WIDTH_PROPERTY    "brcm,gpio-bank-widths"
#define INTR_CONTROLLER_PROPERTY    "interrupt-controller"
#define PIN_ID_MAX                  512
#define NUM_GPIOS                   PIN_ID_MAX

/**
 * struct brcmrpc_gpio_bank: represent a controller pin-bank.
 * @node: Bank list head to track all bank information.
 * @id: Bank id.
 * @gc: GPIO chip of the bank.
 * @width: Bank width(Number of GPIO per bank).
 * @icg: Generic irq chip data structure.
 * @wake_active: Active wake flag to store irq mask.
 * @regs: Bank register storage.
 * @gpio_pins:  boards/machines provide information on their
 *		pins, pads or other muxable units in this struct.
 * @data: Generic data pointer to store any information.
 */
struct brcmrpc_gpio_bank {
	struct list_head node;
	int id;
	struct gpio_chip gc;
	struct brcmrpc_gpio_priv *parent_priv;
	u32 width;
	struct irq_chip_generic *icg;
	u32 regs[GIO_REG_STAT];
	struct pinctrl_pin_desc *gpio_pins;
	void *data;
};

/**
 * struct brcmrpc_pin_group: represent pinctrl supported by the
 * driver.
 * @pin_desc:  boards/machines provide information on their.
 * pins, pads or other muxable units in this struct.
 * @nr_pins: number of pins included in the driver.
 * @nr_pin_names: number of pin names included in the driver.
 */
struct brcmrpc_pin_group {
	struct pinctrl_pin_desc *pin_desc;
	u32 nr_pins;
	u32 nr_pin_names;
};

/**
 * struct brcmrpc_gpio_info: represent GPIO information of all
 * banks.
 * @names: GPIO pin names provided in the device tree to use.
 * @offset: GPIO Pin starting number.
 * @ngpios: Number of pin supported.
 * @nformat: Pin number naming format.
 */
struct brcmrpc_gpio_info {
	const char *names;
	u32 offset;
	u32 ngpios;
	u32 nformat;
	u32 muxfunc;
};

/**
 * struct brcmrpc_pinctrl: represent pinctrl information.
 * @dev: Device instance representing the controller.
 * @base: Register base address of the controller.
 * @pdesc: Pin controller descriptor registered with the pinctrl
 *		subsystem.
 * @pctl_dev: Representing pinctrl class device instance.
 * @gpio_range: Representing pinctrl subranges of the GPIO
 *		number space to be handled by the controller.
 * @bank: Represent a controller pin-bank.
 * @pin_group: List of pin group available to the driver.
 * @groups: Pinctrl group names.
 * @gpio_info: Represent GPIO information of all
 *		banks.
 */
struct brcmrpc_pinctrl {
	struct device *dev;
	void __iomem *base;
	struct pinctrl_desc pdesc;
	struct pinctrl_dev *pctl_dev;
	struct pinctrl_gpio_range gpio_range;
	struct brcmrpc_gpio_bank *bank;
	struct brcmrpc_pin_group *pin_group;
	const char **groups;
	u32 dt_ngpio;
	u32 dt_ngpio_names;
	struct brcmrpc_gpio_info *gpio_info;
};

/**
 * struct brcmrpc_gpio_priv: wrapper for holding driver data
 *		together.
 * @bank_list: Global bank list node.
 * @reg_base: Register base address of the controller.
 * @pdev: device instance representing the controller.
 * @pc: Represent pinctrl information.
 * @irq_domain: Represent IRQ translation domains.
 * @parent_irq: Platform irq information.
 * @gpio_base: Representing base GPIO number.
 * @can_wake: Flag to wake events.
 * @wake_mask: Mask to vakeup events.
 * @parent_wake_irq: Representing irq wake information.
 * @lock: Mutex lock.
 */
struct brcmrpc_gpio_priv {
	struct list_head bank_list;
	void __iomem *reg_base;
	struct platform_device *pdev;
	struct brcmrpc_pinctrl *pc;
	struct irq_domain *irq_domain;
	int parent_irq;
	int gpio_base;
	int num_gpios;
	struct mutex lock;
};

enum gpio_chip_type {
	GIO_GPIO_CHIP = 0,
	TM_GPO_CHIP,
	GPIO_CHIP_MAX
};

/**
 * enum pinconf_param_idx - possible pin configuration
 *		params supported.
 * @PINCONF_PARAM_DRV: Drive strength configuration.
 * @PINCONF_PARAM_SLEW: Slew rate configuration.
 * @PINCONF_PARAM_HYST: Hysteresis configuration.
 * @PINCONF_PARAM_INDIS: Input Enable/Disable configuration.
 * @PINCONF_PARAM_PULL: Pull up/down configuration.
 */
enum pinconf_param_idx {
	PINCONF_PARAM_DRV = 0,
	PINCONF_PARAM_SLEW,
	PINCONF_PARAM_HYST,
	PINCONF_PARAM_INDIS,
	PINCONF_PARAM_PULL,
	PINCONF_PARAM_AMP,
	PINCONF_PARAM_MHV,
	PINCONF_PARAM_GMII,
	PINCONF_PARAM_MPV,
	PINCONF_PARAM_SRC,
	PINCONF_PARAM_DGEN,
	PINCONF_PARAM_RSEL,
	PINCONF_PARAM_GFIL,
	PINCONF_PARAM_PWR,
	PINCONF_PARAM_MAX
};

enum pinconf_pwr_idx {
	PINCONFIG_PWR_DISABLE,
	PINCONFIG_PWR_ENABLE,
};

enum pinconf_gfil_idx {
	PINCONFIG_GFIL_DISABLE,
	PINCONFIG_GFIL_ENABLE,
};

enum pinconf_rsel_idx {
	PINCONFIG_RSEL_10K,
	PINCONFIG_RSEL_50K,
};

enum pinconf_dgen_idx {
	PINCONFIG_DGEN_DISABLE,
	PINCONFIG_DGEN_ENABLE,
};

enum pinconf_src_idx {
	PINCONFIG_SRC_DISABLE,
	PINCONFIG_SRC_ENABLE,
};

enum pinconf_mpv_idx {
	PINCONFIG_MPV_STANDARD,
	PINCONFIG_MPV_FAST,
};

enum pinconf_gmii_idx {
	PINCONFIG_GMII_DISABLE,	//RGMII25
	PINCONFIG_GMII_ENABLE,	//GMII33
};

enum pinconf_mhv_idx {
	PINCONFIG_MHV_DISABLE,
	PINCONFIG_MHV_ENABLE,
};

enum pinconf_amp_idx {
	PINCONFIG_AMP_DISABLE,
	PINCONFIG_AMP_ENABLE,
};

enum pinconf_pull_idx {
	PINCONFIG_PULL_NONE,
	PINCONFIG_PULL_DOWN,
	PINCONFIG_PULL_UP,
};

enum pinconf_indis_idx {
	PINCONFIG_INPUT_ENABLE,
	PINCONFIG_INPUT_DISABLE,
};

enum pinconf_hyst_idx {
	PINCONFIG_HYST_TTL,
	PINCONFIG_HYST_SCHTRIG,
};

enum pinconf_slew_idx {
	PINCONFIG_SLEW_FEDGE,
	PINCONFIG_SLEW_SEDGE,
};

enum pinconf_drv_idx {
	PINCONFIG_DRV_1MA,
	PINCONFIG_DRV_2MA,
	PINCONFIG_DRV_3MA,
	PINCONFIG_DRV_4MA,
	PINCONFIG_DRV_5MA,
	PINCONFIG_DRV_6MA,
	PINCONFIG_DRV_7MA,
	PINCONFIG_DRV_8MA,
};

/**
 * pin configuration (pull up/down, drive strength, slew, hyst etc) type and its
 * value are packed together into a 32-bits. The upper 16-bits 
 * represent the configuration type and the lower 16-bits hold 
 * the value of the configuration type. 
 */
#define PINCONF_PACK(_param_, _arg_) ((_param_) << 16 | (_arg_))
#define PINCONF_UNPACK_PARAM(_conf_) ((_conf_) >> 16)
#define PINCONF_UNPACK_ARG(_conf_) ((_conf_) & 0xffff)

/**
 * enum pimux_fsel - possible pin function select for pinmux
 *		supported.
 * @FSEL_ALT0: Select ALT0 level as pinmux function.
 * @FSEL_ALT1: Select ALT1 level as pinmux function.
 * @FSEL_ALT2: Select ALT2 leve2 as pinmux function.
 * @FSEL_ALT3: Select ALT3 leve3 as pinmux function. 
 * @FSEL_ALT4: Select ALT4 leve4 as pinmux function.
 * @FSEL_ALT5: Select ALT5 leve5 as pinmux function.
 * @FSEL_ALT6: Select ALT6 leve6 as pinmux function.
 * @FSEL_ALT7: Select ALT7 leve7 as pinmux function. 
 * @FSEL_COUNT: Total pinmux function supported.
 */

enum pinmux_fsel {
	FSEL_ALT0 = 0,
	FSEL_ALT1 = 1,
	FSEL_ALT2 = 2,
	FSEL_ALT3 = 3,
	FSEL_ALT4 = 4,
	FSEL_ALT5 = 5,
	FSEL_ALT6 = 6,
	FSEL_ALT7 = 7,
	FSEL_COUNT = 8,
	FSEL_MASK = 0xF,
};

/**
 * enum gpio_pin_info - possible pin information read from
 *		device tree.
 * @GPIO_PIN_OFFSET: Pin offset value.
 * @GPIO_PIN_NGPIOS: Total number of pins supported.
 * @GPIO_PIN_FORMAT: Pin number format. 
 * @GPIO_PIN_MUX_FUNC: GPIO MUX function number. 
 */
enum gpio_pin_info {
	GPIO_PIN_OFFSET = 0,
	GPIO_PIN_NGPIOS,
	GPIO_PIN_FORMAT,
	GPIO_PIN_MUX_FUNC,
	GPIO_PIN_MAX
};

/* RPC Information */
/* RPC global variables */
#define RPC_TIMEOUT	1	/* secs */
#define RPC_GPIO_TIMEOUT	RPC_TIMEOUT
#define RPC_PINCTRL_TIMEOUT	RPC_TIMEOUT
#define RPC_PINCTRL_MUX_OFFSET	8

/**
 * enum rpc_config - possible rpc configuration supported for
 *		individual pins.
 * @RPC_CONFIG_NONE: Pin configuration not done through RPC.
 * @RPC_CONFIG_NOTSUPPORT: Pin configuration not supported
 *		through RPC.
 * @RPC_CONFIG_GPIO: Pin configured as GPIO through RPC.
 * @RPC_CONFIG_PINCTRL: Pin configured as pinctrl through RPC.
 */
enum rpc_config {
	RPC_CONFIG_NONE = 0,
	RPC_CONFIG_NOTSUPPORT,
	RPC_CONFIG_GPIO,
	RPC_CONFIG_PINCTRL,
	RPC_CONFIG_MAX
};

/**
 * enum rpc_action - possible rpc action supported for
 *		individual pins.
 * @RPC_GPIO_ALLOC: Pin allocated as GPIO through RPC.
 * @RPC_GPIO_FREE: GPIO pin free through RPC.
 * @RPC_PINCTRL_ALLOC: Pin allocated as pinmux through RPC.
 * @RPC_CONFIG_PINCTRL: Pinctrl free through RPC.
 */
enum rpc_action {
	RPC_GPIO_ALLOC = 0,
	RPC_GPIO_FREE,
	RPC_PINCTRL_ALLOC,
	RPC_PINCTRL_FREE,
	RPC_ACTION_MAX
};

/**
 * enum rpc_pinctrl_func_idx - possible pinctrl rpc functions
 *		supported for individual pins.
 * @RPC_PINCTRL_FUNC_ALLOC: Pinctrl allocate function.
 * @RPC_PINCTRL_FUNC_FREE: Pinctrl free function.
 * @RPC_PINCTRL_FUNC_MUX_GET: Pinctrl function to get pinmux
 *		configuration.
 * @RPC_PINCTRL_FUNC_MUX_SET: Pinctrl function to set pinmux
 *		configuration.
 * @RPC_PINCTRL_FUNC_PAD_CFG_GET: Pinctrl function to get pad
 *		configuration.
 * @RPC_PINCTRL_FUNC_PAD_CFG_SET: Pinctrl function to set pad
 *		configuration.
 */
enum rpc_pinctrl_func_idx {
	RPC_PINCTRL_FUNC_ALLOC = 0,
	RPC_PINCTRL_FUNC_FREE,
	RPC_PINCTRL_FUNC_MUX_GET,
	RPC_PINCTRL_FUNC_MUX_SET,
	RPC_PINCTRL_FUNC_PAD_CFG_GET,
	RPC_PINCTRL_FUNC_PAD_CFG_SET,
	RPC_PINCTRL_FUNC_MAX
};

/**
 * enum rpc_gpio_func_idx - possible GPIO rpc functions
 *		supported for individual pins.
 * @RPC_GPIO_FUNC_ALLOC: GPIO allocate function.
 * @RPC_GPIO_FUNC_FREE: GPIO free function.
 */
enum rpc_gpio_func_idx {
	RPC_GPIO_FUNC_ALLOC = 0,
	RPC_GPIO_FUNC_FREE,
	RPC_GPIO_FUNC_MAX
};

/**
 * RPC pin configuration (pull up/down, drive strength, slew,
 * hyst etc) type and its value are packed together in required
 * RPC specified format
 */
#define RPC_PINCONF_TYPE_MASK_OFSET 9
#define RPC_PINCONF_TYPE_MASK_SHIFT(type) (type + RPC_PINCONF_TYPE_MASK_OFSET)

#define RPC_PINCONF_ID_MASK 0x1FF
#define RPC_PINCONF_RET_MASK 0xFF
#define RPC_PINCONF_RET_SHIFT 24
#define RPC_PINCONF_MUX_MASK 0xF
#define RPC_PINCONF_MUX_SHIFT 9

#define RPC_PINCONF_DRV_MASK 0xF
#define RPC_PINCONF_DRV_SHIFT 0
#define RPC_PINCONF_SLEW_MASK 0x3
#define RPC_PINCONF_SLEW_SHIFT 4
#define RPC_PINCONF_HYST_MASK 0x3
#define RPC_PINCONF_HYST_SHIFT 6
#define RPC_PINCONF_INDIS_MASK 0x1
#define RPC_PINCONF_INDIS_SHIFT 8
#define RPC_PINCONF_PULL_MASK 0x3
#define RPC_PINCONF_PULL_SHIFT 9
#define RPC_PINCONF_AMP_MASK 0x1
#define RPC_PINCONF_AMP_SHIFT 11
#define RPC_PINCONF_MHV_MASK 0x1
#define RPC_PINCONF_MHV_SHIFT 13
#define RPC_PINCONF_GMII_MASK 0x1
#define RPC_PINCONF_GMII_SHIFT 12
#define RPC_PINCONF_MPV_MASK 0x1
#define RPC_PINCONF_MPV_SHIFT 14
#define RPC_PINCONF_SRC_MASK 0x1
#define RPC_PINCONF_SRC_SHIFT 15
#define RPC_PINCONF_DGEN_MASK 0x1
#define RPC_PINCONF_DGEN_SHIFT 16
#define RPC_PINCONF_RSEL_MASK 0x1
#define RPC_PINCONF_RSEL_SHIFT 17
#define RPC_PINCONF_GFIL_MASK 0x1
#define RPC_PINCONF_GFIL_SHIFT 18
#define RPC_PINCONF_PWR_MASK 0x1
#define RPC_PINCONF_PWR_SHIFT 19

#define RPC_PINCONF_VALUE_BIT_MASK 0xFFFFF /*19- Bits PAD Config Value */
#define RPC_PINCONF_MASK_BIT_MASK 0x7FFF /*14- Bits PAD Config MASK */
/**
 * RPC Pack Format
 * W2: | 31..20 |
 *     |  RSVD  |
 *     |   19   |  18  |  17  |  16  | 15   | 14   | 13   |
 *     |  PWR   | GFIL | RSEL | DGEN | SRC  | MPV  | MHV  |
 *     |   12   |  11  |10..09| 08   |07..06|05..04|03..00|
 *     |  GMII  | AMP  | PULL |INDIS | HYST | SLEW | DRV  |
 *
 */
#define RPC_PINCONF_DRV_PACK(_param_, _arg_) \
(((_param_) & ~(RPC_PINCONF_DRV_MASK << RPC_PINCONF_DRV_SHIFT)) \
	| ((_arg_ & RPC_PINCONF_DRV_MASK) << RPC_PINCONF_DRV_SHIFT))
#define RPC_PINCONF_SLEW_PACK(_param_, _arg_) \
(((_param_) & ~(RPC_PINCONF_SLEW_MASK << RPC_PINCONF_SLEW_SHIFT)) \
	| ((_arg_ & RPC_PINCONF_SLEW_MASK) << RPC_PINCONF_SLEW_SHIFT))
#define RPC_PINCONF_HYST_PACK(_param_, _arg_) \
(((_param_) & ~(RPC_PINCONF_HYST_MASK << RPC_PINCONF_HYST_SHIFT)) \
	| ((_arg_ & RPC_PINCONF_HYST_MASK) << RPC_PINCONF_HYST_SHIFT))
#define RPC_PINCONF_INDIS_PACK(_param_, _arg_) \
(((_param_) & ~(RPC_PINCONF_INDIS_MASK << RPC_PINCONF_INDIS_SHIFT)) \
	| ((_arg_ & RPC_PINCONF_INDIS_MASK) << RPC_PINCONF_INDIS_SHIFT))
#define RPC_PINCONF_PULL_PACK(_param_, _arg_) \
(((_param_) & ~(RPC_PINCONF_PULL_MASK << RPC_PINCONF_PULL_SHIFT)) \
	| ((_arg_ & RPC_PINCONF_PULL_MASK) << RPC_PINCONF_PULL_SHIFT))
#define RPC_PINCONF_AMP_PACK(_param_, _arg_) \
(((_param_) & ~(RPC_PINCONF_AMP_MASK << RPC_PINCONF_AMP_SHIFT)) \
	| ((_arg_ & RPC_PINCONF_AMP_MASK) << RPC_PINCONF_AMP_SHIFT))
#define RPC_PINCONF_MHV_PACK(_param_, _arg_) \
(((_param_) & ~(RPC_PINCONF_MHV_MASK << RPC_PINCONF_MHV_SHIFT)) \
	| ((_arg_ & RPC_PINCONF_MHV_MASK) << RPC_PINCONF_MHV_SHIFT))
#define RPC_PINCONF_GMII_PACK(_param_, _arg_) \
(((_param_) & ~(RPC_PINCONF_GMII_MASK << RPC_PINCONF_GMII_SHIFT)) \
	| ((_arg_ & RPC_PINCONF_GMII_MASK) << RPC_PINCONF_GMII_SHIFT))
#define RPC_PINCONF_MPV_PACK(_param_, _arg_) \
(((_param_) & ~(RPC_PINCONF_MPV_MASK << RPC_PINCONF_MPV_SHIFT)) \
	| ((_arg_ & RPC_PINCONF_MPV_MASK) << RPC_PINCONF_MPV_SHIFT))
#define RPC_PINCONF_SRC_PACK(_param_, _arg_) \
(((_param_) & ~(RPC_PINCONF_SRC_MASK << RPC_PINCONF_SRC_SHIFT)) \
	| ((_arg_ & RPC_PINCONF_SRC_MASK) << RPC_PINCONF_SRC_SHIFT))
#define RPC_PINCONF_DGEN_PACK(_param_, _arg_) \
(((_param_) & ~(RPC_PINCONF_DGEN_MASK << RPC_PINCONF_DGEN_SHIFT)) \
	| ((_arg_ & RPC_PINCONF_DGEN_MASK) << RPC_PINCONF_DGEN_SHIFT))
#define RPC_PINCONF_RSEL_PACK(_param_, _arg_) \
(((_param_) & ~(RPC_PINCONF_RSEL_MASK << RPC_PINCONF_RSEL_SHIFT)) \
	| ((_arg_ & RPC_PINCONF_RSEL_MASK) << RPC_PINCONF_RSEL_SHIFT))
#define RPC_PINCONF_GFIL_PACK(_param_, _arg_) \
(((_param_) & ~(RPC_PINCONF_GFIL_MASK << RPC_PINCONF_GFIL_SHIFT)) \
	| ((_arg_ & RPC_PINCONF_GFIL_MASK) << RPC_PINCONF_GFIL_SHIFT))
#define RPC_PINCONF_PWR_PACK(_param_, _arg_) \
(((_param_) & ~(RPC_PINCONF_PWR_MASK << RPC_PINCONF_PWR_SHIFT)) \
	| ((_arg_ & RPC_PINCONF_PWR_MASK) << RPC_PINCONF_PWR_SHIFT))

#define RPC_PINCONF_MASK_PACK(_param_, _arg_) \
(((_arg_) & ~(0x1 << _param_))\
	| (0x1 << _param_))

#define RPC_PINCONF_DRV_UNPACK(_conf_) \
((_conf_ >> RPC_PINCONF_DRV_SHIFT) \
	& (RPC_PINCONF_DRV_MASK))
#define RPC_PINCONF_SLEW_UNPACK(_conf_) \
((_conf_ >> RPC_PINCONF_SLEW_SHIFT) \
	& (RPC_PINCONF_SLEW_MASK))
#define RPC_PINCONF_HYST_UNPACK(_conf_)  \
((_conf_ >> RPC_PINCONF_HYST_SHIFT) \
	& (RPC_PINCONF_HYST_MASK))
#define RPC_PINCONF_INDIS_UNPACK(_conf_) \
((_conf_ >> RPC_PINCONF_INDIS_SHIFT) \
	& (RPC_PINCONF_INDIS_MASK))
#define RPC_PINCONF_PULL_UNPACK(_conf_) \
((_conf_ >> RPC_PINCONF_PULL_SHIFT) \
	& (RPC_PINCONF_PULL_MASK))
#define RPC_PINCONF_AMP_UNPACK(_conf_) \
((_conf_ >> RPC_PINCONF_AMP_SHIFT) \
	& (RPC_PINCONF_AMP_MASK))
#define RPC_PINCONF_MHV_UNPACK(_conf_) \
((_conf_ >> RPC_PINCONF_MHV_SHIFT) \
	& (RPC_PINCONF_MHV_MASK))
#define RPC_PINCONF_GMII_UNPACK(_conf_) \
((_conf_ >> RPC_PINCONF_GMII_SHIFT) \
	& (RPC_PINCONF_GMII_MASK))
#define RPC_PINCONF_MPV_UNPACK(_conf_) \
((_conf_ >> RPC_PINCONF_MPV_SHIFT) \
	& (RPC_PINCONF_MPV_MASK))
#define RPC_PINCONF_SRC_UNPACK(_conf_) \
((_conf_ >> RPC_PINCONF_SRC_SHIFT) \
	& (RPC_PINCONF_SRC_MASK))
#define RPC_PINCONF_DGEN_UNPACK(_conf_) \
((_conf_ >> RPC_PINCONF_DGEN_SHIFT) \
	& (RPC_PINCONF_DGEN_MASK))
#define RPC_PINCONF_RSEL_UNPACK(_conf_) \
((_conf_ >> RPC_PINCONF_RSEL_SHIFT) \
	& (RPC_PINCONF_RSEL_MASK))
#define RPC_PINCONF_GFIL_UNPACK(_conf_) \
((_conf_ >> RPC_PINCONF_GFIL_SHIFT) \
	& (RPC_PINCONF_GFIL_MASK))
#define RPC_PINCONF_PWR_UNPACK(_conf_) \
((_conf_ >> RPC_PINCONF_PWR_SHIFT) \
	& (RPC_PINCONF_PWR_MASK))

#endif /* __PINCTRL_BCM_RPC_H */
