 /****************************************************************************
 *
 * Broadcom Proprietary and Confidential.
 * (c) 2016 Broadcom. All rights reserved.
 * The term "Broadcom" refers to Broadcom Limited and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Tim Ross <tross@broadcom.com>
 *****************************************************************************/
#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/platform_device.h>
#include "dqnet.h"
#include "dqnet_priv.h"
#include "dqnet_dbg.h"
#include "dqnet_wifi.h"
/*
 * Open a WiFi device
 *
 * Private function for Broadcom WiFi driver called to have DQNet driver
 * handle the Linux TX & RX interfaces for the WiFi driver. When this is
 * called the WiFi driver has already allocated the net_device with a
 * private data size equal to the size of our dqnet_netdev struct. We
 * initialize the dqnet_netdev struct, open the FAP exception channel
 * if it isn't already open, and enable RX callbacks from the DQM driver
 * if they aren't already. It is up to the WiFi driver to start the
 * network stack's netif Q's.
 *
 * Since the normal dqnet_probe and dqnet_open functions are reused
 * and the WiFi is a special case we signal this using the "name"
 * string in the private data. We also fake out the platform_device
 * structure and use it's private data pointer to hand in the net_device
 * pointer. Upon return the platform_device's private data is pointing
 * to the dqnet_netdev structure.
 *
 * Parameters
 *      dev		device to be opened
 *      if_id		FAP assigned ID of the WiFi device (radio instance)
 *      if_sub_id	WiFi assigned sub-ID (BSS/SSID instance)
 *      dhdol_info	DHD offload info that includes callbacks and context.
 *
 * Returns
 *	0 for success, < 0 is error code
 */
int dqnet_wifi_open_ext(struct net_device *dev, int if_id, int if_sub_id,
		    dqnet_dhdol_info_t *dhdol_info)
{
	int status = 0;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	struct platform_device *pdev;
	void *p0, *p1, *p3;
	int *p2;

	pr_debug("%s:-->\n",dev->name);

	pdev = kmalloc(sizeof(struct platform_device), GFP_KERNEL);
	if (!pdev) {
		netdev_err(dev,
			   "Unable to allocate platform device struct.\n");
		status = -ENOMEM;
		goto done;
	}
	memset(pdev, 0, sizeof(struct platform_device));
	pdev->dev.platform_data = dev;

	p0 = ndev->wifi_priv0;
	p1 = ndev->wifi_priv1;
	p2 = ndev->wifi_priv2;
	p3 = ndev->wifi_priv3;

	memset(ndev, 0, sizeof(struct dqnet_netdev));

	ndev->wifi_priv0 = p0;
	ndev->wifi_priv1 = p1;
	ndev->wifi_priv2 = p2;
	ndev->wifi_priv3 = p3;
	ndev->link_type  = DQNET_LINK_TYPE_WIFI;

	ndev->name = DQNET_WIFI_NAME_STRING;

	ndev->demux_type = DQNET_DEMUX_TYPE_IFID_SUBID;
	ndev->if_id = if_id;
	ndev->if_sub_id = if_sub_id;
	ndev->bmcast_enable = true;
	if (dhdol_info) {
		ndev->dhdol_get_flow = dhdol_info->dhdol_get_flow;
		ndev->dhdol_find_sta = dhdol_info->dhdol_find_sta;
		ndev->dhd_cntx = dhdol_info->dhd_cntx;
	} else {
		ndev->dhdol_get_flow = NULL;
		ndev->dhdol_find_sta = NULL;
		ndev->dhd_cntx = NULL;
	}
	status = dqnet_probe(pdev);
	if (status) {
		netdev_err(dev, "Failure probing device.\n");
		goto done;
	}
	status = dqnet_open(dev);
	if (status) {
		netdev_err(dev, "Failure opening device.\n");
		goto err_remove;
	}
	goto done;

err_remove:
	dqnet_remove(pdev);

done:
	pr_debug("%s:<--\n",dev->name);
	return status;
}
EXPORT_SYMBOL(dqnet_wifi_open_ext);

#ifdef DQNET_DOR_API_V2
/*
 * Open a WiFi device
 *
 * Private function for Broadcom WiFi driver called to have DQNet driver
 * handle the Linux TX & RX interfaces for the WiFi driver. When this is
 * called the WiFi driver has already allocated the net_device with a
 * private data size equal to the size of our dqnet_netdev struct. We
 * initialize the dqnet_netdev struct, open the FAP exception channel
 * if it isn't already open, and enable RX callbacks from the DQM driver
 * if they aren't already. It is up to the WiFi driver to start the
 * network stack's netif Q's.
 *
 * Since the normal dqnet_probe and dqnet_open functions are reused
 * and the WiFi is a special case we signal this using the "name"
 * string in the private data. We also fake out the platform_device
 * structure and use it's private data pointer to hand in the net_device
 * pointer. Upon return the platform_device's private data is pointing
 * to the dqnet_netdev structure.
 *
 * Parameters
 *      dev		device to be opened
 *      if_id		FAP assigned ID of the WiFi device (radio instance)
 *      if_sub_id	WiFi assigned sub-ID (BSS/SSID instance)
 *      dhdol_info	DHD offload info that includes callbacks and context.
 *
 * Returns
 *	0 for success, < 0 is error code
 */
int dqnet_wifi_open_ext2(struct net_device *dev, int if_id, int if_sub_id,
		    dqnet_dhdol_info_t *dhdol_info)
{
	int status = 0;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	struct platform_device *pdev;
	void *p0, *p1, *p3;
	int *p2;

	pr_debug("%s:-->\n",dev->name);

	pdev = kmalloc(sizeof(struct platform_device), GFP_KERNEL);
	if (!pdev) {
		netdev_err(dev,
			   "Unable to allocate platform device struct.\n");
		status = -ENOMEM;
		goto done;
	}
	memset(pdev, 0, sizeof(struct platform_device));
	pdev->dev.platform_data = dev;

	p0 = ndev->wifi_priv0;
	p1 = ndev->wifi_priv1;
	p2 = ndev->wifi_priv2;
	p3 = ndev->wifi_priv3;

	memset(ndev, 0, sizeof(struct dqnet_netdev));

	ndev->wifi_priv0 = p0;
	ndev->wifi_priv1 = p1;
	ndev->wifi_priv2 = p2;
	ndev->wifi_priv3 = p3;
	ndev->link_type  = DQNET_LINK_TYPE_WIFI;

	ndev->name = DQNET_WIFI_NAME_STRING;

	ndev->demux_type = DQNET_DEMUX_TYPE_IFID_SUBID;
	ndev->if_id = if_id;
	ndev->if_sub_id = if_sub_id;
	ndev->bmcast_enable = true;
	if (dhdol_info) {
		ndev->dhd_tx_skb_nethook = dhdol_info->dhd_tx_skb_nethook;
		ndev->dhd_rx_skb_nethook = dhdol_info->dhd_rx_skb_nethook;
		ndev->dhdol_get_flow = dhdol_info->dhdol_get_flow;
		ndev->dhd_cntx = dhdol_info->dhd_cntx;
	} else {
		ndev->dhd_tx_skb_nethook = NULL;
		ndev->dhd_rx_skb_nethook = NULL;
		ndev->dhdol_get_flow = NULL;
		ndev->dhd_cntx = NULL;
	}
	status = dqnet_probe(pdev);
	if (status) {
		netdev_err(dev, "Failure probing device.\n");
		goto done;
	}
	status = dqnet_open(dev);
	if (status) {
		netdev_err(dev, "Failure opening device.\n");
		goto err_remove;
	}
	goto done;

err_remove:
	dqnet_remove(pdev);

done:
	pr_debug("%s:<--\n",dev->name);
	return status;
}
EXPORT_SYMBOL(dqnet_wifi_open_ext2);
#endif

int dqnet_wifi_open(struct net_device *dev, int if_id, int if_sub_id)
{
	return dqnet_wifi_open_ext(dev, if_id, if_sub_id, NULL);
}
EXPORT_SYMBOL(dqnet_wifi_open);

/*
 * Close a WiFi device
 *
 * Private function for Broadcom WiFi driver called to have DQNet driver
 * cleanup from it's handling of the Linux TX & RX interfaces for the
 * WiFi driver.
 *
 * Since the normal dqnet_close and dqnet_remove functions are reused
 * and the WiFi is a special case we signal this using the "name"
 * string in the private data. We hand back the faked out platform_device
 * that we used on open so that dqnet_remove can obtain our dqnet_netdev
 * struct in the same manner it would from the kernel.
 *
 * Parameters
 *	dev	device to be closed
 *
 * Returns
 *	0 for success, < 0 is error code
 */
int dqnet_wifi_close(struct net_device *dev)
{
	int status = 0;
	struct dqnet_netdev *ndev = NULL;
	struct platform_device *pdev = NULL;

	if (!dev) {
		pr_err("dev Null\n");
		return -EINVAL;
	}
	pr_debug("%s:-->\n",dev->name);
	ndev = netdev_priv(dev);
	if (!ndev) {
		pr_err("%s ndev Null\n",dev->name);
		return -EINVAL;
	}
	pdev = ndev->pdev;
	if (!pdev) {
		pr_err("%s pdev Null\n",dev->name);
		return -EINVAL;
	}
	status = dqnet_close(dev);
	if (status)
		netdev_err(dev, "Failure (%d) closing device.\n", status);
	pdev = ndev->pdev;
	status += dqnet_remove(pdev);
	if (status)
		netdev_err(dev, "Failure (%d) removing device.\n", status);
	kfree(pdev);

	pr_debug("%s:<--\n",dev->name);
	return status;
}
EXPORT_SYMBOL(dqnet_wifi_close);
