 /****************************************************************************
 *
 * Copyright (c) 2015 Broadcom Corporation
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************/
/***************************************************************************
*	Filename:       bcmunimac.c
*	Author:         Mark Newcomer
*	Creation Date:  5/6/2013
*	PURPOSE: Common Functions for the Unimac driver.
******************************************************************************/
#include <linux/phy.h>
#include <hal_device.h>
#include "unimacdma.h"
#include "bcmunimac_priv.h"

static char macdev_mib_stats_strings[MACDEV_MIB_STATS_MAX][ETH_GSTRING_LEN] =
MACDEV_MIB_STATS_STRINGS;

static int  __init bcmunimac_module_init(void);
static int  bcmunimac_probe(struct platform_device *pdev);
static int  bcmunimac_remove(struct platform_device *pdev);
static void bcmunimac_shutdown(struct platform_device *pdev);
static void __exit bcmunimac_module_cleanup(void);

static int bcmunimac_del_proc_files(struct mac_device *macdev);
static int bcmunimac_add_proc_files(struct mac_device *macdev);

static const struct __initdata of_device_id unimac_of_match[] = {
	{.compatible = UNIMAC_OF_MATCH},
	{}
};
MODULE_DEVICE_TABLE(of, unimac_of_match);
static struct platform_driver unimac_platform_drv = {
	.probe          =   bcmunimac_probe,
	.remove         =   bcmunimac_remove,
	.shutdown       =   bcmunimac_shutdown,
	.suspend        =   NULL,
	.resume         =   NULL,
	.driver         = {
		.name = UNIMAC_DRV,
		.owner = THIS_MODULE,
		.of_match_table = unimac_of_match
	}
};

/*
 * Get the unimac stats.
 *
 * udev:	the unimac_device ptr
 * stats:	statistics struct to fill in
 *
 * returns:	filled in statistics struct
 */
static struct rtnl_link_stats64 *
bcmunimac_link_stats(struct mac_device *macdev, struct rtnl_link_stats64 *stats)
{
	struct unimac_device *udev;
	uint64_t val;

	pr_debug("%s:-->\n", macdev->name);

	udev = macdev_priv(macdev);
	if (stats) {
		volatile struct unimac_mib __iomem *mib =
			(volatile struct unimac_mib __iomem *)(udev->reg_base +
					udev->unimac_mib_offset);
		stats->rx_bytes = (u64)__raw_readl(&mib->grbyt);
		stats->rx_packets = (u64)__raw_readl(&mib->grpkt);
		val = (u64)__raw_readl(&mib->grpok);
		if (val <= stats->rx_packets)
			stats->rx_errors += stats->rx_packets - val;

		stats->multicast = (u64)__raw_readl(&mib->grmca);

		stats->rx_length_errors =
			(u64)__raw_readl(&mib->grflr);
		stats->rx_length_errors +=
			(u64)__raw_readl(&mib->grjbr);
		stats->rx_length_errors +=
			(u64)__raw_readl(&mib->rrpkt);
		stats->rx_length_errors +=
			(u64)__raw_readl(&mib->rrund);
		stats->rx_length_errors +=
			(u64)__raw_readl(&mib->rrfrg);

		stats->rx_over_errors =
			(u64)__raw_readl(&mib->grovr);
		stats->rx_crc_errors =
			(u64)__raw_readl(&mib->grfcs);
		stats->rx_frame_errors =
			(u64)__raw_readl(&mib->graln);

		stats->tx_bytes = (u64)__raw_readl(&mib->gtbyt);
		stats->tx_packets = (u64)__raw_readl(&mib->gtpkt);
		val = (u64)__raw_readl(&mib->gtpok);
		if (val <= stats->tx_packets)
			stats->tx_errors += stats->tx_packets - val;

		stats->collisions = (u64)__raw_readl(&mib->gtncl);
		stats->tx_aborted_errors =
			(u64)__raw_readl(&mib->gtjbr);
		stats->tx_aborted_errors +=
			(u64)__raw_readl(&mib->gtovr);
		stats->tx_aborted_errors +=
			(u64)__raw_readl(&mib->gtfrg);
		stats->tx_aborted_errors +=
			(u64)__raw_readl(&mib->gtfcs);

		stats->tx_window_errors =
			(u64)__raw_readl(&mib->gtlcl);
	} else {
		/* Clear out ALL the MIB counters */
		volatile void __iomem *base = udev->reg_base;
		val = 0;
		SET_FLD(val, IFACE_CLR_MIB_COUNTER, 1);
		__raw_writel(val, base + udev->iface_clear_mib_reg);
		mdelay(1);
		SET_FLD(val, IFACE_CLR_MIB_COUNTER, 0);
		__raw_writel(val, base + udev->iface_clear_mib_reg);
	}

	pr_debug("%s:<--\n", macdev->name);

	return stats;
}

static void bcmunimac_get_strings(struct mac_device *macdev, u32 stringset,
		u8 *strings)
{
	enum ethtool_stringset ss = stringset;

	pr_debug("%s:-->\n", macdev->name);

	if (ss == ETH_SS_STATS)
		memcpy(strings, macdev_mib_stats_strings,
				sizeof(macdev_mib_stats_strings));

	pr_debug("%s:<--\n", macdev->name);
}

static void bcmunimac_get_ethtool_stats(struct mac_device *macdev,
		struct ethtool_stats *stats, u64 *data)
{
	struct net_device *netdev;
	struct phy_device *phydev;
	struct unimac_device *udev;
	volatile struct unimac_mib __iomem *mib;
	int speed = SPEED_1000;

	pr_debug("%s:-->\n", macdev->name);

	netdev = macdev->hal->netdev;
	if (stats->cmd != ETHTOOL_GSTATS) {
		netdev_err(netdev, "Invalid command, %d.\n", stats->cmd);
		goto _ret_func;
	}

	if (stats->n_stats != MACDEV_MIB_STATS_MAX) {
		netdev_err(netdev, "Incorrect stats array size, %d. Must be %d.\n",
				stats->n_stats, MACDEV_MIB_STATS_MAX);
		goto _ret_func;
	}

	phydev = macdev->hal->phydev;
	if (phydev && phydev->link)
		speed = phydev->speed;

	udev = macdev_priv(macdev);
	mib = (volatile struct unimac_mib __iomem *)(udev->reg_base +
			udev->unimac_mib_offset);

	data[MACDEV_MIB_STATS_RX_64_BYTES_PACKETS] = (u64)__raw_readl(&mib->gr64);
	data[MACDEV_MIB_STATS_RX_127_BYTES_PACKETS] = (u64)__raw_readl(&mib->gr127);
	data[MACDEV_MIB_STATS_RX_255_BYTES_PACKETS] = (u64)__raw_readl(&mib->gr255);
	data[MACDEV_MIB_STATS_RX_511_BYTES_PACKETS] = (u64)__raw_readl(&mib->gr511);
	data[MACDEV_MIB_STATS_RX_1023_BYTES_PACKETS] = (u64)__raw_readl(&mib->gr1023);
	data[MACDEV_MIB_STATS_RX_1518_BYTES_PACKETS] = (u64)__raw_readl(&mib->gr1518);
	data[MACDEV_MIB_STATS_RX_1522_BYTES_PACKETS] = (u64)__raw_readl(&mib->gr1522);
	data[MACDEV_MIB_STATS_RX_2047_BYTES_PACKETS] = (u64)__raw_readl(&mib->gr2047);
	data[MACDEV_MIB_STATS_RX_4095_BYTES_PACKETS] = (u64)__raw_readl(&mib->gr4095);
	data[MACDEV_MIB_STATS_RX_9216_BYTES_PACKETS] = (u64)__raw_readl(&mib->gr9216);
	data[MACDEV_MIB_STATS_RX_16383_BYTES_PACKETS] = 0;
	data[MACDEV_MIB_STATS_RX_PACKETS] = (u64)__raw_readl(&mib->grpkt);
	data[MACDEV_MIB_STATS_RX_UNICAST_PACKETS] = (u64)__raw_readl(&mib->gruc);
	data[MACDEV_MIB_STATS_RX_MULTICAST_PACKETS] = (u64)__raw_readl(&mib->grmca);
	data[MACDEV_MIB_STATS_RX_BROADCAST_PACKETS] = (u64)__raw_readl(&mib->grbca);
	data[MACDEV_MIB_STATS_RX_FCS_ERRORS] = (u64)__raw_readl(&mib->grfcs);
	data[MACDEV_MIB_STATS_RX_CONTROL_PACKETS] = (u64)__raw_readl(&mib->grxcf);
	data[MACDEV_MIB_STATS_RX_PAUSE_PACKETS] = (u64)__raw_readl(&mib->grxpf);
	data[MACDEV_MIB_STATS_RX_OPCODE_ERRORS] = (u64)__raw_readl(&mib->grxuo);
	data[MACDEV_MIB_STATS_RX_ALIGNMENT_ERRORS] = (u64)__raw_readl(&mib->graln);
	data[MACDEV_MIB_STATS_RX_LENGTH_ERRORS] = (u64)__raw_readl(&mib->grflr);
	data[MACDEV_MIB_STATS_RX_CODE_ERRORS] = (u64)__raw_readl(&mib->grcde);
	data[MACDEV_MIB_STATS_RX_CARRIER_ERRORS] = (u64)__raw_readl(&mib->grfcr);
	data[MACDEV_MIB_STATS_RX_OVERSIZED_ERRORS] = (u64)__raw_readl(&mib->grovr);
	data[MACDEV_MIB_STATS_RX_JABBER_ERRORS] = (u64)__raw_readl(&mib->grjbr);
	data[MACDEV_MIB_STATS_RX_MTU_ERRORS] = (u64)__raw_readl(&mib->grmtue);
	data[MACDEV_MIB_STATS_RX_MATCHED_CRC_PACKETS] = (u64)__raw_readl(&mib->grcrc);
	data[MACDEV_MIB_STATS_RX_SINGLE_VLAN_PACKETS] = 0;
	data[MACDEV_MIB_STATS_RX_DOUBLE_VLAN_PACKETS] = 0;
	data[MACDEV_MIB_STATS_RX_TRUNCATED_ERRORS] = 0;
	data[MACDEV_MIB_STATS_RX_GOOD_PACKETS] = (u64)__raw_readl(&mib->grpok);
	data[MACDEV_MIB_STATS_RX_BYTES] = (u64)__raw_readl(&mib->grbyt);
	data[MACDEV_MIB_STATS_RX_RUNT_PACKETS] = (u64)__raw_readl(&mib->rrpkt);
	data[MACDEV_MIB_STATS_RX_UNDERSIZED_ERRORS] = (u64)__raw_readl(&mib->rrund);
	data[MACDEV_MIB_STATS_RX_FRAGMENT_ERRORS] = (u64)__raw_readl(&mib->rrfrg);
	data[MACDEV_MIB_STATS_RX_RUNT_BYTES] = (u64)__raw_readl(&mib->rrbyt);
	data[MACDEV_MIB_STATS_RX_ERRORS] = 0;
	if (data[MACDEV_MIB_STATS_RX_GOOD_PACKETS] <=
			data[MACDEV_MIB_STATS_RX_PACKETS])
		data[MACDEV_MIB_STATS_RX_ERRORS] =
			data[MACDEV_MIB_STATS_RX_PACKETS] -
			data[MACDEV_MIB_STATS_RX_GOOD_PACKETS];

	data[MACDEV_MIB_STATS_TX_CAPACITY] = speed;
	data[MACDEV_MIB_STATS_TX_64_BYTES_PACKETS] = (u64)__raw_readl(&mib->tr64);
	data[MACDEV_MIB_STATS_TX_127_BYTES_PACKETS] = (u64)__raw_readl(&mib->tr127);
	data[MACDEV_MIB_STATS_TX_255_BYTES_PACKETS] = (u64)__raw_readl(&mib->tr255);
	data[MACDEV_MIB_STATS_TX_511_BYTES_PACKETS] = (u64)__raw_readl(&mib->tr511);
	data[MACDEV_MIB_STATS_TX_1023_BYTES_PACKETS] = (u64)__raw_readl(&mib->tr1023);
	data[MACDEV_MIB_STATS_TX_1518_BYTES_PACKETS] = (u64)__raw_readl(&mib->tr1518);
	data[MACDEV_MIB_STATS_TX_1522_BYTES_PACKETS] = (u64)__raw_readl(&mib->tr1522);
	data[MACDEV_MIB_STATS_TX_2047_BYTES_PACKETS] = (u64)__raw_readl(&mib->tr2047);
	data[MACDEV_MIB_STATS_TX_4095_BYTES_PACKETS] = (u64)__raw_readl(&mib->tr4095);
	data[MACDEV_MIB_STATS_TX_9216_BYTES_PACKETS] = (u64)__raw_readl(&mib->tr9216);
	data[MACDEV_MIB_STATS_TX_16383_BYTES_PACKETS] = 0;
	data[MACDEV_MIB_STATS_TX_GOOD_PACKETS] = (u64)__raw_readl(&mib->gtpok);
	data[MACDEV_MIB_STATS_TX_PACKETS] = (u64)__raw_readl(&mib->gtpkt);
	data[MACDEV_MIB_STATS_TX_UNICAST_PACKETS] = (u64)__raw_readl(&mib->gtuc);
	data[MACDEV_MIB_STATS_TX_MULTICAST_PACKETS] = (u64)__raw_readl(&mib->gtmca);
	data[MACDEV_MIB_STATS_TX_BROADCAST_PACKETS] = (u64)__raw_readl(&mib->gtbca);
	data[MACDEV_MIB_STATS_TX_PAUSE_PACKETS] = (u64)__raw_readl(&mib->gtxpf);
	data[MACDEV_MIB_STATS_TX_JABBER_ERRORS] = (u64)__raw_readl(&mib->gtjbr);
	data[MACDEV_MIB_STATS_TX_FCS_ERRORS] = (u64)__raw_readl(&mib->gtfcs);
	data[MACDEV_MIB_STATS_TX_CONTROL_PACKETS] = (u64)__raw_readl(&mib->gtxcf);
	data[MACDEV_MIB_STATS_TX_OVERSIZED_ERRORS] = (u64)__raw_readl(&mib->gtovr);
	data[MACDEV_MIB_STATS_TX_SINGLE_DEFERRALS] = (u64)__raw_readl(&mib->gtdrf);
	data[MACDEV_MIB_STATS_TX_MULTIPLE_DEFERRALS] = (u64)__raw_readl(&mib->gtedf);
	data[MACDEV_MIB_STATS_TX_SINGLE_COLLISIONS] = (u64)__raw_readl(&mib->gtscl);
	data[MACDEV_MIB_STATS_TX_MULTIPLE_COLLISIONS] = (u64)__raw_readl(&mib->gtmcl);
	data[MACDEV_MIB_STATS_TX_LATE_COLLISIONS] = (u64)__raw_readl(&mib->gtlcl);
	data[MACDEV_MIB_STATS_TX_EXCESSIVE_COLLISIONS] = (u64)__raw_readl(&mib->gtxcl);
	data[MACDEV_MIB_STATS_TX_FRAGMENT_ERRORS] = (u64)__raw_readl(&mib->gtfrg);
	data[MACDEV_MIB_STATS_TX_SYSTEM_ERRORS] = 0;
	data[MACDEV_MIB_STATS_TX_SINGLE_VLAN_PACKETS] = 0;
	data[MACDEV_MIB_STATS_TX_DOUBLE_VLAN_PACKETS] = 0;
	data[MACDEV_MIB_STATS_TX_RUNT_PACKETS] = 0;
	data[MACDEV_MIB_STATS_TX_UNDERRUN_ERRORS] = 0;
	data[MACDEV_MIB_STATS_TX_TOTAL_COLLISIONS] = (u64)__raw_readl(&mib->gtncl);
	data[MACDEV_MIB_STATS_TX_BYTES] = (u64)__raw_readl(&mib->gtbyt);
	data[MACDEV_MIB_STATS_TX_ERRORS] = 0;
	if (data[MACDEV_MIB_STATS_TX_GOOD_PACKETS] <=
			data[MACDEV_MIB_STATS_TX_PACKETS])
		data[MACDEV_MIB_STATS_TX_ERRORS] =
			data[MACDEV_MIB_STATS_TX_PACKETS] -
			data[MACDEV_MIB_STATS_TX_GOOD_PACKETS];

_ret_func:
	pr_debug("%s:<--\n", macdev->name);
}

static int bcmunimac_get_sset_count(struct mac_device *macdev, int stringset)
{
	enum ethtool_stringset ss = stringset;
	int count = 0;

	pr_debug("%s:-->\n", macdev->name);

	if (ss == ETH_SS_STATS)
		count = MACDEV_MIB_STATS_MAX;

	pr_debug("%s:<--\n", macdev->name);

	return count;
}

static int bcmunimac_set_mac_address(struct mac_device *macdev, void *addr)
{
	struct unimac_device *udev;
	volatile void __iomem *base;
	unsigned char *mac = addr;
	u32 val;

	pr_debug("%s:-->\n", macdev->name);

	udev = macdev_priv(macdev);
	base = udev->reg_base;

	/* Set the MAC address bits [47:16] */
	val = ((u32)mac[0] << 24) | ((u32)mac[1] << 16) |
		((u32)mac[2] << 8) | mac[3];
	__raw_writel(val, base + udev->core_mac0_reg);

	/* Set the MAC address bits [15:0] */
	val = ((u32)mac[4] << 8) | mac[5];
	__raw_writel(val, base + udev->core_mac1_reg);

	pr_debug("%s:<--\n", macdev->name);

	return 0;
}

/*
 * bcmunimac_init: Initialize the ethernet mac in
 * Ethernet mode.
 *
 * @author newcomer (6/24/2013)
 */
static int bcmunimac_init(struct mac_device *macdev)
{
	struct net_device *netdev;
	struct phy_device *phydev;
	struct unimac_device *udev;
	volatile void __iomem *base;
	u32 val = 0;

	pr_debug("%s:-->\n", macdev->name);

	netdev = macdev->hal->netdev;
	phydev = macdev->hal->phydev;
	udev = macdev_priv(macdev);
	if (phydev && phydev->interface == PHY_INTERFACE_MODE_GMII)
		udev->onchip_phy = 1;

	base = udev->reg_base;

	SET_FLD(val, CORE_CMD_SW_RESET, 1);
	__raw_writel(val, base + udev->core_cmd_reg);

	/* Clear out ALL the MIB counters */
	val = 0;
	SET_FLD(val, IFACE_CLR_MIB_COUNTER, 1);
	__raw_writel(val, base + udev->iface_clear_mib_reg);
	mdelay(1);
	SET_FLD(val, IFACE_CLR_MIB_COUNTER, 0);
	__raw_writel(val, base + udev->iface_clear_mib_reg);

	__raw_writel(0, base + udev->core_cmd_reg);

	val = __raw_readl(base + udev->core_cmd_reg);
	/* Tell Unimac to strip CRCs */
	SET_FLD(val, CORE_CMD_CRC_FWD, 0);
	/* Turn on promiscuous mode */
	SET_FLD(val, CORE_CMD_PROMIS_EN, 1);
	/* Turn on length checking */
	SET_FLD(val, CORE_CMD_NO_LGTH_CHECK, 0);
	__raw_writel(val, base + udev->core_cmd_reg);

	bcmunimac_set_mac_address(macdev, netdev->dev_addr);
#if 0
	/* Set the clock freq to 2.5Mhz based on 160Mhz UBUS clock */
	val = __raw_readl(base + IFACE_MDIO_CFG);
	SET_FLD(val, IFACE_MDIO_CFG_CLK_DIV, GMAC_CLK_DIVIDER);
	__raw_writel(val, base + IFACE_MDIO_CFG);
#endif

	/* Let the UNIMAC drop errored packets */
	val = __raw_readl(base + udev->iface_enable_drop_pkt_reg);
	SET_FLD(val, IFACE_EN_DROP_PKT_RX_ERR, 1);
	SET_FLD(val, IFACE_EN_DROP_PKT_CRC_ERR, 1);
	SET_FLD(val, IFACE_EN_DROP_PKT_FRAME_TRUNC, 1);
	SET_FLD(val, IFACE_EN_DROP_PKT_RX_FULL, 1);
	__raw_writel(val, base + udev->iface_enable_drop_pkt_reg);

	/*
	 *  Set UnimacInterface Control to use External Phy Connected
	 *  by RGMII.
	 *
	 *  For the unimacs that connect to the switch in 3385/7145,
	 *  set this even though there is no phy.
	 */
	if (udev->onchip_phy) {
		val = __raw_readl(base + udev->iface_control_reg);
		SET_FLD(val, IFACE_CTRL_ONCHIP_PHY, 1);
		__raw_writel(val, base + udev->iface_control_reg);
	}

	dma_init((void *)udev);

	/*
	 * Here we allocate in the Unimac the amount of extra room (pad)
	 * to place our internal headers
	 */
	val = __raw_readl(base + udev->iface_control_reg);
	SET_FLD(val, IFACE_CTRL_RX_BYTE_PAD, (udev->fpm_rxbuffer_offset +
					      udev->fpm_rxbuffer_pad));
	__raw_writel(val, base + udev->iface_control_reg);


	/*
	 * Setup and Enable the following UniMac interrupts
	 *  IMPORTANT: This needs to be done after the MBDMA otherwise
	 *  there could be a race condition that showed up when using external
	 *  PHY's where UniMac generates an interrupt before DMA is setup
	 *  and we check DMA regs in the ISR
	 */
	val = __raw_readl(base + udev->iface_enable_irq_reg);
	SET_FLD(val, IFACE_EN_IRQ_LINKUP, 1);
	SET_FLD(val, IFACE_EN_IRQ_LINKDOWN, 1);
	SET_FLD(val, IFACE_EN_IRQ_PPP_VAL, 1);
	SET_FLD(val, IFACE_EN_IRQ_RX_FULL, 1);
	SET_FLD(val, IFACE_EN_IRQ_RUNT_TRUNC, 1);
	SET_FLD(val, IFACE_EN_IRQ_FRAME_TRUNC, 1);
	SET_FLD(val, IFACE_EN_IRQ_CRC_ERR, 1);
	SET_FLD(val, IFACE_EN_IRQ_RX_ERR, 1);
#ifdef BCMUNIMAC_MBDMA_INTERRUPT_SUPPORT
	__raw_writel(val, base + udev->iface_enable_irq_reg);
#endif

	pr_debug("%s:<--\n", macdev->name);

	return 0;
}

static void bcmunimac_uninit(struct mac_device *macdev)
{
	struct unimac_device *udev;
	volatile void __iomem *base;
	u32 val;

	pr_debug("%s:-->\n", macdev->name);

	udev = macdev_priv(macdev);
	base = udev->reg_base;

	val = __raw_readl(base + udev->iface_control_reg);
	if (udev->onchip_phy)
		SET_FLD(val, IFACE_CTRL_ONCHIP_PHY, 0);

	SET_FLD(val, IFACE_CTRL_RX_BYTE_PAD, 0);
	__raw_writel(val, base + udev->iface_control_reg);

	val = 0;
	SET_FLD(val, CORE_CMD_SW_RESET, 1);
	__raw_writel(val, base + udev->core_cmd_reg);

	/* Clear out ALL the MIB counters */
	val = 0;
	SET_FLD(val, IFACE_CLR_MIB_COUNTER, 1);
	__raw_writel(val, base + udev->iface_clear_mib_reg);
	mdelay(1);
	SET_FLD(val, IFACE_CLR_MIB_COUNTER, 0);
	__raw_writel(val, base + udev->iface_clear_mib_reg);

	pr_debug("%s:<--\n", macdev->name);
}

/*
 * Enable UniMAC datapath
 * Set the unimac link parameters.
 *
 * macdev: unimac's mac_device
 * udev:	the unimac_device ptr
 * link:	link up (1) or down (0)
 * speed: 	the link speed, one of SPEED_10, SPEED_100, or SPEED_1000
 * duplex:	full or half, one of DUPLEX_FULL or DUPLEX_HALF
 * pause:	pause enabled (1) or disabled (0)
 *
 * returns: 0 for success, non-0 for failure
 */
static int bcmunimac_open(struct mac_device *macdev)
{
	struct phy_device *phydev;
	struct unimac_device *udev;
	volatile void __iomem *base;
	int speed = SPEED_1000, duplex = DUPLEX_FULL, pause = 0, link = 1;
	u32 val;

	pr_debug("%s:-->\n", macdev->name);

	phydev = macdev->hal->phydev;
	if (phydev) {
		speed = phydev->speed;
		duplex = phydev->duplex;
		pause = phydev->pause;
		link = phydev->link;
	}

	udev = macdev_priv(macdev);
	base = udev->reg_base;
	val = __raw_readl(base + udev->core_cmd_reg);

	switch (speed) {
	case SPEED_10:
		SET_FLD(val, CORE_CMD_ETH_SPEED, 0);
		break;
	case SPEED_100:
		SET_FLD(val, CORE_CMD_ETH_SPEED, 1);
		break;
	case SPEED_1000:
		SET_FLD(val, CORE_CMD_ETH_SPEED, 2);
		break;
	case SPEED_2500:
		SET_FLD(val, CORE_CMD_ETH_SPEED, 3);
		break;
	default:
		pr_err("Invalid speed, %d, specified.\n", speed);
		return -EINVAL;
	}

	switch (duplex) {
	case DUPLEX_HALF:
		SET_FLD(val, CORE_CMD_HD_ENA, 1);
		break;
	case DUPLEX_FULL:
		SET_FLD(val, CORE_CMD_HD_ENA, 0);
		break;
	default:
		pr_err("Invalid duplex, %d, specified.\n", duplex);
		return -EINVAL;
	}

	if (pause) {
		SET_FLD(val, CORE_CMD_RX_PAUSE_IGNORE, 0);
		SET_FLD(val, CORE_CMD_TX_PAUSE_IGNORE, 0);
	} else {
		SET_FLD(val, CORE_CMD_RX_PAUSE_IGNORE, 1);
		SET_FLD(val, CORE_CMD_TX_PAUSE_IGNORE, 1);
	}

	__raw_writel(val, base + udev->core_cmd_reg);

	if (udev->onchip_phy) {
		val = __raw_readl(base + udev->iface_rgmii_ctrl_reg);
		SET_FLD(val, IFACE_RGMII_CTRL_LINK, link ? 1 : 0);
		__raw_writel(val, base + udev->iface_rgmii_ctrl_reg);
	}

	/* Enable Tx/Rx */
	val = __raw_readl(base + udev->core_cmd_reg);
	SET_FLD(val, CORE_CMD_RX_ENA, 1);
	SET_FLD(val, CORE_CMD_TX_ENA, 1);
	__raw_writel(val, base + udev->core_cmd_reg);

	pr_debug("%s:<--\n", macdev->name);

	return 0;
}

/*
 * Disable UniMAC datapath
 *
 * macdev: unimac's mac_device
 *
 * returns: 0 for success, non-0 for failure
 */
static int bcmunimac_close(struct mac_device *macdev)
{
	struct unimac_device *udev;
	volatile void __iomem *base;
	u32 val;

	pr_debug("%s:-->\n", macdev->name);

	udev = macdev_priv(macdev);
	base = udev->reg_base;

	/* Disable Tx/Rx */
	val = __raw_readl(base + udev->core_cmd_reg);
	SET_FLD(val, CORE_CMD_RX_ENA, 0);
	SET_FLD(val, CORE_CMD_TX_ENA, 0);
	__raw_writel(val, base + udev->core_cmd_reg);

	pr_debug("%s:<--\n", macdev->name);

	return 0;
}

static int bcmunimac_mtu(struct mac_device *macdev, int new_mtu)
{
	struct unimac_device *udev;
	volatile void __iomem *base;
	u32 val;

	pr_debug("%s:-->\n", macdev->name);

	udev = macdev_priv(macdev);
	base = udev->reg_base;

	/* Set the frame length */
	val = __raw_readl(base + udev->core_frame_len_reg);
	SET_FLD(val, CORE_FRM_LEN_FL, new_mtu);
	__raw_writel(val, base + udev->core_frame_len_reg);

	/* Set the mib frame length */
	val = __raw_readl(base + udev->iface_mib_pktmax);
	SET_FLD(val, IFACE_MIB_PKTMAX_SIZE, new_mtu);
	__raw_writel(val, base + udev->iface_mib_pktmax);

	pr_debug("%s:<--\n", macdev->name);

	return 0;
}

static struct mac_device_ops bcmunimac_macdev_ops = {
	.init			= bcmunimac_init,
	.uninit			= bcmunimac_uninit,
	.open			= bcmunimac_open,
	.stop			= bcmunimac_close,
	.set_mac_address	= bcmunimac_set_mac_address,
	.change_mtu		= bcmunimac_mtu,
	.link_stats		= bcmunimac_link_stats,
	.get_strings		= bcmunimac_get_strings,
	.get_ethtool_stats	= bcmunimac_get_ethtool_stats,
	.get_sset_count		= bcmunimac_get_sset_count
};

/*
 * bcmunimac_module_init: - register the unimac driver as a
 * platform device.
 *
 * @author newcomer (6/24/2013)
 *
 * @return int code from kernel registration call
 */
static int __init bcmunimac_module_init(void)
{
	/* print the driver module version info */
	pr_info("%s driver v%s\n", UNIMAC_DRV, MODULE_VER);
	haldev_move(UNIMAC_OF_MATCH);
	return(platform_driver_register(&unimac_platform_drv));
}
module_init(bcmunimac_module_init);

/*
 * bcmunimac_probe: - unimac hardware init is done here.
 *
 * @author newcomer (6/24/2013)
 *
 * @param pdev
 *
 * @return int error code if probe of device fails, otherwise 0
 */
static int bcmunimac_probe(struct platform_device *pdev)
{
	struct device *pdd = &pdev->dev;
	struct resource *mem;
	struct fpm_hw_info fpm_hwinfo;
	struct mac_device *macdev;
	struct unimac_device *udev;
	int status;

	status = macdev_alloc(pdd, sizeof(struct unimac_device));
	if (status)
		goto done;

	status = bcmunimac_parse_dt_node(pdev);
	if (status) {
		pr_err("bcmunimac: Device Tree Parse Error: Exiting");
		goto err_free_udev;
	}

	macdev = pdd->platform_data;
	macdev->macdev_ops = &bcmunimac_macdev_ops;
	udev = macdev_priv(macdev);
	pr_info("bcmunimac_probe: instance %d, %s\n",
		   udev->instance, pdev->name);

	mem = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!mem) {
		pr_err("bcmunimac: unable to retrieve reg base\n");
		status = -EFAULT;
		goto err_free_dev;
	}

	udev->reg_base = ioremap(mem->start, mem->end - mem->start);
	if (!udev->reg_base) {
		pr_err("bcmunimac: unable to ioremap reg base\n");
		status = -EFAULT;
		goto err_free_dev;
	}

	udev->iface_control_reg = udev->unimac_iface_offset + IFACE_CTRL;
	udev->iface_clear_mib_reg = udev->unimac_iface_offset + IFACE_CLR_MIB;
	udev->iface_mib_pktmax = udev->unimac_iface_offset + IFACE_MIB_PKTMAX;
	udev->iface_enable_irq_reg = udev->unimac_iface_offset + IFACE_EN_IRQ;
	udev->iface_mdio_cfg_clk_div_reg = udev->unimac_iface_offset + IFACE_MDIO_CFG_CLK_DIV;
	udev->iface_rgmii_ctrl_reg = udev->unimac_iface_offset + IFACE_RGMII_CTRL;
	udev->iface_enable_drop_pkt_reg = udev->unimac_iface_offset + IFACE_EN_DROP_PKT;
	udev->core_cmd_reg = udev->unimac_core_offset + CORE_CMD;
	udev->core_mac0_reg = udev->unimac_core_offset + CORE_MAC0;
	udev->core_mac1_reg = udev->unimac_core_offset + CORE_MAC1;
	udev->core_frame_len_reg = udev->unimac_core_offset + CORE_FRM_LEN;

	status = fpm_get_hw_info(&fpm_hwinfo);
	if (status)
		goto err_free_dev;

	udev->fpm_rxbuffer_offset = fpm_hwinfo.net_buf_head_pad;
	udev->fpm_rxbuffer_pad = fpm_hwinfo.net_buf_tail_pad;
	pr_debug("bcmunimac_probe: fpm offset = %d, fpm padd = %d\n",
			 udev->fpm_rxbuffer_offset,
			 udev->fpm_rxbuffer_pad);

	bcmunimac_add_proc_files(macdev);

	pr_debug("bcmunimac: %s initialized successfully\n", pdev->name);
	goto done;

err_free_dev:
	if (udev->reg_base)
		iounmap(udev->reg_base);

err_free_udev:
	macdev_free(pdd);

done:
	return status;
}

/*
 * Part of the /proc filesystem. This file access returns
 * information on the unimac and dma status and settings.
 *
 * @author newcomer (6/25/2013)
 *
 * @param page
 * @param start
 * @param off
 * @param cnt
 * @param eof
 * @param data
 *
 * @return int
 */
static ssize_t proc_get_dma_summary(struct file *filp, char __user *page,
				size_t cnt, loff_t *data)
{
	struct unimac_device *udev = (struct unimac_device *)filp->private_data;

	dma_dump_status((void *)udev);
	return 0;
}

/*
 * Part of the /proc filesystem. This file access sets the debug
 * output verbosity of the unimac/dma driver.
 *
 * TODO: list settings!
 *
 * @author newcomer (6/25/2013)
 *
 * @param file
 * @param buffer
 * @param count
 * @param data
 *
 * @return int
 */
static ssize_t proc_set_verbose(struct file *file, const char __user *buffer,
				size_t count, loff_t *data)
{
	static char buf[2];
	struct unimac_device *udev = (struct unimac_device *)data;

	if (!count)
		return 0;
	if (count > 2)
		return -EINVAL;

	if (copy_from_user(buf, buffer, count))
		return -EFAULT;

	udev->verbosity = buf[0] - '0';
	return count;
}

/*
 * Part of the /proc filesystem. This file access returns the
 * debug output verbosity setting of the unimac/dma driver.
 *
 * @author newcomer (6/25/2013)
 *
 * @param page
 * @param start
 * @param off
 * @param cnt
 * @param eof
 * @param data
 *
 * @return int
 */
static ssize_t proc_get_verbose(struct file *filp, char __user *page,
				size_t cnt, loff_t *data)
{
	struct unimac_device *udev = (struct unimac_device *)filp->private_data;
	int r;

	if (*data)
		return 0;

	r = snprintf(page, cnt, "verbose mode is %d\n", udev->verbosity);
	*data += r;
	return r;
}

static int bcmunimac_del_proc_files(struct mac_device *macdev)
{
	char tmp[32];

	snprintf(tmp, sizeof(tmp), "driver/%s/verbose", macdev->name);
	remove_proc_entry(tmp, NULL);

	snprintf(tmp, sizeof(tmp), "driver/%s/dma_summary", macdev->name);
	remove_proc_entry(tmp, NULL);

	snprintf(tmp, sizeof(tmp), "driver/%s", macdev->name);
	remove_proc_entry(tmp, NULL);
	return 0;
}

static int proc_open(struct inode *inode, struct file *f)
{
	f->private_data = PDE_DATA(inode);
	if (!f->private_data)
		return -EPERM;
	return 0;
}

static const struct file_operations bcmunimac_proc_verbose_ops = {
	.owner  = THIS_MODULE,
	.open = proc_open,
	.read = proc_get_verbose,
	.write  = proc_set_verbose,
};

static const struct file_operations bcmunimac_proc_summary_ops = {
	.owner  = THIS_MODULE,
	.open = proc_open,
	.read = proc_get_dma_summary,
};

static int bcmunimac_add_proc_files(struct mac_device *macdev)
{
	char tmp[64];
	struct proc_dir_entry *entry;
	struct unimac_device *udev = macdev_priv(macdev);

	snprintf(tmp, sizeof(tmp), "driver/%s", macdev->name);
	proc_mkdir (tmp, NULL);

	snprintf(tmp, sizeof(tmp), "driver/%s/dma_summary", macdev->name);
	proc_create_data(tmp, 0, NULL, &bcmunimac_proc_summary_ops, udev);

	snprintf(tmp, sizeof(tmp), "driver/%s/verbose", macdev->name);
	entry = proc_create_data(tmp, 0, NULL, &bcmunimac_proc_verbose_ops, udev);
	return 0;
}

#ifdef BCMUNIMAC_USE_POWERCONTROL
/*
 * Power down the entire Unimac/DMA block.
 * WARNING: full powerdown not yet complete on the 7145/3385.
 *
 * @author newcomer (6/25/2013)
 */
void bcmunimac_power_down( void )
{
	if (!unimac_powered_up)
		return;
	unimac_powered_up = false;
}

/*
 * Power down the entire Unimac/DMA block.
 * WARNING: full powerdown not yet complete on the 7145/3385.
 *
 * @author newcomer (6/25/2013)
 */
void bcmunimac_power_up(void)
{
	if (unimac_powered_up)
		return;
	unimac_powered_up = true;

	/* Delaying approsx 200 clocks (1 us), for 108Mhz Unimac */
	Delayus(2);
}
#endif

static void __exit bcmunimac_module_cleanup(void)
{
	pr_debug("bcmunimac: module_cleanup called\n");
	platform_driver_unregister(&unimac_platform_drv);
}
module_exit(bcmunimac_module_cleanup);

static int bcmunimac_remove(struct platform_device *pdev)
{
	struct device *pdd = &pdev->dev;
	struct mac_device *macdev = pdd->platform_data;
	struct unimac_device *udev;

	if (!macdev)
		goto done;

	pr_debug("bcmunimac_remove: instance %d, %s\n",
			 udev->instance, pdev->name);

	bcmunimac_del_proc_files(macdev);
	udev = macdev_priv(macdev);
	iounmap(udev->reg_base);
	macdev_free(pdd);

done:
	return 0;
}

static void bcmunimac_shutdown(struct platform_device *pdev)
{
	pr_debug("bcmunimac_shutdown: %s\n", pdev->name);
	bcmunimac_remove(pdev);
}

MODULE_LICENSE("GPL v2");
