 /****************************************************************************
 *
 * Copyright (c) 2015-2018 Broadcom. All rights reserved
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Jayesh Patel <jayeshp@broadcom.com>
 ****************************************************************************/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/inetdevice.h>
#include "flowmgr.h"
#include "flowmgr_fap_ops.h"

/* Defaults to HW first */
static u32 fap_type = NFAP;

static int param_set_fap(const char *val, const struct kernel_param *kp)
{
	int ret = 0;
	u32 new_fap;

	ret = kstrtouint(val, 10, &new_fap);
	if (ret)
		return ret;
	if (new_fap > 2) {
		pr_err("Out of recommended range %u, between %u-%u\n",
		       new_fap,
		       0, 2);
		pr_err("Select FAP: 0->No FAP, 1->SW FAP, 2->GFAP 3->RFAP\n");
		ret = -EINVAL;
	}
	new_fap = clamp(new_fap, (u32)0, (u32)2);
	switch (new_fap) {
	case NFAP:
		break;
	case SFAP:
		break;
	case GFAP:
		break;
	case RFAP:
		break;

	default:
		pr_info("FLOWMGR: Switching to FAP type %d not supported\n",
			new_fap);
		new_fap = fap_type;
		break;
	}
	if (new_fap != fap_type) {
		pr_info("FLOWMGR: FAP changed from %d -> %d\n",
			fap_type, new_fap);
		fap_type = new_fap;
	}
	return ret;
}

static int param_get_fap(char *buffer, const struct kernel_param *kp)
{
	u32 type;

	type = *(u32 *)kp->arg;
	/* Buffer can be max 4k */
	return snprintf(buffer, 32, "%u:%s", type, fap()->name);
}

static const struct kernel_param_ops param_ops_fap = {
	.set = param_set_fap,
	.get = param_get_fap,
};

module_param_cb(fap, &param_ops_fap, &fap_type, 0644);
MODULE_PARM_DESC(fap, "Select FAP: 0->NFAP, 1->SFAP, 2>HW FAP");

static int fap_enable(bool enable)
{
	return 0;
}

static int fap_config(char *pri_lan_dev_name, char *wan_dev_name,
		      char *sec_lan_dev_name)
{
	return 0;
}

static int fap_config_dslite(void *params, int oindex)
{
	return 0;
}

static int fap_config_gre(void *params, int oindex)
{
	return 0;
}

static int fap_config_mapt(void *params)
{
	return 0;
}

static int fap_flow_add(void *ptr, int *flow_id)
{
	return 0;
}

static int fap_flow_remove(int flow_id)
{
	return 0;
}

static int fap_flow_remove_mcast(int flow_id, __be32 *group_id,  int oindex)
{
	return 0;
}

static int fap_flow_get_counter(int flow_id, u32 *packets, u32 *bytes,
				bool reset)
{
	return 0;
}

static int fap_flow_get_mcast_counter(int flow_id, u32 *packets, u32 *bytes,
				bool reset)
{
	return 0;
}

static struct fap_ops nfapops = {
	.type = NFAP,
	.name = "No FAP",
	.features = 0,
	.enable = fap_enable,
	.config = fap_config,
	.config_dslite = fap_config_dslite,
	.config_gre = fap_config_gre,
	.config_mapt = fap_config_mapt,
	.flow_add = fap_flow_add,
	.flow_remove = fap_flow_remove,
	.flow_remove_mcast = fap_flow_remove_mcast,
	.flow_get_counter = fap_flow_get_counter,
	.flow_get_mcast_counter = fap_flow_get_mcast_counter
};

struct fap_ops *nfap(void)
{
	return &nfapops;
}
struct fap_ops *fapops;

struct fap_ops *fap(void)
{
	if (!fapops || (fap_type == NFAP))
		fapops = nfap();
	return fapops;
}

void flowmgr_register_fap(struct fap_ops *ops)
{
	if (ops)
		fapops = ops;
	if (flowmgr.flow_usage)	{
		kfree(flowmgr.flow_usage);
		flowmgr.flow_usage = NULL;
	}
	if (fapops && fapops->flow_max)
		flowmgr.max_fap_flows = fapops->flow_max();
	if (fapops) {
		fap_type = fapops->type;
		flowmgr.hw_features = fapops->features;
	}
	flowmgr.features = flowmgr.sw_features & flowmgr.hw_features;
	pr_info("FLOWMGR Register fap: %d\n", fap_type);
}
EXPORT_SYMBOL(flowmgr_register_fap);

void flowmgr_unregister_fap(struct fap_ops *ops)
{
	if (ops == fapops)
		fapops = nfap();
	if (flowmgr.flow_usage)
		kfree(flowmgr.flow_usage);
	fap_type = fapops->type;
}
EXPORT_SYMBOL(flowmgr_unregister_fap);
