/*
 * Broadcom oops capture and display (boops)
 *
 * Copyright (C) 2017 Broadcom
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#include <linux/kernel.h>
#include <linux/mtd/ubi.h>
#include <linux/vmalloc.h>



#include <linux/err.h>
#include <linux/module.h>
#include <linux/kmsg_dump.h>
#include <linux/time.h>
#include <linux/err.h>
#include <linux/io.h>
#include <linux/ioport.h>
#include <linux/platform_device.h>
#include <linux/slab.h>
#include <linux/mtd/ubi.h>
#include <linux/vmalloc.h>
#include <linux/version.h>
#include <linux/file.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/scatterlist.h>
#include <linux/moduleparam.h>
#include <linux/errno.h>
#include <linux/hdreg.h>
#include <linux/kdev_t.h>
#include <linux/blkdev.h>
#include <linux/mutex.h>
#include <linux/string_helpers.h>
#include <linux/delay.h>
#include <linux/capability.h>
#include <linux/compat.h>
#include <linux/pm_runtime.h>
#include <linux/uaccess.h>
#include <linux/proc_fs.h>
#include <linux/random.h>


#include "boops_k_storage.h"
#include "boops_k_common.h"


#define BOOPS_UBI_OPEN_MAX_ATTEMPTS 5
#define BOOPS_UBI_OPEN_DELAY 5000

struct boops_storage_ubi_t {
    struct boops_storage_t generic;
    struct ubi_device_info device_info;
    struct ubi_volume_info volume_info;
    struct ubi_volume_desc *fd;
    unsigned long device;
    unsigned long volume;

    bool write_done;
};

static
int boops_storage_ubi_write_blocks(const struct boops_storage_t *storage, uint8_t *buf, uint32_t block_count) {
    struct boops_storage_ubi_t *ubi = container_of(storage, struct boops_storage_ubi_t, generic);
    uint32_t block_written = 0;
    int err = 0;

    if (!ubi->fd) {
        boops_err("The UBI volume is closed. Are you trying to write a second time? This won't work for UBI\n");
        return -1;
    }

    boops_info("Starting UBI transfer of %u blocks\n", block_count);
    boops_ubi_set_panic_writes(ubi->device, true);

    err = ubi_leb_write(ubi->fd, 0, buf, 0, block_count*storage->block.bytes);
    if (err) {
        boops_err("Error writing block %u (%d)\n", block_written, err);
    }
    boops_info("Completed UBI transfer\n");

    ubi_close_volume(ubi->fd);
    ubi->fd=NULL;

    return 0;
}

static
void boops_storage_ubi_destroy(struct boops_storage_t *storage) {
    struct boops_storage_ubi_t *ubi = container_of(storage, struct boops_storage_ubi_t, generic);
    if (ubi->fd)
        ubi_close_volume(ubi->fd);

    vfree(ubi);
}

struct boops_storage_t* boops_storage_ubi_create(const uint32_t ubi_device, const uint32_t ubi_volume) {
    struct boops_storage_ubi_t *ubi = NULL;
    struct boops_storage_t *storage = NULL;
    int err = 0;
    uint32_t open_attempt = 10;

    ubi = vmalloc(sizeof(*ubi));
    if (!ubi) {
        boops_err("Failed to allocate storage\n");
        return NULL;
    }
    memset(ubi, 0, sizeof(*ubi));
    storage = &ubi->generic;
    ubi->device = ubi_device;
    ubi->volume = ubi_volume;

    /*Get the ubi device info and its min IO size */
    err = ubi_get_device_info(ubi->device, &ubi->device_info);
    if (err) {
        boops_err("Failed to get device info (%d)\n", err);
        vfree(ubi);
        return NULL;
    }

    while( IS_ERR((ubi->fd = ubi_open_volume(ubi->device, ubi->volume, UBI_READWRITE))) ) {
        switch(PTR_ERR(ubi->fd)) {
        case -EBUSY:
            if (--open_attempt <= 0) {
                boops_err("Open volume still busy after maximum retries. Aborting\n");
                vfree(ubi);
                return NULL;
            }
            boops_err("Partiton is busy. Will retry %u more times in %ums...\n", open_attempt, BOOPS_UBI_OPEN_DELAY);
            msleep(BOOPS_UBI_OPEN_DELAY);
            break;
        default:
            boops_err("Open volume failed with unrecoverable error %ld. Aborting\n", PTR_ERR(ubi->fd));
            vfree(ubi);
            return NULL;
        }
    }

    ubi_get_volume_info(ubi->fd, &ubi->volume_info);

    err = ubi_leb_erase(ubi->fd, 0);
    if (err) {
        boops_err("Failed to erase the LEB (%d). Cannot dump OOPS!\n", err);
        ubi_close_volume(ubi->fd);
        return NULL;
    }

    storage->write_blocks = boops_storage_ubi_write_blocks;
    storage->destroy = boops_storage_ubi_destroy;
    storage->block.bytes = ubi->device_info.min_io_size;
    storage->block.count = (ubi->volume_info.usable_leb_size * ubi->volume_info.size) / storage->block.bytes;
    return &ubi->generic;
}
