 /****************************************************************************
 *
 * Copyright (c) 2015-2018 Broadcom. All rights reserved
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Jayesh Patel <jayeshp@broadcom.com>
 ****************************************************************************/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/kthread.h>
#include <linux/timer.h>
#include <linux/etherdevice.h>
#include <linux/socket.h>
#include <net/neighbour.h>
#include <net/netevent.h>
#include <linux/if_arp.h>
#include <net/ipv6.h>
#include <net/tcp.h>
#include <net/udp.h>
#include <net/sock.h>
#include <net/inet_sock.h>
#include <linux/netfilter.h>
#include <linux/netfilter_bridge.h>
#include <net/netfilter/nf_conntrack_core.h>
#include <net/netfilter/nf_conntrack_l4proto.h>
#include <net/netfilter/nf_conntrack_l4proto.h>
#include <net/netfilter/nf_conntrack_ecache.h>
#include <net/netfilter/nf_conntrack_acct.h>
#include <net/netfilter/nf_conntrack_offload.h>
#include <net/netfilter/nf_conntrack_helper.h>
#include <net/netfilter/nf_conntrack_timeout.h>
#include <linux/rculist_nulls.h>
#include <linux/spinlock.h>
#include "flowmgr.h"
#include "flowmgr_fap.h"
#include "flowmgr_fap_ops.h"
#include "flowmgr_map.h"
#include "flowmgr_tunnel.h"
#include "dqnet_priv.h"
#include "dqnet_brcmtag.h"
#include <proc_cmd.h>
#include <net/bonding.h>

#include <net/ip6_tunnel.h>
#include <net/ip_tunnels.h>
#include <net/dsfield.h>

#define VERSION     "1.0"
#define VER_STR     "v" VERSION

#ifdef CONFIG_BCM_FLOWMGR_TUNNEL
#define FLOW_F_ALL		(FLOW_F_L4_ALL | FLOW_F_TUNNEL_ALL | \
				 FLOW_F_MACBR | FLOW_F_MCAST)
#else
#define FLOW_F_ALL		(FLOW_F_L4_ALL | \
				 FLOW_F_MACBR | FLOW_F_MCAST)
#endif

#ifdef CONFIG_BCM_FLOWMGR_NETHOOK
#define  FLOWMGR_USE_BCM_NETHOOKS	1
#else
#define  FLOWMGR_USE_BCM_NETHOOKS	0
#endif

#define OFFLOAD_MARK 0x00000000 /* 0xWXYZABCD: WXYZ-Flow ID1, ABCD-Flow ID2 */

#define MASTER_TIMEOUT_LOGIC 1

struct flowmgr flowmgr;
static char *def_net_interface_list[] = {
	"eth0",
	"eth1",
	"eth2",
	"eth3",
	"eth4",
	"eth5",
	"eth7",
	"stb0",
	"moca0",
	"cm0"
};
static char *net_interface_list[MAX_INF];
static unsigned int interfaces_c;

#define MAX_IGNORE_PORT 32
static __be16 ignore_port_list[MAX_IGNORE_PORT] = {
	htons(21),                    /* FTP     */
	htons(53),                    /* DNS     */
	htons(67),  htons(68),        /* DHCP    */
	htons(69),                    /* TFTP    */
	htons(123),                   /* NTP     */
	htons(137), htons(138),       /* NetBIOS */
	htons(161), htons(162),       /* SNMP    */
	htons(389),                   /* LDAP    */
	htons(546), htons(547),       /* DHCPv6  */
	htons(554),                   /* RTSP    */
	htons(1720),                  /* H.323   */
	htons(1863),                  /* MSN     */
	htons(5060),                  /* SIP     */
};

typedef int (*flowmgr_ovs_hook_fn)(struct sk_buff *skb, struct net_device *in, struct net_device *out);

static inline int is_brnf_call_iptables(void)
{
	struct file *f;
	unsigned char data = '0';
	int ret;

	f = procsys_file_open("/proc/sys/net/bridge/bridge-nf-call-iptables",
			      O_RDONLY, 0);
	if (!f)
		return -ENOENT;
	ret = procsys_file_read(f, 0, &data, 1);
	procsys_file_close(f);
	if (ret > 0)
		return data == '0' ? 0 : 1;
	return ret;
}

static inline int brnf_call_iptables_enable(int enable)
{
	unsigned char *path = "/proc/sys/net/bridge/bridge-nf-call-iptables";
	unsigned char *data = enable ? "1\n" : "0\n";
	return procsys_file_write_string(path, data);
}

static inline int is_brnf_call_ip6tables(void)
{
	struct file *f;
	unsigned char data = '0';
	int ret;

	f = procsys_file_open("/proc/sys/net/bridge/bridge-nf-call-ip6tables",
			      O_RDONLY, 0);
	if (!f)
		return -ENOENT;
	ret = procsys_file_read(f, 0, &data, 1);
	procsys_file_close(f);
	if (ret > 0)
		return data == '0' ? 0 : 1;

	return ret;
}

static inline int brnf_call_ip6tables_enable(int enable)
{
	unsigned char *path = "/proc/sys/net/bridge/bridge-nf-call-ip6tables";
	unsigned char *data = enable ? "1\n" : "0\n";
	return procsys_file_write_string(path, data);
}

void flowmgr_enable_br_conntrack(int enable)
{
	flowmgr.enable_br_conntrack_call = enable;
	brnf_call_iptables_enable(!enable);
	brnf_call_ip6tables_enable(!enable);
}

#ifdef CONFIG_SYSCTL

static
int enable_call(struct ctl_table *ctl, int write,
		void __user *buffer, size_t *lenp, loff_t *ppos)
{
	int *valp = ctl->data;
	int val = *valp;
	loff_t pos = *ppos;
	struct ctl_table lctl;
	int ret;

	lctl = *ctl;
	lctl.data = &val;

	ret = proc_dointvec(&lctl, write, buffer, lenp, ppos);

	if (write)
		*valp = val;
	if (ret)
		*ppos = pos;
	return ret;
}

static
int enable_br_conntrack_call(struct ctl_table *ctl, int write,
			     void __user *buffer, size_t *lenp, loff_t *ppos)
{
	int *valp = ctl->data;
	int val = *valp;
	loff_t pos = *ppos;
	struct ctl_table lctl;
	int ret;

	lctl = *ctl;
	lctl.data = &val;

	ret = proc_dointvec(&lctl, write, buffer, lenp, ppos);

	if (write)
		flowmgr_enable_br_conntrack(val);
	if (ret)
		*ppos = pos;
	return ret;
}

static
int tcp_pkt_threshold_call(struct ctl_table *ctl, int write,
		void __user *buffer, size_t *lenp, loff_t *ppos)
{
	int *valp = ctl->data;
	int val = *valp;
	loff_t pos = *ppos;
	struct ctl_table lctl;
	int ret;

	lctl = *ctl;
	lctl.data = &val;

	ret = proc_dointvec(&lctl, write, buffer, lenp, ppos);

	if (write && (!is_flowmgr_manual_enable() || val))
		*valp = val;
	if (ret)
		*ppos = pos;
	return ret;
}

static
int use_3tuple_br_flow_call(struct ctl_table *ctl, int write,
			    void __user *buffer, size_t *lenp, loff_t *ppos)
{
	int *valp = ctl->data;
	int val = *valp;
	loff_t pos = *ppos;
	struct ctl_table lctl;
	int ret;

	lctl = *ctl;
	lctl.data = &val;

	ret = proc_dointvec(&lctl, write, buffer, lenp, ppos);

	if (write) {
		if (flowmgr.hw_features & FLOW_F_MACBR)
			*valp = val;
	}
	if (write)
		ret = flowmgr_map_disable(val);
	if (ret)
		*ppos = pos;
	return ret;
}

static
int disable_mapt_call(struct ctl_table *ctl, int write,
		      void __user *buffer, size_t *lenp, loff_t *ppos)
{
	int *valp = ctl->data;
	int val = *valp;
	loff_t pos = *ppos;
	struct ctl_table lctl;
	int ret;

	lctl = *ctl;
	lctl.data = &val;

	ret = proc_dointvec(&lctl, write, buffer, lenp, ppos);

	if (write)
		ret = flowmgr_map_disable(val);
	if (ret)
		*ppos = pos;
	return ret;
}

static
int promote_mode_call(struct ctl_table *ctl, int write,
		void __user *buffer, size_t *lenp, loff_t *ppos)
{
	int *valp = ctl->data;
	int val = *valp;
	loff_t pos = *ppos;
	struct ctl_table lctl;
	int ret;

	lctl = *ctl;
	lctl.data = &val;

	ret = proc_dointvec(&lctl, write, buffer, lenp, ppos);

	if (write) {
		flowmgr_manual_enable(val != AUTO_PROMOTE);
		*valp = val;
	}
	if (ret)
		*ppos = pos;
	return ret;
}

static char *promote_dir_ctl_str(int ctl)
{
	char *str;
	switch (ctl) {
	case 1:
		str = "LAN2LAN";
		break;
	case 2:
		str = "WAN2LAN";
		break;
	case 3:
		str = "LAN2LAN WAN2LAN";
		break;
	case 4:
		str = "LAN2WAN";
		break;
	case 5:
		str = "LAN2LAN LAN2WAN";
		break;
	case 6:
		str = "WAN2LAN LAN2WAN";
		break;
	case 7:
		str = "LAN2LAN WAN2LAN LAN2WAN";
		break;
	default:
		str = "Invalid";
		break;
	}
	return str;
}

static
int promote_dir_ctl_call(struct ctl_table *ctl, int write,
		void __user *buffer, size_t *lenp, loff_t *ppos)
{
	int *valp = ctl->data;
	int val = *valp;
	loff_t pos = *ppos;
	struct ctl_table lctl;
	int ret;

	lctl = *ctl;
	lctl.data = &val;

	ret = proc_dointvec(&lctl, write, buffer, lenp, ppos);

	if (write) {
		if ((val <= 0) || (val > 7)) {
			pr_alert("FLOWMGR: Changing promote dir ctrl <= 0 or ctrl > 7 not allowed\n");
			pr_alert("FLOWMGR: Bit0: LAN2LAN, Bit1: WAN2LAN, Bit2: LAN2WAN\n");
			goto done;
		}
		pr_alert("FLOWMGR: Changing promote dir ctrl from (%s) to (%s)\n",
			 promote_dir_ctl_str(flowmgr.promote_dir_ctl),
			 promote_dir_ctl_str(val));
		*valp = val;
	}
done:
	if (ret)
		*ppos = pos;
	return ret;
}

static struct ctl_table flowmgr_sysctl_table[] = {
	{
		.procname	= "enable",
		.data		= &flowmgr.enable,
		.maxlen		= sizeof(int),
		.mode		= 0644,
		.proc_handler	= enable_call,
	},
	{
		.procname	= "udp_pkt_threshold",
		.data		= &flowmgr.udp_pkt_threshold,
		.maxlen		= sizeof(int),
		.mode		= 0644,
		.proc_handler	= proc_dointvec,
	},
	{
		.procname	= "tcp_pkt_threshold",
		.data		= &flowmgr.tcp_pkt_threshold,
		.maxlen		= sizeof(int),
		.mode		= 0644,
		.proc_handler	= tcp_pkt_threshold_call,
	},
	{
		.procname	= "use_3tuple_br_flow",
		.data		= &flowmgr.use_3tuple_br_flow,
		.maxlen		= sizeof(int),
		.mode		= 0644,
		.proc_handler	= use_3tuple_br_flow_call,
	},
	{
		.procname	= "nosock_drop",
		.data		= &flowmgr.nosock_drop,
		.maxlen		= sizeof(int),
		.mode		= 0644,
		.proc_handler	= proc_dointvec,
	},
	{
		.procname	= "fast_death_timeout",
		.data		= &flowmgr.fast_death_timeout,
		.maxlen		= sizeof(int),
		.mode		= 0644,
		.proc_handler	= proc_dointvec,
	},
	{
		.procname	= "use_tcp_3way_handshake",
		.data		= &flowmgr.use_tcp_3way_handshake,
		.maxlen		= sizeof(int),
		.mode		= 0644,
		.proc_handler	= proc_dointvec,
	},
	{
		.procname	= "disable_gre_accel",
		.data		= &flowmgr.disable_gre_accel,
		.maxlen		= sizeof(int),
		.mode		= 0644,
		.proc_handler	= proc_dointvec,
	},
	{
		.procname	= "disable_mapt_accel",
		.data		= &flowmgr.disable_mapt_accel,
		.maxlen		= sizeof(int),
		.mode		= 0644,
		.proc_handler	= disable_mapt_call,
	},
	{
		.procname	= "enable_br_conntrack_call",
		.data		= &flowmgr.enable_br_conntrack_call,
		.maxlen		= sizeof(int),
		.mode		= 0644,
		.proc_handler	= enable_br_conntrack_call,
	},
	{
		.procname	= "enable_expected_flow",
		.data		= &flowmgr.enable_expected_flow,
		.maxlen		= sizeof(int),
		.mode		= 0644,
		.proc_handler	= proc_dointvec,
	},
	{
		.procname	= "expected_flow_timeout",
		.data		= &flowmgr.expected_flow_timeout,
		.maxlen		= sizeof(int),
		.mode		= 0644,
		.proc_handler	= proc_dointvec,
	},
	{
		.procname	= "check_status_timeout",
		.data		= &flowmgr.check_status_timeout,
		.maxlen		= sizeof(int),
		.mode		= 0644,
		.proc_handler	= proc_dointvec,
	},
	{
		.procname	= "promote_mode",
		.data		= &flowmgr.promote_mode,
		.maxlen 	= sizeof(int),
		.mode		= 0644,
		.proc_handler	= promote_mode_call,
	},
	{
		.procname	= "auto_flush_mac_mode",
		.data		= &flowmgr.auto_flush_mac_mode,
		.maxlen		= sizeof(int),
		.mode		= 0644,
		.proc_handler	= proc_dointvec,
	},
	{
		.procname	= "auto_flush_inf_mode",
		.data		= &flowmgr.auto_flush_inf_mode,
		.maxlen		= sizeof(int),
		.mode		= 0644,
		.proc_handler	= proc_dointvec,
	},
	{
		.procname	= "promote_dir_ctl",
		.data		= &flowmgr.promote_dir_ctl,
		.maxlen		= sizeof(int),
		.mode		= 0644,
		.proc_handler	= promote_dir_ctl_call,
	},
	{
		.procname	= "enable_tu_port_track",
		.data		= &flowmgr.enable_tu_port_track,
		.maxlen		= sizeof(int),
		.mode		= 0644,
		.proc_handler	= proc_dointvec,
	},
	{}
};
#else
module_param_named(udp_pkt_threshold, flowmgr.udp_pkt_threshold, int, 0644);
MODULE_PARM_DESC(udp_pkt_threshold, "UDP Packet Threshold for Promotion");

module_param_named(enable, flowmgr.enable, int, 0644);
MODULE_PARM_DESC(enable, "Master Enable (0 or 1)");
#endif /* CONFIG_SYSCTL */

/*static struct net_device *wandev;*/

void flowmgr_set_wandev(struct net_device *dev)
{
	if (dev) {
		struct net_device *active;
		struct dqnet_netdev *ndev;
		pr_alert("FLOWMGR: Changing wandev from %s to %s\n", flowmgr.wandev->name, dev->name);
		flowmgr.wandev = dev;
		active = flowmgr_get_wandev();
		ndev = netdev_priv(active);
		/* DOCSIS WAN Link type is RPC */
		if (ndev->link_type != DQNET_LINK_TYPE_RPC)
			flowmgr.wantype = 1;
	}
}

int flowmgr_set_wandev_by_name(char *name)
{
	struct net_device *dev;
	dev = __dev_get_by_name(&init_net, name);
	if (!dev)
		return -1;
	if (netif_is_bond_master(dev)) {
		flowmgr_set_wandev(dev);
		return 0;
	} else if (BCM_NETDEVICE_GROUP_TYPE(dev->group) ==
		   BCM_NETDEVICE_GROUP_WAN) {
		if (netif_is_bond_slave(dev)) {
			rtnl_lock();
			flowmgr_set_wandev(
			   netdev_master_upper_dev_get(dev));
			rtnl_unlock();
		} else
			flowmgr_set_wandev(dev);
		return 0;
	}
	return -1;
}

struct net_device *flowmgr_get_wandev(void)
{
	if (!flowmgr.wandev)
		return NULL;

	if (netif_is_bond_master(flowmgr.wandev)) {
		struct bonding *bond;
		bond = netdev_priv(flowmgr.wandev);
		if (!bond)
			return flowmgr.wandev;
		return bond->curr_active_slave ? bond->curr_active_slave->dev
			: bond->primary_slave->dev;
	}
	return flowmgr.wandev;
}

int flowmgr_get_vlan_id(struct net_device *dev)
{
	if (dev == NULL)
		return 0;
	/* return the VLAN ID for VLAN devices (usually a bridge object) */
	if (is_vlan_dev((struct net_device *)dev))
		return vlan_dev_vlan_id(dev);
	else
		return 0;
}

static inline
bool is_netdevice_registered(char *name)
{
	int i;
	for (i = 0; i < MAX_INF; i++) {
		if (net_interface_list[i] &&
		    strncmp(net_interface_list[i], name, IFNAMSIZ) == 0)
			return true;
	}
	return false;
}

static inline
bool is_ignore_port(__be16 port)
{
	int i;
	for (i = 0; i < MAX_IGNORE_PORT; i++) {
		if (ignore_port_list[i] && (ignore_port_list[i] == port))
			return true;
	}
	return false;
}

static inline
bool flowmgr_is_dev_veth(struct net_device *dev)
{
	if (!dev)
		return false;
	if (dev->rtnl_link_ops && dev->rtnl_link_ops->kind) {
		if (!strcasecmp(dev->rtnl_link_ops->kind, "veth"))
			return true;
	}
	return false;
}

bool flowmgr_is_dev_ip_tunnel(struct net_device *dev)
{
	if (!dev)
		return false;
	if (dev->type == ARPHRD_TUNNEL6)
		return true;
	if (dev->type == ARPHRD_TUNNEL)
		return true;
	if (dev->type == ARPHRD_IP6GRE)
		return true;
	if (dev->type == ARPHRD_IPGRE)
		return true;
	return false;
}

bool flowmgr_is_dev_gretap_tunnel(struct net_device *dev)
{
	if (dev->rtnl_link_ops && dev->rtnl_link_ops->kind) {
		if (!strcasecmp(dev->rtnl_link_ops->kind, "gretap"))
			return true;
	}
	return false;
}

bool flowmgr_is_dev_ip6gretap_tunnel(struct net_device *dev)
{
	if (dev->rtnl_link_ops && dev->rtnl_link_ops->kind) {
		if (!strcasecmp(dev->rtnl_link_ops->kind, "ip6gretap"))
			return true;
	}
	return false;
}

bool flowmgr_is_dev_tunnel(struct net_device *dev)
{
	if (!dev)
		return false;
	if (flowmgr_is_dev_ip_tunnel(dev))
		return true;
	else if (flowmgr_is_dev_gretap_tunnel(dev))
		return true;
	else if (flowmgr_is_dev_ip6gretap_tunnel(dev))
		return true;
	return false;
}

bool flowmgr_is_tunnel_allowed(struct net_device *dev, int type)
{
	if (!dev)
		return false;

	/* Default Tunnel Type is WAN */
	if (type == 0)
		type = BCM_NETDEVICE_GROUP_WAN;

	if (flowmgr_is_dev_ip_tunnel(dev)) {
		if (flowmgr_is_feature_enabled(FLOW_F_DSLITE))
			return true;
		pr_info("FLOWMGR: Tunnel %s type %d not supported\n",
			dev->name, type);
	} else if (flowmgr_is_dev_gretap_tunnel(dev)) {
		if (type == BCM_NETDEVICE_GROUP_WAN) {
			if (!flowmgr.wantype &&
			    flowmgr_is_feature_enabled(FLOW_F_GRETAP4WAN))
				return true;
			if (flowmgr.wantype &&
			    flowmgr_is_feature_enabled(FLOW_F_GRETAP4ETHWAN))
				return true;
		} else if ((type == BCM_NETDEVICE_GROUP_LAN) &&
			 flowmgr_is_feature_enabled(FLOW_F_GRETAP4LAN)) {
			return true;
		}
		pr_info("FLOWMGR: Tunnelv4 %s type %d not supported\n",
			dev->name, type);
	} else if (flowmgr_is_dev_ip6gretap_tunnel(dev)) {
		if (type == BCM_NETDEVICE_GROUP_WAN) {
			if (!flowmgr.wantype &&
			    flowmgr_is_feature_enabled(FLOW_F_GRETAP6WAN))
				return true;
			if (flowmgr.wantype &&
			    flowmgr_is_feature_enabled(FLOW_F_GRETAP6ETHWAN))
				return true;
		} else if ((type == BCM_NETDEVICE_GROUP_LAN) &&
			 flowmgr_is_feature_enabled(FLOW_F_GRETAP6LAN)) {
			return true;
		}
		pr_info("FLOWMGR: Tunnelv6 %s type %d not supported\n",
			dev->name, type);
	}
	return false;
}

bool flowmgr_is_flow_allowed(__be16 l3, int l4)
{
	if (l4 == IPPROTO_TCP) {
		if ((l3 == ntohs(ETH_P_IP)) &&
		    flowmgr_is_feature_enabled(FLOW_F_TCP4))
			return true;
		if ((l3 == ntohs(ETH_P_IPV6)) &&
		    flowmgr_is_feature_enabled(FLOW_F_TCP6))
			return true;
	} else if (l4 == IPPROTO_UDP) {
		if ((l3 == ntohs(ETH_P_IP)) &&
		    flowmgr_is_feature_enabled(FLOW_F_UDP4))
			return true;
		if ((l3 == ntohs(ETH_P_IPV6)) &&
		    flowmgr_is_feature_enabled(FLOW_F_UDP6))
			return true;
	} else if (l4 == IPPROTO_ESP) {
		if ((l3 == ntohs(ETH_P_IP)) &&
		    flowmgr_is_feature_enabled(FLOW_F_ESP4))
			return true;
		if ((l3 == ntohs(ETH_P_IPV6)) &&
		    flowmgr_is_feature_enabled(FLOW_F_ESP6))
			return true;
	} else if (l4 == IPPROTO_AH) {
		if ((l3 == ntohs(ETH_P_IP)) &&
		    flowmgr_is_feature_enabled(FLOW_F_AH4))
			return true;
		if ((l3 == ntohs(ETH_P_IPV6)) &&
		    flowmgr_is_feature_enabled(FLOW_F_AH6))
			return true;
	}
	return false;
}

bool flowmgr_is_db_stats_update_allowed(struct net_device *in,
					struct net_device *out)
{
	if (flowmgr.inf_db_info)
		return true;
	if (out && out->group && (BCM_NETDEVICE_GROUP_TYPE(out->group) ==
		BCM_NETDEVICE_GROUP_WAN))
		return true;
	if (in && in->group && (BCM_NETDEVICE_GROUP_TYPE(in->group) ==
		BCM_NETDEVICE_GROUP_WAN))
		return true;
	return false;
}

int flowmgr_ignore_port_add(u16 port)
{
	int i;
	if (is_ignore_port(htons(port)))
		return -1;
	for (i = 0; i < MAX_IGNORE_PORT; i++) {
		if (ignore_port_list[i] == 0) {
			ignore_port_list[i] = htons(port);
			return 0;
		}
	}
	return -1;
}

int flowmgr_ignore_port_del(u16 port)
{
	int i;
	for (i = 0; i < MAX_IGNORE_PORT; i++) {
		if (ignore_port_list[i] == htons(port))
			ignore_port_list[i] = 0;
	}
	return -1;
}

void flowmgr_ignore_port_clear(void)
{
	int i;
	for (i = 0; i < MAX_IGNORE_PORT; i++)
		ignore_port_list[i] = 0;
}

void flowmgr_ignore_port_show(struct seq_file *s)
{
	int i;
	pr_seq(s, "Ignore Port List:\n");
	for (i = 0; i < MAX_IGNORE_PORT; i += 2) {
		if (ignore_port_list[i] && ignore_port_list[i+1])
			pr_seq(s, " %-5d %-5d\n", ntohs(ignore_port_list[i]),
				ntohs(ignore_port_list[i+1]));
		else if (ignore_port_list[i])
			pr_seq(s, " %-5d\n", ntohs(ignore_port_list[i]));
		else if (ignore_port_list[i+1])
			pr_seq(s, " %-5d\n", ntohs(ignore_port_list[i+1]));
	}
	pr_seq(s, "----------------------\n");
}

static inline
struct nf_bridge_info *get_nf_bridge(struct sk_buff *skb)
{
	enum ip_conntrack_info ctinfo;
	struct nf_conn *ct = nf_ct_get(skb, &ctinfo);
	struct nf_bridge_info  *nf_bridge;
	struct nf_conn_offload *ct_offload;
	struct offload_info *ct_offload_info;

	nf_bridge = nf_bridge_info_get(skb);
	if ((!nf_bridge || flowmgr.enable_br_conntrack_call) && ct) {
		ct_offload = nf_conn_offload_find(ct);
		if (!ct_offload)
			return nf_bridge;
		ct_offload_info = &ct_offload->info[CTINFO2DIR(ctinfo)];
		nf_bridge = ct_offload_info->nf_bridge;
	}
	return nf_bridge;
}

static inline
struct net_device *in_device(struct sk_buff *skb)
{
	struct net_device *in = NULL;
	struct nf_bridge_info  *nf_bridge;

	if (skb->dev_in)
		return skb->dev_in;

	nf_bridge = get_nf_bridge(skb);
	if (nf_bridge)
		in = nf_bridge->physindev;
	else
		in = __dev_get_by_index(&init_net, skb->skb_iif);

	if (in && netif_is_bond_master(in)) {
		struct bonding *bond;
		bond = netdev_priv(in);
		if (bond->params.mode == BOND_MODE_ACTIVEBACKUP)
			in = bond->curr_active_slave->dev;
	}

	return in;
}

static inline
void update_vlan_info(struct sk_buff *skb, struct offload_info *ct_offload_info)
{
	/* skb->protocol: Is initialized when packet is rcvd by device driver
	   and will be modified after vlan tag is stripped.
	   skb->vlan_proto: Is initialized when packet has vlan tag */
	if ((skb->protocol != htons(ETH_P_8021Q)) &&
	    (skb->vlan_proto == htons(ETH_P_8021Q))) {
		/* VLAN tag is stripped from the packet */
		ct_offload_info->vlan_untag = 1;
	} else if ((skb->protocol == htons(ETH_P_8021Q)) &&
		   (skb->vlan_proto == htons(ETH_P_8021Q))) {
		/* VLAN tag is added to the packet */
		struct vlan_ethhdr *veth;
		u16 vlan_tci;
		veth = vlan_eth_hdr(skb);
		vlan_tci = htons(veth->h_vlan_TCI);
		if (ct_offload_info->vlan_id &&
		    (ct_offload_info->vlan_id != vlan_tci)) {
			pr_err("update_vlan_info: vlan mismatch %d (from check_for_vlan) != %d\n",
			       ct_offload_info->vlan_id, vlan_tci);
		}
		ct_offload_info->vlan_id = vlan_tci;
	}
	/* We don't detect the case where VLAN tag is stripped and
	   new VLAN tag is added to the packet */
}

static int new_expected(struct sk_buff *skb,
			struct net_device *in,
			struct net_device *out,
			struct nf_conn *ct,
			struct nf_conn_offload *ct_offload,
			int protonum)
{
	struct nf_conntrack_tuple *tuple;

	if (!flowmgr.enable_expected_flow)
		return 0;

	if (BCM_NETDEVICE_GROUP_TYPE(out->group) !=
	    BCM_NETDEVICE_GROUP_WAN)
		return 0;

	if ((BCM_NETDEVICE_GROUP_CAT(out->group) !=
	     BCM_NETDEVICE_GROUP_CAT_PHY))
		return 0;

	if (ct_offload_repl.flow_type || ct_offload_orig.flow_type)
		return 0;

	if (ct_offload_repl.create_err || ct_offload_orig.create_err)
		return 0;

	if ((ct_offload_repl.packets_slow > 0) ||
	    (ct_offload_orig.packets_slow > 0))
		return 0;

	if (CTINFO2DIR(nfctinfo(skb)) == IP_CT_DIR_ORIGINAL) {
		tuple = nf_ct_tuple(ct, IP_CT_DIR_REPLY);
		ct_offload_repl.iif = out->ifindex;
		/* Enable debug per conntrack match */
		flowmgr_ctdebug_check(ct, ct_offload, IP_CT_DIR_REPLY);
		if (!flowmgr_expected_promote(flow_expected, protonum, ct,
					     tuple, IP_CT_DIR_REPLY,
					      &ct_offload_repl)) {
			if (flowmgr.expected_flow_timeout)
				nf_ct_refresh(ct, skb,
				      flowmgr.expected_flow_timeout * HZ);
		}
	} else {
		/* No need to created expected flow for closed or reset tcp connection */
		if (protonum == IPPROTO_TCP) {
			pr_debug("ct->proto.tcp.state %d flag 0x%x\n",
				 ct->proto.tcp.state,
				 skb->cb[0]);
			if (ct->proto.tcp.state >= TCP_CONNTRACK_CLOSE_WAIT)
				return 0;
			if (skb->cb[0] & 0x05) /* RST or FIN */
				return 0;
		}
		tuple = nf_ct_tuple(ct, IP_CT_DIR_ORIGINAL);
		ct_offload_orig.iif = out->ifindex;
		/* Enable debug per conntrack match */
		flowmgr_ctdebug_check(ct, ct_offload, IP_CT_DIR_ORIGINAL);
		if (!flowmgr_expected_promote(flow_expected, protonum, ct,
					      tuple, IP_CT_DIR_ORIGINAL,
					      &ct_offload_orig)) {
			if (flowmgr.expected_flow_timeout)
				nf_ct_refresh(ct, skb,
				      flowmgr.expected_flow_timeout * HZ);
		}
	}

	return 0;
}

static inline
int new_offload(struct sk_buff *skb,
		struct net_device *in,
		struct net_device *out,
		struct nf_conn *ct,
		struct nf_conn_offload *ct_offload,
		int protonum)
{
	struct nf_conntrack_tuple *tuple;
	struct flowmgr_db_entry *odb;
	struct flowmgr_db_entry *idb;

	new_expected(skb, in, out, ct, ct_offload, protonum);

	if (CTINFO2DIR(nfctinfo(skb)) == IP_CT_DIR_ORIGINAL) {
		ct_offload_orig.packets_slow++;
		if (ct_offload_orig.flow_id == 0xDEFE22ED)
			return 0;
		ct_offload_orig.dscp_new = ip_get_slow_dscp_bp(skb);
		if (ct_offload_orig.ctinfo != -1)
			return 1;
		if ((ct_offload_orig.flow_type>>16) == ft_ignore)
			return 0;
		tuple = nf_ct_tuple(ct, IP_CT_DIR_ORIGINAL);
		if (!tuple)
			return 0;
		if (is_ignore_port(tuple->dst.u.tcp.port)) {
			ct_offload_orig.flow_type = ft_ignore<<16;
			return 0;
		}
		if (is_ignore_port(tuple->src.u.tcp.port)) {
			ct_offload_orig.flow_type = ft_ignore<<16;
			return 0;
		}
		idb = flowmgr_db_find(ct_offload_orig.ehb.h_source);
		if (idb && !idb->accel) {
			ct_offload_orig.flow_type = ft_ignore<<16;
			return 0;
		}
		odb = flowmgr_db_find(eth_hdr(skb)->h_dest);
		if (odb && !odb->accel) {
			ct_offload_orig.flow_type = ft_ignore<<16;
			return 0;
		}
		ct_offload_orig.ctinfo = nfctinfo(skb);
		ct_offload_orig.idb = idb;
		memcpy(&(ct_offload_orig.eh), eth_hdr(skb), ETH_HLEN);
		ct_offload_orig.oif = out->ifindex;
		ct_offload_orig.iif = in->ifindex;
		ct_offload_orig.odb = odb;
		ct_offload_repl.idb = ct_offload_orig.odb;
		update_vlan_info(skb, &ct_offload_orig);
		/* Enable debug per conntrack match */
		flowmgr_ctdebug_check(ct, ct_offload, IP_CT_DIR_ORIGINAL);
		if (ct_offload_orig.debug) {
			pr_alert(
			   "FLOWMGR: New orig ct-mark %X in_dev(%s) out_dev(%s) vlan_proto %x skb proto %x\n",
			   ct->mark, in->name, out->name,
			   ntohs(skb->vlan_proto), ntohs(skb->protocol));
			nf_ct_tuple_dump(pr_alert, tuple, "FLOWMGR: New orig");
			nf_ct_offload_dump(pr_alert, &ct_offload_orig, "FLOWMGR: New orig");
		}
	} else {
		ct_offload_repl.packets_slow++;
		if (ct_offload_repl.flow_id == 0xDEFE22ED)
			return 0;
		ct_offload_repl.dscp_new = ip_get_slow_dscp_bp(skb);
		if (ct_offload_repl.ctinfo != -1)
			return 1;
		if ((ct_offload_repl.flow_type>>16) == ft_ignore)
			return 0;
		tuple = nf_ct_tuple(ct, IP_CT_DIR_REPLY);
		if (!tuple)
			return 0;
		if (is_ignore_port(tuple->dst.u.tcp.port)) {
			ct_offload_repl.flow_type = ft_ignore<<16;
			return 0;
		}
		if (is_ignore_port(tuple->src.u.tcp.port)) {
			ct_offload_repl.flow_type = ft_ignore<<16;
			return 0;
		}
		idb = flowmgr_db_find(ct_offload_repl.eh.h_source);
		if (idb && !idb->accel) {
			ct_offload_repl.flow_type = ft_ignore<<16;
			return 0;
		}
		odb = flowmgr_db_find(eth_hdr(skb)->h_dest);
		if (odb && !odb->accel) {
			ct_offload_repl.flow_type = ft_ignore<<16;
			return 0;
		}
		ct_offload_repl.ctinfo = nfctinfo(skb);
		ct_offload_repl.idb = idb;
		memcpy(&(ct_offload_repl.eh), eth_hdr(skb), ETH_HLEN);
		ct_offload_repl.oif = out->ifindex;
		ct_offload_repl.iif = in->ifindex;
		ct_offload_repl.odb = odb;
		ct_offload_orig.idb = ct_offload_repl.odb;
		update_vlan_info(skb, &ct_offload_repl);
		/* Enable debug per conntrack match */
		flowmgr_ctdebug_check(ct, ct_offload, IP_CT_DIR_REPLY);
		if (ct_offload_repl.debug) {
			pr_alert(
			   "FLOWMGR: New repl: ct-mark %X in_dev(%s) out_dev(%s) vlan_proto %x skb proto %x\n",
			   ct->mark, in->name, out->name,
			   ntohs(skb->vlan_proto), ntohs(skb->protocol));
			nf_ct_tuple_dump(pr_alert, tuple, "FLOWMGR: New repl");
			nf_ct_offload_dump(pr_alert, &ct_offload_repl, "FLOWMGR: New repl");
		}
	}
	/* Set peeked, so intermediate netfilter hooks does not analyze
	   this packet */
	skb->peeked = 1;
	return 1;
}

static inline
int promote_duplex(struct sk_buff *skb,
		struct net_device *in,
		struct net_device *out,
		struct nf_conn *ct,
		int prot)
{
	struct nf_conntrack_tuple *tuple_orig;
	struct nf_conntrack_tuple *tuple_repl;
	struct nf_conn_offload *ct_offload;
	struct offload_info *ct_offload_info;
	struct nf_bridge_info	*nf_bridge;
	int idev_type = BCM_NETDEVICE_GROUP_TYPE(in->group);
	int odev_type = BCM_NETDEVICE_GROUP_TYPE(out->group);
	ct_offload = nf_conn_offload_find(ct);
	tuple_orig = nf_ct_tuple(ct, IP_CT_DIR_ORIGINAL);
	tuple_repl = nf_ct_tuple(ct, IP_CT_DIR_REPLY);

	nf_bridge = get_nf_bridge(skb);
	ct_offload_info = &ct_offload->info[CTINFO2DIR(nfctinfo(skb))];
	if (ct_offload_info->map) {
#ifdef CONFIG_BCM_FLOWMGR_TUNNEL
		if (flowmgr_promote_mapt_duplex(prot, skb,
						ct, ct_offload_info) == 0) {
						ct_offload_orig.tstamp = jiffies;
						ct_offload_repl.tstamp = jiffies;
		}
#else
		pr_err("FLOWMGR: MAP-T offload set without support (duplex)\n");
		return 0;
#endif
	} else if (odev_type == BCM_NETDEVICE_GROUP_WAN) {
		/* Promote bidirectional flow to FAP */
		if (flowmgr_promote(flow_wanlan, prot,
				    skb->priority,
				    ct, nf_bridge,
				    tuple_repl, tuple_orig,
				    0, &ct_offload_repl) == 0)
			ct_offload_repl.tstamp = jiffies;
		if (flowmgr_promote(flow_lanwan, prot,
				    skb->priority,
				    ct, nf_bridge,
				    tuple_orig, tuple_repl,
				    1, &ct_offload_orig) == 0)
			ct_offload_orig.tstamp = jiffies;
	} else if (odev_type == BCM_NETDEVICE_GROUP_LAN) {
		if (idev_type == BCM_NETDEVICE_GROUP_LAN) {
			/* Promote bidirectional flow to FAP */
			if (flowmgr_promote(flow_lanlan, prot,
					    skb->priority,
					    ct, nf_bridge,
					    tuple_orig, tuple_repl,
					    0, &ct_offload_orig) == 0)
				ct_offload_orig.tstamp = jiffies;
			if (flowmgr_promote(flow_lanlan, prot,
					    skb->priority,
					    ct, nf_bridge,
					    tuple_repl, tuple_orig,
					    1, &ct_offload_repl) == 0){
				ct_offload_repl.tstamp = jiffies;
			}
		} else {
			/* Promote bidirectional flow to FAP */
			if (flowmgr_promote(flow_wanlan, prot,
					    skb->priority,
					    ct, nf_bridge,
					    tuple_orig, tuple_repl,
					    0, &ct_offload_orig) == 0)
				ct_offload_orig.tstamp = jiffies;
			if (flowmgr_promote(flow_lanwan, prot,
					    skb->priority,
					    ct, nf_bridge,
					    tuple_repl, tuple_orig,
					    1, &ct_offload_repl) == 0){
				ct_offload_repl.tstamp = jiffies;
			}
		}
	}
	if (prot == IPPROTO_TCP)
		atomic_add(2, &flowmgr.flow_tcp_cnt);
	else if (prot == IPPROTO_UDP)
		atomic_add(2, &flowmgr.flow_udp_cnt);

	ct_offload->check_status_timeout = nfct_time_stamp +
		flowmgr.check_status_timeout * HZ;
	skb->peeked = 1;
	return 0;
}

static inline
int promote_3way_tcp(struct sk_buff *skb,
		struct net_device *in,
		struct net_device *out,
		struct nf_conn *ct)
{
	promote_duplex(skb, in, out, ct, IPPROTO_TCP);
	return 0;
}

static inline
int promote_2way_tcp(struct sk_buff *skb,
		     struct net_device *in,
		     struct net_device *out,
		     struct nf_conn *ct)
{
	const struct nf_conntrack_l4proto *l4proto;
	struct nf_tcp_net *tn = nf_tcp_pernet(nf_ct_net(ct));
	unsigned int *timeouts;
	if (CTINFO2DIR(nfctinfo(skb)) == IP_CT_DIR_ORIGINAL)
		promote_duplex(skb, in, out, ct, IPPROTO_TCP);
	else
		promote_duplex(skb, out, in, ct, IPPROTO_TCP);

	ct->proto.tcp.state = TCP_CONNTRACK_ESTABLISHED;
	set_bit(IPS_ASSURED_BIT, &ct->status);
	l4proto = nf_ct_l4proto_find(IPPROTO_TCP);
	timeouts = nf_ct_timeout_lookup(ct);
	if (!timeouts)
		timeouts = tn->timeouts;
	if (timeouts) {
		unsigned long timeout;
		timeout = timeouts[TCP_CONNTRACK_ESTABLISHED];
		ct->timeout = nfct_time_stamp + timeout;
	}
	return 0;
}

static inline
int tcp_stats_update_orig(struct sk_buff *skb,
			  struct net_device *in,
			  struct net_device *out,
			  struct nf_conn *ct,
			  struct nf_conn_offload *ct_offload)
{
	atomic64_add(1, &flowmgr.in_flight_tcp_pkt_cnt[IP_CT_DIR_ORIGINAL]);
	flowmgr_dec_dev_slow_stats(out, 1, skb->len, 0,
				   ip_get_slow_dscp_bp(skb));
	if (flowmgr_is_db_stats_update_allowed(in, out))
		flowmgr_db_dec_tx_slow(out, ct_offload_orig.eh.h_dest,
			1, skb->len, ip_get_slow_dscp_bp(skb));
	if (BCM_NETDEVICE_GROUP_TYPE(in->group) == BCM_NETDEVICE_GROUP_WAN) {
		flowmgr_dec_dev_slow_stats(in, 1, skb->len, 1,
			ip_get_slow_dscp_bp(skb));
		if (!ct_offload_repl.expected && ct_offload_repl.flow_type)
			flowmgr_db_dec_rx_slow(in, ct_offload_repl.eh.h_dest,
				1, skb->len, ip_get_slow_dscp_bp(skb));
	}
	return 0;
}

static inline
int tcp_stats_update_repl(struct sk_buff *skb,
			  struct net_device *in,
			  struct net_device *out,
			  struct nf_conn *ct,
			  struct nf_conn_offload *ct_offload)
{
	atomic64_add(1, &flowmgr.in_flight_tcp_pkt_cnt[IP_CT_DIR_REPLY]);
	flowmgr_dec_dev_slow_stats(out, 1, skb->len, 0,
				   ip_get_slow_dscp_bp(skb));
	if (flowmgr_is_db_stats_update_allowed(in, out))
		flowmgr_db_dec_tx_slow(out, ct_offload_repl.eh.h_dest,
			1, skb->len, ip_get_slow_dscp_bp(skb));
	if (BCM_NETDEVICE_GROUP_TYPE(in->group) == BCM_NETDEVICE_GROUP_WAN) {
		flowmgr_dec_dev_slow_stats(in, 1, skb->len, 1,
			ip_get_slow_dscp_bp(skb));
		if (!ct_offload_orig.expected && ct_offload_orig.flow_type)
			flowmgr_db_dec_rx_slow(in, ct_offload_orig.eh.h_dest,
				1, skb->len, ip_get_slow_dscp_bp(skb));
	}
	return 0;
}

static inline
int process_packet_tcp_duplex(struct sk_buff *skb,
			      struct net_device *in,
			      struct net_device *out,
			      struct nf_conn *ct,
			      struct nf_conn_offload *ct_offload)
{
	enum ip_conntrack_info ctinfo = nfctinfo(skb);

	if (!new_offload(skb, in, out, ct, ct_offload, IPPROTO_TCP))
		return 0;

	if (CTINFO2DIR(ctinfo) == IP_CT_DIR_ORIGINAL) {
		ct_offload_orig.packets_slow++;

		if (!ct_offload_orig.expected && ct_offload_orig.flow_type)
			return tcp_stats_update_orig(skb, in, out,
						     ct, ct_offload);

		ct_offload_orig.ctinfo = ctinfo;

	} else {
		ct_offload_repl.packets_slow++;

		if (!ct_offload_repl.expected && ct_offload_repl.flow_type)
			return tcp_stats_update_repl(skb, in, out,
						     ct, ct_offload);

		ct_offload_repl.ctinfo = ctinfo;
	}

	spin_lock_bh(&ct->lock);
	if (ct_offload_orig.map || ct_offload_repl.map) {
		promote_3way_tcp(skb, in, out, ct);
	} else if (flowmgr.use_tcp_3way_handshake) {
		if (ctinfo == IP_CT_ESTABLISHED) {
			if ((ct_offload_orig.packets_slow > 0) &&
				 (ct_offload_repl.packets_slow > 1))
				promote_3way_tcp(skb, in, out, ct);
			else if ((ct_offload_orig.packets_slow > 1) &&
				 (ct_offload_repl.packets_slow > 0))
				promote_3way_tcp(skb, in, out, ct);
		}
	} else {
		if (ct_offload_orig.packets_slow &&
		    ct_offload_repl.packets_slow)
			promote_2way_tcp(skb, in, out, ct);
	}
	spin_unlock_bh(&ct->lock);

	return 0;
}

static inline
int promote_single(struct sk_buff *skb,
		struct net_device *in,
		struct net_device *out,
		struct nf_conn *ct,
		int prot)
{
	struct nf_conntrack_tuple *tuple_orig;
	struct nf_conntrack_tuple *tuple_repl;
	struct nf_conn_offload *ct_offload;
	struct offload_info *ct_offload_info;
	struct nf_bridge_info	*nf_bridge;
	int idev_type = BCM_NETDEVICE_GROUP_TYPE(in->group);
	int odev_type = BCM_NETDEVICE_GROUP_TYPE(out->group);

	ct_offload = nf_conn_offload_find(ct);
	nf_bridge = get_nf_bridge(skb);
	ct_offload_info = &ct_offload->info[CTINFO2DIR(nfctinfo(skb))];
	if (!flowmgr_manual_trigger_check(ct, skb,
				    ct_offload_info->start_promote))
		return 0;

	if (ct_offload_info->map) {
#ifdef CONFIG_BCM_FLOWMGR_TUNNEL
		if ((odev_type == BCM_NETDEVICE_GROUP_WAN) ||
			(idev_type == BCM_NETDEVICE_GROUP_WAN)) {
			/* Promote unidirectional MAP-T */
			if (flowmgr_promote_mapt(prot, skb,
						ct, ct_offload_info) == 0) {
				ct_offload_info->tstamp = jiffies;
			}
		} else {
			pr_err("FLOWMGR: Invalid MAP-T direction\n");
			return 0;
		}
#else
		pr_err("FLOWMGR: MAP-T offload set without support\n");
		return 0;
#endif
	} else if (CTINFO2DIR(nfctinfo(skb)) == IP_CT_DIR_ORIGINAL) {
		tuple_orig = nf_ct_tuple(ct, IP_CT_DIR_ORIGINAL);
		tuple_repl = nf_ct_tuple(ct, IP_CT_DIR_REPLY);
		if (odev_type == BCM_NETDEVICE_GROUP_WAN) {
			/* Promote unidirectional flow to FAP */
			if (flowmgr_promote(flow_lanwan, prot,
					    skb->priority,
					    ct, nf_bridge,
					    tuple_orig, tuple_repl,
					    1, &ct_offload_orig) == 0)
				ct_offload_orig.tstamp = jiffies;
		} else if (idev_type == BCM_NETDEVICE_GROUP_LAN) {
			/* Promote unidirectional flow to FAP */
			if (flowmgr_promote(flow_lanlan, prot,
					    skb->priority,
					    ct, nf_bridge,
					    tuple_orig, tuple_repl,
					    0, &ct_offload_orig) == 0)
				ct_offload_orig.tstamp = jiffies;
		} else {
			/* Promote unidirectional flow to FAP */
			if (flowmgr_promote(flow_wanlan, prot,
					    skb->priority,
					    ct, nf_bridge,
					    tuple_orig, tuple_repl,
					    0, &ct_offload_orig) == 0)
				ct_offload_orig.tstamp = jiffies;
		}
	} else {
		tuple_orig = nf_ct_tuple(ct, IP_CT_DIR_ORIGINAL);
		tuple_repl = nf_ct_tuple(ct, IP_CT_DIR_REPLY);
		if (odev_type == BCM_NETDEVICE_GROUP_WAN) {
			/* Promote unidirectional flow to FAP */
			if (flowmgr_promote(flow_lanwan, prot,
					    skb->priority,
					    ct, nf_bridge,
					    tuple_repl, tuple_orig,
					    1, &ct_offload_repl) == 0)
				ct_offload_repl.tstamp = jiffies;
		} else if (idev_type == BCM_NETDEVICE_GROUP_LAN) {
			/* Promote unidirectional flow to FAP */
			if (flowmgr_promote(flow_lanlan, prot,
					    skb->priority,
					    ct, nf_bridge,
					    tuple_repl, tuple_orig,
					    1, &ct_offload_repl) == 0)
				ct_offload_repl.tstamp = jiffies;
		} else {
			/* Promote unidirectional flow to FAP */
			if (flowmgr_promote(flow_wanlan, prot,
					    skb->priority,
					    ct, nf_bridge,
					    tuple_repl, tuple_orig,
					    0, &ct_offload_repl) == 0)
				ct_offload_repl.tstamp = jiffies;
		}
	}
	if (prot == IPPROTO_TCP)
		atomic_inc(&flowmgr.flow_tcp_cnt);
	else if (prot == IPPROTO_UDP)
		atomic_inc(&flowmgr.flow_udp_cnt);
	else if (prot == IPPROTO_ESP)
		atomic_inc(&flowmgr.flow_esp_cnt);
	else if (prot == IPPROTO_AH)
		atomic_inc(&flowmgr.flow_ah_cnt);
	skb->peeked = 1;
	ct_offload->check_status_timeout = nfct_time_stamp +
		flowmgr.check_status_timeout * HZ;
	return 0;
}

static inline int flowmgr_is_dev_dhd(struct net_device *dev)
{
	struct dqnet_netdev *ndev;

	if ((BCM_NETDEVICE_GROUP_CAT(dev->group) !=
			   BCM_NETDEVICE_GROUP_CAT_PHY))
		return 0;

	ndev = netdev_priv(dev);
	if (!ndev)
		return 0;

	if (!ndev->dhdol_get_flow)
		return 0;

	return 1;
}

static inline
int process_packet_single(struct sk_buff *skb,
			  struct net_device *in,
			  struct net_device *out,
			  struct nf_conn *ct,
			  struct nf_conn_offload *ct_offload,
			  int protonum)
{
	enum ip_conntrack_info ctinfo = nfctinfo(skb);
	struct nf_conn_acct *acct;
	struct nf_conn_counter *nf_counter;
	int pkt_threshold = 0;

	if (!new_offload(skb, in, out, ct, ct_offload, protonum))
		return 0;

	if (protonum == IPPROTO_TCP)
		pkt_threshold = flowmgr.tcp_pkt_threshold;
	else if (protonum == IPPROTO_UDP)
		pkt_threshold = flowmgr.udp_pkt_threshold;
	else if (protonum == IPPROTO_ESP)
		pkt_threshold = flowmgr.esp_pkt_threshold;
	else if (protonum == IPPROTO_AH)
		pkt_threshold = flowmgr.ah_pkt_threshold;

	acct = nf_conn_acct_find(ct);
	if (!acct)
		return 0;
	if (CTINFO2DIR(ctinfo) == IP_CT_DIR_ORIGINAL) {
		struct nf_conntrack_tuple *tuple;
		u64 ct_packets;

		if (ct_offload_orig.skip & NF_CT_OFFLOAD_SKIP_ACTIVE)
			return 0;

		tuple = nf_ct_tuple(ct, IP_CT_DIR_ORIGINAL);
		nf_counter = &(acct->counter[IP_CT_DIR_ORIGINAL]);
		ct_packets = atomic64_read(&(nf_counter->packets));
		if (flowmgr_is_dev_dhd(out)) {
			if (ct_offload_orig.wifi_flowring == 0xFFFF) {
				ct_offload_orig.ctinfo = -1;
				if (ct_offload_orig.debug)
					 pr_alert("FLOWMGR: Promote orig flow dhd flowring not initialized at packets %lld\n",
						  ct_packets);
				return 0;
			}
		}
		if (!(ct_offload_orig.skip & NF_CT_OFFLOAD_SKIP_ENABLE) &&
		    (ct_packets < (u64)pkt_threshold))
			return 0;
		if (!ct_offload_orig.expected && ct_offload_orig.flow_type) {
			struct net_device *idev;
			struct net_device *odev;
			idev = __dev_get_by_index(&init_net, ct_offload_orig.iif);
			odev = __dev_get_by_index(&init_net, ct_offload_orig.oif);
			skb->peeked = 1;
			if (in->ifindex != ct_offload_orig.iif) {
				pr_debug(
				   "FLOWMGR: prot %d Pkt rcvd (%s -> %s) for orig promoted flow %d (%s -> %s)\n",
				   protonum, in->name, out->name,
				   ct_offload_orig.flow_id,
				   idev->name,
				   odev->name);
				/* Flush this flow if transitioning from tunnel */
				if (flowmgr_is_dev_tunnel((struct net_device *)in)) {
					flowmgr_demote(ct, ct_offload);
					atomic_inc(&flowmgr.flow_flush_orig_cnt);
					return 0;
				} else if (flowmgr_is_dev_tunnel((struct net_device *)idev)) {
					flowmgr_demote(ct, ct_offload);
					atomic_inc(&flowmgr.flow_flush_orig_cnt);
					return 0;
				}
			} else if (out->ifindex != ct_offload_orig.oif) {
				pr_debug(
				   "FLOWMGR: prot %d Pkt rcvd (%s -> %s) for orig promoted flow %d (%s -> %s)\n",
				   protonum, in->name, out->name,
				   ct_offload_orig.flow_id,
				   idev->name,
				   odev->name);
				/* Flush this flow if transitioning from tunnel */
				if (flowmgr_is_dev_tunnel((struct net_device *)out)) {
					flowmgr_demote(ct, ct_offload);
					atomic_inc(&flowmgr.flow_flush_orig_cnt);
					return 0;
				} else if (flowmgr_is_dev_tunnel((struct net_device *)odev)) {
					flowmgr_demote(ct, ct_offload);
					atomic_inc(&flowmgr.flow_flush_orig_cnt);
					return 0;
				}
			} else {
				if (protonum == IPPROTO_TCP) {
					tcp_stats_update_orig(skb, in, out,
							      ct, ct_offload);
				} else if (protonum == IPPROTO_UDP)
					atomic64_add(1,
					   &flowmgr.in_flight_udp_pkt_cnt[IP_CT_DIR_ORIGINAL]);
				else if (protonum == IPPROTO_ESP)
					atomic_inc(
					   &flowmgr.in_flight_esp_pkt_cnt[IP_CT_DIR_ORIGINAL]);
				else if (protonum == IPPROTO_AH)
					atomic_inc(
					   &flowmgr.in_flight_ah_pkt_cnt[IP_CT_DIR_ORIGINAL]);
				return 0; /* Flow already promoted */
			}
		} else if (ct_offload_orig.flow_id == -1 &&
		    (in->ifindex != ct_offload_orig.iif ||
		    (out->ifindex != ct_offload_orig.oif))) {
			struct net_device *idev;
			struct net_device *odev;
			idev = __dev_get_by_index(&init_net, ct_offload_orig.iif);
			odev = __dev_get_by_index(&init_net, ct_offload_orig.oif);
			skb->peeked = 1;
			pr_debug(
			   "FLOWMGR: prot %d Pkt rcvd (%s -> %s) for orig recorded flow (%s -> %s)\n",
			   protonum, in->name, out->name,
			   idev->name,
			   odev->name);
			ct_offload_orig.ctinfo = -1;
			return 0;
		}
		if (ct_offload_orig.debug) {
			nf_ct_tuple_dump(pr_alert, tuple, "FLOWMGR: Promote orig");
			pr_alert("FLOWMGR: Promote orig flow at packet %lld bytes %lld skip %x\n",
				 ct_packets, atomic64_read(&(nf_counter->bytes)),
				 ct_offload_orig.skip);
		}
		promote_single(skb, in, out, ct, protonum);
	} else {
		struct nf_conntrack_tuple *tuple;
		u64 ct_packets;

		if (ct_offload_repl.skip & NF_CT_OFFLOAD_SKIP_ACTIVE)
			return 0;

		tuple = nf_ct_tuple(ct, IP_CT_DIR_REPLY);
		nf_counter = &(acct->counter[IP_CT_DIR_REPLY]);
		ct_packets = atomic64_read(&(nf_counter->packets));

		if (flowmgr_is_dev_dhd(out)) {
			if (ct_offload_repl.wifi_flowring == 0xFFFF) {
				ct_offload_repl.ctinfo = -1;
				if (ct_offload_repl.debug)
					 pr_alert("FLOWMGR: Promote repl flow dhd flowring not initialized at packets %lld\n",
						  ct_packets);
				return 0;
			}
		}
		if (!(ct_offload_repl.skip & NF_CT_OFFLOAD_SKIP_ENABLE) &&
		    (ct_packets < (u64)pkt_threshold))
			return 0;
		if (!ct_offload_repl.expected && ct_offload_repl.flow_type) {
			struct net_device *idev;
			struct net_device *odev;
			idev = __dev_get_by_index(&init_net, ct_offload_repl.iif);
			odev = __dev_get_by_index(&init_net, ct_offload_repl.oif);
			skb->peeked = 1;
			if (in->ifindex != ct_offload_repl.iif) {
				pr_debug(
				   "FLOWMGR: prot %d Pkt rcvd (%s -> %s) for promoted repl flow %d (%s -> %s)\n",
				   protonum, in->name, out->name,
				   ct_offload_repl.flow_id,
				   idev->name,
				   odev->name);
				/* Demote this flow if transitioning from tunnel */
				if (flowmgr_is_dev_tunnel((struct net_device *)in)) {
					flowmgr_demote(ct, ct_offload);
					atomic_inc(&flowmgr.flow_flush_repl_cnt);
					return 0;
				} else if (flowmgr_is_dev_tunnel((struct net_device *)idev)) {
					flowmgr_demote(ct, ct_offload);
					atomic_inc(&flowmgr.flow_flush_repl_cnt);
					return 0;
				}
			} else if (out->ifindex != ct_offload_repl.oif) {
				pr_debug(
				   "FLOWMGR: prot %d Pkt rcvd (%s -> %s) for promoted repl flow %d (%s -> %s)\n",
				   protonum, in->name, out->name,
				   ct_offload_repl.flow_id,
				   idev->name,
				   odev->name);
				/* Demote this flow if transitioning from tunnel */
				if (flowmgr_is_dev_tunnel((struct net_device *)out)) {
					flowmgr_demote(ct, ct_offload);
					atomic_inc(&flowmgr.flow_flush_repl_cnt);
					return 0;
				} else if (flowmgr_is_dev_tunnel((struct net_device *)odev)) {
					flowmgr_demote(ct, ct_offload);
					atomic_inc(&flowmgr.flow_flush_repl_cnt);
					return 0;
				}
			} else {
				if (protonum == IPPROTO_TCP) {
					tcp_stats_update_repl(skb, in, out,
							      ct, ct_offload);
				} else if (protonum == IPPROTO_UDP)
					atomic64_add(1,
					   &flowmgr.in_flight_udp_pkt_cnt[IP_CT_DIR_REPLY]);
				else if (protonum == IPPROTO_ESP)
					atomic_inc(
					   &flowmgr.in_flight_esp_pkt_cnt[IP_CT_DIR_REPLY]);
				else if (protonum == IPPROTO_AH)
					atomic_inc(
					   &flowmgr.in_flight_ah_pkt_cnt[IP_CT_DIR_REPLY]);
				return 0; /* Flow already promoted */
			}
		} else if (ct_offload_repl.flow_id == -1 &&
		    (in->ifindex != ct_offload_repl.iif ||
		    (out->ifindex != ct_offload_repl.oif))) {
			struct net_device *idev;
			struct net_device *odev;
			idev = __dev_get_by_index(&init_net, ct_offload_repl.iif);
			odev = __dev_get_by_index(&init_net, ct_offload_repl.oif);
			skb->peeked = 1;
			pr_debug(
			   "FLOWMGR: prot %d Pkt rcvd (%s -> %s) for orig recorded flow (%s -> %s)\n",
			   protonum, in->name, out->name,
			   idev->name,
			   odev->name);
			ct_offload_repl.ctinfo = -1;
			return 0;
		}
		if (ct_offload_repl.debug) {
			nf_ct_tuple_dump(pr_alert, tuple, "FLOWMGR: Promote repl");
			pr_alert("FLOWMGR: Promote repl flow at packet %lld bytes %lld skip %x\n",
				 ct_packets, atomic64_read(&(nf_counter->bytes)),
				 ct_offload_repl.skip);
		}
		promote_single(skb, in, out, ct, protonum);
	}
	return 0;
}

#define FAST_DEATH_TIMEOUT (flowmgr.fast_death_timeout * HZ) /* In Sec */

static inline
int kill_ct(struct nf_conn *ct)
{
	nf_ct_kill(ct);
	return 0;
}

static inline
int update_timeout_if_master_dying(struct nf_conn *ct)
{
	if (!ct->master)
		return 0;
	if (nf_ct_is_dying(ct->master)) {
		pr_debug("update_timeout: Master dying %px\n",
			 ct->master);
		ct->timeout = nfct_time_stamp + FAST_DEATH_TIMEOUT;
	}
	return 0;
}

static int flowtype_cmp(struct nf_conn *ct, void *data)
{
	struct nf_conn_offload *ct_offload;
	ct_offload = nf_conn_offload_find(ct);
	if (!ct_offload)
		return 0;
	if (ct_offload_orig.flow_type)
		return 1;
	if (ct_offload_repl.flow_type)
		return 1;
	return 0;
}
void flowmgr_conntrack_clean_all(void)
{
	rcu_read_lock();
	nf_ct_iterate_cleanup_net(&init_net, flowtype_cmp,
				  NULL, 0, 0);
	rcu_read_unlock();
}

struct flow_flush_work {
	struct work_struct work;
	struct net *net;
	int (*iter)(struct nf_conn *i, void *data);
	char mac[ETH_ALEN];
	int ifindex;
	struct flowmgr_db_entry *db;
	int wifi_flowring;
	/* 1: Delete from FAP
	   2: Delete from FAP and Unlearn flow
	   3: Remove conntrack */
	int mode;
};

static void flowmgr_unlearn(struct nf_conn *ct, struct nf_conn_offload *ct_offload)
{
	if (ct_offload_orig.debug) {
		struct nf_conntrack_tuple *tuple;

		tuple = nf_ct_tuple(ct, IP_CT_DIR_ORIGINAL);
		nf_ct_tuple_dump(pr_alert, tuple, "FLOWMGR: Orig unlearn");
	}
	if (ct_offload_repl.debug) {
		struct nf_conntrack_tuple *tuple;

		tuple = nf_ct_tuple(ct, IP_CT_DIR_REPLY);
		nf_ct_tuple_dump(pr_alert, tuple, "FLOWMGR: Repl unlearn");
	}

	/* Clear learning for unpromoted flows */
	if (ct_offload_orig.flow_id == -1) {
		memset(&ct_offload_orig, 0, offsetof(struct offload_info, debug));
		ct_offload_orig.flow_id = -1;
		atomic_inc(&flowmgr.flow_unlearn_orig_cnt);
	}
	if (ct_offload_repl.flow_id == -1) {
		memset(&ct_offload_repl, 0, offsetof(struct offload_info, debug));
		ct_offload_repl.flow_id = -1;
		atomic_inc(&flowmgr.flow_unlearn_repl_cnt);
	}
	/* Clear learning for promoted flows */
	flowmgr_demote(ct, ct_offload);
	/* Clear larning for expected flows */
	ct_offload_orig.ctinfo = -1;
	ct_offload_orig.wifi_flowring = 0xFFFF;
	ct_offload_orig.wifi_pri = 0;
	ct_offload_repl.ctinfo = -1;
	ct_offload_repl.wifi_flowring = 0xFFFF;
	ct_offload_repl.wifi_pri = 0;
	atomic_inc(&flowmgr.flow_unlearn_cnt);
}

static int inf_cmp_flow_flush(struct nf_conn *ct, void *data)
{
	struct nf_conn_offload *ct_offload;
	struct flow_flush_work *f = (struct flow_flush_work *) data;
	ct_offload = nf_conn_offload_find(ct);
	if (!ct_offload)
		return 0;
	if (ct_offload_orig.ctinfo == -1)
		goto check_repl;
	if (f->ifindex == ct_offload_orig.iif)
		goto done;

	if (f->ifindex == ct_offload_orig.oif)
		goto done;

check_repl:
	if (ct_offload_repl.ctinfo == -1)
		return 0;
	if (f->ifindex == ct_offload_repl.iif)
		goto done;

	if (f->ifindex == ct_offload_repl.oif)
		goto done;
	return 0;
done:
	atomic_inc(&flowmgr.flow_flush_inf_cnt);
	if (f->mode == FLOW_FLUSH_FAP) {
		flowmgr_demote(ct, ct_offload);
		ct_offload_orig.flow_type = ft_ignore<<16;
		ct_offload_repl.flow_type = ft_ignore<<16;
	} else if (f->mode == FLOW_FLUSH_UNLEARN)
		flowmgr_unlearn(ct, ct_offload);
	else if (f->mode == FLOW_FLUSH_NF)
		return 1;
	return 0;
}

void flowmgr_conntrack_clean_by_mapt_domain(int domain)
{
	rcu_read_lock();
	nf_ct_iterate_cleanup_net(&init_net, flowmgr_mapt_domain_cmp,
				  (void *)(long)&domain, 0, 0);
	rcu_read_unlock();
}

void flowmgr_conntrack_clean_by_inf(struct net *net,
				    struct net_device *dev)
{
	struct flow_flush_work w;
	w.net = net;
	w.ifindex = dev->ifindex;
	w.mode = flowmgr.auto_flush_inf_mode;
	rcu_read_lock();
	nf_ct_iterate_cleanup_net(net, inf_cmp_flow_flush,
				  (void *)&w, 0, 0);
	rcu_read_unlock();
}

static int mac_cmp_flow_flush(struct nf_conn *ct, void *data)
{
	struct nf_conn_offload *ct_offload;
	struct flow_flush_work *f = (struct flow_flush_work *) data;
	ct_offload = nf_conn_offload_find(ct);
	if (!ct_offload)
		return 0;
	if (ct_offload_orig.ctinfo == -1)
		goto check_repl;
	if (ether_addr_equal(f->mac,
			     ct_offload_orig.eh.h_source) &&
	    (f->ifindex == ct_offload_orig.iif))
		goto done;

	if (ether_addr_equal(f->mac,
			     ct_offload_orig.eh.h_dest) &&
	    (f->ifindex == ct_offload_orig.oif))
		goto done;

check_repl:
	if (ct_offload_repl.ctinfo == -1)
		return 0;
	if (ether_addr_equal(f->mac,
			     ct_offload_repl.eh.h_source) &&
	    (f->ifindex == ct_offload_repl.iif))
		goto done;

	if (ether_addr_equal(f->mac,
			     ct_offload_repl.eh.h_dest) &&
	    (f->ifindex == ct_offload_repl.oif))
		goto done;
	return 0;
done:
	atomic_inc(&flowmgr.flow_flush_mac_cnt);
	if (f->mode == FLOW_FLUSH_FAP) {
		flowmgr_demote(ct, ct_offload);
		ct_offload_orig.flow_type = ft_ignore<<16;
		ct_offload_repl.flow_type = ft_ignore<<16;
	} else if (f->mode == FLOW_FLUSH_UNLEARN)
		flowmgr_unlearn(ct, ct_offload);
	else if (f->mode == FLOW_FLUSH_NF)
		return 1;
	return 0;
}

static int wifi_cmp_flow_flush(struct nf_conn *ct, void *data)
{
	struct nf_conn_offload *ct_offload;
	struct flow_flush_work *f = (struct flow_flush_work *) data;
	ct_offload = nf_conn_offload_find(ct);
	if (!ct_offload)
		return 0;
	if (ct_offload_orig.ctinfo == -1)
		goto check_repl;
	if (ether_addr_equal(f->mac,
			     ct_offload_orig.eh.h_source) &&
	    (f->wifi_flowring == ct_offload_orig.wifi_flowring))
		goto done;

	if (ether_addr_equal(f->mac,
			     ct_offload_orig.eh.h_dest) &&
	    (f->wifi_flowring == ct_offload_orig.wifi_flowring))
		goto done;

check_repl:
	if (ct_offload_repl.ctinfo == -1)
		return 0;
	if (ether_addr_equal(f->mac,
			     ct_offload_repl.eh.h_source) &&
	    (f->wifi_flowring == ct_offload_repl.wifi_flowring))
		goto done;

	if (ether_addr_equal(f->mac,
			     ct_offload_repl.eh.h_dest) &&
	    (f->wifi_flowring == ct_offload_repl.wifi_flowring))
		goto done;
	return 0;
done:
	atomic_inc(&flowmgr.flow_flush_wifi_cnt);
	if (f->mode == FLOW_FLUSH_FAP) {
		flowmgr_demote(ct, ct_offload);
		ct_offload_orig.flow_type = ft_ignore<<16;
		ct_offload_repl.flow_type = ft_ignore<<16;
	} else if (f->mode == FLOW_FLUSH_UNLEARN)
		flowmgr_unlearn(ct, ct_offload);
	else if (f->mode == FLOW_FLUSH_NF)
		return 1;
	return 0;
}

static int db_cmp_flow_flush(struct nf_conn *ct, void *data)
{
	struct nf_conn_offload *ct_offload;
	struct flow_flush_work *f = (struct flow_flush_work *) data;
	ct_offload = nf_conn_offload_find(ct);
	if (!ct_offload)
		return 0;
	if (ct_offload_orig.ctinfo == -1)
		goto check_repl;
	if (f->db == ct_offload_orig.idb)
		goto done;
	if (f->db == ct_offload_orig.odb)
		goto done;

check_repl:
	if (ct_offload_repl.ctinfo == -1)
		return 0;
	if (f->db == ct_offload_repl.idb)
		goto done;
	if (f->db == ct_offload_repl.odb)
		goto done;
	return 0;
done:
	atomic_inc(&flowmgr.flow_flush_mac_cnt);
	if (f->mode == FLOW_FLUSH_FAP) {
		flowmgr_demote(ct, ct_offload);
		ct_offload_orig.flow_type = ft_ignore<<16;
		ct_offload_repl.flow_type = ft_ignore<<16;
	} else if (f->mode == FLOW_FLUSH_UNLEARN)
		flowmgr_unlearn(ct, ct_offload);
	else if (f->mode == FLOW_FLUSH_NF)
		return 1;
	atomic_inc(&f->db->nsteers);
	return 0;
}

void flowmgr_flow_flush_work(struct work_struct *work)
{
	struct flow_flush_work *w;
	w = container_of(work, struct flow_flush_work, work);
	if (w->wifi_flowring == 0xFFFF)
		pr_debug("FLOWMGR: Flow flush mode %d for mac %pM ifindex %d\n",
			 w->mode, w->mac, w->ifindex);
	else
		pr_debug("FLOWMGR: Flow flush mode %d for mac %pM ifindex %d wifi flring %d\n",
			 w->mode, w->mac, w->ifindex, w->wifi_flowring);
	rcu_read_lock();
	nf_ct_iterate_cleanup_net(w->net, w->iter, (void *)w, 0, 0);
	rcu_read_unlock();
	kfree(w);
	atomic_dec(&flowmgr.flush_worker_count);
	module_put(THIS_MODULE);
}

int flowmgr_flow_flush(struct net *net, const char *addr,
		       int ifindex, int mode)
{
	struct flow_flush_work *w;

	if (!try_module_get(THIS_MODULE))
		goto fail;
	w = kzalloc(sizeof(*w), GFP_ATOMIC);
	if (w) {
		atomic_inc(&flowmgr.flush_worker_count);

		INIT_WORK(&w->work, flowmgr_flow_flush_work);
		w->net = net;
		w->ifindex = ifindex;
		w->mode = mode;
		w->wifi_flowring = 0xFFFF;
		if (addr) {
			/* MAC and INF based flushing */
			memcpy(w->mac, addr, ETH_ALEN);
			w->iter = mac_cmp_flow_flush;
		} else
			w->iter = inf_cmp_flow_flush;
		queue_work_on(WORK_CPU_UNBOUND, kflowmgr_wq, &w->work);

		return 0;
	}
	module_put(THIS_MODULE);
fail:
	pr_err("FLOWMGR: Error %s\n", __func__);
	return -1;
}

int flowmgr_flow_flush_wifi(struct net *net, const char *addr,
			    int flring, int mode)
{
	struct flow_flush_work *w;

	if (!addr)
		goto fail;

	if (!try_module_get(THIS_MODULE))
		goto fail;
	w = kzalloc(sizeof(*w), GFP_ATOMIC);
	if (w) {
		atomic_inc(&flowmgr.flush_worker_count);

		INIT_WORK(&w->work, flowmgr_flow_flush_work);
		w->net = net;
		w->wifi_flowring = flring;
		w->mode = mode;
		memcpy(w->mac, addr, ETH_ALEN);
		w->iter = wifi_cmp_flow_flush;
		schedule_work(&w->work);

		return 0;
	}
	module_put(THIS_MODULE);
fail:
	pr_err("FLOWMGR: Error %s\n", __func__);
	return -1;
}

int flowmgr_flow_flush_db(struct net *net, struct flowmgr_db_entry *db,
			  int mode)
{
	struct flow_flush_work *w;

	if (!try_module_get(THIS_MODULE))
		goto fail;
	w = kzalloc(sizeof(*w), GFP_ATOMIC);
	if (w) {
		atomic_inc(&flowmgr.flush_worker_count);

		INIT_WORK(&w->work, flowmgr_flow_flush_work);
		w->net = net;
		w->db = db;
		w->mode = mode;
		w->wifi_flowring = 0xFFFF;
		w->iter = db_cmp_flow_flush;
		queue_work_on(WORK_CPU_UNBOUND, kflowmgr_wq, &w->work);

		return 0;
	}
	module_put(THIS_MODULE);
fail:
	pr_err("FLOWMGR: Error %s\n", __func__);
	return -1;
}

static int db_check(struct sk_buff *skb,
		    struct net_device *in,
		    int src)
{
	struct flowmgr_db_entry *db;
	struct net_device *dev;
	struct ethhdr *eh = (struct ethhdr *) eth_hdr(skb);
	int eth_proto = eh->h_proto;
	u16 vid = 0;
	unsigned char *mac;

	if (!flowmgr.enable)
		return 0;

	if (src)
		mac = eh->h_source;
	else
		mac = eh->h_dest;

	if (is_multicast_ether_addr(mac) || !is_flowmgr_db_ifenable(in))
		return 0;

	db = flowmgr_db_find(mac);

	if (eth_proto == ntohs(ETH_P_8021Q)) {
		struct vlan_ethhdr *veth = vlan_eth_hdr(skb);
		vid = htons(veth->h_vlan_TCI) & VLAN_VID_MASK;
		eth_proto = veth->h_vlan_encapsulated_proto;
	}

	/* Use only IP packets to detect and trigger steering */
	if ((eth_proto != ntohs(ETH_P_IP)) && (eth_proto != ntohs(ETH_P_IPV6)))
		return 0;

	if (!db) {
		flowmgr_db_update(in, mac, vid);
		return 0;
	}

	dev = __dev_get_by_index(&init_net, db->iif);
	if (!dev)
		return 0;

	if (!db->steer)
		return 0;

	if (db->iif != in->ifindex) {
		/* Change in interface index */
		pr_debug("FLOWMGR: %pM moved from interface %s -> %s\n",
			 mac, dev->name, in->name);
		if (flowmgr.auto_flush_mac_mode)
			flowmgr_flow_flush_db(&init_net, db,
					   flowmgr.auto_flush_mac_mode);
		flowmgr_db_update(in, mac, vid);
	}

	if (!db->vlan_id && vid) {
		/* Change from non vlan to vlan */
		pr_debug("FLOWMGR: %pM moved from non vlan -> vlan %x on interface %s\n",
			 mac, vid, dev->name);
		if (flowmgr.auto_flush_mac_mode)
			flowmgr_flow_flush_db(&init_net, db,
					   flowmgr.auto_flush_mac_mode);
		flowmgr_db_update(in, mac, vid);
		atomic_inc(&db->nsteers);
	} else if (db->vlan_id != vid) {
		/* Change in vlan */
		pr_debug("FLOWMGR: %pM moved from vlan %x -> %x on interface %s\n",
			 mac, db->vlan_id, vid, dev->name);
		if (flowmgr.auto_flush_mac_mode)
			flowmgr_flow_flush_db(&init_net, db,
					   flowmgr.auto_flush_mac_mode);
		flowmgr_db_update(in, mac, vid);
	}

	return 0;
}

static int flow_create_err_check(const struct nf_conn *ct,
				 struct nf_conn_offload *ct_offload,
				 const enum ip_conntrack_info ctinfo)
{
	struct offload_info *ct_offload_info = &ct_offload->info[CTINFO2DIR(ctinfo)];

	return ct_offload_info->create_err;
}

static int process_packet(struct sk_buff *skb,
			  struct net_device *in,
			  struct net_device *out,
			  struct nf_conn *ct,
			  struct nf_conn_offload *ct_offload,
			  int protonum)
{
	struct nf_conn_help *help;

	if (!skb_get_nfct(skb))
		goto _ret_process_packet;

	if (skb->pkt_flooded) {
		atomic_inc(&flowmgr.pkt_flood_cnt);
		goto _ret_process_packet;
	}

#ifdef MMEOUT_LOGIC
	if (ctinfo == IP_CT_ESTABLISHED)
		update_timeout_if_master_dying(ct);
#endif
	/* If connection is using helper extension and
	 * is not MAP, don't promote */
	help = nfct_help(ct);
	if ((help && help->helper) &&
	    (!ct_offload_orig.map && !ct_offload_repl.map))
		goto _ret_process_packet;
	if (protonum == IPPROTO_TCP) {
		if (flowmgr.tcp_pkt_threshold)
			process_packet_single(skb, in, out, ct,
					      ct_offload, IPPROTO_TCP);
		else if (is_nf_ct_offload_skip(ct, ct_offload,
					       nfctinfo(skb)))
			process_packet_single(skb, in, out, ct,
					      ct_offload, IPPROTO_TCP);
		else
			process_packet_tcp_duplex(skb, in, out, ct,
						  ct_offload);
	} else if (protonum == IPPROTO_UDP)
		process_packet_single(skb, in, out, ct,
				      ct_offload, IPPROTO_UDP);
	else if (protonum == IPPROTO_ESP)
		process_packet_single(skb, in, out, ct,
				      ct_offload, IPPROTO_ESP);
	else if (protonum == IPPROTO_AH)
		process_packet_single(skb, in, out, ct,
				      ct_offload, IPPROTO_AH);
	atomic64_add(1, &flowmgr.pkt_processed_cnt);

_ret_process_packet:
	return 0;
}

static int process_packet_lan2lan(struct sk_buff *skb,
				  struct net_device *in,
				  struct net_device *out,
				  struct nf_conn *ct,
				  struct nf_conn_offload *ct_offload,
				  int protonum)
{
	process_packet(skb, in, out, ct, ct_offload, protonum);
	return 0;
}

static enum bcm_nethook_result bcm_nethook_rx(
	struct net_device *in, enum bcm_nethook_type type, void *buf)
{
	struct sk_buff *skb = (struct sk_buff *)buf;
	struct ethhdr *eh = (struct ethhdr *) eth_hdr(skb);
	int eth_proto = eh->h_proto;
	unsigned int byte_len = skb->len+ETH_HLEN;

	if (eth_proto == ntohs(ETH_P_8021Q))
		byte_len += sizeof(struct vlan_hdr);

	if (BCM_NETDEVICE_GROUP_TYPE(in->group) == BCM_NETDEVICE_GROUP_LAN) {
		struct dqnet_netdev *ndev;
		int queue = skb_get_rx_queue(skb);
		ndev = netdev_priv(in);
		if (ndev->chan->rx_q_info[queue].q_type == DQNET_QUEUE_EXP)
			return BCM_NETHOOK_PASS;
	}
	flowmgr_update_dev_slow_stats(in, 1, byte_len, 1,
				      ip_get_slow_dscp_bp(skb));
	db_check(skb, in, 1);

	return BCM_NETHOOK_PASS;
}

static int l4_protonum(struct sk_buff *skb, __be16 *l3proto, bool *ipmulti)
{
	struct ethhdr *eh = (struct ethhdr *) eth_hdr(skb);
	struct ipv6hdr *ipv6h = NULL;
	struct iphdr *iph = NULL;
	char *tcph = NULL;
	struct vlan_hdr *vhdr = NULL;
	int eth_proto = eh->h_proto;
	int protonum = -1;
	int offset = ETH_HLEN;

	*ipmulti = false;
repeat:
	switch (eth_proto) {
	// coverity [bad_constant_function_call]
	case ntohs(ETH_P_IP):
		iph =  (struct iphdr *) ((void *) eh + offset);
		protonum = iph->protocol;
		if (ipv4_is_multicast(iph->daddr))
			*ipmulti = true;
		if (protonum == IPPROTO_TCP) {
			tcph = (__u8 *)(iph) + (iph->ihl << 2);
			skb->cb[0] = tcph[13];
		}
		break;
	// coverity [bad_constant_function_call]
	case ntohs(ETH_P_IPV6):
		ipv6h =  (struct ipv6hdr *) ((void *) eh + offset);
		protonum = ipv6h->nexthdr;
#ifdef CONFIG_BCM_FLOWMGR_TUNNEL
		flowmgr_map_ipv6_frag_check(ipv6h, NULL, &protonum);
#endif
		if (ipv6h->nexthdr == NEXTHDR_DEST) {
			struct ipv6_opt_hdr *opt;
			offset += sizeof(*ipv6h);
			opt = (struct ipv6_opt_hdr *)((void *) eh + offset);
			protonum = opt->nexthdr;
		}
		if (ipv6h->daddr.s6_addr[0] == 0xff)
			*ipmulti = true;
		if (protonum == IPPROTO_TCP) {
			tcph = (__u8 *)(ipv6h) + sizeof(struct ipv6hdr);
			skb->cb[0] = tcph[13];
		}
		break;
	// coverity [bad_constant_function_call]
	case ntohs(ETH_P_8021Q):
		vhdr =  (struct vlan_hdr *) ((void *) eh + offset);
		eth_proto = vhdr->h_vlan_encapsulated_proto;
		offset += sizeof(struct vlan_hdr);
		goto repeat;
		break;
	default:
		break;
	}

	*l3proto = eth_proto;
	return protonum;
}

static enum bcm_nethook_result flowmgr_tx(
	struct net_device *out, enum bcm_nethook_type type, void *buf)
{
	struct sk_buff *skb = (struct sk_buff *)buf;
	const unsigned char *dest = eth_hdr(skb)->h_dest;
	struct net_device *in = NULL;
	int odev_type = BCM_NETDEVICE_GROUP_TYPE(out->group);
	int idev_type = 0;
	struct nf_conn *ct;
	struct nf_conn_offload *ct_offload;
	int protonum;
	__be16 l3proto = 0;
	bool ipmulti = false;

	if (!flowmgr.enable)
		goto out;
	/* group format: 0xCCTT - Must be initialized */
	if (!out->group) {
		pr_debug("No group Out %s\n", out->name);
		atomic_inc(&flowmgr.pkt_ignore_nogroup_out_cnt);
		goto out;
	}
	/* Skip processing of L2 broadcast and multicast packets */
	if (is_multicast_ether_addr(dest)) {
		atomic_inc(&flowmgr.pkt_ignore_mc_cnt);
		goto out;
	}

	in = in_device(skb);
	protonum = l4_protonum(skb, &l3proto, &ipmulti);
	/* Skip processing of L3 multicast packets with unicast mac addr */
	if (ipmulti) {
		atomic_inc(&flowmgr.pkt_ignore_ipmc_cnt);
		goto out;
	}

#ifdef CONFIG_BCM_FLOWMGR_TUNNEL
	if ((protonum == IPPROTO_IPIP) &&
	    flowmgr_is_feature_enabled(FLOW_F_DSLITE)) {
		flowmgr_tunnel_process(skb, in, out, protonum);
		goto out;
	}
	if ((protonum == IPPROTO_GRE) &&
	    flowmgr_is_feature_enabled(FLOW_F_GRE_ALL)) {
		flowmgr_tunnel_process(skb, in, out, protonum);
		goto out;
	}
#endif
	ct = (struct nf_conn *) skb_nfct(skb);
	if (!ct)
		goto out;
	if (!flowmgr_is_flow_allowed(l3proto, protonum))
		goto out;

	ct_offload = nf_conn_offload_find(ct);
	if (!ct_offload)
		goto out;

	if (flow_create_err_check(ct, ct_offload, nfctinfo(skb)))
		goto out;

	if (!in) {
		/* Packet originated from Host */
		if (!skb->peeked) {
			spin_lock(&flowmgr.lock);
			new_offload(skb, out, out, ct,
				    ct_offload, protonum);
			spin_unlock(&flowmgr.lock);
		}
		atomic_inc(&flowmgr.pkt_ignore_null_in_cnt);
		goto out;
	}
	if (!in->group) {
		/* Packet originated from non accelerated device like USB */
		if (!skb->peeked) {
			spin_lock(&flowmgr.lock);
			new_offload(skb, out, out, ct,
				    ct_offload, protonum);
			spin_unlock(&flowmgr.lock);
		}
		atomic_inc(&flowmgr.pkt_ignore_nogroup_in_cnt);
		goto out;
	}
	/* Overwrite skb_iif so that it can be used later */
	skb->skb_iif = in->ifindex;

#ifdef CONFIG_BCM_FLOWMGR_TUNNEL
	if (flowmgr_is_feature_enabled(FLOW_F_MAPT))
		flowmgr_mapt_process(skb, ct, ct_offload, out);
#endif

	idev_type = BCM_NETDEVICE_GROUP_TYPE(in->group);

	if ((idev_type == BCM_NETDEVICE_GROUP_LAN) &&
	    (odev_type == BCM_NETDEVICE_GROUP_LAN))
		process_packet_lan2lan(skb, in, out, ct, ct_offload, protonum);
	else
		process_packet(skb, in, out, ct, ct_offload, protonum);
out:
	return BCM_NETHOOK_PASS;
}

static enum bcm_nethook_result bcm_nethook_tx(
	struct net_device *out, enum bcm_nethook_type type, void *buf)
{
	struct sk_buff *skb = (struct sk_buff *)buf;
	struct net_device *in = in_device(skb);
	enum ip_conntrack_info ctinfo;
	struct nf_conn *ct;
	struct nf_conn_offload *ct_offload;
	struct offload_info *offload;

	if (flowmgr_is_db_stats_update_allowed(in, out)) {
		flowmgr_db_update_tx_slow(out, eth_hdr(skb)->h_dest, 1,
				skb->len, ip_get_slow_dscp_bp(skb));

		ct = nf_ct_get(skb, &ctinfo);
		if (in && skb_mac_header_was_set(skb) && ct) {
			ct_offload = nf_conn_offload_find(ct);
			if (ct_offload) {
				offload = &ct_offload->info[CTINFO2DIR(ctinfo)];
				flowmgr_db_update_rx_slow(in,
					offload->ehb.h_source, 1, skb->len,
					ip_get_slow_dscp_bp(skb));
			}
		}
	}

	if (!flowmgr_is_dev_tunnel(out))
		flowmgr_update_dev_slow_stats(out, 1, skb->len, 0,
				ip_get_slow_dscp_bp(skb));

	return flowmgr_tx(out, type, buf);
}

static inline
unsigned int flowmgr_nfbr_forward(void *priv,
				  struct sk_buff *skb,
				  const struct nf_hook_state *state)
{
	enum ip_conntrack_info ctinfo;
	struct nf_conn *ct;
	struct nf_conn_offload *ct_offload;
	struct offload_info *ct_offload_info;
	int pf = 0, ret;
	struct nf_hook_state s = *state;
	struct net_device *in = state->in;
	struct net_device *out = state->out;

	if (!flowmgr.enable)
		return NF_ACCEPT;

	if (!in || !out)
		return NF_ACCEPT;

	if (!flowmgr.enable_br_conntrack_call)
		return NF_ACCEPT;

	if (skb->pkt_type != PACKET_OTHERHOST &&
		skb->pkt_type != PACKET_HOST)
		return NF_ACCEPT;

	if (skb->protocol == htons(ETH_P_IP)) {
		pf = PF_INET;
		if ((ip_hdr(skb)->protocol != IPPROTO_TCP) &&
		    (ip_hdr(skb)->protocol != IPPROTO_UDP) &&
		    (ip_hdr(skb)->protocol != IPPROTO_ESP) &&
		    (ip_hdr(skb)->protocol != IPPROTO_AH))
			return NF_ACCEPT;
	}

	if (skb->protocol == htons(ETH_P_IPV6)) {
		pf = PF_INET6;
		if ((ipv6_hdr(skb)->nexthdr != IPPROTO_TCP) &&
		    (ipv6_hdr(skb)->nexthdr != IPPROTO_UDP) &&
		    (ipv6_hdr(skb)->nexthdr != IPPROTO_ESP) &&
		    (ipv6_hdr(skb)->nexthdr != IPPROTO_AH))
			return NF_ACCEPT;
	}

	if (!pf)
		return NF_ACCEPT;

	s.pf = pf;
	ret = nf_conntrack_in(skb, &s);
	nf_conntrack_confirm(skb);
	ct = nf_ct_get(skb, &ctinfo);
	if (!ct) {
		net_dbg_ratelimited("flowmgr_nfbr_forward: Error %d creating conntrack %d\n",
			 ret, pf);
		return NF_ACCEPT;
	}
	ct_offload = nf_conn_offload_find(ct);
	if (!ct_offload)
		return NF_ACCEPT;

	ct_offload_info = &ct_offload->info[CTINFO2DIR(nfctinfo(skb))];

	if (!ct_offload_info->nf_bridge)
		ct_offload_info->nf_bridge = kzalloc(sizeof(struct nf_bridge_info),
						     GFP_ATOMIC);

	if (!ct_offload_info->nf_bridge)
		return NF_ACCEPT;

	ct_offload_info->nf_bridge->bridged = 1;
	ct_offload_info->nf_bridge->physindev = (struct net_device *) in;
	ct_offload_info->nf_bridge->physoutdev = (struct net_device *) out;
	if (!ct_offload_info->vlan_untag) {
		if (netif_is_bond_master(in)) {
			struct bonding *bond = netdev_priv(in);
			struct slave *slave;
			struct list_head *iter;
			bond_for_each_slave(bond, slave, iter) {
				if (is_vlan_dev(slave->dev) && skb->dev_in == vlan_dev_real_dev(slave->dev)) {
					ct_offload_info->vlan_untag = 1;
					break;
				}
			}
		} else
			ct_offload_info->vlan_untag = flowmgr_get_vlan_id(ct_offload_info->nf_bridge->physindev) ? 1 : 0;
	}
	if (!ct_offload_info->vlan_id)
		ct_offload_info->vlan_id = flowmgr_get_vlan_id(out);

	return NF_ACCEPT;
}

static struct net_device *get_real_dev(struct net_device *in)
{
	if (is_vlan_dev(in))
		return vlan_dev_real_dev(in);
	return in;
}

static inline
unsigned int flowmgr_nfbr_pre_routing(void *priv,
				      struct sk_buff *skb,
				      const struct nf_hook_state *state)
{
	struct net_device *dev_in = get_real_dev((struct net_device *) state->in);
	int idev_type = BCM_NETDEVICE_GROUP_TYPE(dev_in->group);

	if (!flowmgr.enable)
		return NF_ACCEPT;

	if (!flowmgr_is_dev_tunnel((struct net_device *)dev_in))
		return NF_ACCEPT;

	if ((dev_in == state->in) && (idev_type == BCM_NETDEVICE_GROUP_LAN))
		skb->vlan_proto = 0;

	skb->dev_in = (struct net_device *) dev_in;
	return NF_ACCEPT;
}

#if !FLOWMGR_USE_BCM_NETHOOKS
static inline
unsigned int flowmgr_nfbr_post_routing(void *priv,
				       struct sk_buff *skb,
				       const struct nf_hook_state *state)
{
	if (!skb->peeked)
		flowmgr_tx((struct net_device *)skb->dev,
			0, (void *)skb);
	return NF_ACCEPT;
}
#endif

static inline
unsigned int flowmgr_nf_forwarding(void *priv,
				  struct sk_buff *skb,
				  const struct nf_hook_state *state)
{
	struct net_device *in = state->in;
	enum ip_conntrack_info ctinfo;
	struct nf_conn *ct;
	struct nf_conn_offload *ct_offload;
	struct offload_info *ct_offload_info;

	if (!flowmgr.enable)
		return NF_ACCEPT;

	ct = nf_ct_get(skb, &ctinfo);
	if (!ct)
		return NF_ACCEPT;

	ct_offload = nf_conn_offload_find(ct);
	if (!ct_offload)
		return NF_ACCEPT;

	ct_offload_info = &ct_offload->info[CTINFO2DIR(nfctinfo(skb))];

	update_vlan_info(skb, ct_offload_info);

	if (!flowmgr_is_dev_ip_tunnel((struct net_device *)in))
		return NF_ACCEPT;

	skb->dev_in = (struct net_device *) in;
	return NF_ACCEPT;
}

static struct sock *
xt_socket_get_sock_v4(struct net *net, const u8 protocol,
		      const __be32 saddr, const __be32 daddr,
		      const __be16 sport, const __be16 dport,
		      struct net_device *in)
{
	struct sock *sk = NULL;
	bool refcounted = 0;
	switch (protocol) {
	case IPPROTO_TCP:
		sk = __inet_lookup(net, &tcp_hashinfo,
				   NULL, 0, saddr, sport, daddr, dport,
				   in->ifindex, 0, &refcounted);
		if (refcounted && sk)
			sock_gen_put(sk);
		break;
	case IPPROTO_UDP:
		sk = udp4_lib_lookup(net, saddr, sport, daddr, dport,
				     in->ifindex);
		if (sk)
			sock_gen_put(sk);
		break;
	}
	return sk;
}

#define NOSOCK_DROP_ACTIVE_MASK 0x10
static inline
void flowmgr_nosock_drop_activate(int enable)
{
	if (!flowmgr.nosock_drop)
		return;
	if (enable && !(flowmgr.nosock_drop & NOSOCK_DROP_ACTIVE_MASK)) {
		flowmgr.nosock_drop |= NOSOCK_DROP_ACTIVE_MASK;
		pr_alert("FLOWMGR: Enable Dropping Host Packets from WAN without any socket listening\n");
	}
	if (!enable && (flowmgr.nosock_drop & NOSOCK_DROP_ACTIVE_MASK)) {
		flowmgr.nosock_drop &= ~NOSOCK_DROP_ACTIVE_MASK;
		pr_alert("FLOWMGR: Disable Dropping Host Packets from WAN without any socket listening\n");
	}
}

static inline
unsigned int flowmgr_local_in(void *priv,
			      struct sk_buff *skb,
			      const struct nf_hook_state *state)
{
	enum ip_conntrack_info ctinfo;
	struct nf_conn *ct = nf_ct_get(skb, &ctinfo);
	const struct iphdr *iph = ip_hdr(skb);
	struct udphdr _hdr, *hp = NULL;
	struct sock *sk = skb->sk;
	__be32 daddr, saddr;
	__be16 dport, sport;
	u8 uninitialized_var(protocol);
	struct net_device *in_skb = NULL;
	int igroup;
	int igroup_type;
	struct net_device *in = state->in;
	struct ethhdr *eh = (struct ethhdr *) eth_hdr(skb);

	in_skb = in_device(skb);

	if (!in_skb)
		in_skb = in;

	if (!in_skb->group)
		return NF_ACCEPT;

	if (flowmgr.inf_db_info != 0)
		flowmgr_db_update_rx_slow(in_skb, eh->h_source, 1,
				skb->len+ETH_HLEN, ip_get_slow_dscp_bp(skb));

	if (!(flowmgr.nosock_drop & NOSOCK_DROP_ACTIVE_MASK))
		return NF_ACCEPT;

	if (!ct)
		return NF_ACCEPT;

	if (ctinfo == IP_CT_UNTRACKED)
		return NF_ACCEPT;

	igroup = in_skb->group;

	igroup_type = BCM_NETDEVICE_GROUP_TYPE(igroup);
	if (igroup_type != BCM_NETDEVICE_GROUP_WAN)
		return NF_ACCEPT;

	if (!netif_running(in_skb) || !netif_oper_up(in_skb))
		return NF_ACCEPT;

	if (skb->pkt_type == PACKET_MULTICAST)
		return NF_ACCEPT;

	if (iph->protocol != IPPROTO_UDP && iph->protocol != IPPROTO_TCP)
		return NF_ACCEPT;

	hp = skb_header_pointer(skb, ip_hdrlen(skb),
				sizeof(_hdr), &_hdr);
	if (hp == NULL)
		return NF_ACCEPT;

	protocol = iph->protocol;
	saddr = iph->saddr;
	sport = hp->source;
	daddr = iph->daddr;
	dport = hp->dest;

	sk = xt_socket_get_sock_v4(dev_net(skb->dev), protocol,
				   saddr, daddr, sport, dport,
				   in);

	if (!sk) {
		pr_debug("No socket for protocol %d sip %pI4 dip %pI4 sport %d dport %d pkt_type %d ct_status 0x%lx\n",
			 protocol, &saddr, &daddr, ntohs(sport), ntohs(dport), skb->pkt_type,
			 ct->status);
		if (nf_ct_is_confirmed(ct)) {
			kill_ct(ct);
			atomic_inc(&flowmgr.pkt_nosock_cnt);
		}
	}

	return NF_ACCEPT;
}

static struct nf_hook_ops flowmgr_nf_ops[] __read_mostly = {
#if !FLOWMGR_USE_BCM_NETHOOKS
	{
		.pf       = NFPROTO_BRIDGE,
		.priority = INT_MAX,
		.hooknum  = NF_BR_POST_ROUTING,
		.hook     = flowmgr_nfbr_post_routing,
	},
#endif
	{
		.pf       = NFPROTO_BRIDGE,
		.priority = INT_MAX,
		.hooknum  = NF_BR_FORWARD,
		.hook     = flowmgr_nfbr_forward,
	},
	{
		.pf       = NFPROTO_BRIDGE,
		.priority = INT_MAX,
		.hooknum  = NF_BR_PRE_ROUTING,
		.hook     = flowmgr_nfbr_pre_routing,
	},
	{
		.pf       = NFPROTO_IPV4,
		.priority = INT_MAX,
		.hooknum  = NF_INET_FORWARD,
		.hook     = flowmgr_nf_forwarding,
	},
	{
		.pf       = NFPROTO_IPV4,
		.priority = INT_MAX,
		.hooknum  = NF_INET_LOCAL_IN,
		.hook     = flowmgr_local_in,
	},
	{
		.pf       = NFPROTO_IPV6,
		.priority = INT_MAX,
		.hooknum  = NF_INET_FORWARD,
		.hook     = flowmgr_nf_forwarding,
	},
};

#undef CONFIG_NF_CONNTRACK_EVENTS
#ifdef CONFIG_NF_CONNTRACK_EVENTS
static inline
int conntrack_event_handler(unsigned int events, struct nf_ct_event *item)
{
	struct nf_conn *ct = item->ct;
	struct nf_conn_offload *ct_offload = nf_conn_offload_find(ct);

	/* ignore our fake conntrack entry */
	if (ctinfo == IP_CT_UNTRACKED)
		return 0;

	if (!ct_offload ||
		((ct_offload_orig.ctinfo != -1) &&
		 (ct_offload_repl.ctinfo != -1)))
		return 0;

	if (events & (1 << IPCT_DESTROY)) {
		pr_debug(
		   "FLOWMGR CT Events(%x %x): Delete Flow M(%X) E(%X) S(%lX)\n",
		   item->portid, item->report,
		   ct->mark, events, ct->status);
	} else  if (events & ((1 << IPCT_NEW) | (1 << IPCT_RELATED))) {
		pr_debug(
		   "FLOWMGR CT Events(%x %x): Create Flow M(%X) E(%X) S(%lX)\n",
		   item->portid, item->report,
		   ct->mark, events, ct->status);
	} /*else {
		pr_debug(
		   "FLOWMGR CT Events(%x %x): Update Flow M(%X) E(%X) S(%lX)\n",
		   item->portid, item->report,
		   ct->mark, events, ct->status);
	}*/

	return 0;
}
#endif

int flowmgr_update_dev_fast_dscp_stats(struct net_device *dev,
					u32 packets, u32 bytes,
					int rx, u8 dscp)
{
	struct pcpu_sw_netstats *dscp_tstats = NULL;
	struct bcm_nethooks *nethooks = bcm_nethooks_priv();

	if (dscp >= DSCP_MAX_LIMIT) {
		return -EINVAL;
	}
	if (nethooks->sig == BCM_NETHOOKS_SIG && nethooks->offload_stats.dscp_fast[dscp]) {
		dscp_tstats = this_cpu_ptr(nethooks->offload_stats.dscp_fast[dscp]);
	} else {
		return -ENOMEM;
	}

	if (dscp_tstats) {
		u64_stats_update_begin(&dscp_tstats->syncp);
		if (rx) {
			dscp_tstats->rx_bytes += bytes;
			dscp_tstats->rx_packets += packets;
		} else {
			dscp_tstats->tx_bytes += bytes;
			dscp_tstats->tx_packets += packets;
		}
		u64_stats_update_end(&dscp_tstats->syncp);
	}
	return 0;
}

int flowmgr_update_dev_fast_stats(struct net_device *dev,
				  u32 packets, u32 bytes,
				  int rx, u8 dscp)
{
	struct pcpu_sw_netstats *tstats = NULL;
	struct bcm_nethooks *nethooks = bcm_nethooks_priv();

	if (dscp >= DSCP_MAX_LIMIT) {
		return -EINVAL;
	}
	if (dev->tstats) {
		tstats = this_cpu_ptr(dev->tstats);
	} else if (nethooks->sig == BCM_NETHOOKS_SIG && nethooks->offload_stats.fast) {
		tstats = this_cpu_ptr(nethooks->offload_stats.fast);
	} else {
		return -ENOMEM;
	}

	if (tstats) {
		u64_stats_update_begin(&tstats->syncp);
		if (rx) {
			tstats->rx_bytes += bytes;
			tstats->rx_packets += packets;
		} else {
			tstats->tx_bytes += bytes;
			tstats->tx_packets += packets;
		}
		u64_stats_update_end(&tstats->syncp);
	}
	return 0;
}

int flowmgr_update_dev_stats(struct net_device *dev,
			     u32 packets, u32 bytes,
			     int rx, u8 dscp)
{
	int ret = 0;

	if (dscp >= DSCP_MAX_LIMIT) {
		return -EINVAL;
	}
	ret = flowmgr_update_dev_fast_stats(dev, packets, bytes, rx, dscp);
	if (ret < 0)
		pr_debug("FLOWMGR Dev Fast stats update failure\n");
	ret = flowmgr_update_dev_fast_dscp_stats(dev, packets, bytes, rx, dscp);
	if (ret < 0)
		pr_debug("FLOWMGR Dev DSCP Fast stats update failure\n");
	return ret;
}

int flowmgr_update_dev_slow_stats(struct net_device *dev,
			     u32 packets, u32 bytes,
			     int rx, u8 dscp)
{
	struct pcpu_sw_netstats *dscp_tstats = NULL;
	struct bcm_nethooks *nethooks = bcm_nethooks_priv();

	if (dscp >= DSCP_MAX_LIMIT) {
		return -EINVAL;
	}
	if (nethooks->sig == BCM_NETHOOKS_SIG && nethooks->offload_stats.dscp_slow[dscp]) {
		dscp_tstats = this_cpu_ptr(nethooks->offload_stats.dscp_slow[dscp]);
	} else {
		return -ENOMEM;
	}

	if (dscp_tstats) {
		u64_stats_update_begin(&dscp_tstats->syncp);
		if (rx) {
			dscp_tstats->rx_bytes += bytes;
			dscp_tstats->rx_packets += packets;
		} else {
			dscp_tstats->tx_bytes += bytes;
			dscp_tstats->tx_packets += packets;
		}
		u64_stats_update_end(&dscp_tstats->syncp);
	}
	if (nethooks->offload_stats.slow) {
		dscp_tstats = this_cpu_ptr(nethooks->offload_stats.slow);
	} else {
		return -ENOMEM;
	}
	if (dscp_tstats) {
		u64_stats_update_begin(&dscp_tstats->syncp);
		if (rx) {
			dscp_tstats->rx_bytes += bytes;
			dscp_tstats->rx_packets += packets;
		} else {
			dscp_tstats->tx_bytes += bytes;
			dscp_tstats->tx_packets += packets;
		}
		u64_stats_update_end(&dscp_tstats->syncp);
	}
	return 0;
}

int flowmgr_dec_dev_slow_stats(struct net_device *dev,
			       u32 packets, u32 bytes,
			       int rx, u8 dscp)
{
	struct pcpu_sw_netstats *tstats = NULL;
	struct bcm_nethooks *nethooks = bcm_nethooks_priv();

	if (dev->tstats) {
		tstats = this_cpu_ptr(dev->tstats);
		if (!tstats)
			return -ENOMEM;
		u64_stats_update_begin(&tstats->syncp);
		if (rx) {
			tstats->rx_bytes -= bytes;
			tstats->rx_packets -= packets;
		} else {
			tstats->tx_bytes -= bytes;
			tstats->tx_packets -= packets;
		}
		u64_stats_update_end(&tstats->syncp);
		return 0;
	} else if (nethooks->sig == BCM_NETHOOKS_SIG && nethooks->offload_stats.slow) {
		tstats = this_cpu_ptr(nethooks->offload_stats.slow);
		if (!tstats)
			return -ENOMEM;
		u64_stats_update_begin(&tstats->syncp);
		if (rx) {
			tstats->rx_bytes -= bytes;
			tstats->rx_packets -= packets;
		} else {
			tstats->tx_bytes -= bytes;
			tstats->tx_packets -= packets;
		}
		u64_stats_update_end(&tstats->syncp);
		if (nethooks->offload_stats.dscp_slow[dscp]) {
			tstats = this_cpu_ptr(nethooks->offload_stats.dscp_slow[dscp]);
		} else {
			return -ENOMEM;
		}
		if (!tstats)
			return -ENOMEM;
		u64_stats_update_begin(&tstats->syncp);
		if (rx) {
			tstats->rx_bytes -= bytes;
			tstats->rx_packets -= packets;
		} else {
			tstats->tx_bytes -= bytes;
			tstats->tx_packets -= packets;
		}
		u64_stats_update_end(&tstats->syncp);
		return 0;
	}

	return -ENOMEM;
}

int flowmgr_update_all_stats(const struct nf_conn *ct,
			     struct offload_info *offload,
			     int direction,
			     u32 packets, u32 bytes)
{
	struct nf_conn_acct *acct;
	struct net_device *in, *out;
	struct net_device *tout;
	u32 packets_diff;
	u32 bytes_diff;
	unsigned int byte_adj = 0;
	struct flow_tunnel_params params;

	if (offload->expected)
		return 0;

	spin_lock_bh(&flowmgr.lock_counter);
	if (packets < offload->packets[0]) {
		/* Fap Counter Overflow */
		packets_diff = U32_MAX - offload->packets[0];
		packets_diff += packets;
	} else {
		packets_diff = packets - offload->packets[0];
	}
	if (bytes < offload->bytes) {
		/* Fap Counter Overflow */
		bytes_diff = U32_MAX - offload->bytes;
		bytes_diff += bytes;
	} else {
		bytes_diff = bytes - offload->bytes;
	}
	in = __dev_get_by_index(&init_net, offload->iif);
	/* Update device stats*/
	if (flowmgr_is_dev_tunnel(in)) {
		flowmgr_update_dev_fast_stats(in, packets_diff,
					 bytes_diff, 1, offload->dscp_new);
		tout = __dev_get_by_index(&init_net,
			flowmgr_tunnel_out_ifindex(in, &params, NULL));
		if (tout)
			flowmgr_update_dev_stats(tout,
					 packets_diff,
					 bytes_diff, 1, offload->dscp_new);
	} else if (in) {
		/* Do not include BRCM tag and Ethernet CRC bytes in rx byte count */
		if (BCM_NETDEVICE_GROUP_TYPE(in->group) == BCM_NETDEVICE_GROUP_LAN) {
			struct dqnet_netdev *ndev = netdev_priv(in);
			byte_adj = dqnet_get_brcm_tag_len(ndev);
			if (byte_adj)
				byte_adj += 4;
			bytes_diff -= packets_diff*byte_adj;
		}
		flowmgr_update_dev_stats(in, packets_diff,
					 bytes_diff, 1, offload->dscp_new);
	}

	out = __dev_get_by_index(&init_net, offload->oif);
	if (flowmgr_is_dev_tunnel(out)) {
		flowmgr_update_dev_fast_stats(out, packets_diff,
					 bytes_diff, 0, offload->dscp_new);
		tout = __dev_get_by_index(&init_net,
			flowmgr_tunnel_out_ifindex(out, &params, NULL));
		if (tout)
			flowmgr_update_dev_stats(tout,
					 packets_diff,
					 bytes_diff, 0, offload->dscp_new);
	} else if (out)
		flowmgr_update_dev_stats(out, packets_diff,
					 bytes_diff, 0, offload->dscp_new);

	/* Update MAC stats */
	if (flowmgr_is_db_stats_update_allowed(in, out)) {
		if (offload->odb) {
			flowmgr_db_update_txstats_fast(offload->odb,
							packets_diff,
							bytes_diff,
							offload->dscp_new);
		}
		if (offload->idb) {
			flowmgr_db_update_rxstats_fast(offload->idb,
						   packets_diff,
						   bytes_diff,
						   offload->dscp_new);
		}
	}
	/* Update nf connection acc stats*/
	acct = nf_conn_acct_find(ct);
	if (acct) {
		struct nf_conn_counter *nf_counter;
		nf_counter = &(acct->counter[direction]);
		atomic64_add(packets_diff,
			     &nf_counter->packets);
		atomic64_add(bytes_diff,
			     &nf_counter->bytes);
	}
	offload->packets[0] = packets;
	offload->bytes = bytes;
	spin_unlock_bh(&flowmgr.lock_counter);
	return 0;
}


int flowmgr_update_flow_stats(const struct nf_conn *ct, int direction, int update)
{
	struct nf_conn_offload *ct_offload = nf_conn_offload_find(ct);
	struct offload_info *offload;
	int flow_id;

	if (!ct_offload)
		return -1;
	offload = &ct_offload->info[direction];
	flow_id = offload->flow_id;
	if (flow_id != -1) {
		u32 packets = 0;
		u32 bytes = 0;
		u32 packets_diff;
		fap()->flow_get_counter(flow_id, &packets, &bytes, false);

		packets_diff = packets - offload->packets[0];
		if (!packets_diff)
			return -1;

		if (update)
			flowmgr_update_all_stats(ct, offload, direction,
						 packets, bytes);
		return 0;
	}
	return -1;
}

int flowmgr_check_flow_active(struct nf_conn *ct, int direction, int update)
{
	struct nf_conn_offload *ct_offload = nf_conn_offload_find(ct);
	struct offload_info *offload = &ct_offload->info[direction];
	int flow_id = offload->flow_id;

	if (flow_id != -1) {
		u32 packets = 0;
		u32 bytes = 0;
		u32 packets_diff;
		fap()->flow_get_counter(flow_id, &packets, &bytes, false);

		packets_diff = packets - offload->packets[1];
		if (!packets_diff)
			return -1;

		offload->packets[1] = packets;

		if (update)
			flowmgr_update_all_stats((const struct nf_conn *) ct, offload, direction,
						 packets, bytes);
		return 0;
	}
	return -1;
}

void update_offload_retry_timeout(struct nf_conn *ct)
{
	u_int8_t protonum;
	unsigned int *timeouts;
	const struct nf_conntrack_l4proto *l4proto;
	struct net *net = nf_ct_net(ct);

	protonum = nf_ct_protonum(ct);

	if (protonum == IPPROTO_TCP) {
		struct nf_tcp_net *tn = nf_tcp_pernet(net);

		l4proto = nf_ct_l4proto_find(protonum);
		timeouts = nf_ct_timeout_lookup(ct);
		if (!timeouts)
			timeouts = tn->timeouts;

		if (!timeouts)
			return;

		ct->timeout = nfct_time_stamp + timeouts[ct->proto.tcp.state];
	} else if (protonum == IPPROTO_UDP) {
		struct nf_udp_net *tn = nf_udp_pernet(net);

		l4proto = nf_ct_l4proto_find(protonum);
		timeouts = nf_ct_timeout_lookup(ct);
		if (!timeouts)
			timeouts = tn->timeouts;

		if (!timeouts)
			return;

		if (test_bit(IPS_SEEN_REPLY_BIT, &ct->status)) {
			ct->timeout = nfct_time_stamp +
				timeouts[UDP_CT_REPLIED];
		} else {
			ct->timeout = nfct_time_stamp +
				timeouts[UDP_CT_UNREPLIED];
		}
	} else if (protonum == IPPROTO_ESP) {
		struct nf_esp_net *tn = nf_esp_pernet(net);

		l4proto = nf_ct_l4proto_find(protonum);
		timeouts = nf_ct_timeout_lookup(ct);
		if (!timeouts)
			timeouts = tn->timeouts;

		if (!timeouts)
			return;

		if (test_bit(IPS_SEEN_REPLY_BIT, &ct->status)) {
			ct->timeout = nfct_time_stamp +
				timeouts[UDP_CT_REPLIED];
		} else {
			ct->timeout = nfct_time_stamp +
				timeouts[UDP_CT_UNREPLIED];
		}
	}
}

static inline
int conntrack_get_status(struct nf_conn *ct)
{
	struct nf_conn_offload *ct_offload = nf_conn_offload_find(ct);
	u_int8_t protonum;
	int ret = -1;

	if (!ct_offload)
		return __nf_ct_is_expired(ct);

	if ((ct_offload_orig.flow_id == -1) &&
	    (ct_offload_repl.flow_id == -1)) {
		/* Slowpath Flows */
#ifdef MASTER_TIMEOUT_LOGIC
		if (ct->master) {
			/* If connection has master then, check health of
			   master. If master is dying then accelerate timeout
			   to check if slave connection is still alive. */
			if (nf_ct_is_dying(ct->master) && !__nf_ct_is_expired(ct))
				ct->timeout = nfct_time_stamp +
				FAST_DEATH_TIMEOUT;
		} else if (atomic_read(&ct_offload->slavecnt)) {
			/* If slave connections are promoted,
			   keep master alive */
			ret = 0;
		}
#endif
		if (ret == 0) {
			return 0;
		} else {
			return __nf_ct_is_expired(ct);
		}
	}

	/* Fastpath flows */

	/* For TCP flow in close state, kill it */
	protonum = nf_ct_protonum(ct);
	if ((protonum == IPPROTO_TCP) &&
	    (ct->proto.tcp.state > TCP_CONNTRACK_CLOSE_WAIT))
		return __nf_ct_is_expired(ct);

	if (fap()) {
		if (!__nf_ct_is_expired(ct)) {
			if ((__s32)(ct_offload->check_status_timeout - nfct_time_stamp) > 0)
				return 0;
			ct_offload->check_status_timeout = nfct_time_stamp +
				flowmgr.check_status_timeout * HZ;
		}

		/* Check fap flow stats in original direction */
		ret = flowmgr_check_flow_active(ct, IP_CT_DIR_ORIGINAL,
						1);
		/* Check fap flow stats in reply direction */
		if (ret != 0)
			ret = flowmgr_check_flow_active(ct, IP_CT_DIR_REPLY,
							1);
#ifdef MASTER_TIMEOUT_LOGIC
		if (ct->master) {
			/* If connection has master then, check health of
			   master. If master is dying then accelerate timeout
			   to check if slave connection is still alive. */
			if ((ret == 0) && nf_ct_is_dying(ct->master))
				ct->timeout = nfct_time_stamp +
				FAST_DEATH_TIMEOUT;
		} else if (atomic_read(&ct_offload->slavecnt)) {
			/* If slave connections are promoted,
			   keep master alive */
			ret = 0;
		}
#endif
		if (ct_offload_orig.force_del || ct_offload_orig.force_del)
			ret = -1;
	}

	if (ret == 0) {
		update_offload_retry_timeout(ct);
		return 0;
	} else {
		return __nf_ct_is_expired(ct);
	}
}

static int
flowmgr_tunnel_start_xmit(struct sk_buff *skb,
			  struct net_device *dev)
{
	struct flowmgr_tdb_entry *tdb;

	tdb = flowmgr_tdb_find(dev->ifindex);
	if (!tdb) {
		pr_err("flowmgr_tunnel_start_xmit: Error in finding tunnel device\n");
		return NETDEV_TX_OK;
	}

	if (flowmgr_is_dev_ip_tunnel(dev)) {
		skb_set_mac_header(skb, -ETH_HLEN);
		skb_reset_network_header(skb);
		skb_reset_transport_header(skb);
	}

	if (skb->peeked)
		return tdb->netdev_ops->ndo_start_xmit(skb, dev);

	if (!flowmgr_is_dev_tunnel(skb->dev_in) &&
	    BCM_NETDEVICE_GROUP_TYPE(dev->group) ==
	    BCM_NETDEVICE_GROUP_LAN) {
		if (flowmgr_is_dev_gretap_tunnel(dev))
			db_check(skb, dev, 0);
		if (flowmgr_is_dev_ip6gretap_tunnel(dev))
			db_check(skb, dev, 0);
	}

	if (skb->dev_in && skb->dev_in->group) {
	/* Update Offload Info */
		struct nf_conn *ct;
		struct nf_conn_offload *ct_offload = NULL;
		enum ip_conntrack_info ctinfo;
		ct = nf_ct_get(skb, &ctinfo);
		if (ct) {
			ct_offload = nf_conn_offload_find(ct);
			if (ct_offload) {
				if (CTINFO2DIR(nfctinfo(skb)) ==
				    IP_CT_DIR_ORIGINAL) {
					ct_offload_orig.encap_tid = tdb->id;
					ct_offload_repl.decap_tid = tdb->id;
				} else {
					ct_offload_repl.encap_tid = tdb->id;
					ct_offload_orig.decap_tid = tdb->id;
				}
			}
		}
		flowmgr_tunnel_insert_v4link(dev, skb, ct_offload);
		flowmgr_tunnel_insert_v6link(dev, skb, ct_offload);
		bcm_nethook_tx(dev, 0, (void *)skb);
		skb->dev_in = dev;
	}

	return tdb->netdev_ops->ndo_start_xmit(skb, dev);
}

/* Override netdev ops for tunnel device */
void flowmgr_tunnel_override_netdev_ops(struct net_device *dev)
{
	struct net_device_ops *netdev_ops;
	struct flowmgr_tdb_entry *tdb;
	tdb = flowmgr_tdb_find(dev->ifindex);
	if (tdb) {
		netdev_ops = kmemdup(dev->netdev_ops,
				     sizeof(struct net_device_ops),
				     GFP_KERNEL);
		netdev_ops->ndo_start_xmit = flowmgr_tunnel_start_xmit;
		tdb->netdev_ops = dev->netdev_ops;
		dev->netdev_ops = netdev_ops;
	}
}

/* Restore netdev ops for tunnel device */
void flowmgr_tunnel_restore_netdev_ops(struct net_device *dev)
{
	struct flowmgr_tdb_entry *tdb;
	tdb = flowmgr_tdb_find(dev->ifindex);
	if (tdb) {
		if (tdb->netdev_ops != dev->netdev_ops) {
			kfree(dev->netdev_ops);
			dev->netdev_ops = tdb->netdev_ops;
		}
	}
}

int flowmgr_register_netdevice(struct net_device *dev, int type)
{
	if (BCM_NETDEVICE_GROUP_TYPE(dev->group) &&
	    (BCM_NETDEVICE_GROUP_CAT(dev->group) ==
			   BCM_NETDEVICE_GROUP_CAT_PHY)) {
#if FLOWMGR_USE_BCM_NETHOOKS
		bcm_nethook_register_hook(dev, BCM_NETHOOK_TX_SKB,
					  TX_SKB_PRIO_FLOWMGR, "FlowMgr",
					  bcm_nethook_tx);
		bcm_nethook_enable_hook(dev, BCM_NETHOOK_TX_SKB,
					bcm_nethook_tx, true);
		bcm_nethook_register_hook(dev, BCM_NETHOOK_RX_SKB,
					  RX_SKB_PRIO_FLOWMGR, "FlowMgr",
					  bcm_nethook_rx);
		bcm_nethook_enable_hook(dev, BCM_NETHOOK_RX_SKB,
					bcm_nethook_rx, true);
#endif
		if (BCM_NETDEVICE_GROUP_TYPE(dev->group) ==
		    BCM_NETDEVICE_GROUP_WAN) {
			flowmgr_set_wandev_by_name(dev->name);
			if (netif_running(dev) && netif_oper_up(dev))
				flowmgr_nosock_drop_activate(1);
		}
#ifdef CONFIG_BCM_FLOWMGR_MCAST
		bcm_nethook_dev_mcast_init(dev, flowmgr_mdb_update_dev_mcast_stats);
#endif
		flowmgr_db_ifenable(dev, 1);
	} else {

#ifdef CONFIG_BCM_FLOWMGR_TUNNEL
		if (!flowmgr_is_tunnel_allowed(dev, type))
			return -1;

		flowmgr_tunnel_insert(dev);
		flowmgr_tunnel_override_netdev_ops(dev);
		flowmgr_db_ifenable(dev, 1);
#endif
	}

	return 0;
}
int flowmgr_unregister_netdevice(struct net_device *dev)
{
	if (BCM_NETDEVICE_GROUP_TYPE(dev->group) &&
	    (BCM_NETDEVICE_GROUP_CAT(dev->group) ==
			   BCM_NETDEVICE_GROUP_CAT_PHY)) {
#ifdef CONFIG_BCM_FLOWMGR_MCAST
		bcm_nethook_dev_mcast_fini(dev);
#endif
#if FLOWMGR_USE_BCM_NETHOOKS
		bcm_nethook_unregister_hook(dev, BCM_NETHOOK_TX_SKB,
					    bcm_nethook_tx);
		bcm_nethook_unregister_hook(dev, BCM_NETHOOK_RX_SKB,
					    bcm_nethook_rx);
#endif
		if (BCM_NETDEVICE_GROUP_TYPE(dev->group) ==
		    BCM_NETDEVICE_GROUP_WAN) {
			flowmgr_nosock_drop_activate(0);
		}
		flowmgr_db_ifenable(dev, 0);
	} else {
#ifdef CONFIG_BCM_FLOWMGR_TUNNEL

		if (!flowmgr_is_dev_tunnel(dev))
			return -1;

		flowmgr_conntrack_clean_by_inf(dev_net(dev), dev);
		flowmgr_tunnel_restore_netdev_ops(dev);
		flowmgr_tunnel_delete(dev);
		flowmgr_db_ifenable(dev, 0);
#endif
	}
	return 0;
}

void flowmgr_netdevice_show(struct seq_file *s)
{
	int i;
	pr_seq(s, "Network Interface List:\n");
	for (i = 0; i < MAX_INF; i += 2) {
		if (net_interface_list[i] && net_interface_list[i+1])
			pr_seq(s, " %-10s  %-10s\n", net_interface_list[i],
				net_interface_list[i+1]);
		else if (net_interface_list[i])
			pr_seq(s, " %-10s\n", net_interface_list[i]);
		else if (net_interface_list[i+1])
			pr_seq(s, " %-10s\n", net_interface_list[i+1]);
	}
	pr_seq(s, "----------------------\n");
}

char **flowmgr_netdevice_list(void)
{
	return net_interface_list;
}
EXPORT_SYMBOL(flowmgr_netdevice_list);

int flowmgr_netdevice_type2num(char *type)
{
	if (!type)
		return 0;

	if (strcasecmp(type, "wan") == 0) {
		return BCM_NETDEVICE_GROUP_WAN;
	} else if (strcasecmp(type, "lan") == 0) {
		return BCM_NETDEVICE_GROUP_LAN;
	}
	return 0;
}

void flowmgr_netdevice_set_type(struct net_device *dev, int type)
{
	if (type == BCM_NETDEVICE_GROUP_WAN) {
		if (BCM_NETDEVICE_GROUP_TYPE(dev->group) ==
		     BCM_NETDEVICE_GROUP_LAN) {
			pr_debug("FLOWMGR: Netdevice %s type changed from lan to wan\n", dev->name);
			dev->group = BCM_NETDEVICE_GROUP_TYPE_CLR(dev->group) |
				BCM_NETDEVICE_GROUP_WAN;
		}
	} else if (type == BCM_NETDEVICE_GROUP_LAN) {
		if (BCM_NETDEVICE_GROUP_TYPE(dev->group) ==
		     BCM_NETDEVICE_GROUP_WAN) {
			pr_debug("FLOWMGR: Netdevice %s type changed from wan to lan\n", dev->name);
			dev->group = BCM_NETDEVICE_GROUP_TYPE_CLR(dev->group) |
				BCM_NETDEVICE_GROUP_LAN;
		}
	}
}

int flowmgr_netdevice_add(char *name, char *type)
{
	int i;
	struct net_device *dev;
	int typenum = flowmgr_netdevice_type2num(type);
	dev = __dev_get_by_name(&init_net, name);
	if (!dev)
		return -1;
	/* Check if already in list */
	if (is_netdevice_registered(name)) {
		if (typenum)
			flowmgr_netdevice_set_type(dev, typenum);
		return 0;
	}
	for (i = 0; i < MAX_INF; i++) {
		if (net_interface_list[i] == NULL) {
			if (flowmgr_register_netdevice(dev, typenum) == 0) {
				net_interface_list[i] = dev->name;
				if ((i + 1) > interfaces_c)
					interfaces_c = i + 1;
				if (typenum)
					flowmgr_netdevice_set_type(dev, typenum);
			}
			return 0;
		}
	}
	return -1;
}

int flowmgr_netdevice_del(char *name)
{
	int i;

	for (i = 0; i < MAX_INF; i++) {
		if (net_interface_list[i] &&
			strncmp(net_interface_list[i], name, IFNAMSIZ) == 0) {
			struct net_device *dev;
			dev = __dev_get_by_name(&init_net, name);
			if (!dev)
				continue;
			flowmgr_conntrack_clean_by_inf(dev_net(dev), dev);
#ifdef CONFIG_BCM_FLOWMGR_MCAST
			flowmgr_mdb_delete_by_out_port(dev);
#endif
			flowmgr_unregister_netdevice(dev);
			net_interface_list[i] = NULL;
			return 0;
		}
	}
	return -1;
}

int flowmgr_netdevice_del_internal(struct net_device *dev)
{
	int i;

	for (i = 0; i < MAX_INF; i++) {
		if (net_interface_list[i] &&
			strncmp(net_interface_list[i], dev->name,
					IFNAMSIZ) == 0) {
			flowmgr_unregister_netdevice(dev);
			net_interface_list[i] = NULL;
			return 0;
		}
	}
	return -1;
}

static inline
int netevent_callback(struct notifier_block *self,
		      unsigned long event, void *ctx)
{
	/* Should cache APR entries here ?
	if (event == NETEVENT_NEIGH_UPDATE) {
		struct neighbour *neigh = ctx;
		if (neigh->nud_state & NUD_CONNECTED) {
		}
	}
	*/
	return 0;
}

static inline
int netdevevent_callback(struct notifier_block *self,
			 unsigned long event, void *ctx)
{
	struct net_device *dev = netdev_notifier_info_to_dev(ctx);

	if (!BCM_NETDEVICE_GROUP_TYPE(dev->group))
		return 0;

	pr_debug(
	   "FLOWMGR: Net device %s event %ld\n",
	   dev->name, event);

	if (event == NETDEV_UNREGISTER) {
		flowmgr_conntrack_clean_by_inf(dev_net(dev), dev);
#ifdef CONFIG_BCM_FLOWMGR_MCAST
		flowmgr_mdb_delete_by_out_port(dev);
#endif
		flowmgr_netdevice_del_internal(dev);
		flowmgr_db_delete_by_port(dev->ifindex);
	} else if (event == NETDEV_CHANGE) {
		if (!netif_carrier_ok(dev)) {
			pr_alert(
			   "FLOWMGR: Net device %s down - Cleaning flows\n",
			   dev->name);
			flowmgr_conntrack_clean_by_inf(dev_net(dev), dev);
#ifdef CONFIG_BCM_FLOWMGR_MCAST
			flowmgr_mdb_delete_by_out_port(dev);
#endif
			if (BCM_NETDEVICE_GROUP_TYPE(dev->group) ==
			     BCM_NETDEVICE_GROUP_WAN) {
				flowmgr_nosock_drop_activate(0);
			}
		} else if (BCM_NETDEVICE_GROUP_TYPE(dev->group) ==
			    BCM_NETDEVICE_GROUP_WAN) {
			flowmgr_nosock_drop_activate(1);
		}
	} else if (event == NETDEV_UP) {
		if (BCM_NETDEVICE_GROUP_TYPE(dev->group) ==
		     BCM_NETDEVICE_GROUP_WAN) {
			flowmgr_nosock_drop_activate(1);
		}
	} else if (event == NETDEV_GOING_DOWN) {
		pr_alert(
		   "FLOWMGR: Net device %s down - Cleaning flows\n",
		   dev->name);
		flowmgr_conntrack_clean_by_inf(dev_net(dev), dev);
#ifdef CONFIG_BCM_FLOWMGR_MCAST
		flowmgr_mdb_delete_by_out_port(dev);
#endif
		if (BCM_NETDEVICE_GROUP_TYPE(dev->group) ==
		     BCM_NETDEVICE_GROUP_WAN) {
			flowmgr_nosock_drop_activate(0);
		}
	}

	return 0;
}

static inline
int flowmgr_ovs_hook_in(struct sk_buff *skb, struct net_device *in, struct net_device *out)
{
	if (!skb || !in) {
		pr_err("%s: Invalid input parameters\n", __func__);
		return -EINVAL;
	}

	if (!skb->dev_in)
		return 0;

	if (!flowmgr_is_dev_veth(in) && !netif_is_bond_master(in)) {
		/* Update input device info if it's not updated in the other module */
		if (skb->dev_in && skb->dev_in->ifindex != in->ifindex) {
			if (is_vlan_dev(in))
				skb->dev_in = vlan_dev_real_dev(in);
			else
				skb->dev_in = in;
		}

		if (flowmgr_is_dev_tunnel(skb->dev_in) && (skb->dev_in == in) &&
	    	(BCM_NETDEVICE_GROUP_TYPE(skb->dev_in->group) == BCM_NETDEVICE_GROUP_LAN))
			skb->vlan_proto = 0;
	} else {
		/* For bonding or pseudo-bridge mode, save real input to skb mark and restore it later in hook_ovs_cmd.
	    	If skb mark is in use, print warning here and user can look for unused bits to save it */
		if (unlikely(skb->mark))
			pr_warning("%s: skb mark 0x%x is in use\n", __func__, skb->mark);
		skb->mark = skb->dev_in->ifindex;
	}

	return 0;
}

static inline
int flowmgr_ovs_hook_cmd(struct sk_buff *skb, struct net_device *in, struct net_device *out)
{
	if (!skb || !in) {
		pr_err("%s: Invalid input parameters\n", __func__);
		return -EINVAL;
	}

	/* Restore input device info lost during user space process */
	if (is_vlan_dev(in))
		skb->dev_in = vlan_dev_real_dev(in);
	else if ((flowmgr_is_dev_veth(in) || netif_is_bond_master(in)) && skb->mark) {
		skb->dev_in = __dev_get_by_index(&init_net, skb->mark);
		skb->mark = 0;
	}
	else
		skb->dev_in = in;

	if (skb->dev_in)
		skb->skb_iif = skb->dev_in->ifindex;
	else
		return -EINVAL;

	return 0;
}

static inline
int flowmgr_ovs_hook_ct(struct sk_buff *skb, struct net_device *in, struct net_device *out)
{
	if (!skb) {
		pr_err("%s: Invalid input parameters\n", __func__);
		return -EINVAL;
	}

	if ((skb->protocol == htons(ETH_P_8021Q)) && (skb->vlan_proto != htons(ETH_P_8021Q))) {
		struct nf_conn *ct = (struct nf_conn *)skb_nfct(skb);
		struct nf_conn_offload *ct_offload;
		if (ct) {
			ct_offload = nf_conn_offload_find(ct);
			if (ct_offload)
				ct_offload->info[CTINFO2DIR(nfctinfo(skb))].vlan_untag = 1;
		}
	}

	return 0;
}

static inline
int flowmgr_ovs_hook_out(struct sk_buff *skb, struct net_device *in, struct net_device *out)
{
	enum ip_conntrack_info ctinfo;
	struct nf_conn *ct;
	struct nf_conn_offload *ct_offload;
	struct offload_info *ct_offload_info;
	int pf = 0, ret;

	if (!skb || !in || !out) {
		pr_err("%s: Invalid input parameters\n", __func__);
		return -EINVAL;
	}

	if (!netif_is_bond_master(out)) {
		struct net_device *real_dev = NULL;
		if (is_vlan_dev(out))
			real_dev = vlan_dev_real_dev(out);
		else
			real_dev = out;
		if (!is_netdevice_registered(real_dev->name))
			return 0;
	}

	if (skb->pkt_type != PACKET_OTHERHOST &&
		skb->pkt_type != PACKET_HOST)
		return 0;

	if (skb->protocol == htons(ETH_P_IP)) {
		pf = PF_INET;
		if ((ip_hdr(skb)->protocol != IPPROTO_TCP) &&
		    (ip_hdr(skb)->protocol != IPPROTO_UDP) &&
		    (ip_hdr(skb)->protocol != IPPROTO_ESP) &&
		    (ip_hdr(skb)->protocol != IPPROTO_AH))
			return 0;
	}

	if (skb->protocol == htons(ETH_P_IPV6)) {
		pf = PF_INET6;
		if ((ipv6_hdr(skb)->nexthdr != IPPROTO_TCP) &&
		    (ipv6_hdr(skb)->nexthdr != IPPROTO_UDP) &&
		    (ipv6_hdr(skb)->nexthdr != IPPROTO_ESP) &&
		    (ipv6_hdr(skb)->nexthdr != IPPROTO_AH))
			return 0;
	}

	if (!pf)
		return 0;

	ct = nf_ct_get(skb, &ctinfo);
	if (!ct) {
		struct nf_hook_state state = {
			.hook = NF_INET_FORWARD,
			.pf = pf,
			.net = dev_net(in),
		};
		ret = nf_conntrack_in(skb, &state);
		nf_conntrack_confirm(skb);
		ct = nf_ct_get(skb, &ctinfo);
		if (!ct) {
			net_dbg_ratelimited("hook_ovs_out: Error %d creating conntrack %d\n",
			 	ret, pf);
			return 0;
		}
	}

	ct_offload = nf_conn_offload_find(ct);
	if (!ct_offload)
		return 0;

	ct_offload_info = &ct_offload->info[CTINFO2DIR(nfctinfo(skb))];
	if (!ct_offload_info->nf_bridge)
		ct_offload_info->nf_bridge = kzalloc(sizeof(struct nf_bridge_info),
						     GFP_ATOMIC);
	else
		return 0;

	if (!ct_offload_info->nf_bridge)
		return 0;

	ct_offload_info->nf_bridge->bridged = 1;
	ct_offload_info->nf_bridge->physindev = (struct net_device *) in;
	ct_offload_info->nf_bridge->physoutdev = (struct net_device *) out;
	if (!ct_offload_info->vlan_untag) {
		if (netif_is_bond_master(in)) {
			struct bonding *bond = netdev_priv(in);
			struct slave *slave;
			struct list_head *iter;
			bond_for_each_slave(bond, slave, iter) {
				if (is_vlan_dev(slave->dev) && skb->dev_in == vlan_dev_real_dev(slave->dev)) {
					ct_offload_info->vlan_untag = 1;
					break;
				}
			}
		} else
			ct_offload_info->vlan_untag = flowmgr_get_vlan_id(ct_offload_info->nf_bridge->physindev) ? 1 : 0;
	}
	if (!ct_offload_info->vlan_id)
		ct_offload_info->vlan_id = flowmgr_get_vlan_id(out);

	return 0;
}

static flowmgr_ovs_hook_fn flowmgr_ovs_hook_table[OVS_HOOK_MAX] __read_mostly = {
	flowmgr_ovs_hook_in,
	flowmgr_ovs_hook_cmd,
	flowmgr_ovs_hook_ct,
	flowmgr_ovs_hook_out
};

int flowmgr_ovs_call_hook(unsigned int hook, struct sk_buff *skb, struct net_device *in, struct net_device *out)
{
	if (!flowmgr.enable)
		return 0;

	if (hook >= OVS_HOOK_MAX) {
		pr_err("%s: unsupported hook %d\n", __func__, hook);
		return -EINVAL;
	}

	return flowmgr_ovs_hook_table[hook](skb, in, out);
}

#ifdef CONFIG_NF_CONNTRACK_EVENTS
static struct nf_ct_event_notifier ctflowmgr_notifier __read_mostly = {
	.fcn = conntrack_event_handler,
};
#endif
static struct nf_ct_offload_ops ctflowmgr_ops __read_mostly = {
	.get_status = conntrack_get_status,
#ifdef CONFIG_NF_CONNTRACK_EVENTS
	.event_cb   = &ctflowmgr_notifier,
#endif
	.ovs_hook = flowmgr_ovs_call_hook,
};
static struct notifier_block nb_netevent = {
	.notifier_call = netevent_callback
};

static struct notifier_block nb_netdevevent = {
	.notifier_call = netdevevent_callback
};

void flowmgr_features_show(struct seq_file *s, char *name, int features)
{
	pr_seq(s, "%s: 0x%x\n", name, features);
	pr_seq(s, " ipv4: ");
	if (features & (FLOW_F_TCP4))
		pr_seq(s, "tcp ");
	if (features & (FLOW_F_UDP4))
		pr_seq(s, "udp ");
	if (features & (FLOW_F_ESP4))
		pr_seq(s, "esp ");
	if (features & (FLOW_F_AH4))
		pr_seq(s, "ah ");
	pr_seq(s, "\n ipv6: ");
	if (features & (FLOW_F_TCP6))
		pr_seq(s, "tcp ");
	if (features & (FLOW_F_UDP6))
		pr_seq(s, "udp ");
	if (features & (FLOW_F_ESP6))
		pr_seq(s, "esp ");
	if (features & (FLOW_F_AH6))
		pr_seq(s, "ah ");
	pr_seq(s, "\n tunnels: ");
	if (features & (FLOW_F_GRETAP4WAN))
		pr_seq(s, "gretap4wan ");
	if (features & (FLOW_F_GRETAP6WAN))
		pr_seq(s, "gretap6wan ");
	if (features & (FLOW_F_GRETAP4ETHWAN))
		pr_seq(s, "gretap4ethwan ");
	if (features & (FLOW_F_GRETAP6ETHWAN))
		pr_seq(s, "gretap6ethwan ");
	if (features & (FLOW_F_GRETAP4LAN))
		pr_seq(s, "gretap4lan ");
	if (features & (FLOW_F_GRETAP6LAN))
		pr_seq(s, "gretap6lan ");
	if (features & (FLOW_F_DSLITE))
		pr_seq(s, "dslite ");
	if (features & (FLOW_F_MAPT))
		pr_seq(s, "mapt ");
	if (features & (FLOW_F_DSLITE_GRETAP4))
		pr_seq(s, "dslite->gretap4 ");
	if (features & (FLOW_F_DSLITE_GRETAP6))
		pr_seq(s, "dslite->gretap6 ");
	if (features & (FLOW_F_GRETAP4_DSLITE))
		pr_seq(s, "gretap4->dslite ");
	if (features & (FLOW_F_GRETAP6_DSLITE))
		pr_seq(s, "gretap6->dslite ");
	if (features & (FLOW_F_MAPT_GRETAP4))
		pr_seq(s, "mapt->gretap4 ");
	if (features & (FLOW_F_MAPT_GRETAP6))
		pr_seq(s, "mapt->gretap6 ");
	if (features & (FLOW_F_GRETAP4_MAPT))
		pr_seq(s, "gretap4->mapt ");
	if (features & (FLOW_F_GRETAP6_MAPT))
		pr_seq(s, "gretap6->mapt ");
	pr_seq(s, "\n l2: ");
	if (features & (FLOW_F_MACBR))
		pr_seq(s, "macbr ");
	if (features & (FLOW_F_MCAST))
		pr_seq(s, "mcast ");
	pr_seq(s, "\n");
}

void flowmgr_debug_counter_clear(void)
{
	if (flowmgr.enable || flowmgr.enable_expected_flow)
		return;
	memset(&flowmgr.cnt_start[0], 0,
	       offsetof(struct flowmgr, cnt_end) -
	       offsetof(struct flowmgr, cnt_start));
}

void flowmgr_debug_counter_show(struct seq_file *s)
{
	pr_seq(s, "Flows Created: %u\n",
		atomic_read(&flowmgr.flow_create_cnt));
	pr_seq(s, "Flows Deleted: %u\n",
		atomic_read(&flowmgr.flow_delete_cnt));
	pr_seq(s, "Flows Create Err: %u\n",
		atomic_read(&flowmgr.flow_create_err_cnt));
	pr_seq(s, "Flows Delete Err: %u\n",
		atomic_read(&flowmgr.flow_delete_err_cnt));
	pr_seq(s, "Flows Ignored: %u\n",
		atomic_read(&flowmgr.flow_ignore_cnt));
	pr_seq(s, "Flows TCP: %u\n",
		atomic_read(&flowmgr.flow_tcp_cnt));
	pr_seq(s, "Flows UDP: %u\n",
		atomic_read(&flowmgr.flow_udp_cnt));
	pr_seq(s, "Flows ESP: %u\n",
		atomic_read(&flowmgr.flow_esp_cnt));
	pr_seq(s, "Flows AH: %u\n",
		atomic_read(&flowmgr.flow_ah_cnt));
	pr_seq(s, "Multicast Flows Created: %u\n",
		atomic_read(&flowmgr.flow_mcast_create_cnt));
	pr_seq(s, "Multicast Flows Deleted: %u\n",
		atomic_read(&flowmgr.flow_mcast_delete_cnt));
	pr_seq(s, "Multicast Flows Create Err: %u\n",
		atomic_read(&flowmgr.flow_mcast_create_err_cnt));
	pr_seq(s, "Flows Expected Created: %u\n",
		atomic_read(&flowmgr.flow_expected_create_cnt));
	pr_seq(s, "Flows Expected Deleted: %u\n",
		atomic_read(&flowmgr.flow_expected_delete_cnt));
	pr_seq(s, "Flows Expected Create Err: %u\n",
		atomic_read(&flowmgr.flow_expected_create_err_cnt));
	pr_seq(s, "Flows Expected Delete Err: %u\n",
		atomic_read(&flowmgr.flow_expected_delete_err_cnt));
#if defined (CONFIG_ARM64)
	pr_seq(s, "Total packets: %lld\n",
		atomic64_read(&flowmgr.pkt_processed_cnt));
#else
	pr_seq(s, "Total packets: %llu\n",
		atomic64_read(&flowmgr.pkt_processed_cnt));
#endif
	pr_seq(s, "Flood packets: %u\n",
		atomic_read(&flowmgr.pkt_flood_cnt));
#if defined (CONFIG_ARM64)
	pr_seq(s, "Skip offload packets: %lld\n",
		atomic64_read(&flowmgr.pkt_skipoffload_cnt));
#else
	pr_seq(s, "Skip offload packets: %llu\n",
		atomic64_read(&flowmgr.pkt_skipoffload_cnt));
#endif
	pr_seq(s, "No group out packets: %u\n",
			atomic_read(&flowmgr.pkt_ignore_nogroup_out_cnt));
	pr_seq(s, "No group in packets : %u\n",
			atomic_read(&flowmgr.pkt_ignore_nogroup_in_cnt));
	pr_seq(s, "Null in dev packets : %u\n",
			atomic_read(&flowmgr.pkt_ignore_null_in_cnt));
	pr_seq(s, "Multicast packets   : %u\n",
			atomic_read(&flowmgr.pkt_ignore_mc_cnt));
	pr_seq(s, "IP Multicast packets: %u\n",
			atomic_read(&flowmgr.pkt_ignore_ipmc_cnt));
	pr_seq(s, "Zero port packets   : %u\n",
			atomic_read(&flowmgr.pkt_ignore_0port_cnt));
	pr_seq(s, "No socket packets from WAN: %u\n",
			atomic_read(&flowmgr.pkt_nosock_cnt));
#if defined (CONFIG_ARM64)
	pr_seq(s, "UDP  Packets through slow path: O:%lld R:%lld\n",
		atomic64_read(&flowmgr.in_flight_udp_pkt_cnt[0]),
		atomic64_read(&flowmgr.in_flight_udp_pkt_cnt[1]));
	pr_seq(s, "TCP  Packets through slow path: O:%lld R:%lld\n",
		atomic64_read(&flowmgr.in_flight_tcp_pkt_cnt[0]),
		atomic64_read(&flowmgr.in_flight_tcp_pkt_cnt[1]));
#else
	pr_seq(s, "UDP  Packets through slow path: O:%llu R:%llu\n",
		atomic64_read(&flowmgr.in_flight_udp_pkt_cnt[0]),
		atomic64_read(&flowmgr.in_flight_udp_pkt_cnt[1]));
	pr_seq(s, "TCP  Packets through slow path: O:%llu R:%llu\n",
		atomic64_read(&flowmgr.in_flight_tcp_pkt_cnt[0]),
		atomic64_read(&flowmgr.in_flight_tcp_pkt_cnt[1]));
#endif
	pr_seq(s, "Flow Flush orig: cnt-%u\n",
		atomic_read(&flowmgr.flow_flush_orig_cnt));
	pr_seq(s, "Flow Flush repl: cnt-%u\n",
		atomic_read(&flowmgr.flow_flush_repl_cnt));
	pr_seq(s, "Flow Flush mac: auto-mode(%d) cnt-%u\n",
		flowmgr.auto_flush_mac_mode,
		atomic_read(&flowmgr.flow_flush_mac_cnt));
	pr_seq(s, "Flow Flush inf: auto-mode(%d) cnt-%u\n",
		flowmgr.auto_flush_inf_mode,
		atomic_read(&flowmgr.flow_flush_inf_cnt));
	pr_seq(s, "Flow Flush wifi: cnt-%u\n",
		atomic_read(&flowmgr.flow_flush_wifi_cnt));
	pr_seq(s, "Flow Unlearn : cnt-%u orig-%u repl-%u\n",
		atomic_read(&flowmgr.flow_unlearn_cnt),
		atomic_read(&flowmgr.flow_unlearn_orig_cnt),
		atomic_read(&flowmgr.flow_unlearn_repl_cnt));
	pr_seq(s, "Flow Flush Worker: Active(%u)\n",
		atomic_read(&flowmgr.flush_worker_count));
	pr_seq(s, "----------------------\n");
}

void flowmgr_debug_show(struct seq_file *s)
{
	struct net_device *dev = flowmgr_get_wandev();
	pr_seq(s, "Enable: %d\n", flowmgr.enable);
	pr_seq(s, "Flow promotion mode  : %d\n", flowmgr.promote_mode);
	pr_seq(s, "Promotion dir ctl    : %s\n",
	       promote_dir_ctl_str(flowmgr.promote_dir_ctl));
	pr_seq(s, "GRE Acceleration     : %d\n",
		!flowmgr.disable_gre_accel);
	pr_seq(s, "MAPT Acceleration    : %d (%d:%d)\n",
		(flowmgr_is_feat_map_enable() && !flowmgr.disable_mapt_accel), flowmgr_is_feat_map_enable(), flowmgr.disable_mapt_accel);
	pr_seq(s, "3-tuple Bridge Flow Promotion : %d\n",
		flowmgr.use_3tuple_br_flow);
	pr_seq(s, "TCP Flow Promotion   : %s\n",
		flowmgr.tcp_pkt_threshold ? "Threshold" :
		(flowmgr.use_tcp_3way_handshake ? "3way" : "2Way"));
	pr_seq(s, "UDP Flow promotion threshold: %d\n",
		flowmgr.udp_pkt_threshold);
	if (flowmgr.tcp_pkt_threshold)
	pr_seq(s, "TCP Flow promotion threshold: %d\n",
		flowmgr.tcp_pkt_threshold);
	pr_seq(s, "WAN Dev: %s\n",
		flowmgr.wandev ? flowmgr.wandev->name:"Uninitialized");
	pr_seq(s, "Active WAN Dev: %s type %d\n",
		dev ? dev->name:"Uninitialized", flowmgr.wantype);
	pr_seq(s, "Bridge Conntrack Call: flowmgr=%d | ebtables_ipv4/ipv6=%d/%d\n",
		flowmgr.enable_br_conntrack_call,
		is_brnf_call_iptables(),
		is_brnf_call_ip6tables());
	pr_seq(s, "----------------------\n");
	flowmgr_debug_counter_show(s);
}

static int panic_callback(struct notifier_block *self,
			  unsigned long event, void *ctx)
{
	time64_t now;
	struct tm tm_val;

	now = ktime_get_real_seconds();
	time64_to_tm(now, 0, &tm_val);

	pr_emerg("---[ Flowmgr dump: Start %d/%d/%ld %02d:%02d:%02d ]---\n",
		 tm_val.tm_mon + 1, tm_val.tm_mday, 1900 + tm_val.tm_year,
		 tm_val.tm_hour, tm_val.tm_min, tm_val.tm_sec);
	flowmgr_fap_dbg_show(NULL);
	flowmgr_debug_show(NULL);
	flowmgr_trace_show(NULL);
	pr_emerg("---[ Flowmgr dump: End ]---\n");
	return NOTIFY_OK;
}

static struct notifier_block nb_panic = {
	.notifier_call  = panic_callback,
};

static int __init flowmgr_init(void)
{
	int ret = 0;
	int i;
#ifdef CONFIG_NF_CONNTRACK_EVENTS
	struct nf_ct_event_notifier *notify;
	struct net *net = &init_net;
#endif
	for (i = 0; i < ARRAY_SIZE(def_net_interface_list); i++)
		flowmgr_netdevice_add(def_net_interface_list[i], NULL);
#ifdef CONFIG_NF_CONNTRACK_EVENTS
	rcu_read_lock();
	notify = rcu_dereference(net->ct.nf_conntrack_event_cb);
	rcu_read_unlock();
	if (notify == NULL) {
		ret = nf_conntrack_register_notifier(&init_net,
						     &ctflowmgr_notifier);
	}
#endif
	ret = nf_conntrack_register_offload(&init_net, &ctflowmgr_ops);
	if (ret < 0) {
		pr_err("FLOWMGR Init: Cannot register conntrack notifier.\n");
		goto err_unreg_hooks;
	}
#ifdef CONFIG_SYSCTL
	flowmgr.sysctl_header =
		register_net_sysctl(&init_net,
				    "net/flowmgr",
				    flowmgr_sysctl_table);
	if (!flowmgr.sysctl_header)
		goto err_unreg_notifier;
#endif
	register_netevent_notifier(&nb_netevent);
	register_netdevice_notifier(&nb_netdevevent);
	atomic_notifier_chain_register(&panic_notifier_list,
				       &nb_panic);

	flowmgr_procfs_init();
	flowmgr_db_init();
#ifdef CONFIG_BCM_FLOWMGR_MCAST
	flowmgr_mcast_init();
#endif
#ifdef CONFIG_BCM_FLOWMGR_TUNNEL
	flowmgr_tunnel_init();
	flowmgr_map_init();
#endif
#ifdef CONFIG_BCM_FLOWMGR_IOCTL
	flowmgr_cdev_init();
#endif
#ifdef CONFIG_BCM_FLOWMGR_ARL
	flowmgr_arl_init();
#endif
	flowmgr.udp_pkt_threshold = 10;
	flowmgr.tcp_pkt_threshold = 15;
	flowmgr.enable_expected_flow = 1;
	flowmgr.expected_flow_timeout = 0;
	flowmgr.nosock_drop = 1;
	flowmgr.fast_death_timeout = 1;
	flowmgr.check_status_timeout = 10;
	flowmgr.promote_mode = AUTO_PROMOTE;
	flowmgr.auto_flush_mac_mode = FLOW_FLUSH_UNLEARN;
	flowmgr.auto_flush_inf_mode = FLOW_FLUSH_UNLEARN;
	flowmgr.promote_dir_ctl = 1<<flow_lanlan |
		1<<flow_wanlan | 1<<flow_lanwan;
#ifdef CONFIG_BCM_FLOWMGR_MCAST
	//flowmgr_mcast_counter_start();
#endif
	flowmgr_manual_init();
	flowmgr_enable_br_conntrack(1);
	nf_register_net_hooks(&init_net, flowmgr_nf_ops,
			      ARRAY_SIZE(flowmgr_nf_ops));
	flowmgr.lock = __SPIN_LOCK_UNLOCKED(flowmgr.lock);
	flowmgr.lock_counter = __SPIN_LOCK_UNLOCKED(flowmgr.lock_counter);
	flowmgr_trace_init();
	flowmgr.fap_wq = alloc_workqueue("flowmgr_fap", WQ_SYSFS | WQ_CPU_INTENSIVE, 1);
	if (!flowmgr.fap_wq)
		goto err_unreg_notifier;
	flowmgr_debug_init();
	flowmgr.sw_features = FLOW_F_ALL;
	flowmgr.features = (flowmgr.sw_features & flowmgr.hw_features);
	flowmgr.enable_tu_port_track = 1;
	flowmgr.tu_port_track_timeout = 90;
	pr_alert("FLOWMGR Init: %s\n", VER_STR);
	return 0;
err_unreg_notifier:
#ifdef CONFIG_NF_CONNTRACK_EVENTS
	rcu_read_lock();
	notify = rcu_dereference(net->ct.nf_conntrack_event_cb);
	rcu_read_unlock();
	if (notify == &ctflowmgr_notifier)
		nf_conntrack_unregister_notifier(&init_net,
						 &ctflowmgr_notifier);
#endif
	nf_conntrack_unregister_offload(&init_net);
err_unreg_hooks:
	for (i = 0; i < interfaces_c; i++)
		flowmgr_netdevice_del(net_interface_list[i]);
	return ret;
}

static void __exit flowmgr_exit(void)
{
	int i;
#ifdef CONFIG_NF_CONNTRACK_EVENTS
	struct nf_ct_event_notifier *notify;
	struct net *net = &init_net;
	rcu_read_lock();
	notify = rcu_dereference(net->ct.nf_conntrack_event_cb);
	rcu_read_unlock();
	if (notify == &ctflowmgr_notifier)
		nf_conntrack_unregister_notifier(&init_net,
						 &ctflowmgr_notifier);
#endif
	flowmgr.enable = 0;
	flowmgr_db_flush();
#ifdef CONFIG_BCM_FLOWMGR_MCAST
	flowmgr_mdb_flush();
#endif
#ifdef CONFIG_BCM_FLOWMGR_TUNNEL
	flowmgr_tdb_flush();
#endif
	flowmgr_conntrack_clean_all();
	while (flowmgr_fap_flows_active()) {
		msleep(1000);
		flowmgr_fap_dbg_show(NULL);
	}
	unregister_netevent_notifier(&nb_netevent);
	unregister_netdevice_notifier(&nb_netdevevent);
	atomic_notifier_chain_unregister(&panic_notifier_list,
					 &nb_panic);
	nf_conntrack_unregister_offload(&init_net);
	for (i = 0; i < interfaces_c; i++)
		flowmgr_netdevice_del(net_interface_list[i]);
#ifdef CONFIG_SYSCTL
	unregister_net_sysctl_table(flowmgr.sysctl_header);
#endif
	flowmgr_manual_fini();
#ifdef CONFIG_BCM_FLOWMGR_IOCTL
	flowmgr_cdev_exit();
#endif
#ifdef CONFIG_BCM_FLOWMGR_ARL
	flowmgr_arl_exit();
#endif
	flowmgr_debug_exit();
	flowmgr_trace_exit();
	flowmgr_procfs_exit();
	flowmgr_db_fini();
#ifdef CONFIG_BCM_FLOWMGR_MCAST
	flowmgr_mcast_fini();
#endif
#ifdef CONFIG_BCM_FLOWMGR_TUNNEL
	flowmgr_tunnel_fini();
	flowmgr_map_fini();
#endif
	nf_unregister_net_hooks(&init_net, flowmgr_nf_ops,
				ARRAY_SIZE(flowmgr_nf_ops));
	flush_workqueue(flowmgr.fap_wq);
	destroy_workqueue(flowmgr.fap_wq);
	pr_info("FLOWMGR Exit:\n");
}

module_init(flowmgr_init);
module_exit(flowmgr_exit);
MODULE_LICENSE("GPL");
MODULE_ALIAS("flowmgr");
