 /****************************************************************************
 *
 * Copyright (c) 2015-2018 Broadcom. All rights reserved
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Jayesh Patel <jayeshp@broadcom.com>
 ****************************************************************************/

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/rculist.h>
#include <linux/spinlock.h>
#include <linux/times.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/jhash.h>
#include <linux/random.h>
#include <linux/slab.h>
#include <linux/atomic.h>
#include <asm/unaligned.h>
#include <linux/u64_stats_sync.h>
#include "flowmgr.h"

/* Following code is derived from br_fdb.c implementation */

#define FLOWMGR_HASH_BITS 8
#define FLOWMGR_HASH_SIZE (1 << FLOWMGR_HASH_BITS)
#define FLOWMGR_MAX_DB_ENTRY 256

static spinlock_t			hash_lock;
static struct hlist_head		hash[FLOWMGR_HASH_SIZE];

static struct kmem_cache *flowmgr_db_cache __read_mostly;

static u32 db_salt __read_mostly;
static atomic_t max_entries;
u64 gdscp_notrack;

int flowmgr_db_init(void)
{
	flowmgr_db_cache = kmem_cache_create("flowmgr_db_cache",
					     sizeof(struct flowmgr_db_entry),
					     0,
					     SLAB_HWCACHE_ALIGN, NULL);
	if (!flowmgr_db_cache)
		return -ENOMEM;

	get_random_bytes(&db_salt, sizeof(db_salt));

	return 0;
}

void flowmgr_db_fini(void)
{
	kmem_cache_destroy(flowmgr_db_cache);
}

static inline int flowmgr_mac_hash(const unsigned char *mac)
{
	/* use 1 byte of OUI cnd 3 bytes of NIC */
	u32 key = get_unaligned((u32 *)(mac + 2));

	return jhash_1word(key, db_salt) & (FLOWMGR_HASH_SIZE - 1);
}

static void db_rcu_free(struct rcu_head *head)
{
	struct flowmgr_db_entry *ent
		= container_of(head, struct flowmgr_db_entry, rcu);

	kmem_cache_free(flowmgr_db_cache, ent);
	atomic_sub(1, &max_entries);
}

static void db_delete(struct flowmgr_db_entry *f)
{
	if (f && atomic_dec_and_test(&f->use)) {
		hlist_del_rcu(&f->hlist);
		call_rcu(&f->rcu, db_rcu_free);
	}
}


/* Completely flush all dynamic entries in client database */
void flowmgr_db_flush(void)
{
	int i;

	if (flowmgr.enable)
		return;
	spin_lock_bh(&hash_lock);
	for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
		struct flowmgr_db_entry *f;
		struct hlist_node *n;

		hlist_iterate_safe(f, n, &hash[i], hlist) {
			db_delete(f);
		}
	}
	spin_unlock_bh(&hash_lock);
}

/* Flush all entries referring to a specific port.
 * if do_all is set also flush static entries
 */
void flowmgr_db_delete_by_port(int iif)
{
	int i;

	spin_lock_bh(&hash_lock);
	for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
		struct hlist_node *h, *g;

		hlist_for_each_safe(h, g, &hash[i]) {
			struct flowmgr_db_entry *f;

			f = hlist_entry(h, struct flowmgr_db_entry, hlist);
			if (f->iif != iif)
				continue;

			db_delete(f);
		}
	}
	spin_unlock_bh(&hash_lock);
}

/* Internal Function: Find entry based on mac address in specific list */
static struct flowmgr_db_entry *db_find(struct hlist_head *head,
					const unsigned char *addr)
{
	struct flowmgr_db_entry *db;

	hlist_iterate(db, head, hlist) {
		if (ether_addr_equal(db->addr, addr))
			return db;
	}
	return NULL;
}

/* Find entry based on mac address */
struct flowmgr_db_entry *flowmgr_db_find(const unsigned char *addr)
{
	struct hlist_head *head = &hash[flowmgr_mac_hash(addr)];
	struct flowmgr_db_entry *db = NULL;

	spin_lock_bh(&hash_lock);
	db = db_find(head, addr);
	spin_unlock_bh(&hash_lock);
	return db;
}

/* Internal Function: Create entry based on input interface and mac address
 * in specific list
 */
static struct flowmgr_db_entry *db_create(struct hlist_head *head,
					  int iif,
					  const unsigned char *addr)
{
	struct flowmgr_db_entry *db;

	if (atomic_read(&max_entries) >= FLOWMGR_MAX_DB_ENTRY)
		return NULL;

	db = kmem_cache_alloc(flowmgr_db_cache, GFP_ATOMIC);
	if (db) {
		pr_debug("FLOWMGR: db create %pM\n", addr);

		memset(db, 0, sizeof(struct flowmgr_db_entry));
		memcpy(db->addr, addr, ETH_ALEN);
		db->iif = iif;
		atomic_set(&db->use, 1);
		db->created = jiffies;
		db->updated = db->created;
		db->accel = 1;
		db->steer = 1;
		hlist_add_head_rcu(&db->hlist, head);
		atomic_add(1, &max_entries);
	}
	return db;
}

/* Internal Function: Delete entry based on mac address */
static int db_delete_by_addr(const u8 *addr)
{
	struct hlist_head *head = &hash[flowmgr_mac_hash(addr)];
	struct flowmgr_db_entry *db;

	db = db_find(head, addr);
	if (!db)
		return -ENOENT;

	db_delete(db);
	return 0;
}

/* Delete entry based on mac address */
int flowmgr_db_delete(const unsigned char *addr)
{
	int err;

	spin_lock_bh(&hash_lock);
	err = db_delete_by_addr(addr);
	spin_unlock_bh(&hash_lock);
	return err;
}

void flowmgr_db_ifenable(struct net_device *dev, int enable)
{
	if (!dev)
		return;
	pr_debug("FLOWMGR: %s ifdb enable %d\n",
		 dev->name, enable);
	dev->group = BCM_NETDEVICE_GROUP_FLAG_CLR(dev->group) | ((!enable)<<16);
}

int is_flowmgr_db_ifenable(struct net_device *dev)
{
	if (dev)
		return !(BCM_NETDEVICE_GROUP_FLAG(dev->group) & 1);
	else
		return 0;
}

int flowmgr_db_inc_rxflow(struct flowmgr_db_entry *db)
{
	if (!db)
		return -1;
	atomic_inc(&db->rxflows);
	atomic_inc(&db->totrxflows);
	return 0;
}

int flowmgr_db_dec_rxflow(struct flowmgr_db_entry *db)
{
	if (!db)
		return -1;
	atomic_dec(&db->rxflows);
	return 0;
}

int flowmgr_db_inc_txflow(struct flowmgr_db_entry *db)
{
	if (!db)
		return -1;
	atomic_inc(&db->txflows);
	atomic_inc(&db->tottxflows);
	return 0;
}

int flowmgr_db_dec_txflow(struct flowmgr_db_entry *db)
{
	if (!db)
		return -1;
	atomic_dec(&db->txflows);
	return 0;
}

/* Internal Function: Update entry based on input interface and mac address */
static struct flowmgr_db_entry *db_update(int iif, const unsigned char *addr,
					  u16 vid)
{
	struct hlist_head *head = &hash[flowmgr_mac_hash(addr)];
	struct flowmgr_db_entry *db;

	if (!is_valid_ether_addr(addr))
		return NULL;

	db = db_find(head, addr);
	if (!db)
		db = db_create(head, iif, addr);
	if (!db)
		return NULL;
	db->iif = iif;
	db->created = jiffies;
	db->vlan_id = vid;

	return db;
}

/* Update entry based on input interface and mac address */
int flowmgr_db_update(const struct net_device *in, const unsigned char *addr,
		      u16 vid)
{
	struct flowmgr_db_entry *db;

	spin_lock_bh(&hash_lock);
	db = db_update(in->ifindex, addr, vid);
	spin_unlock_bh(&hash_lock);
	return db ? 0 : -1;
}

/* Update slow path rx stats based db entry */
int flowmgr_db_update_rxstats_slow(struct flowmgr_db_entry *db,
				   unsigned int packets,
				   unsigned int bytes,
				   u8 dscp)
{
	if (dscp >= DSCP_MAX_LIMIT)
		return -EINVAL;
	if (db->notrack)
		return 0;
	atomic64_add(packets, &db->stats.rx_packets);
	atomic64_add(bytes, &db->stats.rx_bytes);
	if (!((gdscp_notrack>>dscp)&0x1ULL)) {
		atomic64_add(packets, &db->dscp_stats[dscp].rx_packets);
		atomic64_add(bytes, &db->dscp_stats[dscp].rx_bytes);
	}
	db->updated = jiffies;
	return 0;
}

int flowmgr_db_dec_rxstats_slow(struct flowmgr_db_entry *db,
				   unsigned int packets,
				   unsigned int bytes,
				   u8 dscp)
{
	if (dscp >= DSCP_MAX_LIMIT)
		return -EINVAL;
	if (db->notrack)
		return 0;
	if (atomic64_read(&db->stats.rx_packets) < packets)
		return 0;
	if (atomic64_read(&db->stats.rx_bytes) < bytes)
		return 0;
	atomic64_sub(packets, &db->stats.rx_packets);
	atomic64_sub(bytes, &db->stats.rx_bytes);
	if (!((gdscp_notrack>>dscp)&0x1ULL)) {
		atomic64_sub(packets, &db->dscp_stats[dscp].rx_packets);
		atomic64_sub(bytes, &db->dscp_stats[dscp].rx_bytes);
	}
	db->updated = jiffies;
	return 0;
}

/* Update fast path rx stats based db entry */
int flowmgr_db_update_rxstats_fast(struct flowmgr_db_entry *db,
				   unsigned int packets,
				   unsigned int bytes,
				   u8 dscp)
{
	if (dscp >= DSCP_MAX_LIMIT)
		return -EINVAL;
	if (db->notrack)
		return 0;
	atomic64_add(packets, &db->fstats.rx_packets);
	atomic64_add(bytes, &db->fstats.rx_bytes);
	if (!((gdscp_notrack>>dscp)&0x1ULL)) {
		atomic64_add(packets, &db->dscp_fstats[dscp].rx_packets);
		atomic64_add(bytes, &db->dscp_fstats[dscp].rx_bytes);
	}
	db->updated = jiffies;
	return 0;
}

/* Update entry based on input interface and mac address */
int flowmgr_db_update_rx_slow(const struct net_device *dev,
			      const unsigned char *addr,
			      unsigned int packets,
			      unsigned int bytes,
			      u8 dscp)
{
	struct flowmgr_db_entry *db;
	struct hlist_head *head = &hash[flowmgr_mac_hash(addr)];

	if (!is_valid_ether_addr(addr))
		return -EINVAL;
	if (!dev)
		return -EINVAL;
	db = db_find(head, addr);
	if (!db)
		db = db_create(head, dev->ifindex, addr);
	if (!db)
		return -ENOMEM;
	flowmgr_db_update_rxstats_slow(db, packets, bytes, dscp);

	return 0;
}

int flowmgr_db_dec_rx_slow(const struct net_device *dev,
			      const unsigned char *addr,
			      unsigned int packets,
			      unsigned int bytes,
			      u8 dscp)
{
	struct flowmgr_db_entry *db;
	struct hlist_head *head = &hash[flowmgr_mac_hash(addr)];

	if (!is_valid_ether_addr(addr))
		return -EINVAL;

	db = db_find(head, addr);
	if (!db)
		db = db_create(head, dev->ifindex, addr);
	if (!db)
		return -ENOMEM;
	flowmgr_db_dec_rxstats_slow(db, packets, bytes, dscp);

	return 0;
}

/* Update slow path tx stats based db entry */
int flowmgr_db_update_txstats_slow(struct flowmgr_db_entry *db,
				   unsigned int packets,
				   unsigned int bytes,
				   u8 dscp)
{
	if (dscp >= DSCP_MAX_LIMIT)
		return -EINVAL;
	if (db->notrack)
		return 0;
	atomic64_add(packets, &db->stats.tx_packets);
	atomic64_add(bytes, &db->stats.tx_bytes);
	if (!((gdscp_notrack>>dscp)&0x1ULL)) {
		atomic64_add(packets, &db->dscp_stats[dscp].tx_packets);
		atomic64_add(bytes, &db->dscp_stats[dscp].tx_bytes);
	}
	db->updated = jiffies;
	return 0;
}

int flowmgr_db_dec_txstats_slow(struct flowmgr_db_entry *db,
				   unsigned int packets,
				   unsigned int bytes,
				   u8 dscp)
{
	if (dscp >= DSCP_MAX_LIMIT)
		return -EINVAL;
	if (db->notrack)
		return 0;
	if (atomic64_read(&db->stats.tx_packets) < packets)
		return 0;
	if (atomic64_read(&db->stats.tx_bytes) < bytes)
		return 0;
	atomic64_sub(packets, &db->stats.tx_packets);
	atomic64_sub(bytes, &db->stats.tx_bytes);
	if (!((gdscp_notrack>>dscp)&0x1ULL)) {
		atomic64_sub(packets, &db->dscp_stats[dscp].tx_packets);
		atomic64_sub(bytes, &db->dscp_stats[dscp].tx_bytes);
	}
	db->updated = jiffies;
	return 0;
}

/* Update fast path tx stats based db entry */
int flowmgr_db_update_txstats_fast(struct flowmgr_db_entry *db,
				   unsigned int packets,
				   unsigned int bytes,
				   u8 dscp)
{
	if (dscp >= DSCP_MAX_LIMIT)
		return -EINVAL;
	if (db->notrack)
		return 0;
	atomic64_add(packets, &db->fstats.tx_packets);
	atomic64_add(bytes, &db->fstats.tx_bytes);
	if (!((gdscp_notrack>>dscp)&0x1ULL)) {
		atomic64_add(packets, &db->dscp_fstats[dscp].tx_packets);
		atomic64_add(bytes, &db->dscp_fstats[dscp].tx_bytes);
	}
	db->updated = jiffies;
	return 0;
}

/* Update entry based on input interface and mac address */
int flowmgr_db_update_tx_slow(const struct net_device *dev,
			      const unsigned char *addr,
			      unsigned int packets,
			      unsigned int bytes,
			      u8 dscp)
{
	struct flowmgr_db_entry *db;
	struct hlist_head *head = &hash[flowmgr_mac_hash(addr)];

	if (!is_valid_ether_addr(addr))
		return -EINVAL;

	db = db_find(head, addr);
	if (!db)
		db = db_create(head, dev->ifindex, addr);

	if (!db)
		return -ENOMEM;

	flowmgr_db_update_txstats_slow(db, packets, bytes, dscp);

	return 0;
}

int flowmgr_db_dec_tx_slow(const struct net_device *dev,
			      const unsigned char *addr,
			      unsigned int packets,
			      unsigned int bytes,
			      u8 dscp)
{
	struct flowmgr_db_entry *db;
	struct hlist_head *head = &hash[flowmgr_mac_hash(addr)];

	if (!is_valid_ether_addr(addr))
		return -EINVAL;

	db = db_find(head, addr);
	if (!db)
		db = db_create(head, dev->ifindex, addr);

	if (!db)
		return -ENOMEM;

	flowmgr_db_dec_txstats_slow(db, packets, bytes, dscp);

	return 0;
}

int flowmgr_db_get_stats64_fast(struct flowmgr_db_entry *db,
				struct link_stats *stats)
{
	stats->tx_bytes   = atomic64_read(&db->fstats.tx_bytes);
	stats->tx_packets = atomic64_read(&db->fstats.tx_packets);
	stats->rx_bytes   = atomic64_read(&db->fstats.rx_bytes);
	stats->rx_packets = atomic64_read(&db->fstats.rx_packets);

	return 0;
}

int flowmgr_db_get_stats64_slow(struct flowmgr_db_entry *db,
				struct link_stats *stats,
				int init)
{
	if (init) {
		stats->tx_bytes   = 0;
		stats->tx_packets = 0;
		stats->rx_bytes   = 0;
		stats->rx_packets = 0;
	}

	stats->tx_bytes   += atomic64_read(&db->stats.tx_bytes);
	stats->tx_packets += atomic64_read(&db->stats.tx_packets);
	stats->rx_bytes   += atomic64_read(&db->stats.rx_bytes);
	stats->rx_packets += atomic64_read(&db->stats.rx_packets);

	return 0;
}

int flowmgr_db_get_stats64(struct flowmgr_db_entry *db,
			   struct link_stats *stats)
{
	flowmgr_db_get_stats64_fast(db, stats);
	flowmgr_db_get_stats64_slow(db, stats, 0);
	return 0;
}

int flowmgr_db_get_dscp_stats64_fast(struct flowmgr_db_entry *db,
				struct link_stats *stats, u64 *dscp_cgz)
{
	int cnt = 0;

	for (cnt = 0; cnt < DSCP_MAX_LIMIT; cnt++) {
		stats[cnt].tx_bytes   = atomic64_read(&db->dscp_fstats[cnt].tx_bytes);
		stats[cnt].tx_packets = atomic64_read(&db->dscp_fstats[cnt].tx_packets);
		stats[cnt].rx_bytes   = atomic64_read(&db->dscp_fstats[cnt].rx_bytes);
		stats[cnt].rx_packets = atomic64_read(&db->dscp_fstats[cnt].rx_packets);
		if (stats[cnt].tx_packets > 0 || stats[cnt].rx_packets > 0)
			*dscp_cgz |= (0x1ULL<<cnt);
		else
			*dscp_cgz &= ~(0x1ULL<<cnt);
	}

	return 0;
}

int flowmgr_db_get_dscp_stats64_slow(struct flowmgr_db_entry *db,
				     struct link_stats *stats,
				     int init, u64 *dscp_cgz)
{
	int cnt = 0;

	for (cnt = 0; cnt < DSCP_MAX_LIMIT; cnt++) {
		if (init) {
			stats[cnt].tx_bytes   = 0;
			stats[cnt].tx_packets = 0;
			stats[cnt].rx_bytes   = 0;
			stats[cnt].rx_packets = 0;
		}
		stats[cnt].tx_bytes   += atomic64_read(&db->dscp_stats[cnt].tx_bytes);
		stats[cnt].tx_packets += atomic64_read(&db->dscp_stats[cnt].tx_packets);
		stats[cnt].rx_bytes   += atomic64_read(&db->dscp_stats[cnt].rx_bytes);
		stats[cnt].rx_packets += atomic64_read(&db->dscp_stats[cnt].rx_packets);
		if (stats[cnt].tx_packets > 0 || stats[cnt].rx_packets > 0)
			*dscp_cgz |= (0x1ULL<<cnt);
		else
			*dscp_cgz &= ~(0x1ULL<<cnt);
	}

	return 0;
}

int flowmgr_db_get_dscp_stats64(struct flowmgr_db_entry *db,
				struct link_stats *stats, u64 *dscp_cgz)
{
	flowmgr_db_get_dscp_stats64_fast(db, stats, dscp_cgz);
	flowmgr_db_get_dscp_stats64_slow(db, stats, 0, dscp_cgz);
	return 0;
}

int flowmgr_db_clear_stats(struct flowmgr_db_entry *db)
{
	int cnt = 0;

	atomic64_set(&db->stats.tx_packets, 0);
	atomic64_set(&db->stats.tx_bytes, 0);
	atomic64_set(&db->stats.rx_packets, 0);
	atomic64_set(&db->stats.rx_bytes, 0);
	atomic64_set(&db->fstats.tx_packets, 0);
	atomic64_set(&db->fstats.tx_bytes, 0);
	atomic64_set(&db->fstats.rx_packets, 0);
	atomic64_set(&db->fstats.rx_bytes, 0);
	for (cnt = 0; cnt < DSCP_MAX_LIMIT; cnt++) {
		atomic64_set(&db->dscp_stats[cnt].tx_packets, 0);
		atomic64_set(&db->dscp_stats[cnt].tx_bytes, 0);
		atomic64_set(&db->dscp_stats[cnt].rx_packets, 0);
		atomic64_set(&db->dscp_stats[cnt].rx_bytes, 0);
		atomic64_set(&db->dscp_fstats[cnt].tx_packets, 0);
		atomic64_set(&db->dscp_fstats[cnt].tx_bytes, 0);
		atomic64_set(&db->dscp_fstats[cnt].rx_packets, 0);
		atomic64_set(&db->dscp_fstats[cnt].rx_bytes, 0);
	}
	return 0;
}

/* Clear stats of client by specific address */
int flowmgr_db_clear_stats_by_addr(unsigned char *addr)
{
	struct flowmgr_db_entry *db;
	struct hlist_head *head = &hash[flowmgr_mac_hash(addr)];

	if (!is_valid_ether_addr(addr))
		return -EINVAL;

	spin_lock_bh(&hash_lock);
	db = db_find(head, addr);
	spin_unlock_bh(&hash_lock);
	if (!db)
		return -ENOMEM;

	flowmgr_db_clear_stats(db);

	return 0;
}

/* Clear stats of all clients from specific network device */
int flowmgr_db_clear_stats_by_port(int  iif)
{
	int i;

	spin_lock_bh(&hash_lock);
	for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
		struct hlist_node *h, *g;

		hlist_for_each_safe(h, g, &hash[i]) {
			struct flowmgr_db_entry *f;

			f = hlist_entry(h, struct flowmgr_db_entry, hlist);
			if (f->iif != iif)
				continue;

			flowmgr_db_clear_stats(f);
		}
	}
	spin_unlock_bh(&hash_lock);
	return 0;
}

/* Clear stats of all clients */
void flowmgr_db_clear(void)
{
	int i;

	spin_lock_bh(&hash_lock);
	for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
		struct flowmgr_db_entry *f;
		struct hlist_node *n;

		hlist_iterate_safe(f, n, &hash[i], hlist) {
			flowmgr_db_clear_stats(f);
		}
	}
	spin_unlock_bh(&hash_lock);
}

int flowmgr_db_set_notrack(unsigned char *addr,
			   enum flow_notrack_type notrack)
{
	int i;
	struct flowmgr_db_entry *db = NULL;
	struct hlist_head *head = NULL;
	struct hlist_node *n = NULL;

	if (notrack != RESTORE_DEFAULT_NOTRACK) {
		if (!is_valid_ether_addr(addr))
			return -EINVAL;
	}
	switch (notrack) {
	case DISABLE_NOTRACK:
	case ENABLE_NOTRACK:
		head = &hash[flowmgr_mac_hash(addr)];
		spin_lock_bh(&hash_lock);
		db = db_find(head, addr);
		spin_unlock_bh(&hash_lock);
		if (!db)
			return -ENOMEM;

		db->notrack = notrack;
		break;
	case ENABLE_ALL_NOTRACK:
		spin_lock_bh(&hash_lock);
		for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
			hlist_iterate_safe(db, n, &hash[i], hlist) {
				if (ether_addr_equal(db->addr, addr))
					db->notrack = 0;
				else
					db->notrack = 1;
			}
		}
		spin_unlock_bh(&hash_lock);
		break;
	case DISABLE_ALL_NOTRACK:
		spin_lock_bh(&hash_lock);

		for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
			hlist_iterate_safe(db, n, &hash[i], hlist) {
				if (ether_addr_equal(db->addr, addr))
					db->notrack = 1;
				else
					db->notrack = 0;
			}
		}
		spin_unlock_bh(&hash_lock);
		break;
	case RESTORE_DEFAULT_NOTRACK:
		spin_lock_bh(&hash_lock);
		for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
			hlist_iterate_safe(db, n, &hash[i], hlist) {
				/* Initialize all mac notrack to default */
				db->notrack = 0;
			}
		}
		spin_unlock_bh(&hash_lock);
		break;
		break;
	default:
		return -EINVAL;
	}
	return 0;
}

int flowmgr_db_set_dscp_notrack(u8 dscp,
				enum flow_notrack_type notrack)
{
	int i;
	struct flowmgr_db_entry *db = NULL;
	struct hlist_node *n = NULL;

	if (dscp >= DSCP_MAX_LIMIT)
		return -EINVAL;
	switch (notrack) {
	case DISABLE_NOTRACK:
	case ENABLE_NOTRACK:
		spin_lock_bh(&hash_lock);
		for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
			hlist_iterate_safe(db, n, &hash[i], hlist) {
				if (notrack == ENABLE_NOTRACK)
					gdscp_notrack |= (0x1ULL<<dscp);
				else
					gdscp_notrack &= ~(0x1ULL<<dscp);
			}
		}
		spin_unlock_bh(&hash_lock);
		break;
	case ENABLE_ALL_NOTRACK:
		spin_lock_bh(&hash_lock);
		for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
			hlist_iterate_safe(db, n, &hash[i], hlist) {
				/* Initialize all dscp not to track */
				gdscp_notrack = U64_MAX;
				/* Enable dscp tracking only for the requested */
				gdscp_notrack &= ~(0x1ULL<<dscp);
			}
		}
		spin_unlock_bh(&hash_lock);
		break;
	case DISABLE_ALL_NOTRACK:
		spin_lock_bh(&hash_lock);
		for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
			hlist_iterate_safe(db, n, &hash[i], hlist) {
				/* Initialize all dscp to track */
				gdscp_notrack = 0x0ULL;
				/* Disable dscp tracking only for the requested */
				gdscp_notrack |= (0x1ULL<<dscp);
			}
		}
		spin_unlock_bh(&hash_lock);
		break;
	case RESTORE_DEFAULT_NOTRACK:
		spin_lock_bh(&hash_lock);
		for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
			hlist_iterate_safe(db, n, &hash[i], hlist) {
				/* Initialize all dscp notrack to default */
				gdscp_notrack = 0x0ULL;
			}
		}
		spin_unlock_bh(&hash_lock);
		break;
	default:
		return -EINVAL;
	}
	return 0;
}

/* Get first entry in specific list */
struct hlist_node  *flowmgr_db_get_first(int *hashid)
{
	struct hlist_node *h;
	int i;

	for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
		h = rcu_dereference(hlist_first_rcu(&hash[i]));
		if (h) {
			*hashid = i;
			return h;
		}
	}
	return NULL;
}

/* Get next entry in specific list */
struct hlist_node *flowmgr_db_get_next(int *hashid,
				       struct hlist_node *head)
{
	head = rcu_dereference(hlist_next_rcu(head));
	while (!head) {
		if (++*hashid >= FLOWMGR_HASH_SIZE)
			return NULL;
		head = rcu_dereference(
				hlist_first_rcu(
				   &hash[*hashid]));
	}
	return head;
}

/* Get index in specific list */
struct hlist_node *flowmgr_db_get_idx(int *hashid, loff_t pos)
{
	struct hlist_node *head = flowmgr_db_get_first(hashid);

	if (head)
		while (pos && (head = flowmgr_db_get_next(hashid, head)))
			pos--;
	return pos ? NULL : head;
}

int flowmgr_dbflow_clear_steer_stats(struct flowmgr_db_entry *db)
{
	atomic_set(&db->nsteers, 0);
	return 0;
}

/* Clear steer stats of client by specific address */
int flowmgr_dbflow_clear_steer_stats_by_addr(unsigned char *addr)
{
	struct flowmgr_db_entry *db;
	struct hlist_head *head = &hash[flowmgr_mac_hash(addr)];

	if (!is_valid_ether_addr(addr))
		return -EINVAL;

	spin_lock_bh(&hash_lock);
	db = db_find(head, addr);
	spin_unlock_bh(&hash_lock);
	if (!db)
		return -ENOMEM;

	flowmgr_dbflow_clear_steer_stats(db);

	return 0;
}

/* Clear steer stats of all clients from specific network device */
int flowmgr_dbflow_clear_steer_stats_by_port(int iif)
{
	int i;

	spin_lock_bh(&hash_lock);
	for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
		struct hlist_node *h, *g;

		hlist_for_each_safe(h, g, &hash[i]) {
			struct flowmgr_db_entry *f;

			f = hlist_entry(h, struct flowmgr_db_entry, hlist);
			if (f->iif != iif)
				continue;

			flowmgr_dbflow_clear_steer_stats(f);
		}
	}
	spin_unlock_bh(&hash_lock);
	return 0;
}

/* Clear stats of all clients */
void flowmgr_dbflow_clear_steer(void)
{
	int i;

	spin_lock_bh(&hash_lock);
	for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
		struct flowmgr_db_entry *f;
		struct hlist_node *n;

		hlist_iterate_safe(f, n, &hash[i], hlist) {
			flowmgr_dbflow_clear_steer_stats(f);
		}
	}
	spin_unlock_bh(&hash_lock);
}

int flowmgr_dbflow_clear_stats(struct flowmgr_db_entry *db)
{
	atomic_set(&db->rxflows, 0);
	atomic_set(&db->totrxflows, 0);
	atomic_set(&db->txflows, 0);
	atomic_set(&db->tottxflows, 0);
	return 0;
}

/* Clear flow stats of client by specific address */
int flowmgr_dbflow_clear_stats_by_addr(unsigned char *addr)
{
	struct flowmgr_db_entry *db;
	struct hlist_head *head = &hash[flowmgr_mac_hash(addr)];

	if (!is_valid_ether_addr(addr))
		return -EINVAL;

	spin_lock_bh(&hash_lock);
	db = db_find(head, addr);
	spin_unlock_bh(&hash_lock);
	if (!db)
		return -ENOMEM;

	flowmgr_dbflow_clear_stats(db);

	return 0;
}

/* Clear flow stats of all clients from specific network device */
int flowmgr_dbflow_clear_stats_by_port(int iif)
{
	int i;

	spin_lock_bh(&hash_lock);
	for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
		struct hlist_node *h, *g;

		hlist_for_each_safe(h, g, &hash[i]) {
			struct flowmgr_db_entry *f;

			f = hlist_entry(h, struct flowmgr_db_entry, hlist);
			if (f->iif != iif)
				continue;

			flowmgr_dbflow_clear_stats(f);
		}
	}
	spin_unlock_bh(&hash_lock);
	return 0;
}

/* Clear flow stats of all clients */
void flowmgr_dbflow_clear(void)
{
	int i;

	spin_lock_bh(&hash_lock);
	for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
		struct flowmgr_db_entry *f;
		struct hlist_node *n;

		hlist_iterate_safe(f, n, &hash[i], hlist) {
			flowmgr_dbflow_clear_stats(f);
		}
	}
	spin_unlock_bh(&hash_lock);
}

u32 flowmgr_dbflow_get_accel(struct flowmgr_db_entry *db)
{
	return db ? db->accel : 0;
}

int flowmgr_dbflow_set_accel(struct flowmgr_db_entry *db, u8 accel)
{
	pr_debug("FLOWMGR: %pM flow acceleration enable changed from %s -> %s\n",
		 db->addr,
		 db->accel ? "Y" : "N",
		 accel ? "Y" : "N");
	db->accel = accel;
	return 0;
}

/* Set flow accel of client by specific address */
int flowmgr_dbflow_set_accel_by_addr(unsigned char *addr, u8 accel)
{
	struct flowmgr_db_entry *db;
	struct hlist_head *head = &hash[flowmgr_mac_hash(addr)];

	if (!is_valid_ether_addr(addr))
		return -EINVAL;

	spin_lock_bh(&hash_lock);
	db = db_find(head, addr);
	if (!db)
		db = db_create(head, 0, addr);
	spin_unlock_bh(&hash_lock);
	if (!db)
		return -ENOMEM;

	flowmgr_dbflow_set_accel(db, accel);

	return 0;
}

/* Set flow accel of all clients from specific network device */
int flowmgr_dbflow_set_accel_by_port(int iif, u8 accel)
{
	int i;

	spin_lock_bh(&hash_lock);
	for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
		struct hlist_node *h, *g;

		hlist_for_each_safe(h, g, &hash[i]) {
			struct flowmgr_db_entry *f;

			f = hlist_entry(h, struct flowmgr_db_entry, hlist);
			if (f->iif != iif)
				continue;

			flowmgr_dbflow_set_accel(f, accel);
		}
	}
	spin_unlock_bh(&hash_lock);
	return 0;
}

/* Set flow accel for all clients */
void flowmgr_dbflow_set_accel_all(u8 accel)
{
	int i;

	spin_lock_bh(&hash_lock);
	for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
		struct flowmgr_db_entry *f;
		struct hlist_node *n;

		hlist_iterate_safe(f, n, &hash[i], hlist) {
			flowmgr_dbflow_set_accel(f, accel);
		}
	}
	spin_unlock_bh(&hash_lock);
}

int flowmgr_dbflow_set_steer(struct flowmgr_db_entry *db, u8 steer)
{
	pr_debug("FLOWMGR: %pM flow steering enable changed from %s -> %s\n",
		 db->addr,
		 db->steer ? "Y" : "N",
		 steer ? "Y" : "N");
	db->steer = steer;
	return 0;
}

/* Set flow steering of client by specific address */
int flowmgr_dbflow_set_steer_by_addr(unsigned char *addr, u8 steer)
{
	struct flowmgr_db_entry *db;
	struct hlist_head *head = &hash[flowmgr_mac_hash(addr)];

	if (!is_valid_ether_addr(addr))
		return -EINVAL;

	spin_lock_bh(&hash_lock);
	db = db_find(head, addr);
	if (!db)
		db = db_create(head, 0, addr);
	spin_unlock_bh(&hash_lock);
	if (!db)
		return -ENOMEM;

	flowmgr_dbflow_set_steer(db, steer);

	return 0;
}

/* Set flow steering of all clients from specific network device */
int flowmgr_dbflow_set_steer_by_port(int iif, u8 steer)
{
	int i;

	spin_lock_bh(&hash_lock);
	for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
		struct hlist_node *h, *g;

		hlist_for_each_safe(h, g, &hash[i]) {
			struct flowmgr_db_entry *f;

			f = hlist_entry(h, struct flowmgr_db_entry, hlist);
			if (f->iif != iif)
				continue;

			flowmgr_dbflow_set_steer(f, steer);
		}
	}
	spin_unlock_bh(&hash_lock);
	return 0;
}

/* Set flow steering for all clients */
void flowmgr_dbflow_set_steer_all(u8 steer)
{
	int i;

	spin_lock_bh(&hash_lock);
	for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
		struct flowmgr_db_entry *f;
		struct hlist_node *n;

		hlist_iterate_safe(f, n, &hash[i], hlist) {
			flowmgr_dbflow_set_steer(f, steer);
		}
	}
	spin_unlock_bh(&hash_lock);
}
