 /****************************************************************************
 *
 * Broadcom Proprietary and Confidential.
 * (c) 2015-2019 Broadcom. All rights reserved.
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************/

#ifndef __PROC_CMD__
#define __PROC_CMD__
#include <linux/version.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/uaccess.h>
#include <linux/sched.h>
#include <linux/seq_file.h>

#define PROC_CMD_STRING_SIZE	200
#define PROC_CMD_ARGS_MAX	20
#define PROC_CMD_TABLE_SIZE_MAX	40

#define pr_seq(s, ...) \
do { \
	if (s) \
		seq_printf(s,  __VA_ARGS__); \
	else \
		pr_emerg(__VA_ARGS__); \
} while (0)

#define pr_seq_cont(s, ...) \
do { \
	if (s) \
		seq_printf(s,  __VA_ARGS__); \
	else \
		pr_cont(__VA_ARGS__); \
} while (0)

#define PROC_CMD_INIT(str, func) { .name = str, .do_command = func, .do_help = func##_help }
#define PROC_CMD_DATA_INIT(str, func) { .name = str, .do_command_data = func, .do_help = func##_help }

struct proc_cmd_ops {
	const char *name;
	int (*do_command)(int argc, char *argv[]);
	int (*do_command_data)(void *data, int argc, char *argv[]);
	void (*do_help)(char *str);
};

struct proc_cmd_table {
	const char *module_name;
	int size;
	struct proc_cmd_ops *ops;
	void *data;
	void *data_seq_read;
	struct mutex lock;
};

static int proc_cmd_open(struct inode *inode, struct file *f)
{
	int ret;
	struct proc_cmd_table *table = PDE_DATA(inode);

	ret = seq_open(f, table->data_seq_read);
	if (ret == 0)
		((struct seq_file *)f->private_data)->private = table;
	return ret;
};

static ssize_t proc_cmd_write(struct file *file, const char __user *buffer,
			  size_t count, loff_t *pos)
{
	char cmd_line[PROC_CMD_STRING_SIZE];
	char *ptr = cmd_line;
	char *argv[PROC_CMD_ARGS_MAX];
	int  i, argc = 0;
	struct proc_cmd_table *table;
	struct seq_file *seq;
	seq = (struct seq_file *) file->private_data;
	table = (struct proc_cmd_table *) seq->private;
	if ((count <= 1) || (buffer == NULL))
		goto help;
	if (count > PROC_CMD_STRING_SIZE)
		count = PROC_CMD_STRING_SIZE;
	memset(cmd_line, 0, PROC_CMD_STRING_SIZE);
	if (copy_from_user(cmd_line, buffer, count))
		return count;

	if (table && table->module_name)
		pr_debug("%s [pid-%d] [cpu-%d]: proc cmd - %s",
				table->module_name, current->pid,
				task_cpu(current), cmd_line);

	/* Parse command line */
	while ((argc < PROC_CMD_ARGS_MAX) &&
	       ((argv[argc] = strsep(&ptr, " ")) != NULL)) {
		if ((argv[argc][0] != 0x00) &&
		    (argv[argc][0] != 0x0A)) {
			/* Ignore white spaces and newline*/
			argc++;
		}
	}

	/* last argument may contain newline, remove it */
	if (argc) {
		ptr = strstr(argv[argc-1], "\n");
		if (ptr)
			*ptr = 0x00;
	}

	if (!argc)
		goto help;

	if (!table)
		return count;

	for (i = 0; table && i < table->size; i++) {
		if (strcasecmp(table->ops[i].name, argv[0]) == 0) {
			mutex_lock(&table->lock);
			if (table->ops[i].do_command)
				table->ops[i].do_command(argc, argv);
			else if (table->ops[i].do_command_data)
				table->ops[i].do_command_data(table->data, argc, argv);
			mutex_unlock(&table->lock);
			goto done;
		}
	}
help:
	if (table) {
		pr_err("List of Commands:\n");
		for (i = 0; i < table->size; i++) {
			if (table->ops[i].do_help)
				table->ops[i].do_help(" ");
			else
				pr_alert("  %s\n", table->ops[i].name);
			pr_alert("--------------------------------\n");
		}
	}
done:
	return count;

}

static ssize_t proc_cmd_read(struct file *file, char __user *buffer,
			     size_t count, loff_t *pos)
{
	int i;
	struct proc_cmd_table *table;
	struct seq_file *seq;
	seq = (struct seq_file *) file->private_data;
	table = (struct proc_cmd_table *) seq->private;
	if (table) {
		pr_alert("List of Commands:\n");
		for (i = 0; i < table->size; i++) {
			if (table->ops[i].do_help)
				table->ops[i].do_help(" ");
			else
				pr_alert("  %s\n", table->ops[i].name);
			pr_alert("--------------------------------\n");
		}
	}
	return 0;
}

static int proc_cmd_release(struct inode *inode, struct file *f)
{
	return seq_release(inode, f);
};

static const struct file_operations proc_cmd_fops = {
	.owner    = THIS_MODULE,
	.open     = proc_cmd_open,
	.read     = proc_cmd_read,
	.write    = proc_cmd_write,
	.llseek   = seq_lseek,
	.release  = proc_cmd_release,
};

static const struct file_operations proc_cmd_read_fops = {
	.owner    = THIS_MODULE,
	.open     = proc_cmd_open,
	.read     = seq_read,
	.write    = proc_cmd_write,
	.llseek   = seq_lseek,
	.release  = proc_cmd_release,
};

static inline
struct proc_dir_entry *proc_create_cmd(const char *name,
				       struct proc_dir_entry *parent,
				       struct proc_cmd_table *table)
{
	if (!table || !name || !parent)
		return NULL;
	if (table->size > PROC_CMD_TABLE_SIZE_MAX)
		return NULL;
	mutex_init(&table->lock);
	if (table->data_seq_read)
		return proc_create_data(name,
					S_IRUGO | S_IWUGO,
					parent,
					&proc_cmd_read_fops,
					table);
	else
		return proc_create_data(name,
					S_IRUGO | S_IWUGO,
					parent,
					&proc_cmd_fops,
					table);
}

static inline
struct file *procsys_file_open(const char *path, int flags, int rights)
{
	struct file *filp = NULL;
	mm_segment_t oldfs;
	int err = 0;

	oldfs = get_fs();
	set_fs(KERNEL_DS);
	filp = filp_open(path, flags, rights);
	set_fs(oldfs);
	if (IS_ERR(filp)) {
		err = PTR_ERR(filp);
		return NULL;
	}
	return filp;
}

static inline
void procsys_file_close(struct file *file)
{
	filp_close(file, NULL);
}

static inline
int procsys_file_read(struct file *file, unsigned long long offset,
		      unsigned char *data, unsigned int size)
{
	mm_segment_t oldfs;
	int ret;

	oldfs = get_fs();
	set_fs(KERNEL_DS);

	ret = kernel_read(file, data, size, &offset);

	set_fs(oldfs);
	return ret;
}

static inline
int procsys_file_read_string(unsigned char *path, unsigned char *data,
		      unsigned int len)
{
	struct file *f;
	int ret;
	f = procsys_file_open(path, O_RDONLY, 0);
	if (!f)
		return -ENOENT;
	ret = procsys_file_read(f, 0, data, len);
	if (ret > 0) {
		procsys_file_close(f);
		return 0;
	}

	procsys_file_close(f);
	return ret;
}

static inline
int procsys_file_write(struct file *file, unsigned long long offset,
		       unsigned char *data, unsigned int size)
{
	mm_segment_t oldfs;
	int ret;

	oldfs = get_fs();
	set_fs(KERNEL_DS);

	ret = kernel_write(file, data, size, &offset);

	set_fs(oldfs);
	return ret;
}

static inline
int procsys_file_write_string(unsigned char *path, unsigned char *data)
{
	struct file *f;
	int ret, len;
	f = procsys_file_open(path, O_WRONLY, 0);
	if (!f)
		return -ENOENT;
	len = strlen(data);
	ret = procsys_file_write(f, 0, data, len);
	if (ret > 0) {
		procsys_file_close(f);
		return 0;
	}

	procsys_file_close(f);
	return ret;
}
#endif
