 /****************************************************************************
 *
 * Broadcom Proprietary and Confidential.
 * (c) 2016 Broadcom. All rights reserved.
 * The term "Broadcom" refers to Broadcom Limited and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Jayesh Patel <jayesh.patel@broadcom.com>
 ****************************************************************************/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <proc_cmd.h>
#include "mdqm_dev.h"

#define PROC_DIR_NAME	"driver/"

static void cmd_show_help(char *str)
{
	pr_alert("%s show: Show MDQM Stats\n", str);
}

static int cmd_show(void *data, int argc, char *argv[])
{
	int i;
	struct mdqm_dev *mdev;
	mdev = (struct mdqm_dev *) data;
	pr_info("Name: %s\n", mdev->name);
	pr_info("Napi Weight: %d\n", mdev->napi.weight);
	pr_info("Int mask   : %d\n", mdev->int_mask);
	pr_info("refcnt     : %d\n", mdev->refcnt);
	pr_info("DQM Info:\n");
	pr_info(" Type : %s\n", (mdev->type == MDEV_TYPE_TX) ? "TX" : "RX");
	pr_info(" Dev  : %s\n", mdev->dqm_dev);
	pr_info(" Num | Pri | Size | Int Count  | Handle\n");
	for (i = 0; i < mdev->q_count; i++) {
		pr_info(" %03d | %03d | %04d | %10d | %p\n",
			mdev->q[i].num,
			mdev->q[i].priority, mdev->q[i].tok_size,
			mdev->q[i].count, mdev->q[i].handle);
	}

	if (mdev->refcnt) {
		pr_info("User Info:\n");
		pr_info(" Num | Name\n");
		for (i = 0; i < MDQM_MAX_USER; i++) {
			if (mdev->cb[i].handler)
				pr_info(" %03d | %s\n",
					i, mdev->cb[i].name);
		}
	}
	pr_info("\n");
	return 0;
}

static void cmd_napi_weight_help(char *str)
{
	pr_alert("%s napi_weight: Show/set NAPI weight for queue\n", str);
	pr_alert("%s  Show/Set NAPI weight for queue\n", str);
	pr_alert("%s   napi_weight\n", str);
	pr_alert("%s  Set NAPI weight for queue\n", str);
	pr_alert("%s   napi_weight <napi_weigh>t\n", str);
}

static int cmd_napi_weight(void *data, int argc, char *argv[])
{
	struct mdqm_dev *mdev;
	mdev = (struct mdqm_dev *) data;
	if (argc == 2) {
		int num;
		if (kstrtoint(argv[1], 0, &num) == 0) {
			mdev->napi.weight = num;
		}
	}
	dev_info(&(mdev->pdev->dev), "Napi Weight=%d\n", mdev->napi.weight);
	return 0;
}

static struct proc_cmd_ops command_entries[] = {
	PROC_CMD_DATA_INIT("show", cmd_show),
	PROC_CMD_DATA_INIT("napi_weight", cmd_napi_weight),
};

static struct proc_cmd_table command_table_tmpl = {
	.module_name = MDQM_MODULE_NAME,
	.size = ARRAY_SIZE(command_entries),
	.ops = command_entries
};

int mdqm_procfs_init(struct mdqm_dev *mdev)
{
	char str[32];
	snprintf(str, sizeof(str), "%s%s", PROC_DIR_NAME, mdev->name);
	mdev->proc_dir = proc_mkdir(str, NULL);
	if (mdev->proc_dir == NULL) {
		pr_warn("mdqm Warning: cannot create /proc/%s\n",
			str);
		return -1;
	}

	mdev->cmd_tbl = kzalloc(sizeof(struct proc_cmd_table), GFP_KERNEL);
	mdev->cmd_tbl->module_name = command_table_tmpl.module_name;
	mdev->cmd_tbl->size = command_table_tmpl.size;
	mdev->cmd_tbl->ops = command_table_tmpl.ops;
	mdev->cmd_tbl->data = (void *) mdev;
	proc_create_cmd("cmd", mdev->proc_dir, mdev->cmd_tbl);
	return 0;
}

void mdqm_procfs_exit(struct mdqm_dev *mdev)
{
	char str[32];
	if (mdev->proc_dir) {
		snprintf(str, sizeof(str), "%s%s", PROC_DIR_NAME, mdev->name);
		remove_proc_entry("cmd", mdev->proc_dir);
		kfree(mdev->cmd_tbl);
		remove_proc_entry(str, NULL);
	}
}
