 /****************************************************************************
  *
  * Broadcom Proprietary and Confidential. (c) 2020 Broadcom.
  * All rights reserved.
  * The term "Broadcom" refers to Broadcom Limited and/or its subsidiaries.
  *
  * Unless you and Broadcom execute a separate written software license
  * agreement governing use of this software, this software is licensed to
  * you under the terms of the GNU General Public License version 2 (the
  * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
  * the following added to such license:
  *
  * As a special exception, the copyright holders of this software give you
  * permission to link this software with independent modules, and to copy
  * and distribute the resulting executable under terms of your choice,
  * provided that you also meet, for each linked independent module, the
  * terms and conditions of the license of that module. An independent
  * module is a module which is not derived from this software. The special
  * exception does not apply to any modifications of the software.
  *
  * Notwithstanding the above, under no circumstances may you combine this
  * software in any way with any other Broadcom software provided under a
  * license other than the GPL, without Broadcom's express prior written
  * consent.
  *
  ****************************************************************************/

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <asm/byteorder.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/string.h>
#include <linux/uaccess.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/netfilter.h>
#include <linux/netfilter_ipv4.h>
#include <linux/netfilter_ipv6.h>
#include <linux/inet.h>
#include <linux/ip.h>
#include <linux/version.h>
#include <net/ip.h>
#include <net/ipv6.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>

MODULE_DESCRIPTION("Netfilter: dnswall implementation");
MODULE_LICENSE("GPL");
MODULE_AUTHOR("Broadcom Corporation");
MODULE_ALIAS("nf_dnswall");


#define CONFIG_TOKEN_MAX_SIZE  32
#define CONFIG_VALUE_MAX_SIZE  32
#define CONFIG_BUFFER_SIZE     70

#define DNS_SERVER_PORT 53

#define OFFENDING_SRV_ARRAY_SIZE 4

struct dnshdr {
	__be16 id;
#if defined(__LITTLE_ENDIAN_BITFIELD)
	__be16 rd : 1;
	__be16 tc : 1;
	__be16 aa : 1;
	__be16 opcode : 4;
	__be16 qr : 1;
	__be16 rcode : 4;
	__be16 cd : 1;
	__be16 ad : 1;
	__be16 z : 1;
	__be16 ra : 1;
#elif defined(__BIG_ENDIAN_BITFIELD)
	__be16 qr : 1;
	__be16 opcode : 4;
	__be16 aa : 1;
	__be16 tc : 1;
	__be16 rd : 1;
	__be16 ra : 1;
	__be16 z : 1;
	__be16 ad : 1;
	__be16 cd : 1;
	__be16 rcode : 4;
#else
#error "Adjust your <asm/byteorder.h> defines"
#endif
	__be16 qdcount;
	__be16 ancount;
	__be16 nscount;
	__be16 arcount;
};

struct dnsrr {
	__be16 type;
	__be16 class;
	__be32 ttl;
	__be16 rdlength;
} __attribute__((packed));	/* required for some archs */

struct ipv6_address_pref {
	struct in6_addr network;
	__u8 prefix;
};

enum {
	A_TYPE = 1,
	AAAA_TYPE = 28,
};

enum {
	IN_CLASS = 1,
};

/* Configuration and Global variables */
static struct proc_dir_entry *proc_ent = NULL;
static int dnswall_cfg_enable = 1;
static int dnswall_cfg_skip_private_srv = 0;
static int ipv4_priv_addr_drop_count = 0;
static int ipv6_priv_addr_drop_count = 0;


/* Functions */
static __u8 * skip_name_label(__u8 *paddr) {
	__u8 * ptr;
	int len, tlen;

	ptr = paddr;
	/* check for compressed format */
	if ((*ptr & 0xc0) == 0xc0)
		return (ptr + 2);

	/* Maxium 255 bytes for name */
	tlen = 0;
	while (*ptr != 0) {
		len = (*ptr) + 1;
		tlen += len;
		ptr += len;
		/* Check if name exceeds 255 bytes */
		if (tlen >= 255)
			return NULL;
	}
	return (ptr + 1);
}

static bool is_ipv4_private_address(const __u8 *paddr)
{
	int i;
	const struct {
		__be32 network;
		__be32 mask;
	} private_networks[] = {
		{ 0x00000000, 0xFFFFFFFF },   // 0.0.0.0/32 invalid
		{ 0x7F000000, 0xFF000000 },   // 127.0.0.0/8 loopback/node
		{ 0xA9FE0000, 0xFFFF0000 },   // 169.254.0.0/16 link-local
		{ 0xC0A80000, 0xFFFF0000 },   // 192.168.0.0/16 private
		{ 0xAC100000, 0xFFF00000 },   // 172.16.0.0/12 private
		{ 0x0A000000, 0xFF000000 },   // 10.0.0.0/8 private
	};
	__u32 addr = ntohl(*(__be32*)paddr);

	for (i = 0; i < sizeof(private_networks)/sizeof(private_networks[0]); i++)
		if ((addr & private_networks[i].mask) == private_networks[i].network)
			return true;
	return false;
}

static bool is_ipv6_private_address(const __u8 *paddr)
{
	int i;
	const struct ipv6_address_pref private_networks[] = {
		{ in6addr_any, 128 },                                        // ::0/128 unspecified
		{ in6addr_loopback, 128 },                                   // ::1/128 loopback
		{ { { { 0xfe,0x80,0,0,0,0,0,0,0,0,0,0,0,0,0,0 } } }, 10 },   // fe80::/10 link-local
		{ { { { 0xfc,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0 } } }, 7 },       // fc00::/7 ULAs
	};

	const struct ipv6_address_pref ipv4_mapped = { { { { 0,0,0,0,0,0,0,0,0,0,0xff,0xff,0,0,0,0 } } }, 96 }; // ::ffff:0:0/96 ipv4 mapped
	const struct ipv6_address_pref depr_ipv4_mapped = { { { { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0 } } }, 96 }; // ::/96 depreciated ipv4 mapped

	struct in6_addr *addr = (struct in6_addr *)paddr;

	for (i = 0; i < sizeof(private_networks)/sizeof(private_networks[0]); i++)
		if (ipv6_prefix_equal(addr, &private_networks[i].network, private_networks[i].prefix))
			return true;

	if (ipv6_prefix_equal(addr, &ipv4_mapped.network, ipv4_mapped.prefix) ||
	    ipv6_prefix_equal(addr, &depr_ipv4_mapped.network, depr_ipv4_mapped.prefix))
		return is_ipv4_private_address(paddr + 12);

	return false;
}



static
#if LINUX_VERSION_CODE > KERNEL_VERSION(3, 17, 0)
unsigned int nf_dnswall(void *priv,
			     struct sk_buff *skb,
			     const struct nf_hook_state *state)
#else
unsigned int nf_dnswall(const struct nf_hook_ops *ops,
			     struct sk_buff *skb,
			     const struct net_device *in,
			     const struct net_device *out,
			     int (*okfn)(struct sk_buff *))
#endif
{
	const __be16 *pport;
	__be16 _ports[2];    /* source and dest port */
	__u16 srcport;
	__u8 *psrv_addr;

	__be16 eth_type;
	const struct iphdr *iph;
	struct iphdr _iph;
	const struct ipv6hdr *ip6h;
	struct ipv6hdr _ip6h;
	__u8 protocol;
	int frag_off;
	__u8 *ptr;

	__u8 *dns;
	int offset_pdns;
	int dns_len;

	const struct dnshdr *pdh;
	__u16 qdcount;
	__u16 ancount;

	struct dnsrr rr;
	__u16 rr_type;
	__u16 rr_class;
	__u16 rr_rdlen;
	__u8 *paddr;

	int ret;

	ret = NF_ACCEPT;
	dns = NULL;

	if (!dnswall_cfg_enable)
		goto out;   /* filter is disable, just accept  */

	eth_type = ntohs(eth_hdr(skb)->h_proto);
	if (eth_type == ETH_P_IP) {
		iph = skb_header_pointer(skb, 0, sizeof(_iph), &_iph);
		if (iph == NULL) {
			ret = NF_ACCEPT;   /* ipv4 header is too small */
			goto out;
		}
		psrv_addr = (__u8*)&(iph->saddr);
		protocol = iph->protocol;
		frag_off = ntohs(iph->frag_off) & IP_OFFSET;
	}
#if IS_ENABLED(CONFIG_IPV6)
	else if (eth_type == ETH_P_IPV6) {
		__u8 nexthdr;
		__be16 _frag_off;
		int offset_ph;

		ip6h = skb_header_pointer(skb, 0, sizeof(_ip6h), &_ip6h);
		if (ip6h == NULL) {
			ret = NF_ACCEPT;  /* ipv6 header is too small */
			goto out;
		}
		nexthdr = ip6h->nexthdr;
		offset_ph = ipv6_skip_exthdr(skb, sizeof(_ip6h), &nexthdr, &_frag_off);
		if (offset_ph == -1) {
			ret = NF_ACCEPT;  /* missing ipv6 protocol header */
			goto out;
		}
		psrv_addr = (__u8*)&(ip6h->saddr);
		protocol = nexthdr;
		frag_off = ntohs(_frag_off) & IP6_OFFSET;
	}
#endif
	else
		goto out;   /* unknown ethernet type, just accept */

	if (frag_off != 0)
		goto out;   /* not from first fragment, just accept */

	if ((protocol != IPPROTO_TCP) && (protocol != IPPROTO_UDP))
		goto out;   /* not TCP or UDP, just accept */

	pport = skb_header_pointer(skb, skb_transport_offset(skb), sizeof(_ports), _ports);
	if (pport == NULL) {
		ret = NF_ACCEPT;  /* TCP/UDP header is too small */
		goto out;
	}

	srcport = ntohs(pport[0]);
	/* If source port is not 53 (from DNS server) */
	if (srcport != DNS_SERVER_PORT)
		goto out;   /* not a dns packet, just accept */

	/* If DNS server has a private address, don't filter
	 * The Private DNS server will most likely answer with private
	 * address. */
	if (dnswall_cfg_skip_private_srv) {
		if (eth_type == ETH_P_IP) {
			if (is_ipv4_private_address(psrv_addr)) {
				goto out;  /* coming from a private server, just accept */
			}
		}
		else if(eth_type == ETH_P_IPV6) {
			if (is_ipv6_private_address(psrv_addr)) {
				goto out;  /* coming from a private server, just accept */
			}
		}
	}

	if (protocol == IPPROTO_UDP)
		offset_pdns = skb_transport_offset(skb) + sizeof(struct udphdr);
	else
		offset_pdns = skb_transport_offset(skb) + tcp_hdrlen(skb);

	/* DNS payload length */
	dns_len = skb->len - offset_pdns;
	if (dns_len <= 0)  /* Skip TCP SYN/FIN message */
		goto out;   /* no DNS payload, just accept */

	/* Allocate buffer if needed - packet is spreaded out in fragments */
	if (unlikely(skb_is_nonlinear(skb))) {
		dns = kmalloc(dns_len, GFP_KERNEL);
		if (dns == NULL)
			goto out;  /* not enough resource to process, just accept */
	}

	/* Fetch or copy the entire DNS message */
	ptr = skb_header_pointer(skb, offset_pdns, dns_len, dns);
	if (ptr == NULL)
		goto out;  /* Not the entire packet, just accept */

	/* For DNS over TCP, check and skip the 2 bytes length field */
	if (protocol == IPPROTO_TCP) {
		if (ntohs(*(__u16*)ptr) != (dns_len-2)) {
			goto out;  /* DNS response is splitted, just accept */
		}
		ptr += 2;
	}

	/* Parse the DNS header */
	pdh = (struct dnshdr *) ptr;
	qdcount = ntohs(pdh->qdcount);
	ancount = ntohs(pdh->ancount);

	/* Only check if this is a reply, with question and answer */
	if ((pdh->qr == 1) && qdcount > 0 && ancount > 0) {
		ptr = (__u8 *)(pdh + 1);

		/* skip all the questions */
		while (qdcount) {
			ptr = skip_name_label(ptr);
			if (unlikely(ptr == NULL)) {
				pr_debug("DNS name is invalid.\n");
				ret = NF_DROP;  /* invalid name */
				goto out;
			}
			ptr += 4; /* skip next 4 bytes - type and class */
			qdcount--;
		}

		/* check each answer */
		while (ancount) {
			ptr = skip_name_label(ptr);
			if (unlikely(ptr == NULL)) {
				pr_debug("DNS name is invalid.\n");
				ret = NF_DROP;  /* invalid packet */
				goto out;
			}

			/* copy out answer recorder, aligning the data */
			memcpy(&rr, ptr, sizeof(rr));
			paddr = ptr + sizeof(rr);
			rr_type = ntohs(rr.type);
			rr_class = ntohs(rr.class);
			rr_rdlen = ntohs(rr.rdlength);
			/* IPv4 record in answer */
			if (rr_type == A_TYPE &&
			    rr_class == IN_CLASS &&
			    rr_rdlen == sizeof(struct in_addr) &&
			    is_ipv4_private_address(paddr)) {
				ipv4_priv_addr_drop_count++;
				ret = NF_DROP;  /* rebind attack */
				break;
			}
			/* IPv6 record in answer */
			if (rr_type == AAAA_TYPE &&
			    rr_class == IN_CLASS &&
			    rr_rdlen == sizeof(struct in6_addr) &&
			    is_ipv6_private_address(paddr)) {
				ipv6_priv_addr_drop_count++;
				ret = NF_DROP;  /* rebind attack */
				break;
			}
			ptr += sizeof(rr) + ntohs(rr.rdlength);
			ancount--;
		}
	}

	if (ret == NF_DROP) {
		char srvaddr[INET6_ADDRSTRLEN];
		char ansaddr[INET6_ADDRSTRLEN];

		snprintf(srvaddr, sizeof(srvaddr),
			 (eth_type == ETH_P_IP) ? "%pI4" : "%pI6c", psrv_addr);
		snprintf(ansaddr, sizeof(ansaddr),
			 (rr_type == A_TYPE) ? "%pI4" : "%pI6c", paddr);

		pr_warn("Possible DNS rebind attack, dropping DNS reply from %s - answer address %s\n",
			srvaddr, ansaddr);
	}
out:
	if (dns) kfree(dns);  /* free buffer if needed */
	return ret;
}


static struct nf_hook_ops nf_dnswall_ops[] __read_mostly = {
	{
		.pf       = NFPROTO_IPV4,
		.priority = NF_IP_PRI_FILTER,
		.hooknum  = NF_INET_PRE_ROUTING,
		.hook     = nf_dnswall,
	},
#if IS_ENABLED(CONFIG_IPV6)
	{
		.pf       = NFPROTO_IPV6,
		.priority = NF_IP6_PRI_FILTER,
		.hooknum  = NF_INET_PRE_ROUTING,
		.hook     = nf_dnswall,
	},
#endif
};

static ssize_t proc_dnswall_write(struct file *file, const char __user
				  *ubuf,size_t count, loff_t *ppos)
{
	char buffer[CONFIG_BUFFER_SIZE] = {0};
	char token[CONFIG_TOKEN_MAX_SIZE] = {0};   /* 32 bytes buffer */
	char value[CONFIG_VALUE_MAX_SIZE] = {0};   /* 32 bytes buffer */
	if (count > CONFIG_BUFFER_SIZE)
		return -EINVAL;

	if (copy_from_user(buffer, ubuf, count))
		return -EFAULT;

	/* coverity [secure_coding] sscanf using exact precision specifiers*/
	if (sscanf(buffer, "%31s %31s", token, value) == 2) {
		if (strcmp(token, "enable") == 0) {
			/* coverity [secure_coding] sscanf using exact precision specifiers*/
			if (sscanf(value, "%31d", &dnswall_cfg_enable) == 1)
				return count;
		}
		else if (strcmp(token, "skip_private_srv") == 0) {
			/* coverity [secure_coding] sscanf using exact precision specifiers*/
			if (sscanf(value, "%31d", &dnswall_cfg_skip_private_srv) == 1)
				return count;
		}
	}
	return -EINVAL;
}

/* Return 0 means success, SEQ_SKIP ignores previous prints, negative for error. */
static int proc_dnswall_show(struct seq_file *s, void *v)
{
	seq_printf(s, "Configuration:\n");
	seq_printf(s, "\tenable %d\n", dnswall_cfg_enable);
	seq_printf(s, "\tskip_private_srv %d\n", dnswall_cfg_skip_private_srv);
	seq_printf(s, "Statistic:\n");
	seq_printf(s, "\tipv6_priv_addr_drop_count %d\n", ipv6_priv_addr_drop_count);
	seq_printf(s, "\tipv4_priv_addr_drop_count %d\n", ipv4_priv_addr_drop_count);
	return 0;
}

static int proc_dnswall_open(struct inode *inode, struct file *file)
{
    return single_open(file, proc_dnswall_show, NULL);
}

static struct file_operations proc_dnswall_ops = {
    .owner   = THIS_MODULE,
    .open    = proc_dnswall_open,
    .llseek  = seq_lseek,
    .read    = seq_read,
    .write   = proc_dnswall_write,
    .release = single_release,
};


static int __init init(void)
{
	int ret;
	dnswall_cfg_enable = 1;
	dnswall_cfg_skip_private_srv = 0;
	proc_ent = proc_create("driver/dnswall", 0660, NULL, &proc_dnswall_ops);
	ret = nf_register_net_hooks(&init_net, nf_dnswall_ops,
				ARRAY_SIZE(nf_dnswall_ops));

	return ret;
}

static void __exit fini(void)
{
	nf_unregister_net_hooks(&init_net, nf_dnswall_ops,
			    ARRAY_SIZE(nf_dnswall_ops));
	if (proc_ent)
		proc_remove(proc_ent);
}

module_init(init);
module_exit(fini);
