 /****************************************************************************
 *
 * Copyright (c) 2015-2018 Broadcom. All rights reserved
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************/
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/module.h>
#include <linux/miscdevice.h>
#include <linux/uaccess.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/phy.h>
#include <linux/kmod.h>
#include <linux/kernel.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/kthread.h>
#include <linux/delay.h>
#include <linux/of.h>
#include <linux/err.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/nand.h>
#include <linux/dma-mapping.h>
#include <linux/vmalloc.h>
#include <linux/proc_fs.h>
#include <linux/stat.h>

#include <linux/bcm_media_gw/itc_rpc/itc_rpc.h>
#include "vflash.h"

/**
* The vflash server driver consists of a set of RPC functions that are called
* (via ITC/RPC driver) from the client driver.	At init time, the driver
* creates a list of shareable partitions based on the entries that it reads
* from the device tree or arguments that are passed as part of module
* installation.
**/

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("Virtual Flash Server");
MODULE_AUTHOR("Gregg Kamosa");

#define CLASS_NAME "vfs"
static struct class *vfs_class;

#define PROCFS_NAME "vfs"

/*this param defines the size of the block cache, queried by boot assist*/
#define MAX_BLOCK_CACHES	512//2048
#define BLOCK_CACHE_BUFFER_SIZE 8192//4096

/*max number of clients talking to the server over itc/rpc */
#define MAX_CLIENTS 2

/*max partitions that can be shared from each client*/
#define MAX_PARTITION_SHARE 4

static char *part_names0[MAX_PARTITION_SHARE];
static int name_cnt0;

module_param_array(part_names0, charp, &name_cnt0, S_IRUGO);
MODULE_PARM_DESC(part_names,
	"List of partition names managed by vFlash for client 0");

static char *part_names1[MAX_PARTITION_SHARE];
static int name_cnt1;

module_param_array(part_names1, charp, &name_cnt1, S_IRUGO);
MODULE_PARM_DESC(part_names,
	"List of partition names managed by vFlash for client 1");

static int g_rpc_tunnel[MAX_CLIENTS];
static u8 *gp_bcache[MAX_CLIENTS];
static void *gp_bcache_phys[MAX_CLIENTS];

#define DRIVER_NAME "brcm-vflashserver"

/*//////////////// Prototypes //////////////////////////*/
static struct mtd_info *get_mtd_handle(int dqm_tunnel, struct vflash_msg *msg, int *idx);

static void put_mtd_handle(struct mtd_info *pmtd);

static long vfs_ctrl_ioctl(struct file *file, unsigned int cmd,
			unsigned long arg);
static int vfs_init(void);
static int vflash_server_init_dev(void);
static void vflash_server_remove(void);

static int vf_nor_read(int dqm_tunnel, struct vflash_msg *msg);

static int vf_nand_read(int dqm_tunnel, struct vflash_msg *msg);

static int vf_write(int dqm_tunnel, struct vflash_msg *msg);

static int vf_erase(int dqm_tunnel, struct vflash_msg *msg);

static int vf_nand_block_isbad(int dqm_tunnel, struct vflash_msg *msg);

static int vf_nand_block_markbad(int dqm_tunnel, struct vflash_msg *msg);

static int vf_info_device_cnt(int dqm_tunnel,
	struct vflash_msg_device_cnt *msg);

static int vf_info_device_info(int dqm_tunnel,
	struct vflash_msg_device_info *msg);

static int vf_info_partition_info(int dqm_tunnel,
	struct vflash_msg_partition_info *msg);

static int vf_info_partition_name(int dqm_tunnel,
	struct vflash_msg_partition_name *msg);

static int vf_info_client_register(int dqm_tunnel,
	struct vflash_msg_client_register *msg);

static rpc_function vf_service_table[] = {
	{(rpc_rx_handler)vf_nor_read,   		  0 },
	{(rpc_rx_handler)vf_nand_read,  		  0 },
	{(rpc_rx_handler)vf_write,  			  0 },
	{(rpc_rx_handler)vf_erase,  			  0 },
	{(rpc_rx_handler)vf_nand_block_isbad,     0 },
	{(rpc_rx_handler)vf_nand_block_markbad,   0 },
	{(rpc_rx_handler)vf_info_device_cnt,	  0 },
	{(rpc_rx_handler)vf_info_device_info,     0 },
	{(rpc_rx_handler)vf_info_partition_info,  0 },
	{(rpc_rx_handler)vf_info_partition_name,  0 },
	{(rpc_rx_handler)vf_info_client_register, 0 },
};

const struct file_operations vfs_ctrl_cdev_operations = {
	.owner		 = THIS_MODULE,
	.unlocked_ioctl	= vfs_ctrl_ioctl,
};

/* VFS control character device */
static struct miscdevice vfs_ctrl_cdev = {
	.minor = MISC_DYNAMIC_MINOR,
	.name = "vfs_ctrl",
	.fops = &vfs_ctrl_cdev_operations,
};

static long vfs_ctrl_ioctl(struct file *file, unsigned int cmd,
			    unsigned long arg)
{
	int err = 0;
	void __user *argp = (void __user *)arg;

	FUNC_TRACE(1);

	switch (cmd) {
	/* Attach an MTD device command */
	case VFS_BUFFSIZE:
	{
		u32 k_size;

		MSG_TRACEI("VFS_BUFFSIZE ioctl");
		err = copy_from_user(&k_size, argp, sizeof(u32));
		if (err) {
			err = -EFAULT;
			break;
		}
		k_size =  MAX_BLOCK_CACHES * BLOCK_CACHE_BUFFER_SIZE;
		/* @err contains UBI device number */
		err = put_user(k_size, (__user u32 *)argp);
		if (err) {
			err = -EFAULT;
			break;
		}

		break;
	}

	default:
		err = -ENOTTY;
		break;
	}

	FUNC_TRACE(0);
	return err;
}

/* Private Helper functions */
static struct mtd_info *get_mtd_handle(int dqm_tunnel, struct vflash_msg *msg, int *idx)
{
	struct mtd_info *mtd_info;
	char **names;
	FUNC_TRACE(1);

	if(dqm_tunnel == g_rpc_tunnel[0]) {
		names = part_names0;
		if(idx) *idx = 0;
	} else if(dqm_tunnel == g_rpc_tunnel[1]){
		names = part_names1;
		if(idx) *idx = 1;
	} else {
		MSG_TRACEE("Tunnel %d doesn't match any registered client\n", dqm_tunnel);
		FUNC_TRACE(0);
		return NULL;
	}

	MSG_TRACEI("idx: %d\n", idx);

	mtd_info = get_mtd_device_nm(names[msg->region]);
	if (IS_ERR(mtd_info)) {
		MSG_TRACEE("Can't retrieve requested partition: %s \n", names[msg->region]);
		FUNC_TRACE(0);
		return NULL;
	}

	FUNC_TRACE(0);
	return mtd_info;
}

static void put_mtd_handle(struct mtd_info *pmtd)
{
	FUNC_TRACE(1);

	put_mtd_device(pmtd);

	FUNC_TRACE(0);
}

extern void vflash_flush_buffer(void *va, u32 size);
extern void vflash_invalidate_buffer(void *va, u32 size);

#if 0
static void flush_buffer(void *va, u32 size)
{
	void *pa = (va - (void *)gp_bcache) + (void *)gp_bcache_phys;
	dma_sync_single_for_device(NULL, (dma_addr_t) pa, size, DMA_TO_DEVICE);
}

static void invalidate_buffer(void *va, u32 size)
{
	void *pa = (va - (void *)gp_bcache) + (void *)gp_bcache_phys;
	dma_sync_single_for_cpu(NULL, (dma_addr_t) pa, size, DMA_FROM_DEVICE);
}
#endif

#ifdef VFLASH_PROC_DEBUG
/* Debug code to allow data to be pushed through shared memory without
involving vflash/ITC/RPC/DQM */
#ifdef USE_SDMA_MAP
dma_addr_t debug_dma_addr;
int
vfs_proc_read(char *buffer,
	      char **buffer_location,
	      off_t offset, int buffer_length, int *eof, void *data)
{
	int ret;
	if (offset)
		return 0;
	MSG_TRACEI("vfs_proc_read (%s) called\n", PROCFS_NAME);

	dma_unmap_single(NULL, debug_dma_addr, (1024*1024), DMA_TO_DEVICE);

	*eof = 1;
	return 1;
}

 #define PATTERN1 0xdeadbeef
 #define PATTERN2 0xcafebabe
int vfs_proc_write(struct file *file, const char *buffer, unsigned long count,
		   void *data)
{
	int i;
	static int trial;

	MSG_TRACEI("vfs_proc_write (%s) called\n", PROCFS_NAME);

	for (i = 0; i < (1024*1024)/4; i++)
		((u32 *)gp_bcache)[i] = trial ? PATTERN2 : PATTERN1;

	MSG_TRACEN("First value: 0x%x\n", ((u32 *)gp_bcache)[0]);

	for (i = 0; i < (1024*1024)/4; i++) {
		if (trial == 0) {
			if (((u32 *)gp_bcache)[i] != PATTERN1) {
				MSG_TRACEE("FAILS at [%d]: x%x\n", i, ((u32 *)gp_bcache)[i]);
				/*break;*/
			}
		} else {
			if (((u32 *)gp_bcache)[i] != PATTERN2) {
				MSG_TRACEE("FAILS at [%d]: x%x\n", i, ((u32 *)gp_bcache)[i]);
				/*break;*/
			}
		}

		if (i % (64*1024) == 0)
			MSG_TRACEN("[%d]: x%x\n", i, ((u32 *)gp_bcache)[i]);
	}
	debug_dma_addr = dma_map_single(NULL, (void *)gp_bcache, (1024*1024),
						DMA_TO_DEVICE);

	if (dma_mapping_error(NULL, debug_dma_addr))
		MSG_TRACEE("\n\n\ndma mapping fails\n\n\n");
	else
		MSG_TRACEI("mapped to 0x%lx\n", debug_dma_addr);

	if (trial == 0)
		trial = 1;
	else
		trial = 0;

	return count;
}
#else
int
vfs_proc_read(char *buffer,
	      char **buffer_location,
	      off_t offset, int buffer_length, int *eof, void *data)
{
	int ret;

	if (offset)
		return 0;

	MSG_TRACEI("vfs_proc_read (%s) called\n", PROCFS_NAME);

	*eof = 1;
	return 1;
}

 #define PATTERN1 0xdeadbeef
 #define PATTERN2 0xcafebabe
int vfs_proc_write(struct file *file, const char *buffer, unsigned long count,
		   void *data)
{
	int i;
	static int trial;

	MSG_TRACEI("vfs_proc_write (%s) called\n", PROCFS_NAME);

	for (i = 0; i < (1024*1024)/4; i++)
		((u32 *)gp_bcache)[i] = trial ? PATTERN2 : PATTERN1;

	MSG_TRACEN("First value: 0x%x\n", ((u32 *)gp_bcache)[0]);

	for (i = 0; i < (1024*1024)/4; i++) {
		if (trial == 0) {
			if (((u32 *)gp_bcache)[i] != PATTERN1) {
				MSG_TRACEE("FAILS at [%d]: x%x\n", i, ((u32 *)gp_bcache)[i]);
				/*break;*/
			}
		} else {
			if (((u32 *)gp_bcache)[i] != PATTERN2) {
				MSG_TRACEE("FAILS at [%d]: x%x\n", i, ((u32 *)gp_bcache)[i]);
				/*break;*/
			}
		}

		if (i % (64*1024) == 0)
			MSG_TRACEN("[%d]: x%x\n", i, ((u32 *)gp_bcache)[i]);
	}
	vflash_flush_buffer((void *)gp_bcache, 1024*1024);
	if (trial == 0)
		trial = 1;
	else
		trial = 0;

	return count;
}

#endif
#endif

/* End Private Helper functions */

static int vf_nor_read(int dqm_tunnel, struct vflash_msg *msg)
{
#ifdef USE_SDMA_MAP
		dma_addr_t dma_addr = 0;
#endif
		u32 bytes, offset;
		size_t retlen;
		u8 *buffer;
		int rc, idx;
		struct mtd_info *pmtd = get_mtd_handle(dqm_tunnel, msg, &idx);

		FUNC_TRACE(1);

		if (pmtd == NULL)
			return 0;

		offset	 = vflash_msg_offset(msg);
		bytes	 = vflash_msg_pagesize(msg) * (1 << msg->pages);

		MSG_TRACEI("vf_nor_read->region %d offset 0x%08x size 0x%08x\n",
			msg->region, offset, bytes);
		MSG_TRACEI("offset: %px computed address: %px\n",	vflash_msg_buf(msg),
			(u8 *)((u32)vflash_msg_buf(msg) + gp_bcache[idx]));
		buffer	 = (u8 *)((u32)vflash_msg_buf(msg) + gp_bcache[idx]);
		rc = mtd_read(pmtd, offset, bytes, &retlen, buffer);
		if (rc >= 0) {
#ifdef USE_SDMA_MAP
			dma_addr = dma_map_single(NULL, (void *)buffer, bytes,
						DMA_TO_DEVICE);
#else
			vflash_flush_buffer((void *)buffer, bytes);
#endif
		} else {
			vflash_msg_set_error(msg, true);
			MSG_TRACEE("Error mtd_info->read\n");
		}
		rpc_send_reply(dqm_tunnel, (rpc_msg *)msg);

		put_mtd_handle(pmtd);

#ifdef USE_SDMA_MAP
		dma_unmap_single(NULL, dma_addr, bytes, DMA_TO_DEVICE);
#endif

		FUNC_TRACE(0);

		return 0;
}

static int vf_nand_read(int dqm_tunnel, struct vflash_msg *msg)
{
#ifdef USE_SDMA_MAP
	dma_addr_t dma_addr = 0;
#endif
	u32 bytes, offset;
	size_t retlen;
	u8 *buffer;
	int i, rc, idx;
	struct mtd_info *pmtd = get_mtd_handle(dqm_tunnel, msg, &idx);

	FUNC_TRACE(1);

	if (pmtd == NULL)
		return 0;

	offset	 = vflash_msg_offset(msg);
	bytes	 = vflash_msg_bytes(msg);

	MSG_TRACEI("vf_nand_read->region %d offset 0x%08x size 0x%08x\n",
		msg->region, offset, bytes);

	MSG_TRACEI("offset: %px computed address: %px\n",	vflash_msg_buf(msg),
		(u8 *)((u32)vflash_msg_buf(msg) + gp_bcache[idx]));
	buffer	 = (u8 *)((u32)vflash_msg_buf(msg) + gp_bcache[idx]);
	rc = mtd_read(pmtd, offset, bytes, &retlen, buffer);
	if (rc >= 0) {

#ifdef USE_SDMA_MAP
		dma_addr = dma_map_single(NULL, (void *)buffer, bytes,
							  DMA_TO_DEVICE);

		if (dma_mapping_error(NULL, dma_addr))
			MSG_TRACEE("\n\n\ndma mapping fails\n\n\n");
#else
		vflash_flush_buffer((void *)buffer, bytes);
#endif

	} else {
		vflash_msg_set_error(msg, true);
		MSG_TRACEE("Error mtd_info->read\n");
		MSG_TRACEE("region %d offset 0x%08x size 0x%08x\n",
			msg->region, offset, bytes);
	}
	rpc_send_reply(dqm_tunnel, (rpc_msg *)msg);

	put_mtd_handle(pmtd);

#ifdef USE_SDMA_MAP
	dma_unmap_single(NULL, dma_addr, bytes, DMA_TO_DEVICE);
#endif
	for (i = 0; i < 64; i += 4)
		MSG_TRACEN("%d: 0x%x 0x%x 0x%x 0x%x\n", i, buffer[i], buffer[i+1], buffer[i+2], buffer[i+3]);

	FUNC_TRACE(0);

	return 0;
}

static int vf_write(int dqm_tunnel, struct vflash_msg *msg)
{
#ifdef USE_SDMA_MAP
	dma_addr_t dma_addr;
#endif
	size_t retlen;
	u32 bytes, offset;
	u8 *buffer;
	int i, idx;
	struct mtd_info *pmtd = get_mtd_handle(dqm_tunnel, msg, &idx);

	FUNC_TRACE(1);

	MSG_TRACEI("msg->info: 0x%x\n", msg->info);

	if (pmtd == NULL)
		return 0;

	offset	 = vflash_msg_offset(msg);
	buffer	 = (u8 *)((u32)vflash_msg_buf(msg) + gp_bcache[idx]);
	bytes	 = vflash_msg_bytes(msg);

	MSG_TRACEI("vf_write->region %d offset 0x%08x size 0x%08x\n",
		msg->region, offset, bytes);
	MSG_TRACEI("offset: %px computed address: %px\n",	vflash_msg_buf(msg),
		(u8 *)((u32)vflash_msg_buf(msg) + gp_bcache[idx]));
#ifdef USE_SDMA_MAP
	dma_addr = dma_map_single(NULL, (void *)buffer, bytes,
						DMA_FROM_DEVICE);
#else
	vflash_invalidate_buffer((void *)buffer, bytes);
#endif

	for (i = 0; i < 64; i += 4)
		MSG_TRACEN("%d: 0x%x 0x%x 0x%x 0x%x\n", i, buffer[i], buffer[i+1], buffer[i+2], buffer[i+3]);
	if (mtd_write(pmtd, offset, bytes, &retlen, buffer) < 0) {
		vflash_msg_set_error(msg, true);
		MSG_TRACEE("Write failed\n");
	}

	rpc_send_reply(dqm_tunnel, (rpc_msg *)msg);

	put_mtd_handle(pmtd);

#ifdef USE_SDMA_MAP
	dma_unmap_single(NULL, dma_addr, bytes, DMA_FROM_DEVICE);
#endif

	FUNC_TRACE(0);

	return 0;
}

static int vf_nand_block_isbad(int dqm_tunnel, struct vflash_msg *msg)
{
	int ret = 0;
	u32 rawOffset = vflash_msg_offset(msg);
	struct mtd_info *pmtd = get_mtd_handle(dqm_tunnel, msg, NULL);

	FUNC_TRACE(1);

	if (pmtd == NULL)
		return 0;

	MSG_TRACEI("vf_nand_block_isbad: erase block: %d\n",
	(u16)(rawOffset/pmtd->erasesize));

	if (pmtd->_block_isbad) {
		ret = mtd_block_isbad(pmtd, rawOffset);
	} else {
		MSG_TRACEE("block_isbad not defined for this MTD\n");
		return -EFAULT;
	}

	if (ret) {
		MSG_TRACEI("block_isbad returns TRUE... block is bad!\n");
		vflash_msg_set_true(msg, true);
	} else {
		vflash_msg_set_true(msg, false);
	}

	rpc_send_reply(dqm_tunnel, (rpc_msg *)msg);

	put_mtd_handle(pmtd);

	FUNC_TRACE(0);

	return 0;
}

static int vf_nand_block_markbad(int dqm_tunnel, struct vflash_msg *msg)
{
	int ret = 0;
	u32 rawOffset = vflash_msg_offset(msg);
	struct mtd_info *pmtd = get_mtd_handle(dqm_tunnel, msg, NULL);

	FUNC_TRACE(1);

	if (pmtd == NULL)
		return 0;

	MSG_TRACEI("MEMMARKBAD Raw offset: %d\n", rawOffset);
	MSG_TRACEI("erase block: %d\n", (u16)(rawOffset/pmtd->erasesize));
	if (pmtd->_block_markbad) {
		ret = mtd_block_markbad(pmtd, rawOffset);
	} else {
		MSG_TRACEE("block_markbad not defined for this MTD\n");
		FUNC_TRACE(0);
		return -EFAULT;
	}

	if (!ret) {
		MSG_TRACEI("block_markbad success!\n");
		vflash_msg_set_true(msg, true);
	} else {
		MSG_TRACEE("block_markbad fail!\n");
		vflash_msg_set_error(msg, true);
	}
	rpc_send_reply(dqm_tunnel, (rpc_msg *)msg);

	put_mtd_handle(pmtd);

	FUNC_TRACE(0);

	return 0;
}

static int vf_info_device_cnt(int dqm_tunnel, struct vflash_msg_device_cnt *msg)
{
	FUNC_TRACE(1);

	if(dqm_tunnel == g_rpc_tunnel[0]) {
		msg->devicecnt = name_cnt0;
	} else if(dqm_tunnel == g_rpc_tunnel[1]){
		msg->devicecnt = name_cnt1;
	} else {
		MSG_TRACEE("Tunnel %d doesn't match any registered client\n", dqm_tunnel);
		FUNC_TRACE(0);
		return -1;
	}

	rpc_send_reply(dqm_tunnel, (rpc_msg *)msg);
	FUNC_TRACE(0);
	return 0;
}

static int vf_info_device_info(int dqm_tunnel, struct vflash_msg_device_info *msg)
{
	struct mtd_info *mtd_info;
	char **names;
	FUNC_TRACE(1);

	/* error checking */
	/*TBD:  is the partition and device index within range */

	/* use device # as index into partnames array*/
	MSG_TRACEI("%s dqm_tunnel: %d\n", __FUNCTION__, dqm_tunnel);
	MSG_TRACEI("msg->device: %d\n", msg->device);

	if(dqm_tunnel == g_rpc_tunnel[0]) {
		names = part_names0;
	} else if(dqm_tunnel == g_rpc_tunnel[1]){
		names = part_names1;
	} else {
		MSG_TRACEE("Tunnel %d doesn't match any registered client\n", dqm_tunnel);
		vflash_msg_device_info_set_error(msg);
		rpc_send_reply(dqm_tunnel, (rpc_msg *)msg);
		FUNC_TRACE(0);
		return 0;
	}

	mtd_info = get_mtd_device_nm(names[msg->device]);
	if (IS_ERR(mtd_info)) {
		MSG_TRACEE("Can't retrieve requested mtd: %s \n", names[msg->device]);
		vflash_msg_device_info_set_error(msg);
	} else {
		msg->type = ((mtd_info->type == MTD_NANDFLASH) || (mtd_info->type == MTD_MLCNANDFLASH)) ?
		kFlashTypeNand : kFlashTypeNor;
		MSG_TRACEI("mtd_info->type: %d, msg->type: %d\n", mtd_info->type, msg->type);
		msg->partition_cnt = 1;
		msg->first_partition = 0;
		msg->page_size = mtd_info->writesize;
		/* TBD: check for div/0 */
		msg->block_size_in_pages = ilog2((u32)mtd_info->erasesize/(u32)mtd_info->writesize);
		msg->size_in_blocks = (u32)mtd_info->size / mtd_info->erasesize;
		MSG_TRACEI("msg->page_size: %ld\n", msg->page_size);
		MSG_TRACEI("msg->block_size_in_pages: %ld\n", msg->block_size_in_pages);
		MSG_TRACEI("msg->size_in_blocks: %ld\n", msg->size_in_blocks);

		put_mtd_device(mtd_info);
	}
	rpc_send_reply(dqm_tunnel, (rpc_msg *)msg);

	FUNC_TRACE(0);
	return 0;
}

static int vf_erase(int dqm_tunnel, struct vflash_msg *msg)
{
	struct erase_info ei;
	int ret;
	struct mtd_info *pmtd = get_mtd_handle(dqm_tunnel, msg, NULL);

	FUNC_TRACE(1);

	if (pmtd == NULL) {
		FUNC_TRACE(0);
		return 0;
	}

	/*explicitly return success until otherwise indicated*/
	vflash_msg_set_error(msg, false);

	MSG_TRACEI("vf_erase block: %d\n", vflash_msg_block(msg));

	ei.addr = vflash_msg_block(msg) * pmtd->erasesize;
	ei.len = pmtd->erasesize;
	ret = mtd_erase(pmtd, &ei);

	if (ret != 0) {
		vflash_msg_set_error(msg, true);
		MSG_TRACEE("vf_erase failed block %08x region = %d\n",
				vflash_msg_block(msg), msg->region);
	}
	rpc_send_reply(dqm_tunnel, (rpc_msg *)msg);
	put_mtd_handle(pmtd);
	FUNC_TRACE(0);
	return 0;
}

static int vf_info_partition_info(int dqm_tunnel, struct vflash_msg_partition_info *msg)
{
	struct mtd_info *mtd_info;
	char **names;
	FUNC_TRACE(1);

	/* error checking */
	/*TBD:  is the partition and device index within range */

	/* use device # as index into partnames array*/
	MSG_TRACEI("%s dqm_tunnel: %d\n", __FUNCTION__, dqm_tunnel);
	MSG_TRACEI("msg->device: %d\n", msg->device);
	MSG_TRACEI("msg->partition: %d\n", msg->partition);

	if(dqm_tunnel == g_rpc_tunnel[0]) {
		names = part_names0;
	} else if(dqm_tunnel == g_rpc_tunnel[1]){
		names = part_names1;
	} else {
		MSG_TRACEE("Tunnel %d doesn't match any registered client\n", dqm_tunnel);
		vflash_msg_partition_info_set_error(msg);
		rpc_send_reply(dqm_tunnel, (rpc_msg *)msg);
		FUNC_TRACE(0);
		return 0;
	}

	mtd_info = get_mtd_device_nm(names[msg->device]);
	if (IS_ERR(mtd_info)) {
		MSG_TRACEE("Can't retrieve requested partition: %s \n", names[msg->device]);
		vflash_msg_partition_info_set_error(msg);
	} else {
		msg->start_block = 0;
		/* TBD: check for div/0 */
		msg->end_block = (u32)mtd_info->size / mtd_info->erasesize;
		msg->read_only = 0;
		/*in our case, region and partition are the same thing*/
		msg->region = msg->device;
		MSG_TRACEI("msg->end_block: %ld\n", msg->end_block);
		MSG_TRACEI("msg->device: %d\n", msg->device);
		MSG_TRACEI("msg->partition: %d\n", msg->partition);
		put_mtd_device(mtd_info);
	}
	rpc_send_reply(dqm_tunnel, (rpc_msg *)msg);
	FUNC_TRACE(0);
	return 0;
}

static int vf_info_partition_name(int dqm_tunnel,
	struct vflash_msg_partition_name *msg)
{
	struct mtd_info *pmtd = get_mtd_handle(dqm_tunnel, (struct vflash_msg *)msg, NULL);
	char tmp[11];

	FUNC_TRACE(1);
	if (!pmtd)
		return -1;

	if (pmtd->name) {
		snprintf(tmp, 11, "vf%d", msg->region);
		strncpy(msg->name, tmp, sizeof(tmp)-1);
	} else {
		vflash_msg_partition_name_set_error(msg);
		MSG_TRACEE("Error vf_info_partition_name: FlashDriverGetRegionName %d\n",
			msg->region);
	}
	put_mtd_handle(pmtd);
	rpc_send_reply(dqm_tunnel, (rpc_msg *)msg);
	FUNC_TRACE(0);
	return 0;
}

static int vf_info_client_register(int dqm_tunnel, struct vflash_msg_client_register *msg)
{
	unsigned long physaddr = (u32)msg->sharedmem_addr;
	unsigned long size = (u32)msg->sharedmem_size; /*in bytes*/

	FUNC_TRACE(1);

	if(msg->client_id == 0)
		g_rpc_tunnel[0] = dqm_tunnel;
	else if (msg->client_id == 1)
		g_rpc_tunnel[1] = dqm_tunnel;
	else {
		FUNC_TRACE(0);
		MSG_TRACEE("client ID %d out of bounds, exiting registration\n", msg->client_id);
		return -1;
	}

	gp_bcache[msg->client_id] = ioremap(physaddr, size);
	if (!gp_bcache[msg->client_id]) {
		FUNC_TRACE(0);
		MSG_TRACEE("ioremap fails on client %d at physaddr: 0x%lx", msg->client_id, physaddr);
		return -1;
	}

	MSG_TRACEI("Registering Client %d on tunnel: %d at physAddr: 0x%x size: 0x%x, maps to VA: 0x%x\n",
		msg->client_id, dqm_tunnel, msg->sharedmem_addr, msg->sharedmem_size, (u32)gp_bcache[msg->client_id]);

	gp_bcache_phys[msg->client_id] = (u8*)physaddr;

	rpc_send_reply(dqm_tunnel, (rpc_msg *)msg);
	FUNC_TRACE(0);
	return 0;
}

static int vfs_init()
{
	/*Do a basic error check here to ensure that all partitions specified in insmod exist*/
	int i, j, cnt;
	char **names;
	struct mtd_info *mtd_info;

	FUNC_TRACE(1);
	MSG_TRACEI("Validating shared partitions for clients 0, 1\n");
	MSG_TRACEI("name_cnt0: %d name_cnt1: %d\n", name_cnt0, name_cnt1);
	for (i = 0; i < MAX_CLIENTS; i++) {
		cnt = (i == 0) ? name_cnt0: name_cnt1;
		names = (i == 0) ? part_names0: part_names1;
		for (j = 0; j < cnt; j++) {
			/* See if names[j] exists in mtd */
			mtd_info = get_mtd_device_nm(names[j]);
			if (IS_ERR(mtd_info)) {
				MSG_TRACEE("Device %d: Can't retrieve requested partition: %s \n", i, names[j]);
				FUNC_TRACE(0);
				return -1;
			} else {
				put_mtd_device(mtd_info);
			}
		}
	}

	FUNC_TRACE(0);
	return 0;

}

static int vflash_server_init_dev()
{
	int ret;

	FUNC_TRACE(1);

	ret = vfs_init();
	if (ret < 0) {
		MSG_TRACEE("vflash_server_init_dev catastrophic ERROR in vfs_init\n");
		FUNC_TRACE(0);
		return ret;
	}

	/*We don't want these initialized to 0, as that's a valid tunnel_id*/
	g_rpc_tunnel[0] = -1;
	g_rpc_tunnel[1] = -1;

	/* TODO: determine number of clients supported, and do what is necessary here to register for each, possibly nothing required */
	rpc_register_functions(RPC_SERVICE_FLASH,
			vf_service_table,
			sizeof(vf_service_table)/sizeof(rpc_function));

	FUNC_TRACE(0);
	return 0;
}

static ssize_t vfs_buffsize_show(struct class *class,
				struct class_attribute *attr, char *buf)
{
	return snprintf(buf, (sizeof(u32)*2), "%u\n", (u32) (MAX_BLOCK_CACHES * BLOCK_CACHE_BUFFER_SIZE));
}

static struct class_attribute vfs_buffsize =
	__ATTR(buffsize, S_IRUGO, vfs_buffsize_show, NULL);

static int __init vflash_server_init(void)
{
	int err = 0;
#ifdef VFLASH_PROC_DEBUG
	struct proc_dir_entry *entry;
#endif
	FUNC_TRACE(1);

	/* Create base sysfs directory and sysfs files */
	vfs_class = class_create(THIS_MODULE, CLASS_NAME);
	if (IS_ERR(vfs_class)) {
		MSG_TRACEE("cannot create UBI class\n");
		goto out;
	}

	err = class_create_file(vfs_class, &vfs_buffsize);
	if (err) {
		MSG_TRACEE("cannot create sysfs file\n");
		goto out_class;
	}

	err = misc_register(&vfs_ctrl_cdev);
	if (err) {
		MSG_TRACEE("cannot register device\n");
		goto out_version;
	}

	err = vflash_server_init_dev();
	if (err) {
		goto out_register;
	}

#ifdef VFLASH_PROC_DEBUG
	entry = create_proc_entry(PROCFS_NAME, 0644, NULL);
	if (entry) {
		entry->read_proc = vfs_proc_read;
		entry->write_proc = vfs_proc_write;
	}
	MSG_TRACEI("vfs_procfile: %px\n", entry);
#endif

	FUNC_TRACE(0);
	return 0;
	out_register:
		misc_deregister(&vfs_ctrl_cdev);
	out_version:
		class_remove_file(vfs_class, &vfs_buffsize);
	out_class:
		class_destroy(vfs_class);
	out:
		MSG_TRACEE("VFS error: cannot initialize VFS, error %d", err);
		FUNC_TRACE(0);
		return err;

}
module_init(vflash_server_init);

static void __exit vflash_server_remove(void)
{
	FUNC_TRACE(1);

#ifdef VFLASH_PROC_DEBUG
	remove_proc_entry(PROCFS_NAME, NULL);
#endif

	misc_deregister(&vfs_ctrl_cdev);
	class_remove_file(vfs_class, &vfs_buffsize);
	class_destroy(vfs_class);

	FUNC_TRACE(0);
}
module_exit(vflash_server_remove);
