/****************************************************************************
*
* Broadcom Proprietary and Confidential. (c) 2016 Broadcom.
* All rights reserved.
* The term Broadcom refers to Broadcom Inc. and/or its subsidiaries.
*
* Unless you and Broadcom execute a separate written software license
* agreement governing use of this software, this software is licensed to
* you under the terms of the GNU General Public License version 2 (the
* "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
* the following added to such license:
*
* As a special exception, the copyright holders of this software give you
* permission to link this software with independent modules, and to copy
* and distribute the resulting executable under terms of your choice,
* provided that you also meet, for each linked independent module, the
* terms and conditions of the license of that module. An independent
* module is a module which is not derived from this software. The special
* exception does not apply to any modifications of the software.
*
* Notwithstanding the above, under no circumstances may you combine this
* software in any way with any other Broadcom software provided under a
* license other than the GPL, without Broadcom's express prior written
* consent.
*
****************************************************************************
* Author: Venky <venky.selvaraj@broadcom.com>
*****************************************************************************/
#include <linux/init.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/gpio/consumer.h>
#include <linux/stat.h>
#include <linux/string.h>
#include <linux/fs.h>
#include <linux/debugfs.h>
#include <linux/cdev.h>
#include <linux/of.h>
#include <linux/io.h>
#include <linux/types.h>
#include <uapi/linux/bcm_media_gw/wrfpwr/wrfpwr_api.h>
#include <linux/uaccess.h>
#include <linux/regulator/consumer.h>
#include <linux/slab.h>
#if defined(CONFIG_BCM_I2C_WBID)
#include <linux/wbid.h>
#endif
#if defined(CONFIG_BCM_MBOX)
#include <brcm_mbox.h>
#elif defined(CONFIG_BCM_PWR_RPC)
#include <brcm_pwr_rpc.h>
#endif
#include <linux/delay.h>
#include <linux/proc_fs.h>
#include <proc_cmd.h>
#include "wrfpwr.h"

static int node_count;
static int supplies;

struct dev_pwr_supply {
	struct list_head node;
	char name[32];
	struct regulator *regulator;
};

struct wrfpwr_priv {
	struct list_head list;
	struct device *dev;
	struct platform_device *pdev;
	struct task_struct *task;
	dev_t ndev;
	struct cdev c_dev;
	struct class *wrfpwr_char_class;
	struct device *wrfpwr_char_device;
	struct list_head	pwr_supplies;
};

static struct list_head priv_list;
static struct proc_dir_entry *proc_dir;
#define PROC_DIR		"driver/wrfpwr"
#define CMD_PROC_FILE	"cmd"

#define WIFI_RF_PARAM_INIT(index, proc_name, reg_name) {proc_name, reg_name},
wifi_rf_param_struct wifi_rf_param_list[] = {
	WIFI_RF_PARAM_LIST
};

#if defined(CONFIG_BCM_MBOX)
static int wrfpwr_mbox_event(
		struct notifier_block *this,
		unsigned long event, void *ptr)
{
	struct list_head *pos, *pos_priv;
	struct dev_pwr_supply *supply;
	int status = 0;
	struct brcm_mbox_info *states = ptr;
	struct wrfpwr_priv *priv;

	switch (event) {
	case MBOX_CHANGE_EVENT:
	{
		if (BATT_STATE(states->mbox[MBOX_BMU])!= BS_BATTERY) {
			pr_debug("Power not operated through battery\n");
			return status;
		}
		if (states->delay_battery_mode) {
			pr_debug("Power operated through battery, delay_battery_mode "
					 "enabled, so no action on WiFi-RF power control\n");
			return status;
		}
		list_for_each(pos_priv, &priv_list)
		{
			priv = list_entry(pos_priv, struct wrfpwr_priv, list);
			list_for_each(pos, &priv->pwr_supplies) {
				supply = list_entry(pos,
						struct dev_pwr_supply, node);
				if (!supply) {
					pr_err("Regulator Supply info is empty\n");
					status = -ENODEV;
					goto done;
				}
				if (regulator_is_enabled(supply->regulator)) {
					status =
					regulator_gpio_force_disable_atomic(
							supply->regulator);
					if (status) {
						pr_err("Unable to turn off %s supply.\n",
							supply->name);
						goto done;
					}
				}
			}
		}
		break;
	}

	default:
		status = -EOPNOTSUPP;
		goto done;
		break;
	}

done:
	pr_debug("<--\n");
	return status;
}

#elif defined(CONFIG_BCM_PWR_RPC)
static int wrfpwr_mbox_event(
		struct notifier_block *this,
		unsigned long event, void *ptr)
{
	struct list_head *pos, *pos_priv;
	struct dev_pwr_supply *supply;
	int status = 0;
	struct brcm_pwr_rpc_mbox_info *states = ptr;
	struct wrfpwr_priv *priv;

	switch (event) {
	case PWR_MBOX_CHANGE_EVENT:
	{
		if (BATT_STATE(states->mbox[PWR_MBOX_BS])!= BS_BATTERY) {
			pr_debug("Power not operated through battery\n");
			return status;
		}
		if (states->delay_battery_mode) {
			pr_debug("Power operated through battery, delay_battery_mode "
					 "enabled, so no action on WiFi-RF power control\n");
			return status;
		}
		list_for_each(pos_priv, &priv_list)
		{
			priv = list_entry(pos_priv, struct wrfpwr_priv, list);
			list_for_each(pos, &priv->pwr_supplies) {
				supply = list_entry(pos,
						struct dev_pwr_supply, node);
				if (!supply) {
					pr_err("Regulator Supply info is empty\n");
					status = -ENODEV;
					goto done;
				}
				if (regulator_is_enabled(supply->regulator)) {
					status =
					regulator_gpio_force_disable_atomic(
							supply->regulator);
					if (status) {
						pr_err("Unable to turn off %s supply.\n",
							supply->name);
						goto done;
					}
				}
			}
		}
		break;
	}

	default:
		status = -EOPNOTSUPP;
		goto done;
		break;
	}

done:
	pr_debug("<--\n");
	return status;
}
#endif

#if defined(CONFIG_BCM_MBOX) || defined(CONFIG_BCM_PWR_RPC)
static struct notifier_block wrfpwr_mbox_notifier = {
	.notifier_call  = wrfpwr_mbox_event,
};
#endif

int wifi_rf_regulator_pwr_cntrl(const char *name, bool state)
{
	struct list_head *pos, *pos_priv;
	struct dev_pwr_supply *supply;
	int status = 0;
	struct wrfpwr_priv *priv;

	list_for_each(pos_priv, &priv_list)
	{
		priv = list_entry(pos_priv, struct wrfpwr_priv, list);
		list_for_each(pos, &priv->pwr_supplies) {
			supply = list_entry(pos,
					struct dev_pwr_supply, node);
			if (!supply) {
				pr_err("WiFi RF regulator info is empty.\n");
				status = -ENODEV;
				goto done;
			}
			if (!strncmp(supply->name, name, strlen(supply->name))) {
				if (state) {
					if (regulator_is_enabled(supply->regulator)) {
						pr_debug("WiFi RF regulator %s is already enabled!\n",
							supply->name);
						status = 0;
						goto done;
					}
					pr_info("Enable WiFi RF regulator %s.\n",
							supply->name);
					status = regulator_enable(supply->regulator);
					if (status) {
						pr_debug("Failed to enable WiFi RF regulator %s.\n",
							supply->name);
						goto done;
					}
					if (regulator_is_enabled(supply->regulator))
						pr_debug("WiFi RF regulator %s was enabled successfully.\n",
							supply->name);
					goto done;
				} else {
					if (!regulator_is_enabled(supply->regulator)) {
						pr_debug("WiFi RF regulator %s is already disabled!\n",
							supply->name);
						status = 0;
						goto done;
					}
					pr_info("Disable WiFi RF regulator %s.\n",
							supply->name);
					status = regulator_disable(supply->regulator);
					if (status) {
						pr_debug("Failed to disable WiFi RF regulator %s.\n",
							supply->name);
						goto done;
					}
					if (!regulator_is_enabled(supply->regulator))
						pr_debug("WiFi RF regulator %s was disabled successfully.\n",
							supply->name);
					goto done;
				}

			}
		}
	}
done:
	pr_debug("<--\n");
	return status;
}EXPORT_SYMBOL(wifi_rf_regulator_pwr_cntrl);

static int wrfpwr_open(struct inode *inode, struct file *flp)
{
	struct wrfpwr_priv *priv; /* device information */
	pr_debug("-->\n");
	priv = container_of(inode->i_cdev, struct wrfpwr_priv, c_dev);
	flp->private_data = priv; /* for other methods */
	pr_debug("<--\n");
	return 0;
}

static int wrfpwr_release(struct inode *inode, struct file *flp)
{
	return 0;
}

static int wrfpwr_update_cntrl_info(void *arg,
			struct radio_regulator_info *regulator_info)
{
	struct wrfpwr_priv *priv = arg;
	struct list_head *pos;
	struct dev_pwr_supply *supply;
	int supply_cnt = 0;
	pr_debug("-->\n");
	list_for_each(pos, &priv->pwr_supplies) {
		supply = list_entry(pos, struct dev_pwr_supply, node);
		/* Update Name */
		strncpy(regulator_info->cntrl_info[supply_cnt].name,
				supply->name, sizeof(supply->name)-1);
		/* Update Regulator State */
		regulator_info->cntrl_info[supply_cnt].regulator_state =
			regulator_is_enabled(supply->regulator);
		supply_cnt++;
	}
	pr_debug("<--\n");
	return 0;
}
static int wrfpwr_regulator_cntrl(void *arg,
			struct radio_regulator_info *regulator_info)
{
	struct wrfpwr_priv *priv = arg;
	struct list_head *pos;
	struct dev_pwr_supply *supply;
	int status = 0, supply_cnt = 0;
	pr_debug("-->\n");

	if (regulator_info->num_regulators > supplies) {
		dev_err(priv->dev,
				"Number of regulators to control greater than supplies.\n");
		status =  -EINVAL;
		goto done;
	}
	list_for_each(pos, &priv->pwr_supplies) {
		supply = list_entry(pos, struct dev_pwr_supply, node);
		/* Check for Regulator Supply info */
		if (!supply) {
			dev_err(priv->dev,
					"Regulator Supply info is empty\n");
			status = -ENODEV;
			goto done;
		}
		for (supply_cnt = 0;
			supply_cnt < regulator_info->num_regulators;
			 supply_cnt++) {
			/* Take action on regulator power supply */
			if (strcmp(regulator_info->cntrl_info[supply_cnt].name,
				supply->name))
					continue;

			if (
		regulator_info->cntrl_info[supply_cnt].regulator_state) {
				if (regulator_is_enabled(supply->regulator)) {
					dev_dbg(priv->dev,
							"Regulator %s supply already Enabled!\n",
							supply->name);
					continue;
				}
				dev_info(priv->dev,
							"Enable %s supply.\n",
							supply->name);
				status = regulator_enable(supply->regulator);
				if (status) {
					dev_err(priv->dev,
							"Unable to turn on %s supply.\n",
							supply->name);
					goto done;
				}
				if (regulator_is_enabled(supply->regulator))
					dev_dbg(priv->dev,
							"Enabled regulator(%s).\n",
							supply->name);
			} else {
				if (!regulator_is_enabled(supply->regulator)) {
					dev_dbg(priv->dev,
							"Regulator %s supply already Disabled!\n",
							supply->name);
					continue;
				}
				status = regulator_disable(supply->regulator);
				if (status) {
					dev_err(priv->dev,
							"Unable to turn off %s supply.\n",
							supply->name);
					goto done;
				}
				if (!regulator_is_enabled(supply->regulator))
					dev_dbg(priv->dev,
							"Disabled regulator(%s).\n",
							supply->name);
			}
		}
	}
done:
	pr_debug("<--\n");
	return status;
}

static long wrfpwr_ioctl(struct file *filep,
			unsigned int cmd, unsigned long arg)
{
	struct wrfpwr_priv *priv = filep->private_data;
	struct radio_regulator_info regulator_info;
	int status = 0;
	pr_debug("-->\n");
    /* Initialize memory */
	memset(&regulator_info, 0, sizeof(struct radio_regulator_info));
    /* Process Command */
	switch (cmd) {
	case BCM_WRFPWR_VREG_CNTRL:
	{
		/* Copy data from user */
		struct radio_regulator_info *ui_regulator_info =
		(struct radio_regulator_info *)arg;
		if (ui_regulator_info->num_regulators <= 0 ||
			ui_regulator_info->num_regulators > supplies) {
			dev_err(priv->dev,
			"Invalid number of regulators : %d\n",
			ui_regulator_info->num_regulators);
			status = -EOPNOTSUPP;
			goto done;
		}
		status = copy_from_user(&regulator_info,
			ui_regulator_info,
			sizeof(struct radio_regulator_info));
		if (status < 0) {
			dev_err(priv->dev,
			"Copy data from user failed : %d\n", status);
			goto done;
		}
		/* Allocate Memory to copy DT information */
		regulator_info.cntrl_info =
			kmalloc(sizeof(struct regulator_cntrl_info)*
			ui_regulator_info->num_regulators, GFP_KERNEL);
		memset(regulator_info.cntrl_info, 0,
			sizeof(struct regulator_cntrl_info)*
			ui_regulator_info->num_regulators);
		status = copy_from_user(regulator_info.cntrl_info,
			ui_regulator_info->cntrl_info,
			sizeof(struct regulator_cntrl_info)*
			ui_regulator_info->num_regulators);
		if (status < 0) {
			dev_err(priv->dev,
			"Copy data from user failed : %d\n", status);
			goto done;
		}
		/* Control the Regulator Power*/
		status = wrfpwr_regulator_cntrl(priv, &regulator_info);
		if (status < 0) {
			dev_err(priv->dev,
					"Power Control failed\n");
			goto done;
		}
		break;
	}

	case BCM_WRFPWR_VREG_GET_NUM:
	{
		status = copy_to_user((void *)arg, &supplies,
			sizeof(supplies));
		if (status < 0) {
			dev_err(priv->dev,
			"Copy data to user failed : %d\n", status);
			goto done;
		}
		break;
	}

	case BCM_WRFPWR_VREG_GET_INFO:
	{
		struct radio_regulator_info *ui_regulator_info =
			(struct radio_regulator_info *)arg;
		/* Allocate Memory to copy DT information */
		regulator_info.cntrl_info =
			kmalloc(sizeof(struct regulator_cntrl_info)*supplies,
			GFP_KERNEL);
		memset(regulator_info.cntrl_info, 0,
			sizeof(struct regulator_cntrl_info)*supplies);
		/* Get DT info and Copy it */
		status = wrfpwr_update_cntrl_info(priv, &regulator_info);
		if (status < 0) {
			dev_err(priv->dev,
					"Update DT info failed\n");
			goto done;
		}
		status = copy_to_user(ui_regulator_info->cntrl_info,
			regulator_info.cntrl_info,
			sizeof(struct regulator_cntrl_info)*supplies);
		if (status < 0) {
			dev_err(priv->dev,
					"copy_to_user failed : %d\n", status);
			goto done;
		}
		break;
	}
#if defined(CONFIG_BCM_I2C_WBID)
	case BCM_WRFPWR_VREG_WBID:
	{
		int wbid = 0;
		wbid = wbid_read();
		status = copy_to_user((void *)arg, &wbid,
			sizeof(wbid));
		if (status < 0) {
			dev_err(priv->dev,
			"Copy data to user failed : %d\n", status);
			goto done;
		}
		break;
	}
#endif
	default:
		status = -EOPNOTSUPP;
		goto done;
		break;
	}
done:
	if (regulator_info.cntrl_info)
		kfree(regulator_info.cntrl_info);
	pr_debug("<--\n");
	return status;
}

static const struct file_operations wrfpwr_file_operations = {
	.owner = THIS_MODULE,
	.open = wrfpwr_open,
	.release = wrfpwr_release,
	.unlocked_ioctl = wrfpwr_ioctl,
};

static void wrfpwr_proc_cmd_regpwrcntrl_help(char *str)
{
	pr_alert("%s regpwrcntrl: Enable/disable WiFi RF power\n", str);
	pr_alert("%s  regpwrcntrl <WiFi RF Name> <enable|disable>\n", str);
	pr_alert("%s  WiFi RF name (e.g. wifirf0pwr)\n", str);
}

static int wrfpwr_proc_cmd_regpwrcntrl(int argc, char *argv[])
{
	int status = 0;
	int cnt = 0;

	pr_debug("-->\n");
	if (argc == 3) {
		for (cnt = 0; cnt < MAX_WIFI_RF_PARAM_LIST; cnt++) {
			if (!strncmp(argv[1], wifi_rf_param_list[cnt].proc_name,
						strlen(wifi_rf_param_list[cnt].proc_name))) {

				if (!strncmp(argv[2], PROC_ENABLE_CMD_STR,
					 strlen(PROC_ENABLE_CMD_STR)))
					wifi_rf_regulator_pwr_cntrl(wifi_rf_param_list[cnt].reg_name, true);
				else if (!strncmp(argv[2], PROC_DISABLE_CMD_STR,
					 strlen(PROC_DISABLE_CMD_STR)))
					wifi_rf_regulator_pwr_cntrl(wifi_rf_param_list[cnt].reg_name, false);
				else {
					status = -EINVAL;
					pr_info("WiFi RF state : %s not supported\n", argv[2]);
					goto help;
				}
					
				goto done;
			}
		}
		if (cnt >= MAX_WIFI_RF_PARAM_LIST) {
			pr_info("WiFi RF name : %s not supported\n", argv[1]);
			status = -EINVAL;
			goto help;
		}
	}

help:
	wrfpwr_proc_cmd_regpwrcntrl_help("");
done:
	pr_debug("<--\n");
	return status;
}

static void wrfpwr_proc_cmd_help_help(char *str)
{
	pr_alert("%s help: Help on individual command\n", str);
}

static int wrfpwr_proc_cmd_help(int argc, char *argv[])
{
	int status = 0;

	if (argc != 2) {
		pr_err("Invalid # of arguments.\n");
		status = -EINVAL;
		goto done;
	}

	if (!strncmp(argv[1], "regpwrcntrl", sizeof("regpwrcntrl"))) {
		wrfpwr_proc_cmd_regpwrcntrl_help("");
	} else {
		pr_err("Unrecognized command: %s\n", argv[1]);
		wrfpwr_proc_cmd_help_help("");
		status = -EINVAL;
	}

done:
	return status;
}

static struct proc_cmd_ops command_entries[] = {
	PROC_CMD_INIT("help", wrfpwr_proc_cmd_help),
	PROC_CMD_INIT("regpwrcntrl", wrfpwr_proc_cmd_regpwrcntrl),
};

struct proc_cmd_table wrfpwr_command_table = {
	.module_name = "wrfpwr",
	.size = ARRAY_SIZE(command_entries),
	.ops = command_entries
};

static struct proc_dir_entry *proc_dir;
static struct proc_dir_entry *cmd_proc_file;

static void wrfpwr_proc_exit(void)
{
	if (cmd_proc_file) {
		remove_proc_entry(CMD_PROC_FILE, proc_dir);
		cmd_proc_file = NULL;
	}
	if (proc_dir) {
		remove_proc_entry(PROC_DIR, NULL);
		proc_dir = NULL;
	}
}

static int wrfpwr_proc_init(void)
{
	int status  = 0;

	proc_dir = proc_mkdir(PROC_DIR, NULL);
	if (!proc_dir) {
		pr_err("Failed to create PROC directory %s.\n",
		       PROC_DIR);
		status = -EIO;
		goto done;
	}
	cmd_proc_file = proc_create_cmd(CMD_PROC_FILE, proc_dir,
					&wrfpwr_command_table);
	if (!cmd_proc_file) {
		pr_err("Failed to create %s\n", CMD_PROC_FILE);
		status = -EIO;
		wrfpwr_proc_exit();
		goto done;
	}

done:
	return status;
}

static int wrfpwr_probe(struct platform_device *pdev)
{
	int status = 0;
	struct device *dev = &pdev->dev;
	struct wrfpwr_priv *priv;

	struct device_node *of_node = pdev->dev.of_node;
	const char *ptr = NULL;
	char device_name[10];

	int i;
	const char *name;
	struct dev_pwr_supply *supply;
	dev_dbg(dev, "-->\n");
	dev_dbg(dev, "WRFPWR: Initializing the WRFPWR.....\n");
	node_count++;

	INIT_LIST_HEAD(&priv_list);
	priv = devm_kzalloc(dev, sizeof(*priv), GFP_KERNEL);
	if (IS_ERR_OR_NULL(priv)) {
		dev_err(dev, "unable to allocate private data\n");
		status = -ENOMEM;
		goto done;
	}
	list_add_tail(&priv->list, &priv_list);
	INIT_LIST_HEAD(&priv->pwr_supplies);
	supplies = of_property_count_strings(of_node, "supply-names");
	if (supplies <= 0) {
		dev_err(&pdev->dev,
		"supply-names property of node missing or empty\n");
		status = -ENODEV;
		goto done;
	}
	/*Add some delay before enabling RF_Disable_L on bootup
	Relay recommed between Pcie_RST and RF_Disable_L is 100ms
	Already there is enough delay on bootup between those
	100ms added for saftey*/
	msleep(100);
	for (i = 0; i < supplies; i++) {
		if (of_property_read_string_index(of_node,
				"supply-names", i, &name)) {
			status = -EINVAL;
			goto done;
		}
		supply = devm_kzalloc(&pdev->dev, sizeof(*supply), GFP_KERNEL);
		if (!supply)
			return -ENOMEM;
		strncpy(supply->name, name, sizeof(supply->name));
		supply->name[sizeof(supply->name) - 1] = '\0';
		supply->regulator = devm_regulator_get(&pdev->dev, name);
		if (IS_ERR(supply->regulator)) {
			dev_err(&pdev->dev,
			"Unable to get %s supply, err=%d\n",
			name, (int)PTR_ERR(supply->regulator));
			status = -ENOENT;
			goto done;
		}
		if (regulator_enable(supply->regulator))
			dev_err(&pdev->dev, "Unable to enable %s supply.\n",
				name);
		dev_info(&pdev->dev, "enabled %s supply.\n",name);
		list_add_tail(&supply->node, &priv->pwr_supplies);
	}

	if (of_property_read_string(of_node, "dev-name", &ptr)) {
		/* Couldn't find the entry */
		snprintf(device_name, sizeof(device_name),
				"wrfpwr%d", node_count);
		device_name[sizeof(device_name) - 1] = '\0';
	} else {
		snprintf(device_name, sizeof(device_name), "%s", ptr);
		device_name[sizeof(device_name) - 1] = '\0';
	}

	platform_set_drvdata(pdev, priv);
	priv->pdev = pdev;

	status = alloc_chrdev_region(&priv->ndev, 0, 1, device_name);
	if (status) {
		dev_err(dev,
		"Failed registering char device number with status %d!\n",
		status);
		goto done;
	}

	cdev_init(&priv->c_dev, &wrfpwr_file_operations);
	(&priv->c_dev)->owner = THIS_MODULE;
	cdev_add(&priv->c_dev, priv->ndev, node_count);

	priv->wrfpwr_char_class = class_create(THIS_MODULE, device_name);
	if (IS_ERR(priv->wrfpwr_char_class)) {
		cdev_del(&priv->c_dev);
		unregister_chrdev_region(priv->ndev, 1);
		status = PTR_ERR(priv->wrfpwr_char_class);
		goto done;
	}

	priv->wrfpwr_char_device = device_create(
			priv->wrfpwr_char_class, NULL, priv->ndev,
			NULL, device_name);
	if (IS_ERR(priv->wrfpwr_char_device)) {
		class_destroy(priv->wrfpwr_char_class);
		cdev_del(&priv->c_dev);
		unregister_chrdev_region(priv->ndev, 1);
		status = PTR_ERR(priv->wrfpwr_char_device);
		goto done;
	}
#if defined(CONFIG_BCM_MBOX)
	brcm_mbox_register_notifier(&wrfpwr_mbox_notifier);
#elif defined(CONFIG_BCM_PWR_RPC)
	brcm_pwr_rpc_mbox_register_notifier(&wrfpwr_mbox_notifier);
#endif
	wrfpwr_proc_init();
done:
	dev_dbg(dev, "<--\n");
	return status;
}

static int wrfpwr_remove(struct platform_device *pdev)
{
	int status = 0;
	struct device *dev = &pdev->dev;
	struct wrfpwr_priv *priv = platform_get_drvdata(pdev);
	struct wrfpwr_priv *priv_tmp;
	struct list_head *pos;

	dev_dbg(dev, "-->\n");
	if (!priv) {
		dev_err(dev, "Release called with uninitialized\n");
		status = -EINVAL;
		goto done;
	}
	wrfpwr_proc_exit();
#if defined(CONFIG_BCM_MBOX)
	brcm_mbox_unregister_notifier(&wrfpwr_mbox_notifier);
#elif defined(CONFIG_BCM_PWR_RPC)
	brcm_pwr_rpc_mbox_unregister_notifier(&wrfpwr_mbox_notifier);
#endif
	class_destroy(priv->wrfpwr_char_class);
	cdev_del(&priv->c_dev);
	unregister_chrdev_region(priv->ndev, 1);
	list_for_each(pos, &priv_list) {
		priv_tmp = list_entry(pos, struct wrfpwr_priv, list);
		if (priv_tmp == priv) {
			list_del(pos);
			break;
		}
	}
done:
	dev_dbg(dev, "<--\n");
	return status;
}

static const struct of_device_id wrfpwr_of_match[] = {
	{ .compatible = "brcm,wrfpwr" },
	{ }
};

MODULE_DEVICE_TABLE(of, wrfpwr_of_match);

static struct platform_driver wrfpwr_driver = {
	.probe	= wrfpwr_probe,
	.remove = wrfpwr_remove,
	.driver = {
		.name		= "wrfpwr",
		.owner		= THIS_MODULE,
		.of_match_table	= wrfpwr_of_match
	}
};

__init int wrfpwr_init(void)
{
	int status = 0;
	pr_debug("-->\n");
	status = platform_driver_register(&wrfpwr_driver);
	pr_debug("<--\n");
	return status;
}

void wrfpwr_exit(void)
{
	pr_debug("-->\n");
	platform_driver_unregister(&wrfpwr_driver);
	pr_debug("<--\n");
}

late_initcall(wrfpwr_init);
module_exit(wrfpwr_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Venky");
MODULE_DESCRIPTION("Broadcom CM WIFI-RF-POWER Driver");
MODULE_VERSION("1.0");
