/*
 * BCM SPU crypto testing module based on tcrypt and testmgr code
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 */

#include <crypto/aead.h>
#include <crypto/hash.h>
#include <crypto/skcipher.h>
#include <linux/err.h>
#include <linux/fips.h>
#include <linux/init.h>
#include <linux/gfp.h>
#include <linux/module.h>
#include <linux/scatterlist.h>
#include <linux/string.h>
#include <linux/moduleparam.h>
#include <linux/jiffies.h>
#include <linux/timex.h>
#include <linux/interrupt.h>
#include "tcrypt.h"

/*
 * Need slab memory for testing (size in number of pages).
 */
#define TVMEMSIZE	4

/*
* Used by test_cipher_speed()
*/
#define ENCRYPT 1
#define DECRYPT 0

#define MAX_DIGEST_SIZE		64

/*
 * return a string with the driver name
 */
#define get_driver_name(tfm_type, tfm) crypto_tfm_alg_driver_name(tfm_type ## _tfm(tfm))

/*
 * Used by test_cipher_speed()
 */
static unsigned int sec;

static int mode;
static char *tvmem[TVMEMSIZE];

struct tcrypt_result {
	struct completion completion;
	int err;
};

/*
 * Need slab memory for testing (size in number of pages).
 */
#define XBUFSIZE	8

/*
 * Indexes into the xbuf to simulate cross-page access.
 */
#define IDX1		32
#define IDX2		32400
#define IDX3		1
#define IDX4		8193
#define IDX5		22222
#define IDX6		17101
#define IDX7		27333
#define IDX8		3000

static unsigned int IDX[8] = { IDX1, IDX2, IDX3, IDX4, IDX5, IDX6, IDX7, IDX8 };


static void hexdump(unsigned char *buf, unsigned int len)
{
	print_hex_dump(KERN_CONT, "", DUMP_PREFIX_OFFSET,
			16, 1,
			buf, len, false);
}

static int wait_async_op(struct tcrypt_result *tr, int ret)
{
	if (ret == -EINPROGRESS || ret == -EBUSY) {
		wait_for_completion(&tr->completion);
		reinit_completion(&tr->completion);
		ret = tr->err;
	}
	return ret;
}

static u32 block_sizes[] = { 16, 64, 256, 1024, 8192, 0 };
static u32 aead_sizes[] = { 16, 64, 256, 512, 1024, 2048, 4096, 8192, 0 };

#define XBUFSIZE 8
#define MAX_IVLEN 32

static int testmgr_alloc_buf(char *buf[XBUFSIZE])
{
	int i;

	for (i = 0; i < XBUFSIZE; i++) {
		buf[i] = (void *)__get_free_page(GFP_KERNEL);
		if (!buf[i])
			goto err_free_buf;
	}

	return 0;

err_free_buf:
	while (i-- > 0)
		free_page((unsigned long)buf[i]);

	return -ENOMEM;
}

static void testmgr_free_buf(char *buf[XBUFSIZE])
{
	int i;

	for (i = 0; i < XBUFSIZE; i++)
		free_page((unsigned long)buf[i]);
}

static void tcrypt_complete(struct crypto_async_request *req, int err)
{
	struct tcrypt_result *res = req->data;

	if (err == -EINPROGRESS)
		return;

	res->err = err;
	complete(&res->completion);
}

static int __test_aead(struct crypto_aead *tfm, int enc,
		       struct aead_testvec *template, unsigned int tcount,
		       const bool diff_dst, const int align_offset)
{
	const char *algo = crypto_tfm_alg_driver_name(crypto_aead_tfm(tfm));
	unsigned int i, j, k, n, temp;
	int ret = -ENOMEM;
	char *q;
	char *key;
	struct aead_request *req;
	struct scatterlist *sg;
	struct scatterlist *sgout;
	const char *e, *d;
	struct tcrypt_result result;
	unsigned int authsize, iv_len;
	void *input;
	void *output;
	void *assoc;
	char *iv;
	char *xbuf[XBUFSIZE];
	char *xoutbuf[XBUFSIZE];
	char *axbuf[XBUFSIZE];

	iv = kzalloc(MAX_IVLEN, GFP_KERNEL);
	if (!iv)
		return ret;
	key = kmalloc(MAX_KEYLEN, GFP_KERNEL);
	if (!key)
		goto out_noxbuf;
	if (testmgr_alloc_buf(xbuf))
		goto out_noxbuf;
	if (testmgr_alloc_buf(axbuf))
		goto out_noaxbuf;
	if (diff_dst && testmgr_alloc_buf(xoutbuf))
		goto out_nooutbuf;

	/* avoid "the frame size is larger than 1024 bytes" compiler warning */
	sg = kmalloc(sizeof(*sg) * 8 * (diff_dst ? 4 : 2), GFP_KERNEL);
	if (!sg)
		goto out_nosg;
	sgout = &sg[16];

	if (diff_dst)
		d = "-ddst";
	else
		d = "";

	if (enc == ENCRYPT)
		e = "encryption";
	else
		e = "decryption";

	init_completion(&result.completion);

	req = aead_request_alloc(tfm, GFP_KERNEL);
	if (!req) {
		pr_err("alg: aead%s: Failed to allocate request for %s\n",
		       d, algo);
		goto out;
	}

	aead_request_set_callback(req, CRYPTO_TFM_REQ_MAY_BACKLOG,
				  tcrypt_complete, &result);

	iv_len = crypto_aead_ivsize(tfm);

	for (i = 0, j = 0; i < tcount; i++) {
		if (template[i].np)
			continue;

		j++;

		/* some templates have no input data but they will
		 * touch input
		 */
		input = xbuf[0];
		input += align_offset;
		assoc = axbuf[0];

		ret = -EINVAL;
		if (WARN_ON(align_offset + template[i].ilen >
			    PAGE_SIZE || template[i].alen > PAGE_SIZE))
			goto out;

		memcpy(input, template[i].input, template[i].ilen);
		memcpy(assoc, template[i].assoc, template[i].alen);
		if (template[i].iv)
			memcpy(iv, template[i].iv, iv_len);
		else
			memset(iv, 0, iv_len);

		crypto_aead_clear_flags(tfm, ~0);
		if (template[i].wk)
			crypto_aead_set_flags(tfm, CRYPTO_TFM_REQ_FORBID_WEAK_KEYS);

		if (template[i].klen > MAX_KEYLEN) {
			pr_err("alg: aead%s: setkey failed on test %d for %s: key size %d > %d\n",
			       d, j, algo, template[i].klen,
			       MAX_KEYLEN);
			ret = -EINVAL;
			goto out;
		}
		memcpy(key, template[i].key, template[i].klen);

		ret = crypto_aead_setkey(tfm, key, template[i].klen);
		if (template[i].fail == !ret) {
			pr_err("alg: aead%s: setkey failed on test %d for %s: flags=%x\n",
			       d, j, algo, crypto_aead_get_flags(tfm));
			goto out;
		} else if (ret)
			continue;

		authsize = abs(template[i].rlen - template[i].ilen);
		ret = crypto_aead_setauthsize(tfm, authsize);
		if (ret) {
			pr_err("alg: aead%s: Failed to set authsize to %u on test %d for %s\n",
			       d, authsize, j, algo);
			goto out;
		}

		k = !!template[i].alen;
		sg_init_table(sg, k + 1);
		sg_set_buf(&sg[0], assoc, template[i].alen);
		sg_set_buf(&sg[k], input,
			   template[i].ilen + (enc ? authsize : 0));
		output = input;

		if (diff_dst) {
			sg_init_table(sgout, k + 1);
			sg_set_buf(&sgout[0], assoc, template[i].alen);

			output = xoutbuf[0];
			output += align_offset;
			sg_set_buf(&sgout[k], output,
				   template[i].rlen + (enc ? 0 : authsize));
		}

		aead_request_set_crypt(req, sg, (diff_dst) ? sgout : sg,
				       template[i].ilen, iv);

		aead_request_set_ad(req, template[i].alen);

		ret = enc ? crypto_aead_encrypt(req) : crypto_aead_decrypt(req);

		switch (ret) {
		case 0:
			if (template[i].novrfy) {
				/* verification was supposed to fail */
				pr_err("alg: aead%s: %s failed on test %d for %s: ret was 0, expected -EBADMSG\n",
				       d, e, j, algo);
				/* so really, we got a bad message */
				ret = -EBADMSG;
				goto out;
			}
			break;
		case -EINPROGRESS:
		case -EBUSY:
			wait_for_completion(&result.completion);
			reinit_completion(&result.completion);
			ret = result.err;
			if (!ret)
				break;
			fallthrough;
		case -EBADMSG:
			if (template[i].novrfy)
				/* verification failure was expected */
				continue;
			fallthrough;
		default:
			pr_err("alg: aead%s: %s failed on test %d for %s: ret=%d\n",
			       d, e, j, algo, -ret);
			goto out;
		}

		q = output;
		if (memcmp(q, template[i].result, template[i].rlen)) {
			pr_err("alg: aead%s: Test %d failed on %s for %s\n",
			       d, j, e, algo);
			hexdump(q, template[i].rlen);
			ret = -EINVAL;
			goto out;
		} else {
			pr_info("alg: aead%s: Test %d passed on %s for %s\n",
				d, j, e, algo);
		}
	}

	for (i = 0, j = 0; i < tcount; i++) {
		/* alignment tests are only done with continuous buffers */
		if (align_offset != 0)
			break;

		if (!template[i].np)
			continue;

		j++;

		if (template[i].iv)
			memcpy(iv, template[i].iv, iv_len);
		else
			memset(iv, 0, MAX_IVLEN);

		crypto_aead_clear_flags(tfm, ~0);
		if (template[i].wk)
			crypto_aead_set_flags(tfm, CRYPTO_TFM_REQ_FORBID_WEAK_KEYS);
		if (template[i].klen > MAX_KEYLEN) {
			pr_err("alg: aead%s: setkey failed on test %d for %s: key size %d > %d\n",
			       d, j, algo, template[i].klen, MAX_KEYLEN);
			ret = -EINVAL;
			goto out;
		}
		memcpy(key, template[i].key, template[i].klen);

		ret = crypto_aead_setkey(tfm, key, template[i].klen);
		if (template[i].fail == !ret) {
			pr_err("alg: aead%s: setkey failed on chunk test %d for %s: flags=%x\n",
			       d, j, algo, crypto_aead_get_flags(tfm));
			goto out;
		} else if (ret)
			continue;

		authsize = abs(template[i].rlen - template[i].ilen);

		ret = -EINVAL;
		sg_init_table(sg, template[i].anp + template[i].np);
		if (diff_dst)
			sg_init_table(sgout, template[i].anp + template[i].np);

		ret = -EINVAL;
		for (k = 0, temp = 0; k < template[i].anp; k++) {
			if (WARN_ON(offset_in_page(IDX[k]) +
				    template[i].atap[k] > PAGE_SIZE))
				goto out;
			sg_set_buf(&sg[k],
				   memcpy(axbuf[IDX[k] >> PAGE_SHIFT] +
					  offset_in_page(IDX[k]),
					  template[i].assoc + temp,
					  template[i].atap[k]),
				   template[i].atap[k]);
			if (diff_dst)
				sg_set_buf(&sgout[k],
					   axbuf[IDX[k] >> PAGE_SHIFT] +
					   offset_in_page(IDX[k]),
					   template[i].atap[k]);
			temp += template[i].atap[k];
		}

		for (k = 0, temp = 0; k < template[i].np; k++) {
			if (WARN_ON(offset_in_page(IDX[k]) +
				    template[i].tap[k] > PAGE_SIZE))
				goto out;

			q = xbuf[IDX[k] >> PAGE_SHIFT] + offset_in_page(IDX[k]);
			memcpy(q, template[i].input + temp, template[i].tap[k]);
			sg_set_buf(&sg[template[i].anp + k],
				   q, template[i].tap[k]);

			if (diff_dst) {
				q = xoutbuf[IDX[k] >> PAGE_SHIFT] +
				    offset_in_page(IDX[k]);

				memset(q, 0, template[i].tap[k]);

				sg_set_buf(&sgout[template[i].anp + k],
					   q, template[i].tap[k]);
			}

			n = template[i].tap[k];
			if (k == template[i].np - 1 && enc)
				n += authsize;
			if (offset_in_page(q) + n < PAGE_SIZE)
				q[n] = 0;

			temp += template[i].tap[k];
		}

		ret = crypto_aead_setauthsize(tfm, authsize);
		if (ret) {
			pr_err("alg: aead%s: Failed to set authsize to %u on chunk test %d for %s\n",
			       d, authsize, j, algo);
			goto out;
		}

		if (enc) {
			if (WARN_ON(sg[template[i].anp + k - 1].offset +
				    sg[template[i].anp + k - 1].length +
				    authsize > PAGE_SIZE)) {
				ret = -EINVAL;
				goto out;
			}

			if (diff_dst)
				sgout[template[i].anp + k - 1].length +=
					authsize;
			sg[template[i].anp + k - 1].length += authsize;
		}

		aead_request_set_crypt(req, sg, (diff_dst) ? sgout : sg,
				       template[i].ilen,
				       iv);

		aead_request_set_ad(req, template[i].alen);

		ret = enc ? crypto_aead_encrypt(req) : crypto_aead_decrypt(req);

		switch (ret) {
		case 0:
			if (template[i].novrfy) {
				/* verification was supposed to fail */
				pr_err("alg: aead%s: %s failed on chunk test %d for %s: ret was 0, expected -EBADMSG\n",
				       d, e, j, algo);
				/* so really, we got a bad message */
				ret = -EBADMSG;
				goto out;
			}
			break;
		case -EINPROGRESS:
		case -EBUSY:
			wait_for_completion(&result.completion);
			reinit_completion(&result.completion);
			ret = result.err;
			if (!ret)
				break;
			fallthrough;
		case -EBADMSG:
			if (template[i].novrfy)
				/* verification failure was expected */
				continue;
			fallthrough;
		default:
			pr_err("alg: aead%s: %s failed on chunk test %d for %s: ret=%d\n",
			       d, e, j, algo, -ret);
			goto out;
		}

		ret = -EINVAL;
		for (k = 0, temp = 0; k < template[i].np; k++) {
			if (diff_dst)
				q = xoutbuf[IDX[k] >> PAGE_SHIFT] +
				    offset_in_page(IDX[k]);
			else
				q = xbuf[IDX[k] >> PAGE_SHIFT] +
				    offset_in_page(IDX[k]);

			n = template[i].tap[k];
			if (k == template[i].np - 1)
				n += enc ? authsize : -authsize;

			if (memcmp(q, template[i].result + temp, n)) {
				pr_err("alg: aead%s: Chunk test %d failed on %s at page %u for %s\n",
				       d, j, e, k, algo);
				hexdump(q, n);
				goto out;
			}

			q += n;
			if (k == template[i].np - 1 && !enc) {
				if (!diff_dst &&
					memcmp(q, template[i].input +
					      temp + n, authsize))
					n = authsize;
				else
					n = 0;
			} else {
				for (n = 0; offset_in_page(q + n) && q[n]; n++)
					;
			}
			if (n) {
				pr_err("alg: aead%s: Result buffer corruption in chunk test %d on %s at page %u for %s: %u bytes:\n",
				       d, j, e, k, algo, n);
				hexdump(q, n);
				goto out;
			} else {
				pr_info("alg: aead%s: Chunk test %d passed on %s at page %u for %s\n",
					d, j, e, k, algo);
			}

			temp += template[i].tap[k];
		}
	}

	ret = 0;

out:
	aead_request_free(req);
	kfree(sg);
out_nosg:
	if (diff_dst)
		testmgr_free_buf(xoutbuf);
out_nooutbuf:
	testmgr_free_buf(axbuf);
out_noaxbuf:
	testmgr_free_buf(xbuf);
out_noxbuf:
	kfree(key);
	kfree(iv);
	return ret;
}

static int test_aead(struct crypto_aead *tfm, int enc,
		     struct aead_testvec *template, unsigned int tcount)
{
	unsigned int alignmask;
	int ret;

	/* test 'dst == src' case */
	ret = __test_aead(tfm, enc, template, tcount, false, 0);
	if (ret)
		return ret;

	/* test 'dst != src' case */
	ret = __test_aead(tfm, enc, template, tcount, true, 0);
	if (ret)
		return ret;

	/* test unaligned buffers, check with one byte offset */
	ret = __test_aead(tfm, enc, template, tcount, true, 1);
	if (ret)
		return ret;

	alignmask = crypto_tfm_alg_alignmask(&tfm->base);
	if (alignmask) {
		/* Check if alignment mask for tfm is correctly set. */
		ret = __test_aead(tfm, enc, template, tcount, true,
				  alignmask + 1);
		if (ret)
			return ret;
	}

	return 0;
}

static int test_aead_common(const char *algo,
			    int enc,
			    struct aead_testvec *template,
			    unsigned int tcount,
			    u32 mask)
{
	struct crypto_aead *tfm;
	int err = 0;

	tfm = crypto_alloc_aead(algo, CRYPTO_ALG_INTERNAL, mask);
	if (IS_ERR(tfm)) {
		printk(KERN_ERR "alg: aead: Failed to load transform for %s: "
		       "%ld\n", algo, PTR_ERR(tfm));
		return PTR_ERR(tfm);
	}

	printk(KERN_INFO "\ntesting aead encrypt(%d) of %s\n", enc,
	       algo);

	err = test_aead(tfm, enc, template, tcount);

	crypto_free_aead(tfm);
	return err;
}

static inline int do_one_aead_op(struct aead_request *req, int ret)
{
	if (ret == -EINPROGRESS || ret == -EBUSY) {
		struct tcrypt_result *tr = req->base.data;

		ret = wait_for_completion_interruptible(&tr->completion);
		if (!ret)
			ret = tr->err;
		reinit_completion(&tr->completion);
	}

	return ret;
}

static int test_aead_jiffies(struct aead_request *req, int enc,
				int blen, int secs)
{
	unsigned long start, end;
	int bcount;
	int ret;

	for (start = jiffies, end = start + secs * HZ, bcount = 0;
	     time_before(jiffies, end); bcount++) {
		if (enc)
			ret = do_one_aead_op(req, crypto_aead_encrypt(req));
		else
			ret = do_one_aead_op(req, crypto_aead_decrypt(req));

		if (ret)
			return ret;
	}

	printk("%d operations in %d seconds (%ld bytes)\n",
	       bcount, secs, (long)bcount * blen);
	return 0;
}

static int test_aead_cycles(struct aead_request *req, int enc, int blen)
{
	unsigned long cycles = 0;
	int ret = 0;
	int i;

	local_irq_disable();

	/* Warm-up run. */
	for (i = 0; i < 4; i++) {
		if (enc)
			ret = do_one_aead_op(req, crypto_aead_encrypt(req));
		else
			ret = do_one_aead_op(req, crypto_aead_decrypt(req));

		if (ret)
			goto out;
	}

	/* The real thing. */
	for (i = 0; i < 8; i++) {
		cycles_t start, end;

		start = get_cycles();
		if (enc)
			ret = do_one_aead_op(req, crypto_aead_encrypt(req));
		else
			ret = do_one_aead_op(req, crypto_aead_decrypt(req));
		end = get_cycles();

		if (ret)
			goto out;

		cycles += end - start;
	}

out:
	local_irq_enable();

	if (ret == 0)
		printk("1 operation in %lu cycles (%d bytes)\n",
		       (cycles + 4) / 8, blen);

	return ret;
}

static void sg_init_aead(struct scatterlist *sg, char *xbuf[XBUFSIZE],
			unsigned int buflen)
{
	int np = (buflen + PAGE_SIZE - 1)/PAGE_SIZE;
	int k, rem;

	if (np > XBUFSIZE) {
		rem = PAGE_SIZE;
		np = XBUFSIZE;
	} else {
		rem = buflen % PAGE_SIZE;
	}

	sg_init_table(sg, np + 1);
	if (rem)
		np--;
	for (k = 0; k < np; k++)
		sg_set_buf(&sg[k + 1], xbuf[k], PAGE_SIZE);

	if (rem)
		sg_set_buf(&sg[k + 1], xbuf[k], rem);
}

static void test_aead_speed(const char *algo, int enc, unsigned int secs,
			    struct aead_speed_template *template,
			    unsigned int tcount, u8 authsize,
			    unsigned int aad_size, u8 *keysize)
{
	unsigned int i, j;
	struct crypto_aead *tfm;
	int ret = -ENOMEM;
	const char *key;
	struct aead_request *req;
	struct scatterlist *sg;
	struct scatterlist *sgout;
	const char *e;
	void *assoc;
	char *iv;
	char *xbuf[XBUFSIZE];
	char *xoutbuf[XBUFSIZE];
	char *axbuf[XBUFSIZE];
	unsigned int *b_size;
	unsigned int iv_len;
	struct tcrypt_result result;

	iv = kzalloc(MAX_IVLEN, GFP_KERNEL);
	if (!iv)
		return;

	if (aad_size >= PAGE_SIZE) {
		pr_err("associate data length (%u) too big\n", aad_size);
		goto out_noxbuf;
	}

	if (enc == ENCRYPT)
		e = "encryption";
	else
		e = "decryption";

	if (testmgr_alloc_buf(xbuf))
		goto out_noxbuf;
	if (testmgr_alloc_buf(axbuf))
		goto out_noaxbuf;
	if (testmgr_alloc_buf(xoutbuf))
		goto out_nooutbuf;

	sg = kmalloc(sizeof(*sg) * 9 * 2, GFP_KERNEL);
	if (!sg)
		goto out_nosg;
	sgout = &sg[9];

	tfm = crypto_alloc_aead(algo, 0, 0);

	if (IS_ERR(tfm)) {
		pr_err("alg: aead: Failed to load transform for %s: %ld\n", algo,
		       PTR_ERR(tfm));
		goto out_notfm;
	}

	init_completion(&result.completion);
	printk(KERN_INFO "\ntesting speed of %s (%s) %s\n", algo,
			get_driver_name(crypto_aead, tfm), e);

	req = aead_request_alloc(tfm, GFP_KERNEL);
	if (!req) {
		pr_err("alg: aead: Failed to allocate request for %s\n",
		       algo);
		goto out_noreq;
	}

	aead_request_set_callback(req, CRYPTO_TFM_REQ_MAY_BACKLOG,
				  tcrypt_complete, &result);

	i = 0;
	do {
		b_size = aead_sizes;
		do {
			assoc = axbuf[0];
			memset(assoc, 0xff, aad_size);

			if ((*keysize + *b_size) > TVMEMSIZE * PAGE_SIZE) {
				pr_err("template (%u) too big for tvmem (%lu)\n",
				       *keysize + *b_size,
					TVMEMSIZE * PAGE_SIZE);
				goto out;
			}

			key = tvmem[0];
			for (j = 0; j < tcount; j++) {
				if (template[j].klen == *keysize) {
					key = template[j].key;
					break;
				}
			}
			ret = crypto_aead_setkey(tfm, key, *keysize);
			ret = crypto_aead_setauthsize(tfm, authsize);

			iv_len = crypto_aead_ivsize(tfm);
			if (iv_len)
				memset(iv, 0xff, iv_len);

			crypto_aead_clear_flags(tfm, ~0);
			printk(KERN_INFO "test %u (%d bit key, %d byte blocks): ",
					i, *keysize * 8, *b_size);


			memset(tvmem[0], 0xff, PAGE_SIZE);

			if (ret) {
				pr_err("setkey() failed flags=%x\n",
						crypto_aead_get_flags(tfm));
				goto out;
			}

			sg_init_aead(sg, xbuf,
				    *b_size + (enc ? 0 : authsize));

			sg_init_aead(sgout, xoutbuf,
				    *b_size + (enc ? authsize : 0));

			sg_set_buf(&sg[0], assoc, aad_size);
			sg_set_buf(&sgout[0], assoc, aad_size);

			aead_request_set_crypt(req, sg, sgout,
					       *b_size + (enc ? 0 : authsize),
					       iv);
			aead_request_set_ad(req, aad_size);

			if (secs)
				ret = test_aead_jiffies(req, enc, *b_size,
							secs);
			else
				ret = test_aead_cycles(req, enc, *b_size);

			if (ret) {
				pr_err("%s() failed return code=%d\n", e, ret);
				break;
			}
			b_size++;
			i++;
		} while (*b_size);
		keysize++;
	} while (*keysize);

out:
	aead_request_free(req);
out_noreq:
	crypto_free_aead(tfm);
out_notfm:
	kfree(sg);
out_nosg:
	testmgr_free_buf(xoutbuf);
out_nooutbuf:
	testmgr_free_buf(axbuf);
out_noaxbuf:
	testmgr_free_buf(xbuf);
out_noxbuf:
	kfree(iv);
	return;
}

static void test_hash_sg_init(struct scatterlist *sg)
{
	int i;

	sg_init_table(sg, TVMEMSIZE);
	for (i = 0; i < TVMEMSIZE; i++) {
		sg_set_buf(sg + i, tvmem[i], PAGE_SIZE);
		memset(tvmem[i], 0xff, PAGE_SIZE);
	}
}

static inline int do_one_ahash_op(struct ahash_request *req, int ret)
{
	if (ret == -EINPROGRESS || ret == -EBUSY) {
		struct tcrypt_result *tr = req->base.data;

		wait_for_completion(&tr->completion);
		reinit_completion(&tr->completion);
		ret = tr->err;
	}
	return ret;
}

struct test_mb_ahash_data {
	struct scatterlist sg[TVMEMSIZE];
	char result[64];
	struct ahash_request *req;
	struct tcrypt_result tresult;
	char *xbuf[XBUFSIZE];
};

static void test_mb_ahash_speed(const char *algo, unsigned int sec,
				struct hash_speed *speed)
{
	struct test_mb_ahash_data *data;
	struct crypto_ahash *tfm;
	unsigned long start, end;
	unsigned long cycles;
	unsigned int i, j, k;
	int ret;

	data = kzalloc(sizeof(*data) * 8, GFP_KERNEL);
	if (!data)
		return;

	tfm = crypto_alloc_ahash(algo, 0, 0);
	if (IS_ERR(tfm)) {
		pr_err("failed to load transform for %s: %ld\n",
			algo, PTR_ERR(tfm));
		goto free_data;
	}

	for (i = 0; i < 8; ++i) {
		if (testmgr_alloc_buf(data[i].xbuf))
			goto out;

		init_completion(&data[i].tresult.completion);

		data[i].req = ahash_request_alloc(tfm, GFP_KERNEL);
		if (!data[i].req) {
			pr_err("alg: hash: Failed to allocate request for %s\n",
			       algo);
			goto out;
		}

		ahash_request_set_callback(data[i].req, 0,
					   tcrypt_complete, &data[i].tresult);
		test_hash_sg_init(data[i].sg);
	}

	pr_info("\ntesting speed of multibuffer %s (%s)\n", algo,
		get_driver_name(crypto_ahash, tfm));

	for (i = 0; speed[i].blen != 0; i++) {
		/* For some reason this only tests digests. */
		if (speed[i].blen != speed[i].plen)
			continue;

		if (speed[i].blen > TVMEMSIZE * PAGE_SIZE) {
			pr_err("template (%u) too big for tvmem (%lu)\n",
			       speed[i].blen, TVMEMSIZE * PAGE_SIZE);
			goto out;
		}

		if (speed[i].klen)
			crypto_ahash_setkey(tfm, tvmem[0], speed[i].klen);

		for (k = 0; k < 8; k++)
			ahash_request_set_crypt(data[k].req, data[k].sg,
						data[k].result, speed[i].blen);

		pr_info("test%3u "
			"(%5u byte blocks,%5u bytes per update,%4u updates): ",
			i, speed[i].blen, speed[i].plen,
			speed[i].blen / speed[i].plen);

		start = get_cycles();

		for (k = 0; k < 8; k++) {
			ret = crypto_ahash_digest(data[k].req);
			if (ret == -EINPROGRESS) {
				ret = 0;
				continue;
			}

			if (ret)
				break;

			complete(&data[k].tresult.completion);
			data[k].tresult.err = 0;
		}

		for (j = 0; j < k; j++) {
			struct tcrypt_result *tr = &data[j].tresult;

			wait_for_completion(&tr->completion);
			if (tr->err)
				ret = tr->err;
		}

		end = get_cycles();
		cycles = end - start;
		pr_cont("%6lu cycles/operation, %4lu cycles/byte\n",
			cycles, cycles / (8 * speed[i].blen));

		if (ret) {
			pr_err("At least one hashing failed ret=%d\n", ret);
			break;
		}
	}

out:
	for (k = 0; k < 8; ++k)
		ahash_request_free(data[k].req);

	for (k = 0; k < 8; ++k)
		testmgr_free_buf(data[k].xbuf);

	crypto_free_ahash(tfm);

free_data:
	kfree(data);
}

static int test_ahash_jiffies_digest(struct ahash_request *req, int blen,
				     char *out, int secs)
{
	unsigned long start, end;
	int bcount;
	int ret;

	for (start = jiffies, end = start + secs * HZ, bcount = 0;
	     time_before(jiffies, end); bcount++) {
		ret = do_one_ahash_op(req, crypto_ahash_digest(req));
		if (ret)
			return ret;
	}

	printk("%6u opers/sec, %9lu bytes/sec\n",
	       bcount / secs, ((long)bcount * blen) / secs);

	return 0;
}

static int test_ahash_jiffies(struct ahash_request *req, int blen,
			      int plen, char *out, int secs)
{
	unsigned long start, end;
	int bcount, pcount;
	int ret;

	if (plen == blen)
		return test_ahash_jiffies_digest(req, blen, out, secs);

	for (start = jiffies, end = start + secs * HZ, bcount = 0;
	     time_before(jiffies, end); bcount++) {
		ret = do_one_ahash_op(req, crypto_ahash_init(req));
		if (ret)
			return ret;
		for (pcount = 0; pcount < blen; pcount += plen) {
			ret = do_one_ahash_op(req, crypto_ahash_update(req));
			if (ret)
				return ret;
		}
		/* we assume there is enough space in 'out' for the result */
		ret = do_one_ahash_op(req, crypto_ahash_final(req));
		if (ret)
			return ret;
	}

	pr_cont("%6u opers/sec, %9lu bytes/sec\n",
		bcount / secs, ((long)bcount * blen) / secs);

	return 0;
}

static int test_ahash_cycles_digest(struct ahash_request *req, int blen,
				    char *out)
{
	unsigned long cycles = 0;
	int ret, i;

	/* Warm-up run. */
	for (i = 0; i < 4; i++) {
		ret = do_one_ahash_op(req, crypto_ahash_digest(req));
		if (ret)
			goto out;
	}

	/* The real thing. */
	for (i = 0; i < 8; i++) {
		cycles_t start, end;

		start = get_cycles();

		ret = do_one_ahash_op(req, crypto_ahash_digest(req));
		if (ret)
			goto out;

		end = get_cycles();

		cycles += end - start;
	}

out:
	if (ret)
		return ret;

	pr_cont("%6lu cycles/operation, %4lu cycles/byte\n",
		cycles / 8, cycles / (8 * blen));

	return 0;
}

static int test_ahash_cycles(struct ahash_request *req, int blen,
			     int plen, char *out)
{
	unsigned long cycles = 0;
	int i, pcount, ret;

	if (plen == blen)
		return test_ahash_cycles_digest(req, blen, out);

	/* Warm-up run. */
	for (i = 0; i < 4; i++) {
		ret = do_one_ahash_op(req, crypto_ahash_init(req));
		if (ret)
			goto out;
		for (pcount = 0; pcount < blen; pcount += plen) {
			ret = do_one_ahash_op(req, crypto_ahash_update(req));
			if (ret)
				goto out;
		}
		ret = do_one_ahash_op(req, crypto_ahash_final(req));
		if (ret)
			goto out;
	}

	/* The real thing. */
	for (i = 0; i < 8; i++) {
		cycles_t start, end;

		start = get_cycles();

		ret = do_one_ahash_op(req, crypto_ahash_init(req));
		if (ret)
			goto out;
		for (pcount = 0; pcount < blen; pcount += plen) {
			ret = do_one_ahash_op(req, crypto_ahash_update(req));
			if (ret)
				goto out;
		}
		ret = do_one_ahash_op(req, crypto_ahash_final(req));
		if (ret)
			goto out;

		end = get_cycles();

		cycles += end - start;
	}

out:
	if (ret)
		return ret;

	pr_cont("%6lu cycles/operation, %4lu cycles/byte\n",
		cycles / 8, cycles / (8 * blen));

	return 0;
}

#if 0
static int ahash_partial_update(struct ahash_request **preq,
	struct crypto_ahash *tfm, struct hash_testvec *template,
	void *hash_buff, int k, int temp, struct scatterlist *sg,
	const char *algo, char *result, struct tcrypt_result *tresult)
{
	char *state;
	struct ahash_request *req;
	int statesize, ret = -EINVAL;
	const char guard[] = { 0x00, 0xba, 0xad, 0x00 };

	req = *preq;
	statesize = crypto_ahash_statesize(
			crypto_ahash_reqtfm(req));
	state = kmalloc(statesize + sizeof(guard), GFP_KERNEL);
	if (!state) {
		pr_err("alt: hash: Failed to alloc state for %s\n", algo);
		goto out_nostate;
	}
	memcpy(state + statesize, guard, sizeof(guard));
	ret = crypto_ahash_export(req, state);
	WARN_ON(memcmp(state + statesize, guard, sizeof(guard)));
	if (ret) {
		pr_err("alt: hash: Failed to export() for %s\n", algo);
		goto out;
	}
	ahash_request_free(req);
	req = ahash_request_alloc(tfm, GFP_KERNEL);
	if (!req) {
		pr_err("alg: hash: Failed to alloc request for %s\n", algo);
		goto out_noreq;
	}
	ahash_request_set_callback(req,
		CRYPTO_TFM_REQ_MAY_BACKLOG,
		tcrypt_complete, tresult);

	memcpy(hash_buff, template->plaintext + temp,
		template->tap[k]);
	sg_init_one(&sg[0], hash_buff, template->tap[k]);
	ahash_request_set_crypt(req, sg, result, template->tap[k]);
	ret = crypto_ahash_import(req, state);
	if (ret) {
		pr_err("alg: hash: Failed to import() for %s\n", algo);
		goto out;
	}
	ret = wait_async_op(tresult, crypto_ahash_update(req));
	if (ret)
		goto out;
	*preq = req;
	ret = 0;
	goto out_noreq;
out:
	ahash_request_free(req);
out_noreq:
	kfree(state);
out_nostate:
	return ret;
}
#endif

static int __test_hash(struct crypto_ahash *tfm, struct hash_testvec *template,
		       unsigned int tcount, bool use_digest,
		       const int align_offset)
{
	const char *algo = crypto_tfm_alg_driver_name(crypto_ahash_tfm(tfm));
	unsigned int i, j, k, temp;
	struct scatterlist sg[8];
	char *result;
	char *key;
	struct ahash_request *req;
	struct tcrypt_result tresult;
	void *hash_buff;
	char *xbuf[XBUFSIZE];
	int ret = -ENOMEM;

	result = kmalloc(MAX_DIGEST_SIZE, GFP_KERNEL);
	if (!result)
		return ret;
	key = kmalloc(MAX_KEYLEN, GFP_KERNEL);
	if (!key)
		goto out_nobuf;
	if (testmgr_alloc_buf(xbuf))
		goto out_nobuf;

	init_completion(&tresult.completion);

	req = ahash_request_alloc(tfm, GFP_KERNEL);
	if (!req) {
		printk(KERN_ERR "alg: hash: Failed to allocate request for "
		       "%s\n", algo);
		goto out_noreq;
	}
	ahash_request_set_callback(req, CRYPTO_TFM_REQ_MAY_BACKLOG,
				   tcrypt_complete, &tresult);

	j = 0;
	for (i = 0; i < tcount; i++) {
		if (template[i].np)
			continue;

		ret = -EINVAL;
		if (WARN_ON(align_offset + template[i].psize > PAGE_SIZE))
			goto out;

		j++;
		memset(result, 0, MAX_DIGEST_SIZE);

		hash_buff = xbuf[0];
		hash_buff += align_offset;

		memcpy(hash_buff, template[i].plaintext, template[i].psize);
		sg_init_one(&sg[0], hash_buff, template[i].psize);

		if (template[i].ksize) {
			crypto_ahash_clear_flags(tfm, ~0);
			if (template[i].ksize > MAX_KEYLEN) {
				pr_err("alg: hash: setkey failed on test %d for %s: key size %d > %d\n",
				       j, algo, template[i].ksize, MAX_KEYLEN);
				ret = -EINVAL;
				goto out;
			}
			memcpy(key, template[i].key, template[i].ksize);
			ret = crypto_ahash_setkey(tfm, key, template[i].ksize);
			if (ret) {
				printk(KERN_ERR "alg: hash: setkey failed on "
				       "test %d for %s: ret=%d\n", j, algo,
				       -ret);
				goto out;
			}
		}

		ahash_request_set_crypt(req, sg, result, template[i].psize);
		if (use_digest) {
			ret = wait_async_op(&tresult, crypto_ahash_digest(req));
			if (ret) {
				pr_err("alg: hash: digest failed on test %d "
				       "for %s: ret=%d\n", j, algo, -ret);
				goto out;
			}
		} else {
			ret = wait_async_op(&tresult, crypto_ahash_init(req));
			if (ret) {
				pr_err("alt: hash: init failed on test %d "
				       "for %s: ret=%d\n", j, algo, -ret);
				goto out;
			}
			ret = wait_async_op(&tresult, crypto_ahash_update(req));
			if (ret) {
				pr_err("alt: hash: update failed on test %d "
				       "for %s: ret=%d\n", j, algo, -ret);
				goto out;
			}
			ret = wait_async_op(&tresult, crypto_ahash_final(req));
			if (ret) {
				pr_err("alt: hash: final failed on test %d "
				       "for %s: ret=%d\n", j, algo, -ret);
				goto out;
			}
		}

		if (memcmp(result, template[i].digest,
			   crypto_ahash_digestsize(tfm))) {
			printk(KERN_ERR "alg: hash: Test %d failed for %s\n",
			       j, algo);
			hexdump(result, crypto_ahash_digestsize(tfm));
			ret = -EINVAL;
			goto out;
		} else {
			pr_info("alg: hash: Test %d "
				"passed (psize %5u ksize %5u)\n",
				i, template[i].psize, template[i].ksize);
		}
	}

	j = 0;
	for (i = 0; i < tcount; i++) {
		/* alignment tests are only done with continuous buffers */
		if (align_offset != 0)
			break;

		if (!template[i].np)
			continue;

		j++;
		memset(result, 0, MAX_DIGEST_SIZE);

		temp = 0;
		sg_init_table(sg, template[i].np);
		ret = -EINVAL;
		for (k = 0; k < template[i].np; k++) {
			if (WARN_ON(offset_in_page(IDX[k]) +
				    template[i].tap[k] > PAGE_SIZE))
				goto out;
			sg_set_buf(&sg[k],
				   memcpy(xbuf[IDX[k] >> PAGE_SHIFT] +
					  offset_in_page(IDX[k]),
					  template[i].plaintext + temp,
					  template[i].tap[k]),
				   template[i].tap[k]);
			temp += template[i].tap[k];
		}

		if (template[i].ksize) {
			if (template[i].ksize > MAX_KEYLEN) {
				pr_err("alg: hash: setkey failed on test %d for %s: key size %d > %d\n",
				       j, algo, template[i].ksize, MAX_KEYLEN);
				ret = -EINVAL;
				goto out;
			}
			crypto_ahash_clear_flags(tfm, ~0);
			memcpy(key, template[i].key, template[i].ksize);
			ret = crypto_ahash_setkey(tfm, key, template[i].ksize);

			if (ret) {
				printk(KERN_ERR "alg: hash: setkey "
				       "failed on chunking test %d "
				       "for %s: ret=%d\n", j, algo, -ret);
				goto out;
			}
		}

		ahash_request_set_crypt(req, sg, result, template[i].psize);
		ret = crypto_ahash_digest(req);
		switch (ret) {
		case 0:
			break;
		case -EINPROGRESS:
		case -EBUSY:
			wait_for_completion(&tresult.completion);
			reinit_completion(&tresult.completion);
			ret = tresult.err;
			if (!ret)
				break;
			/* fall through */
		default:
			printk(KERN_ERR "alg: hash: digest failed "
			       "on chunking test %d for %s: "
			       "ret=%d\n", j, algo, -ret);
			goto out;
		}

		if (memcmp(result, template[i].digest,
			   crypto_ahash_digestsize(tfm))) {
			printk(KERN_ERR "alg: hash: Chunking test %d "
			       "failed for %s\n", j, algo);
			hexdump(result, crypto_ahash_digestsize(tfm));
			ret = -EINVAL;
			goto out;
		} else {
			pr_info("alg: hash: Chunking Test %d "
				"passed (psize %5u ksize %5u)\n",
				i, template[i].psize, template[i].ksize);
		}
	}
#if 0
	/* partial update exercise */
	j = 0;
	for (i = 0; i < tcount; i++) {
		pr_info("%s () %d\n", __func__, i);
		/* alignment tests are only done with continuous buffers */
		if (align_offset != 0)
			break;

		if (template[i].np < 2)
			continue;

		j++;
		memset(result, 0, MAX_DIGEST_SIZE);

		ret = -EINVAL;
		hash_buff = xbuf[0];
		memcpy(hash_buff, template[i].plaintext,
			template[i].tap[0]);
		sg_init_one(&sg[0], hash_buff, template[i].tap[0]);

		if (template[i].ksize) {
			crypto_ahash_clear_flags(tfm, ~0);
			if (template[i].ksize > MAX_KEYLEN) {
				pr_err("alg: hash: setkey failed on test %d for %s: key size %d > %d\n",
					j, algo, template[i].ksize, MAX_KEYLEN);
				ret = -EINVAL;
				goto out;
			}
			memcpy(key, template[i].key, template[i].ksize);
			ret = crypto_ahash_setkey(tfm, key, template[i].ksize);
			if (ret) {
				pr_err("alg: hash: setkey failed on test %d for %s: ret=%d\n",
					j, algo, -ret);
				goto out;
			}
		}

		ahash_request_set_crypt(req, sg, result, template[i].tap[0]);
		ret = wait_async_op(&tresult, crypto_ahash_init(req));
		if (ret) {
			pr_err("alt: hash: init failed on test %d for %s: ret=%d\n",
				j, algo, -ret);
			goto out;
		}
		ret = wait_async_op(&tresult, crypto_ahash_update(req));
		if (ret) {
			pr_err("alt: hash: update failed on test %d for %s: ret=%d\n",
				j, algo, -ret);
			goto out;
		}

		temp = template[i].tap[0];
		for (k = 1; k < template[i].np; k++) {
			ret = ahash_partial_update(&req, tfm, &template[i],
				hash_buff, k, temp, &sg[0], algo, result,
				&tresult);
			if (ret) {
				pr_err("hash: partial update failed on test %d for %s: ret=%d\n",
					j, algo, -ret);
				goto out_noreq;
			}
			temp += template[i].tap[k];
		}
		ret = wait_async_op(&tresult, crypto_ahash_final(req));
		if (ret) {
			pr_err("alt: hash: final failed on test %d for %s: ret=%d\n",
				j, algo, -ret);
			goto out;
		}
		if (memcmp(result, template[i].digest,
			   crypto_ahash_digestsize(tfm))) {
			pr_err("alg: hash: Partial Test %d failed for %s\n",
			       j, algo);
			hexdump(result, crypto_ahash_digestsize(tfm));
			ret = -EINVAL;
			goto out;
		}
	}
#endif
	ret = 0;

out:
	ahash_request_free(req);
out_noreq:
	testmgr_free_buf(xbuf);
out_nobuf:
	kfree(key);
	kfree(result);
	return ret;
}

static int test_hash(struct crypto_ahash *tfm, struct hash_testvec *template,
		     unsigned int tcount, bool use_digest)
{
	unsigned int alignmask;
	int ret;

	ret = __test_hash(tfm, template, tcount, use_digest, 0);
	if (ret)
		return ret;

	/* test unaligned buffers, check with one byte offset */
	ret = __test_hash(tfm, template, tcount, use_digest, 1);
	if (ret)
		return ret;

	alignmask = crypto_tfm_alg_alignmask(&tfm->base);
	if (alignmask) {
		pr_info("%s () alignmask %d\n", __func__, alignmask);

		/* Check if alignment mask for tfm is correctly set. */
		ret = __test_hash(tfm, template, tcount, use_digest,
				  alignmask + 1);
		if (ret)
			return ret;
	}

	return 0;
}

static int test_hash_common(const char *algo,
			    struct hash_testvec *template,
			    unsigned int tcount,
			    u32 mask)
{
	struct crypto_ahash *tfm;
	int err;

	tfm = crypto_alloc_ahash(algo, 0, mask);
	if (IS_ERR(tfm)) {
		printk(KERN_ERR "alg: hash: Failed to load transform for %s: "
		       "%ld\n", algo, PTR_ERR(tfm));
		return PTR_ERR(tfm);
	}

	printk(KERN_INFO "\ntesting hash of %s (%s)\n", algo,
			get_driver_name(crypto_ahash, tfm));

	err = test_hash(tfm, template,
			tcount, true);
	if (!err)
		err = test_hash(tfm, template,
				tcount, false);

	crypto_free_ahash(tfm);
	return err;
}

static void test_ahash_speed_common(const char *algo, unsigned int secs,
				    struct hash_speed *speed, unsigned mask)
{
	struct scatterlist sg[TVMEMSIZE];
	struct tcrypt_result tresult;
	struct ahash_request *req;
	struct crypto_ahash *tfm;
	char *output;
	int i, ret;

	tfm = crypto_alloc_ahash(algo, 0, mask);
	if (IS_ERR(tfm)) {
		pr_err("failed to load transform for %s: %ld\n",
		       algo, PTR_ERR(tfm));
		return;
	}

	printk(KERN_INFO "\ntesting speed of async %s (%s)\n", algo,
			get_driver_name(crypto_ahash, tfm));

	if (crypto_ahash_digestsize(tfm) > MAX_DIGEST_SIZE) {
		pr_err("digestsize(%u) > %d\n", crypto_ahash_digestsize(tfm),
		       MAX_DIGEST_SIZE);
		goto out;
	}

	test_hash_sg_init(sg);
	req = ahash_request_alloc(tfm, GFP_KERNEL);
	if (!req) {
		pr_err("ahash request allocation failure\n");
		goto out;
	}

	init_completion(&tresult.completion);
	ahash_request_set_callback(req, CRYPTO_TFM_REQ_MAY_BACKLOG,
				   tcrypt_complete, &tresult);

	output = kmalloc(MAX_DIGEST_SIZE, GFP_KERNEL);
	if (!output)
		goto out_nomem;

	for (i = 0; speed[i].blen != 0; i++) {
		if (speed[i].blen > TVMEMSIZE * PAGE_SIZE) {
			pr_err("template (%u) too big for tvmem (%lu)\n",
			       speed[i].blen, TVMEMSIZE * PAGE_SIZE);
			break;
		}

		if (speed[i].klen)
			crypto_ahash_setkey(tfm, tvmem[0], speed[i].klen);

		pr_info("test%3u "
			"(%5u byte blocks,%5u bytes per update,%4u updates): ",
			i, speed[i].blen, speed[i].plen, speed[i].blen / speed[i].plen);

		ahash_request_set_crypt(req, sg, output, speed[i].plen);

		if (secs)
			ret = test_ahash_jiffies(req, speed[i].blen,
						 speed[i].plen, output, secs);
		else
			ret = test_ahash_cycles(req, speed[i].blen,
						speed[i].plen, output);

		if (ret) {
			pr_err("hashing failed ret=%d\n", ret);
			break;
		}
	}

	kfree(output);

out_nomem:
	ahash_request_free(req);

out:
	crypto_free_ahash(tfm);
}

static void test_ahash_speed(const char *algo, unsigned int secs,
			     struct hash_speed *speed)
{
	return test_ahash_speed_common(algo, secs, speed, 0);
}

static int __test_skcipher(struct crypto_skcipher *tfm, int enc,
			   struct cipher_testvec *template, unsigned int tcount,
			   const bool diff_dst, const int align_offset)
{
	const char *algo =
		crypto_tfm_alg_driver_name(crypto_skcipher_tfm(tfm));
	unsigned int i, j, k, n, temp;
	char *q;
	struct skcipher_request *req;
	struct scatterlist sg[8];
	struct scatterlist sgout[8];
	const char *e, *d;
	struct tcrypt_result result;
	void *data;
	char iv[MAX_IVLEN];
	char *xbuf[XBUFSIZE];
	char *xoutbuf[XBUFSIZE];
	int ret = -ENOMEM;
	unsigned int ivsize = crypto_skcipher_ivsize(tfm);

	if (testmgr_alloc_buf(xbuf))
		goto out_nobuf;

	if (diff_dst && testmgr_alloc_buf(xoutbuf))
		goto out_nooutbuf;

	if (diff_dst)
		d = "-ddst";
	else
		d = "";

	if (enc == ENCRYPT)
	        e = "encryption";
	else
		e = "decryption";

	init_completion(&result.completion);

	req = skcipher_request_alloc(tfm, GFP_KERNEL);
	if (!req) {
		pr_err("alg: skcipher%s: Failed to allocate request for %s\n",
		       d, algo);
		goto out;
	}

	skcipher_request_set_callback(req, CRYPTO_TFM_REQ_MAY_BACKLOG,
				      tcrypt_complete, &result);

	j = 0;
	for (i = 0; i < tcount; i++) {
		if (template[i].np && !template[i].also_non_np)
			continue;

		if (fips_enabled && template[i].fips_skip)
			continue;

		if (template[i].iv)
			memcpy(iv, template[i].iv, ivsize);
		else
			memset(iv, 0, MAX_IVLEN);

		j++;
		ret = -EINVAL;
		if (WARN_ON(align_offset + template[i].ilen > PAGE_SIZE))
			goto out;

		data = xbuf[0];
		data += align_offset;
		memcpy(data, template[i].input, template[i].ilen);

		crypto_skcipher_clear_flags(tfm, ~0);
		if (template[i].wk)
			crypto_skcipher_set_flags(tfm,
						  CRYPTO_TFM_REQ_FORBID_WEAK_KEYS);

		ret = crypto_skcipher_setkey(tfm, template[i].key,
					     template[i].klen);
		if (template[i].fail == !ret) {
			pr_err("alg: skcipher%s: setkey failed on test %d for %s: flags=%x\n",
			       d, j, algo, crypto_skcipher_get_flags(tfm));
			goto out;
		} else if (ret)
			continue;

		sg_init_one(&sg[0], data, template[i].ilen);
		if (diff_dst) {
			data = xoutbuf[0];
			data += align_offset;
			sg_init_one(&sgout[0], data, template[i].ilen);
		}

		skcipher_request_set_crypt(req, sg, (diff_dst) ? sgout : sg,
					   template[i].ilen, iv);
		ret = enc ? crypto_skcipher_encrypt(req) :
			    crypto_skcipher_decrypt(req);

		switch (ret) {
		case 0:
			break;
		case -EINPROGRESS:
		case -EBUSY:
			wait_for_completion(&result.completion);
			reinit_completion(&result.completion);
			ret = result.err;
			if (!ret)
				break;
			/* fall through */
		default:
			pr_err("alg: skcipher%s: %s failed on test %d for %s: ret=%d\n",
			       d, e, j, algo, -ret);
			goto out;
		}

		q = data;
		if (memcmp(q, template[i].result, template[i].rlen)) {
			pr_err("alg: skcipher%s: Test %d failed (invalid result) on %s for %s\n",
			       d, j, e, algo);
			hexdump(q, template[i].rlen);
			ret = -EINVAL;
			goto out;
		} else {
			pr_info("alg: skcipher%s: Test %d passed on %s for %s\n",
			       d, j, e, algo);
		}

		if (template[i].iv_out &&
		    memcmp(iv, template[i].iv_out,
			   crypto_skcipher_ivsize(tfm))) {
			pr_err("alg: skcipher%s: Test %d failed (invalid output IV) on %s for %s\n",
			       d, j, e, algo);
			hexdump(iv, crypto_skcipher_ivsize(tfm));
			ret = -EINVAL;
			goto out;
		}
	}

	j = 0;
	for (i = 0; i < tcount; i++) {
		/* alignment tests are only done with continuous buffers */
		if (align_offset != 0)
			break;

		if (!template[i].np)
			continue;

		if (fips_enabled && template[i].fips_skip)
			continue;

		if (template[i].iv)
			memcpy(iv, template[i].iv, ivsize);
		else
			memset(iv, 0, MAX_IVLEN);

		j++;
		crypto_skcipher_clear_flags(tfm, ~0);
		if (template[i].wk)
			crypto_skcipher_set_flags(tfm,
						  CRYPTO_TFM_REQ_FORBID_WEAK_KEYS);

		ret = crypto_skcipher_setkey(tfm, template[i].key,
					     template[i].klen);
		if (template[i].fail == !ret) {
			pr_err("alg: skcipher%s: setkey failed on chunk test %d for %s: flags=%x\n",
			       d, j, algo, crypto_skcipher_get_flags(tfm));
			goto out;
		} else if (ret)
			continue;

		temp = 0;
		ret = -EINVAL;
		sg_init_table(sg, template[i].np);
		if (diff_dst)
			sg_init_table(sgout, template[i].np);
		for (k = 0; k < template[i].np; k++) {
			if (WARN_ON(offset_in_page(IDX[k]) +
				    template[i].tap[k] > PAGE_SIZE))
				goto out;

			q = xbuf[IDX[k] >> PAGE_SHIFT] + offset_in_page(IDX[k]);

			memcpy(q, template[i].input + temp, template[i].tap[k]);

			if (offset_in_page(q) + template[i].tap[k] < PAGE_SIZE)
				q[template[i].tap[k]] = 0;

			sg_set_buf(&sg[k], q, template[i].tap[k]);
			if (diff_dst) {
				q = xoutbuf[IDX[k] >> PAGE_SHIFT] +
				    offset_in_page(IDX[k]);

				sg_set_buf(&sgout[k], q, template[i].tap[k]);

				memset(q, 0, template[i].tap[k]);
				if (offset_in_page(q) +
				    template[i].tap[k] < PAGE_SIZE)
					q[template[i].tap[k]] = 0;
			}

			temp += template[i].tap[k];
		}

		skcipher_request_set_crypt(req, sg, (diff_dst) ? sgout : sg,
					   template[i].ilen, iv);

		ret = enc ? crypto_skcipher_encrypt(req) :
			    crypto_skcipher_decrypt(req);

		switch (ret) {
		case 0:
			break;
		case -EINPROGRESS:
		case -EBUSY:
			wait_for_completion(&result.completion);
			reinit_completion(&result.completion);
			ret = result.err;
			if (!ret)
				break;
			/* fall through */
		default:
			pr_err("alg: skcipher%s: %s failed on chunk test %d for %s: ret=%d\n",
			       d, e, j, algo, -ret);
			goto out;
		}

		temp = 0;
		ret = -EINVAL;
		for (k = 0; k < template[i].np; k++) {
			if (diff_dst)
				q = xoutbuf[IDX[k] >> PAGE_SHIFT] +
				    offset_in_page(IDX[k]);
			else
				q = xbuf[IDX[k] >> PAGE_SHIFT] +
				    offset_in_page(IDX[k]);

			if (memcmp(q, template[i].result + temp,
				   template[i].tap[k])) {
				pr_err("alg: skcipher%s: Chunk test %d failed on %s at page %u for %s\n",
				       d, j, e, k, algo);
				hexdump(q, template[i].tap[k]);
				goto out;
			}

			q += template[i].tap[k];
			for (n = 0; offset_in_page(q + n) && q[n]; n++)
				;
			if (n) {
				pr_err("alg: skcipher%s: Result buffer corruption in chunk test %d on %s at page %u for %s: %u bytes:\n",
				       d, j, e, k, algo, n);
				hexdump(q, n);
				goto out;
			} else {
				pr_info("alg: skcipher%s: Chunk test %d passed on %s at page %u for %s\n",
					d, j, e, k, algo);
			}
			temp += template[i].tap[k];
		}
	}

	ret = 0;

out:
	skcipher_request_free(req);
	if (diff_dst)
		testmgr_free_buf(xoutbuf);
out_nooutbuf:
	testmgr_free_buf(xbuf);
out_nobuf:
	return ret;
}

static int test_skcipher(struct crypto_skcipher *tfm, int enc,
			 struct cipher_testvec *template, unsigned int tcount)
{
	unsigned int alignmask;
	int ret;

	/* test 'dst == src' case */
	ret = __test_skcipher(tfm, enc, template, tcount, false, 0);
	if (ret)
		return ret;

	/* test 'dst != src' case */
	ret = __test_skcipher(tfm, enc, template, tcount, true, 0);
	if (ret)
		return ret;

	/* test unaligned buffers, check with one byte offset */
	ret = __test_skcipher(tfm, enc, template, tcount, true, 1);
	if (ret)
		return ret;

	alignmask = crypto_tfm_alg_alignmask(&tfm->base);
	if (alignmask) {
		/* Check if alignment mask for tfm is correctly set. */
		ret = __test_skcipher(tfm, enc, template, tcount, true,
				      alignmask + 1);
		if (ret)
			return ret;
	}

	return 0;
}

static int test_skcipher_common(const char *algo,
				int enc,
				struct cipher_testvec *template,
				unsigned int tcount,
				u32 mask)
{
	struct crypto_skcipher *tfm;
	int err = 0;

	tfm = crypto_alloc_skcipher(algo, CRYPTO_ALG_INTERNAL, mask);
	if (IS_ERR(tfm)) {
		printk(KERN_ERR "alg: skcipher: Failed to load transform for "
		       "%s: %ld\n", algo, PTR_ERR(tfm));
		return PTR_ERR(tfm);
	}

	printk(KERN_INFO "\ntesting skchipher encrypt(%d) of %s\n", enc,
	       algo);

	err = test_skcipher(tfm, enc, template, tcount);

	crypto_free_skcipher(tfm);
	return err;
}

static inline int do_one_acipher_op(struct skcipher_request *req, int ret)
{
	if (ret == -EINPROGRESS || ret == -EBUSY) {
		struct tcrypt_result *tr = req->base.data;

		wait_for_completion(&tr->completion);
		reinit_completion(&tr->completion);
		ret = tr->err;
	}

	return ret;
}

static int test_acipher_jiffies(struct skcipher_request *req, int enc,
				int blen, int secs)
{
	unsigned long start, end;
	int bcount;
	int ret;

	for (start = jiffies, end = start + secs * HZ, bcount = 0;
	     time_before(jiffies, end); bcount++) {
		if (enc)
			ret = do_one_acipher_op(req,
						crypto_skcipher_encrypt(req));
		else
			ret = do_one_acipher_op(req,
						crypto_skcipher_decrypt(req));

		if (ret)
			return ret;
	}

	pr_cont("%d operations in %d seconds (%ld bytes)\n",
		bcount, secs, (long)bcount * blen);
	return 0;
}

static int test_acipher_cycles(struct skcipher_request *req, int enc,
			       int blen)
{
	unsigned long cycles = 0;
	int ret = 0;
	int i;

	/* Warm-up run. */
	for (i = 0; i < 4; i++) {
		if (enc)
			ret = do_one_acipher_op(req,
						crypto_skcipher_encrypt(req));
		else
			ret = do_one_acipher_op(req,
						crypto_skcipher_decrypt(req));

		if (ret)
			goto out;
	}

	/* The real thing. */
	for (i = 0; i < 8; i++) {
		cycles_t start, end;

		start = get_cycles();
		if (enc)
			ret = do_one_acipher_op(req,
						crypto_skcipher_encrypt(req));
		else
			ret = do_one_acipher_op(req,
						crypto_skcipher_decrypt(req));
		end = get_cycles();

		if (ret)
			goto out;

		cycles += end - start;
	}

out:
	if (ret == 0)
		pr_cont("1 operation in %lu cycles (%d bytes)\n",
			(cycles + 4) / 8, blen);

	return ret;
}

static void test_skcipher_speed(const char *algo, int enc, unsigned int secs,
				struct cipher_speed_template *template,
				unsigned int tcount, u8 *keysize, bool async)
{
	unsigned int ret, i, j, k, iv_len;
	struct tcrypt_result tresult;
	const char *key;
	char iv[128];
	struct skcipher_request *req;
	struct crypto_skcipher *tfm;
	const char *e;
	u32 *b_size;

	if (enc == ENCRYPT)
		e = "encryption";
	else
		e = "decryption";

	init_completion(&tresult.completion);

	tfm = crypto_alloc_skcipher(algo, 0, async ? 0 : CRYPTO_ALG_ASYNC);

	if (IS_ERR(tfm)) {
		pr_err("failed to load transform for %s: %ld\n", algo,
		       PTR_ERR(tfm));
		return;
	}

	pr_info("\ntesting speed of async %s (%s) %s\n", algo,
			get_driver_name(crypto_skcipher, tfm), e);

	req = skcipher_request_alloc(tfm, GFP_KERNEL);
	if (!req) {
		pr_err("tcrypt: skcipher: Failed to allocate request for %s\n",
		       algo);
		goto out;
	}

	skcipher_request_set_callback(req, CRYPTO_TFM_REQ_MAY_BACKLOG,
				      tcrypt_complete, &tresult);

	i = 0;
	do {
		b_size = block_sizes;

		do {
			struct scatterlist sg[TVMEMSIZE];

			if ((*keysize + *b_size) > TVMEMSIZE * PAGE_SIZE) {
				pr_err("template (%u) too big for "
				       "tvmem (%lu)\n", *keysize + *b_size,
				       TVMEMSIZE * PAGE_SIZE);
				goto out_free_req;
			}

			pr_info("test %u (%d bit key, %d byte blocks): ", i,
				*keysize * 8, *b_size);

			memset(tvmem[0], 0xff, PAGE_SIZE);

			/* set key, plain text and IV */
			key = tvmem[0];
			for (j = 0; j < tcount; j++) {
				if (template[j].klen == *keysize) {
					key = template[j].key;
					break;
				}
			}

			crypto_skcipher_clear_flags(tfm, ~0);

			ret = crypto_skcipher_setkey(tfm, key, *keysize);
			if (ret) {
				pr_err("setkey() failed flags=%x\n",
					crypto_skcipher_get_flags(tfm));
				goto out_free_req;
			}

			k = *keysize + *b_size;
			sg_init_table(sg, DIV_ROUND_UP(k, PAGE_SIZE));

			if (k > PAGE_SIZE) {
				sg_set_buf(sg, tvmem[0] + *keysize,
				   PAGE_SIZE - *keysize);
				k -= PAGE_SIZE;
				j = 1;
				while (k > PAGE_SIZE) {
					sg_set_buf(sg + j, tvmem[j], PAGE_SIZE);
					memset(tvmem[j], 0xff, PAGE_SIZE);
					j++;
					k -= PAGE_SIZE;
				}
				sg_set_buf(sg + j, tvmem[j], k);
				memset(tvmem[j], 0xff, k);
			} else {
				sg_set_buf(sg, tvmem[0] + *keysize, *b_size);
			}

			iv_len = crypto_skcipher_ivsize(tfm);
			if (iv_len)
				memset(&iv, 0xff, iv_len);

			skcipher_request_set_crypt(req, sg, sg, *b_size, iv);

			if (secs)
				ret = test_acipher_jiffies(req, enc,
							   *b_size, secs);
			else
				ret = test_acipher_cycles(req, enc,
							  *b_size);

			if (ret) {
				pr_err("%s() failed flags=%x\n", e,
				       crypto_skcipher_get_flags(tfm));
				break;
			}
			b_size++;
			i++;
		} while (*b_size);
		keysize++;
	} while (*keysize);

out_free_req:
	skcipher_request_free(req);
out:
	crypto_free_skcipher(tfm);
}

static void test_acipher_speed(const char *algo, int enc, unsigned int secs,
			       struct cipher_speed_template *template,
			       unsigned int tcount, u8 *keysize)
{
	return test_skcipher_speed(algo, enc, secs, template, tcount, keysize,
				   true);
}

static int do_test(int m)
{
	int i;
	int ret = 0;

	switch (m) {

	/* CRYPTO_ALG_TYPE_AEAD */
	case 0:
	case 1:
		/* Test Crashing - Skip the test */
		/*
		ret = test_aead_common("gcm-aes-iproc",
				       ENCRYPT,
				       aes_gcm_enc_tv_template,
				       AES_GCM_ENC_TEST_VECTORS,
				       0);
		*/
		if (m > 0 && m < 100) break;
		fallthrough;

	case 2:
		ret = test_aead_common("gcm-aes-iproc",
				       DECRYPT,
				       aes_gcm_dec_tv_template,
				       AES_GCM_DEC_TEST_VECTORS,
				       0);
		if (m > 0 && m < 100) break;
		fallthrough;

	case 3:
		ret = test_aead_common("ccm-aes-iproc",
				       ENCRYPT,
				       aes_ccm_enc_tv_template,
				       AES_CCM_ENC_TEST_VECTORS,
				       0);
		if (m > 0 && m < 100) break;
		fallthrough;

	case 4:
		ret = test_aead_common("ccm-aes-iproc",
				       DECRYPT,
				       aes_ccm_dec_tv_template,
				       AES_CCM_DEC_TEST_VECTORS,
				       0);
		if (m > 0 && m < 100) break;
		fallthrough;

	case 5:
		ret = test_aead_common("gcm-aes-esp-iproc",
				       ENCRYPT,
				       aes_gcm_rfc4106_enc_tv_template,
				       AES_GCM_4106_ENC_TEST_VECTORS,
				       0);
		if (m > 0 && m < 100) break;
		fallthrough;

	case 6:
		ret = test_aead_common("gcm-aes-esp-iproc",
				       DECRYPT,
				       aes_gcm_rfc4106_dec_tv_template,
				       AES_GCM_4106_DEC_TEST_VECTORS,
				       0);
		if (m > 0 && m < 100) break;
		fallthrough;

	case 7:
		ret = test_aead_common("ccm-aes-esp-iproc",
				       ENCRYPT,
				       aes_ccm_rfc4309_enc_tv_template,
				       AES_CCM_4309_ENC_TEST_VECTORS,
				       0);
		if (m > 0 && m < 100) break;
		fallthrough;

	case 8:
		ret = test_aead_common("ccm-aes-esp-iproc",
				       DECRYPT,
				       aes_ccm_rfc4309_dec_tv_template,
				       AES_CCM_4309_DEC_TEST_VECTORS,
				       0);
		if (m > 0 && m < 100) break;
		fallthrough;

	case 9:
		ret = test_aead_common("gmac-aes-esp-iproc",
				       ENCRYPT,
				       aes_gcm_rfc4543_enc_tv_template,
				       AES_GCM_4543_ENC_TEST_VECTORS,
				       0);
		if (m > 0 && m < 100) break;
		fallthrough;

	case 10:
		ret = test_aead_common("gmac-aes-esp-iproc",
				       DECRYPT,
				       aes_gcm_rfc4543_dec_tv_template,
				       AES_GCM_4543_DEC_TEST_VECTORS,
				       0);
		if (m > 0 && m < 100) break;
		fallthrough;

	case 11:
		ret = test_aead_common("authenc-hmac-sha1-cbc-aes-iproc",
				       ENCRYPT,
				       hmac_sha1_aes_cbc_enc_tv_temp,
				       HMAC_SHA1_AES_CBC_ENC_TEST_VEC,
				       0);
		if (m > 0 && m < 100) break;
		fallthrough;

	case 12:
		ret = test_aead_common("authenc-hmac-sha256-cbc-aes-iproc",
				       ENCRYPT,
				       hmac_sha256_aes_cbc_enc_tv_temp,
				       HMAC_SHA256_AES_CBC_ENC_TEST_VEC,
				       0);
		if (m > 0 && m < 100) break;
		fallthrough;

	case 13:
		ret = test_aead_common("authenc-hmac-sha1-cbc-des-iproc",
				       ENCRYPT,
				       hmac_sha1_des_cbc_enc_tv_temp,
				       HMAC_SHA1_DES_CBC_ENC_TEST_VEC,
				       0);
		if (m > 0 && m < 100) break;
		fallthrough;

	case 14:
		ret = test_aead_common("authenc-hmac-sha224-cbc-des-iproc",
				       ENCRYPT,
				       hmac_sha224_des_cbc_enc_tv_temp,
				       HMAC_SHA224_DES_CBC_ENC_TEST_VEC,
				       0);
		if (m > 0 && m < 100) break;
		fallthrough;

	case 15:
		ret = test_aead_common("authenc-hmac-sha256-cbc-des-iproc",
				       ENCRYPT,
				       hmac_sha256_des_cbc_enc_tv_temp,
				       HMAC_SHA256_DES_CBC_ENC_TEST_VEC,
				       0);
		if (m > 0 && m < 100) break;
		fallthrough;

	case 16:
		ret = test_aead_common("authenc-hmac-sha384-cbc-des-iproc",
				       ENCRYPT,
				       hmac_sha384_des_cbc_enc_tv_temp,
				       HMAC_SHA384_DES_CBC_ENC_TEST_VEC,
				       0);
		if (m > 0 && m < 100) break;
		fallthrough;

	case 17:
		ret = test_aead_common("authenc-hmac-sha512-cbc-des-iproc",
				       ENCRYPT,
				       hmac_sha512_des_cbc_enc_tv_temp,
				       HMAC_SHA512_DES_CBC_ENC_TEST_VEC,
				       0);
		if (m > 0 && m < 100) break;
		fallthrough;

	case 18:
		ret = test_aead_common("authenc-hmac-sha1-cbc-des3-iproc",
				       ENCRYPT,
				       hmac_sha1_des3_ede_cbc_enc_tv_temp,
				       HMAC_SHA1_DES3_EDE_CBC_ENC_TEST_VEC,
				       0);
		if (m > 0 && m < 100) break;
		fallthrough;

	case 19:
		ret = test_aead_common("authenc-hmac-sha224-cbc-des3-iproc",
				       ENCRYPT,
				       hmac_sha224_des3_ede_cbc_enc_tv_temp,
				       HMAC_SHA224_DES3_EDE_CBC_ENC_TEST_VEC,
				       0);
		if (m > 0 && m < 100) break;
		fallthrough;

	case 20:
		ret = test_aead_common("authenc-hmac-sha256-cbc-des3-iproc",
				       ENCRYPT,
				       hmac_sha256_des3_ede_cbc_enc_tv_temp,
				       HMAC_SHA256_DES3_EDE_CBC_ENC_TEST_VEC,
				       0);
		if (m > 0 && m < 100) break;
		fallthrough;

	case 21:
		ret = test_aead_common("authenc-hmac-sha384-cbc-des3-iproc",
				       ENCRYPT,
				       hmac_sha384_des3_ede_cbc_enc_tv_temp,
				       HMAC_SHA384_DES3_EDE_CBC_ENC_TEST_VEC,
				       0);
		if (m > 0 && m < 100) break;
		fallthrough;

	case 22:
		ret = test_aead_common("authenc-hmac-sha512-cbc-des3-iproc",
				       ENCRYPT,
				       hmac_sha512_des3_ede_cbc_enc_tv_temp,
				       HMAC_SHA512_DES3_EDE_CBC_ENC_TEST_VEC,
				       0);
		if (m > 0 && m < 100) break;
		fallthrough;

	case 99:
		break;

	/* CRYPTO_ALG_TYPE_ABLKCIPHER */
	case 100:
	case 101:
		ret = test_skcipher_common("ecb-arc4-iproc",
					   ENCRYPT,
					   arc4_enc_tv_template,
					   ARC4_ENC_TEST_VECTORS,
					   0);
		if (m > 100 && m < 200) break;
		fallthrough;

	case 102:
		ret = test_skcipher_common("ecb-arc4-iproc",
					   DECRYPT,
					   arc4_dec_tv_template,
					   ARC4_DEC_TEST_VECTORS,
					   0);
		if (m > 100 && m < 200) break;
		fallthrough;

	case 103:
		ret = test_skcipher_common("cbc-des-iproc",
					   ENCRYPT,
					   des_cbc_enc_tv_template,
					   DES_CBC_ENC_TEST_VECTORS,
					   0);
		if (m > 100 && m < 200) break;
		fallthrough;

	case 104:
		ret = test_skcipher_common("cbc-des-iproc",
					   DECRYPT,
					   des_cbc_dec_tv_template,
					   DES_CBC_DEC_TEST_VECTORS,
					   0);
		if (m > 100 && m < 200) break;
		fallthrough;

	case 105:
		ret = test_skcipher_common("ecb-des-iproc",
					   ENCRYPT,
					   des_enc_tv_template,
					   DES_ENC_TEST_VECTORS,
					   0);
		if (m > 100 && m < 200) break;
		fallthrough;

	case 106:
		ret = test_skcipher_common("ecb-des-iproc",
					   DECRYPT,
					   des_dec_tv_template,
					   DES_DEC_TEST_VECTORS,
					   0);
		if (m > 100 && m < 200) break;
		fallthrough;

	case 107:
		ret = test_skcipher_common("cbc-des3-iproc",
					   ENCRYPT,
					   des3_ede_cbc_enc_tv_template,
					   DES3_EDE_CBC_ENC_TEST_VECTORS,
					   0);
		if (m > 100 && m < 200) break;
		fallthrough;

	case 108:
		ret = test_skcipher_common("cbc-des3-iproc",
					   DECRYPT,
					   des3_ede_cbc_dec_tv_template,
					   DES3_EDE_CBC_DEC_TEST_VECTORS,
					   0);
		if (m > 100 && m < 200) break;
		fallthrough;

	case 109:
		ret = test_skcipher_common("ecb-des3-iproc",
					   ENCRYPT,
					   des3_ede_enc_tv_template,
					   DES3_EDE_ENC_TEST_VECTORS,
					   0);
		if (m > 100 && m < 200) break;
		fallthrough;

	case 110:
		ret = test_skcipher_common("ecb-des3-iproc",
					   DECRYPT,
					   des3_ede_dec_tv_template,
					   DES3_EDE_DEC_TEST_VECTORS,
					   0);
		if (m > 100 && m < 200) break;
		fallthrough;

	case 111:
		ret = test_skcipher_common("ofb-aes-iproc",
					   ENCRYPT,
					   aes_ofb_enc_tv_template,
					   AES_OFB_ENC_TEST_VECTORS,
					   0);
		if (m > 100 && m < 200) break;
		fallthrough;

	case 112:
		ret = test_skcipher_common("ofb-aes-iproc",
					   DECRYPT,
					   aes_ofb_dec_tv_template,
					   AES_OFB_DEC_TEST_VECTORS,
					   0);
		if (m > 100 && m < 200) break;
		fallthrough;

	case 113:
		ret = test_skcipher_common("cbc-aes-iproc",
					   ENCRYPT,
					   aes_cbc_enc_tv_template,
					   AES_CBC_ENC_TEST_VECTORS,
					   0);
		if (m > 100 && m < 200) break;
		fallthrough;

	case 114:
		ret = test_skcipher_common("cbc-aes-iproc",
					   DECRYPT,
					   aes_cbc_dec_tv_template,
					   AES_CBC_DEC_TEST_VECTORS,
					   0);
		if (m > 100 && m < 200) break;
		fallthrough;

	case 115:
		ret = test_skcipher_common("ecb-aes-iproc",
					   ENCRYPT,
					   aes_enc_tv_template,
					   AES_ENC_TEST_VECTORS,
					   0);
		if (m > 100 && m < 200) break;
		fallthrough;

	case 116:
		ret = test_skcipher_common("ecb-aes-iproc",
					   DECRYPT,
					   aes_dec_tv_template,
					   AES_DEC_TEST_VECTORS,
					   0);
		if (m > 100 && m < 200) break;
		fallthrough;

	case 117:
		ret = test_skcipher_common("ctr-aes-iproc",
					   ENCRYPT,
					   aes_ctr_enc_tv_template,
					   AES_CTR_ENC_TEST_VECTORS,
					   0);
		if (m > 100 && m < 200) break;
		fallthrough;

	case 118:
		ret = test_skcipher_common("ctr-aes-iproc",
					   DECRYPT,
					   aes_ctr_dec_tv_template,
					   AES_CTR_DEC_TEST_VECTORS,
					   0);
		if (m > 100 && m < 200) break;
		fallthrough;

	case 119:
		ret = test_skcipher_common("xts-aes-iproc",
					   ENCRYPT,
					   aes_xts_enc_tv_template,
					   AES_XTS_ENC_TEST_VECTORS,
					   0);
		if (m > 100 && m < 200) break;
		fallthrough;

	case 120:
		ret = test_skcipher_common("xts-aes-iproc",
					   DECRYPT,
					   aes_xts_dec_tv_template,
					   AES_XTS_DEC_TEST_VECTORS,
					   0);
		if (m > 100 && m < 200) break;
		fallthrough;

	case 199:
		break;

	/* CRYPTO_ALG_TYPE_AHASH */
	case 200:
	case 201:
		ret = test_hash_common("md5-iproc",
				       md5_tv_template,
				       MD5_TEST_VECTORS,
				       0);
		if (m > 200 && m < 300) break;
		fallthrough;

	case 202:
		ret = test_hash_common("hmac-md5-iproc",
				       hmac_md5_tv_template,
				       HMAC_MD5_TEST_VECTORS,
				       0);
		if (m > 200 && m < 300) break;
		fallthrough;

	case 203:
		ret = test_hash_common("sha1-iproc",
				       sha1_tv_template,
				       SHA1_TEST_VECTORS,
				       0);
		if (m > 200 && m < 300) break;
		fallthrough;

	case 204:
		ret = test_hash_common("hmac-sha1-iproc",
				       hmac_sha1_tv_template,
				       HMAC_SHA1_TEST_VECTORS,
				       0);
		if (m > 200 && m < 300) break;
		fallthrough;

	case 205:
		ret = test_hash_common("sha224-iproc",
				       sha224_tv_template,
				       SHA224_TEST_VECTORS,
				       0);
		if (m > 200 && m < 300) break;
		fallthrough;

	case 206:
		ret = test_hash_common("hmac-sha224-iproc",
				       hmac_sha224_tv_template,
				       HMAC_SHA224_TEST_VECTORS,
				       0);
		if (m > 200 && m < 300) break;
		fallthrough;

	case 207:
		ret = test_hash_common("sha256-iproc",
				       sha256_tv_template,
				       SHA256_TEST_VECTORS,
				       0);
		if (m > 200 && m < 300) break;
		fallthrough;

	case 208:
		ret = test_hash_common("hmac-sha256-iproc",
				       hmac_sha256_tv_template,
				       HMAC_SHA256_TEST_VECTORS,
				       0);
		if (m > 200 && m < 300) break;
		fallthrough;

	case 209:
		ret = test_hash_common("sha384-iproc",
				       sha384_tv_template,
				       SHA384_TEST_VECTORS,
				       0);
		if (m > 200 && m < 300) break;
		fallthrough;

	case 210:
		ret = test_hash_common("hmac-sha384-iproc",
				       hmac_sha384_tv_template,
				       HMAC_SHA384_TEST_VECTORS,
				       0);
		if (m > 200 && m < 300) break;
		fallthrough;

	case 211:
		ret = test_hash_common("sha512-iproc",
				       sha512_tv_template,
				       SHA512_TEST_VECTORS,
				       0);
		if (m > 200 && m < 300) break;
		fallthrough;

	case 212:
		ret = test_hash_common("hmac-sha512-iproc",
				       hmac_sha512_tv_template,
				       HMAC_SHA512_TEST_VECTORS,
				       0);
		if (m > 200 && m < 300) break;
		fallthrough;

	case 213:
		ret = test_hash_common("xcbc-aes-iproc",
				       aes_xcbc128_tv_template,
				       XCBC_AES_TEST_VECTORS,
				       0);
		if (m > 200 && m < 300) break;
		fallthrough;

	case 299:
		break;

	/* CRYPTO_ALG_TYPE_AEAD Speed Test */
	case 1000:
	case 1001:
		test_aead_speed("gcm-aes-iproc", ENCRYPT, sec,
				NULL, 0, 16, 8, speed_template_16_24_32);
		if (m > 1000 && m < 1100) break;
		fallthrough;

	case 1003:
		test_aead_speed("gcm-aes-esp-iproc", ENCRYPT, sec,
				NULL, 0, 16, 16, aead_speed_template_20);
		if (m > 1000 && m < 1100) break;
		fallthrough;

	case 1004:
		test_aead_speed("ccm-aes-esp-iproc", ENCRYPT, sec,
				NULL, 0, 16, 16, aead_speed_template_19);
		if (m > 1000 && m < 1100) break;
		fallthrough;

	case 1099:
		break;

	/* CRYPTO_ALG_TYPE_ABLKCIPHER Speed Test */
	case 1100:
	case 1101:
		test_acipher_speed("ecb-arc4-iproc", ENCRYPT, sec, NULL, 0,
				   speed_template_8);
		if (m > 1100 && m < 1200) break;
		fallthrough;

	case 1102:
		test_acipher_speed("ofb-des-iproc", ENCRYPT, sec, NULL, 0,
				   speed_template_8);
		fallthrough;
	case 1103:
		test_acipher_speed("ofb-des-iproc", DECRYPT, sec, NULL, 0,
				   speed_template_8);
		if (m > 1100 && m < 1200) break;
		fallthrough;

	case 1104:
		test_acipher_speed("cbc-des-iproc", ENCRYPT, sec, NULL, 0,
				   speed_template_8);
		fallthrough;
	case 1105:
		test_acipher_speed("cbc-des-iproc", DECRYPT, sec, NULL, 0,
				   speed_template_8);
		if (m > 1100 && m < 1200) break;
		fallthrough;

	case 1106:
		test_acipher_speed("ecb-des-iproc", ENCRYPT, sec, NULL, 0,
				   speed_template_8);
		fallthrough;
	case 1107:
		test_acipher_speed("ecb-des-iproc", DECRYPT, sec, NULL, 0,
				   speed_template_8);
		if (m > 1100 && m < 1200) break;
		fallthrough;

	case 1108:
		test_acipher_speed("ofb-des3-iproc", ENCRYPT, sec,
				   des3_speed_template, DES3_SPEED_VECTORS,
				   speed_template_24);
		fallthrough;
	case 1109:
		test_acipher_speed("ofb-des3-iproc", DECRYPT, sec,
				   des3_speed_template, DES3_SPEED_VECTORS,
				   speed_template_24);
		if (m > 1100 && m < 1200) break;
		fallthrough;

	case 1110:
		test_acipher_speed("cbc-des3-iproc", ENCRYPT, sec,
				   des3_speed_template, DES3_SPEED_VECTORS,
				   speed_template_24);
		fallthrough;
	case 1111:
		test_acipher_speed("cbc-des3-iproc", DECRYPT, sec,
				   des3_speed_template, DES3_SPEED_VECTORS,
				   speed_template_24);
		if (m > 1100 && m < 1200) break;
		fallthrough;

	case 1112:
		test_acipher_speed("ecb-des3-iproc", ENCRYPT, sec,
				   des3_speed_template, DES3_SPEED_VECTORS,
				   speed_template_24);
		fallthrough;
	case 1113:
		test_acipher_speed("ecb-des3-iproc", DECRYPT, sec,
				   des3_speed_template, DES3_SPEED_VECTORS,
				   speed_template_24);
		if (m > 1100 && m < 1200) break;
		fallthrough;

	case 1114:
		test_acipher_speed("ofb-aes-iproc", ENCRYPT, sec, NULL, 0,
				   speed_template_16_24_32);
		fallthrough;
	case 1115:
		test_acipher_speed("ofb-aes-iproc", DECRYPT, sec, NULL, 0,
				   speed_template_16_24_32);
		if (m > 1100 && m < 1200) break;
		fallthrough;

	case 1116:
		test_acipher_speed("cbc-aes-iproc", ENCRYPT, sec, NULL, 0,
				   speed_template_16_24_32);
		fallthrough;
	case 1117:
		test_acipher_speed("cbc-aes-iproc", DECRYPT, sec, NULL, 0,
				   speed_template_16_24_32);
		if (m > 1100 && m < 1200) break;
		fallthrough;

	case 1118:
		test_acipher_speed("ecb-aes-iproc", ENCRYPT, sec, NULL, 0,
				   speed_template_16_24_32);
		fallthrough;
	case 1119:
		test_acipher_speed("ecb-aes-iproc", DECRYPT, sec, NULL, 0,
				   speed_template_16_24_32);
		if (m > 1100 && m < 1200) break;
		fallthrough;

	case 1120:
		test_acipher_speed("ctr-aes-iproc", ENCRYPT, sec, NULL, 0,
				   speed_template_16_24_32);
		fallthrough;
	case 1121:
		test_acipher_speed("ctr-aes-iproc", DECRYPT, sec, NULL, 0,
				   speed_template_16_24_32);
		if (m > 1100 && m < 1200) break;
		fallthrough;

	case 1122:
		test_acipher_speed("xts-aes-iproc", ENCRYPT, sec, NULL, 0,
				   speed_template_32_48_64);
		fallthrough;
	case 1123:
		test_acipher_speed("xts-aes-iproc", DECRYPT, sec, NULL, 0,
				   speed_template_32_48_64);
		if (m > 1100 && m < 1200) break;
		fallthrough;

	case 1199:
		break;

	/* CRYPTO_ALG_TYPE_AHASH Speed test */
	case 1200:
	case 1201:
		test_ahash_speed("md5-iproc", sec, generic_hash_speed_template);
		if (m > 1200 && m < 1300) break;
		fallthrough;

	case 1202:
		test_ahash_speed("hmac-md5-iproc", sec, hash_speed_template_16);
		if (m > 1200 && m < 1300) break;
		fallthrough;

	case 1203:
		test_ahash_speed("sha1-iproc", sec, generic_hash_speed_template);
		if (m > 1200 && m < 1300) break;
		fallthrough;

	case 1204:
		test_ahash_speed("hmac-sha1-iproc", sec, hash_speed_template_16);
		if (m > 1200 && m < 1300) break;
		fallthrough;

	case 1205:
		test_ahash_speed("sha224-iproc", sec, generic_hash_speed_template);
		if (m > 1200 && m < 1300) break;
		fallthrough;

	case 1206:
		test_ahash_speed("hmac-sha224-iproc", sec, hash_speed_template_16);
		if (m > 1200 && m < 1300) break;
		fallthrough;

	case 1207:
		test_ahash_speed("sha256-iproc", sec, generic_hash_speed_template);
		if (m > 1200 && m < 1300) break;
		fallthrough;

	case 1208:
		test_ahash_speed("hmac-sha256-iproc", sec, hash_speed_template_16);
		if (m > 1200 && m < 1300) break;
		fallthrough;

	case 1209:
		test_ahash_speed("sha384-iproc", sec, generic_hash_speed_template);
		if (m > 1200 && m < 1300) break;
		fallthrough;

	case 1210:
		test_ahash_speed("hmac-sha384-iproc", sec, hash_speed_template_16);
		if (m > 1200 && m < 1300) break;
		fallthrough;

	case 1211:
		test_ahash_speed("sha512-iproc", sec, generic_hash_speed_template);
		if (m > 1200 && m < 1300) break;
		fallthrough;

	case 1212:
		test_ahash_speed("hmac-sha512-iproc", sec, hash_speed_template_16);
		if (m > 1200 && m < 1300) break;
		fallthrough;

	case 1213:
		test_ahash_speed("xcbc-aes-iproc", sec, hash_speed_template_16);
		if (m > 1200 && m < 1300) break;
		fallthrough;

	case 1214:
		test_mb_ahash_speed("sha1-iproc", sec, generic_hash_speed_template);
		if (m > 1200 && m < 1300) break;
		fallthrough;

	case 1216:
		test_mb_ahash_speed("sha256-iproc", sec, generic_hash_speed_template);
		if (m > 1200 && m < 1300) break;
		fallthrough;

	case 1217:
		test_mb_ahash_speed("sha512-iproc", sec, generic_hash_speed_template);
		if (m > 1200 && m < 1300) break;
		fallthrough;

	case 1299:
		break;

	case 2000:
		for (i = 1; i < 2000; i++)
			if (i % 100)
				ret += do_test(i);
		break;
	}

	return ret;
}

static int __init bcm_tcrypt_mod_init(void)
{
	int err = -ENOMEM;
	int i;

	for (i = 0; i < TVMEMSIZE; i++) {
		tvmem[i] = (void *)__get_free_page(GFP_KERNEL);
		if (!tvmem[i])
			goto err_free_tv;
	}

	err = do_test(mode);

	if (err) {
		printk(KERN_ERR "tcrypt: one or more tests failed!\n");
		goto err_free_tv;
	}

	err = -EAGAIN;

err_free_tv:
	for (i = 0; i < TVMEMSIZE && tvmem[i]; i++)
		free_page((unsigned long)tvmem[i]);

	return err;
}

/*
 * If an init function is provided, an exit function must also be provided
 * to allow module unload.
 */
static void __exit bcm_tcrypt_mod_fini(void) { }

module_init(bcm_tcrypt_mod_init);
module_exit(bcm_tcrypt_mod_fini);

module_param(mode, int, 0);
module_param(sec, uint, 0);
MODULE_PARM_DESC(sec, "Length in seconds of speed tests "
		      "(defaults to zero which uses CPU cycles instead)");

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("BCM crypto testing module");
