// SPDX-License-Identifier: GPL-2.0-only
/*
 * Copyright © 2014 NVIDIA Corporation
 * Copyright © 2015 Broadcom Corporation
 */

#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/slab.h>
#include <linux/soc/brcmstb/brcmstb.h>
#include <linux/sys_soc.h>

#include <soc/brcmstb/common.h>

#define MFG_REG_INFO "mfg-reg-info"
static u32 family_id;
static u32 product_id;
static u32 manufact_id;
static u32 manufact_bits;
static u32 manufact_mask;
static u32 manufact_shift;
static u32 cm_variant_id;

enum {
	DT_MANUFACT_BITS_POS,
	DT_MANUFACT_MASK_POS,
	DT_MANUFACT_SHIFT_POS,
	DT_MANUFACT_COUNT,
};

static const struct of_device_id brcmstb_machine_match[] = {
	{ .compatible = "brcm,brcmstb", },
	{ }
};

bool soc_is_brcmstb(void)
{
	const struct of_device_id *match;
	struct device_node *root;

	root = of_find_node_by_path("/");
	if (!root)
		return false;

	match = of_match_node(brcmstb_machine_match, root);
	of_node_put(root);

	return match != NULL;
}

u32 brcmstb_get_family_id(void)
{
	return family_id;
}
EXPORT_SYMBOL(brcmstb_get_family_id);

u32 brcmstb_get_product_id(void)
{
	return product_id;
}
EXPORT_SYMBOL(brcmstb_get_product_id);

u32 brcmstb_get_manufact_id(void)
{
	return manufact_id;
}
EXPORT_SYMBOL(brcmstb_get_manufact_id);

u32 brcmstb_get_manufact_bits(void)
{
	return manufact_bits;
}
EXPORT_SYMBOL(brcmstb_get_manufact_bits);

u32 brcmstb_get_manufact_mask(void)
{
	return manufact_mask;
}
EXPORT_SYMBOL(brcmstb_get_manufact_mask);

u32 brcmstb_get_manufact_shift(void)
{
	return manufact_shift;
}
EXPORT_SYMBOL(brcmstb_get_manufact_shift);

u32 brcmcm_get_variant_id(void)
{
	return cm_variant_id;
}
EXPORT_SYMBOL(brcmcm_get_variant_id);

static const struct of_device_id sun_top_ctrl_match[] = {
	{ .compatible = "brcm,bcm7125-sun-top-ctrl", },
	{ .compatible = "brcm,bcm7346-sun-top-ctrl", },
	{ .compatible = "brcm,bcm7358-sun-top-ctrl", },
	{ .compatible = "brcm,bcm7360-sun-top-ctrl", },
	{ .compatible = "brcm,bcm7362-sun-top-ctrl", },
	{ .compatible = "brcm,bcm7420-sun-top-ctrl", },
	{ .compatible = "brcm,bcm7425-sun-top-ctrl", },
	{ .compatible = "brcm,bcm7429-sun-top-ctrl", },
	{ .compatible = "brcm,bcm7435-sun-top-ctrl", },
	{ .compatible = "brcm,brcmstb-sun-top-ctrl", },
	{ }
};

static const struct of_device_id jtag_otp_match[] = {
	{ .compatible = "brcm,brcmstb-jtag-otp", },
	{ }
};

static int __init brcmstb_soc_device_early_init(void)
{
	struct device_node *sun_top_ctrl;
	void __iomem *sun_top_ctrl_base;
	int ret = 0;

	/* We could be on a multi-platform kernel, don't make this fatal but
	 * bail out early
	 */
	sun_top_ctrl = of_find_matching_node(NULL, sun_top_ctrl_match);
	if (!sun_top_ctrl)
		return ret;

	sun_top_ctrl_base = of_iomap(sun_top_ctrl, 0);
	if (!sun_top_ctrl_base) {
		ret = -ENODEV;
		goto out;
	}

	family_id = readl(sun_top_ctrl_base);
	product_id = readl(sun_top_ctrl_base + 0x4);
	iounmap(sun_top_ctrl_base);
out:
	of_node_put(sun_top_ctrl);
	return ret;
}
early_initcall(brcmstb_soc_device_early_init);

static int __init brcmstb_soc_device_init(void)
{
	struct soc_device_attribute *soc_dev_attr;
	struct soc_device *soc_dev;
	struct device_node *fam_prod = NULL;
	void __iomem *fam_prod_base = NULL;
	struct device_node *jtag_otp = NULL;
	void __iomem *jtag_otp_base = NULL;
	int ret = 0;
	unsigned int val[DT_MANUFACT_COUNT] = {0x20, 0xf000, 0xc};

	fam_prod = of_find_matching_node(NULL, sun_top_ctrl_match);
	if (fam_prod) {
		fam_prod_base = of_iomap(fam_prod, 0);
		if (!fam_prod_base) {
			ret = -ENODEV;
			goto out;
		}

		family_id = readl(fam_prod_base);
		product_id = readl(fam_prod_base + 0x4);
	} else {
		fam_prod = of_find_node_by_name(NULL, "bolt");
		if (!fam_prod) {
			ret = -ENODEV;
			goto out;
		}
		ret = of_property_read_u32(fam_prod, "family-id",
					   &family_id);
		if (ret)
			goto out;
		ret = of_property_read_u32(fam_prod, "product-id",
					   &product_id);
		if (ret)
			goto out;
	}
	pr_debug("family ID: 0x%08x\n", family_id);
	pr_debug("product ID: 0x%08x\n", product_id);

	jtag_otp = of_find_matching_node(NULL, jtag_otp_match);
	if (jtag_otp)
		jtag_otp_base = of_iomap(jtag_otp, 0);

	if (jtag_otp && jtag_otp_base) {
		if (of_property_read_u32_array(jtag_otp,
			MFG_REG_INFO, val, DT_MANUFACT_COUNT)) {
			printk(KERN_ERR "Missing %s property!"
				  "setting default value\n",
				   MFG_REG_INFO);
		}
		manufact_bits = readl(jtag_otp_base + val[DT_MANUFACT_BITS_POS]);
		if (of_machine_is_compatible("brcm,bcm3390") &&
				((family_id >> 16) == 0x3390))
			cm_variant_id = (readl(jtag_otp_base + 0x24) &
					GENMASK(25, 22)) >> 22;
	}
	manufact_mask = val[DT_MANUFACT_MASK_POS];
	manufact_shift = val[DT_MANUFACT_SHIFT_POS];
	manufact_id = (manufact_bits & manufact_mask) >> manufact_shift;

	soc_dev_attr = kzalloc(sizeof(*soc_dev_attr), GFP_KERNEL);
	if (!soc_dev_attr) {
		ret = -ENOMEM;
		goto out;
	}

	soc_dev_attr->family = kasprintf(GFP_KERNEL, "%x",
					 family_id >> 28 ?
					 family_id >> 16 : family_id >> 8);
	soc_dev_attr->soc_id = kasprintf(GFP_KERNEL, "%x",
					 product_id >> 28 ?
					 product_id >> 16 : product_id >> 8);
	soc_dev_attr->revision = kasprintf(GFP_KERNEL, "%c%d",
					 ((product_id & 0xf0) >> 4) + 'A',
					   product_id & 0xf);

	soc_dev = soc_device_register(soc_dev_attr);
	if (IS_ERR(soc_dev)) {
		kfree(soc_dev_attr->family);
		kfree(soc_dev_attr->soc_id);
		kfree(soc_dev_attr->revision);
		kfree(soc_dev_attr);
		ret = -ENOMEM;
	}
out:
	if (fam_prod)
		of_node_put(fam_prod);
	if (fam_prod_base)
		iounmap(fam_prod_base);
	return ret;
}
arch_initcall(brcmstb_soc_device_init);
