/* nf_conntrack handling for offloading flows to acceleration engine
 *
 * Copyright (c) 2016 Broadcom
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation (or any later at your option).
 *
 * Author: Jayesh Patel <jayeshp@broadcom.com>
 */

#include <linux/netfilter.h>
#include <linux/slab.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>

#include <net/netfilter/nf_conntrack.h>
#include <net/netfilter/nf_conntrack_core.h>
#include <net/netfilter/nf_conntrack_l4proto.h>
#include <net/netfilter/nf_conntrack_expect.h>
#include <net/netfilter/nf_conntrack_helper.h>
#include <net/netfilter/nf_conntrack_acct.h>
#include <net/netfilter/nf_conntrack_zones.h>
#include <net/netfilter/nf_conntrack_timestamp.h>
#include <linux/rculist_nulls.h>
#include <net/netfilter/nf_conntrack_offload.h>

static bool nf_ct_offload __read_mostly = 1;

module_param_named(offload, nf_ct_offload, bool, 0644);
MODULE_PARM_DESC(offload, "Enable connection tracking flow offload.");
static DEFINE_MUTEX(nf_ct_offload_mutex);
struct nf_ct_event_notifier __rcu *nf_conntrack_offload_event_cb;

int nf_conntrack_offload_register_notifier(struct net *net,
					   struct nf_ct_event_notifier *new)
{
	int ret;
	struct nf_ct_event_notifier *notify;
	mutex_lock(&nf_ct_offload_mutex);
	notify = rcu_dereference_protected(nf_conntrack_offload_event_cb,
					   lockdep_is_held(&nf_ct_offload_mutex));
	if (notify != NULL) {
		ret = -EBUSY;
		goto out_unlock;
	}
	rcu_assign_pointer(nf_conntrack_offload_event_cb, new);
	ret = 0;

out_unlock:
	mutex_unlock(&nf_ct_offload_mutex);
	return ret;
}
EXPORT_SYMBOL(nf_conntrack_offload_register_notifier);

void nf_conntrack_offload_unregister_notifier(struct net *net,
					      struct nf_ct_event_notifier *new)
{
	struct nf_ct_event_notifier *notify;

	mutex_lock(&nf_ct_offload_mutex);
	notify = rcu_dereference_protected(nf_conntrack_offload_event_cb,
					   lockdep_is_held(&nf_ct_offload_mutex));
	BUG_ON(notify != new);
	RCU_INIT_POINTER(nf_conntrack_offload_event_cb, NULL);
	mutex_unlock(&nf_ct_offload_mutex);
}
EXPORT_SYMBOL(nf_conntrack_offload_unregister_notifier);

void nf_conntrack_offload_destroy(struct nf_conn *ct)
{
	struct nf_conn_offload *ct_offload = nf_conn_offload_find(ct);
	struct nf_ct_event_notifier *notify;
	struct nf_ct_event item = {
		.ct	= ct,
	};
	unsigned int eventmask = 1<<IPCT_DESTROY;
	if (!ct_offload)
		return;
	if (ct_offload->destructor) {
		if (!ct_offload_orig.expected &&
		    ct_offload_orig.flow_id > 0)
			eventmask |= (1<<IPCT_OFFLOAD);
		else if (!ct_offload_repl.expected &&
			 ct_offload_repl.flow_id > 0)
			eventmask |= (1<<IPCT_OFFLOAD);
		ct_offload->destructor(ct, ct_offload);
	}
	rcu_read_lock();
	notify = rcu_dereference(nf_conntrack_offload_event_cb);
	rcu_read_unlock();
	if (notify)
		notify->fcn(eventmask, &item);
	kfree(ct_offload_orig.nf_bridge);
	kfree(ct_offload_repl.nf_bridge);
	kfree(ct_offload_orig.map);
	kfree(ct_offload_repl.map);
}

static struct nf_ct_ext_type offload_extend __read_mostly = {
	.destroy = &nf_conntrack_offload_destroy,
	.len	 = sizeof(struct nf_conn_offload),
	.align	 = __alignof__(struct nf_conn_offload),
	.id	 = NF_CT_EXT_OFFLOAD,
};

#ifdef CONFIG_SYSCTL
static struct ctl_table offload_sysctl_table[] = {
	{
		.procname	= "nf_conntrack_offload",
		.data		= &init_net.ct.sysctl_offload,
		.maxlen		= sizeof(unsigned int),
		.mode		= 0644,
		.proc_handler	= proc_dointvec,
	},
	{}
};

static int nf_conntrack_offload_init_sysctl(struct net *net)
{
	struct ctl_table *table;

	table = kmemdup(offload_sysctl_table, sizeof(offload_sysctl_table),
			GFP_KERNEL);
	if (!table)
		goto out;

	table[0].data = &net->ct.sysctl_offload;

	/* Don't export sysctls to unprivileged users */
	if (net->user_ns != &init_user_ns)
		table[0].procname = NULL;

	net->ct.offload_sysctl_header =
		register_net_sysctl(net, "net/netfilter", table);
	if (!net->ct.offload_sysctl_header) {
		pr_err("nf_conntrack_offload: can't register to sysctl.\n");
		goto out_register;
	}
	return 0;

out_register:
	kfree(table);
out:
	return -ENOMEM;
}

static void nf_conntrack_offload_fini_sysctl(struct net *net)
{
	struct ctl_table *table;

	table = net->ct.offload_sysctl_header->ctl_table_arg;
	unregister_net_sysctl_table(net->ct.offload_sysctl_header);
	kfree(table);
}
#else
static int nf_conntrack_offload_init_sysctl(struct net *net)
{
	return 0;
}

static void nf_conntrack_offload_fini_sysctl(struct net *net)
{
}
#endif

struct ct_iter_state {
	struct seq_net_private p;
	struct hlist_nulls_head *hash;
	unsigned int htable_size;
	unsigned int bucket;
	u_int64_t time_now;
};

static struct hlist_nulls_node *ct_get_first(struct seq_file *seq)
{
	struct ct_iter_state *st = seq->private;
	struct hlist_nulls_node *n;

	for (st->bucket = 0;
	     st->bucket < st->htable_size;
	     st->bucket++) {
		n = rcu_dereference(
			hlist_nulls_first_rcu(&st->hash[st->bucket]));
		if (!is_a_nulls(n))
			return n;
	}
	return NULL;
}

static struct hlist_nulls_node *ct_get_next(struct seq_file *seq,
				      struct hlist_nulls_node *head)
{
	struct ct_iter_state *st = seq->private;

	head = rcu_dereference(hlist_nulls_next_rcu(head));
	while (is_a_nulls(head)) {
		if (likely(get_nulls_value(head) == st->bucket)) {
			if (++st->bucket >= st->htable_size)
				return NULL;
		}
		head = rcu_dereference(
			hlist_nulls_first_rcu(&st->hash[st->bucket]));
	}
	return head;
}

static struct hlist_nulls_node *ct_get_idx(struct seq_file *seq, loff_t pos)
{
	struct hlist_nulls_node *head = ct_get_first(seq);

	if (head)
		while (pos && (head = ct_get_next(seq, head)))
			pos--;
	return pos ? NULL : head;
}

static void *ct_seq_start(struct seq_file *seq, loff_t *pos)
	__acquires(RCU)
{
	struct ct_iter_state *st = seq->private;

	st->time_now = ktime_to_ns(ktime_get_real());
	rcu_read_lock();
	nf_conntrack_get_ht(&st->hash, &st->htable_size);
	return ct_get_idx(seq, *pos);
}

static void *ct_seq_next(struct seq_file *s, void *v, loff_t *pos)
{
	(*pos)++;
	return ct_get_next(s, v);
}

static void ct_seq_stop(struct seq_file *s, void *v)
	__releases(RCU)
{
	rcu_read_unlock();
}

#ifdef CONFIG_NF_CONNTRACK_SECMARK
static void ct_show_secctx(struct seq_file *s, const struct nf_conn *ct)
{
	int ret;
	u32 len;
	char *secctx;

	ret = security_secid_to_secctx(ct->secmark, &secctx, &len);
	if (ret)
		return;

	seq_printf(s, "secctx=%s ", secctx);

	security_release_secctx(secctx, len);
}
#else
static inline void ct_show_secctx(struct seq_file *s, const struct nf_conn *ct)
{
}
#endif

#ifdef CONFIG_NF_CONNTRACK_ZONES
static void ct_show_zone(struct seq_file *s, const struct nf_conn *ct,
			 int dir)
{
	const struct nf_conntrack_zone *zone = nf_ct_zone(ct);

	if (zone->dir != dir)
		return;
	switch (zone->dir) {
	case NF_CT_DEFAULT_ZONE_DIR:
		seq_printf(s, "zone=%u ", zone->id);
		break;
	case NF_CT_ZONE_DIR_ORIG:
		seq_printf(s, "zone-orig=%u ", zone->id);
		break;
	case NF_CT_ZONE_DIR_REPL:
		seq_printf(s, "zone-reply=%u ", zone->id);
		break;
	default:
		break;
	}
}
#else
static inline void ct_show_zone(struct seq_file *s, const struct nf_conn *ct,
				int dir)
{
}
#endif

#ifdef CONFIG_NF_CONNTRACK_TIMESTAMP
static void ct_show_delta_time(struct seq_file *s, const struct nf_conn *ct)
{
	struct ct_iter_state *st = s->private;
	struct nf_conn_tstamp *tstamp;
	s64 delta_time;

	tstamp = nf_conn_tstamp_find(ct);
	if (tstamp) {
		delta_time = st->time_now - tstamp->start;
		if (delta_time > 0)
			delta_time = div_s64(delta_time, NSEC_PER_SEC);
		else
			delta_time = 0;

		seq_printf(s, "delta-time=%llu ",
			   (unsigned long long)delta_time);
	}
	return;
}
#else
static inline void
ct_show_delta_time(struct seq_file *s, const struct nf_conn *ct)
{
}
#endif

static const char* l3proto_name(u16 proto)
{
	switch (proto) {
	case AF_INET: return "ipv4";
	case AF_INET6: return "ipv6";
	}

	return "unknown";
}

static const char* l4proto_name(u16 proto)
{
	switch (proto) {
	case IPPROTO_ICMP: return "icmp";
	case IPPROTO_TCP: return "tcp";
	case IPPROTO_UDP: return "udp";
	case IPPROTO_DCCP: return "dccp";
	case IPPROTO_GRE: return "gre";
	case IPPROTO_SCTP: return "sctp";
	case IPPROTO_UDPLITE: return "udplite";
	case IPPROTO_ESP: return "esp";
	case IPPROTO_AH: return "ah";
	}

	return "unknown";
}

static unsigned int
seq_print_acct(struct seq_file *s, const struct nf_conn *ct, int dir)
{
	struct nf_conn_acct *acct;
	struct nf_conn_counter *counter;

	acct = nf_conn_acct_find(ct);
	if (!acct)
		return 0;

	counter = acct->counter;
	seq_printf(s, "packets=%llu bytes=%llu ",
		   (unsigned long long)atomic64_read(&counter[dir].packets),
		   (unsigned long long)atomic64_read(&counter[dir].bytes));

	return 0;
}

/* return 0 on success, 1 in case of error */
static int ct_seq_show(struct seq_file *s, void *v)
{
	struct nf_conntrack_tuple_hash *hash = v;
	struct nf_conn *ct = nf_ct_tuplehash_to_ctrack(hash);
	const struct nf_conntrack_l4proto *l4proto;
	struct nf_conn_offload *ct_offload = nf_conn_offload_find(ct);
	struct net_device *in = NULL;
	struct net_device *out = NULL;
	struct timeval tv;
	int ret = 0;

	WARN_ON(!ct);
	if (unlikely(!atomic_inc_not_zero(&ct->ct_general.use)))
		return 0;

	if (!ct_offload)
		goto release;

	/* we only want to print DIR_ORIGINAL and flow is offloaded */
	if (NF_CT_DIRECTION(hash) ||
	    ((ct_offload_orig.flow_id == -1) &&
	     (ct_offload_repl.flow_id == -1)))
		goto release;

	if ((ct_offload_orig.flow_id == -1) &&
	     (ct_offload_repl.expected))
		goto release;

	if ((ct_offload_repl.flow_id == -1) &&
	     (ct_offload_orig.expected))
		goto release;

	if (ct_offload->update_stats)
		ct_offload->update_stats(ct);

	l4proto = nf_ct_l4proto_find(nf_ct_protonum(ct));

	ret = -ENOSPC;
	seq_printf(s, "%-8s %u %-8s %u ",
		   l3proto_name(nf_ct_l3num(ct)), nf_ct_l3num(ct),
		   l4proto_name(l4proto->l4proto), nf_ct_protonum(ct));

	if (!test_bit(IPS_OFFLOAD_BIT, &ct->status))
		seq_printf(s, "%ld ", nf_ct_expires(ct)  / HZ);

	if (ct_offload_orig.flow_id == -1) {
		in = __dev_get_by_index(&init_net, ct_offload_orig.iif);
		out = __dev_get_by_index(&init_net, ct_offload_orig.oif);
		seq_printf(s,
		           "flow=no type=%u|%u exp=%d lag=0 src=%pM dst=%pM in=%s out=%s pkt_slow=%d ",
		           ct_offload_orig.flow_type>>16,
		           ct_offload_orig.flow_type&0xFFFF,
		           ct_offload_orig.expected,
		           ct_offload_orig.eh.h_source,
		           ct_offload_orig.eh.h_dest,
		           in ? in->name:"null",
		           out ? out->name:"null",
			   ct_offload_orig.packets_slow);
	} else {
		int wifi_flowring;
		u8  wifi_pri;

		in = __dev_get_by_index(&init_net, ct_offload_orig.iif);
		out = __dev_get_by_index(&init_net, ct_offload_orig.oif);
		if (ct_offload_orig.wifi_flowring != 0xFFFF) {
			wifi_flowring = ct_offload_orig.wifi_flowring;
			wifi_pri = ct_offload_orig.wifi_pri;
		} else {
			wifi_flowring = -1;
			wifi_pri = 0;
		}
		seq_printf(s,
		           "flow=%u type=%u|%u exp=%d lag=%u wifi=%d|%d src=%pM dst=%pM in=%s out=%s pkt_slow=%d ",
		           ct_offload_orig.flow_id,
		           ct_offload_orig.flow_type>>16,
		           ct_offload_orig.flow_type&0xFFFF,
		           ct_offload_orig.expected,
		           ct_offload_orig.lag,
		           wifi_flowring, wifi_pri,
		           ct_offload_orig.eh.h_source,
		           ct_offload_orig.eh.h_dest,
		           in ? in->name:"null",
		           out ? out->name:"null",
			   ct_offload_orig.packets_slow);
		if (ct_offload_orig.dscp_old != ct_offload_orig.dscp_new) {
			seq_printf(s, "dscp=%d->%d ",
			           ct_offload_orig.dscp_old,
			           ct_offload_orig.dscp_new);
		}
		jiffies_to_timeval(jiffies - ct_offload_orig.tstamp, &tv);
		seq_printf(s, "dur=%ld ", tv.tv_sec);
	}

	if (l4proto->print_conntrack)
		l4proto->print_conntrack(s, ct);

	print_tuple(s, &ct->tuplehash[IP_CT_DIR_ORIGINAL].tuple,
		    l4proto);

	ct_show_zone(s, ct, NF_CT_ZONE_DIR_ORIG);

	if (seq_has_overflowed(s))
		goto release;

	if (seq_print_acct(s, ct, IP_CT_DIR_ORIGINAL))
		goto release;

	if (!(test_bit(IPS_SEEN_REPLY_BIT, &ct->status)))
		seq_printf(s, "[UNREPLIED] ");

	seq_printf(s, "-> ");

	if (ct_offload_repl.flow_id == -1) {
		in = __dev_get_by_index(&init_net, ct_offload_repl.iif);
		out = __dev_get_by_index(&init_net, ct_offload_repl.oif);
		seq_printf(s,
		           "flow=no type=%u|%u exp=%d lag=0 src=%pM dst=%pM in=%s out=%s pkt_slow=%d ",
		           ct_offload_repl.flow_type>>16,
		           ct_offload_repl.flow_type&0xFFFF,
		           ct_offload_repl.expected,
		           ct_offload_repl.eh.h_source,
		           ct_offload_repl.eh.h_dest,
		           in ? in->name:"null",
		           out ? out->name:"null",
			   ct_offload_repl.packets_slow);
	} else {
		int wifi_flowring;
		u8  wifi_pri;

		in = __dev_get_by_index(&init_net, ct_offload_repl.iif);
		out = __dev_get_by_index(&init_net, ct_offload_repl.oif);
		if (ct_offload_repl.wifi_flowring != 0xFFFF) {
			wifi_flowring = ct_offload_repl.wifi_flowring;
			wifi_pri = ct_offload_repl.wifi_pri;
		} else {
			wifi_flowring = -1;
			wifi_pri = 0;
		}
		seq_printf(s,
		           "flow=%u type=%u|%u exp=%d lag=%u wifi=%d|%d src=%pM dst=%pM in=%s out=%s pkt_slow=%d ",
		           ct_offload_repl.flow_id,
		           ct_offload_repl.flow_type>>16,
		           ct_offload_repl.flow_type&0xFFFF,
		           ct_offload_repl.expected,
		           ct_offload_repl.lag,
		           wifi_flowring, wifi_pri,
		           ct_offload_repl.eh.h_source,
		           ct_offload_repl.eh.h_dest,
		           in ? in->name:"null",
			   out ? out->name:"null",
			   ct_offload_repl.packets_slow);

		if (ct_offload_repl.dscp_old != ct_offload_repl.dscp_new) {
			seq_printf(s, "dscp=%d->%d ",
			           ct_offload_repl.dscp_old,
				   ct_offload_repl.dscp_new);
		}
		jiffies_to_timeval(jiffies - ct_offload_repl.tstamp, &tv);
		seq_printf(s, "dur=%ld ", tv.tv_sec);
	}

	print_tuple(s, &ct->tuplehash[IP_CT_DIR_REPLY].tuple,
		    l4proto);

	ct_show_zone(s, ct, NF_CT_ZONE_DIR_REPL);

	if (seq_print_acct(s, ct, IP_CT_DIR_REPLY))
		goto release;

	if (test_bit(IPS_OFFLOAD_BIT, &ct->status))
		seq_puts(s, "[OFFLOAD] ");
	if (test_bit(IPS_ASSURED_BIT, &ct->status))
		seq_printf(s, "[ASSURED] ");

	if (seq_has_overflowed(s))
		goto release;

#if defined(CONFIG_NF_CONNTRACK_MARK)
	seq_printf(s, "mark=%u ",
	           ct->mark);
#endif

	ct_show_secctx(s, ct);
	ct_show_zone(s, ct, NF_CT_DEFAULT_ZONE_DIR);
	ct_show_delta_time(s, ct);

	seq_printf(s, "use=%u\n", atomic_read(&ct->ct_general.use));

	if (seq_has_overflowed(s))
		goto release;

	ret = 0;
release:
	nf_ct_put(ct);
	return ret;
}

static int ct_expect_seq_show(struct seq_file *s, void *v)
{
	struct nf_conntrack_tuple_hash *hash = v;
	struct nf_conn *ct = nf_ct_tuplehash_to_ctrack(hash);
	const struct nf_conntrack_l4proto *l4proto;
	struct nf_conn_offload *ct_offload = nf_conn_offload_find(ct);
	struct net_device *in = NULL;
	struct net_device *out = NULL;
	struct timeval tv;
	int ret = 0;

	WARN_ON(!ct);
	if (unlikely(!atomic_inc_not_zero(&ct->ct_general.use)))
		return 0;

	if (!ct_offload)
		goto release;

	/* we only want to print DIR_ORIGINAL and flow is offloaded */
	if (NF_CT_DIRECTION(hash) ||
	    ((ct_offload_orig.flow_id == -1) &&
	     (ct_offload_repl.flow_id == -1)) ||
	    ((!ct_offload_orig.expected) &&
	     (!ct_offload_repl.expected)))
		goto release;

	if (ct_offload->update_stats)
		ct_offload->update_stats(ct);

	l4proto = nf_ct_l4proto_find(nf_ct_protonum(ct));

	ret = -ENOSPC;
	seq_printf(s, "%-8s %u %-8s %u ",
		   l3proto_name(nf_ct_l3num(ct)), nf_ct_l3num(ct),
		   l4proto_name(l4proto->l4proto), nf_ct_protonum(ct));

	if (!test_bit(IPS_OFFLOAD_BIT, &ct->status))
		seq_printf(s, "%ld ", nf_ct_expires(ct)  / HZ);

	if (ct_offload_orig.flow_id == -1)
		goto skip_orig;

	in = __dev_get_by_index(&init_net, ct_offload_orig.iif);
	out = __dev_get_by_index(&init_net, ct_offload_orig.oif);
	seq_printf(s,
	           "in=%s out=%s ",
	           in ? in->name:"null",
	           out ? out->name:"null");

	jiffies_to_timeval(jiffies - ct_offload_orig.tstamp, &tv);
	seq_printf(s, "dur=%ld ", tv.tv_sec);

skip_orig:
	if (l4proto->print_conntrack)
		l4proto->print_conntrack(s, ct);

	print_tuple(s, &ct->tuplehash[IP_CT_DIR_ORIGINAL].tuple,
		    l4proto);

	if (seq_print_acct(s, ct, IP_CT_DIR_ORIGINAL))
		goto release;

	if (!(test_bit(IPS_SEEN_REPLY_BIT, &ct->status)))
		seq_printf(s, "[UNREPLIED] ");

	seq_printf(s, "-> ");

	if (ct_offload_repl.flow_id == -1)
		goto skip_reply;

	in = __dev_get_by_index(&init_net, ct_offload_repl.iif);
	out = __dev_get_by_index(&init_net, ct_offload_repl.oif);
	seq_printf(s,
	           "in=%s out=%s ",
	           in ? in->name:"null",
	           out ? out->name:"null");

skip_reply:
	print_tuple(s, &ct->tuplehash[IP_CT_DIR_REPLY].tuple,
		    l4proto);

	if (seq_print_acct(s, ct, IP_CT_DIR_REPLY))
		goto release;

	if (test_bit(IPS_ASSURED_BIT, &ct->status))
		seq_printf(s, "[ASSURED] ");

#if defined(CONFIG_NF_CONNTRACK_MARK)
	seq_printf(s, "mark=%u ",
	           ct->mark);
#endif

	jiffies_to_timeval(jiffies - ct_offload_repl.tstamp, &tv);
	seq_printf(s, "dur=%ld ", tv.tv_sec);
	seq_printf(s, "use=%u\n", atomic_read(&ct->ct_general.use));

	ret = 0;
release:
	nf_ct_put(ct);
	return ret;
}

static const struct seq_operations ct_seq_ops = {
	.start = ct_seq_start,
	.next  = ct_seq_next,
	.stop  = ct_seq_stop,
	.show  = ct_seq_show
};

static const struct seq_operations ct_expect_seq_ops = {
	.start = ct_seq_start,
	.next  = ct_seq_next,
	.stop  = ct_seq_stop,
	.show  = ct_expect_seq_show
};

static int nf_conntrack_offload_init_proc(struct net *net)
{
	struct proc_dir_entry *pde;

	pde = proc_create_net("nf_conntrack_offload", 0440,
			  net->proc_net, &ct_seq_ops, sizeof(struct ct_iter_state));
	if (!pde)
		return -ENOMEM;

	pde = proc_create_net("nf_conntrack_offload_expect", 0440,
			  net->proc_net, &ct_expect_seq_ops, sizeof(struct ct_iter_state));
	if (!pde)
		return -ENOMEM;

	return 0;
}

static void nf_conntrack_offload_fini_proc(struct net *net)
{
	remove_proc_entry("nf_conntrack_offload", net->proc_net);
}

int nf_conntrack_offload_pernet_init(struct net *net)
{
	int ret = 0;
	net->ct.sysctl_offload = nf_ct_offload;
	ret = nf_conntrack_offload_init_sysctl(net);
	if (ret < 0)
		return ret;

	ret = nf_conntrack_offload_init_proc(net);
	if (ret < 0)
		nf_conntrack_offload_fini_sysctl(net);

	return ret;
}

void nf_conntrack_offload_pernet_fini(struct net *net)
{
	nf_conntrack_offload_fini_proc(net);
	nf_conntrack_offload_fini_sysctl(net);
}

int nf_conntrack_offload_init(void)
{
	int ret = 0;

	ret = nf_ct_extend_register(&offload_extend);
	if (ret < 0) {
		pr_err(KERN_ERR "nf_ct_offload: Unable to register nf extension\n");
		return ret;
	}

	return ret;
}
EXPORT_SYMBOL(nf_conntrack_offload_init);

void nf_conntrack_offload_fini(void)
{
	nf_ct_extend_unregister(&offload_extend);
}
EXPORT_SYMBOL(nf_conntrack_offload_fini);
