 /****************************************************************************
 *
 * Copyright (c) 2016 Broadcom Corporation
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Tim Ross <tross@broadcom.com>
 *****************************************************************************/
#include <linux/kernel.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/io.h>
#include <linux/bcm_media_gw/bpcm.h>

#include "pmb.h"
#include "bpcm_priv.h"

#define PROC_DIR		"driver/bpcm"
#define STATUS_PROC_FILE	"status"

static void *bpcm_status_proc_start(struct seq_file *seq, loff_t *pos)
{
	void *start = NULL;

	if (*pos == 0)
		start = bpcms.next;

	return start;
}

static void bpcm_status_proc_stop(struct seq_file *seq, void *v)
{
}

static void *bpcm_status_proc_next(struct seq_file *seq, void *v,
				loff_t *pos)
{
	struct list_head *l = v;

	if (!l || list_empty(l) || list_is_last(l, &bpcms))
		return NULL;
	(*pos)++;
	return l->next;
}

static int bpcm_status_proc_show(struct seq_file *seq, void *v)
{
	int status = 0;
	struct bpcm_device *bpcm = list_entry(v, struct bpcm_device, list);
	int zone;
	u32 val;
	bool freq_scale_used;

	seq_printf(seq, "%s\n", bpcm->name);
	for (zone = 0; zone < bpcm->zones; zone++) {
		status = pmb_read(bpcm->pmb_dev, bpcm->id, ZONE_CTRL(zone),
				  &val);
		if (status)
			goto done;
		val &= ZONE_CTRL_PWR_ON_STATE_MASK;
		seq_printf(seq, "   zone%02d pwr:           %s\n", zone,
			   val ? "on" : "off");
	}
	status = pmb_read(bpcm->pmb_dev, bpcm->id, ZONE_CTRL(0), &val);
	if (status)
		goto done;
	freq_scale_used = !(!(val & ZONE_CTRL_FREQ_SCALE_USED_MASK));
	status = pmb_read(bpcm->pmb_dev, bpcm->id, ZONE_FREQ_CTRL(0), &val);
	if (status)
		goto done;
	seq_printf(seq, "   dynamic freq scaling: ");
	if (freq_scale_used) {
		seq_printf(seq, "%s\n", val & ZONE_FREQ_USE_DYN_GEAR_SEL_MASK ?
			   "on" : "off");
		seq_printf(seq, "   high gear clk div:    1/%lu\n",
			   BIT((val & ZONE_FREQ_HIGH_GEAR_DIV_MASK) >>
			   ZONE_FREQ_HIGH_GEAR_DIV_SHIFT));
		seq_printf(seq, "   low gear clk div:     1/%lu\n",
			   BIT((val & ZONE_FREQ_LOW_GEAR_DIV_MASK) >>
			   ZONE_FREQ_LOW_GEAR_DIV_SHIFT));
	} else {
		seq_printf(seq, "unavailable\n");
	}


done:
	return status;
}

static const struct seq_operations status_seq_ops = {
	.start	= bpcm_status_proc_start,
	.stop	= bpcm_status_proc_stop,
	.next	= bpcm_status_proc_next,
	.show	= bpcm_status_proc_show,
};

static int bpcm_status_proc_open(struct inode *inode, struct file *file)
{
	return seq_open(file, &status_seq_ops);
}

static const struct file_operations status_fops = {
	.owner		= THIS_MODULE,
	.open		= bpcm_status_proc_open,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= seq_release,
};

static struct proc_dir_entry *proc_dir;
static struct proc_dir_entry *status_proc_file;

void bpcm_proc_exit(void)
{
	if (proc_dir) {
		remove_proc_entry(PROC_DIR, NULL);
		proc_dir = NULL;
	}
}

int __init bpcm_proc_init(void)
{
	int status = 0;

	proc_dir = proc_mkdir(PROC_DIR, NULL);
	if (!proc_dir) {
		pr_err("Failed to create PROC directory %s.\n",
		       PROC_DIR);
		status = -EIO;
		goto done;
	}
	status_proc_file = proc_create(STATUS_PROC_FILE, S_IRUGO,
					proc_dir, &status_fops);
	if (!status_proc_file) {
		pr_err("Failed to create %s\n", STATUS_PROC_FILE);
		status = -EIO;
		bpcm_proc_exit();
		goto done;
	}

done:
	return status;
}
