 /****************************************************************************
 *
 * Broadcom Proprietary and Confidential.
 * (c) 2016 Broadcom. All rights reserved.
 * The term "Broadcom" refers to Broadcomf Limited and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************/

#ifndef _ETHSW_H_
#define _ETHSW_H_

#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/brcmstb/brcmstb.h>

#define ETHSW_MAX_SWITCHS 2

#define ETHSW_INTERNAL_DEVICE_MAX  1
#define ETHSW_EXTERNAL_DEVICE_MAX  1
#define ETHSW_IRQ_MAX           2
#define ETHSW_PORT_MAX          9
#define ETHSW_PHY_PORT_MAX      4
#define SYSTEM_MAX_PORTS  ETHSW_MAX_SWITCHS * ETHSW_PORT_MAX

#define ETHSW_53134_PORT_OFFSET 0
#define ETHSW_53124_PORT_OFFSET 1

#define ETHSW_PORT_ID_INVALID   0xFFFF
#define ETHSW_PORT_MAP_INVALID  0xFFFF

#define ETHSW_SF2_SUPPORT       1
#define ETHSW_SF2_TO_EXT_SWITCH_PORT 4

#define MAC_ADDR_SIZE           6 /* 48 bits */

struct ethsw_device;

/* port config and status */
enum {
	PORT_SPEED_10M = 0,
	PORT_SPEED_100M,
	PORT_SPEED_1000M,
	PORT_SPEED_2500M,
	PORT_SPEED_MAX
};

struct ethsw_port_config {
	bool auto_neg;
	bool full_duplex;
	int  speed;
};

struct ethsw_port_status {
	bool link_up;
	bool full_duplex;
	u32  speed;
};

/* APD (Auto Power Down) params */
struct ethsw_apd_params {
	u32 sleep_timer;            /* in units of 2.7s, max value is 2 */
	u32 wakeup_timer;           /* in units of 84ms, max value is 15 */
};

/* EEE params */
struct ethsw_eee_params {
	bool enable;    	     /* enabled EEE or not */
	u32  advertised;	     /* specify to advertise EEE or not */
	u32  sleep_timer_g;          /* in units of 1us */
	u32  sleep_timer_h;          /* in units of 1us */
	u32  min_lp_timer_g;         /* in units of 1us */
	u32  min_lp_timer_h;         /* in units of 1us */
	u16  wake_timer_g;           /* in units of 1us */
	u16  wake_timer_h;           /* in units of 1us */
};

/* power config and status */
struct ethsw_power_config {
	bool auto_power;
	bool auto_dll;
	bool afe_pwrdn;

	/* If any of the following params is uninitialized (i.e. set to 0),
	 * the default value for the param will be used instead.
	 */
	struct ethsw_apd_params apd_params;
	struct ethsw_eee_params eee_params;
};

struct ethsw_power_status {
	bool power_up;
	bool dll_up;

	bool eee_capable;
	bool eee_active;         /* information on if eee was negotiated ?? */
	bool eee_assert;         /* Tx LPI enabled? */
	bool eee_indicate;       /* Rx LPI enabled? */
	u32  eee_supported;      /* HARD CODED for DEVICE */
	u32  eee_lp_advertised;  /* link partner info for EEE */
	                         /* need more information ?? */
	bool afe_pwrdn;          /* Switch QPHY forced power off. */
};

#define ENABLE_PHY_LPI          (1 << 1)

/* ARL table configuration */
struct ethsw_arl_config {
	bool age_accelerate;
	u16  port_limit[ETHSW_PORT_MAX];
};

enum {
	MIB_COUNTER_START = 0,
	MIB_TX_COUNTER_START = MIB_COUNTER_START,

	/* transmit counters */
	MIB_TX_OCTETS = MIB_TX_COUNTER_START,
	MIB_TX_OCTETS_RSV,          /* reserved for 64-bit counter */
	MIB_TX_DROP_PKTS,
	MIB_TX_BROADCAST_PKTS,
	MIB_TX_MULTICAST_PKTS,
	MIB_TX_UNICAST_PKTS,
	MIB_TX_PAUSE_PKTS,
	MIB_TX_COLLISIONS,
	MIB_TX_SINGLE_COLLISION,
	MIB_TX_MULTIPLE_COLLISION,
	MIB_TX_LATE_COLLISION,
	MIB_TX_EXCESSIVE_COLLISION,
	MIB_TX_DEFERRED_TRANSMIT,
	MIB_TX_FRAME_IN_DISC,

	MIB_TX_Q0_PKT,
	MIB_TX_Q1_PKT,
	MIB_TX_Q2_PKT,
	MIB_TX_Q3_PKT,
	MIB_TX_Q4_PKT,
	MIB_TX_Q5_PKT,

#if ETHSW_SF2_SUPPORT
	MIB_TX_Q6_PKT,
	MIB_TX_Q7_PKT,
	MIB_TX_PKTS_64_OCTETS,
	MIB_TX_PKTS_65TO127_OCTETS,
	MIB_TX_PKTS_128TO255_OCTETS,
	MIB_TX_PKTS_256TO511_OCTETS,
	MIB_TX_PKTS_512TO1023_OCTETS,
	MIB_TX_PKTS_1024TOMAX_OCTETS,
#endif

	MIB_TX_COUNTER_MAX,
	MIB_RX_COUNTER_START = MIB_TX_COUNTER_MAX,

	/* receive counters */
	MIB_RX_OCTETS = MIB_RX_COUNTER_START,
	MIB_RX_OCTETS_RSV,          /* reserved for 64-bit counter */
	MIB_RX_GOOD_OCTETS,
	MIB_RX_GOOD_OCTETS_RSV,     /* reserved for 64-bit counter */
	MIB_RX_DROP_PKTS,
	MIB_RX_BROADCAST_PKTS,
	MIB_RX_MULTICAST_PKTS,
	MIB_RX_UNICAST_PKTS,
	MIB_RX_PAUSE_PKTS,
	MIB_RX_SA_CHANGES,

	MIB_RX_UNDERSIZE_PKTS,
	MIB_RX_PKTS_64_OCTETS,
	MIB_RX_PKTS_65TO127_OCTETS,
	MIB_RX_PKTS_128TO255_OCTETS,
	MIB_RX_PKTS_256TO511_OCTETS,
	MIB_RX_PKTS_512TO1023_OCTETS,
	MIB_RX_PKTS_1024TOMAX_OCTETS,
	MIB_RX_OVERSIZE_PKTS,
	MIB_RX_JUMBO_PKTS,

	MIB_RX_FRAGMENTS,
	MIB_RX_JABBERS,
	MIB_RX_DISCARD,
	MIB_RX_ALIGNMENT_ERRORS,
	MIB_RX_FCS_ERRORS,
	MIB_RX_SYMBOL_ERRORS,
	MIB_RX_IN_RANGE_ERRORS,
	MIB_RX_OUT_OF_RANGE_ERRORS,

	MIB_RX_COUNTER_MAX,
	MIB_EEE_COUNTER_START = MIB_RX_COUNTER_MAX,

	/* EEE counters */
	MIB_EEE_LPI_EVENT = MIB_EEE_COUNTER_START,
	MIB_EEE_LPI_DURATION,

	MIB_EEE_COUNTER_MAX,
	MIB_COUNTER_MAX = MIB_EEE_COUNTER_MAX
};

/* port API functions */
typedef void (* ethsw_port_cb_t)(
	void *cb_priv,
	struct ethsw_port_status *port_status);
int ethsw_port_connect(
	struct net_device *net_dev,
	int *port_id,
	const char *port_name,
	ethsw_port_cb_t cb,
	void *cb_priv);

int ethsw_port_disconnect(int port_id);

int ethsw_port_enable(int port_id);
int ethsw_port_disable(int port_id);

int ethsw_port_config_set(int port_id,
	struct ethsw_port_config *port_config);

int ethsw_port_config_get(int port_id,
	struct ethsw_port_config *port_config);

int ethsw_port_tx_enable(struct ethsw_device *swdev, int port_id,
						int enable);
int ethsw_port_rx_enable(struct ethsw_device *swdev, int port_id,
						int enable);

int ethsw_port_status_get(int port_id,
	struct ethsw_port_status *port_status);

int ethsw_port_status_get_sf2(struct ethsw_device *swdev, int port_id,
	struct ethsw_port_status *port_status);
int ethsw_port_status_get_531xx(struct ethsw_device *swdev, int port_id,
	struct ethsw_port_status *port_status);

bool ethsw_port_link_up(int port_id);
bool ethsw_port_full_duplex(int port_id);

int ethsw_port_power_config_set(int port_id,
	struct ethsw_power_config *power_config);

int ethsw_port_power_config_get(int port_id,
	struct ethsw_power_config *power_config);

int ethsw_port_power_status_get(int port_id,
	struct ethsw_power_status *power_status);

bool ethsw_port_power_up(int port_id);
bool ethsw_port_dll_up(int port_id);
bool ethsw_port_eee_capable(int port_id);
bool ethsw_port_eee_assert(int port_id);
bool ethsw_port_eee_indicate(int port_id);

int ethsw_port_is_enabled(int port_id);

/* switch control functions */
int ethsw_switch_reset(void);
int ethsw_switch_mode_set(bool managed);
int ethsw_switch_mode_set_sf2(bool managed);
int ethsw_switch_mode_set_531xx(bool managed);
int ethsw_switch_mode_get(bool *managed);

int ethsw_switch_cert_set(bool enable);
int ethsw_switch_cert_get(bool *enable);

int ethsw_switch_ethwan_set(bool enable, int port_id);
int ethsw_switch_ethwan_get(bool *enable, int *port_id);
int ethsw_switch_alt_ethwan_get(struct ethsw_device *swdev, int *imp_port, int *lan_port,  bool *enable);
/* ARL control functions */
int ethsw_arl_config_set(struct ethsw_arl_config *arl_cfg);
int ethsw_arl_config_get(struct ethsw_arl_config *arl_cfg);
int ethsw_arl_learning_enable(bool enable);
int ethsw_arl_swlearn_enable(u16 port, bool enable);
int ethsw_arl_hwlearn_enable(u16 port, bool enable);

/* ARL/VTBL access functions */
int ethsw_arl_table_add(u8 *mac, u16 vid, u16 port);
int ethsw_arl_table_delete(u8 *mac, u16 vid);
int ethsw_arl_fast_age_set(int portid);
int ethsw_arl_table_find(u8 *mac, u16 vid, u16 *port);
int ethsw_arl_hw_table_add(u8 *mac, u16 vid, u16 port);

/* Cache ARL/VTBL access functions */
int ethsw_arl_cache_table_add(u8 *mac, u16 vid, u16 port);
int ethsw_arl_cache_table_delete(u8 *mac, u16 vid);
int ethsw_arl_cache_table_find(u8 *mac, u16 vid, u16 *port);

/* multiport control functions */
int ethsw_multiport_add(u8 *mac, u16 etype, u16 port);
int ethsw_multiport_delete(u8 *mac, u16 etype);
int ethsw_multiport_find(u8 *mac, u16 etype, u16 *port);

/* VLAN control functions */
int ethsw_vlan_table_add(struct ethsw_device *swdev, u16 vid, u16 untag_map, u16 fwd_map);
int ethsw_vlan_table_delete(struct ethsw_device *swdev,u16 vid);
int ethsw_vlan_table_find(struct ethsw_device *swdev, u16 vid, u16 *untag_map, u16 *fwd_map);

int ethsw_vlan_port_tag_set(int port_id, u16 vid, u8 pcp);
int ethsw_vlan_port_tag_get(int port_id, u16 *vid, u8 *pcp);

int ethsw_vlan_enable(u8 vid_enable);

int ethsw_vlan_port_vid_add(int port_id, u16 vid, bool untag);
int ethsw_vlan_port_vid_delete(int port_id, u16 vid, bool untag);

int ethsw_vlan_enabled(void);
bool ethsw_ipv6_mc_flood_disabled(void);
bool ethsw_drop_packet(int port, int start_port);

/* port MIB functions */
int ethsw_mib_counters_reset(u16 port_map);
int ethsw_mib_counter32_get(int port_id, int mib_counter, u32 *value);
int ethsw_mib_counter64_get(int port_id, int mib_counter, u64 *value);
int ethsw_mib_counters_get(int port_id, int start, int count, u32 *values);
int ethsw_mib_counters_clear(int port_id);
int ethsw_mib_counters_get_base(struct ethsw_device *swdev, int port_id, int start, int count, u32 *values);

int ethsw_port_set_stp_state(int port_id, unsigned int state);

#endif /* _ETHSW_H_ */
