 /****************************************************************************
 *
 * Copyright (c) 2015 Broadcom Corporation
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************/
#include <linux/init.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/clk.h>
#include <linux/platform_device.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/ioport.h>
#include <brcm_mbox.h>

#include "ethsw.h"
#include "ethsw_priv.h"

/*
 This file contains initialization for ethsw driver when the driver is used as a
 platform (as opposed to spi) type driver.
 */

#define MODULE_NAME	"brcm-ethsw"
#define MODULE_VER	"1.3"


/* define log module */
#define LOG_MODULE "ethsw_init_plat"

static const struct of_device_id sun_top_ctrl_match[] = {
	{ .compatible = "brcm,brcmstb-sun-top-ctrl", },
	{ }
};

static int ethsw_pdev_probe(struct platform_device *pdev);
static int ethsw_pdev_remove(struct platform_device *pdev);

static int __init ethsw_swdev_param_init(struct ethsw_device *swdev)
{
	int ret = 0;
	struct platform_device *pdev;
	struct device_node *node;

	FUNC_ENTER();

	pdev = swdev->pdev;
	node = pdev->dev.of_node;

	/* only used for 3384 w/external switch */
	swdev->vlan_ipv6_mc_nf_mode_enabled = false;

	/* get ethsw device name */
	if (of_property_read_string(node, "dev-name", &swdev->name)) {
		LOG_ERR("Unable to obtain name string\n");
		ret = -EINVAL;
		goto EXIT;
	}

	/* get MII port params */
	swdev->pm_params = of_get_property(node, "mii-port", &swdev->pm_count);
	if (!swdev->pm_params) {
		LOG_DEBUG("Unable to obtain MII port properties\n");
		swdev->pm_count = 0;
	} else {
		/* convert size to 3-word group */
		swdev->pm_count /= (sizeof(int) * 3);
	}

	/* get IMP port params */
	swdev->pi_params = of_get_property(node, "imp-port", &swdev->pi_count);
	if (!swdev->pi_params) {
		LOG_ERR("Unable to obtain IMP port properties\n");
		ret = -EINVAL;
		goto EXIT;
	}

	/* convert size to 2-word group */
	swdev->pi_count /= (sizeof(int) * 2);

	/* get LED map */
	if (of_property_read_u32(node, "led-map", &swdev->led_map)) {
		/* default to LED on for all ports */
		swdev->led_map = -1;
	}

	if (of_property_read_u32(node, "eee-support", &swdev->eee_support)) {
		LOG_CRIT("Unable to obtain Energy-Efficient Ethernet settings\n");
		ret = -EINVAL;
		goto EXIT;
	}

	if (of_property_read_u32(node, "phy-port-mask", &swdev->phy_port_mask)) {
		swdev->phy_port_mask = 0xF;
	}

	if (of_property_read_u32(node, "subid-port-offset", &swdev->subid_port_offset)) {
		LOG_CRIT("Unable to obtain subid-port-offset\n");
		ret = -EINVAL;
		goto EXIT;
	}

	if (of_property_read_u32(node, "high-speed-imp", &swdev->high_speed_imp)) {
		LOG_CRIT("Unable to obtain high speed imp settings\n");
		ret = -EINVAL;
		goto EXIT;
	}

	if (!of_find_property(node, "vlan-default-mode", NULL)) {
		/* if we can not find it then we default to vlans on*/
		swdev->vlan_default_mode = 1;
	} else {
		if (of_property_read_u32(node, "vlan-default-mode", &swdev->vlan_default_mode)) {
			LOG_CRIT("Unable to obtain vlan default mode setting\n");
			ret = -EINVAL;
			goto EXIT;
		}
	}

	/* do clock */
	swdev->clk = of_clk_get_by_name(node, "sw_mac");
	if (IS_ERR(swdev->clk)) {
		if (PTR_ERR(swdev->clk) == -EPROBE_DEFER) {
			ret = -EPROBE_DEFER;
			goto EXIT;
		}

		LOG_NOTICE("Unable to get sw_mac clock\n");
		swdev->clk = NULL;
	}

	ret = clk_prepare_enable(swdev->clk);

 EXIT:
	FUNC_LEAVE();
	return(ret);
}

static int __init ethsw_swdev_phy_param_init(struct ethsw_device *swdev)
{
	int i, port_id, ret = 0;
	struct platform_device *pdev;
	struct device_node *node;
	struct ethsw_port *port;
	int num_of_ports_defined = 0;

	FUNC_ENTER();

	pdev = swdev->pdev;
	node = pdev->dev.of_node;
	swdev->phy_min_addr = PHY_MAX_ADDR;

	for (i = 0; i < ETHSW_PHY_PORT_MAX; i++) {
		if (!(swdev->phy_port_mask & (1 << i)))
			continue;

		port_id = swdev->ethsw_port_idx_to_port_id(swdev, i);
		port = &swdev->port[port_id];
		atomic_set(&port->phy.state, ETHSW_PHY_STOP);
		port->phy.node = of_parse_phandle(node, "phy-handles", num_of_ports_defined);
		if (!port->phy.node) {
			break;
		}

		num_of_ports_defined++;

		of_node_put(port->phy.node);
		ret = ethsw_phy_get_property(&swdev->pdev->dev, port, port_id);
		if (ret)
			goto EXIT;

		if (port->phy.id < swdev->phy_min_addr)
			swdev->phy_min_addr = port->phy.id - i;

		port_id = swdev->subid_port_offset + i;
		subid_to_swdev_table[port_id] = swdev;
	}

	port_id = 4;
	port = &swdev->port[port_id];
	port->phy.node = of_parse_phandle(node, "p4-handle", 0);
	if (port->phy.node) {
		of_node_put(port->phy.node);
		ret = ethsw_phy_get_property(&swdev->pdev->dev, port, port_id);
		if (ret)
			goto EXIT;

		subid_to_swdev_table[port_id] = swdev;
	}

	port_id = 7;
	port = &swdev->port[port_id];
	port->phy.node = of_parse_phandle(node, "xcvr-handles", 0);
	if (port->phy.node) {
		of_node_put(port->phy.node);
		ret = ethsw_phy_get_property(&swdev->pdev->dev, port, port_id);
		if (ret)
			goto EXIT;

		subid_to_swdev_table[port_id] = swdev;
		port->ethsw_phy_set_afe_power_state = ethsw_phy_set_afe_power_state;
		port->ethsw_phy_get_afe_power_state = ethsw_phy_get_afe_power_state;
		ret = ethsw_phy_get_pwr_rst_pin(&swdev->pdev->dev, port);
		if (ret)
			goto EXIT;

		if (port->ethsw_phy_set_power_state)
			ret = port->ethsw_phy_set_power_state(swdev,
					port_id, false);
	} else {
		port->phy.node = of_parse_phandle(node, "p7-handle", 0);
		if (port->phy.node) {
			of_node_put(port->phy.node);
			ret = ethsw_phy_get_property(&swdev->pdev->dev, port, port_id);
			if (ret)
				goto EXIT;

			subid_to_swdev_table[port_id] = swdev;
		}
	}

 EXIT:
	FUNC_LEAVE();
	return(ret);
}

static int ethsw_pdev_probe(struct platform_device *pdev)
{
	int ret = 0;
	int i;
	struct ethsw_device *swdev;
	struct resource *mres;
	struct device_node *sun_top_ctrl;

	FUNC_ENTER();

	/* alloc ethsw device control block */
	swdev = kzalloc(sizeof(struct ethsw_device), GFP_KERNEL);
	if (!swdev) {
		LOG_CRIT("Ethsw device control block alloc failed\n");
		ret = -ENOMEM;
		goto ERROR;
	}

	pdev->dev.platform_data = swdev;
	swdev->pdev = pdev;
	swdev->ethsw_creg_read = ethsw_creg_read_sf2;
	swdev->ethsw_creg_write = ethsw_creg_write_sf2;
	swdev->ethsw_phy_board_init = ethsw_phy_board_init_sf2;
	swdev->ethsw_led_board_init = ethsw_led_board_init_sf2;
	swdev->ethsw_int_board_init = ethsw_int_board_init_sf2;
	swdev->ethsw_switch_set_port_stp_state_specific = ethsw_switch_set_port_stp_state_specific_sf2;
	swdev->ethsw_int_board_isr = ethsw_int_board_isr_sf2;
	swdev->ethsw_port_idx_to_phy_id = ethsw_port_idx_to_phy_id_sf2;
	swdev->ethsw_port_idx_to_port_id = ethsw_port_idx_to_port_id_sf2;
	swdev->ethsw_port_id_to_port_idx = ethsw_port_id_to_port_idx_sf2;
	swdev->ethsw_led_board_exit = ethsw_led_board_exit_sf2;
	swdev->ethsw_led_powercntrl = ethsw_led_powercntrl_sf2;
	swdev->ethsw_phy_board_exit = ethsw_phy_board_exit_sf2;
	swdev->ethsw_int_board_proc = ethsw_int_board_proc_sf2;
	swdev->ethsw_swdev_board_init = ethsw_swdev_board_init_sf2;
	swdev->ethsw_swdev_board_exit = ethsw_swdev_board_exit_sf2;
	swdev->ethsw_int_board_exit = ethsw_int_board_exit_sf2;
	swdev->ethsw_port_status_get = ethsw_port_status_get_sf2;
	swdev->ethsw_low_pwr_mode = ethsw_core_low_power_mode_sf2;
	swdev->ethsw_normal_pwr_mode = ethsw_core_normal_power_mode_sf2;
	swdev->imp_port_count = 0;

	for (i = 0; i < ETHSW_PHY_PORT_MAX; i++) {
		swdev->port[i].ethsw_phy_set_afe_power_state = ethsw_phy_set_afe_power_state;
		swdev->port[i].ethsw_phy_get_afe_power_state = ethsw_phy_get_afe_power_state;
	}

	/* update ethsw driver */
	for (i = 0 ; i < ETHSW_INTERNAL_DEVICE_MAX; i++) {
		if (!ethsw_drv[INTERNAL_SW]->swdev[i]) {
			ethsw_drv[INTERNAL_SW]->swdev[i] = swdev;
			ethsw_drv[INTERNAL_SW]->swdev_count++;
		}
	}

	/* save ethsw device for global reference */
	if (!ethsw_dev[INTERNAL_SW]) {
		ethsw_dev[INTERNAL_SW] = swdev;
	}

	/* get device params */
	ret = ethsw_swdev_param_init(swdev);
	if (ret) {
		LOG_CRIT("Ethsw device param init failed\n");
		goto ERROR;
	}

	LOG_DEBUG("Device name: %s\n", swdev->name);

	for (i = 0; i < swdev->pm_count; i++) {
		int port_id, mii_speed;
		bool rgmii_en;
		struct ethsw_port *port;

		port_id   = be32_to_cpu(swdev->pm_params[i * 3    ]);
		rgmii_en  = be32_to_cpu(swdev->pm_params[i * 3 + 1]);
		mii_speed = be32_to_cpu(swdev->pm_params[i * 3 + 2]);

		swdev->mii_port_mask |= (1 << port_id);

		port = &swdev->port[port_id];
		port->type = MII_PORT;
		/* get rgmii mode */
		port->mii.rgmii = rgmii_en;
		port->mii.speed = mii_speed;

		LOG_DEBUG("MII port params: <0x%x %x 0x%x>\n",
				port_id, rgmii_en, mii_speed);
	}

	for (i = 0; i < swdev->pi_count; i++) {
		int port_id, imp_speed;
		struct ethsw_port *port;

		port_id   = be32_to_cpu(swdev->pi_params[i * 2    ]);
		imp_speed = be32_to_cpu(swdev->pi_params[i * 2 + 1]);

		swdev->imp_port_mask |= (1 << port_id);
		swdev->imp_port_count++;
		port = &swdev->port[port_id];
		port->type = IMP_PORT;
		port->imp.speed = imp_speed;

		LOG_DEBUG("IMP port params: <0x%x 0x%x>\n",
				port_id, imp_speed);
	}

	LOG_DEBUG("LED map: 0x%x\n", swdev->led_map);

	/* get device resources */
	mres = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!mres) {
		LOG_CRIT("Ethsw device get resources failed\n");
		ret = -EINVAL;
		goto ERROR;
	}
	LOG_DEBUG("Device mem: 0x%x - 0x%x\n", (u32) mres->start, (u32) mres->end);

	/* get device IRQ */
	for (i = 0; i < ETHSW_IRQ_MAX; i++) {
		ret = platform_get_irq(pdev, i);
		if (ret < 0) {
			LOG_CRIT("Ethsw device get resources failed\n");
			ret = -EINVAL;
			goto ERROR;
		}
		swdev->irq[i] = ret;
		LOG_DEBUG("Device irq%d: %d\n", i, swdev->irq[i]);
	}

	/* request memory region */
	swdev->mem_region = request_mem_region(
			mres->start,
			mres->end - mres->start + 1,
			"ethsw");
	if (!swdev->mem_region) {
		LOG_CRIT("Ethsw device request mem region failed\n");
		ret = -ENOMEM;
		goto ERROR;
	}

	/* remap IO */
	swdev->reg_base = ioremap(
			mres->start,
			mres->end - mres->start + 1);
	if (!swdev->reg_base) {
		LOG_CRIT("Ethsw device IO remap failed\n");
		ret = -EIO;
		goto ERROR;
	}

	mres = platform_get_resource(pdev, IORESOURCE_MEM, 1);

	/* remap IO */
	if (mres) {
		swdev->periph_base = ioremap(
				mres->start,
				mres->end - mres->start + 1);
		if (!swdev->periph_base) {
			LOG_CRIT("Ethsw device IO remap failed\n");
			ret = -EIO;
			goto ERROR;
		}
	}

	/* remap IO for SUN_TOP */
	sun_top_ctrl = of_find_matching_node(NULL, sun_top_ctrl_match);
	if (sun_top_ctrl)
		swdev->sun_top_ctrl = of_iomap(sun_top_ctrl, 0);

	/* init ethsw device, platform dependent */
	ret = swdev->ethsw_swdev_board_init(swdev);
	if (ret) {
		LOG_CRIT("Ethsw device board init failed\n");
		goto ERROR;
	}

	/* get PHY port params */
	ethsw_swdev_phy_param_init(swdev);
	ethsw_phy_port_init_ids(swdev);


	ethsw_acb_enable(swdev);

	/* init switch core */
	ret = ethsw_core_init(swdev);
	if (ret) {
		LOG_CRIT("Ethsw core init failed\n");
		goto ERROR;
	}

	/* init switch PHY */
	ret = ethsw_phy_init(swdev);
	if (ret) {
		LOG_CRIT("Ethsw PHY init failed\n");
		goto ERROR;
	}

	/* init interrupt */
	ret = ethsw_int_init(swdev);
	if (ret) {
		LOG_CRIT("Ethsw interrupt init failed\n");
		goto ERROR;
	}

	/* reset config and disable all ports */
	for (i = 0; i < ETHSW_PORT_MAX; i++) {
		if (!((swdev->phy_port_mask | swdev->mii_port_mask) & (1 << i)))
			continue;

		ethsw_port_config_set(i, NULL);
		ethsw_port_power_config_set(i, NULL);
		ethsw_port_disable(i);
	}

#if PROC_IF_SUPPORT
	/* init proc file system */
	ret = ethsw_proc_init(swdev);
	if (ret) {
		LOG_CRIT("Ethsw proc init failed\n");
		goto ERROR;
	}
#endif

#if CDEV_IF_SUPPORT
	/* init cdev device */
	ret = ethsw_cdev_init(swdev);
	if (ret) {
		LOG_CRIT("Ethsw cdev init failed\n");
		goto ERROR;
	}
#endif

#if VNET_IF_SUPPORT
	/* init vnet device */
	ret = ethsw_vnet_init(swdev);
	if (ret) {
		LOG_CRIT("Ethsw vnet init failed\n");
		goto ERROR;
	}
#endif

	LOG_INFO("Ethsw driver probed\n");

	FUNC_LEAVE();
	return(ret);

 ERROR:
	/* remove module properly */
	ethsw_pdev_remove(pdev);

	FUNC_LEAVE();
	return(ret);
}

static int ethsw_pdev_remove(struct platform_device *pdev)
{
	int ret = 0;
	int i;
	struct ethsw_device *swdev;

	FUNC_ENTER();

	swdev = (struct ethsw_device *)pdev->dev.platform_data;
	if (!swdev) {
		goto EXIT;
	}

#if VNET_IF_SUPPORT
	/* exit vnet device */
	ethsw_vnet_exit(swdev);
#endif

#if CDEV_IF_SUPPORT
	/* exit cdev device */
	ethsw_cdev_exit(swdev);
#endif

#if PROC_IF_SUPPORT
	/* exit proc file system */
	ethsw_proc_exit(swdev);
#endif

	/* exit interrupt properly */
	ethsw_int_exit(swdev);

	/* exit switch PHY */
	ethsw_phy_exit(swdev);

	/* exit switch core */
	ethsw_core_exit(swdev);

	/* exit ethsw device, platformat dependent */
	if (swdev->ethsw_swdev_board_exit) {
		swdev->ethsw_swdev_board_exit(swdev);
	}

	/* undo clock */
	clk_disable_unprepare(swdev->clk);
	clk_put(swdev->clk);

	/* unmap IO */
	if (swdev->reg_base)
		iounmap(swdev->reg_base);

	/* release memory region */
	if (swdev->mem_region) {
		struct resource *pregion = swdev->mem_region;
		release_mem_region(
				pregion->start,
				pregion->end - pregion->start + 1);
	}

	/* update ethsw driver */
	for (i = 0 ; i < ETHSW_INTERNAL_DEVICE_MAX; i++) {
		if (ethsw_drv[INTERNAL_SW]->swdev[i] == swdev) {
			ethsw_drv[INTERNAL_SW]->swdev[i] = NULL;
			ethsw_drv[INTERNAL_SW]->swdev_count--;
		}
	}

	/* clear ethsw device for global reference */
	if (ethsw_dev[INTERNAL_SW] == swdev) {
		ethsw_dev[INTERNAL_SW] = NULL;
	}

	/* free ethsw device control block */
	kfree(swdev);
	pdev->dev.platform_data = NULL;

	LOG_INFO("Ethsw driver removed\n");

 EXIT:
	FUNC_LEAVE();
	return(ret);
}


/*
 * Ether Switch Module Functions
 */


static struct of_device_id ethsw_of_match[] = {
	{.compatible = "brcm,ethsw"},
	{}
};

MODULE_DEVICE_TABLE(of, ethsw_of_match);

static struct platform_driver ethsw_pdrv = {
	.probe  = ethsw_pdev_probe,
	.remove = ethsw_pdev_remove,
	.driver = {
		.name  = MODULE_NAME,
		.owner = THIS_MODULE,
		.of_match_table = ethsw_of_match
	},
};

extern struct spi_driver ethsw_sdrv;

static int __init ethsw_module_init(void)
{
	int ret = 0;

	FUNC_ENTER();

	pr_debug("%s driver v%s", MODULE_NAME, MODULE_VER);

	/* alloc ethsw driver control block */
	ethsw_drv[INTERNAL_SW] = kzalloc(sizeof(struct ethsw_driver), GFP_KERNEL);
	if (!ethsw_drv[INTERNAL_SW]) {
		LOG_CRIT("Ethsw driver control block alloc failed\n");
		ret = -ENOMEM;
		goto EXIT;
	}

	ethsw_drv[INTERNAL_SW]->pdrv = &ethsw_pdrv;

	/* register platform driver */
	ret = platform_driver_register(&ethsw_pdrv);
	if (ret) {
		LOG_CRIT("Platform driver register failed\n");
		goto EXIT;
	}

	ret = spi_register_driver(&ethsw_sdrv);
	LOG_INFO("Ethsw module initialized\n");
	if (ret) {
		LOG_CRIT("SPI driver register failed\n");
		goto EXIT;
	}

 EXIT:
	FUNC_LEAVE();
	return(0);
}

static void __exit ethsw_module_exit(void)
{
	FUNC_ENTER();


	/* unregister platform driver */
	platform_driver_unregister(&ethsw_pdrv);

	/* unregister spi driver */
	spi_unregister_driver(&ethsw_sdrv);

	/* free ethsw driver control block */
	kfree(ethsw_drv[INTERNAL_SW]);

	LOG_INFO("Ethsw module exited\n");

	FUNC_LEAVE();
}

module_init(ethsw_module_init);
module_exit(ethsw_module_exit);
MODULE_LICENSE("GPL");
