 /****************************************************************************
 *
 * Copyright (c) 2015-2018 Broadcom. All rights reserved
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************/
#include <linux/kernel.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>

#include "ethsw.h"
#include "ethsw_priv.h"

/*
 * Log Utilities
 */

/* define log module */
#define LOG_MODULE "ethsw_proc"

#define MAX_COPY_COUNT 63

#if PROC_IF_SUPPORT

enum {
	CTL_CMD_NULL        = 0,
	CTL_CMD_STATE       = 1,
	CTL_CMD_MANAGED     = 3,
	CTL_CMD_CERT        = 4,
	CTL_CMD_ETHWAN      = 5,
	CTL_CMD_ALTWAN      = 6
};

enum {
	CREG_CMD_NULL       = 0,
	CREG_CMD_READ       = 1,
	CREG_CMD_WRITE      = 2
};

enum {
	ARL_CMD_NULL        = 0,
	ARL_CMD_FIND        = 1,
	ARL_CMD_ADD         = 2,
	ARL_CMD_DELETE      = 3,
	ARL_CMD_MP_FIND     = 4,
	ARL_CMD_MP_ADD      = 5,
	ARL_CMD_MP_DELETE   = 6,
	ARL_CMD_ENA_LEARN   = 7,
	ARL_CMD_DIS_LEARN   = 8,
	ARL_CMD_AGE_POLL    = 9,
	ARL_CMD_AGE_PERIOD  = 10,
	ARL_CMD_LINK_POLL   = 11,
	ARL_CMD_CACHE_MISS  = 12,
	ARL_CMD_DUMP        = 13,
	ARL_CMD_ENA_SWLEARN = 14,
	ARL_CMD_DIS_SWLEARN = 15,
	ARL_CMD_ENA_HWLEARN = 16,
	ARL_CMD_DIS_HWLEARN = 17,
};

enum {
	VLAN_CMD_NULL       = 0,
	VLAN_CMD_ADD        = 1,
	VLAN_CMD_DELETE     = 2,
	VLAN_CMD_FIND       = 3,
	VLAN_CMD_PTAG_SET   = 4,
	VLAN_CMD_PTAG_GET   = 5,
	VLAN_CMD_PVID_ADD   = 6,
	VLAN_CMD_PVID_DEL   = 7,
	VLAN_CMD_PVID_DEF   = 8,
	VLAN_CMD_ENABLE     = 9,
	VLAN_CMD_IPv6_MC_NF = 10
};

enum {
	HASHCLASH_CMD_NULL       = 0,
	HASHCLASH_CMD_ENABLE     = 1
};

enum {
	RESERVED_MULTICAST_CMD_NULL       = 0,
	RESERVED_MULTICAST_CMD_ENABLE     = 1
};

enum {
	PORT_CMD_NULL       = 0,
	PORT_CMD_ENABLE     = 1,
	PORT_CMD_DISABLE    = 2,
	PORT_CMD_CONFIG_SET = 3,
	PORT_CMD_CONFIG_GET = 4,
	PORT_CMD_STATUS_GET = 5,
	PORT_CMD_PWR_CONFIG_SET = 6,
	PORT_CMD_PWR_CONFIG_GET = 7,
	PORT_CMD_PWR_STATUS_GET = 8,
	PORT_CMD_INFO_GET = 9
};

enum {
	MIB_CMD_NULL        = 0,
	MIB_CMD_RESET       = 1,
	MIB_CMD_GET         = 2,
	MIB_CMD_TX_ALL      = 3,
	MIB_CMD_RX_ALL      = 4,
	MIB_CMD_EEE_ALL     = 5,
	MIB_CMD_ALL         = 6
};

#ifdef ETHSW_FLOW_CONTROL_ENABLED
enum {
	FLOWCTL_CMD_NULL          = 0,
	FLOWCTL_CMD_MAP_SET       = 1,
	FLOWCTL_CMD_MAP_GET       = 2,
	FLOWCTL_CMD_THRESHOLD_SET = 3,
	FLOWCTL_CMD_THRESHOLD_GET = 4
};
#endif

enum {
	ETHSW_PWR_CMD_NULL       = 0,
	ETHSW_PWR_CMD_ENABLE     = 1,
	ETHSW_PWR_CMD_DISABLE    = 2,
	ETHSW_PWR_CMD_INFO_GET   = 3
};

/* ethsw proc control block */
struct ethsw_proc {
	/* proc dir name of the ethsw device */
	char dir_name[32];

	/* ctl cmd and results */
	int ctl_cmd;
	bool ctl_managed;
	bool ctl_cert;
	bool ctl_ethwan;
	int  ctl_ethwan_port;
	bool ctl_altwan_enable;
	int  ctl_altwan_impport;
	int  ctl_altwan_lanport;

	/* creg cmd and results */
	int creg_cmd;
	int creg_page;
	int creg_addr;
	int creg_size;
	u64 creg_data;

	/* ARL cmd and results */
	int arl_cmd;
	u8  arl_mac[6];
	u16 arl_vid;
	u16 arl_port;
	u8  arl_mp_mac[6];
	u16 arl_mp_etype;
	u16 arl_mp_port;
	u16 arl_mac_aging_period;
	u16 arl_mac_aging_poll_period;
	u16 arl_link_status_poll_period;

	/* VLAN cmd and results */
	int vlan_cmd;
	u16 vlan_vid;
	u16 vlan_untag_map;
	u16 vlan_fwd_map;
	int vlan_port_id;
	u16 vlan_port_vid;
	u8  vlan_port_pcp;
	bool vlan_enable;
	bool vlan_ipv6_mc_nf_mode_enabled;

	/* hash clash cmd */
	int hashclash_cmd;
	bool hashclash_enable;

	/* reserved multicast cmd */
	int reserved_multicast_cmd;
	bool reserved_multicast_enable;

	/* port cmd and results */
	int port_cmd;
	int port_id;
	struct ethsw_port_config port_config;
	struct ethsw_port_status port_status;
	struct ethsw_power_config power_config;
	struct ethsw_power_status power_status;

	/* MIB cmd and results */
	int mib_cmd;
	u32 mib_port;
	u32 mib_counter;
	u64 mib_value;
	u32 mib_values[MIB_COUNTER_MAX];

#ifdef ETHSW_FLOW_CONTROL_ENABLED
	/* Flow control */
	u32 flowctl_cmd;
	u32 flowctl_port;
	u32 flowctl_threshold;
	u32 flowctl_tc;
	u32 flowctl_qid;
#define ETHSW_TC_MAX 8
	u32 tc2qid_map[ETHSW_TC_MAX];
	u32 port_qid_thresholds[ETHSW_ACB_QUEUES_PER_PORT];
#endif
    /* pwr cmd and results */
	int ethsw_pwr_cmd;
};

/* MIB counter names */
static char *mib_name_tbl[] =
{
	/* transmit counters */
	"TxOctets",
	"",
	"TxDropPkts",
	"TxBroadcastPkts",
	"TxMulticastPkts",
	"TxUnicastPkts",
	"TxPausePkts",
	"TxCollisions",
	"TxSingleCollision",
	"TxMultipleCollision",
	"TxLateCollision",
	"TxExcessiveCollision",
	"TxDeferredTransmit",
	"TxFrameInDisc",

	"TxQ0Pkt",
	"TxQ1Pkt",
	"TxQ2Pkt",
	"TxQ3Pkt",
	"TxQ4Pkt",
	"TxQ5Pkt",

#if ETHSW_SF2_SUPPORT
	"TxQ6Pkt",
	"TxQ7Pkt",
	"TxPkts64Octets",
	"TxPkts65to127Octets",
	"TxPkts128to255Octets",
	"TxPkts256to511Octets",
	"TxPkts512to1023Octets",
	"TxPkts1024toMaxOctets",
#endif

	/* receive counters */
	"RxOctets",
	"",
	"RxGoodOctets",
	"",
	"RxDropPkts",
	"RxBroadcastPkts",
	"RxMulticastPkts",
	"RxUnicastPkts",
	"RxPausePkts",
	"RxSAChanges",

	"RxUndersizePkts",
	"RxPkts64Octets",
	"RxPkts65to127Octets",
	"RxPkts128to255Octets",
	"RxPkts256to511Octets",
	"RxPkts512to1023Octets",
	"RxPkts1024toMaxOctets",
	"RxOversizePkts",
	"RxJumboPkts",

	"RxFragments",
	"RxJabbers",
	"RxDiscard",
	"RxAlignmentErrors",
	"RxFCSErrors",
	"RxSymbolErrors",
	"RxInRangeErrors",
	"RxOutOfRangeErrors",

	/* EEE counters */
	"EEELpiEvent",
	"EEELpiDuration"
};

static ssize_t ethsw_proc_ctl_write(
	struct file *file, const char __user *buffer, size_t count, loff_t *data)
{
	int len = 0;
	char cmd_buffer[64];
	char cmd_char;
	int ctl_managed;
	int ctl_cert;
	int ctl_ethwan, ctl_ethwan_port;
	int ctl_altwan_enable, ctl_altwan_impport, ctl_altwan_lanport;
	int scan_items;
	struct ethsw_device *swdev;
	struct ethsw_proc *swproc;

	swdev = (struct ethsw_device *)file->private_data;
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	swproc = swdev->swproc;
	memset(cmd_buffer, 0, sizeof(cmd_buffer));
	/* copy from user space */
	len = min((int)count, (int)MAX_COPY_COUNT);
	if (copy_from_user(cmd_buffer, buffer, len)) {
		return(-EFAULT);
	}

	/* force NULL termination */
	cmd_buffer[len] = '\0';

	/* clear previous cmd */
	swproc->ctl_cmd = CTL_CMD_NULL;

	/* parsing cmd */
	/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
	scan_items = sscanf(cmd_buffer, "%1c", &cmd_char);
	if (scan_items != 1) {
		return(len);
	}

	switch (cmd_char) {
	case 's':
	case 'S':
		/* get states */
		ethsw_switch_mode_get(&swproc->ctl_managed);
		ethsw_switch_cert_get(&swproc->ctl_cert);
		if ((swproc->ctl_ethwan_port >=0) && (swproc->ctl_ethwan_port < ETHSW_PORT_MAX))
			ethsw_switch_ethwan_get(&swproc->ctl_ethwan,
									&swproc->ctl_ethwan_port);
		ethsw_switch_alt_ethwan_get(swdev,
                                    &swproc->ctl_altwan_impport,
                                    &swproc->ctl_altwan_lanport,
                                    &swproc->ctl_altwan_enable);
		swproc->ctl_cmd = CTL_CMD_STATE;
		break;

	case 'm':
	case 'M':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + 2, "%8x", &ctl_managed);
		if (scan_items != 1) {
			return(-EINVAL);
		}

		/* execute set managed mode */
		if (ethsw_switch_mode_set(ctl_managed)) {
			return(-EIO);
		}

		swproc->ctl_cmd = CTL_CMD_MANAGED;
		swproc->ctl_managed = ctl_managed;
		break;

	case 'c':
	case 'C':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + 2, "%8x", &ctl_cert);
		if (scan_items != 1) {
			return(-EINVAL);
		}

		/* execute set cert */
		if (ethsw_switch_cert_set(ctl_cert)) {
			return(-EIO);
		}

		swproc->ctl_cmd = CTL_CMD_CERT;
		swproc->ctl_cert = ctl_cert;
		break;

	case 'w':
	case 'W':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + 2, "%1x %8x",
							&ctl_ethwan,
							&ctl_ethwan_port);
		if (scan_items != 2) {
			return(-EINVAL);
		}

		/* execute set managed mode */
		if (ethsw_switch_ethwan_set(ctl_ethwan, ctl_ethwan_port)) {
			return(-EIO);
		}

		swproc->ctl_cmd = CTL_CMD_ETHWAN;
		swproc->ctl_ethwan      = (bool)ctl_ethwan;
		swproc->ctl_ethwan_port = ctl_ethwan_port;
		break;


	case 'a':
	case 'A':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + 2, "%8x %8x %8x",
							&ctl_altwan_enable,
							&ctl_altwan_impport,
							&ctl_altwan_lanport);
		if (scan_items != 3) {
			return(-EINVAL);
		}

		/* execute set managed mode */
		if (ethsw_alt_ethwan_core(swdev, ctl_altwan_impport, ctl_altwan_lanport,
								  ctl_altwan_enable)) {
			return(-EIO);
		}

		swproc->ctl_cmd = CTL_CMD_ALTWAN;
		swproc->ctl_altwan_enable  = ctl_altwan_enable;
		swproc->ctl_altwan_impport = ctl_altwan_impport;
		swproc->ctl_altwan_lanport = ctl_altwan_lanport;
		break;

	case ' ':
	case '\n':
	case '\t':
		return(len);

	default:
		return(-EINVAL);
	}

	return(len);
}

static ssize_t ethsw_proc_ctl_read(
	struct file *file, char __user *buffer, size_t count, loff_t *data)
{
	int len = 0;
	struct ethsw_device *swdev;
	struct ethsw_proc *swproc;
	unsigned long page;

	if (*data)
		return 0;

	swdev = (struct ethsw_device *)file->private_data;
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	swproc = swdev->swproc;

	page = get_zeroed_page(GFP_KERNEL);
	if (!page)
		return -ENOMEM;

	switch (swproc->ctl_cmd) {
	case CTL_CMD_STATE:
		len += snprintf(
			(char *)page + len, count-len,
			"Control get:\n"
			"\n"
			"    managed=%x\n"
			"    cert=%x\n"
			"    ethwan=%x\n"
			"    ethwan_port=%x\n"
			"    altwan_enble=%x\n"
			"    altwan_impport=%x\n"
			"    altwan_lanport=%x\n"
			"\n",
			swproc->ctl_managed,
			swproc->ctl_cert,
			swproc->ctl_ethwan,
			swproc->ctl_ethwan_port,
			swproc->ctl_altwan_enable,
			swproc->ctl_altwan_impport,
			swproc->ctl_altwan_lanport);
		break;

	case CTL_CMD_MANAGED:
		len += snprintf((char *)page + len, count-len, "Control set: managed=%x\n",
				swproc->ctl_managed);
		break;

	case CTL_CMD_CERT:
		len += snprintf((char *)page + len, count-len, "Control set: cert=%x\n",
			       swproc->ctl_cert);
		break;

	case CTL_CMD_ETHWAN:
		len += snprintf((char *)page + len, count-len, "Control set: ethwan=%x, ethwan_port=%x\n",
			       swproc->ctl_ethwan, swproc->ctl_ethwan_port);
		break;

	case CTL_CMD_ALTWAN:
		len += snprintf((char *)page + len, count-len, "Control set: altwan_enble=%x, altwan_impport=%x altwan_lanport=%x\n",
			       swproc->ctl_altwan_enable,
			       swproc->ctl_altwan_impport,
			       swproc->ctl_altwan_lanport);
		break;


	default:
		len += snprintf(
			(char *)page + len, count-len,
			"Ctl cmd is NULL.\n"
			"\n"
			"To issue a ctl cmd, write to /proc/%s/ctl\n"
			"\n"
			"    s                             <- get ctl state\n"
            "    m 0/1                         <- set managed mode\n"
            "    c 0/1                         <- set certification mode\n"
            "    w 0/1 port_id                 <- set ethwan mode and port\n"
            "    a 0/1 impport_id lanport_id   <- set altwan mode and impport and lanport\n"
			"\n"
            "where all values are in hex format.\n"
			"\n",
			swproc->dir_name);
		break;
	}

	if (len >= 0) {
		*data += len;
		if (copy_to_user(buffer, (char *)page, len))
			len = -EFAULT;
	}
	free_page(page);

	return(len);
}

extern struct ethsw_device *ethsw_dev[];

static ssize_t ethsw_proc_creg_write(
	struct file *file, const char __user *buffer, size_t count, loff_t *data)
{
	int len = 0;
	char cmd_buffer[64];
	char cmd_char;
	int creg_page, creg_addr, creg_size;
	u64 creg_data;
	int scan_items;
	struct ethsw_device *swdev;
	struct ethsw_proc *swproc;

	swdev = (struct ethsw_device *)file->private_data;
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	swproc = swdev->swproc;
	memset(cmd_buffer, 0, sizeof(cmd_buffer));
	/* copy from user space */
	len = min((int)count, (int)MAX_COPY_COUNT);
	if (copy_from_user(cmd_buffer, buffer, len)) {
		return(-EFAULT);
	}

	/* force NULL termination */
	cmd_buffer[len] = '\0';

	/* clear previous cmd */
	swproc->creg_cmd = CREG_CMD_NULL;

	/* parsing cmd */
	/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
	scan_items = sscanf(cmd_buffer, "%1c", &cmd_char);
	if (scan_items != 1) {
		return(len);
	}

	switch (cmd_char) {
	case 'r':
	case 'R':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + 2, "%8x:%8x %10d",
							&creg_page, &creg_addr, &creg_size);

		if (scan_items != 3) {
			return(-EINVAL);
		}

		if (creg_page > 0xFF ||
			creg_addr > 0xFF ||
			creg_size > 8) {
			return(-EINVAL);
		}

		/* execute creg read */
		if (ethsw_creg_read(
				swdev,
				creg_page,
				creg_addr,
				(u8 *)&creg_data,
				creg_size)) {
			return(-EIO);
		}

		swproc->creg_cmd = CREG_CMD_READ;
		break;

	case 'w':
	case 'W':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + 2, "%8x:%8x %10d %llx",
							&creg_page, &creg_addr, &creg_size, &creg_data);

		if (scan_items != 4) {
			return(-EINVAL);
		}

		if (creg_page > 0xFF ||
			creg_addr > 0xFF ||
			creg_size > 8) {
			return(-EINVAL);
		}

#ifdef __BIG_ENDIAN
		/* move data to start of the 8-byte array */
		switch (creg_size) {
		case 1:
			*(u8 *)&creg_data = *((u8 *)&creg_data + 7);
			break;
		case 2:
			*(u16 *)&creg_data = *((u16 *)&creg_data + 3);
			break;
		case 4:
			*(u32 *)&creg_data = *((u32 *)&creg_data + 1);
			break;
		case 8:
			break;
		default:
			return(-EINVAL);
		}
#endif

		/* execute creg write */
		if (ethsw_creg_write(
				swdev,
				creg_page,
				creg_addr,
				(u8 *)&creg_data,
				creg_size)) {
			return(-EIO);
		}

		swproc->creg_cmd = CREG_CMD_WRITE;
		break;

	case ' ':
	case '\n':
	case '\t':
		return(len);

	default:
		return(-EINVAL);
	}

	/* record results */
	swproc->creg_page = creg_page;
	swproc->creg_addr = creg_addr;
	swproc->creg_size = creg_size;
	swproc->creg_data = creg_data;

	return(len);
}

static ssize_t ethsw_proc_creg_read(
	struct file *file, char __user *buffer, size_t count, loff_t *data)
{
	int len = 0;
	struct ethsw_device *swdev;
	struct ethsw_proc *swproc;
	unsigned long page;

	if (*data)
		return 0;

	swdev = (struct ethsw_device *)file->private_data;
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	swproc = swdev->swproc;

	page = get_zeroed_page(GFP_KERNEL);
	if (!page)
		return -ENOMEM;

	switch (swproc->creg_cmd) {
	case  CREG_CMD_READ:
		len += snprintf((char *)page + len, count-len, "Core reg read: ");
		break;

	case CREG_CMD_WRITE:
		len += snprintf((char *)page + len, count-len, "Core reg write: ");
		break;

	default:
		len += snprintf(
			(char *)page + len, count-len,
			"Core reg cmd is NULL.\n"
			"\n"
			"To issue a core reg cmd, write to /proc/%s/creg\n"
			"\n"
            "    r PP:AA S              <- read  core reg at page:addr of the given size\n"
            "    w PP:AA S value        <- write core reg at page:addr of the given size\n"
			"\n"
            "where all values are in hex format.\n"
			"\n",
			swproc->dir_name);
		goto EXIT;
	}

	len += snprintf((char *)page + len, count-len, "%02X:%02X=",
			swproc->creg_page, swproc->creg_addr);

	switch (swproc->creg_size) {
	case 1:
		len += snprintf((char *)page + len, count-len, "%02X\n",
				*(u8 *)&swproc->creg_data);
		break;
	case 2:
		len += snprintf((char *)page + len, count-len, "%04X\n",
			       *(u16 *)&swproc->creg_data);
		break;
	case 4:
		len += snprintf((char *)page + len, count-len, "%08X\n",
			       *(u32 *)&swproc->creg_data);
		break;
	case 8:
		len += snprintf((char *)page + len, count-len, "%016llX\n",
			       *(u64 *)&swproc->creg_data);
		break;
	}

 EXIT:
	if (len >= 0) {
		*data += len;
		if (copy_to_user(buffer, (char *)page, len))
			len = -EFAULT;
	}
	free_page(page);

	return(len);
}

static int ethsw_scan_mac(char *buffer, u8 *mac)
{
	int offset = 0;
	int i;
	u32 word;

	for (i = 0; i < 6; i++) {
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		if (sscanf(buffer + offset, "%8x", &word) != 1)
			return(0);
		offset += 3;
		*mac++ = (u8)word;
	}
	return(1);
}

static int ethsw_print_mac(char *buffer, int buffer_len, u8 *mac)
{
	return snprintf(buffer, buffer_len, "%pM", mac);
}

static ssize_t ethsw_proc_arl_write(
	struct file *file, const char __user *buffer, size_t count, loff_t *data)
{
	int len = 0;
	char cmd_buffer[64];
	int cmd_offset;
	char cmd_char;
	u32 cmd_word;
	u8  arl_mac[6];
	u16 arl_vid;
	u16 arl_port;
	u8  arl_mp_mac[6];
	u16 arl_mp_etype;
	u16 arl_mp_port;
	int scan_items;
	u16 period;
	struct ethsw_device *swdev;
	struct ethsw_proc *swproc;

	swdev = (struct ethsw_device *)file->private_data;
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	swproc = swdev->swproc;
	memset(cmd_buffer, 0, sizeof(cmd_buffer));
	memset(arl_mp_mac, 0, sizeof(arl_mp_mac));
	/* copy from user space */
	len = min((int)count, (int)MAX_COPY_COUNT);
	if (copy_from_user(cmd_buffer, buffer, len)) {
		return(-EFAULT);
	}

	/* force NULL termination */
	cmd_buffer[len] = '\0';

	/* clear previous cmd */
	swproc->arl_cmd = ARL_CMD_NULL;

	/* parse cmd */
	/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
	scan_items = sscanf(cmd_buffer, "%1c", &cmd_char);
	if (scan_items != 1) {
		return(len);
	}

	cmd_offset = 2;
	arl_vid = 0;
	arl_mp_etype = 0;
	period = 0;

	switch (cmd_char) {
	case 'f':
	case 'F':
	case 'a':
	case 'A':
	case 'd':
	case 'D':
		/* parse argument, step 1 VID/MAC */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + cmd_offset, "%8x", &cmd_word);
		if (scan_items != 1) {
			return(-EINVAL);
		}

		cmd_offset += 4;
		arl_vid = (u16)cmd_word;

		scan_items = ethsw_scan_mac(cmd_buffer + cmd_offset, arl_mac);
		if (scan_items != 1) {
			return(-EINVAL);
		}

		cmd_offset += 18;
		break;

	case 'l':
	case 'L':
	case 'm':
	case 'n':
	case 'N':
		/* parse argument, step 1 etype/MAC */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + cmd_offset, "%8x", &cmd_word);
		if (scan_items != 1) {
			return(-EINVAL);
		}

		cmd_offset += 5;
		arl_mp_etype = (u16)cmd_word;

		scan_items = ethsw_scan_mac(cmd_buffer + cmd_offset, arl_mp_mac);
		if (scan_items != 1) {
			return(-EINVAL);
		}

		cmd_offset += 18;
		break;

	case 'g':
	case 'G':
	case 'p':
	case 'P':
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + cmd_offset, "%10d", &cmd_word);
		if (scan_items != 1) {
			return(-EINVAL);
		}
		cmd_offset += 5;
		period = (u16)cmd_word;

	default:
		break;
	}

	switch (cmd_char) {
	case 'f':
	case 'F':
		if (cmd_char == 'f') {
			/* ARL table find */
			if (ethsw_arl_switch_table_find(swdev, arl_mac, arl_vid, &arl_port)) {
				return(-EIO);
			}
		} else {
			if (__ethsw_arl_cache_table_find(arl_mac, arl_vid, &arl_port)) {
				return(-EIO);
			}
		}

		swproc->arl_cmd = ARL_CMD_FIND;
		memcpy(swproc->arl_mac, arl_mac, 6);
		swproc->arl_vid  = arl_vid;
		swproc->arl_port = arl_port;
		break;

	case 'a':
	case 'A':
		/* parse argument, step 2 port */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + cmd_offset, "%8x", &cmd_word);
		if (scan_items != 1) {
			return(-EINVAL);
		}

		cmd_offset += 3;
		arl_port = (u16)cmd_word;
		if (cmd_char == 'a') {
			/* ARL table add */
			if (ethsw_arl_switch_table_add(swdev, arl_mac, arl_vid, arl_port)) {
				return(-EIO);
			}
		} else {
			if (ethsw_arl_cache_table_add(arl_mac, arl_vid, arl_port)) {
				return(-EIO);
			}
		}
		swproc->arl_cmd = ARL_CMD_ADD;
		memcpy(swproc->arl_mac, arl_mac, 6);
		swproc->arl_vid  = arl_vid;
		swproc->arl_port = arl_port;

		break;

	case 'd':
	case 'D':
		if (cmd_char == 'd') {
			/* ARL table delete */
			if (ethsw_arl_switch_table_delete(swdev, arl_mac, arl_vid)) {
				return(-EIO);
			}
		} else {
			if (ethsw_arl_cache_table_delete(arl_mac, arl_vid)) {
				return(-EIO);
			}
		}
		swproc->arl_cmd = ARL_CMD_DELETE;
		memcpy(swproc->arl_mac, arl_mac, 6);
		swproc->arl_vid  = arl_vid;
		break;

	case 'l':
	case 'L':
		/* multiport find */
		if (ethsw_multiport_find(arl_mp_mac, arl_mp_etype, &arl_mp_port)) {
			return(-EIO);
		}

		swproc->arl_cmd = ARL_CMD_MP_FIND;
		memcpy(swproc->arl_mp_mac, arl_mp_mac, 6);
		swproc->arl_mp_etype = arl_mp_etype;
		swproc->arl_mp_port  = arl_mp_port;
		break;

	case 'm':
		/* parse argument, step 2 port */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + cmd_offset, "%8x", &cmd_word);
		if (scan_items != 1) {
			return(-EINVAL);
		}

		cmd_offset += 3;
		arl_mp_port = (u16)cmd_word;

		/* multiport add */
		if (ethsw_multiport_add(arl_mp_mac, arl_mp_etype, arl_mp_port)) {
			return(-EIO);
		}

		swproc->arl_cmd = ARL_CMD_MP_ADD;
		memcpy(swproc->arl_mp_mac, arl_mp_mac, 6);
		swproc->arl_mp_etype = arl_mp_etype;
		swproc->arl_mp_port  = arl_mp_port;
		break;

	case 'n':
	case 'N':
		/* multiport delete */
		if (ethsw_multiport_delete(arl_mp_mac, arl_mp_etype)) {
			return(-EIO);
		}

		swproc->arl_cmd = ARL_CMD_MP_DELETE;
		memcpy(swproc->arl_mp_mac, arl_mp_mac, 6);
		swproc->arl_mp_etype = arl_mp_etype;
		break;

	case 'e':
	case 'E':
		/* Enable ARL learning */
		if (ethsw_arl_learning_enable(1)) {
			return(-EIO);
		}
		swproc->arl_cmd = ARL_CMD_ENA_LEARN;
		break;

	case 'x':
	case 'X':
		/* Disable ARL learning */
		if (ethsw_arl_learning_enable(0)) {
			return(-EIO);
		}
		swproc->arl_cmd = ARL_CMD_DIS_LEARN;
		break;

	case 's':
		/* Enable software ARL learning */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + cmd_offset, "%5hu", &arl_port);
		if (scan_items != 1) {
			return(-EINVAL);
		}

		if (ethsw_arl_swlearn_enable(arl_port, 1)) {
			return(-EIO);
		}
		swproc->arl_cmd = ARL_CMD_ENA_SWLEARN;
		swproc->arl_port = arl_port;
		break;

	case 'S':
		/* Disable software ARL learning */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + cmd_offset, "%5hu", &arl_port);
		if (scan_items != 1) {
			return(-EINVAL);
		}

		if (ethsw_arl_swlearn_enable(arl_port, 0)) {
			return(-EIO);
		}
		swproc->arl_cmd = ARL_CMD_DIS_SWLEARN;
		swproc->arl_port = arl_port;
		break;

	case 'h':
		/* Enable hardware ARL learning */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + cmd_offset, "%5hu", &arl_port);
		if (scan_items != 1) {
			return(-EINVAL);
		}

		if (ethsw_arl_hwlearn_enable(arl_port, 1)) {
			return(-EIO);
		}
		swproc->arl_cmd = ARL_CMD_ENA_HWLEARN;
		swproc->arl_port = arl_port;
		break;

	case 'H':
		/* Disable hardware ARL learning */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + cmd_offset, "%5hu", &arl_port);
		if (scan_items != 1) {
			return(-EINVAL);
		}

		if (ethsw_arl_hwlearn_enable(arl_port, 0)) {
			return(-EIO);
		}
		swproc->arl_cmd = ARL_CMD_DIS_HWLEARN;
		swproc->arl_port = arl_port;
		break;

	case 'g':
	case 'G':
		/* set mac aging period */
		ethsw_arl_cache_table_set_aging_period(swdev, period);
		swproc->arl_cmd = ARL_CMD_AGE_PERIOD;
		swproc->arl_mac_aging_period = period;
		break;

	case 'p':
		/* set how often we poll for aged macs */
		ethsw_arl_cache_table_set_aging_poll(swdev, period);
		swproc->arl_cmd = ARL_CMD_AGE_POLL;
		swproc->arl_mac_aging_poll_period = period;
		break;

	case 'P':
		/* set how often we poll for link status changes */
		ethsw_set_link_poll(swdev, period);
		swproc->arl_cmd = ARL_CMD_LINK_POLL;
		swproc->arl_link_status_poll_period = period;
		break;
	case 'M':
		swproc->arl_cmd = ARL_CMD_CACHE_MISS;
		break;
	case ' ':
	case '\n':
	case '\t':
		return(len);

	default:
		return(-EINVAL);
	}

	return(len);
}

static ssize_t ethsw_proc_arl_read(
	struct file *file, char __user *buffer, size_t count, loff_t *data)
{
	int len = 0;
	struct ethsw_device *swdev;
	struct ethsw_proc *swproc;
	unsigned long page;
	unsigned int ethsw_arl_cache_miss = 0;
	unsigned int ethsw_arl_cache_faked = 0;

	if (*data)
		return 0;

	swdev = (struct ethsw_device *)file->private_data;
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);
	ethsw_arl_cache_get_counter(swdev, &ethsw_arl_cache_miss, &ethsw_arl_cache_faked);

	swproc = swdev->swproc;

	page = get_zeroed_page(GFP_KERNEL);
	if (!page)
		return -ENOMEM;

	switch (swproc->arl_cmd) {
	case ARL_CMD_FIND:
		len += snprintf((char *)page + len,  count-len, "ARL table find: vid=%X, mac=",
				swproc->arl_vid);
		len += ethsw_print_mac((char *)page + len,  count-len, swproc->arl_mac);
		len += snprintf((char *)page + len, count-len, ", port=%X\n", swproc->arl_port);
		break;

	case ARL_CMD_ADD:
		len += snprintf((char *)page + len, count-len, "ARL table add: vid=%X, mac=",
				swproc->arl_vid);
		len += ethsw_print_mac((char *)page + len, count-len, swproc->arl_mac);
		len += snprintf((char *)page + len,  count-len, ", port=%X\n", swproc->arl_port);
		break;

	case ARL_CMD_DELETE:
		len += snprintf((char *)page + len, count-len, "ARL table delete: vid=%X, mac=",
				swproc->arl_vid);
		len += ethsw_print_mac((char *)page + len, count-len, swproc->arl_mac);
		len += snprintf((char *)page + len, count-len, "\n");
		break;

	case ARL_CMD_MP_FIND:
		len += snprintf((char *)page + len, count-len, "Multiport find: etype=%X, mac=",
				swproc->arl_mp_etype);
		len += ethsw_print_mac((char *)page + len, count-len, swproc->arl_mp_mac);
		len += snprintf((char *)page + len, count-len, ", port=%X\n",
				swproc->arl_mp_port);
		break;

	case ARL_CMD_MP_ADD:
		len += snprintf((char *)page + len, count-len, "Multiport add: etype=%X, mac=",
				swproc->arl_mp_etype);
		len += ethsw_print_mac((char *)page + len, count-len, swproc->arl_mp_mac);
		len += snprintf((char *)page + len, count-len, ", port=%X\n",
				swproc->arl_mp_port);
		break;

	case ARL_CMD_MP_DELETE:
		len += snprintf((char *)page + len, count-len, "Multiport delete: etype=%X, mac=",
				swproc->arl_mp_etype);
		len += ethsw_print_mac((char *)page + len, count-len, swproc->arl_mp_mac);
		len += snprintf((char *)page + len, count-len, "\n");
		break;

	case ARL_CMD_ENA_LEARN:
		len += snprintf((char *)page + len, count-len, "ARL Learning Enabled\n");
		break;

	case ARL_CMD_DIS_LEARN:
		len += snprintf((char *)page + len, count-len, "ARL Learning Disabled\n");
		break;

	case ARL_CMD_ENA_SWLEARN:
		len += snprintf((char *)page + len, count-len, "ARL SW Learning Enabled for port=%d\n",
				swproc->arl_port);
		break;

	case ARL_CMD_DIS_SWLEARN:
		len += snprintf((char *)page + len,  count-len,"ARL SW Learning Disabled for port=%d\n",
				swproc->arl_port);
		break;

	case ARL_CMD_ENA_HWLEARN:
		len += snprintf((char *)page + len, count-len, "ARL HW Learning Enabled for port=%d\n",
				swproc->arl_port);
		break;

	case ARL_CMD_DIS_HWLEARN:
		len += snprintf((char *)page + len, count-len, "ARL HW Learning Disabled for port=%d\n",
				swproc->arl_port);
		break;

	case ARL_CMD_AGE_PERIOD:
		len += snprintf((char *)page + len, count-len, "MAC aging period %d\n",
				swproc->arl_mac_aging_period);
		break;

	case ARL_CMD_AGE_POLL:
		len += snprintf((char *)page + len, count-len, "MAC aging poll period %d\n",
				swproc->arl_mac_aging_poll_period);
		break;

	case ARL_CMD_LINK_POLL:
		len += snprintf((char *)page + len, count-len, "Link status poll period %d\n",
				swproc->arl_link_status_poll_period);
		break;
	case ARL_CMD_CACHE_MISS:
		len += snprintf((char *)page + len, count-len, "ARL cache misses %u\n",
				ethsw_arl_cache_miss);
		len += snprintf((char *)page + len, count-len, "ARL cache faked %u\n",
				ethsw_arl_cache_faked);
		break;
	break;
	default:
		len += snprintf(
			(char *)page + len, count-len,
			"ARL cmd is NULL.\n"
			"\n"
			"To issue an ARL cmd, write to /proc/%s/arl\n"
			"\n"
			"    e                      <- enable ARL learning\n"
			"    x                      <- disable ARL learning\n"
			"    s port                 <- enable software ARL learning\n"
			"    S port                 <- disable software ARL learning\n"
			"    h port                 <- enable hardware ARL learning\n"
			"    H port                 <- disable hardware ARL learning\n"
			"    f VVV MM:MM:...        <- find ARL entry with VID/MAC (switch)\n"
			"    a VVV MM:MM:... port   <- add ARL entry with VID/MAC and port (switch)\n"
			"    d VVV MM:MM:...        <- delete ARL entry with VID/MAC (switch)\n"
			"    l EEEE MM:MM:...       <- find multiport entry with etype/MAC\n"
			"    m EEEE MM:MM:... port  <- add multiport entry with etype/MAC and port\n"
			"    n EEEE MM:MM:...       <- delete ARL entry with etype/MAC\n"
			"    The entris below are only valid for 3384\n"
			"    F VVV MM:MM:...        <- find ARL entry with VID/MAC (cache)\n"
			"    A VVV MM:MM:... port   <- add ARL entry with VID/MAC and port (cache)\n"
			"    D VVV MM:MM:...        <- delete ARL entry with VID/MAC (cached)\n"
			"    g AAAA                 <- set MAC aging period\n"
			"    p pppp                 <- set how often we poll for aged MACs\n"
			"    P PPPP                 <- set how ofter we poll for link status changes\n"
			"    M                      <- ARL cache misses\n"
			"\n"
			"where all values are in hex format.\n"
			"\n",
			swproc->dir_name);
		break;
	}

	if (len >= 0) {
		*data += len;
		if (copy_to_user(buffer, (char *)page, len))
			len = -EFAULT;
	}
	free_page(page);

	return(len);
}

static ssize_t ethsw_proc_vlan_write(
	struct file *file, const char __user *buffer, size_t count, loff_t *data)
{
	int len = 0;
	char cmd_buffer[64];
	int cmd_offset;
	char cmd_char;
	u32 cmd_word;
	u32 cmd_word0, cmd_word1;
	u16 vlan_vid;
	u16 vlan_untag_map;
	u16 vlan_fwd_map;
	int vlan_port_id;
	u16 vlan_port_vid;
	u8  vlan_port_pcp;
	u8  vlan_enable;
	u8  vlan_ipv6_mc_nf_mode_enabled;
	int scan_items;
	struct ethsw_device *swdev;
	struct ethsw_proc *swproc;

	swdev = (struct ethsw_device *)file->private_data;
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	swproc = swdev->swproc;
	memset(cmd_buffer, 0, sizeof(cmd_buffer));
	/* copy from user space */
	len = min((int)count, (int)MAX_COPY_COUNT);
	if (copy_from_user(cmd_buffer, buffer, len)) {
		return(-EFAULT);
	}

	/* force NULL termination */
	cmd_buffer[len] = '\0';

	/* clear previous cmd */
	swproc->vlan_cmd = VLAN_CMD_NULL;

	/* parse cmd */
	/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
	scan_items = sscanf(cmd_buffer, "%1c", &cmd_char);
	if (scan_items != 1) {
		return(len);
	}

	cmd_offset = 2;
	vlan_vid = 0;

	switch (cmd_char) {
	case 'f':
	case 'F':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + cmd_offset, "%8x", &cmd_word);
		if (scan_items != 1) {
			return(-EINVAL);
		}

		vlan_vid = (u16)cmd_word;

		/* VLAN table find */
		if (ethsw_vlan_table_find(swdev, vlan_vid, &vlan_untag_map, &vlan_fwd_map)) {
			return(-EIO);
		}

		swproc->vlan_cmd = VLAN_CMD_FIND;
		swproc->vlan_vid       = vlan_vid;
		swproc->vlan_untag_map = vlan_untag_map;
		swproc->vlan_fwd_map   = vlan_fwd_map;
		break;

	case 'a':
	case 'A':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + cmd_offset, "%8x %8x %8x",
							&cmd_word, &cmd_word0, &cmd_word1);
		if (scan_items != 3) {
			return(-EINVAL);
		}

		vlan_vid       = (u16)cmd_word;
		vlan_untag_map = (u16)cmd_word0;
		vlan_fwd_map   = (u16)cmd_word1;

		/* VLAN table add */
		if (ethsw_vlan_table_add(swdev, vlan_vid, vlan_untag_map, vlan_fwd_map)) {
			return(-EIO);
		}

		swproc->vlan_cmd = VLAN_CMD_ADD;
		swproc->vlan_vid       = vlan_vid;
		swproc->vlan_untag_map = vlan_untag_map;
		swproc->vlan_fwd_map   = vlan_fwd_map;
		break;

	case 'd':
	case 'D':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + cmd_offset, "%8x", &cmd_word);
		if (scan_items != 1) {
			return(-EINVAL);
		}

		vlan_vid = (u16)cmd_word;

		/* VLAN table delete */
		if (ethsw_vlan_table_delete(swdev, vlan_vid)) {
			return(-EIO);
		}

		swproc->vlan_cmd = VLAN_CMD_DELETE;
		swproc->vlan_vid = vlan_vid;
		break;

	case 'p':
	case 'P':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + cmd_offset, "%8x %8x %8x",
							&vlan_port_id, &cmd_word0, &cmd_word1);
		if (scan_items != 3) {
			return(-EINVAL);
		}

		vlan_port_vid = cmd_word0;
		vlan_port_pcp = cmd_word1;

		/* VLAN port set */
		if (ethsw_vlan_port_tag_set(
				vlan_port_id,
				vlan_port_vid,
				vlan_port_pcp)) {
			return(-EIO);
		}

		swproc->vlan_cmd = VLAN_CMD_PTAG_SET;
		swproc->vlan_port_id  = vlan_port_id;
		swproc->vlan_port_vid = vlan_port_vid;
		swproc->vlan_port_pcp = vlan_port_pcp;
		break;

	case 'q':
	case 'Q':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + cmd_offset, "%8x", &vlan_port_id);
		if (scan_items != 1) {
			return(-EINVAL);
		}

		/* VLAN port get */
		if (ethsw_vlan_port_tag_get(
				vlan_port_id,
				&vlan_port_vid,
				&vlan_port_pcp)) {
			return(-EIO);
		}

		swproc->vlan_cmd = VLAN_CMD_PTAG_GET;
		swproc->vlan_port_id  = vlan_port_id;
		swproc->vlan_port_vid = vlan_port_vid;
		swproc->vlan_port_pcp = vlan_port_pcp;
		break;

	case 's':
	case 'S':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + cmd_offset, "%8x %8x",
							&vlan_port_id, &cmd_word0);
		if (scan_items != 2) {
			return(-EINVAL);
		}

		vlan_port_vid = cmd_word0;

		/* VLAN port add */
		if (ethsw_vlan_port_vid_add(
				vlan_port_id,
				vlan_port_vid, 1)) {
			return(-EIO);
		}

		swproc->vlan_cmd = VLAN_CMD_PVID_ADD;
		swproc->vlan_port_id  = vlan_port_id;
		swproc->vlan_port_vid = vlan_port_vid;
		break;

	case 't':
	case 'T':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + cmd_offset, "%8x %8x",
							&vlan_port_id, &cmd_word0);
		if (scan_items != 2) {
			return(-EINVAL);
		}

		vlan_port_vid = cmd_word0;

		/* VLAN port delete */
		if (ethsw_vlan_port_vid_delete(
				vlan_port_id,
				vlan_port_vid, 1)) {
			return(-EIO);
		}

		swproc->vlan_cmd = VLAN_CMD_PVID_DEL;
		swproc->vlan_port_id  = vlan_port_id;
		swproc->vlan_port_vid = vlan_port_vid;
		break;

	case 'v':
	case 'V':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + cmd_offset, "%d %d %d",
							&vlan_port_id, &cmd_word0, &cmd_word1);
		if (scan_items != 3) {
			return(-EINVAL);
		}

		/* VLAN port default */
		/* VLAN port get */
		if (ethsw_vlan_port_tag_get(
				vlan_port_id,
				&vlan_port_vid,
				&vlan_port_pcp)) {
			return(-EIO);
		}

		if (vlan_port_vid == cmd_word0)
			break;

		/* VLAN port delete */
		if (ethsw_vlan_port_vid_delete(
				vlan_port_id,
				vlan_port_vid, 1)) {
			return(-EIO);
		}

		vlan_port_vid = cmd_word0;
		vlan_port_pcp = cmd_word1;

		/* VLAN port add */
		if (ethsw_vlan_port_vid_add(
				vlan_port_id,
				vlan_port_vid, 1)) {
			return(-EIO);
		}

		/* VLAN port set */
		if (ethsw_vlan_port_tag_set(
				vlan_port_id,
				vlan_port_vid,
				vlan_port_pcp)) {
			return(-EIO);
		}

		swproc->vlan_cmd = VLAN_CMD_PVID_DEF;
		swproc->vlan_port_id  = vlan_port_id;
		swproc->vlan_port_vid = vlan_port_vid;
		swproc->vlan_port_pcp = vlan_port_pcp;
		break;

	case 'e':
	case 'E':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + cmd_offset, "%8x", &cmd_word);
		if (scan_items != 1) {
			return(-EINVAL);
		}

		vlan_enable = (u8)cmd_word;

		/* VLAN enable/disable */
		if (ethsw_vlan_enable(vlan_enable)) {
			return(-EIO);
		}

		swproc->vlan_cmd = VLAN_CMD_ENABLE;
		swproc->vlan_enable       = vlan_enable;
		break;

	case 'm':
	case 'M':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + cmd_offset, "%4x", &cmd_word);
		if (scan_items != 1)
			return(-EINVAL);

		vlan_ipv6_mc_nf_mode_enabled = (bool)cmd_word;

		swproc->vlan_cmd = VLAN_CMD_IPv6_MC_NF;
		swproc->vlan_ipv6_mc_nf_mode_enabled
			= vlan_ipv6_mc_nf_mode_enabled;
		/* IPv6 multicast non-flooding is only
		 * a swdev private_data variable
		 */
		swdev->vlan_ipv6_mc_nf_mode_enabled =
			swproc->vlan_ipv6_mc_nf_mode_enabled;
		break;

	case ' ':
	case '\n':
	case '\t':
		return(len);

	default:
		return(-EINVAL);
	}

	return(len);
}

static ssize_t ethsw_proc_vlan_read(
	struct file *file, char __user *buffer, size_t count, loff_t *data)
{
	int len = 0;
	struct ethsw_device *swdev;
	struct ethsw_proc *swproc;
	unsigned long page;

	if (*data)
		return 0;

	swdev = (struct ethsw_device *)file->private_data;
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	swproc = swdev->swproc;

	page = get_zeroed_page(GFP_KERNEL);
	if (!page)
		return -ENOMEM;

	switch (swproc->vlan_cmd) {
	case VLAN_CMD_FIND:
		len += snprintf((char *)page + len, count-len, "VLAN table find");
		len += snprintf((char *)page + len, count-len, ": vid=%X", swproc->vlan_vid);
		len += snprintf((char *)page + len, count-len, ", untag_map=%03X", swproc->vlan_untag_map);
		len += snprintf((char *)page + len, count-len, ", fwd_map=%03X\n", swproc->vlan_fwd_map);
		break;

	case VLAN_CMD_ADD:
		len += snprintf((char *)page + len, count-len, "VLAN table add");
		len += snprintf((char *)page + len, count-len, ": vid=%X", swproc->vlan_vid);
		len += snprintf((char *)page + len, count-len, ", untag_map=%03X", swproc->vlan_untag_map);
		len += snprintf((char *)page + len, count-len, ", fwd_map=%03X\n", swproc->vlan_fwd_map);
		break;

	case VLAN_CMD_DELETE:
		len += snprintf((char *)page + len, count-len, "VLAN table delete");
		len += snprintf((char *)page + len, count-len, ": vid=%X\n", swproc->vlan_vid);
		break;

	case VLAN_CMD_PTAG_SET:
		len += snprintf((char *)page + len, count-len, "VLAN port default tag set: "
				"port_id=%d, vid=%X, pcp=%X\n",
				swproc->vlan_port_id,
				swproc->vlan_port_vid,
				swproc->vlan_port_pcp);
		break;

	case VLAN_CMD_PTAG_GET:
		len += snprintf((char *)page + len, count-len, "VLAN port default tag get: "
				"port_id=%d, vid=%X, pcp=%X\n",
				swproc->vlan_port_id,
				swproc->vlan_port_vid,
				swproc->vlan_port_pcp);
		break;

	case VLAN_CMD_PVID_ADD:
		len += snprintf((char *)page + len, count-len, "VLAN port vid add: "
				"port_id=%d, vid=%X\n",
				swproc->vlan_port_id,
				swproc->vlan_port_vid);
		break;

	case VLAN_CMD_PVID_DEL:
		len += snprintf((char *)page + len, count-len, "VLAN port vid delete: "
				"port_id=%d, vid=%X\n",
				swproc->vlan_port_id,
				swproc->vlan_port_vid);
		break;

	case VLAN_CMD_PVID_DEF:
		len += snprintf((char *)page + len, count-len, "VLAN port vid default: "
				"port_id=%d, vid=%X, pcp=%X\n",
				swproc->vlan_port_id,
				swproc->vlan_port_vid,
				swproc->vlan_port_pcp);
		break;

	case VLAN_CMD_ENABLE:
		len += snprintf((char *)page + len, count-len, "VLAN enable: =%d\n",
				swproc->vlan_enable);
		break;

	case VLAN_CMD_IPv6_MC_NF:
		len += snprintf((char *)page + len,  count-len, "IPv6 multicast non-flooding: =%d\n",
				swproc->vlan_ipv6_mc_nf_mode_enabled);
		break;

	default:
		len += snprintf(
			(char *)page + len, count-len,
			"VLAN cmd is NULL.\n"
			"\n"
			"To issue a VLAN cmd, write to /proc/%s/vlan\n"
			"\n"
			"    f vid                  <- find VLAN entry "
							"with VID\n"
			"    a vid umap fmap        <- add VLAN entry "
							"with VID and untag/fwd maps\n"
			"    d vid                  <- delete VLAN entry "
							"with VID\n"
			"    p port_id vid pcp      <- set port default "
							"tag VID and PCP\n"
			"    q port_id              <- get port default "
							"tag VID and PCP\n"
			"    s port_id vid          <- add port VID\n"
			"    t port_id vid          <- delete port VID\n"
			"    v port_id vid pcp      <- default port VID and PCP\n"
			"    e 0/1                  <- Enable/disable VLAN\n"
			"    m 0/1                  <- Enable/disable IPv6 "
							"multicast non-flooding\n"
			"\n"
			"where all values are in hex format.\n"
			"\n",
			swproc->dir_name);
		break;
	}

	if (len >= 0) {
		*data += len;
		if (copy_to_user(buffer, (char *)page, len))
			len = -EFAULT;
	}
	free_page(page);

	return(len);
}


static ssize_t ethsw_proc_hashclash_write(
	struct file *file, const char __user *buffer, size_t count, loff_t *data)
{
	int len = 0;
	char cmd_buffer[64];
	int cmd_offset;
	char cmd_char;
	u32 cmd_word;
	u8  hashclash_enable;
	int scan_items;
	struct ethsw_device *swdev;
	struct ethsw_proc *swproc;

	swdev = (struct ethsw_device *)file->private_data;
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	swproc = swdev->swproc;
	memset(cmd_buffer, 0, sizeof(cmd_buffer));
	/* copy from user space */
	len = min((int)count, (int)MAX_COPY_COUNT);
	if (copy_from_user(cmd_buffer, buffer, len)) {
		return(-EFAULT);
	}

	/* force NULL termination */
	cmd_buffer[len] = '\0';

	/* clear previous cmd */
	swproc->hashclash_cmd = HASHCLASH_CMD_NULL;

	/* parse cmd */
	/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
	scan_items = sscanf(cmd_buffer, "%1c", &cmd_char);
	if (scan_items != 1) {
		return(len);
	}

	cmd_offset = 2;

	switch (cmd_char) {
		case 'e':
		case 'E':
			/* parse arguments */
			/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
			scan_items = sscanf(cmd_buffer + cmd_offset, "%8x", &cmd_word);
			if (scan_items != 1) {
				return(-EINVAL);
			}

			hashclash_enable = (u8)cmd_word;

			/* Hashclash enable/disable */
			if (hash_collision_fix_enable(swdev, hashclash_enable)) {
				return(-EIO);
			}

			swproc->hashclash_cmd = HASHCLASH_CMD_ENABLE;
			swproc->hashclash_enable       = hashclash_enable;
			break;

		case ' ':
		case '\n':
		case '\t':
			return(len);

		default:
			return(-EINVAL);
	}

	return(len);
}

static ssize_t ethsw_proc_hashclash_read(
	struct file *file, char __user *buffer, size_t count, loff_t *data)
{
	int len = 0;
	struct ethsw_device *swdev;
	struct ethsw_proc *swproc;
	unsigned long page;

	if (*data)
		return 0;

	swdev = (struct ethsw_device *)file->private_data;
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	swproc = swdev->swproc;

	page = get_zeroed_page(GFP_KERNEL);
	if (!page)
		return -ENOMEM;

	switch (swproc->hashclash_cmd) {

	case HASHCLASH_CMD_ENABLE:
		len += snprintf((char *)page + len, count-len, "Hashclash enable: = %d\n",
				swproc->hashclash_enable);
		break;

	default:
		len += snprintf(
			(char *)page + len, count-len,
			"Hashclash cmd is NULL.\n"
			"\n"
			"Current Hashclash state is = %d\n\n\n"
			"To issue a Hashclash cmd, write to /proc/%s/hashclash\n"
			"\n"
			"    e 0/1                  <- Enable/disable Hashclash fix\n"
			"\n"
			"where all values are in hex format.\n"
			"\n",
			hash_collision_fix_enabled(), swproc->dir_name);
		break;
	}

	if (len >= 0) {
		*data += len;
		if (copy_to_user(buffer, (char *)page, len))
			len = -EFAULT;
	}
	free_page(page);

	return(len);
}


static ssize_t ethsw_proc_reserved_multicast_write(
	struct file *file, const char __user *buffer, size_t count, loff_t *data)
{
	int len = 0;
	char cmd_buffer[64];
	int cmd_offset;
	char cmd_char;
	u32 cmd_word;
	u8  reserved_multicast_fix_enable;
	int scan_items;
	struct ethsw_device *swdev;
	struct ethsw_proc *swproc;

	swdev = (struct ethsw_device *)file->private_data;
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	swproc = swdev->swproc;
	memset(cmd_buffer, 0, sizeof(cmd_buffer));
	/* copy from user space */
	len = min((int)count, (int)MAX_COPY_COUNT);
	if (copy_from_user(cmd_buffer, buffer, len)) {
		return(-EFAULT);
	}

	/* force NULL termination */
	cmd_buffer[len] = '\0';

	/* clear previous cmd */
	swproc->reserved_multicast_cmd = RESERVED_MULTICAST_CMD_NULL;

	/* parse cmd */
	/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
	scan_items = sscanf(cmd_buffer, "%1c", &cmd_char);
	if (scan_items != 1) {
		return(len);
	}

	cmd_offset = 2;

	switch (cmd_char) {
		case 'e':
		case 'E':
			/* parse arguments */
			/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
			scan_items = sscanf(cmd_buffer + cmd_offset, "%8x", &cmd_word);
			if (scan_items != 1) {
				return(-EINVAL);
			}

			reserved_multicast_fix_enable = (u8)cmd_word;

		    /* reservered multicast enable/disable */
		    if (reserved_multicast_enable(swdev, reserved_multicast_fix_enable)) {
		    	return(-EIO);
		    }

			swproc->reserved_multicast_cmd = RESERVED_MULTICAST_CMD_ENABLE;
			swproc->reserved_multicast_enable       = reserved_multicast_fix_enable;
			break;

		case ' ':
		case '\n':
		case '\t':
			return(len);

		default:
			return(-EINVAL);
	}

	return(len);
}

static ssize_t ethsw_proc_reserved_multicast_read(
	struct file *file, char __user *buffer, size_t count, loff_t *data)
{
	int len = 0;
	struct ethsw_device *swdev;
	struct ethsw_proc *swproc;
	unsigned long page;

	if (*data)
		return 0;

	swdev = (struct ethsw_device *)file->private_data;
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	swproc = swdev->swproc;

	page = get_zeroed_page(GFP_KERNEL);
	if (!page)
		return -ENOMEM;

	switch (swproc->reserved_multicast_cmd) {

	case RESERVED_MULTICAST_CMD_ENABLE:
		len += snprintf((char *)page + len, count-len, "Reserved Multicast enable: = %d\n",
				swproc->reserved_multicast_enable);
		break;

	default:
		len += snprintf(
			(char *)page + len, count-len,
			"Reserved Multicast cmd is NULL.\n"
			"\n"
			"Current Reserved Multicast state is = %d\n\n\n"
			"To issue a reserved multicast cmd, write to /proc/%s/rsvrdmulti\n"
			"\n"
			"    e 0/1                  <- Enable/disable reserved multicast\n"
			"\n"
			"where all values are in hex format.\n"
			"\n",
			reserved_multicast_enabled(), swproc->dir_name);
		break;
	}

	if (len >= 0) {
		*data += len;
		if (copy_to_user(buffer, (char *)page, len))
			len = -EFAULT;
	}
	free_page(page);

	return(len);
}

static ssize_t ethsw_proc_port_write(
	struct file *file, const char __user *buffer, size_t count, loff_t *data)
{
	int len = 0;
	char cmd_buffer[64];
	char cmd_char;
	u32  cmd_word, cmd_word0, cmd_word1, cmd_word2;
	int port_id;
	struct ethsw_port_config port_config;
	struct ethsw_port_status port_status;
	struct ethsw_power_config power_config;
	struct ethsw_power_status power_status;
	int scan_items;
	struct ethsw_device *swdev;
	struct ethsw_proc *swproc;
	struct ethsw_port *port;

	swdev = (struct ethsw_device *)file->private_data;
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	swproc = swdev->swproc;
	memset(cmd_buffer, 0, sizeof(cmd_buffer));
	if (swproc->port_id >= ETHSW_PORT_MAX ) {
		return(-EIO);
	}

	/* copy from user space */
	len = min((int)count, (int)MAX_COPY_COUNT);
	if (copy_from_user(cmd_buffer, buffer, len)) {
		return(-EFAULT);
	}

	if (swproc->port_id >= ETHSW_PORT_MAX ) {
		return(-EIO);
	}

	/* force NULL termination */
	cmd_buffer[len] = '\0';

	/* clear previous cmd */
	swproc->port_cmd = PORT_CMD_NULL;

	/* parsing cmd */
	/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
	scan_items = sscanf(cmd_buffer, "%1c", &cmd_char);
	if (scan_items != 1) {
		return(len);
	}

	switch (cmd_char) {
	case 'e':
	case 'E':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + 2, "%8x",
							&port_id);
		if (scan_items != 1) {
			return(-EINVAL);
		}

		/* execute enable port */
		if (ethsw_port_enable(port_id)) {
			return(-EIO);
		}

		swproc->port_cmd = PORT_CMD_ENABLE;
		swproc->port_id  = port_id;
		break;

	case 'd':
	case 'D':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + 2, "%8x",
							&port_id);
		if (scan_items != 1) {
			return(-EINVAL);
		}

		/* execute disable port */
		if (ethsw_port_disable(port_id)) {
			return(-EIO);
		}

		swproc->port_cmd = PORT_CMD_DISABLE;
		swproc->port_id  = port_id;
		break;

	case 'c':
	case 'C':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + 2, "%8x %8x %8x %8x",
							&port_id,
							&cmd_word,
							&cmd_word0,
							&cmd_word1);
		if (scan_items != 4) {
			return(-EINVAL);
		}

		port_config.auto_neg    = cmd_word;
		port_config.full_duplex = cmd_word0;
		port_config.speed       = cmd_word1;

		/* execute set port config */
		if (ethsw_port_config_set(port_id, &port_config)) {
			return(-EIO);
		}

		swproc->port_cmd = PORT_CMD_CONFIG_SET;
		swproc->port_id     = port_id;
		swproc->port_config = port_config;
		break;

	case 'g':
	case 'G':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + 2, "%8x",
							&port_id);
		if (scan_items != 1) {
			return(-EINVAL);
		}

		/* execute get port status */
		if (ethsw_port_config_get(port_id, &port_config)) {
			return(-EIO);
		}

		swproc->port_cmd = PORT_CMD_CONFIG_GET;
		swproc->port_id     = port_id;
		swproc->port_config = port_config;
		break;

	case 's':
	case 'S':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + 2, "%8x",
							&port_id);
		if (scan_items != 1) {
			return(-EINVAL);
		}

		/* execute get port status */
		if (ethsw_port_status_get(port_id, &port_status)) {
			return(-EIO);
		}

		swproc->port_cmd = PORT_CMD_STATUS_GET;
		swproc->port_id     = port_id;
		swproc->port_status = port_status;
		break;

	case 'p':
	case 'P':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + 2, "%8x %8x %8x %8x %8x",
							&port_id,
							&cmd_word,
							&cmd_word0,
							&cmd_word1,
							&cmd_word2);
		if (scan_items != 5) {
			return(-EINVAL);
		}

		memset(&power_config, 0, sizeof(power_config));
		power_config.auto_power = cmd_word;
		power_config.auto_dll   = cmd_word0;
		power_config.eee_params.enable = cmd_word1;
		power_config.afe_pwrdn = cmd_word2;

		/* execute set port config */
		if (ethsw_port_power_config_set(port_id, &power_config)) {
			return(-EIO);
		}
		if(port_id >= ETHSW_PORT_MAX || port_id < 0)
			return(-EIO);

		if((port_id - (int)swdev->subid_port_offset) < 0 ||
		   (port_id - (int)swdev->subid_port_offset) >= ETHSW_PORT_MAX)
			return(-EIO);

		port = &swdev->port[port_id - (int)swdev->subid_port_offset];
		if (port->ethsw_phy_set_afe_power_state) {
			if (port->type && port->ethsw_phy_set_afe_power_state(
						swdev,
						port_id -
						swdev->subid_port_offset,
						cmd_word2)) {
				return(-EIO);
			}
		}
		if (swdev->ethsw_led_powercntrl) {
			if (port->type && swdev->ethsw_led_powercntrl(swdev, port_id, cmd_word2)) {
				return(-EIO);
			}
		}

		swproc->port_cmd = PORT_CMD_PWR_CONFIG_SET;
		swproc->port_id      = port_id;
		swproc->power_config = power_config;
		break;

	case 'q':
	case 'Q':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + 2, "%8x",
							&port_id);
		if (scan_items != 1) {
			return(-EINVAL);
		}

		/* execute get port status */
		if (ethsw_port_power_config_get(port_id, &power_config)) {
			return(-EIO);
		}

		swproc->port_cmd = PORT_CMD_PWR_CONFIG_GET;
		swproc->port_id      = port_id;
		swproc->power_config = power_config;
		break;

	case 'r':
	case 'R':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + 2, "%8x",
							&port_id);
		if (scan_items != 1) {
			return(-EINVAL);
		}

		/* execute get port status */
		if (ethsw_port_power_status_get(port_id, &power_status)) {
			return(-EIO);
		}

		swproc->port_cmd = PORT_CMD_PWR_STATUS_GET;
		swproc->port_id      = port_id;
		swproc->power_status = power_status;
		break;

	case 'i':
	case 'I':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + 2, "%8x",
							&port_id);
		if (scan_items != 1) {
			return(-EINVAL);
		}

		swproc->port_cmd = PORT_CMD_INFO_GET;
		swproc->port_id  = port_id;
		break;

	case ' ':
	case '\n':
	case '\t':
		return(len);

	default:
		return(-EINVAL);
	}

	return(len);
}

static ssize_t ethsw_proc_port_read(
	struct file *file, char __user *buffer, size_t count, loff_t *data)
{
	int len = 0;
	struct ethsw_device *swdev;
	struct ethsw_proc *swproc;
	unsigned long page;

	if (*data)
		return 0;

	swdev = (struct ethsw_device *)file->private_data;
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	swproc = swdev->swproc;

	if (swproc->port_id >= ETHSW_PORT_MAX ) {
		return(-EIO);
	}

	page = get_zeroed_page(GFP_KERNEL);
	if (!page)
		return -ENOMEM;

	switch (swproc->port_cmd) {
	case PORT_CMD_ENABLE:
		len += snprintf((char *)page + len, count-len, "Port enable: port=%d\n",
				swproc->port_id);
		break;

	case PORT_CMD_DISABLE:
		len += snprintf((char *)page + len, count-len, "Port disable: port=%d\n",
			       swproc->port_id);
		break;

	case PORT_CMD_CONFIG_SET:
		len += snprintf((char *)page + len, count-len, "Port config set: port=%d "
				"auto_neg=%d full_duplex=%d speed=%d\n",
				swproc->port_id,
				swproc->port_config.auto_neg,
				swproc->port_config.full_duplex,
				swproc->port_config.speed);
		break;

	case PORT_CMD_CONFIG_GET:
		len += snprintf((char *)page + len, count-len, "Port config get: port=%d "
				"auto_neg=%d full_duplex=%d speed=%d\n",
				swproc->port_id,
				swproc->port_config.auto_neg,
				swproc->port_config.full_duplex,
				swproc->port_config.speed);
		break;

	case PORT_CMD_STATUS_GET:
		len += snprintf((char *)page + len, count-len, "Port status get: port=%d "
				"link_up=%d full_duplex=%d speed=%d\n",
				swproc->port_id,
				swproc->port_status.link_up,
				swproc->port_status.full_duplex,
				swproc->port_status.speed);
		break;

	case PORT_CMD_PWR_CONFIG_SET:
		len += snprintf((char *)page + len, count-len, "Port power config set: port=%d "
				"auto_power=%d auto_dll=%d eee_enable=%d afe_pwrdn=%d\n",
				swproc->port_id,
				swproc->power_config.auto_power,
				swproc->power_config.auto_dll,
				swproc->power_config.eee_params.enable,
				swproc->power_config.afe_pwrdn);
		break;

	case PORT_CMD_PWR_CONFIG_GET:
		len += snprintf((char *)page + len, count-len, "Port power config get: port=%d "
				"auto_power=%d auto_dll=%d eee_enable=%d afe_pwrdn=%d\n",
				swproc->port_id,
				swproc->power_config.auto_power,
				swproc->power_config.auto_dll,
				swproc->power_config.eee_params.enable,
				swproc->power_config.afe_pwrdn);
		break;

	case PORT_CMD_PWR_STATUS_GET:
		len += snprintf((char *)page + len, count-len, "Port power status get: port=%d "
				"power_up=%d dll_up=%d eee_capable=%d eee_assert=%d eee_indicate=%d afe_pwrdn=%d\n",
				swproc->port_id,
				swproc->power_status.power_up,
				swproc->power_status.dll_up,
				swproc->power_status.eee_capable,
				swproc->power_status.eee_assert,
				swproc->power_status.eee_indicate,
				swproc->power_status.afe_pwrdn);
		break;

	case PORT_CMD_INFO_GET:
		len += snprintf((char *)page + len, count-len, "Port info get: port=%d name=%s enable=%d\n",
				swproc->port_id,
				swdev->port[swproc->port_id].name,
				swdev->port[swproc->port_id].enable);
		break;

	default:
		len += snprintf(
			(char *)page + len, count-len,
			"Port cmd is NULL.\n"
			"\n"
			"To issue a port cmd, write to /proc/%s/port\n"
			"\n"
            "    e port_id                                <- enable port\n"
            "    d port_id                                <- disable port\n"
            "    c port_id auto fd spd                    <- set port config\n"
            "    g port_id                                <- get port config\n"
            "    s port_id                                <- get port status\n"
            "    p port_id auto_pwr dll eee afe_pwrdn     <- set port power config\n"
            "    q port_id                                <- get port power config\n"
            "    r port_id                                <- get port power status\n"
            "    i port_id                                <- get port info\n"
			"\n"
            "where all values are in hex format.\n"
			"\n",
			swproc->dir_name);
		break;
	}

	if (len >= 0) {
		*data += len;
		if (copy_to_user(buffer, (char *)page, len))
			len = -EFAULT;
	}
	free_page(page);

	return(len);
}

static ssize_t ethsw_proc_ethsw_pwr_write(
	struct file *file, const char __user *buffer, size_t count, loff_t *data)
{
	int len = 0;
	char cmd_buffer[64];
	char cmd_char;

	int scan_items;
	struct ethsw_device *swdev;
	struct ethsw_proc *swproc;

	swdev = (struct ethsw_device *)file->private_data;
	swproc = swdev->swproc;
	memset(cmd_buffer, 0, sizeof(cmd_buffer));
	/* copy from user space */
	len = min((int)count, (int)MAX_COPY_COUNT);
	if (copy_from_user(cmd_buffer, buffer, len)) {
		return(-EFAULT);
	}

	/* force NULL termination */
	cmd_buffer[len] = '\0';

	/* clear previous cmd */
	swproc->ethsw_pwr_cmd = ETHSW_PWR_CMD_NULL;

	/* parsing cmd */
	/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
	scan_items = sscanf(cmd_buffer, "%1c", &cmd_char);
	if (scan_items != 1) {
		return(len);
	}

	switch (cmd_char) {
	case 'e':
	case 'E':

		/* execute enable ethsw */
		if (swdev->ethsw_pwr_status) {
			LOG_INFO("Power to Ethernet switch already turned on\n");
			return(len);
		}
		if (swdev->ethsw_normal_pwr_mode(swdev)) {
			swproc->ethsw_pwr_cmd = ETHSW_PWR_CMD_DISABLE;
			return(-EIO);
		}
		swproc->ethsw_pwr_cmd = ETHSW_PWR_CMD_ENABLE;
		break;

	case 'd':
	case 'D':
		/* Check the ethernet switch power status */
		if (!swdev->ethsw_pwr_status) {
			LOG_INFO("Power to Ethernet switch already turned off\n");
			return(len);
		}
		/* execute disable ethsw */
		swproc->ethsw_pwr_cmd = ETHSW_PWR_CMD_DISABLE;
		if (swdev->ethsw_low_pwr_mode(swdev)) {
			return(-EIO);
		}
		break;

	case 'i':
	case 'I':
		/* parse arguments */

		swproc->ethsw_pwr_cmd = ETHSW_PWR_CMD_INFO_GET;
		break;

	case ' ':
	case '\n':
	case '\t':
		return(len);

	default:
		return(-EINVAL);
	}

	return(len);
}

static ssize_t ethsw_proc_ethsw_pwr_read(
	struct file *file, char __user *buffer, size_t count, loff_t *data)
{
	int len = 0;
	struct ethsw_device *swdev;
	struct ethsw_proc *swproc;
	unsigned long page;

	if (*data)
		return 0;

	swdev = (struct ethsw_device *)file->private_data;
	VALIDATE_SWDEV(swdev);

	swproc = swdev->swproc;

	page = get_zeroed_page(GFP_KERNEL);
	if (!page)
		return -ENOMEM;

	switch (swproc->ethsw_pwr_cmd) {
	case ETHSW_PWR_CMD_ENABLE:
		len += snprintf((char *)page + len, count-len, "Ethernet switch enabled\n");
		break;

	case ETHSW_PWR_CMD_DISABLE:
		len += snprintf((char *)page + len, count-len, "Ethernet switch disabled\n");
		break;

	case ETHSW_PWR_CMD_INFO_GET:
		len += snprintf((char *)page + len, count-len, "Ethernet switch info get ethsw_status=%s\n",
					(swdev->ethsw_pwr_status ? "enabled" : "disabled"));

		break;

	default:
		len += snprintf(
			(char *)page + len, count-len,
			"pwr cmd is NULL.\n"
			"\n"
			"To issue a pwr cmd, write to /proc/%s/pwr\n"
			"\n"
            "    e               <- enable ethernet switch power\n"
            "    d               <- disable ethernet switch power\n"
            "    i               <- get ethernet switch power info\n"
			"\n"
            "where all values are in hex format.\n"
			"\n",
			swproc->dir_name);
		break;
	}
	if (len >= 0) {
		*data += len;
		if (copy_to_user(buffer, (char *)page, len))
			len = -EFAULT;
	}
	free_page(page);
	
	return(len);
}

static int ethsw_print_mib_counters(
	char *buffer, int buffer_len, int start, int count, u32 *values)
{
	int len = 0;
	int i;

	for (i = start; i < start + count; i++) {
		if (i == MIB_TX_OCTETS_RSV ||
			i == MIB_RX_OCTETS_RSV ||
			i == MIB_RX_GOOD_OCTETS_RSV) {
			/* reserved for 64-bit counter */
			values++;
			continue;
		}
		else if (i == MIB_TX_OCTETS ||
				 i == MIB_RX_OCTETS ||
				 i == MIB_RX_GOOD_OCTETS) {
			/* 64-bit counter */
			len += snprintf(buffer + len, buffer_len-len, "%24s=%-llX\n",
					mib_name_tbl[i], *(u64 *)values++);
		}
		else {
			/* 32-bit counter */
			len += snprintf(buffer + len, buffer_len-len, "%24s=%-X\n",
				       mib_name_tbl[i], *values++);
		}
	}
	return(len);
}

static ssize_t ethsw_proc_mib_write(
	struct file *file, const char __user *buffer, size_t count, loff_t *data)
{
	int len = 0;
	char cmd_buffer[64];
	char cmd_char;
	char mib_name[64];
	u32 mib_port, mib_counter;
	int scan_items;
	struct ethsw_device *swdev;
	struct ethsw_proc *swproc;

	swdev = (struct ethsw_device *)file->private_data;
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	swproc = swdev->swproc;
	memset(cmd_buffer, 0, sizeof(cmd_buffer));
	memset(mib_name, 0, sizeof(mib_name));

	/* copy from user space */
	len = min((int)count, (int)MAX_COPY_COUNT);
	if (copy_from_user(cmd_buffer, buffer, len)) {
		return(-EFAULT);
	}

	/* force NULL termination */
	cmd_buffer[len] = '\0';

	/* clear previous cmd */
	swproc->mib_cmd = MIB_CMD_NULL;

	/* parse cmd */
	/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
	scan_items = sscanf(cmd_buffer, "%1c", &cmd_char);
	if (scan_items != 1) {
		return(len);
	}

	switch (cmd_char) {
	case 'c':
	case 'C':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + 2, "%8x", &mib_port);
		if (scan_items == 0) {
			mib_port = (1 << ETHSW_PORT_MAX) - 1;
		}
		else if (scan_items != 1) {
			return(-EINVAL);
		}

		/* execute MIB reset */
		if (ethsw_mib_counters_reset(mib_port)) {
			return(-EIO);
		}

		swproc->mib_cmd  = MIB_CMD_RESET;
		swproc->mib_port = mib_port;
		break;

	case 'g':
	case 'G':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + 2, "%8x %63s", &mib_port, mib_name);
		if (scan_items != 2) {
			return(-EINVAL);
		}

		/* lookup MIB counter name */
		for (mib_counter = 0; mib_counter < MIB_COUNTER_MAX; mib_counter++) {
			if (strcmp(mib_name, mib_name_tbl[mib_counter]) == 0) {
				break;
			}
		}

		if (mib_port >= ETHSW_PORT_MAX ||
			mib_counter >= MIB_COUNTER_MAX) {
			return(-EINVAL);
		}

		/* execute MIB get */
		if (ethsw_mib_counter64_get(
				mib_port + swdev->subid_port_offset,
				mib_counter,
				&swproc->mib_value)) {
			return(-EIO);
		}

		swproc->mib_cmd     = MIB_CMD_GET;
		swproc->mib_port    = mib_port;
		swproc->mib_counter = mib_counter;
		break;

	case 't':
	case 'T':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + 2, "%8x", &mib_port);
		if (scan_items != 1) {
			return(-EINVAL);
		}

		/* execute MIB get counters */
		if (ethsw_mib_counters_get_base(
				swdev,
				mib_port,
				MIB_TX_COUNTER_START,
				MIB_TX_COUNTER_MAX - MIB_TX_COUNTER_START,
				swproc->mib_values + MIB_TX_COUNTER_START)) {
			return(-EIO);
		}

		swproc->mib_cmd  = MIB_CMD_TX_ALL;
		swproc->mib_port = mib_port;
		break;

	case 'r':
	case 'R':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + 2, "%8x", &mib_port);
		if (scan_items != 1) {
			return(-EINVAL);
		}

		/* execute MIB get counters */
		if (ethsw_mib_counters_get_base(
				swdev,
				mib_port,
				MIB_RX_COUNTER_START,
				MIB_RX_COUNTER_MAX - MIB_RX_COUNTER_START,
				swproc->mib_values + MIB_RX_COUNTER_START)) {
			return(-EIO);
		}

		swproc->mib_cmd  = MIB_CMD_RX_ALL;
		swproc->mib_port = mib_port;
		break;

	case 'e':
	case 'E':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + 2, "%8x", &mib_port);
		if (scan_items != 1) {
			return(-EINVAL);
		}

		/* execute MIB get counters */
		if (ethsw_mib_counters_get_base(
				swdev,
				mib_port,
				MIB_EEE_COUNTER_START,
				MIB_EEE_COUNTER_MAX - MIB_EEE_COUNTER_START,
				swproc->mib_values + MIB_EEE_COUNTER_START)) {
			return(-EIO);
		}

		swproc->mib_cmd  = MIB_CMD_EEE_ALL;
		swproc->mib_port = mib_port;
		break;

	case 'a':
	case 'A':
		/* parse arguments */
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + 2, "%8x", &mib_port);
		if (scan_items != 1) {
			return(-EINVAL);
		}

		/* execute MIB get counters */
		if (ethsw_mib_counters_get_base(
				swdev,
				mib_port,
				MIB_COUNTER_START,
				MIB_COUNTER_MAX - MIB_COUNTER_START,
				swproc->mib_values)) {
			return(-EIO);
		}

		/* write results to buffer */
		swproc->mib_cmd  = MIB_CMD_ALL;
		swproc->mib_port = mib_port;
		break;

	case ' ':
	case '\n':
	case '\t':
		return(len);

	default:
		return(-EINVAL);
	}

	return(len);
}

static ssize_t ethsw_proc_mib_read(
	struct file *file, char __user *buffer, size_t count, loff_t *data)
{
	int len = 0;
	struct ethsw_device *swdev;
	struct ethsw_proc *swproc;
	unsigned long page;

	if (*data)
		return 0;

	swdev = (struct ethsw_device *)file->private_data;
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	swproc = swdev->swproc;

	page = get_zeroed_page(GFP_KERNEL);
	if (!page)
		return -ENOMEM;

	switch (swproc->mib_cmd) {
	case MIB_CMD_RESET:
		len += snprintf((char *)page + len, count-len, "MIB reset: port_map=%X\n",
				swproc->mib_port);
		break;

	case MIB_CMD_GET:
		len += snprintf((char *)page + len, count-len, "MIB get: port_id=%d, %s=%llX\n",
			       swproc->mib_port,
			       mib_name_tbl[swproc->mib_counter],
			       swproc->mib_value);
		break;

	case MIB_CMD_TX_ALL:
		len += snprintf((char *)page + len, count-len, "MIB get TX counters: port_id=%d\n\n",
				swproc->mib_port);

		len += ethsw_print_mib_counters(
			(char *)page + len, count-len,
			MIB_TX_COUNTER_START,
			MIB_TX_COUNTER_MAX - MIB_TX_COUNTER_START,
			swproc->mib_values + MIB_TX_COUNTER_START);
		break;

	case MIB_CMD_RX_ALL:
		len += snprintf((char *)page + len, count-len, "MIB get RX counters: port_id=%d\n\n",
				swproc->mib_port);

		len += ethsw_print_mib_counters(
			(char *)page + len, count-len,
			MIB_RX_COUNTER_START,
			MIB_RX_COUNTER_MAX - MIB_RX_COUNTER_START,
			swproc->mib_values + MIB_RX_COUNTER_START);
		break;

	case MIB_CMD_EEE_ALL:
		len += snprintf((char *)page + len, count-len, "MIB get EEE counters: port_id=%d\n\n",
				swproc->mib_port);

		len += ethsw_print_mib_counters(
			(char *)page + len, count-len,
			MIB_EEE_COUNTER_START,
			MIB_EEE_COUNTER_MAX - MIB_EEE_COUNTER_START,
			swproc->mib_values + MIB_EEE_COUNTER_START);
		break;

	case MIB_CMD_ALL:
		len += snprintf((char *)page + len, count-len, "MIB get all counters: port_id=%d\n\n",
			       swproc->mib_port);

		len += ethsw_print_mib_counters(
			(char *)page + len, count-len,
			MIB_COUNTER_START,
			MIB_COUNTER_MAX - MIB_COUNTER_START,
			swproc->mib_values);
		break;

	default:
		len += snprintf(
			(char *)page + len, count-len,
			"MIB cmd is NULL.\n"
			"\n"
			"To issue a MIB cmd, write to /proc/%s/mib\n"
			"\n"
            "    c [port_map]           <- reset all MIB counters for the port map\n"
            "    g port_id mib_counter  <- get the named MIB counter of the given port\n"
            "    t port_id              <- get all MIB TX counters of the given port\n"
            "    r port_id              <- get all MIB RX counters of the given port\n"
            "    e port_id              <- get all MIB EEE counters of the given port\n"
            "    a port_id              <- get all MIB counters of the given port\n"
			"\n"
            "where all values are in hex format.\n"
			"\n",
			swproc->dir_name);
	}

	if (len >= 0) {
		*data += len;
		if (copy_to_user(buffer, (char *)page, len))
			len = -EFAULT;
	}
	free_page(page);

	return(len);
}


#ifdef ETHSW_FLOW_CONTROL_ENABLED

static ssize_t ethsw_proc_flowctl_write(
	struct file *file, const char __user *buffer, size_t count, loff_t *data)
{
	int len = 0;
	char cmd_buffer[64];
	char cmd_char;
	int scan_items;
	u32 port, tc, qid, threshold, ret;
	struct ethsw_device *swdev;
	struct ethsw_proc *swproc;

	swdev = (struct ethsw_device *)file->private_data;
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	swproc = swdev->swproc;
	memset(cmd_buffer, 0, sizeof(cmd_buffer));
	/* copy from user space */
	len = min((int)count, (int)MAX_COPY_COUNT);
	if (copy_from_user(cmd_buffer, buffer, len)) {
		return(-EFAULT);
	}

	/* force NULL termination */
	cmd_buffer[len] = '\0';

	/* clear previous cmd */
	swproc->flowctl_cmd = FLOWCTL_CMD_NULL;

	/* parse cmd */
	/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
	scan_items = sscanf(cmd_buffer, "%1c", &cmd_char);
	if (scan_items != 1) {
		return(len);
	}

	switch (cmd_char) {
	case 'm':
	case 'M':
		swproc->flowctl_cmd = FLOWCTL_CMD_MAP_SET;
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + 2, "%8x %8x %8x",
				    &port, &tc, &qid);
		if (scan_items != 3) {
			return(-EINVAL);
		}
		if(port >= ETHSW_PORT_MAX) {
			return(-EINVAL);
		}

		ret = ethsw_set_tc2cos_mapping(swdev, port, tc, qid);
		if(ret) return(ret);

		swproc->flowctl_port = port;
		swproc->flowctl_qid = qid;
		swproc->flowctl_tc = tc;
		break;
	case 't':
	case 'T':
		swproc->flowctl_cmd = FLOWCTL_CMD_THRESHOLD_SET;
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + 2, "%8x %8x %8x",
				    &port, &qid, &threshold);
		if (scan_items != 3) {
			return(-EINVAL);
		}
		if(port >= ETHSW_PORT_MAX) {
			return(-EINVAL);
		}

		ret = ethsw_set_acb_threshold(swdev, port, qid, threshold);
		if(ret) return(ret);

		swproc->flowctl_port = port;
		swproc->flowctl_qid = qid;
		swproc->flowctl_threshold = threshold;
		break;
	case 'g':
	case 'G':
		swproc->flowctl_cmd = FLOWCTL_CMD_MAP_GET;
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + 2, "%8x", &port);
		if (scan_items != 1) {
			return(-EINVAL);
		}
		if(port >= ETHSW_PORT_MAX) {
			return(-EINVAL);
		}

		swproc->flowctl_port = port;

		for(tc = 0; tc < ETHSW_TC_MAX; tc ++) {
			ret = ethsw_get_tc2cos_mapping(swdev, port, tc, &swproc->tc2qid_map[tc]);
			if(ret) return ret;
		}
		break;
	case 'f':
	case 'F':
		swproc->flowctl_cmd = FLOWCTL_CMD_THRESHOLD_GET;
		/* coverity [secure_coding] sscanf is OK with correct precision in format specifiers */
		scan_items = sscanf(cmd_buffer + 2, "%8x", &port);
		if (scan_items != 1) {
			return(-EINVAL);
		}
		if(port >= ETHSW_PORT_MAX) {
			return(-EINVAL);
		}

		swproc->flowctl_port = port;

		for(qid = 0; qid < ETHSW_ACB_QUEUES_PER_PORT; qid ++) {
			ret = ethsw_get_acb_threshold(swdev, port, qid, &swproc->port_qid_thresholds[qid]);
			if(ret) return ret;
		}
		break;

	case ' ':
	case '\n':
	case '\t':
		return(len);

	default:
		return(-EINVAL);
	}

	return(len);
}

static ssize_t ethsw_proc_flowctl_read(
	struct file *file, char __user *buffer, size_t count, loff_t *data)
{
	int len = 0;
	struct ethsw_device *swdev;
	struct ethsw_proc *swproc;
	u32 tc, qid;
	unsigned long page;

	if (*data)
		return 0;

	swdev = (struct ethsw_device *)file->private_data;
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	swproc = swdev->swproc;

	page = get_zeroed_page(GFP_KERNEL);
	if (!page)
		return -ENOMEM;

	switch (swproc->flowctl_cmd) {
	case FLOWCTL_CMD_MAP_SET:
		len += snprintf((char *)page + len, count-len, "port %d TC 0x%x mapped to QID 0x%x \n",
				swproc->flowctl_port, swproc->flowctl_tc, swproc->flowctl_qid);
		break;
	case FLOWCTL_CMD_MAP_GET:
		len += snprintf((char *)page + len, count-len, "TC -> QID mapping for port %d\n",
				swproc->flowctl_port);
		len += snprintf((char *)page + len, count-len, "tc\tqid\n");

		for(tc = 0; tc < ETHSW_TC_MAX; tc ++) {
			len += snprintf((char *)page + len, count-len, "0x%x\t0x%x\n", tc, swproc->tc2qid_map[tc]);
		}
		break;
	case FLOWCTL_CMD_THRESHOLD_SET:
		len += snprintf((char *)page + len, count-len, "port %d QID 0x%x threshold 0x%x\n",
				swproc->flowctl_port, swproc->flowctl_qid, swproc->flowctl_threshold);
		break;
	case FLOWCTL_CMD_THRESHOLD_GET:
		len += snprintf((char *)page + len, count-len, "QID thresholds for port %d\n",
				swproc->flowctl_port);
		len += snprintf((char *)page + len, count-len, "qid\tthreshold\n");

		for(qid = 0; qid < ETHSW_ACB_QUEUES_PER_PORT; qid ++) {
			len += snprintf((char *)page + len, count-len, "0x%x\t0x%x\n", qid, swproc->port_qid_thresholds[qid]);
		}

		break;
	default:
		len += snprintf(
			(char *)page + len, count-len,
			"flowctl cmd is NULL.\n"
			"\n"
			"To issue a flowctl cmd, write to /proc/%s/flowctl\n"
			"\n"
            "    m port_id tc qid        <- set port tc->qid map\n"
            "    t port_id qid threshold <- set queue xoff threshold for port\n"
	    "    g port_id               <- get port tc->qid map\n"
            "    f port_id               <- get port xoff thresholds for all queues\n"
			"\n"
            "where all values are in hex format.\n"
			"\n",
			swproc->dir_name);
		break;
	}

	if (len >= 0) {
		*data += len;
		if (copy_to_user(buffer, (char *)page, len))
			len = -EFAULT;
	}
	free_page(page);

	return(len);
};

#endif /* ETHSW_FLOW_CONTROL_ENABLED */


static int ethsw_proc_open(struct inode *inode, struct file *f)
{
	f->private_data = PDE_DATA(inode);
	if (!f->private_data)
		return -EPERM;
	return 0;
}

static const struct file_operations ethsw_proc_ctl_ops = {
	.owner  = THIS_MODULE,
	.open = ethsw_proc_open,
	.read = ethsw_proc_ctl_read,
	.write  = ethsw_proc_ctl_write,
};

static const struct file_operations ethsw_proc_creg_ops = {
	.owner  = THIS_MODULE,
	.open = ethsw_proc_open,
	.read = ethsw_proc_creg_read,
	.write  = ethsw_proc_creg_write,
};

static const struct file_operations ethsw_proc_arl_ops = {
	.owner  = THIS_MODULE,
	.open = ethsw_proc_open,
	.read = ethsw_proc_arl_read,
	.write  = ethsw_proc_arl_write,
};

static ethsw_arl_table_entry tbl_entry[ETHSW_ARL_TBL_ENTRIES_PER_READ];
static void *arl_retval = 0;

static void *dumparl_proc_start(struct seq_file *seq, loff_t *pos)
{
	void *start = NULL;
	struct ethsw_device *swdev;

	swdev = (struct ethsw_device *)seq->private;
	if (!swdev) {
		LOG_ERR("%s Ethsw switch device not initialized\n",__FUNCTION__);
		return NULL;
	}
	if (!swdev->ethsw_pwr_status) {
		return NULL;
	}

	if (*pos == 0) {
		start = (void*)1;
		arl_retval = (void *)ethsw_arl_table_get_next(swdev, tbl_entry);
	}
	return start;
}

static void dumparl_proc_stop(struct seq_file *seq, void *v)
{
}

static void *dumparl_proc_next(struct seq_file *seq, void *v,
					loff_t *pos)
{
	struct ethsw_device *swdev;

	swdev = (struct ethsw_device *)seq->private;
	if (!swdev) {
		LOG_ERR("%s Ethsw switch device not initialized\n",__FUNCTION__);
		return NULL;
	}
	if (!swdev->ethsw_pwr_status) {
		return NULL;
	}

	arl_retval = (void *)ethsw_arl_table_get_next(swdev, tbl_entry);
	if (arl_retval == (void *)ETHSW_ARL_TBL_LAST_ENTRY) {
		arl_retval = NULL;
	} else {
		arl_retval = (void *)1;
	}

	return arl_retval;
}

static int dumparl_proc_show(struct seq_file *seq, void *v)
{
	int i;
	for(i = 0; i < ETHSW_ARL_TBL_ENTRIES_PER_READ; i ++) {
		if(tbl_entry[i].valid) {
			seq_printf(seq,
			"mac=%pM port=%u vid=%04u static=%u\n",
			tbl_entry[i].mac, tbl_entry[i].port,
			tbl_entry[i].vid, tbl_entry[i].stat);
		}
	}

	return 0;
}



static const struct seq_operations ethsw_proc_dumparl_seq_ops = {
	.start	= dumparl_proc_start,
	.stop	= dumparl_proc_stop,
	.next	= dumparl_proc_next,
	.show	= dumparl_proc_show,
};

static int dumparl_proc_open(struct inode *inode, struct file *file)
{
	int fid;

	fid = seq_open(file, &ethsw_proc_dumparl_seq_ops);
	if (!fid)
		((struct seq_file *)(file->private_data))->private = (void *)PDE_DATA(inode);
	return fid;
}
static const struct file_operations dumparl_fops = {
	.owner		= THIS_MODULE,
	.open		= dumparl_proc_open,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= seq_release,
};

static const struct file_operations ethsw_proc_vlan_ops = {
	.owner  = THIS_MODULE,
	.open = ethsw_proc_open,
	.read = ethsw_proc_vlan_read,
	.write  = ethsw_proc_vlan_write,
};

static const struct file_operations ethsw_proc_hashclash_ops = {
	.owner  = THIS_MODULE,
	.open = ethsw_proc_open,
	.read = ethsw_proc_hashclash_read,
	.write  = ethsw_proc_hashclash_write,
};

static const struct file_operations ethsw_proc_reserved_multicast_ops = {
	.owner  = THIS_MODULE,
	.open = ethsw_proc_open,
	.read = ethsw_proc_reserved_multicast_read,
	.write  = ethsw_proc_reserved_multicast_write,
};


static const struct file_operations ethsw_proc_port_ops = {
	.owner  = THIS_MODULE,
	.open = ethsw_proc_open,
	.read = ethsw_proc_port_read,
	.write  = ethsw_proc_port_write,
};

static const struct file_operations ethsw_proc_mib_ops = {
	.owner  = THIS_MODULE,
	.open = ethsw_proc_open,
	.read = ethsw_proc_mib_read,
	.write  = ethsw_proc_mib_write,
};

#ifdef ETHSW_FLOW_CONTROL_ENABLED
static const struct file_operations ethsw_proc_flowctl_ops = {
	.owner  = THIS_MODULE,
	.open = ethsw_proc_open,
	.read = ethsw_proc_flowctl_read,
	.write  = ethsw_proc_flowctl_write,
};
#endif

static const struct file_operations ethsw_proc_pwr_ops = {
	.owner  = THIS_MODULE,
	.open = ethsw_proc_open,
	.read = ethsw_proc_ethsw_pwr_read,
	.write  = ethsw_proc_ethsw_pwr_write,
};

int ethsw_proc_init(struct ethsw_device *swdev)
{
	int ret = 0;
	char file_name[64];
   struct proc_dir_entry *proc_dir;
	struct proc_dir_entry *entry;
	struct ethsw_proc *swproc;

	FUNC_ENTER();

	/* alloc ethsw proc control block */
	swproc = kzalloc(sizeof(struct ethsw_proc), GFP_KERNEL);
	if (!swproc) {
		LOG_ERR("Ethsw proc control block alloc failed\n");
		ret = -ENOMEM;
		goto ERROR;
	}

	swdev->swproc = swproc;

	/* create proc dir and files */
	snprintf(swproc->dir_name, sizeof(swproc->dir_name), "driver/%s", swdev->name);
	proc_dir = proc_mkdir(swproc->dir_name, NULL);
	if (!proc_dir) {
		LOG_ERR("Make /proc/%s dir failed\n", swproc->dir_name);
		ret = -EFAULT;
		goto ERROR;
	}

	snprintf(file_name, sizeof(file_name), "%s/ctl", swproc->dir_name);
	entry = proc_create_data(file_name, 0644, NULL, &ethsw_proc_ctl_ops, swdev);

	snprintf(file_name, sizeof(file_name), "%s/creg", swproc->dir_name);
	entry = proc_create_data(file_name, 0644, NULL, &ethsw_proc_creg_ops, swdev);

	snprintf(file_name, sizeof(file_name), "%s/arl", swproc->dir_name);
	entry = proc_create_data(file_name, 0644, NULL, &ethsw_proc_arl_ops, swdev);

	snprintf(file_name, sizeof(file_name), "%s/dumparl", swproc->dir_name);
	entry = proc_create_data(file_name, 0644,
				     NULL, &dumparl_fops, swdev);
   
	snprintf(file_name, sizeof(file_name), "%s/vlan", swproc->dir_name);
	entry = proc_create_data(file_name, 0644, NULL, &ethsw_proc_vlan_ops, swdev);

	snprintf(file_name, sizeof(file_name), "%s/hashclash", swproc->dir_name);
	entry = proc_create_data(file_name, 0644, NULL, &ethsw_proc_hashclash_ops, swdev);

	snprintf(file_name, sizeof(file_name), "%s/rsvdmulti", swproc->dir_name);
	entry = proc_create_data(file_name, 0644, NULL, &ethsw_proc_reserved_multicast_ops, swdev);

	snprintf(file_name, sizeof(file_name), "%s/port", swproc->dir_name);
	entry = proc_create_data(file_name, 0644, NULL
							 , &ethsw_proc_port_ops, swdev);

	snprintf(file_name, sizeof(file_name), "%s/mib", swproc->dir_name);
	entry = proc_create_data(file_name, 0644, NULL, &ethsw_proc_mib_ops, swdev);
#ifdef ETHSW_FLOW_CONTROL_ENABLED
	snprintf(file_name, sizeof(file_name), "%s/flowctl", swproc->dir_name);
	entry = proc_create_data(file_name, 0644, NULL, &ethsw_proc_flowctl_ops, swdev);
#endif
	snprintf(file_name, sizeof(file_name), "%s/pwr", swproc->dir_name);
	entry = proc_create_data(file_name, 0644, NULL, &ethsw_proc_pwr_ops, swdev);

	LOG_INFO("/proc/%s dir and files created\n", swproc->dir_name);
	LOG_INFO("Ethsw proc initialized\n");

	FUNC_LEAVE();
	return(ret);

 ERROR:
	/* exit proc properly */
	ethsw_proc_exit(swdev);

	FUNC_LEAVE();
	return(ret);
}

void ethsw_proc_exit(struct ethsw_device *swdev)
{
	char file_name[64];
	struct ethsw_proc *swproc;

	FUNC_ENTER();

	swproc = swdev->swproc;
	if (!swproc) {
		goto EXIT;
	}

	/* remove proc dir and files */
	snprintf(file_name, sizeof(file_name), "%s/ctl", swproc->dir_name);
	remove_proc_entry(file_name, NULL);

	snprintf(file_name, sizeof(file_name), "%s/creg", swproc->dir_name);
	remove_proc_entry(file_name, NULL);

	snprintf(file_name, sizeof(file_name), "%s/arl", swproc->dir_name);
	remove_proc_entry(file_name, NULL);

	snprintf(file_name, sizeof(file_name), "%s/dumparl", swproc->dir_name);
	remove_proc_entry(file_name, NULL);

	snprintf(file_name, sizeof(file_name), "%s/vlan", swproc->dir_name);
	remove_proc_entry(file_name, NULL);

	snprintf(file_name, sizeof(file_name), "%s/hashclash", swproc->dir_name);
	remove_proc_entry(file_name, NULL);

	snprintf(file_name, sizeof(file_name), "%s/port", swproc->dir_name);
	remove_proc_entry(file_name, NULL);

	snprintf(file_name, sizeof(file_name), "%s/mib", swproc->dir_name);
	remove_proc_entry(file_name, NULL);
#ifdef ETHSW_FLOW_CONTROL_ENABLED
	snprintf(file_name, sizeof(file_name), "%s/flowctl", swproc->dir_name);
	remove_proc_entry(file_name, NULL);
#endif
	snprintf(file_name, sizeof(file_name), "%s/pwr", swproc->dir_name);
	remove_proc_entry(file_name, NULL);

	remove_proc_entry(swproc->dir_name, NULL);

	/* free ethsw proc control block */
	kfree(swproc);
	swdev->swproc = NULL;

	LOG_INFO("Ethsw proc exited\n");

 EXIT:
	FUNC_LEAVE();
}

#endif
