 /****************************************************************************
 *
 * Copyright (c) 2015-2018 Broadcom. All rights reserved
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Jayesh Patel <jayeshp@broadcom.com>
 ****************************************************************************/

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/rculist.h>
#include <linux/spinlock.h>
#include <linux/times.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/jhash.h>
#include <linux/random.h>
#include <linux/slab.h>
#include <linux/atomic.h>
#include <asm/unaligned.h>
#include "flowmgr.h"
#include "flowmgr_fap.h"

/* Following code is derived from br_fdb.c implementation */

#define FLOWMGR_HASH_SIZE	NUM_MCTX

static spinlock_t		hash_lock;
static struct hlist_head	hash[FLOWMGR_HASH_SIZE];

static struct kmem_cache *flowmgr_mdb_cache __read_mostly;

int flowmgr_mdb_init(void)
{
	flowmgr_mdb_cache = kmem_cache_create("flowmgr_mdb_cache",
					     sizeof(struct flowmgr_mdb_entry),
					     0,
					     SLAB_HWCACHE_ALIGN, NULL);
	if (!flowmgr_mdb_cache)
		return -ENOMEM;

	return 0;
}

void flowmgr_mdb_fini(void)
{
	kmem_cache_destroy(flowmgr_mdb_cache);
}

static inline int flowmgr_mdb_hash(const __u8 iif)
{
	return iif % FLOWMGR_HASH_SIZE;
}

static int flowmgr_mdb_update_dev_fast_mcast_stats(struct net_device *dev,
					u32 packets, u32 bytes, int rx)
{
	struct pcpu_sw_netstats *tstats = NULL;
	struct bcm_nethooks *nethooks = bcm_nethooks_priv();

	if (dev->tstats) {
		tstats = this_cpu_ptr(dev->tstats);
	} else if (nethooks->sig == BCM_NETHOOKS_SIG && nethooks->offload_stats.mcast_fast) {
		tstats = this_cpu_ptr(nethooks->offload_stats.mcast_fast);
	} else {
		return -ENOMEM;
	}

	if (tstats) {
		u64_stats_update_begin(&tstats->syncp);
		if (rx) {
			tstats->rx_bytes += bytes;
			tstats->rx_packets += packets;
		} else {
			tstats->tx_bytes += bytes;
			tstats->tx_packets += packets;
		}
		u64_stats_update_end(&tstats->syncp);
	}
	return 0;
}

static int flowmgr_mdb_update_mcast_counters(struct flowmgr_mdb_entry *f)
{
	int ret = 0;
	struct net_device *dev;

	flowmgr_get_mcast_conters(f);
	if (f->packets > 0) {
		dev = __dev_get_by_index(&init_net, f->iif);
		if (!dev)
			return -1;
		ret = flowmgr_mdb_update_dev_fast_mcast_stats(dev, 
			f->packets, f->bytes, 1);
	}

	return ret;
}

int flowmgr_mdb_update_dev_mcast_stats(struct net_device *dev)
{
	int i, ret = 0;
	struct net_device *mcast_dev;

	spin_lock_bh(&hash_lock);
	for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
		struct flowmgr_mdb_entry *f;
		struct hlist_node *n;
		hlist_iterate_safe(f, n, &hash[i], hlist) {
			mcast_dev = __dev_get_by_index(&init_net, f->iif);
			if (!mcast_dev)
				continue;
			if (strncmp(dev->name, mcast_dev->name, strlen(dev->name)))
				continue;
			ret = flowmgr_mdb_update_mcast_counters(f);
		}
	}
	spin_unlock_bh(&hash_lock);

	return ret;
}

int flowmgr_mdb_update_all_mcast_stats(void)
{
	int i, ret = 0;

	spin_lock_bh(&hash_lock);
	for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
		struct flowmgr_mdb_entry *f;
		struct hlist_node *n;
		hlist_iterate_safe(f, n, &hash[i], hlist) {
			ret = flowmgr_mdb_update_mcast_counters(f);
		}
	}
	spin_unlock_bh(&hash_lock);

	return ret;
}

static void mdb_rcu_free(struct rcu_head *head)
{
	struct flowmgr_mdb_entry *ent
		= container_of(head, struct flowmgr_mdb_entry, rcu);
	kmem_cache_free(flowmgr_mdb_cache, ent);
}

static void mdb_delete(struct flowmgr_mdb_entry *f, const __u8 oif)
{
	flowmgr_mdb_update_mcast_counters(f);
	flowmgr_demote_mcast(f, oif);
	if (atomic_dec_and_test(&f->use) || !oif) {
		hlist_del_rcu(&f->hlist);
		call_rcu(&f->rcu, mdb_rcu_free);
	}
}

/* Completely flush all dynamic entries in multicast database */
void flowmgr_mdb_flush(void)
{
	int i;

	spin_lock_bh(&hash_lock);
	for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
		struct flowmgr_mdb_entry *f;
		struct hlist_node *n;
		hlist_iterate_safe(f, n, &hash[i], hlist) {
			mdb_delete(f, 0);
		}
	}
	spin_unlock_bh(&hash_lock);
}

/* Flush all entries referring to a specific group */
void flowmgr_mdb_delete_by_group(const __be32 *group)
{
	int i;

	spin_lock_bh(&hash_lock);
	for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
		struct hlist_node *h, *g;

		hlist_for_each_safe(h, g, &hash[i]) {
			struct flowmgr_mdb_entry *f;
			f = hlist_entry(h, struct flowmgr_mdb_entry, hlist);
			if (memcmp(f->group, group, sizeof(f->group)))
				continue;

			mdb_delete(f, 0);
		}
	}
	spin_unlock_bh(&hash_lock);
}

/* Internal Function: Find entry based on specific group in specific list */
static struct flowmgr_mdb_entry *mdb_find(struct hlist_head *head,
					  const __be32 *group)
{
	struct flowmgr_mdb_entry *mdb;

	hlist_iterate(mdb, head, hlist) {
		if (memcmp(mdb->group, group, sizeof(mdb->group)) == 0)
			return mdb;
	}
	return NULL;
}

/* Find entry based on specific group in specific list */
struct flowmgr_mdb_entry *flowmgr_mdb_find(const __u8 iif, const __be32 *group)
{
	struct hlist_head *head = &hash[flowmgr_mdb_hash(iif)];
	struct flowmgr_mdb_entry *mdb = NULL;

	spin_lock_bh(&hash_lock);
	mdb = mdb_find(head, group);
	spin_unlock_bh(&hash_lock);
	return mdb;
}

/* Internal Function: Create entry based on specific group,
   and input interface in specific list */
static struct flowmgr_mdb_entry *mdb_create(struct hlist_head *head,
					    int iif,
					    const __be32 *group)
{
	struct flowmgr_mdb_entry *mdb;

	mdb = kmem_cache_alloc(flowmgr_mdb_cache, GFP_ATOMIC);
	if (mdb) {
		memset(mdb, 0, sizeof(struct flowmgr_mdb_entry));
		memcpy(mdb->group, group, sizeof(mdb->group));
		mdb->iif = iif;
		atomic_set(&mdb->use, 1);
		mdb->created = jiffies;
		hlist_add_head_rcu(&mdb->hlist, head);
	}
	return mdb;
}

static void mdb_clear_mc2uc_by_mac(struct flowmgr_mdb_entry *mdb,
				   const __u8 *mc2uc)
{
	int i;
	for (i = 0; i <  mdb->num_mc2uc; i++) {

		if (memcmp(mdb->mc2uc[i].addr, mc2uc, 6))
			continue;

		if (i == (mdb->num_mc2uc - 1)) {
			mdb->num_mc2uc--;
			break;
		} else {
			memcpy(&mdb->mc2uc[i],
			       &mdb->mc2uc[i+1],
			       (NUM_MC2UC-1-i)*sizeof(struct flowmgr_mc2uc));
			mdb->num_mc2uc--;
			break;
		}
	}
}

static int mdb_num_mc2uc_by_port(struct flowmgr_mdb_entry *mdb,
				  const __u8 oif)
{
	int i, num = 0;
	for (i = 0; i <  mdb->num_mc2uc; i++) {
		if (mdb->mc2uc[i].oif == oif)
			num++;
	}
	return num;
}

static void mdb_clear_mc2uc_by_port(struct flowmgr_mdb_entry *mdb,
				    const __u8 oif)
{
	int i;
	for (i = 0; i <  mdb->num_mc2uc; i++) {

		if (mdb->mc2uc[i].oif != oif)
			continue;

		if (i == (mdb->num_mc2uc - 1)) {
			mdb->num_mc2uc--;
			break;
		} else {
			memcpy(&mdb->mc2uc[i],
			       &mdb->mc2uc[i+1],
			       (NUM_MC2UC-1-i)*sizeof(struct flowmgr_mc2uc));
			mdb->num_mc2uc--;
			break;
		}
	}
}

/* Internal Function: Delete entry based on specific group,
   output, input interface and mac address */
static int mdb_delete_by_group(const __u8 iif,
			       const __u8 oif,
			       const __be32 *group,
			       const __u8 *mc2uc)
{
	struct hlist_head *head = &hash[flowmgr_mdb_hash(iif)];
	struct flowmgr_mdb_entry *mdb;

	mdb = mdb_find(head, group);
	if (!mdb)
		return -ENOENT;

	if (mdb->oif[oif])
		mdb->oif[oif]--;

	if (!mc2uc)
		goto skip_m2uc;

	mdb_clear_mc2uc_by_mac(mdb, mc2uc);

skip_m2uc:
	mdb->created = jiffies;
	mdb_delete(mdb, oif);
	return 0;
}

/* Delete entry based on specific group,
   output, input interface and mac address */
int flowmgr_mdb_delete(const struct net_device *in,
		       const struct net_device *out,
		       const __be32 *group,
		       const __u8 *mc2uc)
{
	int err;

	if (out->ifindex >= NUM_MCTX)
		return -1;
	spin_lock_bh(&hash_lock);
	err = mdb_delete_by_group(in->ifindex, out->ifindex, group, mc2uc);
	spin_unlock_bh(&hash_lock);
	return err;
}

/* Internal Function: Insert entry based on specific group,
   output, input interface and mac address */
static int mdb_insert(const __u8 iif,
		      const __u8 oif,
		      const __be32 *group,
		      const __u8 *mc2uc)
{
	struct hlist_head *head = &hash[flowmgr_mdb_hash(iif)];
	struct flowmgr_mdb_entry *mdb;

	mdb = mdb_find(head, group);
	if (mdb)
		atomic_inc(&mdb->use);
	else
		mdb = mdb_create(head, iif, group);
	if (!mdb)
		return -ENOMEM;

	if (!mc2uc) {
		/* No MAC address, so no mc2uc */
		mdb->oif[oif]++;
		goto done;
	}

	if (mdb->oif[oif]) {
		/* oif already part of replcation mask, so no mc2uc */
		mdb->oif[oif]++;
		goto done;
	}

	if (mdb->num_mc2uc < NUM_MC2UC) {
		/* Space in mc2uc, add */
		memcpy(mdb->mc2uc[mdb->num_mc2uc].addr,
		       mc2uc, 6);
		mdb->mc2uc[mdb->num_mc2uc].oif = oif;
		mdb->num_mc2uc++;
	} else {
		/* mc2uc table full, remove entries with oif */
		int i, num;
		num = mdb_num_mc2uc_by_port(mdb, oif);
		for (i = 0; i < num; i++)
			mdb_clear_mc2uc_by_port(mdb, oif);
		mdb->oif[oif] += (num + 1);
	}

done:
	mdb->created = jiffies;

	/* Promote flow */
	flowmgr_promote_mcast(mdb, oif);

	return 0;
}

/* Insert entry based on specific group,
   output, input interface and mac address */
int flowmgr_mdb_insert(const struct net_device *in,
		       const struct net_device *out,
		       const __be32 *group,
		       const __u8 *mc2uc)
{
	int ret;
	if (out->ifindex >= NUM_MCTX)
		return -1;
	spin_lock_bh(&hash_lock);
	ret = mdb_insert(in->ifindex, out->ifindex, group, mc2uc);
	spin_unlock_bh(&hash_lock);
	return ret;
}

/* Delete entry based on output interface */
void flowmgr_mdb_delete_by_out_port(const struct net_device *out)
{
	int i, j;

	spin_lock_bh(&hash_lock);
	for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
		struct hlist_node *h, *g;

		hlist_for_each_safe(h, g, &hash[i]) {
			struct flowmgr_mdb_entry *f;
			f = hlist_entry(h, struct flowmgr_mdb_entry, hlist);
			for (j = 0; j < NUM_MCTX; j++) {
				if (f->oif[j] && (j == out->ifindex))
					mdb_delete_by_group(f->iif,
							    j,
							    f->group,
							    NULL);
			}
		}
	}
	spin_unlock_bh(&hash_lock);
}

/* Get first entry in specific list */
struct hlist_node  *flowmgr_mdb_get_first(int *hashid)
{
	struct hlist_node *h;
	int i;

	for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
		h = rcu_dereference(hlist_first_rcu(&hash[i]));
		if (h) {
			*hashid = i;
			return h;
		}
	}
	return NULL;
}

/* Get next entry in specific list */
struct hlist_node *flowmgr_mdb_get_next(int *hashid,
				       struct hlist_node *head)
{
	head = rcu_dereference(hlist_next_rcu(head));
	while (!head) {
		if (++*hashid >= FLOWMGR_HASH_SIZE)
			return NULL;
		head = rcu_dereference(
				hlist_first_rcu(
				   &hash[*hashid]));
	}
	return head;
}

/* Get index in specific list */
struct hlist_node *flowmgr_mdb_get_idx(int *hashid, loff_t pos)
{
	struct hlist_node *head = flowmgr_mdb_get_first(hashid);

	if (head)
		while (pos && (head = flowmgr_mdb_get_next(hashid, head)))
			pos--;
	return pos ? NULL : head;
}
