 /****************************************************************************
 *
 * Copyright (c) 2021 Broadcom. All rights reserved
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Jayesh Patel <jayeshp@broadcom.com>
 ****************************************************************************/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/netdevice.h>
#include <proc_cmd.h>
#include "macpt.h"

#define MACPT_PROC_DIR_NAME	"driver/macpt"

static int cmd_add(int argc, char *argv[])
{
	if (argc == 2) {
		__u8 mac[6];
		if (mac_pton(argv[1], mac)) {
			macpt_db_add(NULL, NULL, NULL, mac, 0, 0, NULL);
			goto done;
		}
	}
/* help */
	pr_info("%s <mac>\n", argv[0]);
done:
	return 0;
}
static int cmd_del(int argc, char *argv[])
{
	if (argc == 2) {
		__u8 mac[6];
		if (mac_pton(argv[1], mac)) {
			macpt_db_del(NULL, NULL, NULL, mac, 0);
			goto done;
		}
	}
/* help */
	pr_info("%s <mac>\n", argv[0]);
done:
	return 0;
}
static int cmd_reload(int argc, char *argv[])
{
    /* register macpt hook to newly added interfaces if any */
    macpt_register_nethook();
    return 0;
}
static struct proc_cmd_ops command_entries[] = {
	{ .name = "add", .do_command = cmd_add},
	{ .name = "del", .do_command = cmd_del},
	{ .name = "update", .do_command = cmd_reload},
};

static struct proc_cmd_table command_table = {
	.module_name = "MACPT",
	.size = ARRAY_SIZE(command_entries),
	.ops = command_entries
};

static void *macpt_db_seq_start(struct seq_file *seq, loff_t *pos)
	__acquires(RCU)
{
	rcu_read_lock();
	seq->private = 0;
	return macpt_db_get_idx((int *)&seq->private, *pos);
}

static void *macpt_db_seq_next(struct seq_file *seq, void *v,
				      loff_t *pos)
{
	(*pos)++;
	return macpt_db_get_next((int *)&seq->private, v);
}

static void macpt_db_seq_stop(struct seq_file *seq, void *v)
	__releases(RCU)
{
	rcu_read_unlock();
}

static int macpt_db_seq_show(struct seq_file *seq, void *v)
{
	struct macpt_db_entry *db = v;

	if (!v)
		return -1;

	seq_printf(seq, " %pM @ interface=%s\n",
		   db->addr, db->dev ? db->dev->name : "Unknown");

	return 0;

}

static const struct seq_operations macpt_db_seq_ops = {
	.start	= macpt_db_seq_start,
	.next	= macpt_db_seq_next,
	.stop	= macpt_db_seq_stop,
	.show	= macpt_db_seq_show,
};

static int macpt_db_seq_open(struct inode *inode, struct file *file)
{
	int ret = seq_open(file, &macpt_db_seq_ops);
	return ret;
};

static const struct file_operations macpt_proc_db_fops = {
	.owner    = THIS_MODULE,
	.open     = macpt_db_seq_open,
	.read     = seq_read,
	.llseek   = seq_lseek,
	.release  = seq_release,
};

static struct
proc_dir_entry *macpt_proc_dir;

void macpt_procfs_init(void)
{
	macpt_proc_dir = proc_mkdir(MACPT_PROC_DIR_NAME, NULL);
	if (macpt_proc_dir == NULL) {
		pr_warn("MACPT Warning: cannot create /proc/%s\n",
			MACPT_PROC_DIR_NAME);
		return;
	}
	proc_create_cmd("mac", macpt_proc_dir, &command_table);
	proc_create_data("db", S_IRUGO, macpt_proc_dir,
			 &macpt_proc_db_fops, NULL);
}

void macpt_procfs_exit(void)
{
	if (macpt_proc_dir) {
		remove_proc_entry("db", macpt_proc_dir);
		remove_proc_entry("mac", macpt_proc_dir);
		remove_proc_entry(MACPT_PROC_DIR_NAME, NULL);
		macpt_proc_dir = NULL;
	}
}
