 /****************************************************************************
 *
 * Copyright (c) 2017 Broadcom Corporation
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Broadcom BCM8488x Multi-Gigabit Ethernet external transceivers.
 *
 * Author: Ravi Patel <ravi.patel@broadcom.com>
 ****************************************************************************/

#include <linux/module.h>
#include <linux/phy.h>
#include <linux/delay.h>
#include <linux/brcmphy.h>
#include <linux/firmware.h>
#include <linux/hwmon.h>
#include <linux/hwmon-sysfs.h>
#include <linux/netdevice.h>
#include "bcm8488x.h"

/* Speed register. */
#define MDIO_SPEED_5G	0x4000	/* 5G capable */
#define MDIO_SPEED_2P5G	0x2000	/* 2.5G capable */

/* Device present registers. */
#define MDIO_MMD_MII	0
#define MDIO_DEVS_MII	MDIO_DEVS_PRESENT(MDIO_MMD_MII)

/* AN 10GBASE-T control register. */
#define MDIO_AN_10GBT_CTRL_ADV5G	0x0100	/* Advertise 5GBASE-T */
#define MDIO_AN_10GBT_CTRL_ADV2P5G	0x0080	/* Advertise 2.5GBASE-T */
#define MDIO_AN_10GBT_CTRL_ADV		(MDIO_AN_10GBT_CTRL_ADV10G | \
									MDIO_AN_10GBT_CTRL_ADV5G | \
									MDIO_AN_10GBT_CTRL_ADV2P5G)

/* AN 10GBASE-T status register. */
#define MDIO_AN_10GBT_STAT_LP5G		0x0040	/* LP is 5GBT capable */
#define MDIO_AN_10GBT_STAT_LP2P5G	0x0020	/* LP is 2.5GBT capable */

/* one write for all phy */
static unsigned short bcm8488x_halt_seq0[] = {
	/* regnum, value for DEVAD 30 */
	0x418c, 0x0000,
	0x4188, 0x48f0,
	0x4181, 0x017c,
	0x4186, 0x8000,
	0x4181, 0x0040,	/* write till read back 0x0040 */
	0x4186, 0x8000,
};

/* all write for one phy */
static unsigned short bcm5499x_halt_seq0[] = {
	/* regnum, value for DEVAD 30 */
	0x4110, 0x0001,
	0x418c, 0x0000,
	0x4188, 0x48f0,
};

/* one write for all phy */
static unsigned short bcm8488x_halt_seq1[] = {
	/* ADDR_L, ADDR_H, DATA_L, DATA_H and then PROCTL_WRITE */
	0x0000, 0xc300, 0x0010, 0x0000,
};

/* all write for one phy */
static unsigned short bcm5499x_halt_seq1[] = {
	/* ADDR_L, ADDR_H, DATA_L, DATA_H and then PROCTL_WRITE */
	0x3000, 0xf000, 0x0121, 0x0000,
};

/* one write for all phy */
static unsigned short bcm8488x_halt_seq2[] = {
	/* devad, regnum, value */
	MDIO_MMD_VEND1, 0x4181, 0x0000,
};

/* one write for all phy */
static unsigned short bcm5499x_halt_seq2[] = {
	/* devad, regnum, value */
	MDIO_MMD_VEND1,  0x80a6, 0x0000,
	MDIO_MMD_PMAPMD, 0xa010, 0x0000,
	MDIO_MMD_PMAPMD, MDIO_CTRL1, BMCR_RESET,
	/* wait 1 ms */
	MDIO_MMD_VEND1,  0x4110, 0x0001,
};

/* one write for all phy */
static unsigned short bcm8488x_bcast_seq[] = {
	/* regnum, value for DEVAD 30 */
	0x4117, 0xf003,
	0x4107, 0x0401,
};

/* all write for one phy */
static unsigned short bcm5499x_bcast_seq[] = {
	/* regnum, value for DEVAD 30 */
	0x4107, 0x0401,
	0x4117, 0xf001,
};

static unsigned short bcm8488x_start_seq[] = {
	/* devad, regnum, value */
	MDIO_MMD_PMAPMD, 0xa008, 0x0000,
	/* one write for all phy */
	MDIO_MMD_VEND1,  0x8004, 0x5555,
	MDIO_MMD_PMAPMD, 0xa003, 0x0000,
	MDIO_MMD_PMAPMD, MDIO_CTRL1, BMCR_RESET,
};

/* all write for one phy */
static unsigned short bcm5499x_start_seq[] = {
	/* devad, regnum, value */
	MDIO_MMD_PMAPMD, ADDR_H, 0xf000,
	MDIO_MMD_PMAPMD, ADDR_L, 0x3000,
	MDIO_MMD_PMAPMD, DATA_H, 0x0000,
	MDIO_MMD_PMAPMD, DATA_L, 0x0020,
	MDIO_MMD_PMAPMD, PROCTL, PROCTL_WRITE,
	/* wait 2 ms */
};

static unsigned short bcm8488x_super_i_seq[] = {
	/* regnum, mask, wr-value, regnum, mask, rd-value for DEVAD 30 */
	BCM8488X_PINSTRAP, BCM8488X_SUPER_I, 0x0000,
	REQ1_STATUS, DONT_CHANGE_STRAP, 0x0000,
};

static unsigned short bcm5499x_super_i_seq[] = {
	/* regnum, mask, wr-value, regnum, mask, rd-value for DEVAD 30 */
	BCM5499X_PINSTRAP, BCM5499X_SUPER_I, 0x0000,
	REQ1_STATUS, DONT_CHANGE_STRAP, 0x0000,
};

static unsigned short *bcm8488x_led_type = NULL;
static unsigned short *bcm8488x_led_user_seq = NULL;
static unsigned int bcm8488x_led_user_seqlen = 0;

static unsigned short bcm5499x_led_type[] = {
	/* MDIO_CMD */ /* DATA REG BITMAP */ /* MDIO_DATA1 ... MDIO_DATA5 */
	MDIO_CMD_SET_LED_TYPE, MDIO_CMD_MSK_LED_TYPE, MDIO_CMD_VAL_LED_FW,
};

static unsigned short bcm84891_led_user_seq[] = {
	/* regnum, value for DEVAD 1 */
	0xA82C, 0x0080,
	0xA8EF, 0x0004,
	0xA82F, 0x0006,
	0xA8F0, 0x0000,
	0xA832, 0x0000,
	0xA8F1, 0x000C,
	0xA838, 0x0008,
	0xA8F3, 0x0010,
	0xA83B, 0xA48A,
	0xA8EC, 0x83FF,
	0xA82B, 0x00FF,
	0xA82A, 0xFFFF,
};

static unsigned short bcm8488x_pair_swap[] = {
	/* MDIO_CMD */ /* DATA REG BITMAP */ /* MDIO_DATA1 ... MDIO_DATA5 */
	MDIO_CMD_SET_PAIR_SWAP, MDIO_CMD_MSK_PAIR_SWAP, MDIO_CMD_VAL_PAIR_SWAP,
};

static unsigned short bcm8488x_usxgmii_none[] = {
	/* MDIO_CMD */ /* DATA REG BITMAP */ /* MDIO_DATA1 ... MDIO_DATA5 */
	MDIO_CMD_SET_USXGMII, MDIO_CMD_MSK_USXGMII, 0, MDIO_CMD_VAL_USXGMII_AN, MDIO_CMD_VAL_USXGMII_10, MDIO_CMD_VAL_USXGMII_MP,
};

static unsigned short bcm8488x_usxgmii_10[] = {
	/* MDIO_CMD */ /* DATA REG BITMAP */ /* MDIO_DATA1 ... MDIO_DATA5 */
	MDIO_CMD_SET_USXGMII, MDIO_CMD_MSK_USXGMII, MDIO_CMD_VAL_USXGMII_EN, MDIO_CMD_VAL_USXGMII_AN, MDIO_CMD_VAL_USXGMII_10, MDIO_CMD_VAL_USXGMII_MP,
};

static struct bcm8488x_core bcm8488x_cores[] = {
	[BCM8488X_CORE] = {
		.devad_id = BCM8488X_DEVAD,
		.seqlen[BCM8488X_HALT_SEQ0] = ARRAY_SIZE(bcm8488x_halt_seq0),
		.seqlen[BCM8488X_HALT_SEQ1] = ARRAY_SIZE(bcm8488x_halt_seq1),
		.seqlen[BCM8488X_HALT_SEQ2] = ARRAY_SIZE(bcm8488x_halt_seq2),
		.seqlen[BCM8488X_BCAST_SEQ] = ARRAY_SIZE(bcm8488x_bcast_seq),
		.seqlen[BCM8488X_START_SEQ] = ARRAY_SIZE(bcm8488x_start_seq),
		.seqlen[BCM8488X_SUPER_I_SEQ] = ARRAY_SIZE(bcm8488x_super_i_seq),
		.seq[BCM8488X_HALT_SEQ0]    = bcm8488x_halt_seq0,
		.seq[BCM8488X_HALT_SEQ1]    = bcm8488x_halt_seq1,
		.seq[BCM8488X_HALT_SEQ2]    = bcm8488x_halt_seq2,
		.seq[BCM8488X_BCAST_SEQ]    = bcm8488x_bcast_seq,
		.seq[BCM8488X_START_SEQ]    = bcm8488x_start_seq,
		.seq[BCM8488X_SUPER_I_SEQ]  = bcm8488x_super_i_seq,
	},
	[BCM54991E_CORE] = {
		.devad_id = BCM5499X_DEVAD,
		.seqlen[BCM8488X_HALT_SEQ0] = ARRAY_SIZE(bcm5499x_halt_seq0),
		.seqlen[BCM8488X_HALT_SEQ1] = ARRAY_SIZE(bcm5499x_halt_seq1),
		.seqlen[BCM8488X_HALT_SEQ2] = ARRAY_SIZE(bcm5499x_halt_seq2),
		.seqlen[BCM8488X_BCAST_SEQ] = ARRAY_SIZE(bcm5499x_bcast_seq),
		.seqlen[BCM8488X_START_SEQ] = ARRAY_SIZE(bcm5499x_start_seq),
		.seqlen[BCM8488X_SUPER_I_SEQ] = ARRAY_SIZE(bcm8488x_super_i_seq),
		.seq[BCM8488X_HALT_SEQ0]    = bcm5499x_halt_seq0,
		.seq[BCM8488X_HALT_SEQ1]    = bcm5499x_halt_seq1,
		.seq[BCM8488X_HALT_SEQ2]    = bcm5499x_halt_seq2,
		.seq[BCM8488X_BCAST_SEQ]    = bcm5499x_bcast_seq,
		.seq[BCM8488X_START_SEQ]    = bcm5499x_start_seq,
		.seq[BCM8488X_SUPER_I_SEQ]  = bcm8488x_super_i_seq,
	},
	[BCM5499X_CORE] = {
		.devad_id = BCM5499X_DEVAD,
		.seqlen[BCM8488X_HALT_SEQ0] = ARRAY_SIZE(bcm5499x_halt_seq0),
		.seqlen[BCM8488X_HALT_SEQ1] = ARRAY_SIZE(bcm5499x_halt_seq1),
		.seqlen[BCM8488X_HALT_SEQ2] = ARRAY_SIZE(bcm5499x_halt_seq2),
		.seqlen[BCM8488X_BCAST_SEQ] = ARRAY_SIZE(bcm5499x_bcast_seq),
		.seqlen[BCM8488X_START_SEQ] = ARRAY_SIZE(bcm5499x_start_seq),
		.seqlen[BCM8488X_SUPER_I_SEQ] = ARRAY_SIZE(bcm5499x_super_i_seq),
		.seq[BCM8488X_HALT_SEQ0]    = bcm5499x_halt_seq0,
		.seq[BCM8488X_HALT_SEQ1]    = bcm5499x_halt_seq1,
		.seq[BCM8488X_HALT_SEQ2]    = bcm5499x_halt_seq2,
		.seq[BCM8488X_BCAST_SEQ]    = bcm5499x_bcast_seq,
		.seq[BCM8488X_START_SEQ]    = bcm5499x_start_seq,
		.seq[BCM8488X_SUPER_I_SEQ]  = bcm5499x_super_i_seq,
	},
};

static struct bcm8488x_prop bcm8488x_props[] = {
	{
		.phy_id  = PHY_ID_BCM8488X_OUI | BCM84880_A0,
		.lane_max = 1,
		.model   = "BCM84880 A0",
		.fw_name = BCM8488_A0_FILE,
		.core = &bcm8488x_cores[BCM8488X_CORE],
		.max_speed = SPEED_10000,
	},
	{
		.phy_id  = PHY_ID_BCM8488X_OUI | BCM84880_B0,
		.lane_max = 1,
		.model   = "BCM84880 B0",
		.fw_name = BCM8488_B0_FILE,
		.core = &bcm8488x_cores[BCM8488X_CORE],
		.max_speed = SPEED_10000,
	},
	{
		.phy_id  = PHY_ID_BCM8488X_OUI | BCM84886_B0,
		.lane_max = 2,
		.model   = "BCM84886 B0",
		.fw_name = BCM8488_B0_FILE,
		.core = &bcm8488x_cores[BCM8488X_CORE],
		.max_speed = SPEED_10000,
	},
	{
		.phy_id  = PHY_ID_BCM5499X_OUI | BCM54991E_A0,
		.lane_max = 1,
		.model   = "BCM54991E A0",
		.fw_name = BCM5499_A0_FILE,
		.core = &bcm8488x_cores[BCM54991E_CORE],
		.max_speed = SPEED_2500,
	},
	{
		.phy_id  = PHY_ID_BCM5499X_OUI | BCM54991EL_A0,
		.lane_max = 1,
		.model   = "BCM54991EL A0",
		.fw_name = BCM5499_A0_FILE,
		.core = &bcm8488x_cores[BCM5499X_CORE],
		.max_speed = SPEED_2500,
	},
	{
		.phy_id  = PHY_ID_BCM5499X_OUI | BCM84891_A0,
		.lane_max = 1,
		.model	 = "BCM84891 A0",
		.fw_name = BCM5499_A0_FILE,
		.core = &bcm8488x_cores[BCM5499X_CORE],
		.max_speed = SPEED_10000,
	},
	{
		.phy_id  = PHY_ID_BCM5499X_OUI | BCM54991EL_B0,
		.lane_max = 1,
		.model   = "BCM54991EL B0",
		.fw_name = BLACKFIN_B0_FILE,
		.core = &bcm8488x_cores[BCM5499X_CORE],
		.max_speed = SPEED_2500,
	},
	{
		.phy_id  = PHY_ID_BCM5499X_OUI | BCM54991ELM_A0,
		.lane_max = 1,
		.model	 = "BCM54991ELM A0",
		.fw_name = LONGFIN_FILE,
		.core = &bcm8488x_cores[BCM5499X_CORE],
		.max_speed = SPEED_2500,
	},
	{
		.phy_id  = PHY_ID_BCM5499X_OUI | BCM84892_B0,
		.lane_max = 2,
		.model	 = "BCM84892 B0",
		.fw_name = BLACKFIN_B0_FILE,
		.core = &bcm8488x_cores[BCM5499X_CORE],
		.max_speed = SPEED_10000,
	},
	{
		.phy_id  = PHY_ID_BCM5499X_OUI | BCM50991EL_B0,
		.lane_max = 1,
		.model	 = "BCM50991EL B0",
		.fw_name = BLACKFIN_B0_FILE,
		.core = &bcm8488x_cores[BCM5499X_CORE],
		.max_speed = SPEED_2500,
	},
	{
		.phy_id  = PHY_ID_BCM5499X_OUI | BCM50991ELM_B0,
		.lane_max = 1,
		.model	 = "BCM50991ELM B0",
		.fw_name = LONGFIN_FILE,
		.core = &bcm8488x_cores[BCM5499X_CORE],
		.max_speed = SPEED_2500,
	},
	{
		.phy_id  = PHY_ID_BCM5499X_OUI | BCM84891L_B0,
		.lane_max = 1,
		.model	 = "BCM84891L B0",
		.fw_name = BLACKFIN_B0_FILE,
		.core = &bcm8488x_cores[BCM5499X_CORE],
		.max_speed = SPEED_10000,
	},
};

static void bcm8488x_processor_halt(struct phy_device *phydev)
{
	struct device *dev = &phydev->mdio.dev;
	struct brcmphy_xcvr *phyxcvr = dev_get_drvdata(dev);
	struct bcm8488x_lane *lane = phyxcvr->phypriv;
	struct bcm8488x_prop *prop = lane->prop;
	struct bcm8488x_phys *phys = lane->phys;
	unsigned int seqlen;
	unsigned short *seq;
	int i, l, mask, ret;

	/* load halt seq0 in DEVAD 30 */
	seqlen = prop->core->seqlen[BCM8488X_HALT_SEQ0];
	seq = prop->core->seq[BCM8488X_HALT_SEQ0];
	ret = 0;

	/* all write for one phy */
	for (l = 0; l < prop->lane_max; l++) {
		mask = 1 << l;
		if (!(mask & phys->lane_map))
			continue;

		for (i = 0; i < seqlen; i+=2)
			ret |= phy_write_mmd(phys->phydev[l], MDIO_MMD_VEND1,
					seq[i], seq[i + 1]);
	}

	if (ret)
		dev_err(dev, "Error %d loading halt seq0\n", ret);

	/* load halt seq1 in DEVAD 1 */
	seqlen = prop->core->seqlen[BCM8488X_HALT_SEQ1];
	seq = prop->core->seq[BCM8488X_HALT_SEQ1];
	ret = 0;

	/* all write for one phy */
	for (l = 0; l < prop->lane_max; l++) {
		mask = 1 << l;
		if (!(mask & phys->lane_map))
			continue;

		for (i = 0; i < seqlen; i+=4) {
			ret |= phy_write_mmd(phys->phydev[l], MDIO_MMD_PMAPMD,
					ADDR_H, seq[i + 1]);
			ret |= phy_write_mmd(phys->phydev[l], MDIO_MMD_PMAPMD,
					ADDR_L, seq[i]);
			ret |= phy_write_mmd(phys->phydev[l], MDIO_MMD_PMAPMD,
					DATA_H, seq[i + 3]);
			ret |= phy_write_mmd(phys->phydev[l], MDIO_MMD_PMAPMD,
					DATA_L, seq[i + 2]);
			ret |= phy_write_mmd(phys->phydev[l], MDIO_MMD_PMAPMD,
					PROCTL, PROCTL_WRITE);
		}
	}

	if (ret)
		dev_err(dev, "Error %d loading halt seq1\n", ret);

	/* load halt seq2 */
	seqlen = prop->core->seqlen[BCM8488X_HALT_SEQ2];
	seq = prop->core->seq[BCM8488X_HALT_SEQ2];
	ret = 0;

	/* one write for all phy */
	for (i = 0; i < seqlen; i+=3) {
		for (l = 0; l < prop->lane_max; l++) {
			mask = 1 << l;
			if (!(mask & phys->lane_map))
				continue;

			ret |= phy_write_mmd(phys->phydev[l], seq[i],
					seq[i + 1], seq[i + 2]);

			/* Reset issued then wait 1 ms */
			if ((seq[i + 1] == MDIO_CTRL1) && (seq[i + 2] == BMCR_RESET))
				mdelay(1);
		}
	}

	if (ret)
		dev_err(dev, "Error %d loading halt seq2\n", ret);
}

static void bcm8488x_broadcast_mode(struct bcm8488x_lane *lane,
		u32 addr, bool enable)
{
	struct device *edev = NULL;
	struct bcm8488x_prop *prop = lane->prop;
	struct bcm8488x_phys *phys = lane->phys;
	unsigned short *seq, val;
	unsigned int seqlen;
	int i, l, mask, ret;

	/* load broadcast seq */
	seq = prop->core->seq[BCM8488X_BCAST_SEQ];
	seqlen = prop->core->seqlen[BCM8488X_BCAST_SEQ];
	ret = 0;

	/* all write for one phy */
	for (l = 0; l < prop->lane_max; l++) {
		mask = 1 << l;
		if (mask == lane->mask || !(mask & phys->lane_map))
			continue;

		for (i = 0; i < seqlen; i+=2) {
			val = seq[i + 1] * enable;
			if (val == 0x0401)
				val |= (addr << 5);

			ret |= phy_write_mmd(phys->phydev[l], MDIO_MMD_VEND1,
					seq[i], val);
			if (ret)
				edev = &phys->phydev[l]->mdio.dev;
		}
	}

	if (ret)
		dev_err(edev, "Error %d loading broadcast %s seq\n", ret, enable ? "enable" : "disable");
}

static int bcm8488x_verify_loadcrc(struct phy_device *phydev)
{
	struct device *dev = &phydev->mdio.dev;
	int ret, timeout;

	timeout = MDIO_RETRY;
	do {
		msleep(MDIO_DELAY);
		ret = phy_read_mmd(phydev, MDIO_MMD_PMAPMD, MDIO_CTRL1);
		if (ret < 0) {
			dev_err(dev, "Error verifying firmware is loaded\n");
			return ret;
		}

	} while (ret != MDIO_CTRL1_SPEEDSELEXT && --timeout);

	if (ret != MDIO_CTRL1_SPEEDSELEXT) {
		dev_err(dev, "Timeout verifying firmware is loaded\n");
		return -ETIMEDOUT;
	}

	ret = phy_read_mmd(phydev, MDIO_MMD_VEND1, STATUS);
	if (ret < 0) {
		dev_err(dev, "Error verifying firmware crc\n");
		return ret;
	}

	if ((ret & GOOD_CRC) != GOOD_CRC) {
		dev_err(dev, "Firmware loaded has bad crc\n");
		return -EIO;
	}

	dev_info(dev, "PHY Firmware is loaded with Good CRC.\n");

	ret = phy_read_mmd(phydev, MDIO_MMD_VEND1, FW_REV);
	if (ret < 0) {
		dev_err(dev, "Error reading firmware ver\n");
		return ret;
	}

	dev_info(dev, "PHY Firmware Version(Main.Branch.Build): "
			"%d.%02d.%02d\n",
			(ret & FW_MAIN_VER_MASK) >> FW_MAIN_VER_SHIFT,
			(ret & FW_BRANCH_VER_MASK),
			(ret & FW_BUILD_VER_MASK) >> FW_BUILD_VER_SHIFT);

	ret = phy_read_mmd(phydev, MDIO_MMD_VEND1, FW_DATE);
	if (ret < 0) {
		dev_err(dev, "\nError reading firmware date\n");
		return ret;
	}

	dev_info(dev, "PHY Firmware Date(MM/DD/YYYY): "
			"%02d/%02d/%04d\n",
			(ret & FW_MONTH_MASK) >> FW_MONTH_SHIFT,
			(ret & FW_DAY_MASK) >> FW_DAY_SHIFT,
			((ret & FW_YEAR4_MASK) >> (FW_YEAR4_SHIFT - FW_DAY_SHIFT)) +
			(ret & FW_YEAR_MASK) + 2000);

	return 0;
}

static int bcm8488x_processor_start(struct phy_device *phydev)
{
	struct device *dev = &phydev->mdio.dev;
	struct brcmphy_xcvr *phyxcvr = dev_get_drvdata(dev);
	struct bcm8488x_lane *lane = phyxcvr->phypriv;
	struct bcm8488x_prop *prop = lane->prop;
	struct bcm8488x_phys *phys = lane->phys;
	unsigned int seqlen;
	unsigned short *seq;
	int ret, i, l, mask;
	const struct firmware *fw;
	const u8 *fw_data;
	unsigned short data_low, data_high;
	unsigned int fw_cksum = 0;

	/* Update phydev to first phy */
	for (l = 0; l < prop->lane_max; l++) {
		mask = 1 << l;
		if (mask & phys->lane_map) {
			phydev = phys->phydev[l];
			phyxcvr = dev_get_drvdata(&phydev->mdio.dev);
			lane = phyxcvr->phypriv;
			break;
		}
	}

	/* Request BCM8488x firmware from user space */
	ret = request_firmware(&fw, prop->fw_name, dev);
	if (ret) {
		dev_err(dev, "Request firmware %s failed\n",
			prop->fw_name);
		return ret;
	}

	/* Enable broadcast mode */
	if (prop->lane_max > 1)
		bcm8488x_broadcast_mode(lane, phydev->mdio.addr, 1);

	/* load firmware into BCM8488x device OCM */
	ret = phy_write_mmd(phydev, MDIO_MMD_PMAPMD, ADDR_H, 0);
	if (ret) {
		dev_err(dev, "Error %d clearing ADDR_H reg\n", ret);
		goto _release_firmware;
	}

	ret = phy_write_mmd(phydev, MDIO_MMD_PMAPMD, ADDR_L, 0);
	if (ret) {
		dev_err(dev, "Error %d clearing ADDR_L reg\n", ret);
		goto _release_firmware;
	}

	ret = phy_write_mmd(phydev, MDIO_MMD_PMAPMD, PROCTL, PROCTL_DOWNLOAD);
	if (ret) {
		dev_err(dev, "Error %d writing PROCTL_DOWNLOAD cmd\n", ret);
		goto _release_firmware;
	}

	fw_data = fw->data;
	for (i = 0; i < fw->size; i+=4) {
		data_low = (fw_data[i + 1] << 8) | fw_data[i];
		data_high = (fw_data[i + 3] << 8) | fw_data[i + 2];
		fw_cksum += ((data_high << 16) | data_low);
		ret = phy_write_mmd(phydev, MDIO_MMD_PMAPMD, DATA_H, data_high);
		if (ret) {
			dev_err(dev, "Error %d writing DATA_H reg\n", ret);
			break;
		}

		ret = phy_write_mmd(phydev, MDIO_MMD_PMAPMD, DATA_L, data_low);
		if (ret) {
			dev_err(dev, "Error %d writing DATA_L reg\n", ret);
			break;
		}
	}

_release_firmware:
	release_firmware(fw);
	if (ret)
		return ret;

	dev_info(dev, "PHY Firmware Size %d:0x%x bytes\n", i, i);
	dev_info(dev, "PHY Firmware Simple Checksum: 0x%08x\n",
		fw_cksum);
	ret = phy_write_mmd(phydev, MDIO_MMD_PMAPMD, PROCTL, PROCTL_CLEAR);
	if (ret) {
		dev_err(dev, "Error %d writing PROCTL_CLEAR cmd\n", ret);
		return ret;
	}

	/* Disable broadcast mode */
	if (prop->lane_max > 1)
		bcm8488x_broadcast_mode(lane, phydev->mdio.addr, 0);

	/* load start seq */
	seqlen = prop->core->seqlen[BCM8488X_START_SEQ];
	seq = prop->core->seq[BCM8488X_START_SEQ];

	/* all write for one phy */
	for (l = 0; l < prop->lane_max; l++) {
		mask = 1 << l;
		if (!(mask & phys->lane_map))
			continue;

		for (i = 0; i < seqlen; i+=3)
			ret |= phy_write_mmd(phys->phydev[l], seq[i],
					seq[i + 1], seq[i + 2]);
	}

	if (ret) {
		dev_err(dev, "Error %d loading start seq\n", ret);
		return ret;
	}

	mdelay(2);

	for (l = 0; l < prop->lane_max; l++) {
		mask = 1 << l;
		if (!(mask & phys->lane_map))
			continue;

		ret = bcm8488x_verify_loadcrc(phys->phydev[l]);
	}

	return ret;
}

static void bcm8488x_bootstrap(struct phy_device *phydev, bool off)
{
	struct device *dev = &phydev->mdio.dev;
	struct brcmphy_xcvr *phyxcvr = dev_get_drvdata(dev);
	struct bcm8488x_lane *lane = phyxcvr->phypriv;
	struct bcm8488x_prop *prop = lane->prop;
	unsigned short *seq = prop->core->seq[BCM8488X_SUPER_I_SEQ];
	int ret;

	/* load super_i/bootstrap seq */
	ret = phy_read_mmd(phydev, MDIO_MMD_VEND1, seq[0]);
	if ((ret < 0) || ((ret & seq[1]) == seq[2 - off]))
		return;

	ret = (ret & ~seq[1]) | seq[2 - off];
	phy_write_mmd(phydev, MDIO_MMD_VEND1, seq[0], ret);
	if (!seq[4])
		return;

	do {
		ret = phy_read_mmd(phydev, MDIO_MMD_VEND1, seq[3]);
		if (ret < 0)
			break;

	} while ((ret & seq[4]) != seq[5]);
}

static int bcm8488x_mdio_cmdset(struct phy_device *phydev, unsigned short *seq)
{
	int i, j = 0, ret = 0, timeout = 0;
	unsigned short cmd, mask;

	if (!seq)
		return 0;

	/* Check if PHY available to take command */
	timeout = MDIO_CMD_TIMEOUT;
	do {
		ret = phy_read_mmd(phydev, MDIO_MMD_VEND1, MDIO_STATUS);
		if (ret < 0)
			goto _ret_func;

		mdelay(100);
	} while (--timeout && (ret == MDIO_CMD_IN_PROGRESS || ret == MDIO_CMD_SYSTEM_BUSY));

	if (ret == MDIO_CMD_IN_PROGRESS || ret == MDIO_CMD_SYSTEM_BUSY)
		return -1;

	cmd = seq[j];
	mask = seq[++j];
	switch(cmd) {
		case MDIO_CMD_SET_PAIR_SWAP:
		case MDIO_CMD_SET_LED_TYPE:
		case MDIO_CMD_SET_USXGMII:
			for (i = 0; i < 5; i++) {
				if (!(mask & BIT(i)))
					continue;

				phy_write_mmd(phydev, MDIO_MMD_VEND1, MDIO_DATA1 + i, seq[++j]);
			}

			phy_write_mmd(phydev, MDIO_MMD_VEND1, MDIO_CMD, cmd);
			break;
		case MDIO_CMD_GET_USXGMII:
		case MDIO_CMD_GET_CURR_TEMP:
			phy_write_mmd(phydev, MDIO_MMD_VEND1, MDIO_CMD, cmd);
			for (i = 0; i < 5; i++) {
				if (!(mask & BIT(i)))
					continue;

				ret = phy_read_mmd(phydev, MDIO_MMD_VEND1, MDIO_DATA1 + i);
				if (ret < 0)
					goto _ret_func;

				seq[++j] = ret;
			}

			break;
	}

	/* Check if PHY completed processing command */
	timeout = MDIO_CMD_TIMEOUT;
	do {
		ret = phy_read_mmd(phydev, MDIO_MMD_VEND1, MDIO_STATUS);
		if (ret < 0)
			goto _ret_func;

		mdelay(100);
	} while (--timeout && ret != MDIO_CMD_COMPLETE_PASS && ret != MDIO_CMD_COMPLETE_ERROR);

	ret = (ret == MDIO_CMD_COMPLETE_PASS) ? 0 : -1;
	if ((ret == 0) && (cmd == MDIO_CMD_SET_LED_TYPE) && bcm8488x_led_user_seq)
		for (i = 0; i < bcm8488x_led_user_seqlen; i+=2)
			phy_write_mmd(phydev, MDIO_MMD_PMAPMD, bcm8488x_led_user_seq[i], bcm8488x_led_user_seq[i + 1]);

_ret_func:
	return ret;
}

static ssize_t bcm8488x_get_temp(struct device *dev,
		struct device_attribute *devattr, char *buf)
{
	struct phy_device *phydev = dev_get_drvdata(dev);
	unsigned short seq[3];
	seq[0] = MDIO_CMD_GET_CURR_TEMP;
	seq[1] = MDIO_CMD_MSK_CURR_TEMP;
	bcm8488x_mdio_cmdset(phydev, seq);
	return sprintf(buf, "%u\n", seq[2]);
}

static SENSOR_DEVICE_ATTR(temp1_input, 0444, bcm8488x_get_temp, NULL, 0);
static struct attribute *bcm8488x_attrs[] = {
	&sensor_dev_attr_temp1_input.dev_attr.attr,
	NULL
};
ATTRIBUTE_GROUPS(bcm8488x);

static void bcm8488x_hwmon_open(struct phy_device *phydev,
		struct bcm8488x_lane *lane)
{
	struct device *dev = &phydev->mdio.dev;
	if (lane->hwmon_dev)
		return;
	lane->hwmon_dev = hwmon_device_register_with_groups(dev, phydev->attached_dev->name,
							phydev, bcm8488x_groups);
	if (IS_ERR(lane->hwmon_dev)) {
		lane->hwmon_dev = NULL;
		dev_err(dev, "Unable to register hwmon device\n");
	}
}

static void bcm8488x_hwmon_close(struct bcm8488x_lane *lane)
{
	if (lane->hwmon_dev) {
		hwmon_device_unregister(lane->hwmon_dev);
		lane->hwmon_dev = NULL;
	}
}

static int bcm8488x_get_features(struct phy_device *phydev)
{
	__ETHTOOL_DECLARE_LINK_MODE_MASK(advertising);
	struct device *dev = &phydev->mdio.dev;
	struct brcmphy_xcvr *phyxcvr = dev_get_drvdata(dev);
	unsigned long *supported = phydev->supported;

	linkmode_zero(supported);
	linkmode_set_bit(ETHTOOL_LINK_MODE_Pause_BIT, supported);
	linkmode_set_bit(ETHTOOL_LINK_MODE_Asym_Pause_BIT, supported);
	linkmode_set_bit(ETHTOOL_LINK_MODE_FIBRE_BIT, supported);
	linkmode_set_bit(ETHTOOL_LINK_MODE_TP_BIT, supported);
	linkmode_set_bit(ETHTOOL_LINK_MODE_MII_BIT, supported);
	linkmode_set_bit(ETHTOOL_LINK_MODE_Autoneg_BIT, supported);
	linkmode_set_bit(ETHTOOL_LINK_MODE_10000baseT_Full_BIT, supported);
	linkmode_set_bit(ETHTOOL_LINK_MODE_5000baseT_Full_BIT, supported);
	linkmode_set_bit(ETHTOOL_LINK_MODE_2500baseT_Full_BIT, supported);
	linkmode_set_bit(ETHTOOL_LINK_MODE_1000baseT_Full_BIT, supported);
	linkmode_set_bit(ETHTOOL_LINK_MODE_1000baseT_Half_BIT, supported);
	linkmode_set_bit(ETHTOOL_LINK_MODE_100baseT_Full_BIT, supported);
	linkmode_set_bit(ETHTOOL_LINK_MODE_100baseT_Half_BIT, supported);

	linkmode_copy(advertising, phydev->advertising);
	linkmode_copy(phydev->advertising, supported);
	phy_set_max_speed(phydev, phyxcvr->speed);
	linkmode_copy(phydev->advertising, advertising);

	return 0;
}

static int bcm8488x_init_supported(struct phy_device *phydev)
{
	__ETHTOOL_DECLARE_LINK_MODE_MASK(supported);
	struct device *dev = &phydev->mdio.dev;
	struct brcmphy_xcvr *phyxcvr = dev_get_drvdata(dev);
	struct brcmphy_core *phycore = phyxcvr->phycore;
	int ret, ext;

	linkmode_zero(supported);
	linkmode_set_bit(ETHTOOL_LINK_MODE_Pause_BIT, supported);
	linkmode_set_bit(ETHTOOL_LINK_MODE_Asym_Pause_BIT, supported);
	ret = phy_read_mmd(phydev, MDIO_MMD_VEND1, STATUS);
	if (ret < 0) {
		dev_err(dev, "Error %d reading vendor1 status\n", ret);
		return ret;
	}

	if (ret & LINESIDE_XFI_PHY)
		linkmode_set_bit(ETHTOOL_LINK_MODE_FIBRE_BIT, supported);
	else
		linkmode_set_bit(ETHTOOL_LINK_MODE_TP_BIT, supported);

	ret = phy_read_mmd(phydev, MDIO_MMD_PMAPMD, MDIO_DEVS1);
	if (ret < 0) {
		dev_err(dev, "Error %d reading devices in package\n", ret);
		return ret;
	}

	if (ret & MDIO_DEVS_MII)
		linkmode_set_bit(ETHTOOL_LINK_MODE_MII_BIT, supported);

	ret = phy_read_mmd(phydev, MDIO_MMD_AN, MDIO_STAT1);
	if (ret < 0) {
		dev_err(dev, "Error %d reading basic mode status\n", ret);
		return ret;
	}

	if (ret & MDIO_AN_STAT1_ABLE)
		linkmode_set_bit(ETHTOOL_LINK_MODE_Autoneg_BIT, supported);

	ret = phy_read_mmd(phydev, MDIO_MMD_PMAPMD, MDIO_SPEED);
	if (ret < 0) {
		dev_err(dev, "Error %d reading speed ability\n", ret);
		return ret;
	}

	if (ret & MDIO_SPEED_10G)
		linkmode_set_bit(ETHTOOL_LINK_MODE_10000baseT_Full_BIT, supported);

	if (ret & MDIO_SPEED_5G)
		linkmode_set_bit(ETHTOOL_LINK_MODE_5000baseT_Full_BIT, supported);

	if (ret & MDIO_SPEED_2P5G)
		linkmode_set_bit(ETHTOOL_LINK_MODE_2500baseT_Full_BIT, supported);

	ret = phy_read_mmd(phydev, MDIO_MMD_AN, MDIO_MII_BMSR);
	if (ret < 0) {
		dev_err(dev, "Error %d reading basic mode status\n", ret);
		return ret;
	}

	if (ret & BMSR_ANEGCAPABLE)
		linkmode_set_bit(ETHTOOL_LINK_MODE_Autoneg_BIT, supported);

	if (ret & BMSR_ESTATEN) {
		ext = phy_read_mmd(phydev, MDIO_MMD_AN, MDIO_MII_ESTATUS);
		if (ext < 0) {
			dev_err(dev, "Error %d reading extended status\n", ext);
			return ext;
		}

		if (ext & ESTATUS_1000_TFULL)
			linkmode_set_bit(ETHTOOL_LINK_MODE_1000baseT_Full_BIT, supported);

		if (ext & ESTATUS_1000_THALF)
			linkmode_set_bit(ETHTOOL_LINK_MODE_1000baseT_Half_BIT, supported);
	}

	if (ret & BMSR_100FULL)
		linkmode_set_bit(ETHTOOL_LINK_MODE_100baseT_Full_BIT, supported);

	if (ret & BMSR_100HALF)
		linkmode_set_bit(ETHTOOL_LINK_MODE_100baseT_Half_BIT, supported);

	if (phycore) {
		if (phycore->speed && (phyxcvr->speed > phycore->speed))
			phyxcvr->speed = phycore->speed;

		if (phycore->duplex && (phyxcvr->duplex > phycore->duplex))
			phyxcvr->duplex = phycore->duplex;
	}

	bcm8488x_get_features(phydev);

	linkmode_and(phydev->supported, phydev->supported, supported);

	if (phycore && !linkmode_empty(phycore->core->supported))
		linkmode_and(phydev->supported, phydev->supported, phycore->core->supported);

	return 0;
}

static int bcm8488x_restart_aneg(struct phy_device *phydev)
{
	struct device *dev = &phydev->mdio.dev;
	unsigned long *advertising = phydev->advertising;
	int ret = -1;

	if (linkmode_test_bit(ETHTOOL_LINK_MODE_10000baseT_Full_BIT, advertising) ||
			linkmode_test_bit(ETHTOOL_LINK_MODE_5000baseT_Full_BIT, advertising) ||
			linkmode_test_bit(ETHTOOL_LINK_MODE_2500baseT_Full_BIT, advertising)) {
		ret = phy_read_mmd(phydev, MDIO_MMD_AN, MDIO_CTRL1);
		if (ret < 0) {
			dev_err(dev, "Error %d reading AN basic mode control\n", ret);
			return ret;
		}

		ret |= (BMCR_ANENABLE | BMCR_ANRESTART);
		ret = phy_write_mmd(phydev, MDIO_MMD_AN, MDIO_CTRL1, ret);
		if (ret < 0) {
			dev_err(dev, "Error %d writing AN basic mode control\n", ret);
			return ret;
		}

	} else if (linkmode_test_bit(ETHTOOL_LINK_MODE_1000baseT_Half_BIT, advertising) ||
			linkmode_test_bit(ETHTOOL_LINK_MODE_1000baseT_Full_BIT, advertising) ||
			linkmode_test_bit(ETHTOOL_LINK_MODE_100baseT_Half_BIT, advertising) ||
			linkmode_test_bit(ETHTOOL_LINK_MODE_100baseT_Full_BIT, advertising)) {
		ret = phy_read_mmd(phydev, MDIO_MMD_AN, MDIO_MII_BMCR);
		if (ret < 0) {
			dev_err(dev, "Error %d reading basic mode control\n", ret);
			return ret;
		}

		ret |= (BMCR_ANENABLE | BMCR_ANRESTART);
		ret = phy_write_mmd(phydev, MDIO_MMD_AN, MDIO_MII_BMCR, ret);
		if (ret < 0) {
			dev_err(dev, "Error %d writing basic mode control\n", ret);
			return ret;
		}
	}

	return ret;
}

static int bcm8488x_config_advert(struct phy_device *phydev)
{
	struct device *dev = &phydev->mdio.dev;
	unsigned long *advertising = phydev->advertising;
	int ret, adv, changed = 0;

	if (linkmode_empty(advertising))
		linkmode_copy(advertising, phydev->supported);
	else
		linkmode_and(advertising, advertising, phydev->supported);

	ret = phy_read_mmd(phydev, MDIO_MMD_AN, MDIO_AN_10GBT_CTRL);
	if (ret < 0) {
		dev_err(dev, "Error %d reading 10GBASE-T auto-negotiation control\n", ret);
		return ret;
	}

	adv = ret;
	if (linkmode_test_bit(ETHTOOL_LINK_MODE_10000baseT_Full_BIT, advertising))
		adv |= (MDIO_AN_10GBT_CTRL_ADV10G);
	else
		adv &= ~(MDIO_AN_10GBT_CTRL_ADV10G);

	if (linkmode_test_bit(ETHTOOL_LINK_MODE_5000baseT_Full_BIT, advertising))
		adv |= (MDIO_AN_10GBT_CTRL_ADV5G);
	else
		adv &= ~(MDIO_AN_10GBT_CTRL_ADV5G);

	if (linkmode_test_bit(ETHTOOL_LINK_MODE_2500baseT_Full_BIT, advertising))
		adv |= (MDIO_AN_10GBT_CTRL_ADV2P5G);
	else
		adv &= ~(MDIO_AN_10GBT_CTRL_ADV2P5G);

	if (ret != adv) {
		ret = phy_write_mmd(phydev, MDIO_MMD_AN, MDIO_AN_10GBT_CTRL, adv);
		if (ret < 0) {
			dev_err(dev, "Error %d writing 10GBASE-T auto-negotiation control\n", ret);
			return ret;
		}

		changed = 1;
	}

	ret = phy_read_mmd(phydev, MDIO_MMD_AN, MDIO_MII_CTRL1000);
	if (ret < 0) {
		dev_err(dev, "Error %d reading 1000BASE-T control\n", ret);
		return ret;
	}

	adv = ret;
	if (linkmode_test_bit(ETHTOOL_LINK_MODE_1000baseT_Full_BIT, advertising))
		adv |= (ADVERTISE_1000FULL);
	else
		adv &= ~(ADVERTISE_1000FULL);

	if (linkmode_test_bit(ETHTOOL_LINK_MODE_1000baseT_Half_BIT, advertising))
		adv |= (ADVERTISE_1000HALF);
	else
		adv &= ~(ADVERTISE_1000HALF);

	if (ret != adv) {
		ret = phy_write_mmd(phydev, MDIO_MMD_AN, MDIO_MII_CTRL1000, adv);
		if (ret < 0) {
			dev_err(dev, "Error %d writing 1000BASE-T control\n", ret);
			return ret;
		}

		changed = 1;
	}

	ret = phy_read_mmd(phydev, MDIO_MMD_AN, MDIO_MII_ADVERTISE);
	if (ret < 0) {
		dev_err(dev, "Error %d reading Advertisement control\n", ret);
		return ret;
	}

	adv = ret;
	if (linkmode_test_bit(ETHTOOL_LINK_MODE_Asym_Pause_BIT, advertising))
		adv |= (ADVERTISE_PAUSE_ASYM);
	else
		adv &= ~(ADVERTISE_PAUSE_ASYM);

	if (linkmode_test_bit(ETHTOOL_LINK_MODE_Pause_BIT, advertising))
		adv |= (ADVERTISE_PAUSE_CAP);
	else
		adv &= ~(ADVERTISE_PAUSE_CAP);

	if (linkmode_test_bit(ETHTOOL_LINK_MODE_100baseT_Full_BIT, advertising))
		adv |= (ADVERTISE_100FULL);
	else
		adv &= ~(ADVERTISE_100FULL);

	if (linkmode_test_bit(ETHTOOL_LINK_MODE_100baseT_Half_BIT, advertising))
		adv |= (ADVERTISE_100HALF);
	else
		adv &= ~(ADVERTISE_100HALF);

	if (ret != adv) {
		ret = phy_write_mmd(phydev, MDIO_MMD_AN, MDIO_MII_ADVERTISE, adv);
		if (ret < 0) {
			dev_err(dev, "Error %d writing Advertisement control\n", ret);
			return ret;
		}

		changed = 1;
	}

	return changed;
}

static int bcm8488x_sw_init(struct phy_device *phydev)
{
	struct device *dev = &phydev->mdio.dev;
	struct brcmphy_xcvr *phyxcvr = dev_get_drvdata(dev);
	struct bcm8488x_lane *lane = phyxcvr->phypriv;
	struct bcm8488x_prop *prop = lane->prop;
	int ret = 0;

	dev_info(dev, "%s: sw initializing\n", prop->model);

	ret = bcm8488x_mdio_cmdset(phydev, bcm8488x_led_type);
	if (ret < 0)
		goto _ret_func;

	if (lane->pair_swap) {
		ret = bcm8488x_mdio_cmdset(phydev, bcm8488x_pair_swap);
		if (ret < 0)
			goto _ret_func;
	}

	bcm8488x_bootstrap(phydev, false);
	ret = bcm8488x_mdio_cmdset(phydev, bcm8488x_led_type);
	if (ret < 0)
		goto _ret_func;

	ret = bcm8488x_init_supported(phydev);
	if (ret < 0)
		goto _ret_func;

	ret = bcm8488x_config_advert(phydev);
	if (ret < 0)
		goto _ret_func;

	bcm8488x_hwmon_open(phydev, lane);
	dev_info(dev, "%s: sw initialized\n", prop->model);

	return 0;

_ret_func:
	dev_err(dev, "%s: sw initialization error\n", prop->model);

	return ret;
}

static int bcm8488x_fw_init(struct phy_device *phydev)
{
	struct device *dev = &phydev->mdio.dev;
	struct brcmphy_xcvr *phyxcvr = dev_get_drvdata(dev);
	struct bcm8488x_lane *lane = phyxcvr->phypriv;
	struct bcm8488x_prop *prop = lane->prop;
	int ret = 0, retry = FW_RETRY;

	dev_info(dev, "%s: fw initializing\n", prop->model);

_retry_load:
	bcm8488x_processor_halt(phydev);
	ret = bcm8488x_processor_start(phydev);
	if ((ret == -ETIMEDOUT || ret == -EIO) && retry) {
		dev_info(dev, "%s: fw re-initializing\n", prop->model);
		--retry;
		goto _retry_load;
	}

	if (ret < 0) {
		dev_err(dev, "%s: fw initialization error\n", prop->model);
		return ret;
	}

	dev_info(dev, "%s: fw initialized\n", prop->model);

	return bcm8488x_sw_init(phydev);
}

static int bcm8488x_pmapmd_reset(struct phy_device *phydev)
{
	int ret;

	/* Check PHY reset bit */
	ret = phy_write_mmd(phydev, MDIO_MMD_VEND1, 0x4110, 0x0001);
	if (ret < 0)
		return ret;

	ret = phy_read_mmd(phydev, MDIO_MMD_PMAPMD, MDIO_CTRL1);
	if (ret < 0)
		return ret;

	return (ret & MDIO_CTRL1_RESET);
}

static int bcm8488x_config_init(struct phy_device *phydev)
{
	struct device *dev = &phydev->mdio.dev;
	struct brcmphy_xcvr *phyxcvr = dev_get_drvdata(dev);
	struct bcm8488x_lane *lane = phyxcvr->phypriv;
	struct bcm8488x_phys *phys = lane->phys;

	mutex_lock(&phys->lock);
	if (bcm8488x_pmapmd_reset(phydev))
		bcm8488x_fw_init(phydev);
	else
		bcm8488x_sw_init(phydev);
	mutex_unlock(&phys->lock);

	return 0;
}

static int bcm8488x_config_forced(struct phy_device *phydev)
{
	struct device *dev = &phydev->mdio.dev;
	int ctl = 0;

	if (phydev->speed == SPEED_1000)
		ctl |= BMCR_SPEED1000;
	else if (phydev->speed == SPEED_100)
		ctl |= BMCR_SPEED100;
	else {
		dev_err(dev, "Unsupported %d Mbps forced link speed\n",
				phydev->speed);
		return -EINVAL;
	}

	if (phydev->duplex == DUPLEX_FULL)
		ctl |= BMCR_FULLDPLX;

	phydev->pause = 0;
	phydev->asym_pause = 0;

	return phy_write_mmd(phydev, MDIO_MMD_AN, MDIO_MII_BMCR, ctl);
}

static int bcm8488x_config_aneg(struct phy_device *phydev)
{
	int result;

	if (AUTONEG_ENABLE != phydev->autoneg)
		return bcm8488x_config_forced(phydev);

	result = bcm8488x_config_advert(phydev);
	if (result < 0) /* error */
		return result;
	if (result == 0) {
		/* Advertisement hasn't changed, but maybe aneg was never on to
		 * begin with?  Or maybe phy was isolated?
		 */
		int ctl = phy_read_mmd(phydev, MDIO_MMD_AN, MDIO_MII_BMCR);

		if (ctl < 0)
			return ctl;

		if (!(ctl & BMCR_ANENABLE) || (ctl & BMCR_ISOLATE))
			result = 1; /* do restart aneg */
	}

	/* Only restart aneg if we are advertising something different
	 * than we were before.
	 */
	if (result > 0)
		result = bcm8488x_restart_aneg(phydev);

	return result;
}

static int bcm8488x_aneg_done(struct phy_device *phydev)
{
	struct device *dev = &phydev->mdio.dev;
	struct brcmphy_xcvr *phyxcvr = dev_get_drvdata(dev);
	struct bcm8488x_lane *lane = phyxcvr->phypriv;
	unsigned long *advertising = phydev->advertising;
	int ret = -1;

	if (unlikely(lane->fixed_link))
		return BMSR_ANEGCOMPLETE;

	if (linkmode_test_bit(ETHTOOL_LINK_MODE_10000baseT_Full_BIT, advertising) ||
			linkmode_test_bit(ETHTOOL_LINK_MODE_5000baseT_Full_BIT, advertising) ||
			linkmode_test_bit(ETHTOOL_LINK_MODE_2500baseT_Full_BIT, advertising)) {
		ret = phy_read_mmd(phydev, MDIO_MMD_AN, MDIO_STAT1);
		if (ret < 0) {
			dev_err(dev, "Error %d reading AN basic mode status\n", ret);
			return ret;
		}

		ret &= BMSR_ANEGCOMPLETE;
	} else if (linkmode_test_bit(ETHTOOL_LINK_MODE_1000baseT_Half_BIT, advertising) ||
			linkmode_test_bit(ETHTOOL_LINK_MODE_1000baseT_Full_BIT, advertising) ||
			linkmode_test_bit(ETHTOOL_LINK_MODE_100baseT_Half_BIT, advertising) ||
			linkmode_test_bit(ETHTOOL_LINK_MODE_100baseT_Full_BIT, advertising)) {
		ret = phy_read_mmd(phydev, MDIO_MMD_AN, MDIO_MII_BMSR);
		if (ret < 0) {
			dev_err(dev, "Error %d reading MII basic mode control\n", ret);
			return ret;
		}

		ret &= BMSR_ANEGCOMPLETE;
	}

	return ret;
}

static int bcm8488x_usxgmii_update(struct phy_device *phydev, int speed)
{
	int update = 0;
	unsigned short seq[4];

	seq[0] = MDIO_CMD_GET_USXGMII;
	seq[1] = MDIO_CMD_MSK_USXGMII_G;
	bcm8488x_mdio_cmdset(phydev, seq);

	switch (speed) {
	case SPEED_10000:
	case SPEED_5000:
	case SPEED_2500:
		if (seq[2] != MDIO_CMD_VAL_USXGMII_EN || seq[3] != MDIO_CMD_VAL_USXGMII_10) {
			update = 1;
			bcm8488x_mdio_cmdset(phydev, bcm8488x_usxgmii_10);
		}

		break;
	default:
		if (seq[2] == MDIO_CMD_VAL_USXGMII_EN) {
			update = 1;
			bcm8488x_mdio_cmdset(phydev, bcm8488x_usxgmii_none);
		}
	}

	if (update) {
		bcm8488x_bootstrap(phydev, BMCR_PDOWN);
		mdelay(100);
		bcm8488x_bootstrap(phydev, 0);
	}

	return update;
}

static int bcm8488x_update_link(struct phy_device *phydev)
{
	struct device *dev = &phydev->mdio.dev;
	struct brcmphy_xcvr *phyxcvr = dev_get_drvdata(dev);
	struct bcm8488x_lane *lane = phyxcvr->phypriv;

	if (likely(!lane->fixed_link)) {
		/* Read status */
		lane->status = phy_read_mmd(phydev, MDIO_MMD_VEND1, STATUS);
		if (lane->status < 0)
			return lane->status;

		goto _update_link;
	}

	lane->status = GOOD_CRC | MAC_LINK | COPPER_LINK | COPPER_DETECTED;
	switch (lane->fixed_link) {
		case SPEED_10000:
			lane->status |= COPPER_SPEED_10G;
			break;
		case SPEED_5000:
			lane->status |= COPPER_SPEED_5G;
			break;
		case SPEED_2500:
			lane->status |= COPPER_SPEED_2P5G;
			break;
		case SPEED_1000:
			lane->status |= COPPER_SPEED_1G;
			break;
		case SPEED_100:
			lane->status |= COPPER_SPEED_100M;
			break;
	}

_update_link:
	switch (lane->status & COPPER_SPEED_MASK) {
		case COPPER_SPEED_10G:
			lane->speed = SPEED_10000;
			break;
		case COPPER_SPEED_5G:
			lane->speed = SPEED_5000;
			break;
		case COPPER_SPEED_2P5G:
			lane->speed = SPEED_2500;
			break;
		case COPPER_SPEED_1G:
			lane->speed = SPEED_1000;
			break;
		case COPPER_SPEED_100M:
			lane->speed = SPEED_100;
			break;
		default:
			lane->speed = SPEED_UNKNOWN;
	}

	if ((lane->status & COPPER_LINK) && (lane->speed != SPEED_UNKNOWN)) {
		if (phydev->link == 0) {
			if (phydev->interface == PHY_INTERFACE_MODE_USXGMII &&
					bcm8488x_usxgmii_update(phydev, lane->speed))
				return 0;

			dev_info(dev, "Link up status 0x%04X\n", lane->status);
		}

		phydev->link = 1;
	} else {
		if (phydev->link == 1)
			dev_info(dev, "Link down status 0x%04X\n", lane->status);

		phydev->link = 0;
	}

	return 0;
}

static int bcm8488x_read_status(struct phy_device *phydev)
{
	struct device *dev = &phydev->mdio.dev;
	struct brcmphy_xcvr *phyxcvr = dev_get_drvdata(dev);
	struct brcmphy_core *phycore = phyxcvr->phycore;
	struct bcm8488x_lane *lane = phyxcvr->phypriv;
	unsigned long *supported = phydev->supported;
	unsigned long *lp_advertising = phydev->lp_advertising;
	int speed[2];
	int duplex[2];
	int adv;
	int lpa;
	int common_adv_mb;
	int common_adv_gb = 0;
	int common_adv_mgb = 0;

	speed[0] = phydev->speed * phydev->link;
	duplex[0] = phydev->duplex * phydev->link;

	/* Update the link, but return if there was an error */
	lpa = bcm8488x_update_link(phydev);
	if (lpa)
		return lpa;

	linkmode_zero(lp_advertising);

	if (AUTONEG_ENABLE == phydev->autoneg) {
		if (linkmode_test_bit(ETHTOOL_LINK_MODE_10000baseT_Full_BIT, supported) ||
				linkmode_test_bit(ETHTOOL_LINK_MODE_5000baseT_Full_BIT, supported) ||
				linkmode_test_bit(ETHTOOL_LINK_MODE_2500baseT_Full_BIT, supported)) {
			if (likely(!lane->fixed_link))
				lpa = phy_read_mmd(phydev, MDIO_MMD_AN, MDIO_AN_10GBT_STAT);
			else
				lpa = MDIO_AN_10GBT_STAT_LP10G | MDIO_AN_10GBT_STAT_LP5G | MDIO_AN_10GBT_STAT_LP2P5G;

			if (lpa < 0)
				return lpa;

			adv = phy_read_mmd(phydev, MDIO_MMD_AN, MDIO_AN_10GBT_CTRL);
			if (adv < 0)
				return adv;

			adv &= (MDIO_AN_10GBT_CTRL_ADV);
			if (adv & MDIO_AN_10GBT_CTRL_ADV10G)
				adv = (adv & ~MDIO_AN_10GBT_CTRL_ADV10G) |
					(MDIO_AN_10GBT_CTRL_ADV10G << 1);

			common_adv_mgb = lpa & adv >> 2;
			if (lpa & MDIO_AN_10GBT_STAT_LP10G)
				linkmode_set_bit(ETHTOOL_LINK_MODE_10000baseT_Full_BIT, lp_advertising);

			if (lpa & MDIO_AN_10GBT_STAT_LP5G)
				linkmode_set_bit(ETHTOOL_LINK_MODE_5000baseT_Full_BIT, lp_advertising);

			if (lpa & MDIO_AN_10GBT_STAT_LP2P5G)
				linkmode_set_bit(ETHTOOL_LINK_MODE_2500baseT_Full_BIT, lp_advertising);
		}

		if (linkmode_test_bit(ETHTOOL_LINK_MODE_1000baseT_Full_BIT, supported) ||
				linkmode_test_bit(ETHTOOL_LINK_MODE_1000baseT_Half_BIT, supported)) {
			if (likely(!lane->fixed_link))
				lpa = phy_read_mmd(phydev, MDIO_MMD_AN, MDIO_MII_STAT1000);
			else
				lpa = LPA_1000FULL | LPA_1000HALF;

			if (lpa < 0)
				return lpa;

			adv = phy_read_mmd(phydev, MDIO_MMD_AN, MDIO_MII_CTRL1000);
			if (adv < 0)
				return adv;

			common_adv_gb = lpa & adv << 2;
			mii_stat1000_mod_linkmode_lpa_t(lp_advertising, lpa);
		}

		if (likely(!lane->fixed_link)) {
			lpa = LPA_LPACK | ADVERTISE_PAUSE_ASYM | ADVERTISE_PAUSE_CAP;
			lpa |= (ADVERTISE_100FULL | ADVERTISE_100HALF | ADVERTISE_CSMA);
		} else
			lpa = phy_read_mmd(phydev, MDIO_MMD_AN, MDIO_MII_LPA);

		if (lpa < 0)
			return lpa;

		adv = phy_read_mmd(phydev, MDIO_MMD_AN, MDIO_MII_ADVERTISE);
		if (adv < 0)
			return adv;

		common_adv_mb = lpa & adv;
		mii_lpa_mod_linkmode_lpa_t(lp_advertising, lpa);

		phydev->pause = 0;
		phydev->asym_pause = 0;

		if ((common_adv_mgb & MDIO_AN_10GBT_STAT_LP10G) &&
				(lane->speed == SPEED_10000)) {
			phydev->speed = SPEED_10000;
			phydev->duplex = DUPLEX_FULL;

		} else if ((common_adv_mgb & MDIO_AN_10GBT_STAT_LP5G) &&
				(lane->speed == SPEED_5000)) {
			phydev->speed = SPEED_5000;
			phydev->duplex = DUPLEX_FULL;

		} else if ((common_adv_mgb & MDIO_AN_10GBT_STAT_LP2P5G) &&
				(lane->speed == SPEED_2500)) {
			phydev->speed = SPEED_2500;
			phydev->duplex = DUPLEX_FULL;

		} else if ((common_adv_gb & (LPA_1000FULL | LPA_1000HALF)) &&
				(lane->speed == SPEED_1000)) {
			phydev->speed = SPEED_1000;
			if (common_adv_gb & LPA_1000FULL)
				phydev->duplex = DUPLEX_FULL;
			else if (common_adv_gb & LPA_1000HALF)
				phydev->duplex = DUPLEX_HALF;

		} else if ((common_adv_mb & (LPA_100FULL | LPA_100HALF)) &&
				(lane->speed == SPEED_100)) {
			phydev->speed = SPEED_100;
			if (common_adv_mb & LPA_100FULL)
				phydev->duplex = DUPLEX_FULL;
			else if (common_adv_mb & LPA_100HALF)
				phydev->duplex = DUPLEX_HALF;
		}

		if (phydev->duplex == DUPLEX_FULL) {
			phydev->pause = lpa & LPA_PAUSE_CAP ? 1 : 0;
			phydev->asym_pause = lpa & LPA_PAUSE_ASYM ? 1 : 0;
		}
	} else {
		int bmcr = phy_read_mmd(phydev, MDIO_MMD_AN, MDIO_MII_BMCR);

		if (bmcr < 0)
			return bmcr;

		if (bmcr & BMCR_FULLDPLX)
			phydev->duplex = DUPLEX_FULL;
		else
			phydev->duplex = DUPLEX_HALF;

		if (bmcr & BMCR_SPEED1000)
			phydev->speed = SPEED_1000;
		else
			phydev->speed = SPEED_100;

		phydev->pause = 0;
		phydev->asym_pause = 0;
	}

	if (phycore && phycore->media_speed) {
		speed[1] = phydev->speed * phydev->link;
		duplex[1] = phydev->duplex * phydev->link;
		if ((speed[0] != speed[1]) || (duplex[0] != duplex[1])) {
			if (linkmode_test_bit(ETHTOOL_LINK_MODE_FIBRE_BIT, supported))
				lpa = PORT_FIBRE;
			else
				lpa = PORT_TP;

			phycore->media_speed(phycore, lpa, speed[1], duplex[1]);
		}
	}

	return 0;
}

static struct bcm8488x_prop *bcm8488x_phy_id_match(u32 phy_id)
{
	struct bcm8488x_prop *prop = NULL;
	int i;

	for (i = 0; i < ARRAY_SIZE(bcm8488x_props); i++) {
		if (bcm8488x_props[i].phy_id == phy_id) {
			prop = &bcm8488x_props[i];
			break;
		}
	}

	return prop;
}

static int bcm8488x_match_phy_device(struct phy_device *phydev)
{
	struct device *dev = &phydev->mdio.dev;
	u32 phy_id = phydev->c45_ids.device_ids[BCM8488X_DEVAD];
	int match = 0;

	if (((phy_id ^ PHY_ID_BCM8488X_OUI) & PHY_ID_BCM8488X_MASK) == 0) {
		if (!bcm8488x_phy_id_match(phy_id))
			dev_info(dev, "phy_id 0x%08x un-supported\n", phy_id);
		else
			match = 1;
	}

	if (match)
		phydev->phy_id = phy_id;

	return match;
}

static int bcm5499x_match_phy_device(struct phy_device *phydev)
{
	struct device *dev = &phydev->mdio.dev;
	u32 phy_id = phydev->c45_ids.device_ids[BCM5499X_DEVAD];
	int match = 0;

	if (((phy_id ^ PHY_ID_BCM5499X_OUI) & PHY_ID_BCM5499X_MASK) == 0) {
		if (!bcm8488x_phy_id_match(phy_id))
			dev_info(dev, "phy_id 0x%08x un-supported\n", phy_id);
		else
			match = 1;
	}

	if (match) {
		phydev->phy_id = phy_id;
		bcm8488x_led_type = bcm5499x_led_type;
		if (phy_id == (PHY_ID_BCM5499X_OUI | BCM84891L_B0)) {
			bcm5499x_led_type[2] = MDIO_CMD_VAL_LED_USER;
			bcm8488x_led_user_seq = bcm84891_led_user_seq;
			bcm8488x_led_user_seqlen = ARRAY_SIZE(bcm84891_led_user_seq);
		}
	}

	return match;
}

static int bcm8488x_init_lane(struct phy_device *phydev,
		struct bcm8488x_lane *lane)
{
	struct bcm8488x_prop *prop = lane->prop;
	struct bcm8488x_phys *phys = NULL;
	struct brcmphy_xcvr *phyxcvr;
	struct phy_device *phy;
	u32 i, addr;

	addr = phydev->mdio.addr & ~(prop->lane_max - 1);
	lane->idx = phydev->mdio.addr - addr;
	lane->mask = 1 << lane->idx;
	for (i = addr; i < addr + prop->lane_max; i++) {
                phy = mdiobus_get_phy(phydev->mdio.bus, i);
                if (!phy)
			continue;

		phyxcvr = dev_get_drvdata(&phy->mdio.dev);
		if (!phyxcvr)
			continue;

		phys = ((struct bcm8488x_lane *)phyxcvr->phypriv)->phys;
		break;
	}

	if (!phys) {
		phys = kzalloc(sizeof(*phys), GFP_KERNEL);
		if (!phys)
			return -ENOMEM;

		mutex_init(&phys->lock);
	}

	phys->lane_map |= lane->mask;
	phys->lane_set |= lane->mask;
	phys->phydev[lane->idx] = phydev;
	lane->phys = phys;

	return 0;
}

static void bcm8488x_exit_lane(struct phy_device *phydev)
{
	struct device *dev = &phydev->mdio.dev;
	struct brcmphy_xcvr *phyxcvr = dev_get_drvdata(dev);
	struct bcm8488x_lane *lane = phyxcvr->phypriv;
	struct bcm8488x_prop *prop = lane->prop;
	struct bcm8488x_phys *phys = lane->phys;
	struct phy_device *phy;
	u32 i, addr;

	if (!phys)
		return;

	phys->lane_set &= ~lane->mask;
	if (phys->lane_set)
		return;

	bcm8488x_processor_halt(phydev);
	mutex_destroy(&phys->lock);
	bcm8488x_hwmon_close(lane);

	addr = phydev->mdio.addr & ~(prop->lane_max - 1);
	for (i = addr; i < addr + prop->lane_max; i++) {
                phy = mdiobus_get_phy(phydev->mdio.bus, i);
                if (!phy)
			continue;

		phyxcvr = dev_get_drvdata(&phy->mdio.dev);
		if (!phyxcvr)
			continue;

		((struct bcm8488x_lane *)phyxcvr->phypriv)->phys = NULL;
	}

	memset(phys, 0, sizeof(struct bcm8488x_phys));
	kfree(phys);
}

static void bcm8488x_pma_pcs_mii_update(struct phy_device *phydev,
		u16 reg, u16 mask, u16 val)
{
	u16 devad[3] = {MDIO_MMD_PMAPMD, MDIO_MMD_PCS, MDIO_MMD_AN};
	u16 regad[3] = {reg, reg, MDIO_MII_BASE | reg};
	int i, data;

	for (i = 0; i < 3; i++) {
		data = phy_read_mmd(phydev, devad[i], regad[i]);
		data = (data & ~mask) | val;
		phy_write_mmd(phydev, devad[i], regad[i], data);
	}

	bcm8488x_bootstrap(phydev, val);
	bcm8488x_mdio_cmdset(phydev, bcm8488x_led_type);
}

static int bcm8488x_suspend(struct phy_device *phydev)
{
	struct device *dev = &phydev->mdio.dev;
	struct brcmphy_xcvr *phyxcvr = dev_get_drvdata(dev);
	struct brcmphy_core *phycore = phyxcvr->phycore;

	mutex_lock(&phydev->lock);
	bcm8488x_pma_pcs_mii_update(phydev, MDIO_CTRL1, BMCR_PDOWN, BMCR_PDOWN);
	mutex_unlock(&phydev->lock);
	if (phycore && phycore->suspend)
		phycore->suspend(phycore);

	return 0;
}

int bcm8488x_resume(struct phy_device *phydev)
{
	struct device *dev = &phydev->mdio.dev;
	struct brcmphy_xcvr *phyxcvr = dev_get_drvdata(dev);
	struct brcmphy_core *phycore = phyxcvr->phycore;
	struct bcm8488x_lane *lane = phyxcvr->phypriv;
	struct bcm8488x_prop *prop = lane->prop;
	struct bcm8488x_phys *phys = lane->phys;
	u32 phy_id, data;

	if (phycore && phycore->resume)
		phycore->resume(phycore);

	/* Check PHY power by verifying PHYID read */
	data = phy_read_mmd(phydev, prop->core->devad_id, MDIO_DEVID1);
	phy_id = (data & 0xffff) << 16;
	data = phy_read_mmd(phydev, prop->core->devad_id, MDIO_DEVID2);
	phy_id |= (data & 0xffff);
	if (phy_id != prop->phy_id) {
		dev_info(dev, "powered down\n");

		return 0;
	}

	/*mutex_lock(&phydev->lock);*/
	mutex_lock(&phys->lock);
	/* Check PHY reset bit */
	if (bcm8488x_pmapmd_reset(phydev))
		bcm8488x_fw_init(phydev);
	else
		bcm8488x_pma_pcs_mii_update(phydev, MDIO_CTRL1, BMCR_PDOWN, 0);
	mutex_unlock(&phys->lock);
	/*mutex_unlock(&phydev->lock);*/

	return 0;
}

static int bcm8488x_probe(struct phy_device *phydev)
{
	struct device *dev = &phydev->mdio.dev;
	struct device_node *fixed_link_node;
	struct device_node *phan_node;
	struct brcmphy_xcvr *phyxcvr;
	struct bcm8488x_lane *lane;
	struct bcm8488x_prop *prop;
	int rc = -ENOMEM;

	prop = bcm8488x_phy_id_match(phydev->phy_id);
	if (!prop)
		return -ENODEV;

	phyxcvr = devm_kzalloc(dev, sizeof(*phyxcvr), GFP_KERNEL);
	if (!phyxcvr)
		return rc;

	lane = devm_kzalloc(dev, sizeof(*lane), GFP_KERNEL);
	if (!lane)
		return rc;

	lane->prop = prop;
	rc = bcm8488x_init_lane(phydev, lane);
	if (rc)
		return rc;

	phyxcvr->duplex = DUPLEX_FULL;
	phyxcvr->speed = 0;

	phan_node = of_parse_phandle(dev->of_node, "mac-handle", 0);
	if (phan_node) {
		if (!of_property_read_u32(phan_node, "max-speed", &phyxcvr->speed)) {
			switch (phyxcvr->speed) {
				case SPEED_10000:
				case SPEED_5000:
				case SPEED_2500:
				case SPEED_1000:
				case SPEED_100:
					break;
				default:
					phyxcvr->speed = 0;
			}
		}
		of_node_put(phan_node);
	}

	if (!phyxcvr->speed || phyxcvr->speed > prop->max_speed)
		phyxcvr->speed = prop->max_speed;

	lane->pair_swap = of_property_read_bool(dev->of_node,
			"enet-phy-lane-swap");
	fixed_link_node = of_get_child_by_name(dev->of_node, "fixed-link");
	if (fixed_link_node) {
		of_node_put(fixed_link_node);
		if (of_property_read_u32(fixed_link_node, "speed",
					&lane->fixed_link))
			return -EINVAL;
	}

	phyxcvr->phypriv = lane;
	dev_set_drvdata(dev, phyxcvr);
	dev_info(dev, "probed\n");

	return rc;
}

static void bcm8488x_remove(struct phy_device *phydev)
{
	struct device *dev = &phydev->mdio.dev;
	struct brcmphy_xcvr *phyxcvr = dev_get_drvdata(dev);

	if (!phyxcvr)
		return;

	if (dev->type->pm)
		dev->type->pm->suspend(dev);

	bcm8488x_exit_lane(phydev);
	memset(phyxcvr->phypriv, 0, sizeof(struct bcm8488x_lane));
	devm_kfree(dev, phyxcvr->phypriv);
	memset(phyxcvr, 0, sizeof(struct brcmphy_xcvr));
	devm_kfree(dev, phyxcvr);
	dev_set_drvdata(dev, NULL);
	dev_info(dev, "removed\n");
}

static void bcm8488x_shutdown(struct device *dev)
{
	struct phy_device *phydev = to_phy_device(dev);

	bcm8488x_remove(phydev);
	dev_info(dev, "shutdown\n");
}

#define BCM8488X_DRV(_oui, _mask, _name, _match)	\
{							\
	.phy_id			= (_oui),		\
	.name			= _name,		\
	.phy_id_mask		= (_mask),		\
	.soft_reset		= genphy_no_soft_reset,	\
	.get_features		= bcm8488x_get_features,\
	.config_init		= bcm8488x_config_init,	\
	.probe			= bcm8488x_probe,	\
	.suspend		= bcm8488x_suspend,	\
	.resume			= bcm8488x_resume,	\
	.config_aneg		= bcm8488x_config_aneg,	\
	.aneg_done		= bcm8488x_aneg_done,	\
	.update_link		= bcm8488x_update_link,	\
	.read_status		= bcm8488x_read_status,	\
	.remove			= bcm8488x_remove,	\
	.match_phy_device	= _match,		\
	.mdiodrv.driver	= {				\
		.owner = THIS_MODULE,			\
		.shutdown = bcm8488x_shutdown,		\
	},						\
}

static struct phy_driver bcm8488x_drivers[] = {
	BCM8488X_DRV(PHY_ID_BCM8488X_OUI, PHY_ID_BCM8488X_MASK,
		"Broadcom BCM8488X", bcm8488x_match_phy_device),
	BCM8488X_DRV(PHY_ID_BCM5499X_OUI, PHY_ID_BCM5499X_MASK,
		"Broadcom BCM5499X", bcm5499x_match_phy_device),
};

module_phy_driver(bcm8488x_drivers);

static struct mdio_device_id __maybe_unused bcm8488x_tbl[] = {
	{ PHY_ID_BCM8488X_OUI, PHY_ID_BCM8488X_MASK, },
	{ PHY_ID_BCM5499X_OUI, PHY_ID_BCM5499X_MASK, },
	{ }
};

MODULE_DEVICE_TABLE(mdio, bcm8488x_tbl);

MODULE_DESCRIPTION("Broadcom BCM8488x PHY driver");
MODULE_AUTHOR("Ravi Patel <ravi.patel@broadcom.com>");
MODULE_LICENSE("GPL");
