/*
 * Driver for BCM fixed regulator
 *
 * Copyright (C) 2020 Broadcom
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation version 2.
 *
 * This program is distributed "as is" WITHOUT ANY WARRANTY of any
 * kind, whether express or implied; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/err.h>
#include <linux/mutex.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/regulator/driver.h>
#include <linux/regulator/bcm-fixed.h>
#include <linux/delay.h>
#include <linux/gpio.h>
#include <linux/slab.h>
#include <linux/of.h>
#include <linux/of_gpio.h>
#include <linux/regulator/of_regulator.h>
#include <linux/regulator/machine.h>
#include <linux/gpio/consumer.h>

struct bcm_fixed_data {
	struct regulator_desc desc;
	struct regulator_dev *dev;
	struct gpio_desc **power_gpiods;
	struct gpio_desc **reset_gpiods;
	struct bcm_fixed_config *config;
	unsigned long *values_1;
	unsigned long *values_0;
};

/**
 * bcm_fixed_delay - a delay helper function
 * @delay: time to delay in microseconds
 *
 * The assumption here is that regulators will never be enabled in
 * atomic context and therefore sleeping functions can be used.
 */
static void bcm_fixed_delay(unsigned int delay_us)
{
	unsigned int ms = delay_us / 1000;
	unsigned int us = delay_us % 1000;

	if (!ms  && !us)
		return;

	if (ms > 0) {
		/*
		 * For small enough values, handle super-millisecond
		 * delays in the usleep_range() call below.
		 */
		if (ms < 20)
			us += ms * 1000;
		else
			msleep(ms);
	}

	/*
	 * Give the scheduler some room to coalesce with any other
	 * wakeup sources. For delays shorter than 10 us, don't even
	 * bother setting up high-resolution timers and just busy-
	 * loop.
	 */
	if (us >= 10)
		usleep_range(us, us + 100);
	else
		udelay(us);
}

static int bcm_fixed_regulator_is_enabled(struct regulator_dev *dev)
{
	int power = 0, reset = 0;

	struct bcm_fixed_data *drvdata = rdev_get_drvdata(dev);

	if (!drvdata->power_gpiods ||
			gpiod_get_value_cansleep(drvdata->power_gpiods[0]))
		power = 1;

	if (!drvdata->reset_gpiods ||
			gpiod_get_value_cansleep(drvdata->reset_gpiods[0]))
		reset = 1;

	if (power && reset)
		return true;

	return false;
}

static int bcm_fixed_regulator_enable(struct regulator_dev *dev)
{
	struct bcm_fixed_data *drvdata = rdev_get_drvdata(dev);
	struct bcm_fixed_pinctrl *pins = &drvdata->config->pinctrl;
	int ret;

	if (bcm_fixed_regulator_is_enabled(dev))
		return 0;

	if (drvdata->reset_gpiods &&
			gpiod_get_value_cansleep(drvdata->reset_gpiods[0])) {
		gpiod_set_array_value_cansleep(drvdata->config->reset.ngpios,
				drvdata->reset_gpiods, NULL, drvdata->values_0);
		if (drvdata->config->assert_delay)
			bcm_fixed_delay(drvdata->config->assert_delay);
	}

	if (drvdata->power_gpiods) {
		gpiod_set_array_value_cansleep(drvdata->config->power.ngpios,
				drvdata->power_gpiods, NULL, drvdata->values_1);
		if (drvdata->config->startup_delay)
			bcm_fixed_delay(drvdata->config->startup_delay);
	}

	if (pins->pinctrl) {
		ret = pinctrl_select_state(pins->pinctrl, pins->states[1]);
		if (ret)
			pr_err("Couldn't set %s state, err %d\n",
					pins->names[1], ret);
	}

	if (drvdata->reset_gpiods) {
		gpiod_set_array_value_cansleep(drvdata->config->reset.ngpios,
				drvdata->reset_gpiods, NULL, drvdata->values_1);
		if (drvdata->config->deassert_delay)
			bcm_fixed_delay(drvdata->config->deassert_delay);
	}

	return 0;
}

static void bcm_fixed_regulator_set_disable_state(
			struct regulator_dev *dev)
{
	struct bcm_fixed_data *drvdata = rdev_get_drvdata(dev);
	struct bcm_fixed_pinctrl *pins = &drvdata->config->pinctrl;
	int ret;

	if (drvdata->reset_gpiods &&
			gpiod_get_value_cansleep(drvdata->reset_gpiods[0])) {
		gpiod_set_array_value_cansleep(drvdata->config->reset.ngpios,
				drvdata->reset_gpiods, NULL, drvdata->values_0);
		if (drvdata->config->assert_delay)
			bcm_fixed_delay(drvdata->config->assert_delay);
	}

	if (pins->pinctrl) {
		ret = pinctrl_select_state(pins->pinctrl, pins->states[0]);
		if (ret)
			pr_err("Couldn't set %s state, err %d\n",
					pins->names[0], ret);
	}

	if (drvdata->power_gpiods) {
		gpiod_set_array_value_cansleep(drvdata->config->power.ngpios,
				drvdata->power_gpiods, NULL, drvdata->values_0);
		if (drvdata->config->shutdown_delay)
			bcm_fixed_delay(drvdata->config->shutdown_delay);
	}
}

static int bcm_fixed_regulator_disable(struct regulator_dev *dev)
{
	if (!bcm_fixed_regulator_is_enabled(dev))
		return 0;

	bcm_fixed_regulator_set_disable_state(dev);

	return 0;
}

static struct regulator_ops bcm_fixed_ops = {
	.enable          = bcm_fixed_regulator_enable,
	.disable         = bcm_fixed_regulator_disable,
	.is_enabled      = bcm_fixed_regulator_is_enabled,
};

static int bcm_fixed_gpios_get(struct device *dev, char *name,
		struct bcm_fixed_gpios *gpio)
{
	char propname[32];
	int i, count;

	count = gpiod_count(dev, name);
	if (count <= 0)
		return 0;

	gpio->gpios = devm_kzalloc(dev, count * sizeof(int), GFP_KERNEL);
	if (!gpio->gpios) {
		dev_err(dev, "Fail to alloc %s gpios\n", propname);
		return -ENOMEM;
	}

	gpio->ngpios = count;
	snprintf(propname, sizeof(propname), "%s-gpios", name);
	for (i = 0; i < count; i++)
		gpio->gpios[i] = of_get_named_gpio(dev->of_node, propname, i);

	return 0;
}

static int bcm_fixed_pinctrl_get(struct device *dev,
		struct bcm_fixed_pinctrl *pins)
{
	const char *name;
	int i, count, ret = -ENOENT;

	count = of_property_count_strings(dev->of_node, "pinctrl-names");
	if (count != NR_PIN_STATES) {
		dev_warn(dev, "Current supported number of states is %d\n",
			NR_PIN_STATES);
		return 0;
	}

	pins->pinctrl = devm_pinctrl_get(dev);
	if (IS_ERR(pins->pinctrl)) {
		pins->pinctrl = NULL;
		return 0;
	}

	for (i = 0; i < count; i++) {
		ret = of_property_read_string_index(dev->of_node,
				"pinctrl-names", i, &name);
		if (ret)
			break;

		pins->names[i] = name;
		pins->states[i] = pinctrl_lookup_state(pins->pinctrl, name);
		if (IS_ERR(pins->states[i])) {
			ret = PTR_ERR(pins->states[i]);
			break;
		}
	}

	if (ret) {
		devm_pinctrl_put(pins->pinctrl);
		memset(pins, 0, sizeof(struct bcm_fixed_pinctrl));
	}

	return ret;
}

/**
 * bcm_fixed_config_get - extract
 * bcm_fixed_config structure info
 * @dev: device requesting for bcm_fixed_config
 * @desc: regulator description
 *
 * Populates bcm_fixed_config structure by
 * extracting data from device tree node, returns a pointer to
 * the populated structure of NULL if memory alloc fails.
 */
static struct bcm_fixed_config *
	bcm_fixed_config_get(struct device *dev,
			    const struct regulator_desc *desc)
{
	struct bcm_fixed_config *config;
	struct device_node *np = dev->of_node;
	int ret = 0;

	config = devm_kzalloc(dev,
				sizeof(struct bcm_fixed_config),
				GFP_KERNEL);
	if (!config) {
		dev_err(dev,
			"Memory allocation on bcm_fixed_config failure\n");
		return ERR_PTR(-ENOMEM);
	}

	config->init_data = of_get_regulator_init_data(dev,
							dev->of_node, desc);
	if (!config->init_data) {
		dev_err(dev,
			 " of_get_regulator_init_data failure\n");
		return ERR_PTR(-EINVAL);
	}

	ret = bcm_fixed_gpios_get(dev, "power", &config->power);
	if (ret)
		return ERR_PTR(ret);

	ret = bcm_fixed_gpios_get(dev, "reset", &config->reset);
	if (ret)
		return ERR_PTR(ret);

	ret = bcm_fixed_pinctrl_get(dev, &config->pinctrl);
	if (ret)
		return ERR_PTR(ret);

	of_property_read_u32(np, "startup-delay-us", &config->startup_delay);
	of_property_read_u32(np, "shutdown-delay-us", &config->shutdown_delay);
	of_property_read_u32(np, "assert-delay-us", &config->assert_delay);
	of_property_read_u32(np, "deassert-delay-us",
		&config->deassert_delay);

	return config;
}

static void *bcm_fixed_gpiods_get(struct device *dev, char *name,
		int *gpios, int count)
{
	struct gpio_desc **gpiods;
	int i;

	gpiods = devm_kcalloc(dev, count, sizeof(struct gpio_desc *),
			GFP_KERNEL);
	if (!gpiods) {
		dev_err(dev, "Fail to alloc %s gpiods\n", name);
		return ERR_PTR(-ENOMEM);
	}

	for (i = 0; i < count; i++) {
		struct gpio_desc *gpiod;

		if (!gpio_is_valid(gpios[i])) {
			dev_err(dev, "%s gpio %d is not valid\n",
					name, gpios[i]);
			return ERR_PTR(-EINVAL);
		}

		gpiod = devm_gpiod_get_index(dev, name, i, GPIOD_OUT_LOW);
		if (IS_ERR(gpiod)) {
			dev_err(dev, "Failed to get %s gpiod %d\n", name, i);
			return gpiod;
		}

		gpiods[i] = gpiod;
	}

	return gpiods;
}

static int reg_bcm_fixed_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct bcm_fixed_config *config;
	struct bcm_fixed_data *drvdata;
	struct regulator_config cfg = { };
	int count, ret;

	drvdata = devm_kzalloc(dev, sizeof(struct bcm_fixed_data),
			GFP_KERNEL);
	if (!drvdata) {
		dev_err(dev, "Failed to allocate bcm_fixed_data\n");
		return -ENOMEM;
	}

	drvdata->desc.type = REGULATOR_VOLTAGE;
	drvdata->desc.owner = THIS_MODULE;
	drvdata->desc.ops = &bcm_fixed_ops;

	if (dev->of_node)
		config = bcm_fixed_config_get(dev, &drvdata->desc);
	else
		config = dev_get_platdata(dev);

	if (!config || IS_ERR(config)) {
		dev_err(dev, "Getting platform data failure\n");
		return !config ? -ENODEV : PTR_ERR(config);
	}

	if (!config->power.gpios && !config->reset.gpios) {
		dev_err(dev, "No power-gpios or reset-gpios defined\n");
		return -ENODEV;
	}

	drvdata->config = config;
	drvdata->desc.supply_name = "vin";
	drvdata->desc.name = devm_kstrdup(dev,
			config->init_data->constraints.name,
			GFP_KERNEL);
	if (drvdata->desc.name == NULL) {
		dev_err(dev, "Failed to allocate supply name\n");
		return -ENOMEM;
	}

	count = max_t(int, config->power.ngpios, config->reset.ngpios);
	drvdata->values_0 = devm_kzalloc(dev, BITS_TO_LONGS(count) *
			sizeof(unsigned long), GFP_KERNEL);
	if (!drvdata->values_0) {
		dev_err(dev, "Fail to alloc zeros array\n");
		return -ENOMEM;
	}

	drvdata->values_1 = devm_kzalloc(dev, BITS_TO_LONGS(count) *
			sizeof(unsigned long), GFP_KERNEL);
	if (!drvdata->values_1) {
		dev_err(dev, "Fail to alloc ones array\n");
		return -ENOMEM;
	}

	bitmap_fill(drvdata->values_1, count);

	if (config->power.gpios)
		drvdata->power_gpiods = (struct gpio_desc **)
			bcm_fixed_gpiods_get(dev, "power",
					config->power.gpios,
					config->power.ngpios);

	if (config->reset.gpios)
		drvdata->reset_gpiods = (struct gpio_desc **)
			bcm_fixed_gpiods_get(dev, "reset",
					config->reset.gpios,
					config->reset.ngpios);

	cfg.dev = dev;
	cfg.init_data = config->init_data;
	cfg.driver_data = drvdata;
	cfg.of_node = dev->of_node;

	drvdata->dev = devm_regulator_register(dev, &drvdata->desc, &cfg);
	if (IS_ERR(drvdata->dev)) {
		ret = PTR_ERR(drvdata->dev);
		dev_err(dev, "Failed to register regulator: %d\n", ret);
		return ret;
	}

	/*
	 * By default set pwr and reset pins similar to the state when
	 * regulator is disabled.
	 */
	bcm_fixed_regulator_set_disable_state(drvdata->dev);

	platform_set_drvdata(pdev, drvdata);

	dev_dbg(dev, "%s supplying %duV\n", drvdata->desc.name,
			drvdata->desc.fixed_uV);

	return 0;
}

static const struct of_device_id bcm_fixed_of_match[] = {
	{ .compatible = "regulator-bcm-fixed", },
	{},
};
MODULE_DEVICE_TABLE(of, bcm_fixed_of_match);

static struct platform_driver
				regulator_bcm_fixed_driver = {
	.probe		= reg_bcm_fixed_probe,
	.driver		= {
		.name		= "reg-bcm-fixed",
		.of_match_table = of_match_ptr(bcm_fixed_of_match),
	},
};

static int __init regulator_bcm_fixed_init(void)
{
	return platform_driver_register(
			&regulator_bcm_fixed_driver);
}
subsys_initcall(regulator_bcm_fixed_init);

static void __exit regulator_bcm_fixed_exit(void)
{
	platform_driver_unregister(
		&regulator_bcm_fixed_driver);
}
module_exit(regulator_bcm_fixed_exit);

MODULE_AUTHOR("Venky Selvaraj <venky.selvaraj@broadcom.com>");
MODULE_AUTHOR("Ravi Patel <ravi.patel@broadcom.com>");
MODULE_DESCRIPTION("BCM fixed regulator");
MODULE_LICENSE("GPL");
MODULE_ALIAS("platform:reg-bcm-fixed");

