/*
 * Broadcom oops capture and display (boops)
 *
 * Copyright (C) 2017 Broadcom
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#include <linux/kernel.h>
#include <linux/mmc/mmc.h>
#include <linux/mmc/host.h>
#include <linux/scatterlist.h>
#include <linux/vmalloc.h>

#include "boops_k_storage.h"
#include "boops_k_common.h"

/* SDIO controller has a max transfer of 64k per entry (512K total table).
   For now we do only a single transfer so limit to 64Kb (128 blocks of 512b) */
#define BOOPS_MMC_MAX_OOPS_BLOCKS 128

struct boops_storage_mmc_t {
    struct boops_storage_t generic;                                     /* Generic interface to this storage device */
    struct mmc_card*       card;
    uint32_t               write_start;
};

static
int boops_storage_mmc_write_blocks(const struct boops_storage_t *storage, uint8_t *buf, uint32_t block_count) {
    struct boops_storage_mmc_t *mmc = container_of(storage, struct boops_storage_mmc_t, generic);
    struct mmc_request mrq;
    struct mmc_command cmd;
    struct mmc_command stop;
    struct mmc_data data;
    struct scatterlist sg;
    memset(&mrq, 0, sizeof(struct mmc_request));
    memset(&cmd, 0, sizeof(struct mmc_command));
    memset(&stop, 0, sizeof(struct mmc_command));
    memset(&data, 0, sizeof(struct mmc_data));

    boops_mmc_claim_host(mmc->card->host);

    if (!boops_mmc_is_user_part_active(mmc->card)) {
        boops_err("*** The user partition is not currently active! Cannot dump the oops!!! ***\n");
        boops_mmc_release_host(mmc->card->host);
        return -1;
    }

    mrq.cmd = &cmd;
    mrq.data = &data;

    data.blksz = storage->block.bytes;
    data.blocks = block_count;
    data.sg = &sg;
    data.sg_len = 1;
    data.flags = MMC_DATA_WRITE;

    sg_init_one(&sg, buf, data.blksz*data.blocks);

    cmd.opcode = MMC_WRITE_BLOCK;
    cmd.arg = mmc->write_start;
    cmd.flags = MMC_RSP_R1 | MMC_CMD_ADTC;

    if (data.blocks > 1) {
        cmd.opcode = MMC_WRITE_MULTIPLE_BLOCK;
        mrq.stop = &stop;
        stop.opcode = MMC_STOP_TRANSMISSION;
        stop.arg = 0;
        stop.flags = MMC_RSP_R1B | MMC_CMD_AC;
    }

    boops_info("Starting MMC transfer of %u blocks\n", block_count);
    mmc_wait_for_req(mmc->card->host, &mrq);
    boops_mmc_release_host(mmc->card->host);
    if (cmd.error) {
        boops_err("Failed MMC write with cmd error %d\n", cmd.error);
        return cmd.error;
    }
    if (data.error) {
        boops_err("Failed MMC write with data error %d\n", cmd.error);
        return data.error;
    }
    boops_info("Completed MMC transfer\n");
    return 0;
}

static
void boops_storage_mmc_destroy(struct boops_storage_t *storage) {
    struct boops_storage_mmc_t *mmc = container_of(storage, struct boops_storage_mmc_t, generic);
    vfree(mmc);
}

struct boops_storage_t* boops_storage_mmc_create(const uint32_t block_first, const uint32_t block_count) {
    struct boops_storage_mmc_t *mmc = NULL;
    struct boops_storage_t *storage = NULL;

    mmc = vmalloc(sizeof(*mmc));
    if (!mmc) {
        boops_err("Failed to allocate storage\n");
        return NULL;
    }
    memset(mmc, 0, sizeof(*mmc));

    /* Set storage first */
    storage = &mmc->generic;

    /* Setup MMC */
    mmc->card = boops_mmc_blk_mmc_card;
    mmc->write_start = block_first;
    if (!mmc->card) {
        boops_err("card is NULL (%p)\n", boops_mmc_blk_mmc_card);
        vfree(mmc);
        return NULL;
    }

    /* Setup generic storage */
    if (mmc_card_mmc(mmc->card))
        boops_info("Using MMC device\n");
    else if (mmc_card_sd(mmc->card))
        boops_info("Using SD device\n");
    else {
        boops_err("Failed to determine card type\n");
        vfree(mmc);
        return NULL;
    }

    storage->write_blocks=boops_storage_mmc_write_blocks;
    storage->destroy=boops_storage_mmc_destroy;
    storage->block.count = block_count;
    storage->block.bytes = 512; /* For MMC this will be 512 */
    if (storage->block.count > BOOPS_MMC_MAX_OOPS_BLOCKS) {
        boops_info("The boops partition can store %lu blocks, however we will limit to %u blocks\n", storage->block.count, BOOPS_MMC_MAX_OOPS_BLOCKS);
        storage->block.count = BOOPS_MMC_MAX_OOPS_BLOCKS;
    }

    return &mmc->generic;
}

