/****************************************************************************
 *
 * Copyright (c) 2015-2018 Broadcom. All rights reserved
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Tim Ross <tross@broadcom.com>
 *****************************************************************************/
#include <linux/proc_fs.h>
#include "dqnet_priv.h"
#include "dqnet_ethstats.h"
#include "dqnet_dbg.h"
#include "dqnet_switch.h"

#define ETHSTATS_PROC_FILE	"dev_eth"
#define DQNETSTATS_PROC_ALL	"all"
#define DQNETSTATS_PROC_DIR	"dqnet_stats"
#if defined(CONFIG_BCM_ETHSW) || defined(CONFIG_BCM_ETHSW_MODULE)
static int dqnet_ethstats_open(struct inode *inode, struct file *file);
static void *dqnet_ethstats_start(struct seq_file *seq, loff_t *pos);
static void dqnet_ethstats_stop(struct seq_file *seq, void *v);
static void *dqnet_ethstats_next(struct seq_file *seq, void *v, loff_t *pos);
static int dqnet_ethstats_show(struct seq_file *seq, void *v);

static const struct seq_operations ethstats_seq_ops = {
	.start	= dqnet_ethstats_start,
	.stop	= dqnet_ethstats_stop,
	.next	= dqnet_ethstats_next,
	.show	= dqnet_ethstats_show,
};

static const struct proc_ops ethstats_fops = {
	.proc_open		= dqnet_ethstats_open,
	.proc_read		= seq_read,
	.proc_lseek		= seq_lseek,
	.proc_release	= seq_release,
};

static int dqnet_ethstats_open(struct inode *inode, struct file *file)
{
	int status;

	pr_debug("-->\n");

	status = seq_open(file, &ethstats_seq_ops);

	pr_debug("<--\n");
	return status;
}

static void *dqnet_ethstats_start(struct seq_file *seq, loff_t *pos)
{
	struct dqnet_netdev *ndev = NULL;
	struct dqnet_netdev *tmp;
	loff_t off = -1;

	pr_debug("-->\n");

	rcu_read_lock();
	list_for_each_entry_rcu(tmp, &ndevs.list, list) {
		if (tmp->link_type == DQNET_LINK_TYPE_SWITCH)
			off++;
		if (*pos == off) {
			ndev = tmp;
			break;
		}
	}
	rcu_read_unlock();

	pr_debug("<--\n");
	return ndev;
}

static void dqnet_ethstats_stop(struct seq_file *seq, void *v)
{
	pr_debug("-->\n");
	pr_debug("<--\n");
}

static void *dqnet_ethstats_next(struct seq_file *seq, void *v, loff_t *pos)
{
	struct dqnet_netdev *ndev;

	pr_debug("-->\n");

	(*pos)++;
	ndev = dqnet_ethstats_start(seq, pos);

	pr_debug("<--\n");
	return ndev;
}

static int dqnet_ethstats_show(struct seq_file *seq, void *v)
{
	int status = 0;
	struct dqnet_netdev *ndev = v;
	struct dqnet_switch_stats stats;

	pr_debug("-->\n");

	seq_printf(seq, "%s:\n", ndev->dev->name);
	SWITCH->get_mib_counters(ndev->dev, &stats);
	seq_printf(seq, "TXOctets:              %20llu\n",
		   stats.tx_bytes);
	seq_printf(seq, "TxDropPkts:            %20u\n",
		   stats.tx_drop_pkts);
	seq_printf(seq, "TxBroadcastPkts:       %20u\n",
		   stats.tx_broadcast_pkts);
	seq_printf(seq, "TxMulticastPkts:       %20u\n",
		   stats.tx_multicast_pkts);
	seq_printf(seq, "TxUnicastPkts:         %20u\n",
		   stats.tx_unicast_pkts);
	seq_printf(seq, "TxPausePkts:           %20u\n",
		   stats.tx_pause_pkts);
	seq_printf(seq, "TxCollisions:          %20u\n",
		   stats.tx_collisions);
	seq_printf(seq, "TxSingleCollision:     %20u\n",
		   stats.tx_single_collision);
	seq_printf(seq, "TxMultipleCollision:   %20u\n",
		   stats.tx_multiple_collision);
	seq_printf(seq, "TxLateCollision:       %20u\n",
		   stats.tx_late_collision);
	seq_printf(seq, "TxExcessiveCollision:  %20u\n",
		   stats.tx_excessive_collision);
	seq_printf(seq, "TxDeferredTransmit:    %20u\n",
		   stats.tx_deferred_transmit);
	seq_printf(seq, "TxFrameInDisc:         %20u\n",
		   stats.tx_frame_in_disc);
	seq_printf(seq, "TxQ0Pkt:               %20u\n",
		   stats.tx_q0_pkt);
	seq_printf(seq, "TxQ1Pkt:               %20u\n",
		   stats.tx_q1_pkt);
	seq_printf(seq, "TxQ2Pkt:               %20u\n",
		   stats.tx_q2_pkt);
	seq_printf(seq, "TxQ3Pkt:               %20u\n",
		   stats.tx_q3_pkt);
	seq_printf(seq, "TxQ4Pkt:               %20u\n",
		   stats.tx_q4_pkt);
	seq_printf(seq, "TxQ5Pkt:               %20u\n",
		   stats.tx_q5_pkt);
	seq_printf(seq, "TxQ6Pkt:               %20u\n",
		   stats.tx_q6_pkt);
	seq_printf(seq, "TxQ7Pkt:               %20u\n",
		   stats.tx_q7_pkt);
	seq_printf(seq, "TxPkts64Octets:        %20u\n",
		   stats.tx_pkts_64_bytes);
	seq_printf(seq, "TxPkts65to127Octets:   %20u\n",
		   stats.tx_pkts_65to127_bytes);
	seq_printf(seq, "TxPkts128to255Octets:  %20u\n",
		   stats.tx_pkts_128to255_bytes);
	seq_printf(seq, "TxPkts256to511Octets:  %20u\n",
		   stats.tx_pkts_256to511_bytes);
	seq_printf(seq, "TxPkts512to1023Octets: %20u\n",
		   stats.tx_pkts_512to1023_bytes);
	seq_printf(seq, "TxPkts1024toMax:       %20u\n",
		   stats.tx_pkts_1024tomax_bytes);
	seq_printf(seq, "RxOctets:              %20llu\n",
		   stats.rx_bytes);
	seq_printf(seq, "RxGoodOctets:          %20llu\n",
		   stats.rx_good_bytes);
	seq_printf(seq, "RxDropPkts:            %20u\n",
		   stats.rx_drop_pkts);
	seq_printf(seq, "RxBroadcastPkts:       %20u\n",
		   stats.rx_broadcast_pkts);
	seq_printf(seq, "RxMulticastPkts:       %20u\n",
		   stats.rx_multicast_pkts);
	seq_printf(seq, "RxUnicastPkts:         %20u\n",
		   stats.rx_unicast_pkts);
	seq_printf(seq, "RxPausePkts:           %20u\n",
		   stats.rx_pause_pkts);
	seq_printf(seq, "RxSAChanges:           %20u\n",
		   stats.rx_sa_changes);
	seq_printf(seq, "RxUndersizePkts:       %20u\n",
		   stats.rx_undersize_pkts);
	seq_printf(seq, "RxPkts64Octets:        %20u\n",
		   stats.rx_pkts_64_bytes);
	seq_printf(seq, "RxPkts64to127Octets:   %20u\n",
		   stats.rx_pkts_65to127_bytes);
	seq_printf(seq, "RxPkts127To255Octets:  %20u\n",
		   stats.rx_pkts_128to255_bytes);
	seq_printf(seq, "RxPkts256to511Octets:  %20u\n",
		   stats.rx_pkts_256to511_bytes);
	seq_printf(seq, "RxPkts512to1023Octets: %20u\n",
		   stats.rx_pkts_512to1023_bytes);
	seq_printf(seq, "RxPkts1024toMaxOctets: %20u\n",
		   stats.rx_pkts_1024tomax_bytes);
	seq_printf(seq, "RxOversizePkts:        %20u\n",
		   stats.rx_oversize_pkts);
	seq_printf(seq, "RxJumboPkts:           %20u\n",
		   stats.rx_jumbo_pkts);
	seq_printf(seq, "RxFragments:           %20u\n",
		   stats.rx_fragments);
	seq_printf(seq, "RxJabbers:             %20u\n",
		   stats.rx_jabbers);
	seq_printf(seq, "RxDiscard:             %20u\n",
		   stats.rx_discard);
	seq_printf(seq, "RxAlignmentErrors:     %20u\n",
		   stats.rx_alignment_errors);
	seq_printf(seq, "RxFcsErrors:           %20u\n",
		   stats.rx_fcs_errors);
	seq_printf(seq, "RxSymbolErrors:        %20u\n",
		   stats.rx_symbol_errors);
	seq_printf(seq, "RxInRangeErrors:       %20u\n",
		   stats.rx_in_range_errors);
	seq_printf(seq, "RxOutOfRangeErrors:    %20u\n",
		   stats.rx_out_of_range_errors);
	seq_printf(seq, "EeeLpiEvent:           %20u\n",
		   stats.eee_lpi_event);
	seq_printf(seq, "EeeLpiDuration:        %20u\n",
		   stats.eee_lpi_duration);
	seq_printf(seq, "\n");

	pr_debug("<--\n");
	return status;
}
#endif
static void *dqnetstats_all_start(struct seq_file *seq, loff_t *pos)
{
	struct dqnet_netdev *ndev = NULL;
	struct dqnet_netdev *tmp;
	loff_t off = 0;

	if (!*pos)
		return SEQ_START_TOKEN;

	rcu_read_lock();
	list_for_each_entry_rcu(tmp, &ndevs.list, list) {
		off++;
		if (*pos == off) {
			ndev = tmp;
			break;
		}
	}
	rcu_read_unlock();

	return ndev;
}

static void dqnetstats_all_stop(struct seq_file *seq, void *v)
{
}

static void *dqnetstats_all_next(struct seq_file *seq, void *v, loff_t *pos)
{
	struct dqnet_netdev *ndev;

	(*pos)++;
	ndev = dqnetstats_all_start(seq, pos);

	return ndev;
}

static void printf_stat_header(struct seq_file *seq)
{
	seq_puts(seq,   "     Inter-             |    Receive                           "
			"                  |   Transmit\n"
			"      face   DSCP  Path |          bytes   packets errs drop fifo frame "
			"multicast|          bytes   packets errs "
			"drop fifo colls carrier\n");
}

static void printf_stat_footer(struct seq_file *seq)
{
	seq_printf(seq, "========================================================="
			"========================================="
			"=========================================\n");
}

static void printf_stat_divider(struct seq_file *seq)
{
	seq_printf(seq, "---------------------------------------------------------"
			"-----------------------------------------"
			"-----------------------------------------\n");
}

static void printf_stat(struct seq_file *seq, char *name,
			char *path,
			struct rtnl_link_stats64 *stats)
{
	seq_printf(seq, "%*s   sum  %*s | %14llu %9llu %4llu %4llu %4llu %5llu %9llu|"
			" %14llu %9llu %4llu %4llu %4llu %5llu %7llu\n",
			10, name, 5, path,
			stats->rx_bytes,
			stats->rx_packets,
			stats->rx_errors,
			stats->rx_dropped + stats->rx_missed_errors,
			stats->rx_fifo_errors,
			stats->rx_length_errors + stats->rx_over_errors +
			stats->rx_crc_errors + stats->rx_frame_errors,
			stats->multicast,
			stats->tx_bytes, stats->tx_packets,
			stats->tx_errors, stats->tx_dropped,
			stats->tx_fifo_errors, stats->collisions,
			stats->tx_carrier_errors +
			stats->tx_aborted_errors +
			stats->tx_window_errors +
			stats->tx_heartbeat_errors);
}

static void printf_stat_dscp(struct seq_file *seq, char *name,
			     int dscp, char *path,
			     struct rtnl_link_stats64 *stats)
{
	seq_printf(seq, "%*s   %2d   %*s | %14llu %9llu %4llu %4llu %4llu %5llu %9llu|"
			" %14llu %9llu %4llu %4llu %4llu %5llu %7llu\n",
			10, name, dscp, 5, path,
			stats->rx_bytes,
			stats->rx_packets,
			stats->rx_errors,
			stats->rx_dropped + stats->rx_missed_errors,
			stats->rx_fifo_errors,
			stats->rx_length_errors + stats->rx_over_errors +
			stats->rx_crc_errors + stats->rx_frame_errors,
			stats->multicast,
			stats->tx_bytes, stats->tx_packets,
			stats->tx_errors, stats->tx_dropped,
			stats->tx_fifo_errors, stats->collisions,
			stats->tx_carrier_errors +
			stats->tx_aborted_errors +
			stats->tx_window_errors +
			stats->tx_heartbeat_errors);
}

static void printf_stat_offload_fast(struct seq_file *seq,
				     struct dqnet_netdev *dev)
{
	struct rtnl_link_stats64 stats = { 0 };
	struct pcpu_sw_netstats tmp = { 0 };
	unsigned int cpu;

	if (!dev->nethooks.offload_stats.fast)
		return;

	for_each_possible_cpu(cpu) {
		unsigned int start;
		const struct pcpu_sw_netstats *tstats
			= per_cpu_ptr(dev->nethooks.offload_stats.fast, cpu);
		do {
			start = u64_stats_fetch_begin(&tstats->syncp);
			memcpy(&tmp, tstats, sizeof(tmp));
		} while (u64_stats_fetch_retry(&tstats->syncp, start));
		stats.tx_bytes   += tmp.tx_bytes;
		stats.tx_packets += tmp.tx_packets;
		stats.rx_bytes   += tmp.rx_bytes;
		stats.rx_packets += tmp.rx_packets;
	}
	printf_stat(seq, dev->dev->name, "fast", &stats);
}

static void printf_stat_offload_dscp_fast(struct seq_file *seq,
					  struct dqnet_netdev *dev)
{
	struct rtnl_link_stats64 stats = { 0 };
	struct pcpu_sw_netstats tmp = { 0 };
	unsigned int cpu, cnt;

	for (cnt=0; cnt<DSCP_MAX_LIMIT; cnt++) {
		memset(&stats, 0, sizeof(stats));
		if (!dev->nethooks.offload_stats.dscp_fast[cnt])
			continue;
		for_each_possible_cpu(cpu) {
			unsigned int dscp_start;
			const struct pcpu_sw_netstats *dscp_tstats
			= per_cpu_ptr(dev->nethooks.offload_stats.dscp_fast[cnt],
				      cpu);
			do {
				dscp_start =
				u64_stats_fetch_begin(&dscp_tstats->syncp);
				memcpy(&tmp, dscp_tstats, sizeof(tmp));
			} while (u64_stats_fetch_retry(&dscp_tstats->syncp,
						       dscp_start));
			stats.tx_bytes   += tmp.tx_bytes;
			stats.tx_packets += tmp.tx_packets;
			stats.rx_bytes   += tmp.rx_bytes;
			stats.rx_packets += tmp.rx_packets;
		}
		if (!stats.tx_packets && !stats.rx_packets)
			continue;

		printf_stat_dscp(seq, dev->dev->name, cnt,
				 "fast", &stats);
	}
}

static void printf_stat_dqnet(struct seq_file *seq,
			      struct dqnet_netdev *dev)
{
	struct rtnl_link_stats64 *stats = &dev->stats.netdev;

	printf_stat(seq, dev->dev->name, "dqnet", stats);

}

static void printf_stat_offload_slow(struct seq_file *seq,
				     struct dqnet_netdev *dev)
{
	struct rtnl_link_stats64 dscp_stats ;
	struct pcpu_sw_netstats tmp = { 0 };
	unsigned int cpu;

	if (dev->nethooks.offload_stats.slow) {
		memset(&dscp_stats, 0, sizeof(dscp_stats));
		for_each_possible_cpu(cpu) {
			unsigned int dscp_start;
			const struct pcpu_sw_netstats *dscp_tstats
			= per_cpu_ptr(dev->nethooks.offload_stats.slow, cpu);
			do {
				dscp_start =
				u64_stats_fetch_begin(&dscp_tstats->syncp);
				memcpy(&tmp, dscp_tstats, sizeof(tmp));
			} while (u64_stats_fetch_retry(&dscp_tstats->syncp,
						       dscp_start));
			dscp_stats.tx_bytes   += tmp.tx_bytes;
			dscp_stats.tx_packets += tmp.tx_packets;
			dscp_stats.rx_bytes   += tmp.rx_bytes;
			dscp_stats.rx_packets += tmp.rx_packets;
		}
		printf_stat(seq, dev->dev->name, "slow", &dscp_stats);
	}
}

static void printf_stat_offload_dscp_slow(struct seq_file *seq,
					  struct dqnet_netdev *dev)
{
	struct rtnl_link_stats64 dscp_stats ;
	struct pcpu_sw_netstats tmp = { 0 };
	unsigned int cpu, cnt;

	for (cnt=0; cnt<DSCP_MAX_LIMIT; cnt++) {
		memset(&dscp_stats, 0, sizeof(dscp_stats));
		if (!dev->nethooks.offload_stats.dscp_slow[cnt])
			continue;
		for_each_possible_cpu(cpu) {
			unsigned int dscp_start;
			const struct pcpu_sw_netstats *dscp_tstats
			= per_cpu_ptr(dev->nethooks.offload_stats.dscp_slow[cnt],
				      cpu);
			do {
				dscp_start =
				u64_stats_fetch_begin(&dscp_tstats->syncp);
				memcpy(&tmp, dscp_tstats, sizeof(tmp));
			} while (u64_stats_fetch_retry(&dscp_tstats->syncp,
						       dscp_start));
			dscp_stats.tx_bytes   += tmp.tx_bytes;
			dscp_stats.tx_packets += tmp.tx_packets;
			dscp_stats.rx_bytes   += tmp.rx_bytes;
			dscp_stats.rx_packets += tmp.rx_packets;
		}
		if (!dscp_stats.tx_packets && !dscp_stats.rx_packets)
			continue;
		printf_stat_dscp(seq, dev->dev->name, cnt,
				 "slow", &dscp_stats);
	}
}

static void printf_stat_dev(struct seq_file *seq, struct net_device *dev)
{
	struct rtnl_link_stats64 temp;
	struct rtnl_link_stats64 *stats = dev_get_stats(dev, &temp);

	printf_stat(seq, dev->name, "dev", stats);
}

static int dqnetstats_all_show(struct seq_file *seq, void *v)
{
	int status = 0;
	struct dqnet_netdev *ndev = v;

	if (ndev == SEQ_START_TOKEN)
		printf_stat_header(seq);
	else {
		printf_stat_offload_dscp_fast(seq, ndev);
		printf_stat_offload_fast(seq, ndev);
		printf_stat_divider(seq);
		printf_stat_offload_dscp_slow(seq, ndev);
		printf_stat_offload_slow(seq, ndev);
		printf_stat_divider(seq);
		printf_stat_dqnet(seq, ndev);
		printf_stat_dev(seq, ndev->dev);
		printf_stat_footer(seq);
	}

	return status;
}

static const struct seq_operations dqnetstats_all_seq_ops = {
	.start	= dqnetstats_all_start,
	.stop	= dqnetstats_all_stop,
	.next	= dqnetstats_all_next,
	.show	= dqnetstats_all_show,
};

static int dqnetstats_all_open(struct inode *inode, struct file *file)
{
	int status;

	status = seq_open(file, &dqnetstats_all_seq_ops);

	return status;
}

static int dqnetstats_dev_seq_show(struct seq_file *seq, void *v)
{
	struct dqnet_netdev *ndev = (struct dqnet_netdev *)seq->private;

	printf_stat_header(seq);
	printf_stat_offload_dscp_fast(seq, ndev);
	printf_stat_offload_fast(seq, ndev);
	printf_stat_divider(seq);
	printf_stat_offload_dscp_slow(seq, ndev);
	printf_stat_offload_slow(seq, ndev);
	printf_stat_divider(seq);
	printf_stat_dqnet(seq, ndev);
	printf_stat_dev(seq, ndev->dev);
	printf_stat_footer(seq);

	return 0;
}

static ssize_t dqnetstats_all_write(struct file *file, const char *buffer,
				    size_t count, loff_t *pos)
{
	struct dqnet_netdev *ndev;
	rcu_read_lock();
	list_for_each_entry_rcu(ndev, &ndevs.list, list) {
		if (ndev->hal.macdev)
			haldev_link_stats(&ndev->hal, NULL);
		else if (ndev->link_stats)
			ndev->link_stats(ndev->dev, NULL);

		memset(&ndev->stats, 0, sizeof(ndev->stats));
		memset(&ndev->nethooks.offload_stats, 0,
		       sizeof(ndev->nethooks.offload_stats));
		pr_info("%s: Counters cleared\n", ndev->dev->name);
	}
	rcu_read_unlock();
	return count;
}

static int dqnetstats_dev_seq_open(struct inode *inode, struct file *file)
{
	return single_open(file, dqnetstats_dev_seq_show, PDE_DATA(inode));
}

static ssize_t dqnetstats_dev_seq_write(struct file *file, const char *buffer,
					size_t count, loff_t *pos)
{
	struct seq_file *m = file->private_data;
	struct dqnet_netdev *ndev;
	ndev = m->private;
	if (buffer[0] != '0') {
		if (ndev->hal.macdev)
			haldev_link_stats(&ndev->hal, NULL);
		else if (ndev->link_stats)
			ndev->link_stats(ndev->dev, NULL);

		memset(&ndev->stats, 0, sizeof(ndev->stats));
		memset(&ndev->nethooks.offload_stats, 0,
		       sizeof(ndev->nethooks.offload_stats));
		pr_info("%s: Counters cleared\n", ndev->dev->name);
	}
	return count;
}

static const struct proc_ops dqnetstats_all_fops = {
	.proc_open	 = dqnetstats_all_open,
	.proc_write   = dqnetstats_all_write,
	.proc_read	 = seq_read,
	.proc_lseek	 = seq_lseek,
	.proc_release = seq_release,
};

static const struct proc_ops dqnetstats_dev_seq_fops = {
	.proc_open	 = dqnetstats_dev_seq_open,
	.proc_write   = dqnetstats_dev_seq_write,
	.proc_read	 = seq_read,
	.proc_lseek	 = seq_lseek,
	.proc_release = single_release,
};
#if defined(CONFIG_BCM_ETHSW) || defined(CONFIG_BCM_ETHSW_MODULE)
static struct proc_dir_entry *ethstats_pf;
#endif
static struct proc_dir_entry *dqnetstats_all;
static struct proc_dir_entry *dqnetstats_dir;

int dqnetstats_register_dev(struct dqnet_netdev *ndev)
{
	struct proc_dir_entry *p;
	if (!ndev)
		return -EINVAL;

	if (!dqnetstats_dir)
		return -ENOENT;
	p = proc_create_data(ndev->dev->name, S_IRUGO | S_IWUGO,
			     dqnetstats_dir,
			     &dqnetstats_dev_seq_fops, ndev);
	if (!p)
		return -ENOMEM;

	ndev->proc_dir_entry = p;
	return 0;
}

int dqnetstats_unregister_dev(struct dqnet_netdev *ndev)
{
	if (!dqnetstats_dir)
		return -ENOENT;
	if (!ndev->proc_dir_entry)
		return -EINVAL;
	proc_remove(ndev->proc_dir_entry);
	ndev->proc_dir_entry = NULL;
	return 0;
}

int __init dqnet_ethstats_init(void)
{
	int status = 0;

	pr_debug("-->\n");

	dqnetstats_dir = proc_mkdir(DQNETSTATS_PROC_DIR, init_net.proc_net);
	if (!dqnetstats_dir) {
		pr_err("Failed to create %s\n", DQNETSTATS_PROC_DIR);
		status = -EIO;
		remove_proc_entry(ETHSTATS_PROC_FILE, init_net.proc_net);
		goto done;
	}

	dqnetstats_all = proc_create(DQNETSTATS_PROC_ALL, S_IRUGO,
				    dqnetstats_dir, &dqnetstats_all_fops);
	if (!dqnetstats_all) {
		pr_err("Failed to create %s\n", DQNETSTATS_PROC_ALL);
		status = -EIO;
		remove_proc_entry(DQNETSTATS_PROC_DIR, init_net.proc_net);
		dqnetstats_dir = NULL;
		goto done;
	}

#if defined(CONFIG_BCM_ETHSW) || defined(CONFIG_BCM_ETHSW_MODULE)
	ethstats_pf = proc_create(ETHSTATS_PROC_FILE, S_IRUGO,
				  init_net.proc_net, &ethstats_fops);
	if (!ethstats_pf) {
		pr_err("Failed to create %s\n", ETHSTATS_PROC_FILE);
		remove_proc_entry(ETHSTATS_PROC_FILE, init_net.proc_net);
		dqnetstats_dir = NULL;
		remove_proc_entry(DQNETSTATS_PROC_DIR, init_net.proc_net);
		dqnetstats_dir = NULL;
		status = -EIO;
		goto done;
	}
#endif

done:
	pr_debug("<--\n");
	return status;
}

void dqnet_ethstats_exit(void)
{
	pr_debug("-->\n");
#if defined(CONFIG_BCM_ETHSW) || defined(CONFIG_BCM_ETHSW_MODULE)
	remove_proc_entry(ETHSTATS_PROC_FILE, init_net.proc_net);
	ethstats_pf = NULL;
#endif
	remove_proc_entry(DQNETSTATS_PROC_ALL, dqnetstats_dir);
	dqnetstats_all = NULL;
	remove_proc_entry(DQNETSTATS_PROC_DIR, init_net.proc_net);
	dqnetstats_dir = NULL;

	pr_debug("<--\n");
}
