/****************************************************************************
 *
 * Copyright (c) 2015-2018 Broadcom. All rights reserved
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Tim Ross <tross@broadcom.com>
 *****************************************************************************/
#ifndef _BCM_NETHOOKS_H_
#define _BCM_NETHOOKS_H_

#include <linux/atomic.h>
#include <linux/netdevice.h>
#include <linux/list.h>

#define BCM_NETDEVICE_GROUP_LAN	1
#define BCM_NETDEVICE_GROUP_WAN	2
#define BCM_NETDEVICE_GROUP_CAT_PHY	0
#define BCM_NETDEVICE_GROUP_CAT_VIR	1
#define BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP 2
#define BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP6 3

#define BCM_NETDEVICE_GROUP_TYPE(a)	(a&0xFF)
#define BCM_NETDEVICE_GROUP_CAT(a)	((a>>8)&0xFF)
#define BCM_NETDEVICE_GROUP_FLAG(a)	((a>>16)&0xFF)
#define BCM_NETDEVICE_GROUP_TYPE_CLR(a) (a & 0xFFFFFF00)
#define BCM_NETDEVICE_GROUP_CAT_CLR(a)  (a & 0xFFFF00FF)
#define BCM_NETDEVICE_GROUP_FLAG_CLR(a) (a & 0xFF00FFFF)

#define DSCP_BIT_SIZE 6
#define DSCP_MAX_LIMIT (0x1<<DSCP_BIT_SIZE)

/* BCM Nethooks
 *
 * Each packet transmitted and received will be passed to the first function
 * in a priority-ordered list of nethooks. After processing the packet, each
 * nethook function can decide to:
 *
 * 1. pass the packet to the next nethook for further processing, or
 * 2. drop the packet, or
 * 3. skip remaining nethooks, or
 * 4. consume the packet, taking ownership of it and all further processing
 *    including responsibility to free the buffer
 *
 * Multiple hook points are available for registering a nethook handler. The
 * hook points in both the TX and RX packet processing paths occur
 * where an FPM and SKB buffer are available. The TX and RX packet processing
 * flow and hook points are shown below.
 *
 * RX: DQM-->demux-->FPM hooks-->FPM to SKB buf copy-->SKB hooks-->Linux
 * TX: DQM<----------FPM hooks<--SKB to FPM buf copy<--SKB hooks<--Linux
 *
 * Since the nethook lists at each hook point are processed according to the
 * TX or RX packet processing flows shown above, it follows that in the RX path
 * the FPM nethooks are called before the SKB nethooks and in the TX path the
 * SKB nethooks are called before the FPM nethooks.
 *
 * A list of nethooks is maintained for each hook point and device. Each list
 * of nethooks is ordered according to a priority assigned to each snoop when
 * it is added. A nethook added to a hook point with the same priority as an
 * existing nethook(s) is inserted into the hook point list after all existing
 * nethooks of the same priority.
 *
 * An exception to the above description exists in the case of
 * BCM_NETHOOK_RX_COMPLETE:
 * The purpose of this nethook is to inform a driver that RX processing
 * is complete for this particularly napi cycle, and pending time sensitive
 * processing needs to be completed.
 * The hook function of this nethook should always return BCM_NETHOOK_PASS.
 */

/*
 * BCM Nethook Priorities
 *
 * All nethook priorities must be listed below. This is to consolidate all
 * nethook priorities in the system in one place. Please keep priorities
 * for each type of nethook grouped together and sorted in ascending order.
 * This will simplify identification of hook processing order for each hook
 * point when adding/removing nethooks and debugging.
 *
 * Nethook priorities must be <= 255.
 */
enum bcm_nethook_prio {
	RX_SKB_PRIO_DQNET_DROP_PRIV			= 0x71,
	RX_SKB_PRIO_FLOWMGR				= 0x80,
	RX_SKB_PRIO_EPON_OAM				= 0x78,
	RX_SKB_PRIO_RTF 				= 0x90,
	RX_SKB_PRIO_DQNET_ETH_CLIENT_RETRANS_SRC	= 0x98,

	RX_FPM_PRIO_FLOWMGR				= 0x80,
	RX_FPM_PRIO_RTF 				= 0x70,
	RX_FPM_PRIO_SFAP				= 0x80,

	TX_SKB_PRIO_DQNET_DROP_SWITCH_TO_SWITCH		= 0x70,
	TX_SKB_PRIO_DQNET_DROP_PRIV			= 0x71,
	TX_SKB_PRIO_DQNET_DROP_REDUNDANT_SWITCH_MCASTS	= 0x78,
	TX_SKB_PRIO_FLOWMGR				= 0x80,
	TX_SKB_PRIO_DQNET_TAG_SWITCH_MCASTS		= 0x88,
	TX_SKB_PRIO_RTF 				= 0x90,
	TX_SKB_PRIO_WMF 				= 0x98,

	BCM_NETHOOK_PRIO_MAX				= 0xff
};

enum bcm_nethook_type {
	BCM_NETHOOK_RX_SKB = 0,	/* RX SKB hook point */
	BCM_NETHOOK_TX_SKB,	/* TX SKB hook point */
	BCM_NETHOOK_RX_FPM,	/* RX FPM hook point */
	BCM_NETHOOK_TX_FPM,	/* TX FPM hook point */
	BCM_NETHOOK_TYPE_MAX
};

enum bcm_nethook_result {
	BCM_NETHOOK_PASS,	/* pass packet to next nethook */
	BCM_NETHOOK_DROP,	/* drop packet */
	BCM_NETHOOK_SKIP,	/* skip remaining nethooks */
	BCM_NETHOOK_CONSUMED,	/* nethook consumed packet */
	BCM_NETHOOK_MAX_RESULT
};

/*
 * BCM Nethook functions
 *
 * Each client registering a nethook will provide a callback function
 * conforming to the following prototype. This callback will be called
 * with a read lock from within the driver's packet TX or RX function
 * which is running at soft-IRQ level, so no blocking is allowed.
 *
 * dev	hooked net_device to which the packet belongs
 * type	hook point
 * buf	format of the buf parameter will depend on the value of
 *	type; FPM hooks will reference a struct fpm_buff while
 *	SKB hooks will reference a struct sk_buff.
 *
 * returns a bcm_nethook_result to instruct the network driver the next step
 * 	to take in the processing of the packet.
 *
 */
enum buf_type {
	BUF_TYPE_FPM,
	BUF_TYPE_SKB
};

struct fpm_buff {
	u32	if_id;		/* interface ID from pkt descriptor */
	u32	if_sub_id;	/* interface sub-ID from pkt descriptor */
	u32	priority;	/* priority from pkt descriptor */
	u32	cmim;	        /* CMIM */
	u8	queue_id;	/* RX/TX Queue ID */
	int	brcm_tag_len;	/* length of BRCM tag */
	int	fap_tag_len;	/* length of FAP tag */
	int	sv_tag_len;	/* length of SV tag */

	enum buf_type	type;	/* FPM or SKB? */
	struct sk_buff *skb;	/* SKB */
	u32	token;		/* FPM token */
	u8	*buf;		/* FPM buffer */
	u8	*data;		/* FPM buffer valid data */
	u32	len;		/* FPM data length */
	u32	offset;		/* FPM data offset */
	struct net_device *dev_in;
	struct net_device *dev_out;
};

struct netdev_priv_shared {
	void *priv[4];
};

typedef enum bcm_nethook_result (*bcm_nethook_fn)(
	struct net_device *dev, enum bcm_nethook_type type, void *buf);

#define BCM_NETHOOK_NAME_LEN	64
struct bcm_nethook {
	struct list_head list;
	u32  count[BCM_NETHOOK_MAX_RESULT];
	bcm_nethook_fn hook;
	char name[BCM_NETHOOK_NAME_LEN];
	enum bcm_nethook_prio priority;
	bool enabled;
};

struct bcm_offload_stats {
	struct pcpu_sw_netstats __percpu *fast;
	struct pcpu_sw_netstats __percpu *dscp_fast[DSCP_MAX_LIMIT];
	struct pcpu_sw_netstats __percpu *mcast_fast;
	struct pcpu_sw_netstats __percpu *slow;
	struct pcpu_sw_netstats __percpu *dscp_slow[DSCP_MAX_LIMIT];
};

#define BCM_NETHOOKS_SIG	0x484f4f4b	/* "HOOK" */
typedef int (*fpm_tx_fn)(struct fpm_buff *fpmb, struct net_device *dev);
typedef int (*flow_mdb_counter_update_fn)(struct net_device *dev);
struct bcm_nethooks {
	u32 sig;
	struct bcm_nethook hooks[BCM_NETHOOK_TYPE_MAX];	/* lists */
	rwlock_t hook_locks[BCM_NETHOOK_TYPE_MAX];	/* list update locks */
	fpm_tx_fn fpm_tx;
	flow_mdb_counter_update_fn flow_mdb_counter_update;
	struct bcm_offload_stats offload_stats;
};

/*
 * Each driver supporting BCM nethooks must include a struct bcm_nethooks in
 * its netdev_priv data structure. The struct bcm_nethooks must begin at an
 * offset of one pointer from the beginning of the netdev_priv structure as
 * follows:
 *
 * struct driver_private_data {
 *      void *driver_use;
 *      bcm_nethooks nethooks;
 *      ...
 * }
 *
 */
#define bcm_nethooks_priv()	(netdev_priv(dev) + 4 * sizeof(void *))

static inline int bcm_nethook_register_hook(
	struct net_device *dev, enum bcm_nethook_type type,
	enum bcm_nethook_prio priority, char *name, bcm_nethook_fn hook)
{
	int status = 0;
	struct bcm_nethooks *nethooks = bcm_nethooks_priv();
	struct bcm_nethook *nethook, *tmp;
	unsigned long flags;

	pr_debug("%s:-->\n",dev->name);
	if (nethooks->sig != BCM_NETHOOKS_SIG ||
	    type >= BCM_NETHOOK_TYPE_MAX || !name || !hook ||
	    priority > BCM_NETHOOK_PRIO_MAX) {
		netdev_err(dev, "Invalid parameter provided!\n");
		status = -EINVAL;
		goto done;
	}
	nethook = kmalloc(sizeof(struct bcm_nethook), GFP_KERNEL);
	if (!nethook) {
		netdev_err(dev, "Nethook allocation failed!\n");
		status = -ENOMEM;
		goto done;
	}
	memset(nethook, 0, sizeof(struct bcm_nethook));
	strncpy(nethook->name, name, sizeof(nethook->name));
	nethook->name[sizeof(nethook->name)-1] = '\0';
	nethook->hook = hook;
	nethook->priority = priority;
	write_lock_irqsave(&nethooks->hook_locks[type], flags);
	list_for_each_entry(tmp, &nethooks->hooks[type].list, list) {
		if (tmp->priority > priority)
			break;
	}
	list_add_tail(&nethook->list, &tmp->list);
	write_unlock_irqrestore(&nethooks->hook_locks[type], flags);
done:
	pr_debug("%s:<--\n",dev->name);
	return status;
}

static inline int bcm_nethook_unregister_hook(
	struct net_device *dev, enum bcm_nethook_type type,
	bcm_nethook_fn hook)
{
	int status = 0;
	struct bcm_nethooks *nethooks = bcm_nethooks_priv();
	struct bcm_nethook *nethook, *tmp;
	unsigned long flags;
	bool found = false;

	pr_debug("%s:-->\n",dev->name);
	if (nethooks->sig != BCM_NETHOOKS_SIG ||
	    type >= BCM_NETHOOK_TYPE_MAX || !hook) {
		netdev_err(dev, "Invalid parameter provided!\n");
		status = -EINVAL;
		goto done;
	}
	write_lock_irqsave(&nethooks->hook_locks[type], flags);
	list_for_each_entry_safe(nethook, tmp, &nethooks->hooks[type].list, list) {
		if (nethook->hook == hook) {
			found = true;
			list_del(&nethook->list);
			kfree(nethook);
			break;
		}
	}
	write_unlock_irqrestore(&nethooks->hook_locks[type], flags);
	if (!found)
		netdev_err(dev, "Attempt to unregister non-existent hook!\n");
done:
	pr_debug("%s:<--\n",dev->name);
	return status;
}

static inline int bcm_nethook_enable_hook(
	struct net_device *dev, enum bcm_nethook_type type,
	bcm_nethook_fn hook, bool enable)
{
	int status = 0;
	struct bcm_nethooks *nethooks = bcm_nethooks_priv();
	struct bcm_nethook *nethook;
	bool found = false;

	pr_debug("%s:-->\n",dev->name);
	if (nethooks->sig != BCM_NETHOOKS_SIG ||
	    type >= BCM_NETHOOK_TYPE_MAX || !hook) {
		netdev_err(dev, "Invalid parameter provided!\n");
		status = -EINVAL;
		goto done;
	}
	read_lock(&nethooks->hook_locks[type]);
	list_for_each_entry(nethook, &nethooks->hooks[type].list, list) {
		if (nethook->hook == hook) {
			found = true;
			nethook->enabled = enable;
			break;
		}
	}
	read_unlock(&nethooks->hook_locks[type]);
	if (!found)
		netdev_err(dev, "Attempt to enable non-existent hook!\n");
done:
	pr_debug("%s:<--\n",dev->name);
	return status;
}

static inline int bcm_nethook_tx_fpm(struct fpm_buff *fpmb,
	struct net_device *dev)
{
	struct bcm_nethooks *nethooks = bcm_nethooks_priv();
	int	ret;

	pr_debug("%s:-->\n",dev->name);
	if (nethooks->sig != BCM_NETHOOKS_SIG || !nethooks->fpm_tx) {
		netdev_err(dev, "Attempt to call BCM nethook TX FPM function ");
		netdev_err(dev, "on non-nethook device or nethook device ");
		netdev_err(dev, "with NULL TX FPM function!\n");
		return -EIO;
	}
	ret = nethooks->fpm_tx(fpmb, dev);
	pr_debug("%s:<--\n",dev->name);
	return ret;
}

/*
 * BCM Nethooks Driver Functions
 *
 * The following functions are for use by drivers supporting
 * BCM nethooks.
 *
 */
static inline void bcm_nethook_dev_init(
	struct net_device *dev, fpm_tx_fn fpm_tx)
{
	struct bcm_nethooks *nethooks = bcm_nethooks_priv();
	int i;

	pr_debug("-->\n");
	memset(nethooks, 0, sizeof(struct bcm_nethooks));
	nethooks->sig = BCM_NETHOOKS_SIG;
	for (i = 0; i < BCM_NETHOOK_TYPE_MAX; i++) {
		INIT_LIST_HEAD(&nethooks->hooks[i].list);
		rwlock_init(&nethooks->hook_locks[i]);
	}
	nethooks->fpm_tx = fpm_tx;
	nethooks->offload_stats.fast =
		alloc_percpu(struct pcpu_sw_netstats);
	nethooks->offload_stats.slow =
		alloc_percpu(struct pcpu_sw_netstats);
	nethooks->offload_stats.mcast_fast =
		alloc_percpu(struct pcpu_sw_netstats);
	for (i = 0; i < DSCP_MAX_LIMIT; i++) {
		nethooks->offload_stats.dscp_fast[i] =
			alloc_percpu(struct pcpu_sw_netstats);
		nethooks->offload_stats.dscp_slow[i] =
			alloc_percpu(struct pcpu_sw_netstats);
	}
	pr_debug("<--\n");
}

static inline int bcm_nethook_dev_mcast_init(
	struct net_device *dev,
	flow_mdb_counter_update_fn flow_mdb_counter_update)
{
	struct bcm_nethooks *nethooks;

	pr_debug("-->\n");
	if (!dev) {
		pr_err("Invalid Device provided!\n");
		return -EINVAL;
	}
	nethooks = bcm_nethooks_priv();
	if (nethooks->sig != BCM_NETHOOKS_SIG) {
		netdev_err(dev, "Invalid parameter provided!\n");
		return -EINVAL;
	}
	nethooks->flow_mdb_counter_update = flow_mdb_counter_update;

	pr_debug("<--\n");
	return 0;
}

static inline void bcm_nethook_dev_fini(
	struct net_device *dev)
{
	struct bcm_nethooks *nethooks = bcm_nethooks_priv();
	int i;

	pr_debug("-->\n");
	for (i=0; i<DSCP_MAX_LIMIT; i++) {
		if (nethooks->offload_stats.dscp_fast[i]) {
			free_percpu(nethooks->offload_stats.dscp_fast[i]);
			nethooks->offload_stats.dscp_fast[i] = NULL;
		}
		if (nethooks->offload_stats.dscp_slow[i]) {
			free_percpu(nethooks->offload_stats.dscp_slow[i]);
			nethooks->offload_stats.dscp_slow[i] = NULL;
		}
	}
	if (nethooks->offload_stats.mcast_fast) {
		free_percpu(nethooks->offload_stats.mcast_fast);
		nethooks->offload_stats.mcast_fast = NULL;
	}
	if (nethooks->offload_stats.fast) {
		free_percpu(nethooks->offload_stats.fast);
		nethooks->offload_stats.fast = NULL;
	}
	if (nethooks->offload_stats.slow) {
		free_percpu(nethooks->offload_stats.slow);
		nethooks->offload_stats.slow = NULL;
	}
	pr_debug("<--\n");
}

static inline int bcm_nethook_dev_mcast_fini(
	struct net_device *dev)
{
	struct bcm_nethooks *nethooks;

	pr_debug("-->\n");
	if (!dev) {
		pr_err("Invalid Device provided!\n");
		return -EINVAL;
	}
	nethooks = bcm_nethooks_priv();

	nethooks->flow_mdb_counter_update = NULL;

	pr_debug("<--\n");
	return 0;
}

static inline enum bcm_nethook_result bcm_nethook_call_hooks(
	struct net_device *dev, enum bcm_nethook_type type, void *buf)
{
	enum bcm_nethook_result result = BCM_NETHOOK_PASS;
	struct bcm_nethooks *nethooks = bcm_nethooks_priv();
	struct bcm_nethook *nethook;

	pr_debug("%s:-->\n",dev->name);
	read_lock(&nethooks->hook_locks[type]);
	list_for_each_entry(nethook, &nethooks->hooks[type].list, list) {
		if (!nethook->enabled)
			continue;
		pr_debug("hook:%s\n", nethook->name);
		result = nethook->hook(dev, type, buf);
		if (result < BCM_NETHOOK_MAX_RESULT)
			nethook->count[result]++;
		switch (result) {
		case BCM_NETHOOK_PASS:
			continue;
		case BCM_NETHOOK_DROP:
		case BCM_NETHOOK_SKIP:
		case BCM_NETHOOK_CONSUMED:
			goto done;
		default:
			netdev_err(dev, "Erroneous result returned from ");
			netdev_err(dev, "nethook %s.\n", nethook->name);
			netdev_err(dev, "Passing packet to next nethook.\n");
			continue;
		}
	}

done:
	read_unlock(&nethooks->hook_locks[type]);
	pr_debug("%s:<--\n",dev->name);
	return result;
}

extern int bcm_nethook_dev(struct net_device *dev);

extern int bcm_nethook_register_hook_devs(enum bcm_nethook_type type,
					  enum bcm_nethook_prio priority,
					  char *name, bcm_nethook_fn hook);

extern int bcm_nethook_unregister_hook_devs(enum bcm_nethook_type type,
					    bcm_nethook_fn hook);

extern int bcm_nethook_enable_hook_devs(enum bcm_nethook_type type,
					bcm_nethook_fn hook, bool enable);
#endif
