 /****************************************************************************
 *
 * Copyright (c) 2015 Broadcom Corporation
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************/
#include <linux/kernel.h>
#include <linux/netdevice.h>

#include "ethsw.h"
#include "ethsw_priv.h"

/*
 * Log Utilities
 */

/* define log module */
#define LOG_MODULE "ethsw_vnet"

#if VNET_IF_SUPPORT

struct ethsw_vnet_device {
	u16 port_mask;
};

struct ethsw_vnet_port {
	int port_id;
};

/* ethsw vnet control block */
struct ethsw_vnet {
	/* net device for switch */
	struct net_device *dev;

	/* net device for each port */
	struct net_device *port[ETHSW_PORT_MAX];
};

void ethsw_vnet_port_cb(
	void *cb_priv,
	struct ethsw_port_status *port_status)
{
	struct net_device *port;
	struct ethsw_vnet_port *vport;

	FUNC_ENTER();

	/* get net device and private data */
	port = (struct net_device *)cb_priv;
	vport = netdev_priv(port);

	LOG_INFO("vnet port %s status update: "
			 "link_up = %d, full_duplex = %d, speed = %d\n",
			 port->name,
			 port_status->link_up,
			 port_status->full_duplex,
			 port_status->speed);

	FUNC_LEAVE();
}

void ethsw_vnet_device_init(struct net_device *dev)
{
	struct ethsw_vnet_device *vdev;

	FUNC_ENTER();

	/* get private data */
	vdev = netdev_priv(dev);

	FUNC_LEAVE();
}

void ethsw_vnet_port_init(struct net_device *port)
{
	struct ethsw_vnet_port *vport;

	FUNC_ENTER();

	/* get private data */
	vport = netdev_priv(port);

	FUNC_LEAVE();
}

int ethsw_vnet_init(struct ethsw_device *swdev)
{
	int ret = 0;
	int i;
	struct ethsw_vnet *swvnet;

	FUNC_ENTER();

	/* alloc ethsw vnet control block */
	swvnet = kzalloc(sizeof(struct ethsw_vnet), GFP_KERNEL);
	if (!swvnet) {
		LOG_ERR("Ethsw vnet control block alloc failed\n");
		ret = -ENOMEM;
		goto ERROR;
	}

	swdev->swvnet = swvnet;

	/* create net device for switch */
	swvnet->dev = alloc_netdev(
		sizeof(struct ethsw_vnet_device),
		"swvnet",
		ethsw_vnet_device_init);

	if (!swvnet->dev) {
		LOG_ERR("Ethsw vnet net device alloc failed\n");
		ret = -ENOMEM;
		goto ERROR;
	}

	/* alloc and connect net devices for all ports */
	for (i = 0; i < ETHSW_PORT_MAX; i++) {
		/* PHY ports only */
		if (swdev->port[i].type == PHY_PORT) {
			struct net_device *port;
			struct ethsw_vnet_port *vport;

			/* alloc net device */
			swvnet->port[i] = alloc_netdev(
				sizeof(struct ethsw_vnet_port),
				swdev->port[i].name,
				ethsw_vnet_port_init);

			if (!swvnet->port[i]) {
				LOG_ERR("Ethsw vnet port net device alloc failed\n");
				ret = -ENOMEM;
				goto ERROR;
			}

			port = swvnet->port[i];
			vport = netdev_priv(port);

			/* connect to switch port */
			if (ethsw_port_connect(
					port->name,
					&vport->port_id,
					ethsw_vnet_port_cb,
					port)) {
				LOG_ERR("Ethsw vnet port connect failed\n");
				vport->port_id = ETHSW_PORT_ID_INVALID;

				/* continue on */
				continue;
			}

			/* enable port */
			if (ethsw_port_enable(
					vport->port_id)) {
				LOG_ERR("Ethsw vnet port enable failed\n");

				/* continue on */
				continue;
			}
		}
	}

	LOG_INFO("Ethsw vnet initialized\n");

	FUNC_LEAVE();
	return(ret);

 ERROR:
	/* exit vnet properly */
	ethsw_vnet_exit(swdev);

	FUNC_LEAVE();
	return(ret);
}

void ethsw_vnet_exit(struct ethsw_device *swdev)
{
	int i;
	struct ethsw_vnet *swvnet;

	FUNC_ENTER();

	swvnet = swdev->swvnet;
	if (!swvnet) {
		goto EXIT;
	}

	/* free net devices for ports */
	for (i = 0; i < ETHSW_PORT_MAX; i++) {
		if (swvnet->port[i]) {
			struct net_device *port;
			struct ethsw_vnet_port *vport;

			port = swvnet->port[i];
			vport = netdev_priv(port);

			if (vport->port_id != ETHSW_PORT_ID_INVALID) {
				/* disconnect from switch port */
				ethsw_port_disconnect(vport->port_id);
				vport->port_id = ETHSW_PORT_ID_INVALID;
			}

			/* free net device */
			free_netdev(swvnet->port[i]);
			swvnet->port[i] = NULL;
		}
	}

	/* free net device for switch */
	free_netdev(swvnet->dev);
	swvnet->dev = NULL;

	/* free ethsw vnet control block */
	kfree(swvnet);
	swdev->swvnet = NULL;

	LOG_INFO("Ethsw vnet exited\n");

 EXIT:
	FUNC_LEAVE();
}

#endif
