 /****************************************************************************
 *
 * Copyright (c) 2022 Broadcom. All rights reserved
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************/
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/string.h>

#include "ethsw.h"
#include "ethsw_priv.h"
#include "ethsw_core.h"
#include "mso_feat.h"
#include "regops.h"

#define LOG_MODULE "ethsw_core"

static struct mso_feat *feat;

static int mso_feat_ethiso_init(void)
{
	get_mso_feat_fptr get_mso_feat;

	if (!feat) {
		get_mso_feat = get_mso_feat_symbol();
		if (!get_mso_feat) {
			pr_alert("ETHSW: mso_feat module not loaded\n");
			return -1;
		}

		feat = get_mso_feat(mf_ethiso);
		if (!feat) {
			pr_alert("ETHSW: Isolate feature not supported\n");
			return -1;
		}
	}
	return 0;
}

static inline void pr_debug_regops(int i, struct regops *ops)
{
	pr_debug("%d: %d %d %d 0x%x %d %s\n", i,
	       ops->code, ops->page,
	       ops->addr, ops->data16,
	       ops->size, ops->name);
}

static inline void pr_debug_img(struct regops_img *img)
{
	int i;

	pr_debug("%s: %d %d\n", img->name, img->type, img->nops);
	for (i = 0; i < img->nops; i++)
		pr_debug_regops(i, &img->ops[i]);
}

int flash_regops_img(struct ethsw_device *swdev, struct regops_img *img)
{
	int i, loop;
	int ret = 0;
	unsigned char  data8;
	unsigned short data16;
	unsigned int   data32;

	pr_debug("%s: %d %d\n", img->name, img->type, img->nops);
	for (i = 0; i < img->nops; i++) {
		pr_debug_regops(i, &img->ops[i]);
		switch (img->ops[i].code) {
		case REG_WR8_OP:
			ethsw_creg_write(swdev, img->ops[i].page,
					 img->ops[i].addr,
					 (unsigned char *)&(img->ops[i].data8),
					 img->ops[i].size);
			break;
		case REG_RD_OR_WR8_OP:
			ethsw_creg_read(swdev, img->ops[i].page,
					img->ops[i].addr,
					(unsigned char *)&(data8),
					img->ops[i].size);
			data8 |= img->ops[i].data8;
			ethsw_creg_write(swdev, img->ops[i].page,
					 img->ops[i].addr,
					 (unsigned char *)&(data8),
					 img->ops[i].size);
			break;
		case REG_RD_AND_WR8_OP:
			ethsw_creg_read(swdev, img->ops[i].page,
					img->ops[i].addr,
					(unsigned char *)&(data8),
					img->ops[i].size);
			data8 &= img->ops[i].data8;
			ethsw_creg_write(swdev, img->ops[i].page,
					 img->ops[i].addr,
					 (unsigned char *)&(data8),
					 img->ops[i].size);
			break;
		case REG_RD_WAIT8_OP:
			loop = 100;
			do {
				udelay(10);
				ethsw_creg_read(swdev, img->ops[i].page,
						img->ops[i].addr,
						(unsigned char *)&(data8),
						img->ops[i].size);
				loop--;
			} while ((data8 & img->ops[i].data8) && loop);
			break;
		case REG_WR16_OP:
			ethsw_creg_write(swdev, img->ops[i].page,
					 img->ops[i].addr,
					 (unsigned char *)&(img->ops[i].data16),
					 img->ops[i].size);
			break;
		case REG_RD_OR_WR16_OP:
			ethsw_creg_read(swdev, img->ops[i].page,
					img->ops[i].addr,
					(unsigned char *)&(data16),
					img->ops[i].size);
			data16 |= img->ops[i].data16;
			ethsw_creg_write(swdev, img->ops[i].page,
					 img->ops[i].addr,
					 (unsigned char *)&(data16),
					 img->ops[i].size);
			break;
		case REG_RD_AND_WR16_OP:
			ethsw_creg_read(swdev, img->ops[i].page,
					img->ops[i].addr,
					(unsigned char *)&(data16),
					img->ops[i].size);
			data16 &= img->ops[i].data16;
			ethsw_creg_write(swdev, img->ops[i].page,
					 img->ops[i].addr,
					 (unsigned char *)&(data16),
					 img->ops[i].size);
			break;
		case REG_RD_WAIT16_OP:
			loop = 100;
			do {
				udelay(10);
				ethsw_creg_read(swdev, img->ops[i].page,
						img->ops[i].addr,
						(unsigned char *)&(data16),
						img->ops[i].size);
				loop--;
			} while ((data16 & img->ops[i].data16) && loop);
			break;
		case REG_WR32_OP:
			ethsw_creg_write(swdev, img->ops[i].page,
					 img->ops[i].addr,
					 (unsigned char *)&(img->ops[i].data32),
					 img->ops[i].size);
			break;
		case REG_RD_OR_WR32_OP:
			ethsw_creg_read(swdev, img->ops[i].page,
					img->ops[i].addr,
					(unsigned char *)&(data32),
					img->ops[i].size);
			data32 |= img->ops[i].data32;
			ethsw_creg_write(swdev, img->ops[i].page,
					 img->ops[i].addr,
					 (unsigned char *)&(data32),
					 img->ops[i].size);
			break;
		case REG_RD_AND_WR32_OP:
			ethsw_creg_read(swdev, img->ops[i].page,
					img->ops[i].addr,
					(unsigned char *)&(data32),
					img->ops[i].size);
			data32 &= img->ops[i].data32;
			ethsw_creg_write(swdev, img->ops[i].page,
					 img->ops[i].addr,
					 (unsigned char *)&(data32),
					 img->ops[i].size);
			break;
		case REG_RD_WAIT32_OP:
			loop = 100;
			do {
				udelay(10);
				ethsw_creg_read(swdev, img->ops[i].page,
						img->ops[i].addr,
						(unsigned char *)&(data32),
						img->ops[i].size);
				loop--;
			} while ((data32 & img->ops[i].data32) && loop);
			break;
		default:
			break;
		}
	}
	return ret;
}

int ethsw_port_isolate_sf2(struct ethsw_device *swdev, int imp_port,
			   int lan_port, bool enable)
{
	struct ethsw_core *swcore;
	struct regops_img img;
	int ign_port = 0;
	int ret = -1;

	if (!feat || !feat->ethiso || !feat->ethiso->ops1)
		return -1;

	img.nops_alloc = feat->ethiso->ops1();
	img.ops = kzalloc(img.nops_alloc*sizeof(struct regops),
			  GFP_KERNEL);
	if (!img.ops) {
		pr_debug("mso_feat ethsw alloc img ops buffer failed\n");
		return -1;
	}

	swcore = swdev->swcore;

	if (swcore->altwan_enble) {
		ign_port  = (1<<swcore->altwan_impport);
		ign_port |= (1<<swcore->altwan_lanport);
	}
	if (enable && feat && feat->ethiso && feat->ethiso->ops3)
		ret = feat->ethiso->ops3(imp_port, lan_port, ign_port, &img);
	else if (!enable && feat && feat->ethiso && feat->ethiso->ops4)
		ret = feat->ethiso->ops4(imp_port, lan_port, ign_port, &img);

	//pr_debug_img(&img);
	if (ret == 0)
		flash_regops_img(swdev, &img);
	else
		pr_debug("ETHSW: Isolate feature operation 3 or 4 failed\n");

	if (feat && feat->ethiso && feat->ethiso->ops2)
		pr_debug("ETHSW: Isolate feature operation 2 - %x\n",
			 feat->ethiso->ops2());


	kfree(img.ops);

	return 0;
}

int ethsw_port_isolate_core(struct ethsw_device *swdev, int imp_port,
			    int lan_port, bool enable)
{
	struct ethsw_core *swcore;
	int ret = 0;

	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);
	swcore = swdev->swcore;
	VALIDATE_SWCORE(swdev);

	if (swdev != ethsw_dev[INTERNAL_SW])
		return ret;

	ret = ethsw_port_isolate_sf2(swdev, imp_port,
				     lan_port - swdev->subid_port_offset,
				     enable);

	return ret;
}

int ethsw_port_isolate(int imp_port, int lan_port, bool enable)
{
	struct ethsw_device *swdev = subid_to_swdev_table[lan_port];

	if (!swdev)
		swdev = ethsw_dev[INTERNAL_SW];

	if (mso_feat_ethiso_init())
		return -1;

	if (feat) {
		if (!feat->ethiso || !feat->ethiso->ops0) {
			pr_alert("ETHSW: Isolate feature operation not supported\n");
			return -1;
		}

		if (feat->ethiso->ops0()) {
			pr_alert("ETHSW: Isolate feature not allowed\n");
			return -1;
		}
	}

	return ethsw_port_isolate_core(swdev, imp_port, lan_port, enable);
}
EXPORT_SYMBOL(ethsw_port_isolate);
