 /****************************************************************************
 *
 * Copyright (c) 2015 Broadcom Corporation
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Piotr Romanus <promanus@broadcom.com>
 ****************************************************************************/

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/rculist.h>
#include <linux/spinlock.h>
#include <linux/times.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/jhash.h>
#include <linux/random.h>
#include <linux/slab.h>
#include <linux/atomic.h>
#include <asm/unaligned.h>

#include "ethsw.h"
#include "ethsw_priv.h"
#include "ethsw_db.h"

/* define log module */
#define LOG_MODULE "ethsw_db"

/* Following code is derived from flowmgr_db.c implementation */

#define ETHSW_HASH_BITS 8
#define ETHSW_HASH_SIZE (1 << ETHSW_HASH_BITS)

static spinlock_t			hash_lock;
static struct hlist_head		hash[ETHSW_HASH_SIZE];

static struct kmem_cache *ethsw_db_cache __read_mostly = NULL;

static u32 db_salt __read_mostly;

int ethsw_db_init(void)
{
	if (ethsw_db_cache)
		return 0;

	ethsw_db_cache = kmem_cache_create("ethsw_db_cache",
					     sizeof(struct ethsw_db_entry),
					     0,
					     SLAB_HWCACHE_ALIGN, NULL);
	if (!ethsw_db_cache)
		return -ENOMEM;

	get_random_bytes(&db_salt, sizeof(db_salt));

	return 0;
}

void ethsw_db_fini(void)
{
	kmem_cache_destroy(ethsw_db_cache);
}

static inline int ethsw_mac_hash(const unsigned char *mac)
{
	/* use 1 byte of OUI cnd 3 bytes of NIC */
	u32 key = get_unaligned((u32 *)(mac + 2));
	return jhash_1word(key, db_salt) & (ETHSW_HASH_SIZE - 1);
}

static void db_rcu_free(struct rcu_head *head)
{
	struct ethsw_db_entry *ent
		= container_of(head, struct ethsw_db_entry, rcu);
	kmem_cache_free(ethsw_db_cache, ent);
}

static void db_delete(struct ethsw_db_entry *f)
{
#ifdef ETHSW_SPI_SWITCH_ACCESS
	__ethsw_arl_cache_table_delete(f->addr, f->vlanid, f->portid);
#endif
	if (f && atomic_dec_and_test(&f->use)) {
		hlist_del_rcu(&f->hlist);
		call_rcu(&f->rcu, db_rcu_free);
	}
}


/* Completely flush all dynamic entries in client database */
void ethsw_db_flush(void)
{
	int i;
	unsigned long flags;

	spin_lock_irqsave(&hash_lock, flags);
	for (i = 0; i < ETHSW_HASH_SIZE; i++) {
		struct ethsw_db_entry *f;
		struct hlist_node *n;
		hlist_iterate_safe(f, n, &hash[i], hlist) {
			db_delete(f);
		}
	}
	spin_unlock_irqrestore(&hash_lock, flags);
}

/* Flush all entries referring to a specific port. */
void ethsw_db_delete_by_port(int portid)
{
	int i;
	unsigned long flags;

	FUNC_ENTER();

	spin_lock_irqsave(&hash_lock, flags);
	for (i = 0; i < ETHSW_HASH_SIZE; i++) {
		struct hlist_node *h, *g;

		hlist_for_each_safe(h, g, &hash[i]) {
			struct ethsw_db_entry *f;
			f = hlist_entry(h, struct ethsw_db_entry, hlist);
			if (f->portid != portid)
				continue;

			LOG_DEBUG("deleted by port %u mac %pM vid %u\n",
				  f->portid, f->addr, f->vlanid);

			db_delete(f);
		}
	}
	spin_unlock_irqrestore(&hash_lock, flags);
	FUNC_LEAVE();
}

/* Flush all entries older than age jiffies
 * if do_all is set also flush static entries
 */
void ethsw_db_delete_by_age(int age)
{
	int i;
	unsigned long flags;

	spin_lock_irqsave(&hash_lock, flags);
	for (i = 0; i < ETHSW_HASH_SIZE; i++) {
		struct hlist_node *h, *g;

		hlist_for_each_safe(h, g, &hash[i]) {
			struct ethsw_db_entry *f;
			f = hlist_entry(h, struct ethsw_db_entry, hlist);
			if (age > (jiffies - f->used))
				continue;

			LOG_DEBUG("age %d, jiffies %lu, used %lu, mac %pM\n",
				 age, jiffies, f->used, f->addr);
			db_delete(f);
		}
	}
	spin_unlock_irqrestore(&hash_lock, flags);
}

/* Internal Function: Find entry based on mac address in specific list */
static struct ethsw_db_entry *db_find(struct hlist_head *head,
				      const unsigned char *addr, u16 vid)
{
	struct ethsw_db_entry *db;

	hlist_iterate(db, head, hlist) {
		if (ether_addr_equal(db->addr, addr) && db->vlanid == vid)
			return db;
	}
	return NULL;
}

/* Find entry based on mac address */
struct ethsw_db_entry *ethsw_db_find(const unsigned char *addr, u16 vid)
{
	struct hlist_head *head = &hash[ethsw_mac_hash(addr)];
	struct ethsw_db_entry *db = NULL;
	unsigned long flags;

	spin_lock_irqsave(&hash_lock, flags);
	db = db_find(head, addr, vid);
	if (db)
		db->used = jiffies;
	spin_unlock_irqrestore(&hash_lock, flags);
	return db;
}

/* Internal Function: Create entry based on input interface and mac address
   in specific list */
static struct ethsw_db_entry *db_create(struct hlist_head *head,
					  int portid,
					  const unsigned char *addr, u16 vid)
{
	struct ethsw_db_entry *db;

	db = kmem_cache_alloc(ethsw_db_cache, GFP_ATOMIC);
	if (db) {
		memcpy(db->addr, addr, ETH_ALEN);
		db->portid = portid;
		atomic_set(&db->use, 1);
		db->used = jiffies;
		db->vlanid = vid;
		hlist_add_head_rcu(&db->hlist, head);
	}
	return db;
}

/* Internal Function: Delete entry based on mac address */
static int db_delete_by_addr(const u8 *addr, u16 vid)
{
	struct hlist_head *head = &hash[ethsw_mac_hash(addr)];
	struct ethsw_db_entry *db;

	db = db_find(head, addr, vid);
	if (!db)
		return -ENOENT;

	db_delete(db);
	return 0;
}

/* Delete entry based on mac address */
int ethsw_db_delete(const unsigned char *addr, u16 vid)
{
	int err;
	unsigned long flags;

	spin_lock_irqsave(&hash_lock, flags);
	err = db_delete_by_addr(addr, vid);
	spin_unlock_irqrestore(&hash_lock, flags);
	LOG_DEBUG("Removed mac %pM vid %u\n", addr, vid);
	return err;
}

/* Internal Function: Insert entry based on input interface and mac address */
static int db_insert(int portid, const unsigned char *addr, u16 vid)
{
	struct hlist_head *head = &hash[ethsw_mac_hash(addr)];
	struct ethsw_db_entry *db;

	if (!is_valid_ether_addr(addr))
		return -EINVAL;

	db = db_find(head, addr, vid);
	if (!db)
		db = db_create(head, portid, addr, vid);
	if (!db)
		return -ENOMEM;

	return 0;
}

/* Insert entry based on input interface and mac address */
int ethsw_db_insert(int portid, const unsigned char *addr, u16 vid)
{
	int ret;
	unsigned long flags;

	spin_lock_irqsave(&hash_lock, flags);
	ret = db_insert(portid, addr, vid);
	spin_unlock_irqrestore(&hash_lock, flags);
	LOG_DEBUG("Added mac %pM vid %u port %u\n", addr, vid, portid);

	return ret;
}

/* Internal Function: Update entry based on input interface and mac address */
static int db_update(int portid, const unsigned char *addr, u16 vid)
{
	struct hlist_head *head = &hash[ethsw_mac_hash(addr)];
	struct ethsw_db_entry *db;

	if (!is_valid_ether_addr(addr))
		return -EINVAL;

	db = db_find(head, addr, vid);
	db->portid = portid;
	db->used = jiffies;

	return 0;
}

/* Update entry based on input interface and mac address */
int ethsw_db_update(int portid, const unsigned char *addr, u16 vid)
{
	int ret;
	unsigned long flags;

	spin_lock_irqsave(&hash_lock, flags);
	ret = db_update(portid, addr, vid);
	spin_unlock_irqrestore(&hash_lock, flags);
	return ret;
}

/* Get first entry in specific list */
struct hlist_node  *ethsw_db_get_first(int *hashid)
{
	struct hlist_node *h;
	int i;

	for (i = 0; i < ETHSW_HASH_SIZE; i++) {
		h = rcu_dereference(hlist_first_rcu(&hash[i]));
		if (h) {
			*hashid = i;
			return h;
		}
	}
	return NULL;
}

/* Get next entry in specific list */
struct hlist_node *ethsw_db_get_next(int *hashid,
				       struct hlist_node *head)
{
	head = rcu_dereference(hlist_next_rcu(head));
	while (!head) {
		if (++*hashid >= ETHSW_HASH_SIZE)
			return NULL;
		head = rcu_dereference(
				hlist_first_rcu(
				   &hash[*hashid]));
	}
	return head;
}

/* Get index in specific list */
struct hlist_node *ethsw_db_get_idx(int *hashid, loff_t pos)
{
	struct hlist_node *head = ethsw_db_get_first(hashid);

	if (head)
		while (pos && (head = ethsw_db_get_next(hashid, head)))
			pos--;
	return pos ? NULL : head;
}
