 /****************************************************************************
 *
 * Copyright (c) 2015-2019 Broadcom. All rights reserved
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Zibin Yu <zibin.yu@broadcom.com>
 ****************************************************************************/

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/device.h>
#include <linux/netdevice.h>
#include <linux/uaccess.h>
#include <linux/if_vlan.h>
#include <net/netfilter/nf_conntrack_offload.h>
#include <linux/mutex.h>

#include "flowmgr.h"
#include "flowmgr_fap.h"
#include "flowmgr_fap_ops.h"
#include "flowmgr_tunnel.h"
#include <uapi/linux/bcm_media_gw/flowmgr/flowmgr_ioctl.h>

#define FLOWMGR_NAME "flowmgr"
static DEFINE_MUTEX(flowmgr_mutex);

static struct class		*flowmgr_class;
static int				flowmgr_major;
static dev_t			flowmgr_dev;
static struct cdev		flowmgr_cdev;
static struct device		*flowmgr_device;

static int flowmgr_open(struct inode *inode, struct file *file);
static int flowmgr_release(struct inode *inode, struct file *file);
static long flowmgr_ioctl(struct file *file, unsigned int cmd, unsigned long arg);
#ifdef CONFIG_COMPAT
static long flowmgr_compat_ioctl(struct file *file, unsigned int cmd, unsigned long arg);
#endif
static inline
struct nf_conn *nf_conn_find_by_flow_tuple(struct flow_tuple tuple, enum ip_conntrack_dir *dir)
{
	struct nf_conntrack_tuple t;
	struct nf_conntrack_tuple_hash *h;
	int len = (tuple.l3_prot == AF_INET)? 4 : 16;
	memset(&t, 0, sizeof(t));
	t.src.l3num = tuple.l3_prot;
	memcpy(&t.src.u3, &tuple.src_ip, len);
	memcpy(&t.dst.u3, &tuple.dst_ip, len);
	t.dst.protonum = tuple.ip_prot;
	t.src.u.all = tuple.src_port;
	t.dst.u.all = tuple.dst_port;
	h = nf_conntrack_find_get(&init_net, &nf_ct_zone_dflt, &t);
	if (!h)
		return NULL;
	if (dir)
		*dir = NF_CT_DIRECTION(h);
	return nf_ct_tuplehash_to_ctrack(h);
}

static inline
void update_ct_offload(int flow_id, struct flow_params *params,
		       struct offload_info *ct_offload_info,
		       struct nf_conn_offload *ct_offload,
		       struct nf_conn_offload *ct_offload_master)
{
	ct_offload_info->flow_id = flow_id;
	if (params->type == ft_ipv6)
		ct_offload_info->flow_type = ft_ipv6 << 16 | params->ipv6.type;
	else
		ct_offload_info->flow_type = ft_ipv4 << 16 | params->ipv4.type;
	ct_offload_info->oif = params->tx.interface;
	ct_offload_info->iif = params->rx.interface;
	ct_offload_info->vlan_id = params->tx.control.vlan_tci;
	ct_offload_info->lag = params->tx.control.lag;
	ct_offload_info->vlan_untag = params->rx.control.vlan_untag;
	ct_offload_info->expected = 0;
	ct_offload->destructor = flowmgr_demote;
	ct_offload->update_stats = flowmgr_update_stats;
	atomic_inc(&flowmgr.flow_create_cnt);
	if (ct_offload_master)
		atomic_inc(&(ct_offload_master->slavecnt));
	if (flowmgr.flow_usage)
		flowmgr.flow_usage[flow_id].add++;
}

static const struct file_operations flowmgr_fops = {
	.owner =			THIS_MODULE,
	.open =			flowmgr_open,
	.release =		flowmgr_release,
	.unlocked_ioctl =	flowmgr_ioctl,
#ifdef CONFIG_COMPAT
	.compat_ioctl =		flowmgr_compat_ioctl,
#endif
};

static int flowmgr_open(struct inode *inode, struct file *file)
{
	return 0;
}

static int flowmgr_release(struct inode *inode, struct file *file)
{
	return 0;
}

static long flowmgr_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	int ret = 0;
	bool locked = 0;

	switch (cmd) {
	case FLOWMGR_IOCTL_FLOW_ADD: {
		struct flow_config configs;
		struct flow_params params;
		struct net_device *rxdev;
		struct net_device *txdev;
		struct nf_conn *ct = NULL;
		enum ip_conntrack_dir dir;
		struct nf_conn_offload *ct_offload = NULL;
		struct nf_conn_offload *ct_offload_master = NULL;
		struct offload_info *ct_offload_info;
		int flow_id = -1;

		if (flowmgr.promote_mode == AUTO_PROMOTE) {
			pr_err("%s: Operation not supported in auto promote mode\n", __func__);
			ret = -EPERM;
			break;
		}

		ret = copy_from_user(&configs, (void *)arg, sizeof(configs));
		if (ret) {
			pr_err("%s: can't copy flow_config from userspace\n", __func__);
			break;
		}
		if (configs.tuple.l3_prot != AF_INET && configs.tuple.l3_prot != AF_INET6) {
			pr_err("%s: Only ipv4/ipv6 flow type supported\n", __func__);
			ret = -EINVAL;
			break;
		}

		mutex_lock(&flowmgr_mutex);
		locked = 1;

		/* Look up ct_offload by flow params */
		ct = nf_conn_find_by_flow_tuple(configs.tuple, &dir);
		if (!ct) {
			pr_err("%s: no matching ct by flow params\n", __func__);
			ret = -EINVAL;
			break;
		}
		ct_offload = nf_conn_offload_find(ct);
		if (!ct_offload) {
			pr_err("%s: ct_offload not found\n", __func__);
			ret = -1;
			break;
		}
		ct_offload_info = &ct_offload->info[dir];
		if (ct_offload_info->flow_id != -1 && !ct_offload_info->expected) {
			pr_err("%s: flow %d already added\n", __func__, ct_offload_info->flow_id);
			ret = -EINVAL;
			break;
		}
		if (ct->master)
			ct_offload_master = nf_conn_offload_find(ct->master);

		/* demote expected flow */
		flowmgr_expected_demote(ct_offload_info);

		/* map flow_config to flow_params */
		memset(&params, 0, sizeof(params));
		if (configs.tuple.l3_prot == AF_INET6)
		{
			params.type = ft_ipv6;
			params.ipv6.type = configs.flow_type;
			params.ipv6.ip_prot = configs.tuple.ip_prot;
			memcpy((char *)params.ipv6.src_ip,
				configs.tuple.src_ip.in6.s6_addr,
				sizeof(struct in6_addr));
			memcpy((char *)params.ipv6.dst_ip,
				configs.tuple.dst_ip.in6.s6_addr,
				sizeof(struct in6_addr));
			params.ipv6.src_port = configs.tuple.src_port;
			params.ipv6.dst_port = configs.tuple.dst_port;
			memcpy(params.ipv6.replacement_mac_src, configs.replacement_mac_src, ETH_ALEN);
			memcpy(params.ipv6.replacement_mac_dst, configs.replacement_mac_dst, ETH_ALEN);
		} else {
			params.type = ft_ipv4;
			params.ipv4.type = configs.flow_type;
			params.ipv4.ip_prot = configs.tuple.ip_prot;
			params.ipv4.src_ip = configs.tuple.src_ip.in.s_addr;
			params.ipv4.dst_ip = configs.tuple.dst_ip.in.s_addr;
			params.ipv4.src_port = configs.tuple.src_port;
			params.ipv4.dst_port = configs.tuple.dst_port;
			params.ipv4.replacement_ip = configs.replacement_ip;
			params.ipv4.replacement_port = configs.replacement_port;
			memcpy(params.ipv4.replacement_mac_src, configs.replacement_mac_src, ETH_ALEN);
			memcpy(params.ipv4.replacement_mac_dst, configs.replacement_mac_dst, ETH_ALEN);
		}

		/* configure interfaces */
		rxdev = __dev_get_by_index(&init_net, configs.rx_ifindex);
		txdev = __dev_get_by_index(&init_net, configs.tx_ifindex);
		if (!rxdev || !txdev) {
			pr_err("%s: rx net device (or) tx net device invalid\n", __func__);
			ret = -EINVAL;
			break;
    	}
		if (is_vlan_dev(rxdev) && !is_vlan_dev(txdev)) {
			params.tx.control.vlan_tci = 0;
			params.rx.control.vlan_untag = 1;
			params.tx.interface = configs.tx_ifindex;
			params.rx.interface = vlan_dev_priv(rxdev)->real_dev->ifindex;
		} else if (!is_vlan_dev(rxdev) && is_vlan_dev(txdev)) {
			params.tx.control.vlan_tci = vlan_dev_vlan_id(txdev);
			params.rx.control.vlan_untag = 0;
			params.tx.interface = vlan_dev_priv(txdev)->real_dev->ifindex;
			params.rx.interface = configs.rx_ifindex;
		} else {
			params.tx.interface = configs.tx_ifindex;
			if (params.ipv4.type == ft_ipv4_dslite_decap || params.ipv4.type == ft_ipv4_gre_decap)
				params.rx.interface = configs.tunnel.ifindex;
			else
				params.rx.interface = configs.rx_ifindex;
		}
		if (configs.tx_dscp) {
			params.tx.control.dscp_mark = 1;
			params.tx.control.dscp_val = configs.tx_dscp;
		}

		/* configure tunnel */
		if (params.type == ft_ipv4) {
			struct flow_tunnel_params *encap = &params.encap_tun;
			struct flow_tunnel_params *decap = &params.decap_tun;
			struct flowmgr_tdb_entry *db;

			if (params.ipv4.type == ft_ipv4_gre_decap || params.ipv4.type == ft_ipv4_gre_encap) {
				db = flowmgr_tdb_find_by_raddr(&configs.tunnel.remote_ip.ip);
				if (!db) {
					pr_err("%s: no matching db for gre remote ip %pI4\n", __func__, &configs.tunnel.remote_ip.ip);
					ret = -1;
					break;
				}

				encap->gre.tunnel_id = db->id;
				decap->gre.tunnel_id = db->id;
				if (params.ipv4.type == ft_ipv4_gre_encap) {
					encap->type = ARPHRD_ETHER;
					memcpy(encap->gre.remote_mac, db->parms.eh.h_dest,
				       	ETH_ALEN);
					memcpy(encap->gre.local_mac, db->parms.eh.h_source,
				       	ETH_ALEN);
					encap->gre.remote_ip[0] = db->parms.raddr.in.s_addr;
					encap->gre.local_ip[0] = db->parms.laddr.in.s_addr;
					encap->gre.vlan_tag = configs.tunnel.vlan_tag;
					params.tx.control.vlan_tci = encap->gre.vlan_tag;
				} else if (configs.tunnel.vlan_tag) {
					params.rx.control.vlan_untag = 1;
					encap->gre.vlan_tag = 1;
				}
			} else if (params.ipv4.type == ft_ipv4_dslite_encap) {
				db = flowmgr_tdb_find_by_raddr(configs.tunnel.remote_ip.in6.s6_addr32);
				if (!db) {
					pr_err("%s: no matching db for dslite remote ip %pI6\n", __func__, configs.tunnel.remote_ip.in6.s6_addr32);
					ret = -1;
					break;
				}
				encap->type = ARPHRD_TUNNEL6;
				memcpy(encap->dslite.remote_mac, db->parms.eh.h_dest,
			       	ETH_ALEN);
				memcpy(encap->dslite.local_mac, db->parms.eh.h_source,
			       	ETH_ALEN);
				memcpy((char *)encap->dslite.remote_ip,
			       	db->parms.raddr.in6.s6_addr,
			       	sizeof(struct in6_addr));
				memcpy((char *)encap->dslite.local_ip,
			       	db->parms.laddr.in6.s6_addr,
			       	sizeof(struct in6_addr));
			}
		}

		ret = fap()->flow_add(&params, &flow_id);
		if (ret) {
			pr_err("%s: flow_add failed\n", __func__);
			break;
		}

		if (params.ipv4.type == ft_ipv4_dslite_encap || params.ipv4.type == ft_ipv4_gre_encap)
			params.tx.interface = configs.tunnel.ifindex;
		update_ct_offload(flow_id, &params, ct_offload_info, ct_offload, ct_offload_master);
		nf_ct_put(ct);
		ret = flow_id;
		break;
	}

	case FLOWMGR_IOCTL_FLOW_DEL: {
		int flow_id;
		ret = copy_from_user(&flow_id, (void *)arg, sizeof(flow_id));
		if (ret) {
			pr_err("%s: can't copy params from userspace\n", __func__);
			break;
		}

		mutex_lock(&flowmgr_mutex);
		locked = 1;

		ret = fap()->flow_remove(flow_id);
		if (ret) {
			pr_err("%s: flow_remove failed\n", __func__);
			break;
		}
		break;
	}

	case FLOWMGR_IOCTL_FLOW_PROMOTE: {
		struct flow_tuple tuple;
		struct nf_conn *ct = NULL;

		if (flowmgr.promote_mode != MANUAL_PROMOTE_BY_CMD) {
			pr_err("%s: Operation only supported in manual promote by cmd mode\n", __func__);
			ret = -EPERM;
			break;
		}

		ret = copy_from_user(&tuple, (void *)arg, sizeof(tuple));
		if (ret) {
			pr_err("%s: can't copy flow_tuple from userspace\n", __func__);
			break;
		}
		if (tuple.l3_prot != AF_INET && tuple.l3_prot != AF_INET6) {
			pr_err("%s: Only ipv4/ipv6 flow type supported\n", __func__);
			ret = -EINVAL;
			break;
		}

		mutex_lock(&flowmgr_mutex);
		locked = 1;

		/* Look up ct_offload by flow params */
		ct = nf_conn_find_by_flow_tuple(tuple, NULL);
		if (!ct) {
			pr_err("%s: no matching ct by flow params\n", __func__);
			ret = -EINVAL;
			break;
		}
		ret = flowmgr_manual_promote(&tuple, ct);
		nf_ct_put(ct);
		break;
	}

	case FLOWMGR_IOCTL_FLOW_DEMOTE: {
		struct flow_tuple tuple;
		struct nf_conn *ct = NULL;

		ret = copy_from_user(&tuple, (void *)arg, sizeof(tuple));
		if (ret) {
			pr_err("%s: can't copy flow_tuple from userspace\n", __func__);
			break;
		}
		if (tuple.l3_prot != AF_INET && tuple.l3_prot != AF_INET6) {
			pr_err("%s: Only ipv4/ipv6 flow type supported\n", __func__);
			ret = -EINVAL;
			break;
		}

		mutex_lock(&flowmgr_mutex);
		locked = 1;
		/* Look up ct_offload by flow params */
		ct = nf_conn_find_by_flow_tuple(tuple, NULL);
		if (!ct) {
			pr_err("%s: no matching ct by flow params\n", __func__);
			ret = -EINVAL;
			break;
		}
		ret = flowmgr_manual_demote(&tuple, ct);
		nf_ct_put(ct);
		break;
	}

	default:
		pr_err("Unsupported IOCTL %d\n", cmd);
		ret = -EINVAL;
		break;
	}

	if (locked)
		mutex_unlock(&flowmgr_mutex);
	return ret;
}

#ifdef CONFIG_COMPAT
static long flowmgr_compat_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	return flowmgr_ioctl(file, cmd, (unsigned long)compat_ptr(arg));
}
#endif

int flowmgr_cdev_init(void)
{
	int ret = 0;

	flowmgr_class = class_create(THIS_MODULE, FLOWMGR_NAME);
	if (IS_ERR(flowmgr_class)) {
		pr_err("%s: can't create flowmgr class\n", __func__);
		ret = PTR_ERR(flowmgr_class);
		goto ERROR;
	}

	ret = alloc_chrdev_region(&flowmgr_dev, 0, 1, FLOWMGR_NAME);
	if (ret) {
		pr_err("%s: can't alloc chrdev region\n", __func__);
		goto ERROR;
	}
	flowmgr_major = MAJOR(flowmgr_dev);

	cdev_init(&flowmgr_cdev, &flowmgr_fops);
	ret = cdev_add(&flowmgr_cdev, flowmgr_dev, 1);
	if (ret) {
		pr_err("%s: can't add flowmgr cdev\n", __func__);
		goto ERROR;
	}

	flowmgr_device = device_create(flowmgr_class, NULL, MKDEV(flowmgr_major, 0), NULL, FLOWMGR_NAME);
	if (IS_ERR(flowmgr_device)) {
		pr_err("%s: can't create flowmgr device\n", __func__);
		ret = PTR_ERR(flowmgr_device);
		goto ERROR;
	}

	return 0;

ERROR:
	flowmgr_cdev_exit();
	return ret;
}

void flowmgr_cdev_exit(void)
{
	if (flowmgr_device)
		device_destroy(flowmgr_class, MKDEV(flowmgr_major, 0));
	cdev_del(&flowmgr_cdev);
	if (flowmgr_class)
		class_destroy(flowmgr_class);
	if (flowmgr_major)
		unregister_chrdev_region(MKDEV(flowmgr_major, 0), 1);
}
