/*
 * Driver for Broadcom RPC GPIO unit (pinctrl + GPIO)
 *
 * Copyright (C) 2015-2019 Broadcom
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation version 2.
 *
 * This program is distributed "as is" WITHOUT ANY WARRANTY of any
 * kind, whether express or implied; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/bitops.h>
#include <linux/gpio/driver.h>
#include <linux/of_device.h>
#include <linux/of_irq.h>
#include <linux/of_address.h>
#include <linux/of.h>
#include <linux/module.h>
#include <linux/irqdomain.h>
#include <linux/irqchip/chained_irq.h>
#include <linux/interrupt.h>
#include <linux/platform_device.h>
#include <linux/seq_file.h>
#include <linux/slab.h>
#include <linux/spinlock.h>
#include <linux/types.h>
#include <linux/bcm_media_gw/itc_rpc/itc_rpc.h>
#include <linux/delay.h>

#include "../core.h"
#include "../pinctrl-utils.h"
#include "pinctrl-bcm-rpc.h"

static const char * const bcmrpc_pinconf_dtname[PINCONF_PARAM_MAX] = {
	[PINCONF_PARAM_DRV] = "brcm,drv",
	[PINCONF_PARAM_SLEW] = "brcm,slew",
	[PINCONF_PARAM_HYST] = "brcm,hyst",
	[PINCONF_PARAM_INDIS] = "brcm,indis",
	[PINCONF_PARAM_PULL] = "brcm,pull",
	[PINCONF_PARAM_AMP] = "brcm,amp",
	[PINCONF_PARAM_MHV] = "brcm,mhv",
	[PINCONF_PARAM_GMII] = "brcm,gmii",
	[PINCONF_PARAM_MPV] = "brcm,mpv",
	[PINCONF_PARAM_SRC] = "brcm,src",
	[PINCONF_PARAM_DGEN] = "brcm,dgen",
	[PINCONF_PARAM_RSEL] = "brcm,rsel",
	[PINCONF_PARAM_GFIL] = "brcm,gfil",
	[PINCONF_PARAM_PWR] = "brcm,pwr",
};

static const int  bcmrpc_pinconf_arg_max_limit[PINCONF_PARAM_MAX] = {
	[PINCONF_PARAM_DRV] = PINCONFIG_DRV_8MA,
	[PINCONF_PARAM_SLEW] = PINCONFIG_SLEW_SEDGE,
	[PINCONF_PARAM_HYST] = PINCONFIG_HYST_SCHTRIG,
	[PINCONF_PARAM_INDIS] = PINCONFIG_INPUT_DISABLE,
	[PINCONF_PARAM_PULL] = PINCONFIG_PULL_UP,
	[PINCONF_PARAM_AMP] = PINCONFIG_AMP_ENABLE,
	[PINCONF_PARAM_MHV] = PINCONFIG_MHV_ENABLE,
	[PINCONF_PARAM_GMII] = PINCONFIG_GMII_ENABLE,
	[PINCONF_PARAM_MPV] = PINCONFIG_MPV_FAST,
	[PINCONF_PARAM_SRC] = PINCONFIG_SRC_ENABLE,
	[PINCONF_PARAM_DGEN] = PINCONFIG_DGEN_ENABLE,
	[PINCONF_PARAM_RSEL] = PINCONFIG_RSEL_50K,
	[PINCONF_PARAM_GFIL] = PINCONFIG_GFIL_ENABLE,
	[PINCONF_PARAM_PWR] = PINCONFIG_PWR_ENABLE,
};

static const char * const bcmrpc_functions[FSEL_COUNT] = {
	[FSEL_ALT0] = "alt0",
	[FSEL_ALT1] = "alt1",
	[FSEL_ALT2] = "alt2",
	[FSEL_ALT3] = "alt3",
	[FSEL_ALT4] = "alt4",
	[FSEL_ALT5] = "alt5",
	[FSEL_ALT6] = "alt6",
	[FSEL_ALT7] = "alt7",
};

static int g_rpc_tunnel;
static uint8_t g_pin_used[PIN_ID_MAX] = {0};
static uint16_t g_pin_id[PIN_ID_MAX] = {0};
static uint8_t g_gpio_used[PIN_ID_MAX] = {0};
static uint16_t g_gpio_id[PIN_ID_MAX] = {0};

static rpc_function rpc_gpio_services_tbl[] = {
	{ NULL,	0 },
	{ NULL,	0 }
};

static rpc_function rpc_pinctrl_services_tbl[] = {
	{ NULL,	0 },
	{ NULL,	0 },
	{ NULL,	0 },
	{ NULL,	0 },
	{ NULL,	0 },
	{ NULL,	0 }
};

/* -------------------- RPC API functions -------------------- */
/* gpio rpc message manipulation helpers */

/**
 * rpc_gpio_alloc, rpc_gpio_free
 * reply decoding
 * W1: | 31..24 | 23..9 | 08..00 |
 *     |   RC   | RSVD  | GPIO-ID|
 *
 * W2: | 31..0 |
 *     |  RSVD |
 *
 * W3: | 31..0 |
 *     |  RSVD |
 *
 * RSVD:	Reserved
 * GPIO-ID:	GPIO ID provided by SMC #
 */
static inline uint16_t rpc_gpio_decode_gpio_id(rpc_msg *msg)
{
	return (msg->data[0] & RPC_PINCONF_ID_MASK);
}

static inline void rpc_gpio_encode_gpio_id(rpc_msg *msg,
					uint16_t gpio_id)
{
	msg->data[0] = ((msg->data[0] & ~RPC_PINCONF_ID_MASK) |
					 gpio_id);
}

static inline uint8_t rpc_gpio_decode_retcode(rpc_msg *msg)
{
	return ((msg->data[0] >> RPC_PINCONF_RET_SHIFT) &
			RPC_PINCONF_RET_MASK);
}

static inline void rpc_gpio_encode_retcode(rpc_msg *msg, uint8_t v)
{
	msg->data[0] = ((msg->data[0] &
		~(RPC_PINCONF_RET_MASK << RPC_PINCONF_RET_SHIFT)) |
		(v << RPC_PINCONF_RET_SHIFT));
}

/**
 * gpio_alloc request encoding
 * W1: | 31..24  | 23..16  | 15..08  | 07..00  |
 *     | NAME[3] | NAME[2] | NAME[1] | NAME[0] |
 *
 * W2: | 31..24  | 23..16  | 15..08  | 07..00  |
 *     | NAME[7] | NAME[6] | NAME[5] | NAME[4] |
 *
 * W3: | 31..24  | 23..16  | 15..08  | 07..00  |
 *     | NAME[11]| NAME[10]| NAME[9] | NAME[8] |
 *
 * NAME:	GPIO name in ASCII
 */
static inline void rpc_gpio_encode_gpio_name(rpc_msg *msg,
					const char *gpio_name)
{
	if (strlen(gpio_name) > sizeof(msg->data)) {
		pr_err("%s : gpio-name : %s length > than limit %lu",
			   MODULE_NAME, gpio_name,
			   sizeof(msg->data));
	}
	strncpy((char *)msg->data, gpio_name, sizeof(msg->data));
}

static inline int rpc_gpio_request(int tunnel, rpc_msg *msg)
{
	int rc = 0;

	rc = rpc_send_request_timeout(tunnel, msg,
			RPC_GPIO_TIMEOUT);
	if (rc) {
		pr_debug("%s : rpc_send_request failure (%d)",
			MODULE_NAME, rc);
		//rpc_dump_msg(msg);
		goto done;
	}
	rc = rpc_gpio_decode_retcode(msg);
	if (rc) {
		pr_debug("%s : gpio_decode_retcode %d",
			MODULE_NAME, rc);
		//rpc_dump_msg(msg);
	}
done:
	return rc;
}

static int rpc_gpio_sanity_check_alloc(
			struct brcmrpc_pinctrl *pc,
			const char *gpio_name,
			unsigned int gpio_num,
			enum rpc_config config)
{
	int rc = 0, cnt;

	for (cnt = 0; cnt < pc->dt_ngpio_names; cnt++) {
		if (!strncmp(pc->gpio_info[cnt].names, gpio_name,
		   strlen(pc->gpio_info[cnt].names)))
			break;
	}
	if (cnt >= pc->dt_ngpio_names) {
		pr_info("%s : Pin name not valid for GPIO configuration %s\n",
				 MODULE_NAME, gpio_name);
		rc = -EINVAL;
		goto out;
	}

	if (g_gpio_used[gpio_num] == RPC_CONFIG_NONE)
		return rc;

	if (g_gpio_used[gpio_num] == RPC_CONFIG_NOTSUPPORT) {
		pr_debug("%s : GPIO : %d not accessible by this CPU\n",
				MODULE_NAME, gpio_num);
		rc = -EINVAL;
		goto out;
	}
	if (g_gpio_used[gpio_num] != config) {
		pr_debug("%s : GPIO : %d already in use by other feature\n",
			MODULE_NAME, gpio_num);
		rc = -EINVAL;
		goto out;
	}

out:
	return rc;
}

static int rpc_gpio_alloc(struct brcmrpc_pinctrl *pc,
			int tunnel, const char *gpio_name,
		    unsigned int gpio_num, unsigned int *gpio_id,
			enum rpc_config config)
{
	int rc = 0;
	rpc_msg msg;

	/* Sanity check for gpio before allocation */
	rc = rpc_gpio_sanity_check_alloc(pc, gpio_name,
			gpio_num, config);
	if (rc)
		goto out;

	if (g_gpio_used[gpio_num] == config &&
		g_gpio_used[gpio_num] != RPC_CONFIG_NONE) {
		*gpio_id = g_gpio_id[gpio_num];
		pr_debug("%s : GPIO: %d already in use just return the id: %d\n",
			   MODULE_NAME, gpio_num, g_gpio_id[gpio_num]);
		goto out;
	}
	rpc_msg_init(&msg, RPC_SERVICE_GPIO, RPC_GPIO_FUNC_ALLOC, 0, 0, 0, 0);
	rpc_gpio_encode_gpio_name(&msg, gpio_name);
	rc = rpc_gpio_request(tunnel, &msg);
	if (rc) {
		g_gpio_used[gpio_num] = RPC_CONFIG_NOTSUPPORT;
		pr_debug("%s : GPIO request failed to return\n", MODULE_NAME);
		goto out;
	}
	*gpio_id = rpc_gpio_decode_gpio_id(&msg);
	/* Check GPIO-ID validity */
	if (*gpio_id >= PIN_ID_MAX) {
		pr_debug("%s : GPIO-ID: %d invalid greater than supported\n",
				 MODULE_NAME, *gpio_id);
		rc = -EINVAL;
		goto out;
	}
	/* Updated the used GPIO db */
	g_gpio_used[gpio_num] = config;
	g_gpio_id[gpio_num] = *gpio_id;

out:
	return rc;
}

static int rpc_gpio_free(int tunnel, unsigned int gpio_num,
			const char *gpio_name, unsigned int gpio_id)
{
	int rc = 0;
	rpc_msg msg;

	if (g_gpio_used[gpio_id] == RPC_CONFIG_NONE) {
		pr_info("%s : GPIO %s with ID : %d not confiured to be freed\n",
				MODULE_NAME, gpio_name, gpio_id);
		rc = -EINVAL;
		goto out;
	}
	rpc_msg_init(&msg, RPC_SERVICE_GPIO, RPC_GPIO_FUNC_FREE, 0, 0, 0, 0);
	rpc_gpio_encode_gpio_id(&msg, (uint16_t)gpio_id);
	rc = rpc_gpio_request(tunnel, &msg);
	if (rc)
		goto out;
	/* Updated the used GPIO db */
	g_gpio_used[gpio_id] = RPC_CONFIG_NONE;
out:
	return rc;
}

/* pinctrl rpc message manipulation helpers */

/**
 * rpc_pinconf_alloc, rpc_pinconf_free
 * reply decoding
 * W1: | 31..24 | 23..9 | 8..0 |
 *     |   RC   | RSVD  |PIN-ID|
 *
 * W2: | 31..0 |
 *     |  RSVD |
 *
 * W3: | 31..0 |
 *     |  RSVD |
 *
 * RSVD:	Reserved
 * PIN-ID:	Pin ID provided by SMC #
 */
static inline uint16_t rpc_pinconf_decode_pin_id(rpc_msg *msg)
{
	return (msg->data[0] & RPC_PINCONF_ID_MASK);
}

/**
 * rpc_pinconf_alloc request encoding
 * W1: | 31..24  | 23..16  | 15..08  | 07..00  |
 *     | NAME[3] | NAME[2] | NAME[1] | NAME[0] |
 *
 * W2: | 31..24  | 23..16  | 15..08  | 07..00  |
 *     | NAME[7] | NAME[6] | NAME[5] | NAME[4] |
 *
 * W3: | 31..24  | 23..16  | 15..08  | 07..00  |
 *     | NAME[11]| NAME[10]| NAME[9] | NAME[8] |
 *
 * RSVD:	Reserved
 * PIN-ID:	Pin ID provided by SMC #
 */
static inline void rpc_pinconf_encode_pin_name(rpc_msg *msg,
					const char *pin_name)
{
	if (strlen(pin_name) > sizeof(msg->data)) {
		pr_err("%s: pin-name: %s greater than RPC message buffer %lu\n",
			   MODULE_NAME, pin_name, sizeof(msg->data));
	}
	strncpy((char *)msg->data, pin_name, sizeof(msg->data));
}

/**
 * rpc_pinconf_decode_pad_cfg
 * response decoding
 * W1: | 31..24 | 23..09 | 08..00 |
 *     |   RC   | MASK   | PIN-ID |
 *
 * W2: | 31..20 |
 *     |  RSVD  |
 *     |   19   |  18  |  17  |  16  | 15   | 14   | 13   |
 *     |  PWR   | GFIL | RSEL | DGEN | SRC  | MPV  | MHV  |
 *     |   12   |  11  |10..09| 08   |07..06|05..04|03..00|
 *     |  GMII  | AMP  | PULL |INDIS | HYST | SLEW | DRV  |
 *
 * W3: | 31..0 |
 *     |  RSVD |
 *
 * RSVD:	Reserved
 * PIN-ID:	Pin ID provided by SMC #
 */
static inline uint32_t rpc_pinconf_decode_pad_cfg(rpc_msg *msg)
{
	return (uint32_t)(msg->data[1] & RPC_PINCONF_VALUE_BIT_MASK);
}

static inline uint16_t rpc_pinconf_decode_pad_mask(rpc_msg *msg)
{
	return (uint16_t)((msg->data[0] >> RPC_PINCONF_TYPE_MASK_SHIFT(0))
					  & RPC_PINCONF_MASK_BIT_MASK);
}


/**
 * rpc_pinconf_encode_pad_cfg
 * request encoding
 * W1: | 31..24 | 23..09 | 08..00 |
 *     |   RC   | MASK   | PIN-ID |
 *
 * W2: | 31..20 |
 *     |  RSVD  |
 *     |   19   |  18  |  17  |  16  | 15   | 14   | 13   |
 *     |  PWR   | GFIL | RSEL | DGEN | SRC  | MPV  | MHV  |
 *     |   12   |  11  |10..09| 08   |07..06|05..04|03..00|
 *     |  GMII  | AMP  | PULL |INDIS | HYST | SLEW | DRV  |
 *
 * W3: | 31..0 |
 *     |  RSVD |
 *
 * RSVD:	Reserved
 * PIN-ID:	Pin ID provided by SMC #
 */
static inline void rpc_pinconf_encode_pad_cfg(rpc_msg *msg,
					unsigned int func)
{
	msg->data[1] =
		((msg->data[1] & ~RPC_PINCONF_VALUE_BIT_MASK)
		| (func & RPC_PINCONF_VALUE_BIT_MASK));
}

static inline void rpc_pinconf_encode_pad_mask(rpc_msg *msg,
					unsigned int mask)
{
	msg->data[0] = ((msg->data[0] &
		~(mask << RPC_PINCONF_TYPE_MASK_SHIFT(0)))
		| (mask << RPC_PINCONF_TYPE_MASK_SHIFT(0)));
}

/**
 * pinctrl_get_mux_func
 * reply decoding
 * W1: | 31..24 | 23..13 | 12..09 | 08..00 |
 *     |   RC   | RSVD   |  FUNC  | PIN-ID |
 *
 * W2: | 31..0 |
 *     |  RSVD |
 *
 * W3: | 31..0 |
 *     |  RSVD |
 *
 * RSVD:	Reserved
 * PIN-ID:	Pin ID provided by SMC #
 */
static inline uint8_t rpc_pinconf_decode_mux_func(rpc_msg *msg)
{
	return ((msg->data[0] >> RPC_PINCONF_MUX_SHIFT) &
			RPC_PINCONF_MUX_MASK);
}

static inline void rpc_pinconf_encode_pin_id(rpc_msg *msg,
					uint16_t pin_id)
{
	msg->data[0] = ((msg->data[0] & ~RPC_PINCONF_ID_MASK) |
					 pin_id);
}

/**
 * rpc_pinconf_encode_mux_func
 * request encoding
 * W1: | 31..13 | 12..09 | 08..00 |
 *     |  RSVD  |  FUNC  | PIN-ID |
 *
 * W2: | 31..0 |
 *     |  RSVD |
 *
 * W3: | 31..0 |
 *     |  RSVD |
 *
 * RSVD:	Reserved
 * PIN-ID:	Pin ID provided by SMC #
 */
static inline void rpc_pinconf_encode_mux_func(rpc_msg *msg,
					uint8_t func)
{
	msg->data[0] = ((msg->data[0] &
	~(RPC_PINCONF_MUX_MASK << RPC_PINCONF_MUX_SHIFT)) |
	 (func << RPC_PINCONF_MUX_SHIFT));
}

static inline uint8_t rpc_pinconf_decode_retcode(rpc_msg *msg)
{
	return ((msg->data[0] >> RPC_PINCONF_RET_SHIFT) &
			 RPC_PINCONF_RET_MASK);
}

static inline void rpc_pinconf_encode_retcode(
					rpc_msg *msg, uint8_t v)
{
	msg->data[0] = ((msg->data[0] &
		 ~(RPC_PINCONF_RET_MASK << RPC_PINCONF_RET_SHIFT)) |
		 (v << RPC_PINCONF_RET_SHIFT));
}

static inline int rpc_pinconf_request(int tunnel,
					rpc_msg *msg)
{
	int rc = 0;

	rc = rpc_send_request_timeout(tunnel, msg, RPC_PINCTRL_TIMEOUT);
	if (rc) {
		pr_debug("%s : rpc_send_request failure (%d)\n",
			   MODULE_NAME, rc);
		//rpc_dump_msg(msg);
		goto out;
	}
	rc = rpc_pinconf_decode_retcode(msg);
	if (rc) {
		pr_debug("%s : pinctrl_decode_retcode %d\n",
			   MODULE_NAME, rc);
		//rpc_dump_msg(msg);
	}
out:
	return rc;
}

static int rpc_pinconf_alloc(int tunnel,
			int pin,
			const char *pin_name,
			unsigned int *pin_id)
{
	int rc = 0;
	rpc_msg msg;

	if (g_pin_used[pin]) {
		*pin_id = g_pin_id[pin];
		pr_debug("%s : Pin:%d already in use just return the id: %d\n",
			   MODULE_NAME, pin, g_pin_id[pin]);
		return 0;
	}
	/* Allocate Pinctrl */
	rpc_msg_init(&msg, RPC_SERVICE_PINCTRL,
		RPC_PINCTRL_FUNC_ALLOC, 0, 0, 0, 0);
	rpc_pinconf_encode_pin_name(&msg, pin_name);
	rc = rpc_pinconf_request(tunnel, &msg);
	if (rc)
		goto out;
	*pin_id = rpc_pinconf_decode_pin_id(&msg);
	/* Check Pin-ID validity */
	if (*pin_id >= PIN_ID_MAX) {
		pr_err("%s : Pin-ID invalid greater than supported %d\n",
			   MODULE_NAME, *pin_id);
		rc = -EINVAL;
		goto out;
	}
	/* Updated the used GPIO db */
	g_pin_used[pin] = 1;
	g_pin_id[pin] = *pin_id;
out:
	return rc;
}

static int rpc_pinconf_free(int tunnel,
			int pin,
			unsigned int pin_id)
{
	int rc = 0;
	rpc_msg msg;
	/*Check Pin-ID usage */
	if (!g_pin_used[pin])
		goto done;

	rpc_msg_init(&msg, RPC_SERVICE_PINCTRL, RPC_PINCTRL_FUNC_FREE, 0, 0, 0, 0);
	rpc_pinconf_encode_pin_id(&msg, (uint16_t)pin_id);
	rc = rpc_pinconf_request(tunnel, &msg);
	if (rc)
		goto done;
	/* Updated the used GPIO db */
	g_pin_used[pin] = 0;
	g_pin_id[pin] = 0;
done:
	return rc;
}

static int rpc_pinconf_set_mux_func(struct brcmrpc_pinctrl *pc,
			unsigned int pin,
			const char *pin_name,
			unsigned int func, enum rpc_config config)
{
	rpc_msg msg;
	int rc = 0;
	unsigned int pin_id = 0;

	/* Allocate GPIO */
	rc = rpc_gpio_alloc(pc, g_rpc_tunnel,
						pin_name,
						pin,
						&pin_id,
						config);
	if (rc)
		pr_debug("%s : could not allocate gpio:%d\n",
			   MODULE_NAME, pin);

	/* Pin 0-8 doesnot have Pinctrl setting */
	/*if (pin < RPC_PINCTRL_MUX_OFFSET)
		return rc;*/
	/* Allocate Pinmux */
	rc = rpc_pinconf_alloc(g_rpc_tunnel,
				pin, pin_name, &pin_id);
	if (rc) {
		pr_debug("%s : could not allocate gpio:%d pinctrl\n",
			   MODULE_NAME, pin);
		return rc;
	}
	rpc_msg_init(&msg, RPC_SERVICE_PINCTRL,
		RPC_PINCTRL_FUNC_MUX_SET, 0, 0, 0, 0);
	rpc_pinconf_encode_pin_id(&msg, (uint16_t)pin_id);
	rpc_pinconf_encode_mux_func(&msg, (uint8_t)func);

	return rpc_pinconf_request(g_rpc_tunnel, &msg);
}

static int rpc_pinconf_set(struct brcmrpc_pinctrl *pc,
			unsigned int pin,
			const char *pin_name,
			unsigned int mask,
			unsigned int func,
			enum rpc_config config)
{
	rpc_msg msg;
	int rc = 0;
	unsigned int pin_id = 0;
	/* Allocate GPIO */
	rc = rpc_gpio_alloc(pc, g_rpc_tunnel,
						pin_name,
						pin,
						&pin_id,
						config);
	if (rc)
		pr_debug("%s : could not allocate gpio:%d\n",
			   MODULE_NAME, pin);

	rc = rpc_pinconf_alloc(g_rpc_tunnel,
				pin, pin_name, &pin_id);
	if (rc) {
		pr_debug("%s : could not allocate gpio%d pinctrl\n",
			   MODULE_NAME, pin);
		return rc;
	}
	rpc_msg_init(&msg, RPC_SERVICE_PINCTRL,
		RPC_PINCTRL_FUNC_PAD_CFG_SET, 0, 0, 0, 0);
	rpc_pinconf_encode_pin_id(&msg, (uint16_t)pin_id);
	rpc_pinconf_encode_pad_mask(&msg, mask);
	rpc_pinconf_encode_pad_cfg(&msg, func);
	return rpc_pinconf_request(g_rpc_tunnel, &msg);
}

static int rpc_pinconf_get(unsigned int pin_id,
			unsigned int *func,
			unsigned int *mask)
{
	int rc = 0;
	rpc_msg msg;

	rpc_msg_init(&msg, RPC_SERVICE_PINCTRL,
		RPC_PINCTRL_FUNC_PAD_CFG_GET, 0, 0, 0, 0);
	rpc_pinconf_encode_pin_id(&msg, (uint16_t)pin_id);
	rc = rpc_pinconf_request(g_rpc_tunnel, &msg);
	if (rc)
		goto out;
	*func = (unsigned int)rpc_pinconf_decode_pad_cfg(&msg);
	*mask = (unsigned int)rpc_pinconf_decode_pad_mask(&msg);
out:
	return rc;
}

int brcmrpc_get_gpio_mux_func(struct brcmrpc_pinctrl *pc,
					unsigned int gpio_num,
					unsigned int *mux_func)
{
	int cnt = 0;

	for (cnt = 0; cnt < pc->dt_ngpio_names; cnt++) {
		if ((gpio_num < (pc->gpio_info[cnt].offset+
						pc->gpio_info[cnt].ngpios)) &&
			(gpio_num >= pc->gpio_info[cnt].offset))
		{
			*mux_func = pc->gpio_info[cnt].muxfunc;
			return 0;
		}
	}

	return -EINVAL;
}

int rpc_gpio_config(struct brcmrpc_gpio_priv *priv,
					const char *gpio_name,
					unsigned int gpio_num,
					enum rpc_config config,
					enum rpc_action action)
{
	int rc = 0;
	struct brcmrpc_pinctrl *pc = priv->pc;
	unsigned int mux_func = 0;

	mutex_lock(&priv->lock);
	switch (action) {
	case RPC_GPIO_ALLOC:  {
		/* Set to GPIO on Pinctrl and allocate*/
		rc = brcmrpc_get_gpio_mux_func(pc, gpio_num, &mux_func);
		if (rc) {
			pr_err("%s : failed to get gpio mux function value for gpio %d\n",
				   MODULE_NAME, gpio_num);
			return rc;
		}
		rc = rpc_pinconf_set_mux_func(pc, gpio_num,
				gpio_name, mux_func, config);
		if (rc)
			pr_debug("%s : failed to set gpio on pinctrl %d\n",
				   MODULE_NAME, gpio_num);
	}
	break;

	case RPC_GPIO_FREE:  {
		/* Free Pinctrl */
		rc = rpc_pinconf_free(g_rpc_tunnel, gpio_num,
			g_pin_id[gpio_num]);
		if (rc)
			pr_debug("%s : failed to free pinctrl %d\n",
				   MODULE_NAME, gpio_num);
		/* Free GPIO */
		rc = rpc_gpio_free(g_rpc_tunnel, gpio_num,
			gpio_name, g_gpio_id[gpio_num]);
		if (rc)
			pr_debug("%s : failed to free gpio %d\n",
				   MODULE_NAME, gpio_num);
	}
	break;

	case RPC_PINCTRL_ALLOC:
	case RPC_PINCTRL_FREE:
	default:
		rc = -EINVAL;
	break;
	}
	mutex_unlock(&priv->lock);
	return rc;
}

/* -------------------- Pin config functions -------------------- */
static int bcmrpc_pctl_find_pin_offset(struct brcmrpc_pinctrl *pc,
			unsigned int selector)
{
	int cnt;
	struct pinctrl_pin_desc *pcfg_pins = pc->pin_group->pin_desc;
	u32 nr_pins = pc->pin_group->nr_pins;

	for (cnt = 0; cnt < nr_pins; cnt++) {
		if (pcfg_pins[cnt].number == selector)
			return cnt;
	}

	return -EINVAL;
}

static int bcmrpc_pctl_get_groups_count(struct pinctrl_dev *pctldev)
{
	struct brcmrpc_pinctrl *pc = pinctrl_dev_get_drvdata(pctldev);
	u32 nr_groups = pc->pin_group->nr_pins;

	return nr_groups;
}

static const char *bcmrpc_pctl_get_group_name(struct pinctrl_dev *pctldev,
		unsigned int selector)
{
	struct brcmrpc_pinctrl *pc = pinctrl_dev_get_drvdata(pctldev);
	struct pinctrl_pin_desc *pmux_groups = pc->pin_group->pin_desc;
	u32 nr_groups = pc->pin_group->nr_pins;

	if (selector >= nr_groups)
		return NULL;

	return pmux_groups[selector].name;
}

static int bcmrpc_pctl_get_group_pins(struct pinctrl_dev *pctldev,
		unsigned int selector,
		const unsigned int **pins,
		unsigned int *num_pins)
{
	struct brcmrpc_pinctrl *pc = pinctrl_dev_get_drvdata(pctldev);
	struct pinctrl_pin_desc *pmux_groups = pc->pin_group->pin_desc;
	u32 nr_groups = pc->pin_group->nr_pins;

	if (selector >= nr_groups)
		return -EINVAL;

	*pins = &pmux_groups[selector].number;
	*num_pins = 1;

	return 0;
}

static int bcmrpc_pctl_dt_node_to_map_func(struct pinctrl_dev *pctldev,
		struct pinctrl_map **map, unsigned int *reserved_maps,
		unsigned int *num_maps,
		struct device_node *np, u32 group,
		const char *group_name, u32 fnum)
{
	int ret;
	struct device *dev = pctldev->dev;

	if (fnum >= ARRAY_SIZE(bcmrpc_functions)) {
		dev_err(dev, "%s: invalid %s %d\n",
			of_node_full_name(np),
			PMUX_FUNCTION_PROPERTY, fnum);
		return -EINVAL;
	}
	ret = pinctrl_utils_add_map_mux(pctldev, map,
			reserved_maps, num_maps, group_name,
			bcmrpc_functions[fnum]);
	if (ret < 0)
		return ret;

	return 0;
}

static int bcmrpc_pctl_dt_node_to_map_pinconf(struct pinctrl_dev *pctldev,
		struct pinctrl_map **map,
		unsigned int *reserved_maps,
		unsigned int *num_maps,
		struct device_node *np,
		u32 pin,
		const char *pin_name,
		u32 *pinconf_arg,
		int *num_pinconf)
{
	struct device *dev = pctldev->dev;
	int ret = 0, cnt = 0;
	unsigned long config;
	unsigned long *configs = NULL;
	unsigned int num_configs = 0, pinconf_cnt = 0;
	u32 param[PINCONF_PARAM_MAX] = {0};
	u32 arg[PINCONF_PARAM_MAX] = {0};

	for (cnt = 0; cnt < PINCONF_PARAM_MAX; cnt++) {
		if ((num_pinconf[cnt] > 0) &&
			(pinconf_arg[cnt]
				> bcmrpc_pinconf_arg_max_limit[cnt])) {
			dev_err(dev, "%s: invalid value for %s : %d\n",
					of_node_full_name(np),
					bcmrpc_pinconf_dtname[cnt],
					pinconf_arg[cnt]);
			return -EINVAL;
		}
		if (num_pinconf[cnt] > 0) {
			param[pinconf_cnt] = cnt;
			arg[pinconf_cnt] = pinconf_arg[cnt];
			pinconf_cnt++;
		}
	}
	for (cnt = 0; cnt < pinconf_cnt; cnt++) {
		config = PINCONF_PACK(param[cnt], arg[cnt]);
		ret = pinctrl_utils_add_config(pctldev, &configs,
				&num_configs, config);
		if (ret < 0) {
			dev_err(dev, "pinctrl add config failure\n");
			goto out;
		}
	}
	ret = pinctrl_utils_add_map_configs(pctldev, map,
			reserved_maps, num_maps, pin_name,
			configs, num_configs,
			PIN_MAP_TYPE_CONFIGS_PIN);
	if (ret < 0) {
		dev_err(dev, "pinctrl add config failure\n");
		goto out;
	}

out:
	kfree(configs);
	return ret;
}

static int bcmrpc_pinctrl_dt_subnode_to_map(
			struct pinctrl_dev *pctldev,
			struct device_node *np,
			struct pinctrl_map **map,
			unsigned int *reserved_maps,
			unsigned int *num_maps)
{
	struct device *dev = pctldev->dev;
	struct property *pins, *pin_names, *funcs,
		*pinconf[PINCONF_PARAM_MAX] = {NULL};
	unsigned int num_pins, num_funcs, num_pin_names, reserve;
	int num_pinconf[PINCONF_PARAM_MAX] = {0};
	int i, ret, cnt, pinconf_cnt = 0;
	u32 pin, func, pinconf_arg[PINCONF_PARAM_MAX] = {0};
	const char *name = NULL;

	pins = of_find_property(np, PINS_PROPERTY, NULL);
	if (!pins) {
		dev_err(dev, "%s: missing %s property\n",
				of_node_full_name(np),
				PINS_PROPERTY);
		return -EINVAL;
	}
	pin_names = of_find_property(np, PIN_NAMES_PROPERTY, NULL);
	if (!pin_names) {
		dev_err(dev, "%s: missing %s property\n",
				of_node_full_name(np),
				PIN_NAMES_PROPERTY);
		return -EINVAL;
	}
	ret = of_property_count_strings(np, PIN_NAMES_PROPERTY);
	if (ret < 0) {
		dev_err(dev, "%s: could not parse property %s\n",
				of_node_full_name(np), PIN_NAMES_PROPERTY);
		return ret;
	}
	num_pins = pins->length / 4;
	num_pin_names = ret;
	/* Pin number and pin names count should match */
	if (num_pin_names != num_pins) {
		dev_err(dev,
			"%s: number of pins and number of pin names not match\n",
			of_node_full_name(np));
		return -EINVAL;
	}
	funcs = of_find_property(np, PMUX_FUNCTION_PROPERTY, NULL);
	if (!funcs) {
       /* EINVAL=missing, which is fine since it's optional */
		dev_dbg(dev, "could not find property %s\n",
				PMUX_FUNCTION_PROPERTY);
	}
	for (cnt = 0; cnt < PINCONF_PARAM_MAX; cnt++)
		pinconf[cnt] = of_find_property(np,
			bcmrpc_pinconf_dtname[cnt], NULL);
	/* Atleast function (or) any of config property needs to be
	   present */
	if (!funcs) {
		for (cnt = 0; cnt < PINCONF_PARAM_MAX; cnt++) {
			if (pinconf[cnt])
				break;
		}
		if (cnt >= PINCONF_PARAM_MAX) {
			dev_err(dev,
					"%s: neither %s nor pin configs specified\n",
					of_node_full_name(np),
					PMUX_FUNCTION_PROPERTY);
			return -EINVAL;
		}
	}
	num_funcs = funcs ? (funcs->length / 4) : 0;
	for (cnt = 0; cnt < PINCONF_PARAM_MAX; cnt++)
		num_pinconf[cnt] =
		pinconf[cnt] ? (pinconf[cnt]->length / 4) : 0;
	/* If number of function > 1 then it should match the number
	   of pins */
	if (num_funcs > 1 && num_funcs != num_pins) {
		dev_err(dev,
			"%s: %s must have 1 or %d entries\n",
			of_node_full_name(np),
			PMUX_FUNCTION_PROPERTY,
			num_pins);
		return -EINVAL;
	}
	/* If number of pinconf > 1 then it should match the number
	   of pins */
	for (cnt = 0; cnt < PINCONF_PARAM_MAX; cnt++) {
		if (num_pinconf[cnt] > 1 &&
			num_pinconf[cnt] != num_pins) {
			dev_err(dev,
					"%s: %s must have 1 or %d entries\n",
					of_node_full_name(np),
					bcmrpc_pinconf_dtname[cnt],
					num_pins);
			return -EINVAL;
		}
	}
	/* Calculate the number of maps required */
	reserve = 0;
	if (num_funcs)
		reserve++;
	for (cnt = 0; cnt < PINCONF_PARAM_MAX; cnt++) {
		if (num_pinconf[cnt]) {
			reserve++;
			break;
		}
	}
	reserve *= num_pins;
	ret = pinctrl_utils_reserve_map(pctldev, map,
			reserved_maps, num_maps, reserve);
	if (ret < 0) {
		dev_err(dev, "%s: Reserve map failed\n",
			   of_node_full_name(np));
		goto out;
	}
	for (i = 0; i < num_pins; i++) {
		ret = of_property_read_u32_index(np,
				PINS_PROPERTY, i, &pin);
		if (ret)
			goto out;
		if (pin >= PIN_ID_MAX) {
			dev_err(dev, "%s: invalid %s value %d\n",
					of_node_full_name(np),
					PINS_PROPERTY, pin);
			ret = -EINVAL;
			goto out;
		}
		ret = of_property_read_string_index(np,
				PIN_NAMES_PROPERTY, i, &name);
		if (ret) {
			dev_err(dev,
				"%s %s string index read failure.\n",
				of_node_full_name(np),
				PIN_NAMES_PROPERTY);
			goto out;
		}
		if (!strlen(name)) {
			dev_err(dev,
				"%s pins string is empty.\n",
				of_node_full_name(np));
			ret = -EINVAL;
			goto out;
		}
		if (num_funcs) {
			ret = of_property_read_u32_index(np,
					PMUX_FUNCTION_PROPERTY,
					(num_funcs > 1) ? i : 0, &func);
			if (ret) {
				dev_err(dev,
					"%s %s index read failure.\n",
					of_node_full_name(np),
					PMUX_FUNCTION_PROPERTY);
				goto out;
			}
			ret = bcmrpc_pctl_dt_node_to_map_func(pctldev,
					map, reserved_maps, num_maps,
					np, pin, name, func);
			if (ret) {
				dev_err(dev,
					"%s dt node map func failure.\n",
					of_node_full_name(np));
				goto out;
			}
		}
		for (cnt = 0; cnt < PINCONF_PARAM_MAX; cnt++) {
			if (num_pinconf[cnt]) {
				ret = of_property_read_u32_index(np,
					bcmrpc_pinconf_dtname[cnt],
					(num_pinconf[cnt] > 1) ? i : 0,
					&pinconf_arg[cnt]);
				if (ret) {
					dev_err(dev,
						"%s %s index read failure.\n",
						of_node_full_name(np),
						bcmrpc_pinconf_dtname[cnt]);
					goto out;
				}
				pinconf_cnt++;
			}
		}
		if (pinconf_cnt > 0) {
			ret = bcmrpc_pctl_dt_node_to_map_pinconf(
					pctldev, map, reserved_maps,
					num_maps, np, pin, name,
					pinconf_arg, num_pinconf);
			if (ret) {
				dev_err(dev,
					"%s dt node map pinconf failure.\n",
					of_node_full_name(np));
				goto out;
			}
		}
	}

out:
	return ret;
}

static int bcmrpc_pctl_dt_node_to_map(
			struct pinctrl_dev *pctldev,
			struct device_node *np_config,
			struct pinctrl_map **map,
			unsigned int *num_maps)
{
	unsigned int reserved_maps;
	struct device_node *np;
	int ret = 0;

	if (!pctldev->dev) {
		pr_err("%s Device info not valid", __func__);
		return -EINVAL;
	}
	reserved_maps = 0;
	*map = NULL;
	*num_maps = 0;

	if (!of_get_child_count(np_config))
		ret =  bcmrpc_pinctrl_dt_subnode_to_map(pctldev,
				np_config, map,
				&reserved_maps,
				num_maps);
	if (ret < 0) {
		pinctrl_utils_free_map(pctldev, *map, *num_maps);
		return ret;
	}

	for_each_child_of_node(np_config, np) {
		ret = bcmrpc_pinctrl_dt_subnode_to_map(pctldev,
				np, map,
				&reserved_maps,
				num_maps);
		if (ret < 0) {
			pinctrl_utils_free_map(pctldev, *map,
								   *num_maps);
			of_node_put(np);
			return ret;
		}
	}

	return ret;
}

static const struct pinctrl_ops bcmrpc_pctl_ops = {
	.get_groups_count = bcmrpc_pctl_get_groups_count,
	.get_group_name = bcmrpc_pctl_get_group_name,
	.get_group_pins = bcmrpc_pctl_get_group_pins,
	.dt_node_to_map = bcmrpc_pctl_dt_node_to_map,
	.dt_free_map = pinctrl_utils_free_map,
};

static int bcmrpc_pmx_get_functions_count(
		struct pinctrl_dev *pctldev)
{
	return FSEL_COUNT;
}

static const char *bcmrpc_pmx_get_function_name(
		struct pinctrl_dev *pctldev,
		unsigned int selector)
{
	if (selector >= FSEL_COUNT)
		return NULL;

	return bcmrpc_functions[selector];
}

static int bcmrpc_pmx_get_function_groups(
		struct pinctrl_dev *pctldev,
		unsigned int selector,
		const char * const **groups,
		unsigned * const num_groups)
{
	struct brcmrpc_pinctrl *pc = pinctrl_dev_get_drvdata(pctldev);
	u32 nr_groups = pc->pin_group->nr_pins;

	/* every pin can do every function */
	*groups = pc->groups;
	*num_groups = nr_groups;

	return 0;
}

static int bcmrpc_pmx_set(
		struct pinctrl_dev *pctldev,
		unsigned int func_selector,
		unsigned int group_selector)
{
	unsigned int func = func_selector;
	unsigned int group;
	const char *group_name;
	struct brcmrpc_pinctrl *pc = pinctrl_dev_get_drvdata(pctldev);
	struct pinctrl_pin_desc *pmux_groups = pc->pin_group->pin_desc;
	u32 nr_groups = pc->pin_group->nr_pins;
	int rc = 0;

	if (group_selector >= nr_groups)
		return -EINVAL;

	group = (pmux_groups[group_selector].number & PIN_NUMBER_MASK);
	group_name = pmux_groups[group_selector].name;
	rc = rpc_pinconf_set_mux_func(pc, group, group_name,
			func, RPC_CONFIG_PINCTRL);
	if (rc) {
		pr_info("%s: could not config pmux group : %s\n",
			   MODULE_NAME, group_name);
	}

	return rc;
}

static void bcmrpc_pmx_gpio_disable_free(
		struct pinctrl_dev *pctldev,
		struct pinctrl_gpio_range *range,
		unsigned int offset)
{
	unsigned int group;
	struct brcmrpc_pinctrl *pc = pinctrl_dev_get_drvdata(pctldev);
	struct pinctrl_pin_desc *pmux_groups = pc->pin_group->pin_desc;
	u32 nr_groups = pc->pin_group->nr_pins;
	const char *group_name;
	int rc = 0;
	unsigned int mux_func = 0;

	if (offset >= nr_groups)
		return;

	group = (pmux_groups[offset].number & PIN_NUMBER_MASK);
	group_name = pmux_groups[offset].name;
	/* disable by setting to GPIO_IN */
	rc = brcmrpc_get_gpio_mux_func(pc, group, &mux_func);
	if (rc) {
		pr_err("%s : failed to get gpio mux function value for group %d\n",
			   MODULE_NAME, group);
		return;
	}
	rpc_pinconf_set_mux_func(pc, group, group_name,
		mux_func, RPC_CONFIG_PINCTRL);
	rpc_pinconf_free(g_rpc_tunnel, group, g_pin_id[group]);
}

static const struct pinmux_ops bcmrpc_pmx_ops = {
	.get_functions_count = bcmrpc_pmx_get_functions_count,
	.get_function_name = bcmrpc_pmx_get_function_name,
	.get_function_groups = bcmrpc_pmx_get_function_groups,
	.set_mux = bcmrpc_pmx_set,
	.gpio_disable_free = bcmrpc_pmx_gpio_disable_free,
};

static int bcmrpc_pinconf_get(struct pinctrl_dev *pctldev,
			unsigned int pin, unsigned long *config)
{
	enum pinconf_param_idx param =
		PINCONF_UNPACK_PARAM(*config);
	u32 data;
	int rc = 0;
	unsigned int func = 0, mask = 0;

	rc = rpc_pinconf_get(g_pin_id[pin], &func, &mask);

	switch (param) {
	case PINCONF_PARAM_DRV:
		data = RPC_PINCONF_DRV_UNPACK(func);
		break;

	case PINCONF_PARAM_SLEW:
		data = RPC_PINCONF_SLEW_UNPACK(func);
		break;

	case PINCONF_PARAM_HYST:
		data = RPC_PINCONF_HYST_UNPACK(func);
		break;

	case PINCONF_PARAM_INDIS:
		data = RPC_PINCONF_INDIS_UNPACK(func);
		break;

	case PINCONF_PARAM_PULL:
		data = RPC_PINCONF_PULL_UNPACK(func);
		break;

	case PINCONF_PARAM_AMP:
		data = RPC_PINCONF_AMP_UNPACK(func);
		break;

	case PINCONF_PARAM_MHV:
		data = RPC_PINCONF_MHV_UNPACK(func);
		break;

	case PINCONF_PARAM_GMII:
		data = RPC_PINCONF_GMII_UNPACK(func);
		break;

	case PINCONF_PARAM_MPV:
		data = RPC_PINCONF_MPV_UNPACK(func);
		break;

	case PINCONF_PARAM_SRC:
		data = RPC_PINCONF_SRC_UNPACK(func);
		break;

	case PINCONF_PARAM_DGEN:
		data = RPC_PINCONF_DGEN_UNPACK(func);
		break;

	case PINCONF_PARAM_RSEL:
		data = RPC_PINCONF_RSEL_UNPACK(func);
		break;

	case PINCONF_PARAM_GFIL:
		data = RPC_PINCONF_GFIL_UNPACK(func);
		break;

	case PINCONF_PARAM_PWR:
		data = RPC_PINCONF_PWR_UNPACK(func);
		break;

	default:
		return -EINVAL;
	}

	*config = PINCONF_PACK(param, data);

	return rc;
}

static int bcmrpc_pinconf_set(struct pinctrl_dev *pctldev,
			unsigned int pin, unsigned long *configs,
			unsigned int num_configs)
{
	enum pinconf_param_idx param;
	u16 arg = 0;
	u32 func = 0, mask = 0;
	int i, rc = 0;
	int ret;
	const char *pin_name;
	struct brcmrpc_pinctrl *pc = pinctrl_dev_get_drvdata(pctldev);
	struct pinctrl_pin_desc *pcfg_pins = pc->pin_group->pin_desc;

	ret = bcmrpc_pctl_find_pin_offset(pc, pin);
	if (ret < 0)
		return ret;

	pin_name = pcfg_pins[ret].name;
	pin &= PIN_NUMBER_MASK;

	for (i = 0; i < num_configs; i++) {
		param = PINCONF_UNPACK_PARAM(configs[i]);
		arg = PINCONF_UNPACK_ARG(configs[i]);
		mask = RPC_PINCONF_MASK_PACK(param, mask);
		pr_debug("%s : %s : param : %d arg : %d mask : %x\n",
				MODULE_NAME, __func__,
				param, arg, mask);
		switch (param) {
		case PINCONF_PARAM_DRV:
			func = RPC_PINCONF_DRV_PACK(func, arg);
			pr_debug("%s : PINCONF_PARAM_DRV : %x\n",
					MODULE_NAME, func);
			break;

		case PINCONF_PARAM_SLEW:
			func = RPC_PINCONF_SLEW_PACK(func, arg);
			pr_debug("%s : PINCONF_PARAM_SLEW : %x\n",
					MODULE_NAME, func);
			break;

		case PINCONF_PARAM_HYST:
			func = RPC_PINCONF_HYST_PACK(func, arg);
			pr_debug("%s : PINCONF_HYST_PACK : %x\n",
					MODULE_NAME, func);
			break;

		case PINCONF_PARAM_INDIS:
			func = RPC_PINCONF_INDIS_PACK(func, arg);
			pr_debug("%s : PINCONF_INDIS_PACK : %x\n",
					MODULE_NAME, func);
			break;

		case PINCONF_PARAM_PULL:
			func = RPC_PINCONF_PULL_PACK(func, arg);
			pr_debug("%s : PINCONF_PARAM_PULL : %x\n",
					MODULE_NAME, func);
			break;

		case PINCONF_PARAM_AMP:
			func = RPC_PINCONF_AMP_PACK(func, arg);
			pr_debug("%s : PINCONF_PARAM_AMP : %x\n",
					MODULE_NAME, func);
			break;

		case PINCONF_PARAM_MHV:
			func = RPC_PINCONF_MHV_PACK(func, arg);
			pr_debug("%s : PINCONF_PARAM_MHV : %x\n",
					MODULE_NAME, func);
			break;

		case PINCONF_PARAM_GMII:
			func = RPC_PINCONF_GMII_PACK(func, arg);
			pr_debug("%s : PINCONF_PARAM_GMII : %x\n",
					MODULE_NAME, func);
			break;

		case PINCONF_PARAM_MPV:
			func = RPC_PINCONF_MPV_PACK(func, arg);
			pr_debug("%s : PINCONF_PARAM_MPV : %x\n",
					MODULE_NAME, func);
			break;

		case PINCONF_PARAM_SRC:
			func = RPC_PINCONF_SRC_PACK(func, arg);
			pr_debug("%s : PINCONF_PARAM_SRC : %x\n",
					MODULE_NAME, func);
			break;

		case PINCONF_PARAM_PWR:
			func = RPC_PINCONF_PWR_PACK(func, arg);
			pr_debug("%s : PINCONF_PARAM_PWR : %x\n",
					MODULE_NAME, func);
			break;

		case PINCONF_PARAM_DGEN:
			func = RPC_PINCONF_DGEN_PACK(func, arg);
			pr_debug("%s : PINCONF_PARAM_DGEN : %x\n",
					MODULE_NAME, func);
			break;

		case PINCONF_PARAM_RSEL:
			func = RPC_PINCONF_RSEL_PACK(func, arg);
			pr_debug("%s : PINCONF_PARAM_RSEL : %x\n",
					MODULE_NAME, func);
			break;

		case PINCONF_PARAM_GFIL:
			func = RPC_PINCONF_GFIL_PACK(func, arg);
			pr_debug("%s : PINCONF_PARAM_GFIL : %x\n",
					MODULE_NAME, func);
			break;

		default:
			return -EINVAL;
		}
	}

	if (num_configs > 0) {
		rc = rpc_pinconf_set(pc, pin, pin_name, mask, func,
							 RPC_CONFIG_PINCTRL);
		if (rc) {
			pr_info("%s: could not config gpio pin : %s\n",
				   MODULE_NAME, pin_name);
		}
	}

	return rc;
}

static const struct pinconf_ops bcmrpc_pinconf_ops = {
	.pin_config_get = bcmrpc_pinconf_get,
	.pin_config_set = bcmrpc_pinconf_set,
};

static inline struct brcmrpc_gpio_priv *
brcmrpc_gpio_gc_to_priv(struct gpio_chip *gc)
{
	struct brcmrpc_gpio_bank *bank = gpiochip_get_data(gc);

	return bank->parent_priv;
}

static unsigned long
__brcmrpc_gpio_get_active_irqs(struct brcmrpc_gpio_bank *bank)
{
	void __iomem *reg_base = bank->parent_priv->reg_base;

	return bank->gc.read_reg(reg_base + GIO_STAT(bank->id)) &
	       bank->gc.read_reg(reg_base + GIO_MASK(bank->id));
}

static unsigned long
brcmrpc_gpio_get_active_irqs(struct brcmrpc_gpio_bank *bank)
{
	unsigned long status;
	unsigned long flags;

	irq_gc_lock_irqsave(bank->icg, flags);
	status = __brcmrpc_gpio_get_active_irqs(bank);
	irq_gc_unlock_irqrestore(bank->icg, flags);

	return status;
}

static int brcmrpc_gpio_to_irq(struct gpio_chip *gc,
			unsigned int gc_offset)
{
	struct brcmrpc_gpio_priv *priv = brcmrpc_gpio_gc_to_priv(gc);
	/* gc_offset is relative to this gpio_chip; want real offset */
	int offset = gc_offset + (gc->base - priv->gpio_base);

	if (offset >= priv->num_gpios)
		return -ENXIO;
	return irq_create_mapping(priv->irq_domain, offset);
}

/* -------------------- IRQ chip functions -------------------- */

static int brcmrpc_gpio_hwirq_to_offset(irq_hw_number_t hwirq,
					struct brcmrpc_gpio_bank *bank)
{
	return hwirq - (bank->gc.base - bank->parent_priv->gpio_base);
}

static int brcmrpc_gpio_irq_set_type(struct irq_data *d, unsigned int type)
{
	struct irq_chip_generic *icg = irq_data_get_irq_chip_data(d);
	struct brcmrpc_gpio_bank *bank = icg->private;
	struct brcmrpc_gpio_priv *priv = bank->parent_priv;
	u32 mask = BIT(brcmrpc_gpio_hwirq_to_offset(d->hwirq, bank));
	u32 edge_insensitive, iedge_insensitive;
	u32 edge_config, iedge_config;
	u32 level, ilevel;
	unsigned long flags;

	switch (type) {
	case IRQ_TYPE_LEVEL_LOW:
		level = mask;
		edge_config = 0;
		edge_insensitive = 0;
		break;
	case IRQ_TYPE_LEVEL_HIGH:
		level = mask;
		edge_config = mask;
		edge_insensitive = 0;
		break;
	case IRQ_TYPE_EDGE_FALLING:
		level = 0;
		edge_config = 0;
		edge_insensitive = 0;
		break;
	case IRQ_TYPE_EDGE_RISING:
		level = 0;
		edge_config = mask;
		edge_insensitive = 0;
		break;
	case IRQ_TYPE_EDGE_BOTH:
		level = 0;
		edge_config = 0;  /* don't care, but want known value */
		edge_insensitive = mask;
		break;
	default:
		return -EINVAL;
	}

	spin_lock_irqsave(&bank->gc.bgpio_lock, flags);

	iedge_config = bank->gc.read_reg(priv->reg_base +
			GIO_EC(bank->id)) & ~mask;
	iedge_insensitive = bank->gc.read_reg(priv->reg_base +
			GIO_EI(bank->id)) & ~mask;
	ilevel = bank->gc.read_reg(priv->reg_base +
			GIO_LEVEL(bank->id)) & ~mask;

	bank->gc.write_reg(priv->reg_base + GIO_EC(bank->id),
			iedge_config | edge_config);
	bank->gc.write_reg(priv->reg_base + GIO_EI(bank->id),
			iedge_insensitive | edge_insensitive);
	bank->gc.write_reg(priv->reg_base + GIO_LEVEL(bank->id),
			ilevel | level);

	spin_unlock_irqrestore(&bank->gc.bgpio_lock, flags);
	return 0;
}

static void brcmrpc_gpio_irq_bank_handler(struct brcmrpc_gpio_bank *bank)
{
	struct brcmrpc_gpio_priv *priv = bank->parent_priv;
	struct irq_domain *domain = priv->irq_domain;
	int hwbase = bank->gc.base - priv->gpio_base;
	unsigned long status;
	unsigned int irq;

	while ((status = brcmrpc_gpio_get_active_irqs(bank))) {
		for_each_set_bit(irq, &status, 32) {
			if (irq >= bank->width)
				dev_warn(&priv->pdev->dev,
					 "IRQ for invalid GPIO (bank=%d, offset=%d)\n",
					 bank->id, irq);
			irq = irq_linear_revmap(domain, irq + hwbase);
			generic_handle_irq(irq);
		}
	}
}

/* Each UPG GIO block has one IRQ for all banks */
static void brcmrpc_gpio_irq_handler(struct irq_desc *desc)
{
	struct brcmrpc_gpio_priv *priv = irq_desc_get_handler_data(desc);
	struct irq_chip *chip = irq_desc_get_chip(desc);
	struct brcmrpc_gpio_bank *bank;

	/* Interrupts weren't properly cleared during probe */
	BUG_ON(!priv || !chip);

	if (unlikely(irqd_irq_disabled(&desc->irq_data))) {
		/*
		 * This should not happen, but if the parent interrupt
		 * controller does not implement the irq_disable operation
		 * in its irq_chip the interrupt may not be masked.
		 * So, mask it here in the chained handler.
		 */
		chip->irq_mask(&desc->irq_data);
		if (chip->irq_eoi)
			chip->irq_eoi(&desc->irq_data);
		return;
	}

	chained_irq_enter(chip, desc);
	list_for_each_entry(bank, &priv->bank_list, node)
		brcmrpc_gpio_irq_bank_handler(bank);
	chained_irq_exit(chip, desc);
}

static int brcmrpc_gpio_sanity_check_banks(struct device *dev,
		enum gpio_chip_type gpio_chip,
		struct device_node *np, struct resource *res)
{
	u32 res_num_banks, num_banks;

	if (gpio_chip == GIO_GPIO_CHIP)
		res_num_banks = ((u32)resource_size(res) / GIO_BANK_SIZE);
	else
		res_num_banks = ((u32)resource_size(res) / TM_GPO_BANK_SIZE);
	num_banks =
		of_property_count_u32_elems(np, GPIO_BANK_WIDTH_PROPERTY);

	if (res_num_banks != num_banks) {
		dev_err(dev, "Mismatch in banks: res had %d, bank-widths had %d\n",
				res_num_banks, num_banks);
		return -EINVAL;
	}

	return 0;
}

static int bcmrpc_gpio_find_pin_offset(struct gpio_chip *gc,
			unsigned int selector)
{
	int cnt;
	struct brcmrpc_gpio_bank *bank = gpiochip_get_data(gc);

	for (cnt = 0; cnt < bank->width; cnt++) {
		if (bank->gpio_pins[cnt].number == selector)
			return cnt;
	}

	return -EINVAL;
}

static int brcmrpc_gpio_of_xlate(struct gpio_chip *gc,
		const struct of_phandle_args *gpiospec, u32 *flags)
{
	struct brcmrpc_gpio_priv *priv = brcmrpc_gpio_gc_to_priv(gc);
	struct brcmrpc_gpio_bank *bank = gpiochip_get_data(gc);
	int offset;

	if (gc->of_gpio_n_cells != 2) {
		WARN_ON(1);
		return -EINVAL;
	}

	if (WARN_ON(gpiospec->args_count < gc->of_gpio_n_cells))
		return -EINVAL;

	offset = gpiospec->args[0] - (gc->base - priv->gpio_base);
	if (offset >= gc->ngpio || offset < 0)
		return -EINVAL;

	if (unlikely(offset >= bank->width)) {
		dev_warn_ratelimited(&priv->pdev->dev,
			"Received request for invalid GPIO offset %d\n",
			gpiospec->args[0]);
	}

	if (flags)
		*flags = gpiospec->args[1];

	return offset;
}

/* priv->parent_irq and priv->num_gpios must be set before calling */
static int brcmrpc_gpio_irq_setup(struct platform_device *pdev,
		struct brcmrpc_gpio_priv *priv)
{
	struct device *dev = &pdev->dev;
	struct device_node *np = dev->of_node;
	struct brcmrpc_gpio_bank *bank;
	struct irq_chip_generic *gc;
	struct irq_chip_type *ct;
	unsigned int clr = IRQ_NOREQUEST | IRQ_NOPROBE | IRQ_NOAUTOEN;
	unsigned int flags;
	int ret, hw_int;

	priv->irq_domain = irq_domain_add_linear(np, priv->num_gpios,
						&irq_generic_chip_ops,
						priv);
	if (!priv->irq_domain) {
		dev_err(dev, "Couldn't allocate IRQ domain\n");
		return -ENXIO;
	}

	/* Configure the peripheral registers for
	 * CPU-native byte order.
	 */
	flags = IRQ_GC_INIT_MASK_CACHE;
	if (IS_ENABLED(CONFIG_CPU_BIG_ENDIAN))
		flags |= IRQ_GC_BE_IO;

	ret = irq_alloc_domain_generic_chips(priv->irq_domain,
					     MAX_GPIO_PER_BANK, 1,
					     dev_name(dev),
					     handle_level_irq, clr, 0, flags);
	if (ret) {
		dev_err(&pdev->dev, "failed to allocate generic irq chip\n");
		goto out_free_domain;
	}

	list_for_each_entry(bank, &priv->bank_list, node) {
		hw_int = bank->id * MAX_GPIO_PER_BANK;
		gc = irq_get_domain_generic_chip(priv->irq_domain, hw_int);
		gc->reg_base = priv->reg_base + bank->id * GIO_BANK_SIZE;
		if (bank->width < sizeof(gc->unused) * 8)
			gc->unused = -((typeof(gc->unused))1 << bank->width);
		gc->private = bank;
		ct = gc->chip_types;
		ct->regs.mask = GIO_MASK(0);
		ct->regs.ack = GIO_STAT(0);
		ct->chip.irq_ack = irq_gc_ack_set_bit;
		ct->chip.irq_mask = irq_gc_mask_clr_bit;
		ct->chip.irq_unmask = irq_gc_mask_set_bit;
		ct->chip.irq_set_type = brcmrpc_gpio_irq_set_type;
		ct->chip.irq_disable = irq_gc_mask_clr_bit;
		/* Ensure that interrupts are masked when changing type */
		ct->chip.flags = IRQCHIP_SET_TYPE_MASKED;
		bank->icg = gc;
	}

	irq_set_chained_handler_and_data(priv->parent_irq,
					 brcmrpc_gpio_irq_handler, priv);

	return 0;

out_free_domain:
	irq_domain_remove(priv->irq_domain);

	return ret;
}

static void brcmrpc_gpio_bank_save(struct brcmrpc_gpio_priv *priv,
				   struct brcmrpc_gpio_bank *bank)
{
	struct gpio_chip *gc = &bank->gc;
	unsigned int i;

	for (i = 0; i < GIO_REG_STAT; i++)
		bank->regs[i] = gc->read_reg(priv->reg_base +
					     GIO_BANK_OFF(bank->id, i));
}

static void brcmrpc_gpio_quiesce(struct device *dev, bool save)
{
	struct brcmrpc_gpio_priv *priv = dev_get_drvdata(dev);
	struct brcmrpc_gpio_bank *bank;
	struct gpio_chip *gc;

	/* disable interrupt */
	if (priv->parent_irq >= 0)
		disable_irq(priv->parent_irq);

	list_for_each_entry(bank, &priv->bank_list, node) {
		gc = &bank->gc;

		if (save)
			brcmrpc_gpio_bank_save(priv, bank);

		gc->write_reg(priv->reg_base + GIO_MASK(bank->id), 0);
	}
}

static void brcmrpc_pinctrl_shutdown(struct platform_device *pdev)
{
	/* Enable GPIO for S5 cold boot */
	brcmrpc_gpio_quiesce(&pdev->dev, 0);
}

static void brcmrpc_gpio_free(struct gpio_chip *gc,
				unsigned int offset)
{
	int rc;
	unsigned int pin;
	const char *pin_name;
	struct brcmrpc_gpio_bank *bank = gpiochip_get_data(gc);

	if (offset >= gc->ngpio)
		return;

	pin = gc->base + offset;
	rc = bcmrpc_gpio_find_pin_offset(gc, pin);
	if (rc < 0)
		return;

	pin_name = bank->gpio_pins[rc].name;
	/* Free GPIO */
	rc = rpc_gpio_free(g_rpc_tunnel, pin, pin_name,
			g_gpio_id[pin]);
	if (rc)
		pr_info("%s : Couldn't free GPIO: %d\n",
			   MODULE_NAME, pin);

}

static int brcmrpc_gpio_request(struct gpio_chip *gc,
			unsigned int offset)
{
	int rc = 0;
	unsigned int pin;
	struct brcmrpc_gpio_bank *bank = gpiochip_get_data(gc);

	if (offset >= gc->ngpio)
		return -EINVAL;

	pin = gc->base + offset;
	rc = bcmrpc_gpio_find_pin_offset(gc, pin);
	if (rc < 0)
		goto out;

	/* Allocate GPIO */
	rc = rpc_gpio_config(bank->parent_priv,
			bank->gpio_pins[rc].name,
			bank->gpio_pins[rc].number,
			RPC_CONFIG_GPIO, RPC_GPIO_ALLOC);
	if (rc)
		goto out;

out:
	return rc;
}

static int brcmrpc_pinctrl_remove(struct platform_device *pdev)
{
	struct brcmrpc_gpio_priv *priv = platform_get_drvdata(pdev);
	struct brcmrpc_gpio_bank *bank;
	int ret = 0, gpio_cnt;

	if (!priv) {
		dev_err(&pdev->dev, "called %s without drvdata!\n", __func__);
		return -EFAULT;
	}

	/*
	 * You can lose return values below, but we report all errors, and it's
	 * more important to actually perform all of the steps.
	 */
	if (priv->pc->pctl_dev)
		devm_pinctrl_unregister(&pdev->dev, priv->pc->pctl_dev);
	list_for_each_entry(bank, &priv->bank_list, node) {
		for (gpio_cnt = 0; gpio_cnt < bank->gc.ngpio; gpio_cnt++)
			brcmrpc_gpio_free(&bank->gc, gpio_cnt);
		gpiochip_remove(&bank->gc);
		if (bank->gpio_pins)
			devm_kfree(&pdev->dev, bank->gpio_pins);
	}
	if (priv->pc->pin_group && priv->pc->pin_group->pin_desc)
		devm_kfree(&pdev->dev, priv->pc->pin_group->pin_desc);

	return ret;
}

static int brcmrpc_pinctrl_parse_dt_pin_or_group(struct device *dev,
			struct brcmrpc_pinctrl *pc,
			struct device_node *np)
{
	struct brcmrpc_pin_group *pin_group = pc->pin_group;
	struct property *pins, *pin_names, *funcs;
	int pin, cnt, func;
	const char *pin_name;
	static unsigned int nr_pins;
	static unsigned int nr_pin_names;
	static unsigned int base_nr_pins;
	static unsigned int base_nr_pin_names;
	unsigned int npins, npin_names, nfuncs;
	int ret = 0;

	pins = of_find_property(np, PINS_PROPERTY, NULL);
	if (!pins)
		return 0;
	pin_names = of_find_property(np, PIN_NAMES_PROPERTY, NULL);
	if (!pin_names) {
		dev_err(dev, "%s: missing %s property\n",
				of_node_full_name(np), PIN_NAMES_PROPERTY);
		return -EINVAL;
	}
	ret = of_property_count_strings(np, PIN_NAMES_PROPERTY);
	if (ret < 0) {
		dev_err(dev, "%s: could not parse property %s\n",
				of_node_full_name(np), PIN_NAMES_PROPERTY);
		return ret;
	}
	npins = (pins->length/4);
	npin_names = ret;
	if (nr_pins > pin_group->nr_pins) {
		dev_err(dev,
			"Number of pins in property %s greater than calculated\n",
			PINS_PROPERTY);
		return -EINVAL;
	}
	nr_pins += npins;
	nr_pin_names += npin_names;
	if (nr_pin_names != nr_pins) {
		dev_err(dev,
			"%s:number of pins and number of pin names not match\n",
			of_node_full_name(np));
		return -EINVAL;
	}

	funcs = of_find_property(np, PMUX_FUNCTION_PROPERTY, NULL);
	if (!funcs) {
		/* EINVAL=missing, which is fine since it's optional */
		dev_dbg(dev, "could not find property %s\n",
                PMUX_FUNCTION_PROPERTY);
	}
	nfuncs = funcs ? (funcs->length / 4) : 0;
	/* If number of function > 1 then it should match the number
	   of pins */
	if (nfuncs > 1 && nfuncs != npins) {
		dev_err(dev,
			"%s: %s must have 1 or %d entries\n",
			of_node_full_name(np),
			PMUX_FUNCTION_PROPERTY,
			npins);
		return -EINVAL;
	}

	for (cnt = 0; cnt < npins; cnt++) {
		ret = of_property_read_u32_index(
				np, PINS_PROPERTY,
				cnt, &pin);
		if (ret)
			goto out;

		pin_group->pin_desc[(cnt + base_nr_pins)].number =
			(pin & PIN_NUMBER_MASK);
	}
	
	/* Append Function number to the Pin number to support multiple functions
	   for a pin otherwise pinctrl registration will fail */
	for (cnt = 0; cnt < nfuncs; cnt++) {
		ret = of_property_read_u32_index(
				np, PMUX_FUNCTION_PROPERTY,
				cnt, &func);
		if (ret)
			goto out;

		if (nfuncs == 1) {
            func = (func << PIN_FUNC_SHIFT);
			for (cnt = 0; cnt < npins; cnt++) {
				pin_group->pin_desc[(cnt + base_nr_pins)].number |=
                    (func & PIN_FUNC_MASK);
            }
            break;
        } else {
			pin_group->pin_desc[(cnt + base_nr_pins)].number |=
				((func << PIN_FUNC_SHIFT) & PIN_FUNC_MASK);
		}
	}
    base_nr_pins = nr_pins;

	for (cnt = 0; cnt < npin_names; cnt++) {
		ret = of_property_read_string_index(
		   np, PIN_NAMES_PROPERTY, cnt, &pin_name);
		if (ret)
			goto out;
		if (!strlen(pin_name)) {
			dev_err(dev, "pin-names string should not be empty.\n");
			ret = -EINVAL;
			goto out;
		}
		pin_group->pin_desc[(cnt + base_nr_pin_names)].name = pin_name;
	}
	base_nr_pin_names = nr_pin_names;

out:
	return ret;
}

static int brcmrpc_pinctrl_parse_dt(struct platform_device *pdev,
			struct brcmrpc_pinctrl *pc)
{
	struct device *dev = &pdev->dev;
	struct device_node *np = dev->of_node;
	struct device_node *cfg_np, *sub_np;
	struct brcmrpc_pin_group *pin_group = pc->pin_group;
	struct pinctrl_pin_desc *pin_desc;
	int ret = 0;

	/*
	 * Iterate over all the child nodes of the pin controller node
	 * and find pin and pin name count.
	 */
	for_each_child_of_node(np, cfg_np) {
		struct property *pins_np, *pin_names_np;

		if (!of_get_child_count(cfg_np)) {
			pins_np = of_find_property(cfg_np,
				PINS_PROPERTY, NULL);
			pin_names_np = of_find_property(cfg_np,
				PIN_NAMES_PROPERTY, NULL);
			if (!pins_np && !pin_names_np)
				continue;
			if (pins_np)
				pin_group->nr_pins += (pins_np->length/4);
			if (pin_names_np) {
				ret = of_property_count_strings(cfg_np,
						PIN_NAMES_PROPERTY);
				if (ret < 0) {
					dev_err(dev,
						"%s: could not parse property %s\n",
						of_node_full_name(np),
						PIN_NAMES_PROPERTY);
					goto out;
				}
				pin_group->nr_pin_names += ret;
			}
			continue;
		}
		for_each_child_of_node(cfg_np, sub_np) {
			pins_np = of_find_property(sub_np,
				PINS_PROPERTY, NULL);
			pin_names_np = of_find_property(sub_np,
				PIN_NAMES_PROPERTY, NULL);
			if (!pins_np && !pin_names_np)
				continue;
			if (pins_np)
				pin_group->nr_pins += (pins_np->length/4);

			if (pin_names_np) {
				ret = of_property_count_strings(sub_np,
						PIN_NAMES_PROPERTY);
				if (ret < 0) {
					dev_err(dev, "could not parse property %s\n",
						PIN_NAMES_PROPERTY);
					goto out;
				}
				pin_group->nr_pin_names += ret;
			}
			continue;
		}
	}
	if (pin_group->nr_pins != pin_group->nr_pin_names) {
		dev_err(dev,
			"number of pins and number of pin_names not match\n");
		ret = -EINVAL;
		goto out;
	}
	if (pin_group->nr_pins < 1) {
		dev_dbg(dev, "No Pins found in  %s node to register pinctrl\n",
				sub_np->name);
		ret = 0;
		goto out;
	}
	pin_group->pin_desc = devm_kzalloc(dev, pin_group->nr_pins *
			sizeof(*pin_desc), GFP_KERNEL);
	if (!pin_group->pin_desc) {
		ret = -ENOMEM;
		goto out;
	}
	pc->pin_group->pin_desc = pin_group->pin_desc;
	/*
	 * Iterate over all the child nodes of the pin controller node
	 * and create pins and pin name lists.
	 */
	for_each_child_of_node(np, cfg_np) {
		if (!of_get_child_count(cfg_np)) {
			ret = brcmrpc_pinctrl_parse_dt_pin_or_group(dev,
					pc, cfg_np);
			if (ret < 0) {
				dev_err(dev, "Parse dt pin or group failed\n");
				goto out;
			}
		}
		for_each_child_of_node(cfg_np, sub_np) {
			ret = brcmrpc_pinctrl_parse_dt_pin_or_group(dev,
					pc, sub_np);
			if (ret < 0) {
				dev_err(dev, "Parse dt sub pin or group failed\n");
				goto out;
			}
		}
	}

out:
	return ret;
}

static int brcmrpc_pinctrl_update_db(struct device *dev,
			struct brcmrpc_pinctrl *pc)
{
	struct brcmrpc_pin_group *pin_group = pc->pin_group;
	const char **groups = NULL;
	int cnt;

	groups = devm_kzalloc(dev, sizeof(*groups) *
				pin_group->nr_pins, GFP_KERNEL);
	if (!groups)
		return -ENOMEM;

	pc->groups = groups;

	for (cnt = 0; cnt < pin_group->nr_pins; cnt++)
		groups[cnt] = pin_group->pin_desc[cnt].name;

	return 0;
}

static int brcmrpc_pinctrl_register(struct platform_device *pdev,
		struct brcmrpc_gpio_priv *priv)
{
	struct device *dev = &pdev->dev;
	struct brcmrpc_pinctrl *pc = priv->pc;
	struct pinctrl_desc *brcmrpc_pinctrl_desc = &pc->pdesc;
	struct brcmrpc_pin_group *pin_group;
	int ret;

	brcmrpc_pinctrl_desc->name = MODULE_NAME;
	brcmrpc_pinctrl_desc->owner = THIS_MODULE;
	brcmrpc_pinctrl_desc->pctlops = &bcmrpc_pctl_ops;
	brcmrpc_pinctrl_desc->pmxops = &bcmrpc_pmx_ops;
	brcmrpc_pinctrl_desc->confops = &bcmrpc_pinconf_ops;

	pin_group = devm_kzalloc(dev, sizeof(*pin_group), GFP_KERNEL);
	if (!pin_group)
		return -ENOMEM;

	pc->pin_group = pin_group;

	ret = brcmrpc_pinctrl_parse_dt(pdev, pc);
	if (ret) {
		dev_err(dev, "Parse Pincntrl device tree failed\n");
		return ret;
	}
	brcmrpc_pinctrl_desc->pins = pc->pin_group->pin_desc;
	brcmrpc_pinctrl_desc->npins = pc->pin_group->nr_pins;

	ret = brcmrpc_pinctrl_update_db(dev, pc);
	if (ret) {
		dev_err(dev, "Pincntrl group name update failed\n");
		return ret;
	}

	pc->pctl_dev = devm_pinctrl_register(dev, brcmrpc_pinctrl_desc, pc);
	ret = IS_ERR(pc->pctl_dev);
	if (ret) {
		dev_err(dev, "Could not register pinctrl %d\n", ret);
		return ret;
	}

	return 0;
}

static int brcmrpc_gpo_direction_input(struct gpio_chip *gc,
				       unsigned int offset)
{
	/* This device is output only */
	return -EINVAL;
}

static int brcmrpc_pinctrl_update_gpio_db(
			struct device *dev,
			struct brcmrpc_gpio_bank *bank,
			int gpio_base)
{
	struct pinctrl_pin_desc *gpio_pins = bank->gpio_pins;
	struct brcmrpc_pinctrl *pc = bank->parent_priv->pc;
	u32 bank_width = bank->width;
	int cnt, name_cnt;
	char *buff;
	u32 offset = 0, gpio_str_size = 0, max_size = 0;

	/* Calculate for Max size of differnt gpio string name */
	for (name_cnt = 0; name_cnt < pc->dt_ngpio_names; name_cnt++) {
		gpio_str_size = strlen(pc->gpio_info[name_cnt].names) +
					pc->gpio_info[name_cnt].nformat + 1;
		if (max_size < gpio_str_size)
			max_size = gpio_str_size;
	}
	/* Allocate memory from max string length and bank width */
	buff = devm_kzalloc(dev, max_size*bank_width, GFP_KERNEL);
	if (!buff)
		return -ENOMEM;

	bank->data = (char *)buff;
	for (cnt = 0; cnt < bank_width; cnt++) {
		for (name_cnt = 0; name_cnt <
			 pc->dt_ngpio_names; name_cnt++) {
			if ((cnt + gpio_base) <
			   (pc->gpio_info[name_cnt].offset +
				pc->gpio_info[name_cnt].ngpios)) {
				offset = (cnt * max_size);
				snprintf(&buff[offset],
					max_size, "%s%0*u",
					pc->gpio_info[name_cnt].names,
					pc->gpio_info[name_cnt].nformat,
					((cnt + gpio_base) -
					pc->gpio_info[name_cnt].offset));
				gpio_pins[cnt].number = (cnt + gpio_base);
				gpio_pins[cnt].name =
					(const char *)&buff[offset];
				break;
			}
		}
	}

	return 0;
}

static int brcmrpc_parse_dt_gpio_info(struct device *dev,
			struct device_node *np,
			struct brcmrpc_pinctrl *pc)
{
	struct brcmrpc_gpio_info *gpio_info;
	struct device_node *phan_node;
	const char *name;
	struct property *gpio_pins_info;
	int ret;
	u32 name_cnt, cnt,
		info_cnt, offset, info;

	phan_node = of_parse_phandle(np, RPC_CHANNEL_NODE, 0);
	if (!phan_node) {
		dev_err(dev, "Unable to retrieve %s phandle ",
				RPC_CHANNEL_NODE);
		return -EINVAL;
	}

	if (of_property_read_string(phan_node,
		DEVICE_NAME_PROPERTY, &name)) {
		dev_err(dev, "%s: Missing %s property!\n",
				of_node_full_name(np),
				DEVICE_NAME_PROPERTY);
		return -ENOENT;
	}
	of_node_put(phan_node);
	g_rpc_tunnel = rpc_get_fifo_tunnel_id((char *)name);
	if (g_rpc_tunnel < 0) {
		dev_err(dev, "%s: Unable to obtain RPC tunnel ID.\n",
			__func__);
		return -EIO;
	}

	ret = of_property_count_strings(np, GPIO_NAMES_PROPERTY);
	if (ret < 1) {
		dev_err(dev, "%s: could not parse property %s\n",
				of_node_full_name(np), GPIO_NAMES_PROPERTY);
		return -EINVAL;
	}
	name_cnt = ret;
	gpio_pins_info = of_find_property(np, GPIO_PINS_INFO_PROPERTY, NULL);
	if (!gpio_pins_info) {
		dev_err(dev, "%s: missing %s property\n",
				of_node_full_name(np),
				GPIO_PINS_INFO_PROPERTY);
		return -EINVAL;
	}
	if (name_cnt != ((gpio_pins_info->length/4)/GPIO_PIN_MAX)) {
		dev_err(dev, "%s: number of gpio names and pin info property mismatch\n",
				of_node_full_name(np));
		return -EINVAL;
	}

	pc->dt_ngpio_names = name_cnt;
	gpio_info = devm_kzalloc(dev, sizeof(*gpio_info)*name_cnt, GFP_KERNEL);
	if (!gpio_info) {
		dev_err(dev, "gpio info memory allocate failure\n");
		return -ENOMEM;
	}
	pc->gpio_info = gpio_info;

	for (cnt = 0; cnt < name_cnt; cnt++) {
		if (of_property_read_string_index(np,
					GPIO_NAMES_PROPERTY, cnt, &name)) {
			dev_err(dev, "%s: %s property value read failure\n",
				of_node_full_name(np), GPIO_NAMES_PROPERTY);
			return -EINVAL;
		}
		gpio_info[cnt].names = name;

		for (info_cnt = 0; info_cnt < GPIO_PIN_MAX; info_cnt++) {
			offset = (cnt * GPIO_PIN_MAX) + info_cnt;
			if (of_property_read_u32_index(np,
				GPIO_PINS_INFO_PROPERTY, offset, &info)) {
				dev_err(dev, "%s: %s property value read failure\n",
					of_node_full_name(np),
					GPIO_PINS_INFO_PROPERTY);
				return -EINVAL;
			}
			if (info_cnt == GPIO_PIN_OFFSET)
				gpio_info[cnt].offset = info;
			else if (info_cnt == GPIO_PIN_NGPIOS) {
				gpio_info[cnt].ngpios = info;
				pc->dt_ngpio += info;
			} else if (info_cnt == GPIO_PIN_FORMAT)
				gpio_info[cnt].nformat = info;
			else
				gpio_info[cnt].muxfunc = info;
		}
	}

	return 0;
}

static int brcmrpc_gpo_add_data(struct device *dev,
			struct brcmrpc_gpio_bank *bank, int gpio_base)
{
	struct brcmrpc_gpio_priv *priv = bank->parent_priv;
	void __iomem *reg_base = priv->reg_base;
	struct device_node *np = dev->of_node;
	struct gpio_chip *gc = &bank->gc;
	unsigned long flags = 0;
	int ret;

	/*
	 * ARM either do not support big endian, or
	 * else leave I/O in little endian mode.
	 */
#if defined(__BIG_ENDIAN)
	flags = BGPIOF_BIG_ENDIAN_BYTE_ORDER;
#endif
	/*
	 * Regs are 4 bytes wide, have data reg, no set/clear regs,
	 * and direction bits have 0 = output and 1 = input
	 */
	ret = bgpio_init(gc, dev, 4,
			reg_base + TM_GPO_DATA(bank->id),
			NULL, NULL, reg_base + TM_GPO_EN(bank->id),
			NULL, flags);
	if (ret) {
		dev_err(dev, "bgpio_init() failed\n");
		return ret;
	}
	gc->request = brcmrpc_gpio_request;
	gc->free = brcmrpc_gpio_free;
	gc->direction_input = brcmrpc_gpo_direction_input;
	gc->of_node = np;
	gc->owner = THIS_MODULE;
	gc->label = np->full_name;
	gc->base = gpio_base;
	/* not all ngpio lines are valid, will use bank width later */
	gc->ngpio = MAX_GPIO_PER_BANK;

	ret = gpiochip_add_data(gc, bank);
	if (ret) {
		dev_err(dev, "Could not add gpiochip for bank %d\n",
				bank->id);
		return ret;
	}

	return 0;
}

static int brcmrpc_gpio_add_data(struct device *dev,
			struct brcmrpc_gpio_bank *bank, int gpio_base)
{
	struct brcmrpc_gpio_priv *priv = bank->parent_priv;
	struct device_node *np = dev->of_node;
	void __iomem *reg_base = priv->reg_base;
	struct gpio_chip *gc = &bank->gc;
	unsigned long flags = 0;
	int ret;

	/*
	 * ARM either do not support big endian, or
	 * else leave I/O in little endian mode.
	 */
#if defined(__BIG_ENDIAN)
	flags = BGPIOF_BIG_ENDIAN_BYTE_ORDER;
#endif
	/*
	 * Regs are 4 bytes wide, have data reg, no set/clear regs,
	 * and direction bits have 0 = output and 1 = input
	 */
	ret = bgpio_init(gc, dev, 4,
			reg_base + GIO_DATA(bank->id),
			NULL, NULL, NULL,
			reg_base + GIO_IODIR(bank->id), flags);
	if (ret) {
		dev_err(dev, "bgpio_init() failed\n");
		return ret;
	}
	gc->request = brcmrpc_gpio_request;
	gc->free = brcmrpc_gpio_free;
	gc->of_node = np;
	gc->owner = THIS_MODULE;
	gc->label = np->full_name;
	gc->base = gpio_base;
	gc->of_gpio_n_cells = 2;
	gc->of_xlate = brcmrpc_gpio_of_xlate;
	/* not all ngpio lines are valid, will use bank width later */
	gc->ngpio = MAX_GPIO_PER_BANK;
	if (priv->parent_irq >= 0)
		gc->to_irq = brcmrpc_gpio_to_irq;

	/*
	 * Mask all interrupts by default
	 */
	gc->write_reg(reg_base + GIO_MASK(bank->id), 0);

	ret = gpiochip_add_data(gc, bank);
	if (ret) {
		dev_err(dev, "Could not add gpiochip for bank %d\n",
				bank->id);
		return ret;
	}

	return 0;
}

static int brcmrpc_gpio_register(struct platform_device *pdev,
		struct brcmrpc_gpio_priv *priv, struct resource *res)
{
	struct device *dev = &pdev->dev;
	struct device_node *np = dev->of_node;
	struct property *prop;
	const __be32 *p;
	u32 bank_width;
	int num_banks = 0, gpio_cnt = 0;
	int ret;
	static int gpio_base;
	struct brcmrpc_gpio_bank *bank;
	struct gpio_chip *gc;
	struct pinctrl_pin_desc *gpio_pins;
	const char *name;
	enum gpio_chip_type gpio_chip = 0;
	unsigned int pin_id = 0;

	priv->gpio_base = gpio_base;

	if (of_property_read_string(np, COMPATIBLE_PROPERTY, &name)) {
		dev_err(dev, "%s: Missing %s property!\n",
				of_node_full_name(np),
				COMPATIBLE_PROPERTY);
		return -ENOENT;
	}
	if (!strcmp(name, COMPATIBLE_GPIO))
		gpio_chip = GIO_GPIO_CHIP;
	else if (!strcmp(name, COMPATIBLE_GPO))
		gpio_chip = TM_GPO_CHIP;
	else {
		dev_err(dev, "Invalid %s property value\n",
				COMPATIBLE_PROPERTY);
		return -EINVAL;
	}

	/* Get the GPIO information from device tree to process further */
	ret = brcmrpc_parse_dt_gpio_info(dev, np, priv->pc);
	if (ret)
		return ret;

	/* Check for GPIO bank information w.r.t reserved info*/
	ret = brcmrpc_gpio_sanity_check_banks(dev, gpio_chip, np, res);
	if (ret)
		return ret;

	of_property_for_each_u32(np, GPIO_BANK_WIDTH_PROPERTY, prop, p,
			bank_width) {
		bank = devm_kzalloc(dev, sizeof(*bank), GFP_KERNEL);
		if (!bank) {
			dev_err(dev, "bank memory allocate failure\n");
			ret = -ENOMEM;
			goto out;
		}
		bank->parent_priv = priv;
		bank->id = num_banks;
		if (bank_width <= 0 || bank_width > MAX_GPIO_PER_BANK) {
			dev_err(dev, "Invalid bank width %d\n", bank_width);
			ret = -EINVAL;
			goto out;
		} else
			bank->width = bank_width;

		if ((gpio_base + bank_width) > NUM_GPIOS) {
			dev_err(dev, "Total bank width exceed total number of GPIO supported %d\n",
					bank_width);
			ret = -EINVAL;
			goto out;
		}
		gpio_pins = devm_kzalloc(dev, sizeof(*gpio_pins)*bank_width,
					GFP_KERNEL);
		if (!gpio_pins) {
			ret = -ENOMEM;
			goto out;
		}
		bank->gpio_pins = gpio_pins;
		ret = brcmrpc_pinctrl_update_gpio_db(dev,
				bank, gpio_base);
		if (ret) {
			dev_err(dev, "Update GPIO Pin name and number fails\n");
			goto out;
		}
		/* Allocate GPIO for access to initialize */
		for (gpio_cnt = 0; gpio_cnt < bank_width; gpio_cnt++) {
			if (gpio_pins[gpio_cnt].name) {
				/* Allocate GPIO */
				ret = rpc_gpio_alloc(priv->pc, g_rpc_tunnel,
						gpio_pins[gpio_cnt].name,
						gpio_pins[gpio_cnt].number,
						&pin_id,
						RPC_CONFIG_GPIO);
				if (ret)
					dev_dbg(dev, "could not allocate gpio:%d\n",
						 gpio_pins[gpio_cnt].number);
				else {
					/* Allocate Pinmux */
					ret = rpc_pinconf_alloc(g_rpc_tunnel,
						gpio_pins[gpio_cnt].number,
						gpio_pins[gpio_cnt].name,
						&pin_id);
					if (ret)
						dev_dbg(dev, "could not allocate gpio:%d for pinmux\n",
						 gpio_pins[gpio_cnt].number);
				}
			}
		}
		gc = &bank->gc;
		if (gpio_chip == GIO_GPIO_CHIP) {
			ret = brcmrpc_gpio_add_data(dev, bank, gpio_base);
			if (ret) {
				dev_err(dev, "GPIO chip add data failed\n");
				goto out;
			}
		} else {
			ret = brcmrpc_gpo_add_data(dev, bank, gpio_base);
			if (ret) {
				dev_err(dev, "GPO chip add data failed\n");
				goto out;
			}
		}
		for (gpio_cnt = 0; gpio_cnt < bank_width; gpio_cnt++) {
			pin_id = g_gpio_id[gpio_pins[gpio_cnt].number];
			if (g_gpio_used[pin_id] != RPC_CONFIG_NONE)
				rpc_gpio_config(priv,
				   gpio_pins[gpio_cnt].name,
				   gpio_pins[gpio_cnt].number,
				   RPC_CONFIG_GPIO, RPC_GPIO_FREE);
		}
		gpio_base += gc->ngpio;
		dev_dbg(dev, "bank=%d, base=%d, ngpio=%d, width=%d\n",
			bank->id, gc->base, gc->ngpio, bank->width);

		/* Everything looks good, so add bank to list */
		list_add(&bank->node, &priv->bank_list);
		num_banks++;
	}
	priv->num_gpios = gpio_base - priv->gpio_base;
	if (priv->parent_irq >= 0) {
		ret = brcmrpc_gpio_irq_setup(pdev, priv);
		if (ret)
			goto out;
	}

	dev_info(dev, "Registered %d banks (GPIO(s): %d-%d)\n",
			num_banks, priv->gpio_base, gpio_base - 1);

out:
	return ret;

}

static int brcmrpc_pinctrl_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct device_node *np = dev->of_node;
	void __iomem *reg_base;
	struct brcmrpc_gpio_priv *priv;
	struct brcmrpc_pinctrl *pc;
	struct resource *res;
	int ret;

	priv = devm_kzalloc(dev, sizeof(*priv), GFP_KERNEL);
	if (!priv)
		return -ENOMEM;

	pc = devm_kzalloc(dev, sizeof(*pc), GFP_KERNEL);
	if (!pc)
		return -ENOMEM;

	platform_set_drvdata(pdev, priv);
	INIT_LIST_HEAD(&priv->bank_list);

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	reg_base = devm_ioremap_resource(dev, res);
	if (IS_ERR(reg_base))
		return PTR_ERR(reg_base);
	priv->reg_base = reg_base;
	priv->pdev = pdev;
	priv->pc = pc;

	if (of_property_read_bool(np, INTR_CONTROLLER_PROPERTY)) {
		priv->parent_irq = platform_get_irq(pdev, 0);
		if (priv->parent_irq <= 0) {
			dev_err(dev, "Couldn't get IRQ");
			return -ENOENT;
		}
	} else
		priv->parent_irq = -ENOENT;

	mutex_init(&priv->lock);

	ret = brcmrpc_gpio_register(pdev, priv, res);
	if (ret)
		goto out;

	ret = brcmrpc_pinctrl_register(pdev, priv);
	if (ret)
		goto out;

	return 0;

out:
	(void) brcmrpc_pinctrl_remove(pdev);
	return ret;
}

static const struct of_device_id brcmrpc_pinctrl_of_match[] = {
	{ .compatible = "brcm,bcmrpc-pinctrl-gpio" },
	{ .compatible = "brcm,bcmrpc-pinctrl-gpo" },
	{},
};

MODULE_DEVICE_TABLE(of, brcmrpc_gpio_of_match);

static struct platform_driver brcmrpc_pinctrl_driver = {
	.driver = {
		.name = MODULE_NAME,
		.of_match_table = brcmrpc_pinctrl_of_match,
	},
	.probe = brcmrpc_pinctrl_probe,
	.remove = brcmrpc_pinctrl_remove,
	.shutdown = brcmrpc_pinctrl_shutdown,
};

static int __init brcmrpc_pinctrl_init(void)
{
	int rc = 0;

	rc = rpc_register_functions(RPC_SERVICE_GPIO,
			rpc_gpio_services_tbl,
			RPC_GPIO_FUNC_MAX);
	if (rc) {
		pr_err("%s: Failed to register GPIO RPC function(s).\n",
			MODULE_NAME);
		goto out;
	}

	rc = rpc_register_functions(RPC_SERVICE_PINCTRL,
			rpc_pinctrl_services_tbl,
			RPC_PINCTRL_FUNC_MAX);
	if (rc) {
		pr_err("%s: Failed to register PINCTRL RPC function(s).\n",
				MODULE_NAME);
		goto err_unreg_gpio_rpc;
	}

	rc = platform_driver_register(&brcmrpc_pinctrl_driver);
	if (rc) {
		pr_err("%s: Failed to register platform driver.\n",
			   MODULE_NAME);
		goto err_unreg_pinctrl_rpc;
	}
	return 0;

err_unreg_pinctrl_rpc:
	rpc_unregister_functions(RPC_SERVICE_PINCTRL);
err_unreg_gpio_rpc:
	rpc_unregister_functions(RPC_SERVICE_GPIO);
out:
	return rc;
}
subsys_initcall(brcmrpc_pinctrl_init);

static void __exit brcmrpc_pinctrl_exit(void)
{
	platform_driver_unregister(&brcmrpc_pinctrl_driver);
	rpc_unregister_functions(RPC_SERVICE_PINCTRL);
	rpc_unregister_functions(RPC_SERVICE_GPIO);
}
module_exit(brcmrpc_pinctrl_exit);

MODULE_AUTHOR("Venky Selvaraj");
MODULE_DESCRIPTION("BCMRPC Pin control driver");
MODULE_LICENSE("GPL v2");
