 /****************************************************************************
 *
 * Copyright (c) 2023-2023 Broadcom Corporation
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 *******************************************************************************
 *
 * Author: Venky Selvaraj
 *
 ******************************************************************************/

#include <linux/module.h>
#include <linux/cdev.h>
#include <linux/uaccess.h>
#include <linux/dma-mapping.h>
#include <linux/of_platform.h>

#include <linux/bcm_media_gw/itc_rpc/itc_rpc.h>
#include <linux/bcm_media_gw/smc_svn/smc_svn.h>

// RPC events sent from eRouter and processed by SMC
typedef enum {
   RPC_SMC_SYS_SET_SVN_FUNC = 34,                 /* 34 */
   RPC_SMC_SYS_CHECK_SVN_FUNC,                    /* 35 */
   RPC_SMC_SYS_GET_BP3_DEBUG_DATA_FUNC,           /* 36 */
   RPC_SMC_SYS_SVN_FUNC_MAX                       /* 37 */
} rpc_smc_sys_svn_func;

/* Create a user space device for sending messages */
#define SMC_SVN_CLASS        "smc_svn"
#define SMC_SVN_DEVNAME      "smc_svn"
#define SMC_SVN_MAX_DEVS     1

#define SMC_SVN_8BYTE_ALIGN     8
#define SMC_SVN_MAX_SVN_SIZE  4096

#define ANSI_BLACK	"\e[30;1m"
#define ANSI_RED	"\e[31;1m"
#define ANSI_GREEN	"\e[32;1m"
#define ANSI_YELLOW	"\e[33;1m"
#define ANSI_BLUE	"\e[34;1m"
#define ANSI_MAGENTA	"\e[35;1m"
#define ANSI_CYAN	"\e[36;1m"
#define ANSI_WHITE	"\e[37;1m"
#define ANSI_RESET	"\e[0m"
#define BLK(str)	ANSI_BLK str ANSI_RESET
#define RED(str)	ANSI_RED str ANSI_RESET
#define GRN(str)	ANSI_GREEN str ANSI_RESET
#define YLW(str)	ANSI_YELLOW str ANSI_RESET
#define BLU(str)	ANSI_BLUE str ANSI_RESET
#define MAG(str)	ANSI_MAGENTA str ANSI_RESET
#define CYN(str)	ANSI_CYAN str ANSI_RESET
#define WHT(str)	ANSI_WHITE str ANSI_RESET

static struct class     *smc_svn_class;
static int              smc_svn_major;
static dev_t            smc_svn_dev;
static struct cdev      smc_svn_cdev;
static struct device    *smc_svn_device;

static int rpc_rg_smc_tunnel = -1;

static inline void smc_svn_rpc_msg_set_addr(rpc_msg *msg, u64 v)
{
	msg->data[1] = (u32)(v & 0xffffffff);
	msg->data[2] = (msg->data[2] & ~(0xff)) |
		(u32)((v >> 32) & (0xff));
}

static inline void smc_svn_rpc_msg_set_size(rpc_msg *msg, u16 v)
{
	msg->data[2] = (msg->data[2] & ~(0xffff << 8)) |
		(u32)((v & 0xffff) << 8);
}

static inline void smc_svn_rpc_msg_set_bank(rpc_msg *msg, bool v)
{
	msg->data[0] = (msg->data[0] & ~(0x1)) | (v & 0x1);
}

static inline u8 smc_svn_rpc_msg_get_retcode(rpc_msg *msg)
{
	return ((msg->data[0] >> 24) & 0xff);
}

int smc_svn_open(struct inode *inode, struct file *file)
{
	return 0;
}

int smc_svn_release(struct inode *inode, struct file *file)
{
	return 0;
}

ssize_t smc_svn_read(struct file *file,
		                  char __user *buf,
		                  size_t count,
		                  loff_t *ppos)
{
	return 0;
}

ssize_t smc_svn_write(struct file *file,
			                const char __user *buf,
			                size_t count,
			                loff_t *ppos)
{
   return 0;
}


int smc_svn_rpc_send_sys_set_svn(dma_addr_t dma_addr, unsigned int buffer_size)
{
    rpc_msg msg;
    int ret;

    pr_debug("%s, rpc_rg_smc_tunnel:%d", __FUNCTION__, rpc_rg_smc_tunnel);
    rpc_msg_init(&msg, RPC_SERVICE_SYS, (uint32_t)(RPC_SMC_SYS_SET_SVN_FUNC), 0, 0, 0, 0);
    smc_svn_rpc_msg_set_addr(&msg, (u64)dma_addr);
    smc_svn_rpc_msg_set_size(&msg, (u16)buffer_size);
    dma_sync_single_for_device(smc_svn_device, dma_addr, buffer_size, DMA_BIDIRECTIONAL);
    ret = rpc_send_request(rpc_rg_smc_tunnel, (rpc_msg *)&msg);
    if (unlikely(ret)) {
        pr_err(RED("rpc_send_request failure (%d) for svn set\n"),
                ret);
        rpc_dump_msg(&msg);
        return ret;
    }
    ret = smc_svn_rpc_msg_get_retcode(&msg);
    if (unlikely(ret)) {
        pr_err(RED("smc svn set msg retcode %d\n"), (s8)ret);
        rpc_dump_msg(&msg);
        ret = -EIO;
    }
    dma_sync_single_for_cpu(smc_svn_device, dma_addr, buffer_size, DMA_BIDIRECTIONAL);

    return ret;
}

int smc_svn_rpc_send_sys_check_svn(dma_addr_t dma_addr, unsigned int buffer_size, bool bank)
{
    rpc_msg msg;
    int ret;

    pr_debug("%s, rpc_rg_smc_tunnel:%d", __FUNCTION__, rpc_rg_smc_tunnel);
    rpc_msg_init(&msg, RPC_SERVICE_SYS, (uint32_t)(RPC_SMC_SYS_CHECK_SVN_FUNC), 0, 0, 0, 0);
    smc_svn_rpc_msg_set_addr(&msg, (u64)dma_addr);
    smc_svn_rpc_msg_set_size(&msg, (u16)buffer_size);
    smc_svn_rpc_msg_set_bank(&msg, bank);
    dma_sync_single_for_device(smc_svn_device, dma_addr, buffer_size, DMA_BIDIRECTIONAL);
    ret = rpc_send_request(rpc_rg_smc_tunnel, (rpc_msg *)&msg);
    if (unlikely(ret)) {
        pr_err(RED("rpc_send_request failure (%d) for svn check\n"),
                ret);
        rpc_dump_msg(&msg);
        return ret;
    }
    ret = smc_svn_rpc_msg_get_retcode(&msg);
    if (unlikely(ret)) {
        pr_err(RED("smc svn check msg retcode %d\n"), (s8)ret);
        rpc_dump_msg(&msg);
        ret = -EIO;
    }
    dma_sync_single_for_cpu(smc_svn_device, dma_addr, buffer_size, DMA_BIDIRECTIONAL);

    return ret;
}

/* BP3 Debug data request from SMC */
static int smc_svn_rpc_req_sys_get_bp3_debug_data(dma_addr_t dma_addr, unsigned int *buffer_size, unsigned int *info)
{
    rpc_msg msg;
    int ret = 0;
    u32 reply_size = 0;

    pr_debug("%s, rpc_rg_smc_tunnel:%d", __FUNCTION__, rpc_rg_smc_tunnel);
    rpc_msg_init(&msg, RPC_SERVICE_SYS, (uint32_t)(RPC_SMC_SYS_GET_BP3_DEBUG_DATA_FUNC), 0, 0, 0, 0);
    smc_svn_rpc_msg_set_addr(&msg, (u64)dma_addr);
    smc_svn_rpc_msg_set_size(&msg, (u16)*buffer_size);
    dma_sync_single_for_device(smc_svn_device, dma_addr, *buffer_size, DMA_BIDIRECTIONAL);
    ret = rpc_send_request(rpc_rg_smc_tunnel, (rpc_msg *)&msg);
    if (unlikely(ret)) {
        pr_err(RED("rpc_send_request failure (%d) for svn check\n"),
                ret);
        rpc_dump_msg(&msg);
        return ret;
    } else {
        /* Sanity check the reply size and RC */
        reply_size = (msg.data[2] & 0x00FFFF00) >> 8;
        ret = smc_svn_rpc_msg_get_retcode(&msg);
        if (unlikely(ret)) {
            if( (ret == 0xFE) && (reply_size > (*buffer_size)) )
                pr_err(RED("Get BP3 License return error size:%d is larger than buffer %d\n"),
                    reply_size, *buffer_size);
            else {
                pr_err(RED("Get BP3 License return error size:%d  retcode:%d\n"),
                    reply_size, (s8)ret);
            }
            rpc_dump_msg(&msg);
            ret = -EIO;
        }
        *buffer_size = reply_size;
        /* Copy BP3 License Version info */
        *info = (unsigned int)msg.data[1];
    }
    dma_sync_single_for_cpu(smc_svn_device, dma_addr, *buffer_size, DMA_BIDIRECTIONAL);

    return ret;
}

static char *smc_svn_alloc_buffer(struct smc_svn_req *smc_svn_request, unsigned int *buf_size)
{
	char *buffer = NULL;
	unsigned int buffer_size;

	buffer_size = smc_svn_request->buff_length;
	if( buffer_size > SMC_SVN_MAX_SVN_SIZE )
	{
		return NULL;
	}
	buffer_size = ALIGN(buffer_size, SMC_SVN_8BYTE_ALIGN);
	buffer = devm_kzalloc(smc_svn_device, buffer_size, GFP_KERNEL);
	*buf_size = buffer_size;

	return buffer;
}

static long smc_svn_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	int ret = 0;
	unsigned int buffer_size;
	char *buffer = NULL;
	struct smc_svn_req smc_svn_request;
	unsigned int extra_info;
	dma_addr_t dma_addr;

	ret = copy_from_user(&smc_svn_request, (void *)arg, sizeof(smc_svn_request));
	if (ret)
	{
		dev_err(smc_svn_device, "%s: can't copy smc_svn_req from userspace\n", __func__);
		return ret;
	}

	switch (cmd)
	{
		case SMC_SVN_IOCTL_SET_SVN:
		{
			buffer = smc_svn_alloc_buffer( &smc_svn_request, &buffer_size );
			if( !buffer )
			{
				return -EINVAL;
			}
			dma_addr = dma_map_single(smc_svn_device, buffer, buffer_size, DMA_BIDIRECTIONAL);
			if( dma_mapping_error( smc_svn_device, dma_addr ) ) {
				dev_err( smc_svn_device, "DMA address not mapped\n" );
				devm_kfree(smc_svn_device, buffer);
				return -EINVAL;
			}

			ret = copy_from_user(buffer, smc_svn_request.buffer, buffer_size);
			if (ret)
			{
				dev_err(smc_svn_device, "Failure copying SVN data from user. Abort set.");
				dma_unmap_single(smc_svn_device, dma_addr, buffer_size, DMA_BIDIRECTIONAL);
				devm_kfree(smc_svn_device, buffer);
				return ret;
			}

			ret = smc_svn_rpc_send_sys_set_svn(dma_addr, buffer_size);

			dma_unmap_single(smc_svn_device, dma_addr, buffer_size, DMA_BIDIRECTIONAL);
			devm_kfree(smc_svn_device, buffer);

			break;
		}

		case SMC_SVN_IOCTL_CHECK_SVN:
		{
			buffer = smc_svn_alloc_buffer( &smc_svn_request, &buffer_size );
			if( !buffer )
			{
				return -EINVAL;
			}
			dma_addr = dma_map_single(smc_svn_device, buffer, buffer_size, DMA_BIDIRECTIONAL);
			if( dma_mapping_error( smc_svn_device, dma_addr ) ) {
				dev_err( smc_svn_device, "DMA address not mapped\n" );
				devm_kfree(smc_svn_device, buffer);
				return -EINVAL;
			}

			ret = copy_from_user(buffer, smc_svn_request.buffer, buffer_size);
			if (ret)
			{
				dev_err(smc_svn_device, "Failure checking SVN data from user. Abort check.");
				dma_unmap_single(smc_svn_device, dma_addr, buffer_size, DMA_BIDIRECTIONAL);
				devm_kfree(smc_svn_device, buffer);
				return ret;
			}

			ret = smc_svn_rpc_send_sys_check_svn(dma_addr, buffer_size, smc_svn_request.bank);

			dma_unmap_single(smc_svn_device, dma_addr, buffer_size, DMA_BIDIRECTIONAL);
			devm_kfree(smc_svn_device, buffer);

			break;
		}

		case SMC_SVN_IOCTL_GET_BP3_DEBUG_DATA:
		{
			buffer = smc_svn_alloc_buffer( &smc_svn_request, &buffer_size );
			if( !buffer )
			{
				return -EINVAL;
			}

			dma_addr = dma_map_single(smc_svn_device, buffer, buffer_size, DMA_BIDIRECTIONAL);
			if( dma_mapping_error( smc_svn_device, dma_addr ) ) {
				dev_err( smc_svn_device, "DMA address not mapped\n" );
				devm_kfree(smc_svn_device, buffer);
				return -EINVAL;
			}

			ret = smc_svn_rpc_req_sys_get_bp3_debug_data(dma_addr, &buffer_size, &extra_info);
			if( ret )
			{
				dev_err(smc_svn_device, "Failure retrieving BP3 Debug data\n");
				dma_unmap_single(smc_svn_device, dma_addr, buffer_size, DMA_BIDIRECTIONAL);
				devm_kfree(smc_svn_device, buffer);
				return ret;
			}

			smc_svn_request.buff_length = buffer_size;
			ret = copy_to_user(smc_svn_request.buffer, buffer, buffer_size);
			ret = copy_to_user(smc_svn_request.extra_info, &extra_info, sizeof(extra_info));

			dma_unmap_single(smc_svn_device, dma_addr, buffer_size, DMA_BIDIRECTIONAL);
			devm_kfree(smc_svn_device, buffer);

			break;
		}

		default:
		{
			dev_err(smc_svn_device, "Unsupported IOCTL %d\n", cmd);
			ret = -EINVAL;
			break;
		}
	}

	if( ret )
	{
		dev_err(smc_svn_device, "Error during cmd %d\n", cmd);
		return ret;
	}

	// Copy results to user if we have any
	ret = copy_to_user((void *)arg, &smc_svn_request, sizeof(smc_svn_request));
	if (ret)
	{
		dev_err(smc_svn_device, "%s: can't copy smc_svn_req to userspace\n", __func__);
	}

	return ret;
}

static const struct file_operations smc_svn_fops = {
	.owner            = THIS_MODULE,
	.open             = smc_svn_open,
	.release          = smc_svn_release,
	.read             = smc_svn_read,
	.write            = smc_svn_write,
	.unlocked_ioctl   = smc_svn_ioctl,
};

int __init smc_svn_init(void)
{
	int status = 0;

	rpc_rg_smc_tunnel = rpc_get_fifo_tunnel_id("rg-smc");
	if (rpc_rg_smc_tunnel == -1)
		dev_err(smc_svn_device, "Unable to get rg-smc RPC tunnel");

	/*status = rpc_register_functions(RPC_SERVICE_SYS,
                                   smc_svn_table,
                                   sizeof(smc_svn_table)/
                                   sizeof(rpc_function));*/

	smc_svn_class = class_create(THIS_MODULE, SMC_SVN_CLASS);
	if (IS_ERR(smc_svn_class)) {
		dev_err(smc_svn_device, "class_create() failed for smc_svn_class\n");
		return PTR_ERR(smc_svn_class);
	}
	status = alloc_chrdev_region(&smc_svn_dev, 0, SMC_SVN_MAX_DEVS, SMC_SVN_CLASS);
	smc_svn_major = MAJOR(smc_svn_dev);
	if (status < 0) {
		dev_err(smc_svn_device, "%s: can't alloc chrdev region\n", __func__);
		class_destroy(smc_svn_class);
		return status;
	}
	cdev_init(&smc_svn_cdev, &smc_svn_fops);
	status = cdev_add(&smc_svn_cdev, smc_svn_dev, 1);
	if (status < 0) {
		dev_err(smc_svn_device, "can't register major %d\n", smc_svn_major);
		return status;
	}
	smc_svn_device = device_create(smc_svn_class, NULL, MKDEV(smc_svn_major, 0),
                                     NULL, SMC_SVN_DEVNAME);
	if (IS_ERR(smc_svn_device)) {
		dev_err(smc_svn_device, "%s: can't register class device\n", __func__);
		smc_svn_device = NULL;
		return PTR_ERR(smc_svn_device);
	}

	smc_svn_device->dma_mask = &smc_svn_device->coherent_dma_mask;
	dma_set_mask(smc_svn_device, DMA_BIT_MASK(64));

	return status;
}

void smc_svn_cleanup(void)
{
	if (smc_svn_class)
		class_destroy(smc_svn_class);

	if (smc_svn_major)
		unregister_chrdev_region(MKDEV(smc_svn_major, 0), SMC_SVN_MAX_DEVS);
}

module_init(smc_svn_init);
module_exit(smc_svn_cleanup);
MODULE_LICENSE("GPL");
MODULE_AUTHOR("Broadcom Corporation");
MODULE_DESCRIPTION("SMC SVN driver");
