 /****************************************************************************
 *
 * Copyright (c) 2015 Broadcom Corporation
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Piotr Romanus <promanus@broadcom.com>
 ****************************************************************************/

#include <linux/workqueue.h>
#include <linux/spinlock.h>

#include "ethsw.h"
#include "ethsw_priv.h"
#include "ethsw_db.h"

/* define log module */
#define LOG_MODULE "ethsw_arl_cache"

#define ETHSW_MAX_OUTSTANDING_REQUESTS 32
#define ETHSW_MAX_ARL_CACHE 8

struct ethsw_mac_key {
	u8 mac[6];
	u16 vid;
	u16 portmask;
};

/* ethsw arl cache control block */
struct ethsw_arl_cache {
	unsigned int instanceid;

	/* ethsw device  */
	struct ethsw_device *swdev;

	/* work queue  */
	struct workqueue_struct *workqueue;
	struct work_struct arl_update_work;
	struct work_struct arl_hw_update_work;
	struct work_struct arl_delete_work;

	struct ethsw_mac_key mac_key[ETHSW_MAX_OUTSTANDING_REQUESTS];
	int next_free_key_entry;
	int next_used_key_entry;
	struct ethsw_mac_key mac_key_delete[ETHSW_MAX_OUTSTANDING_REQUESTS];
	int next_free_key_entry_delete;
	int next_used_key_entry_delete;
	struct ethsw_mac_key hw_update_mac_key[ETHSW_MAX_OUTSTANDING_REQUESTS];
	int next_free_hw_update_key_entry;
	int next_used_hw_update_key_entry;

	unsigned int arl_cache_miss;
	unsigned int arl_cache_faked;

	int aging_poll_period;
	int aging_period;
	int link_status_poll_period;

	spinlock_t outstanding_req_lock;
	spinlock_t outstanding_hw_req_lock;
	spinlock_t outstanding_req_arl_del_lock;
};

static struct ethsw_arl_cache *swarlcache[ETHSW_MAX_ARL_CACHE] = {NULL};
static unsigned int arlcacheused = 0;

void ethsw_arl_cache_table_set_aging_poll(struct ethsw_device *swdev, int seconds)
{
	if (!swdev->arlcache) {
		LOG_ERR("ARL cache not initialized\n");
		return;
	}
	LOG_DEBUG("setting mac aging period to %d seconds\n", seconds);
	swdev->arlcache->aging_poll_period = seconds * HZ;
}

void ethsw_arl_cache_table_set_aging_period(struct ethsw_device *swdev, int seconds)
{
	if (!swdev->arlcache) {
		LOG_ERR("ARL cache not initialized\n");
		return;
	}
	LOG_DEBUG("setting mac aging poll period to %d seconds\n", seconds);
	swdev->arlcache->aging_period = seconds * HZ;
}

void ethsw_set_link_poll(struct ethsw_device *swdev, int seconds)
{
	if (!swdev->arlcache) {
		LOG_ERR("ARL cache not initialized\n");
		return;
	}
	LOG_DEBUG("setting link status poll period to %d seconds\n", seconds);
	swdev->arlcache->link_status_poll_period = seconds * HZ;
}

void ethsw_link_down(int portid)
{
	FUNC_ENTER();

	ethsw_db_delete_by_port(portid);

	FUNC_LEAVE();
}

u8 ethsw_check_link_status(struct ethsw_device *swdev)
{
	int i;
	u8 port_mask = 0;

	for (i = 0; i < ETHSW_PHY_PORT_MAX; i++) {
		struct ethsw_port *port;
		if (!(swdev->phy_port_mask & (1 << i)))
			continue;

		port = &swdev->port[i];
		if (port->cb) {
			struct ethsw_port_status new_status;
			if (swdev->ethsw_port_status_get){
				swdev->ethsw_port_status_get(swdev, i, &new_status);
			} else
				continue;

			/* anything interesting changed? */
			if (new_status.link_up     == port->status.link_up &&
					new_status.full_duplex == port->status.full_duplex &&
					new_status.speed       == port->status.speed) {
				continue;
			}

			/* update port status */
			port->status = new_status;
			LOG_DEBUG("port: %d, phy: %d, link: %d, duplex: %d, "
					"speed: %d\n", i, port->phy.id,
					port->status.link_up, port->status.full_duplex,
					port->status.speed);

			/* callback with port status */
			(*port->cb)(port->cb_priv, &port->status);
			if (new_status.link_up) {
				port_mask |= 1 << i;
			} else {
				ethsw_db_delete_by_port(i + swdev->subid_port_offset);
			}
		}
	}

	return port_mask;
}

void ethsw_check_link_status_timer_cb(struct work_struct *work)
{
	struct ethsw_device *swdev = container_of(work, struct ethsw_device,
			link_status_work.work);
	struct delayed_work *dwork = &swdev->link_status_work;

	FUNC_ENTER();

	ethsw_check_link_status(swdev);
	queue_delayed_work(swdev->arlcache->workqueue, dwork, swdev->arlcache->link_status_poll_period);

	FUNC_LEAVE();
}


static void ethsw_arl_cache_table_aging_timer_cb(struct work_struct *work)
{
	struct ethsw_device *swdev = container_of(work, struct ethsw_device,
			arl_aging_work.work);
	struct delayed_work *dwork = &swdev->arl_aging_work;

	FUNC_ENTER();

	ethsw_db_delete_by_age(swdev->arlcache->aging_period);
	queue_delayed_work(swdev->arlcache->workqueue, dwork, swdev->arlcache->aging_poll_period);

	FUNC_LEAVE();
}

static void ethsw_arl_cache_table_update(struct work_struct *work)
{
	struct ethsw_arl_cache *arlcache = container_of(work, struct ethsw_arl_cache,
			arl_update_work);
	u16 port;
	int idx = arlcache->next_used_key_entry;

	LOG_DEBUG("jiffies %lu idx %d\n", jiffies, idx);
	ethsw_arl_switch_table_find(arlcache->swdev, arlcache->mac_key[idx].mac, arlcache->mac_key[idx].vid, &port);
	if (port != ETHSW_PORT_MAP_INVALID) {
		ethsw_arl_cache_table_add(arlcache->mac_key[idx].mac, arlcache->mac_key[idx].vid, port);
		arlcache->next_used_key_entry++;
		if (arlcache->next_used_key_entry >= ETHSW_MAX_OUTSTANDING_REQUESTS) {
			arlcache->next_used_key_entry = 0;
		}
	}
}

static void ethsw_arl_cache_table_delete_hw(struct work_struct *work)
{
	struct ethsw_arl_cache *arlcache = container_of(work, struct ethsw_arl_cache,
			arl_delete_work);
	int idx = arlcache->next_used_key_entry_delete;

	LOG_DEBUG("jiffies %lu idx %d\n", jiffies, idx);
	ethsw_arl_switch_table_delete(arlcache->swdev, arlcache->mac_key_delete[idx].mac, arlcache->mac_key_delete[idx].vid);
	arlcache->next_used_key_entry_delete++;
	if (arlcache->next_used_key_entry_delete >= ETHSW_MAX_OUTSTANDING_REQUESTS) {
		arlcache->next_used_key_entry_delete = 0;
	}
}

static void ethsw_arl_hw_table_update(struct work_struct *work)
{
	struct ethsw_arl_cache *arlcache = container_of(work, struct ethsw_arl_cache,
			arl_hw_update_work);
	int idx = arlcache->next_used_hw_update_key_entry;
	LOG_DEBUG("jiffies %lu idx %d\n", jiffies, idx);
	if (arlcache->hw_update_mac_key[idx].portmask) {
		ethsw_multiport_add(arlcache->hw_update_mac_key[idx].mac, 0,
				arlcache->hw_update_mac_key[idx].portmask);
	} else {
		ethsw_multiport_delete(arlcache->hw_update_mac_key[idx].mac, 0);
	}

	arlcache->next_used_hw_update_key_entry++;
	if (arlcache->next_used_hw_update_key_entry >= ETHSW_MAX_OUTSTANDING_REQUESTS) {
		arlcache->next_used_hw_update_key_entry = 0;
	}
}

int ethsw_arl_cache_table_init(struct ethsw_device *swdev)
{
	int ret = 0;
	int port_mask;
	struct ethsw_arl_cache *arlcache;
	int i = 0;

	FUNC_ENTER();
	if (arlcacheused > ETHSW_MAX_ARL_CACHE) {
		LOG_ERR("No free ARL cache instance\n");
		goto ERROR;
	}

	LOG_DEBUG("jiffies %lu HZ %d\n", jiffies, HZ);
	ret = ethsw_db_init();
	if (ret) {
		LOG_ERR("ARL cache db initialization failed\n");
		goto ERROR;
	}

	/* alloc ethsw arl cache control block */
	arlcache = kzalloc(sizeof(struct ethsw_arl_cache), GFP_KERNEL);
	if (!arlcache) {
		LOG_CRIT("Ethsw arl cache control block alloc failed\n");
		ret = -ENOMEM;
		goto ERROR;
	}

	arlcache->workqueue = alloc_ordered_workqueue("ethsw_arl_cache", 0);
	if (!arlcache->workqueue) {
		LOG_ERR("ARL cache workqueue initialization failed\n");
		ret = -ENOMEM;
		kfree(arlcache);
		goto ERROR;
	}

	INIT_WORK(&arlcache->arl_update_work, ethsw_arl_cache_table_update);
	INIT_WORK(&arlcache->arl_hw_update_work, ethsw_arl_hw_table_update);
	INIT_WORK(&arlcache->arl_delete_work, ethsw_arl_cache_table_delete_hw);

	arlcache->aging_poll_period = HZ;
	arlcache->aging_period = 300 * HZ;
	arlcache->link_status_poll_period = HZ / 2;

	INIT_DELAYED_WORK(&swdev->arl_aging_work, ethsw_arl_cache_table_aging_timer_cb);
	queue_delayed_work(arlcache->workqueue, &swdev->arl_aging_work, arlcache->aging_poll_period);
	INIT_DELAYED_WORK(&swdev->link_status_work, ethsw_check_link_status_timer_cb);
	queue_delayed_work(arlcache->workqueue, &swdev->link_status_work, arlcache->link_status_poll_period);

	arlcache->swdev = swdev;
	while (swarlcache[i])
		i++;
	swarlcache[i] = arlcache;
	arlcacheused++;
	swdev->arlcache = arlcache;

	port_mask = ethsw_check_link_status(swdev);
	LOG_INFO("ports up mask %x\n", port_mask);

ERROR:
	FUNC_LEAVE();
	return ret;
}

void ethsw_arl_cache_table_fini(struct ethsw_device *swdev)
{
	int i = 0;
	FUNC_ENTER();
	destroy_workqueue(swdev->arlcache->workqueue);
	while (swarlcache[i] != swdev->arlcache)
		i++;
	swarlcache[i] = NULL;
	arlcacheused--;
	kfree(swdev->arlcache);
	ethsw_db_fini();
	FUNC_LEAVE();
}

int ethsw_arl_cache_table_find(u8 *mac, u16 vid, u16 *port)
{
	int ret;

	ret = __ethsw_arl_cache_table_find(mac, vid, port);
	return ret;
}


int __ethsw_arl_cache_table_find(u8 *mac, u16 vid, u16 *port)
{
	struct ethsw_db_entry *db_entry;

	FUNC_ENTER();

	LOG_DEBUG("ARL cache port lookup for %pM vid=%u\n", mac, vid);

	db_entry = ethsw_db_find(mac, vid);
	if (db_entry) {
		*port = db_entry->portid;
		LOG_DEBUG("port found %u\n", *port);
	} else {
		int i = 0;
		*port = ETHSW_PORT_MAP_INVALID;
		LOG_DEBUG("port not found; jiffies %lu\n", jiffies);
		/* Loop through arl cache instances to find the entry */
		while (swarlcache[i]) {
			spin_lock(&swarlcache[i]->outstanding_req_lock);
			memcpy(swarlcache[i]->mac_key[swarlcache[i]->next_free_key_entry].mac, mac, 6);
			swarlcache[i]->mac_key[swarlcache[i]->next_free_key_entry].vid = vid;
			swarlcache[i]->next_free_key_entry++;
			if (swarlcache[i]->next_free_key_entry >= ETHSW_MAX_OUTSTANDING_REQUESTS) {
				swarlcache[i]->next_free_key_entry = 0;
			}
			spin_unlock(&swarlcache[i]->outstanding_req_lock);
			queue_work(swarlcache[i]->workqueue, &swarlcache[i]->arl_update_work);
			i++;
		}
	}

	FUNC_LEAVE();

	return 0;
}

int ethsw_arl_cache_table_add(u8 *mac, u16 vid, u16 port)
{
	int ret;

	FUNC_ENTER();
	ret = ethsw_db_insert(port, mac, vid);
	FUNC_LEAVE();

	return ret;
}
int __ethsw_arl_cache_table_delete(u8 *mac, u16 vid, u16 port)
{
	struct ethsw_device *swdev = NULL;
	struct ethsw_arl_cache *arlcache = NULL;
	swdev = subid_to_swdev(port);
	VALIDATE_SWDEV(swdev);
	arlcache = swdev->arlcache;
	if (!arlcache) {
		LOG_ERR("ARL cache not initialized\n");
		return -1;
	}

	spin_lock(&arlcache->outstanding_req_arl_del_lock);
	memcpy(arlcache->mac_key_delete[arlcache->next_free_key_entry_delete].mac, mac, 6);
	arlcache->mac_key_delete[arlcache->next_free_key_entry_delete].vid = vid;
	arlcache->next_free_key_entry_delete++;
	if (arlcache->next_free_key_entry_delete >= ETHSW_MAX_OUTSTANDING_REQUESTS) {
		arlcache->next_free_key_entry_delete = 0;
	}

	spin_unlock(&arlcache->outstanding_req_arl_del_lock);
	queue_work(arlcache->workqueue, &arlcache->arl_delete_work);

	return 0;
}

int ethsw_arl_cache_table_delete(u8 *mac, u16 vid)
{
	int ret;

	FUNC_ENTER();
	ret = ethsw_db_delete(mac, vid);
	FUNC_LEAVE();

	return ret;
}

int ethsw_arl_hw_table_add(u8 *mac, u16 vid, u16 portmask)
{
	struct ethsw_arl_cache *arlcache = swarlcache[0];
	if (!arlcache) {
		LOG_ERR("ARL cache not initialized\n");
		return -1;
	}

	spin_lock(&arlcache->outstanding_hw_req_lock);
	memcpy(arlcache->hw_update_mac_key[arlcache->next_free_hw_update_key_entry].mac, mac, 6);
	arlcache->hw_update_mac_key[arlcache->next_free_hw_update_key_entry].vid = vid;
	arlcache->hw_update_mac_key[arlcache->next_free_hw_update_key_entry].portmask = portmask;
	arlcache->next_free_hw_update_key_entry++;
	if (arlcache->next_free_hw_update_key_entry >= ETHSW_MAX_OUTSTANDING_REQUESTS) {
		arlcache->next_free_hw_update_key_entry = 0;
	}

	spin_unlock(&arlcache->outstanding_hw_req_lock);
	queue_work(arlcache->workqueue, &arlcache->arl_hw_update_work);

	return 0;
}
EXPORT_SYMBOL(ethsw_arl_hw_table_add);

void ethsw_arl_cache_get_counter(struct ethsw_device *swdev, unsigned int *arl_cache_miss, unsigned int *arl_cache_faked)
{
	if (swdev->arlcache) {
		*arl_cache_miss = swdev->arlcache->arl_cache_miss;
		*arl_cache_faked = swdev->arlcache->arl_cache_faked;
	}
}
