/****************************************************************************
 *
 * Broadcom Proprietary and Confidential.
 * (c) 2019 Broadcom. All rights reserved.
 * The term "Broadcom" refers to Broadcom Limited and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * RPC RTC driver
 *
 * Author: Venky Selvaraj <venky.selvaraj@broadcom.com>
 *****************************************************************************/

#ifdef CONFIG_BCM_KF_CM

#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/rtc.h>
#include <linux/bcm_media_gw/itc_rpc/itc_rpc.h>

#define MODULE_NAME             "bcmrpc-rtc"
#define COMPATIBLE_RTC          "brcm,bcmrpc-rtc"
#define RPC_CHANNEL_NODE        "rpc-channel"
#define DEVICE_NAME_PROPERTY    "dev-name"

#define RPC_RTC_TIMEOUT	1	/* secs */
#define RTC_ID_INVALID	-1

#define RPC_RTC_ID_MASK 0x0F
#define RPC_RTC_READ_SHIFT 31
#define RPC_RTC_WRITE_SHIFT 30
#define RPC_RTC_RET_MASK 0xFF
#define RPC_RTC_RET_SHIFT 24
#define RPC_RTC_AP_SHIFT 5
#define RPC_RTC_AE_SHIFT 4
#define RPC_RTC_ALARM_NOTIFY_MASK 0xFF
#define LSHIFT(x, y) ((x) << (y))
#define RPC_RTC_READ_MASK(x) LSHIFT(x, RPC_RTC_READ_SHIFT)
#define RPC_RTC_WRITE_MASK(x) LSHIFT(x, RPC_RTC_WRITE_SHIFT)
#define RPC_RTC_ATTR_MASK(x) (RPC_RTC_READ_MASK(x) | RPC_RTC_WRITE_MASK(x))
#define RPC_RTC_AP_MASK(x) LSHIFT(x, RPC_RTC_AP_SHIFT)
#define RPC_RTC_AE_MASK(x) LSHIFT(x, RPC_RTC_AE_SHIFT)
#define RPC_RTC_ALARM_MASK(x) (RPC_RTC_AP_MASK(x) | RPC_RTC_AE_MASK(x))

#define ANSI_RED	"\e[31;1m"
#define ANSI_YELLOW	"\e[33;1m"
#define ANSI_RESET	"\e[0m"
#define RED(str)	ANSI_RED str ANSI_RESET
#define YLW(str)	ANSI_YELLOW str ANSI_RESET

struct rpc_info {
	char *tunnel_name;
	int tunnel;
	int rtc_id;
	bool read_attr;
	bool write_attr;
	unsigned long time;
	unsigned char alarm_enable;
	unsigned char alarm_pending;
};

struct rtc_plat_data {
	struct list_head list;
	struct platform_device *pdev;
	char *dev_name;
	struct rpc_info info;
	struct rtc_device *rtc;
	int exiting;
	struct mutex mutex;
};

/**
 * enum rpc_rtc_func_idx - possible rpc functions
 *		supported for Real Time Clock.
 * @RPC_RTC_FUNC_ID: Function to request for Tunnel ID.
 * @RPC_RTC_FUNC_NAME: Function to request for RTC name.
 * @RPC_RTC_FUNC_ALLOC: Function to request for RTC allocation
 *		for access.
 * @RPC_RTC_FUNC_FREE: Function to request to free RTC
 *		allocation.
 * @RPC_RTC_FUNC_GET_TIME: Function to get RTC time.
 * @RPC_RTC_FUNC_SET_TIME: Function to setRTC time.
 * @RPC_RTC_FUNC_GET_ALARM: Function to get RTC alarm
 *		information.
 * @RPC_RTC_FUNC_GET_ALARM: Function to set RTC alarm
 * @RPC_RTC_FUNC_REQ_NOTIFICATION: Function to request for
 *		notification
 * @RPC_RTC_FUNC_NOTIFY: Function to notify the process that
 *		reuested notification
 */
enum rpc_rtc_func_idx {
	RPC_RTC_FUNC_ID = 0,
	RPC_RTC_FUNC_NAME,
	RPC_RTC_FUNC_ALLOC,
	RPC_RTC_FUNC_FREE,
	RPC_RTC_FUNC_GET_TIME,
	RPC_RTC_FUNC_SET_TIME,
	RPC_RTC_FUNC_GET_ALARM,
	RPC_RTC_FUNC_SET_ALARM,
	RPC_RTC_FUNC_REQ_NOTIFICATION,
	RPC_RTC_FUNC_NOTIFY,
	RPC_RTC_FUNC_MAX
};

static int bcmrpc_rtc_notify(int tunnel, rpc_msg *msg);
static void bcmrpc_alarm_notify(struct rtc_plat_data *pdata);
static rpc_function rpc_rtc_services_tbl[] = {
	{ NULL,			0 },
	{ NULL,			0 },
	{ NULL,			0 },
	{ NULL,			0 },
	{ NULL,			0 },
	{ NULL,			0 },
	{ NULL,			0 },
	{ NULL,			0 },
	{ NULL,			0 },
	{ bcmrpc_rtc_notify,	0 }
};
static struct list_head rtcdev_list;
static spinlock_t rtcdev_list_lock;

/******************************************************************************
 *
 *	RPC Interface Functions
 *
 ******************************************************************************/
/**
 * rpc_pinconf_alloc request encoding
 * W1: | 31..00  |
 *     |   RSVD  |
 *
 * W2: | 31..24  | 23..16  | 15..08  | 07..00  |
 *     | NAME[3] | NAME[2] | NAME[1] | NAME[0] |
 *
 * W3: | 31..24  | 23..16  | 15..08  | 07..00  |
 *     | NAME[7] | NAME[6] | NAME[5] | NAME[4] |
 *
 * RSVD:	Reserved
 * Name:	RTC Device Name passed to SMC #
 */
static inline void bcmrpc_rtc_encode_name(rpc_msg *msg,
					char *dev_name)
{
	strncpy((char *)&(msg->data[1]), dev_name,
			(2*sizeof(msg->data[0])));
}

/**
 * bcmrpc_rtc_alloc request encoding
 * W1: |   31    |    30   | 29..04  | 03..00  |
 *     |   R     |    W    |  RSVD   | RTC ID  |
 *
 * RTC ID:	Returned from rct id request
 */
static inline void bcmrpc_rtc_encode_rtc_id(rpc_msg *msg,
				int rtc_id)
{
	msg->data[0] = ((msg->data[0] & ~RPC_RTC_ID_MASK) |
					(rtc_id & RPC_RTC_ID_MASK));
}

static inline void bcmrpc_rtc_encode_attribute(rpc_msg *msg,
				  uint32_t r_attr, uint32_t w_attr)
{
	u32 attr = (LSHIFT(r_attr, RPC_RTC_READ_SHIFT) |
				LSHIFT(w_attr, RPC_RTC_WRITE_SHIFT));

	msg->data[0] = ((msg->data[0] & ~(RPC_RTC_ATTR_MASK(0x01))) |
					(attr & RPC_RTC_ATTR_MASK(0x01)));
}

/**
 * All reply decoding ret code
 * W1: | 31..24 | 23..04 | 03..00 |
 *     |   RC   | RSVD   | RTC-ID|
 *
 * W2: | 31..0 |
 *     |  RSVD |
 *
 * W3: | 31..0 |
 *     |  RSVD |
 *
 * RSVD:	Reserved
 * RTC-ID:	RTC ID provided by SMC #
 */
static inline u8 bcmrpc_rtc_decode_retcode(rpc_msg *msg)
{
	return ((msg->data[0] >> RPC_RTC_RET_SHIFT) &
			RPC_RTC_RET_MASK);
}

static inline u8 bcmrpc_rtc_decode_rtc_id(rpc_msg *msg)
{
	return (msg->data[0] & RPC_RTC_ID_MASK);
}

/**
 * bcmrpc_rtc_get_time reply decoding
 * W1: | 31..24 | 23..04 |
 *     03..00 | |   RC   | RSVD   | RTC-ID |
 *
 * W2: | 31..0 |
 *     |  Time |
 *
 * W3: | 31..0 |
 *     |  RSVD |
 *
 * RSVD:	Reserved
 * RTC-ID:	RTC ID provided by SMC #
 * Time: RTC Time
 */
static inline u32 bcmrpc_rtc_decode_time(rpc_msg *msg)
{
	return msg->data[1];
}

static inline void bcmrpc_rtc_encode_time(rpc_msg *msg, u32 itime)
{
	msg->data[1] = (u32)itime;
}

/**
 * bcmrpc_rtc_get_alarm reply decoding
 * W1: | 31..24 | 23..06 |  05  |  04  |03..00  |
 *     |   RC   | RSVD   |  AP  |  AE  | RTC-ID |
 *
 * W2: | 31..0 |
 *     |  Time |
 *
 * W3: | 31..0 |
 *     |  RSVD |
 *
 * RSVD:	Reserved
 * RTC-ID:	RTC ID provided by SMC #
 * AP: Alarm pending flag
 * AE: Alarm enable flag
 * Time: RTC Time
 */
static inline void bcmrpc_rtc_decode_alarm(rpc_msg *msg,
				u32 *itime, u32 *ap, u32 *ae)
{
	*ap = (msg->data[0] & RPC_RTC_AP_MASK(0x01)) >> RPC_RTC_AP_SHIFT;
	*ae = (msg->data[0] & RPC_RTC_AE_MASK(0x01)) >> RPC_RTC_AE_SHIFT;
	*itime = (u32)msg->data[1];
}

static inline void bcmrpc_rtc_encode_alarm(rpc_msg *msg,
					u32 itime, u32 ae)
{
	u32 alarm = LSHIFT(ae, RPC_RTC_AE_SHIFT);

	msg->data[0] = ((msg->data[0] & ~(RPC_RTC_AE_MASK(0x01))) |
					(alarm & RPC_RTC_AE_MASK(0x01)));
	msg->data[1] = (u32)itime;
}

static inline void bcmrpc_rtc_encode_alarm_request(
					rpc_msg *msg, u32 ae)
{
	u32 alarm = LSHIFT(ae, RPC_RTC_AE_SHIFT);

	msg->data[0] = ((msg->data[0] & ~(RPC_RTC_AE_MASK(0x01))) |
					(alarm & RPC_RTC_AE_MASK(0x01)));
}

static inline int bcmrpc_rtc_decode_alarm_notify(
					rpc_msg *msg)
{
	return ((msg->data[0] >> RPC_RTC_AE_SHIFT) &
			RPC_RTC_ALARM_NOTIFY_MASK);
}

static inline int bcmrpc_rtc_request(rpc_msg *msg, int tunnel)
{
	int ret = 0;

	ret = rpc_send_request_timeout(tunnel, msg,
			RPC_RTC_TIMEOUT);
	if (ret) {
		pr_err(RED("%s : rpc_send_request failure (%d)\n"),
			   MODULE_NAME, ret);
		rpc_dump_msg(msg);
		goto done;
	}
	ret = bcmrpc_rtc_decode_retcode(msg);
	if (ret) {
		pr_err(RED("%s : bcmrpc_rtc_decode_retcode failure (%d)\n"),
			   MODULE_NAME, ret);
		rpc_dump_msg(msg);
	}
done:
	return ret;
}

static int bcmrpc_rtc_id(struct rtc_plat_data *pdata)
{
	int ret = 0, len = 0;
	rpc_msg msg;
	int tunnel = pdata->info.tunnel;
	int rtc_id = 0;

	len = strlen(pdata->dev_name);
	if (len > (2*sizeof(msg.data[0]))) {
		pr_err(RED("%s: RTC Device Name length > Max length : %lu .\n"),
			   MODULE_NAME, (2*sizeof(msg.data[0])));
		ret = -EINVAL;
		goto out;
	}
	rpc_msg_init(&msg, RPC_SERVICE_RTC, RPC_RTC_FUNC_ID, 0, 0, 0, 0);
	bcmrpc_rtc_encode_name(&msg, pdata->dev_name);
	ret = bcmrpc_rtc_request(&msg, tunnel);
	if (ret) {
		pr_err(RED("%s: RPC request for RTC-ID failed.\n"),
			   MODULE_NAME);
		goto out;
	}
	rtc_id = bcmrpc_rtc_decode_rtc_id(&msg);
	pdata->info.rtc_id = rtc_id;
out:
	return ret;
}

static int bcmrpc_rtc_alloc(struct rtc_plat_data *pdata)
{
	int ret = 0;
	rpc_msg msg;
	int tunnel = pdata->info.tunnel;
	int rtc_id = pdata->info.rtc_id;
	u32 r_attr = (u32)(pdata->info.read_attr & 0x01);
	u32 w_attr = (u32)(pdata->info.write_attr & 0x01);

	rpc_msg_init(&msg, RPC_SERVICE_RTC, RPC_RTC_FUNC_ALLOC, 0, 0, 0, 0);
	bcmrpc_rtc_encode_attribute(&msg, r_attr, w_attr);
	bcmrpc_rtc_encode_rtc_id(&msg, rtc_id);
	ret = bcmrpc_rtc_request(&msg, tunnel);
	if (ret) {
		pr_err(RED("%s: RPC request for alloc failed.\n"),
			   MODULE_NAME);
		goto out;
	}

out:
	return ret;
}

static int bcmrpc_rtc_free(struct rtc_plat_data *pdata)
{
	int ret = 0;
	rpc_msg msg;
	int tunnel = pdata->info.tunnel;
	int rtc_id = pdata->info.rtc_id;

	rpc_msg_init(&msg, RPC_SERVICE_RTC, RPC_RTC_FUNC_FREE, 0, 0, 0, 0);
	bcmrpc_rtc_encode_rtc_id(&msg, rtc_id);
	ret = bcmrpc_rtc_request(&msg, tunnel);
	if (ret) {
		pr_err(RED("%s: RPC request for free failed.\n"),
			   MODULE_NAME);
		goto out;
	}

out:
	return ret;
}

static int bcmrpc_rtc_get_time(struct rtc_plat_data *pdata)
{
	int ret = 0;
	rpc_msg msg;
	int tunnel = pdata->info.tunnel;
	int rtc_id = pdata->info.rtc_id;
	u32 itime;

	rpc_msg_init(&msg, RPC_SERVICE_RTC, RPC_RTC_FUNC_GET_TIME, 0, 0, 0, 0);
	bcmrpc_rtc_encode_rtc_id(&msg, rtc_id);
	ret = bcmrpc_rtc_request(&msg, tunnel);
	if (ret) {
		pr_err(RED("%s: RPC request for get time failed.\n"),
			   MODULE_NAME);
		goto out;
	}
	itime = bcmrpc_rtc_decode_time(&msg);
	pdata->info.time = itime;
out:
	return ret;
}

static int bcmrpc_rtc_set_time(struct rtc_plat_data *pdata)
{
	int ret = 0;
	rpc_msg msg;
	int tunnel = pdata->info.tunnel;
	int rtc_id = pdata->info.rtc_id;
	u32 itime = pdata->info.time;

	rpc_msg_init(&msg, RPC_SERVICE_RTC, RPC_RTC_FUNC_SET_TIME, 0, 0, 0, 0);
	bcmrpc_rtc_encode_rtc_id(&msg, rtc_id);
	bcmrpc_rtc_encode_time(&msg, itime);
	ret = bcmrpc_rtc_request(&msg, tunnel);
	if (ret) {
		pr_err(RED("%s: RPC request for set time failed.\n"),
			   MODULE_NAME);
		goto out;
	}

out:
	return ret;
}

static int bcmrpc_rtc_get_alarm(struct rtc_plat_data *pdata)
{
	int ret = 0;
	rpc_msg msg;
	int tunnel = pdata->info.tunnel;
	int rtc_id = pdata->info.rtc_id;
	u32 itime, ap, ae;

	rpc_msg_init(&msg, RPC_SERVICE_RTC, RPC_RTC_FUNC_GET_ALARM, 0, 0, 0, 0);
	bcmrpc_rtc_encode_rtc_id(&msg, rtc_id);
	ret = bcmrpc_rtc_request(&msg, tunnel);
	if (ret) {
		pr_err(RED("%s: RPC request for get alarm failed.\n"),
			   MODULE_NAME);
		goto out;
	}
	bcmrpc_rtc_decode_alarm(&msg, &itime, &ap, &ae);
	pdata->info.alarm_pending = (unsigned char)ap;
	pdata->info.alarm_enable = (unsigned char)ae;
	pdata->info.time = itime;

out:
	return ret;
}

static int bcmrpc_rtc_set_alarm(struct rtc_plat_data *pdata)
{
	int ret = 0;
	rpc_msg msg;
	int tunnel = pdata->info.tunnel;
	int rtc_id = pdata->info.rtc_id;
	u32 itime = pdata->info.time;
	u32 ae = pdata->info.alarm_enable;

	rpc_msg_init(&msg, RPC_SERVICE_RTC, RPC_RTC_FUNC_SET_ALARM, 0, 0, 0, 0);
	bcmrpc_rtc_encode_rtc_id(&msg, rtc_id);
	bcmrpc_rtc_encode_alarm(&msg, itime, ae);
	ret = bcmrpc_rtc_request(&msg, tunnel);
	if (ret) {
		pr_err(RED("%s: RPC request for set alarm failed.\n"),
			   MODULE_NAME);
		goto out;
	}

out:
	return ret;
}

static int bcmrpc_rtc_request_notification(struct rtc_plat_data *pdata)
{
	int ret = 0;
	rpc_msg msg;
	int tunnel = pdata->info.tunnel;
	int rtc_id = pdata->info.rtc_id;
	u32 ae = pdata->info.alarm_enable;

	rpc_msg_init(&msg, RPC_SERVICE_RTC,
		RPC_RTC_FUNC_REQ_NOTIFICATION, 0, 0, 0, 0);
	bcmrpc_rtc_encode_rtc_id(&msg, rtc_id);
	bcmrpc_rtc_encode_alarm_request(&msg, ae);
	ret = bcmrpc_rtc_request(&msg, tunnel);
	if (ret) {
		pr_err(RED("%s: RPC request for notification failed.\n"),
			   MODULE_NAME);
		goto out;
	}

out:
	return ret;
}

static int bcmrpc_rtc_notify(int tunnel, rpc_msg *msg)
{
	int ret;
	struct list_head *pos;
	struct rtc_plat_data *pdata;
	int rtc_id = 0;

	ret = bcmrpc_rtc_decode_alarm_notify(msg);
	if (ret)
		goto out;

	rtc_id = bcmrpc_rtc_decode_rtc_id(msg);

	spin_lock(&rtcdev_list_lock);
	list_for_each(pos, &rtcdev_list) {
		pdata = list_entry(pos, struct rtc_plat_data, list);
		if (pdata && pdata->info.rtc_id == rtc_id) {
			bcmrpc_alarm_notify(pdata);
			break;
		}
	}
	spin_unlock(&rtcdev_list_lock);
out:
	return ret;
}

/******************************************************************************
 *
 *	Driver API Functions
 *
 ******************************************************************************/
static int bcmrpc_read_time(struct device *dev, struct rtc_time *time)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct rtc_plat_data *pdata = platform_get_drvdata(pdev);
	int ret;

	ret = bcmrpc_rtc_get_time(pdata);
	if (ret) {
		pr_err(RED("%s: RTC bcmrpc_rtc_get_time function failed.\n"),
			   MODULE_NAME);
		goto out;
	}
	rtc_time64_to_tm(pdata->info.time, time);

out:
	return ret;
}

static int bcmrpc_set_time(struct device *dev, struct rtc_time *time)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct rtc_plat_data *pdata = platform_get_drvdata(pdev);
	unsigned long itime;

	if (!pdata->info.write_attr) {
		pr_info(YLW("%s: RTC write access denied.\n"),
			MODULE_NAME);
		return -EINVAL;
	}

	itime = rtc_tm_to_time64(time);
	pdata->info.time = (u32)itime;

	return bcmrpc_rtc_set_time(pdata);
}

static int bcmrpc_read_alarm(struct device *dev, struct rtc_wkalrm *alarm)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct rtc_plat_data *pdata = platform_get_drvdata(pdev);
	int ret = 0;

	mutex_lock(&pdata->mutex);

	ret = bcmrpc_rtc_get_alarm(pdata);
	if (ret) {
		pr_err(RED("%s: RTC bcmrpc_rtc_get_alarm function failed.\n"),
			   MODULE_NAME);
		goto out;
	}

	rtc_time64_to_tm(pdata->info.time, &alarm->time);
	alarm->enabled = pdata->info.alarm_enable;
	alarm->pending = pdata->info.alarm_pending;

out:
	mutex_unlock(&pdata->mutex);
	return ret;
}

static int bcmrpc_set_alarm(struct device *dev, struct rtc_wkalrm *alarm)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct rtc_plat_data *pdata = platform_get_drvdata(pdev);
	unsigned long new_alarm, itime;
	int ret = 0;

	if (!pdata->info.write_attr) {
		pr_info(YLW("%s: RTC write access denied.\n"),
			MODULE_NAME);
		return -EINVAL;
	}

	mutex_lock(&pdata->mutex);

	ret = bcmrpc_rtc_get_time(pdata);
	if (ret) {
		pr_err(RED("%s: RTC bcmrpc_rtc_get_time function failed.\n"),
			   MODULE_NAME);
		goto out;
	}

	itime = pdata->info.time;
	new_alarm = rtc_tm_to_time64(&alarm->time);


	/* This can happen due to races, in addition to dates that are
	 * truly in the past.  To avoid requiring the caller to check for
	 * races, dates in the past are assumed to be in the recent past
	 * (i.e. not something that we'd rather the caller know about via
	 * an error), and the alarm is set to go off as soon as possible.
	 */
	if (time_before_eq(new_alarm, itime))
		new_alarm = itime + 1;

	if (alarm->enabled)
		pdata->info.alarm_enable = 1;
	else
		pdata->info.alarm_enable = 0;
	pdata->info.time = new_alarm;

	/* Set Notification for Alarm */
	ret = bcmrpc_rtc_request_notification(pdata);
	if (ret) {
		pr_err(RED("%s: RPC Request Notification failed.\n"),
			   MODULE_NAME);
		goto out;
	}

	/* Set Alarm */
	ret = bcmrpc_rtc_set_alarm(pdata);
	if (ret) {
		pr_err(RED("%s: RTC bcmrpc_rtc_set_alarm function failed.\n"),
			   MODULE_NAME);
		goto out;
	}

out:
	mutex_unlock(&pdata->mutex);
	return ret;
}

static void bcmrpc_alarm_notify(struct rtc_plat_data *pdata)
{
	if (pdata->exiting)
		return;

	mutex_lock(&pdata->mutex);

	rtc_update_irq(pdata->rtc, 1, RTC_AF | RTC_IRQF);

	mutex_unlock(&pdata->mutex);
}

static int bcmrpc_alarm_notification_enable(struct device *dev, unsigned int enable)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct rtc_plat_data *pdata = platform_get_drvdata(pdev);
	int ret;

	if (!pdata->info.write_attr) {
		pr_info(YLW("%s: RTC write access denied.\n"),
			MODULE_NAME);
		return -EINVAL;
	}

	mutex_lock(&pdata->mutex);

	if (enable)
		pdata->info.alarm_enable = 1;
	else
		pdata->info.alarm_enable = 0;

	ret = bcmrpc_rtc_request_notification(pdata);
	if (ret) {
		pr_err(RED("%s: RPC Request Notification failed.\n"),
			   MODULE_NAME);
		goto out;
	}

out:
	mutex_unlock(&pdata->mutex);
	return ret;
}

static const struct rtc_class_ops bcmrpc_rtc_ops = {
	.read_time = bcmrpc_read_time,
	.set_time = bcmrpc_set_time,
	.read_alarm = bcmrpc_read_alarm,
	.set_alarm = bcmrpc_set_alarm,
	.alarm_irq_enable = bcmrpc_alarm_notification_enable,
};

/******************************************************************************
 *
 *	Driver Interface
 *
 ******************************************************************************/

static int bcmrpc_rtc_remove(struct platform_device *pdev)
{
	struct rtc_plat_data *pdata = platform_get_drvdata(pdev);
	struct list_head *pos;
	struct rtc_plat_data *pdata_tmp;
	struct device *dev = &pdev->dev;

	dev_dbg(dev, "-->\n");

	BUG_ON(!pdata);

	mutex_lock(&pdata->mutex);
	pdata->exiting = 1;
	mutex_unlock(&pdata->mutex);
	if (pdata->info.tunnel >= 0 &&
			pdata->info.rtc_id >= 0)
		bcmrpc_rtc_free(pdata);
	spin_lock(&rtcdev_list_lock);
	list_for_each(pos, &rtcdev_list) {
		pdata_tmp = list_entry(pos, struct rtc_plat_data, list);
		if (pdata_tmp == pdata) {
			list_del(pos);
			break;
		}
	}
	spin_unlock(&rtcdev_list_lock);
	dev_dbg(dev, "<--\n");

	return 0;
}

static int bcmrpc_rtc_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct device_node *np = dev->of_node;
	struct device_node *rnp;
	struct rtc_plat_data *pdata;
	char *str;
	int ret = 0;

	dev_dbg(dev, "-->\n");
	spin_lock_init(&rtcdev_list_lock);
	INIT_LIST_HEAD(&rtcdev_list);

	pdata = devm_kzalloc(dev, sizeof(*pdata), GFP_KERNEL);
	if (!pdata) {
		dev_err(dev, RED("Unable to allocate pdata.\n"));
		ret = -ENOMEM;
		goto done;
	}
	pdata->pdev = pdev;
	pdata->info.tunnel = RPC_INVALID_TUNNEL;
	pdata->info.rtc_id = RTC_ID_INVALID;
	mutex_init(&pdata->mutex);
	platform_set_drvdata(pdev, pdata);

	rnp = of_parse_phandle(np, RPC_CHANNEL_NODE, 0);
	if (!rnp) {
		dev_err(dev, RED("Unable to retrieve %s phandle.\n"),
			RPC_CHANNEL_NODE);
		ret = -EINVAL;
		goto out;
	}
	ret = of_property_read_string(rnp, DEVICE_NAME_PROPERTY,
				(const char **)&str);
	of_node_put(rnp);
	if (ret) {
		dev_err(dev, RED("RPC node %s property missing.\n"),
			DEVICE_NAME_PROPERTY);
		goto out;
	}
	pdata->info.tunnel_name = str;
	pdata->info.tunnel = rpc_get_fifo_tunnel_id(str);
	if (pdata->info.tunnel < 0) {
		dev_err(dev, RED("Unable to obtain RPC tunnel ID.\n"));
		ret = -EIO;
		goto out;
	}

	ret = of_property_read_string(np, DEVICE_NAME_PROPERTY,
				(const char **)&str);
	of_node_put(np);
	if (ret) {
		dev_err(dev, RED("RTC node %s property missing.\n"),
			DEVICE_NAME_PROPERTY);
		goto out;
	}
	pdata->dev_name = str;

	ret = bcmrpc_rtc_id(pdata);
	if (ret) {
		dev_err(dev, RED("Unable to get RTC-ID"
						 " through RPC\n"));
		goto out;
	}

	/* Set the read and write attribute before allocating the RTC */
	pdata->info.read_attr = true;
	pdata->info.write_attr = true;
	ret = bcmrpc_rtc_alloc(pdata);
	if (ret) {
		pdata->info.write_attr = false;
		ret = bcmrpc_rtc_alloc(pdata);
		if (ret) {
			dev_err(dev, RED("Unable to allocate RTC"
							 " through RPC\n"));
			goto out;
		}
		dev_info(dev, YLW("RTC R/W allocation requested but"
						  " Read only RTC through RPC allocated\n"));
	}

	/* SMC handles the alarm notification just register for
	 * wakealarm interface
	 */
	device_set_wakeup_capable(dev, 1);

	pdata->rtc = devm_rtc_device_register(dev, pdev->name,
				  &bcmrpc_rtc_ops, THIS_MODULE);
	if (IS_ERR(pdata->rtc)) {
		dev_err(dev, RED("unable to register the class device\n"));
		ret = PTR_ERR(pdata->rtc);
		goto out;
	}

	/* Add device to the list */
	spin_lock(&rtcdev_list_lock);
	list_add_tail(&pdata->list, &rtcdev_list);
	spin_unlock(&rtcdev_list_lock);

	goto done;

out:
	bcmrpc_rtc_remove(pdev);

done:
	pr_debug("<--\n");
	return ret;
}

static const struct of_device_id bcmrpc_rtc_of_match[] = {
	{ .compatible = COMPATIBLE_RTC },
	{},
};

static struct platform_driver bcmrpc_rtc_driver = {
	.probe		= bcmrpc_rtc_probe,
	.remove		= bcmrpc_rtc_remove,
	.driver		= {
		.name	= MODULE_NAME,
		.of_match_table	= of_match_ptr(bcmrpc_rtc_of_match),
	},
};

static int __init bcmrpc_rtc_init(void)
{
	int rc = 0;

	rc = rpc_register_functions(RPC_SERVICE_RTC,
			rpc_rtc_services_tbl,
			RPC_RTC_FUNC_MAX);
	if (rc) {
		pr_err(RED("%s: Failed to register RTC RPC function(s).\n"),
			   MODULE_NAME);
		goto out;
	}

	rc = platform_driver_register(&bcmrpc_rtc_driver);
	if (rc) {
		pr_err(RED("%s: Failed to register platform driver.\n"),
			   MODULE_NAME);
		goto err_unreg_rtc_rpc;
	}
	return 0;

err_unreg_rtc_rpc:
	rpc_unregister_functions(RPC_SERVICE_RTC);
out:
	return rc;
}
module_init(bcmrpc_rtc_init);

static void __exit bcmrpc_rtc_exit(void)
{
	platform_driver_unregister(&bcmrpc_rtc_driver);
	rpc_unregister_functions(RPC_SERVICE_RTC);
}
module_exit(bcmrpc_rtc_exit);

MODULE_AUTHOR("Venky Selvaraj <venky.selvaraj@broadcom.com>");
MODULE_DESCRIPTION("Broadcom RPC RTC Driver");
MODULE_LICENSE("GPL v2");


#endif /* CONFIG_BCM_KF_CM */
