 /****************************************************************************
 *
 * Broadcom Proprietary and Confidential.
 * (c) 2015-2016 Broadcom. All rights reserved.
 * The term Broadcom refers to Broadcom Limited and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************/
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/string.h>
#include <linux/uaccess.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/netfilter.h>
#include <linux/netfilter_bridge.h>
#include <linux/netfilter_ipv4.h>
#include <linux/ip.h>
#include <linux/udp.h>
#include "sfap.h"
#include "bcmnethooks.h"

#define VERSION     "0.2"
#define VER_STR     "v" VERSION

#ifdef CONFIG_BCM_SFAP_NETHOOK
#define  SFAP_USE_BCM_NETHOOKS		1
#else
#define  SFAP_USE_BCM_NETHOOKS		0
#endif
#ifdef CONFIG_BCM_SFAP_NETHOOK_FPM
#define  SFAP_USE_BCM_NETHOOKS_FPM	1
#else
#define  SFAP_USE_BCM_NETHOOKS_FPM	0
#endif
#if SFAP_USE_BCM_NETHOOKS_FPM
#define  SFAP_BCM_NETHOOKS_RX		BCM_NETHOOK_RX_FPM
#else
#define  SFAP_BCM_NETHOOKS_RX		BCM_NETHOOK_RX_SKB
#endif

int sfap_debug = 1;
module_param_named(debug, sfap_debug, int, 0644);
MODULE_PARM_DESC(debug, "Debug level (0-5)");

static int sfap_max_flows = 1024;
module_param_named(max_flows, sfap_max_flows, int, 0444);
MODULE_PARM_DESC(debug, "Max Flows");

int sfap_master_enable;
#ifdef CONFIG_SYSCTL
struct ctl_table_header	*sfap_sysctl_header;
static struct ctl_table sfap_sysctl_table[] = {
	{
		.procname	= "enable",
		.data		= &sfap_master_enable,
		.maxlen		= sizeof(int),
		.mode		= 0644,
		.proc_handler	= proc_dointvec,
	},
	{}
};
#else
module_param_named(enable, sfap_master_enable, int, 0644);
MODULE_PARM_DESC(enable, "Master Enable (0 or 1)");
#endif /* CONFIG_SYSCTL */

static char *net_interfaces_name[] = {
	"eth0",
	"eth1",
	"eth2",
	"eth3",
	"stb0",
	"moca0",
	"cm0"
};

#if SFAP_USE_BCM_NETHOOKS
static enum bcm_nethook_result
sfap_bcm_nethook_rx(struct net_device *dev, enum bcm_nethook_type type,
		    void *buf)
{
#if SFAP_USE_BCM_NETHOOKS_FPM
	struct fpm_buff *fpmb = (struct fpm_buff *)buf;
	if (!sfap_master_enable)
		return BCM_NETHOOK_PASS;
	if (sfap_process_packet_fpm(fpmb, dev) == 0)
		return BCM_NETHOOK_CONSUMED;
	else
		return BCM_NETHOOK_PASS;
#else
	struct sk_buff *skb = (struct sk_buff *) buf;
	if (!sfap_master_enable)
		return BCM_NETHOOK_PASS;
	if (sfap_process_packet(skb, skb->len, skb->dev) == 0)
		return BCM_NETHOOK_CONSUMED;
	else
		return BCM_NETHOOK_PASS;
#endif
}
#else
static inline
unsigned int sfap_nf_br_pre_routing(const struct nf_hook_ops *ops,
				    struct sk_buff *skb,
				    const struct net_device *in,
				    const struct net_device *out,
				    int (*okfn)(struct sk_buff *))
{
	if (!sfap_master_enable)
		return NF_ACCEPT;
	if (sfap_process_packet(skb, skb->len, skb->dev) == 0)
		return NF_STOLEN;
	else
		return NF_ACCEPT;
}

static inline
unsigned int sfap_nf_inet_pre_routing(const struct nf_hook_ops *ops,
				      struct sk_buff *skb,
				      const struct net_device *in,
				      const struct net_device *out,
				      int (*okfn)(struct sk_buff *))
{
	struct sfap_inf *inf = sfap_inf_get_by_dev(skb->dev);
	if (!sfap_master_enable)
		return NF_ACCEPT;
	if (inf && (inf->type == sfap_inf_type_wan) &&
	    (sfap_process_packet(skb, skb->len, skb->dev) == 0))
		return NF_STOLEN;
	else
		return NF_ACCEPT;
}

static struct nf_hook_ops sfap_nf_ops[] __read_mostly = {
	{
		.pf       = NFPROTO_BRIDGE,
		.priority = INT_MIN,
		.hooknum  = NF_BR_PRE_ROUTING,
		.hook     = sfap_nf_br_pre_routing,
		.owner    = THIS_MODULE,
	},
	{
		.pf       = NFPROTO_IPV4,
		.priority = INT_MIN,
		.hooknum  = NF_INET_PRE_ROUTING,
		.hook     = sfap_nf_inet_pre_routing,
		.owner    = THIS_MODULE,
	},
};
#endif

int sfap_inf_register_netdevice(char *name)
{
	struct net_device *dev;
	int type;
	dev = dev_get_by_name(&init_net, name);
	if (!dev)
		return -1;
	if (dev->group == 0)
		return -1;
	if (dev->group == BCM_NETDEVICE_GROUP_LAN)
		type = sfap_inf_type_lan;
	else
		type = sfap_inf_type_wan;
	if (sfap_inf_add(type, name) == 0) {
#if SFAP_USE_BCM_NETHOOKS
		bcm_nethook_register_hook(dev, SFAP_BCM_NETHOOKS_RX,
			RX_FPM_PRIO_SFAP, "SFAP",
			sfap_bcm_nethook_rx);
		bcm_nethook_enable_hook(dev, SFAP_BCM_NETHOOKS_RX,
			sfap_bcm_nethook_rx, true);
#endif
		pr_info("SFAP Init: Interface %s type %d\n",
			name, type);
		return 0;
	}
	return -1;
}

int sfap_inf_unregister_netdevice(char *name)
{
	struct net_device *dev;
	struct sfap_inf *inf;
	dev = dev_get_by_name(&init_net, name);
	if (!dev)
		return -1;

	if (dev->group == 0)
		return -1;
	inf = sfap_inf_get_by_dev(dev);
	if (!inf)
		return -1;
#if SFAP_USE_BCM_NETHOOKS
	bcm_nethook_unregister_hook(inf->dev, SFAP_BCM_NETHOOKS_RX,
		sfap_bcm_nethook_rx);
#endif
	inf->flags = 0;
	return 0;
}

static int __init sfap_init(void)
{
	int i, ret = 0;
	pr_info("SFAP Init: %s use_fpm(%d)\n",
		VER_STR, SFAP_USE_BCM_NETHOOKS_FPM);

	ret = sfap_flow_init(sfap_max_flows);
	if (ret) {
		pr_info("SFAP Init: sfap_flow_init failed\n");
		return ret;
	}

	for (i = 0; i < ARRAY_SIZE(net_interfaces_name); i++)
		sfap_inf_register_netdevice(net_interfaces_name[i]);

#if !SFAP_USE_BCM_NETHOOKS
	ret = nf_register_hooks(sfap_nf_ops, ARRAY_SIZE(sfap_nf_ops));
#endif

	if (ret == 0) {
#if !SFAP_USE_BCM_NETHOOKS
		pr_info("SFAP Init: NF Hooks Registered\n");
#endif
		sfap_procfs_init();
	}

#ifdef CONFIG_SYSCTL
	sfap_sysctl_header =
		register_net_sysctl(&init_net,
				    "net/sfap",
				    sfap_sysctl_table);
#endif

	return ret;
}

static void __exit sfap_exit(void)
{
#if SFAP_USE_BCM_NETHOOKS
	int i;
	for (i = 0; i < sfap_inf_num(); i++) {
		struct sfap_inf *inf = sfap_inf_get_by_index(i);
		if (inf)
			bcm_nethook_unregister_hook(inf->dev,
				SFAP_BCM_NETHOOKS_RX, sfap_bcm_nethook_rx);
	}
#else
	nf_unregister_hooks(sfap_nf_ops, ARRAY_SIZE(sfap_nf_ops));
#endif
#ifdef CONFIG_SYSCTL
	unregister_net_sysctl_table(sfap_sysctl_header);
#endif
	sfap_procfs_exit();
	sfap_flow_exit();
	pr_info("SFAP Exit:\n");
}

module_init(sfap_init);
module_exit(sfap_exit);
MODULE_DESCRIPTION("Soft Forward Assist Processing Driver (SFAP)");
MODULE_AUTHOR("Jayesh Patel");
MODULE_LICENSE("GPL");
MODULE_ALIAS("sfap");
