#ifndef _UBUS_CAPTURE_H
#define _UBUS_CAPTURE_H

/****************************************************************************
 *
 * Broadcom Proprietary and Confidential. (c) 2017 Broadcom.  All rights reserved.
 * The term “Broadcom” refers to Broadcom Limited and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Peter Sulc <peter.sulc@broadcom.com>
 * Filename: ubus_capture.h
 * Description: Ubus capture driver adapted from Mike Sieweke's
 * UbusCapture3390.js BBS script
 ****************************************************************************/

#define CAPT_BASE_OFFSET	0x400
#define CAPT_ENGINE_OFFSET	0x200
#define CAPT_REPLY_OFFSET	0x100
#define UBUS_REG_SIZE		0x1000

// Capture engine registers
#define CAPTURE_ID        0x00  // ID Register
#define CAPTURE_ENABLE    0x04  // Enable Register
#define CAPTURE_CONTROL   0x08  // Control Register
#define CAPTURE_STATUS    0x0c  // Status Register
#define INT_PENDING       0x10  // Interrupt Pending Register
#define INT_MASK          0x14  // Interrupt Mask Register
#define INT_STATE         0x18  // Interrupt State Register
#define ACCUM_HI          0x20  // Accululator High Register
#define ACCUM_LO          0x24  // Accululator High Register
#define ACCUM_HI_THRESH   0x28  // Accululator High Threshold Register
#define ACCUM_LO_THRESH   0x2c  // Accululator Low Threshold Register
#define RATE_INTV         0x30  // Rate Counter Interval Register.
#define RATE_COUNTER      0x34  // Rate Counter Value Register.
#define RATE_CNTR_THRESH  0x38  // Rate Counter Threshold Register.

// There are 2 configurable filters per engine.
#define F0_BASE           0x40  // Filter 0 registers base
#define F1_BASE           0x60  // Filter 1 registers base

// Control registers per filter.
#define F_CONTROL         0x00  // Filter Control Register.
#define F_ADDR_MIN        0x04  // Filter Address Min Register.
#define F_ADDR_MAX        0x08  // Filter Address Max Register.
#define F_HADDR_FLTR      0x0c  // Filter high address
#define F_PID             0x10  // Filter PID Register.
#define F_HDR_FILTER      0x14  // Filter Header Filter Register.
#define F_HDR_MASK        0x18  // Filter Header Filter Mask Register.

// Capture buffer data.  Read all 4 registers to pop the next entry in the FIFO.
#define CAPT_BUF_WORD3    0x80
#define CAPT_BUF_WORD2    0x84
#define CAPT_BUF_WORD1    0x88
#define CAPT_BUF_WORD0    0x8c


// Capture Enable register bits
#define F1_RESET	(1<< 5) // Writing a 1 forces the filter configuration registers to their reset state. This bit is not sticky.
#define F1_EN           (1<< 4) // Filter 1 Enable.
#define F0_RESET        (1<< 3) // Writing a 1 forces the filter configuration registers to their reset state. This bit is not sticky.
#define F0_EN           (1<< 2) // Filter 0 Enable.
#define CE_RESET        (1<< 1) // Capture Engine Reset.
#define CE_EN           (1<< 0) // Capture Engine Enable.

// Capture Control register bits
#define BLIND_CAP_TRANS (1<<28) // ??
#define BLIND_CAP_HDRS  (1<<27) // ??
#define STOPC_ON_FULL   (1<<26) // Stop capture when capture buffer is full.
#define START_CAPTURE   (1<<25) // Manual start capture strobe
#define STOP_CAPTURE    (1<<24) // Manual stop capture strobe
#define STOPC_ON_EXT    (1<<14) // Stop capture on external input?
#define STOPC_ON_IRQ    (1<<13) // Stop capture on Interrupt
#define STOPC_ON_UBERR  (1<<12) // Stop capture on UBUS_ERROR
#define STOPC_ON_F1     (1<<11) // Stop capture on Filter 1 match
#define STOPC_ON_F0     (1<<10) // Stop capture on Filter 0 match
#define STARTC_ON_IRQ   (1<< 8) // Start capture on interrupt
#define STARTC_ON_F1    (1<< 7) // Start capture on Filter 1 match
#define STARTC_ON_F0    (1<< 6) // Start capture on Filter 0 match
#define ACC_CLR         (1<< 3) // Accumulator Clear. This bit is not sticky.
#define SNGL_INTV       (1<< 2) // Single interval (1) or continuous count (0) select.
#define START_INTV      (1<< 1) // Writing a '1' forces the interval counter to begin counting a rate interval. Non sticky bit.
#define RATE_CLR        (1<< 0) // Writing a 1 forces the rate counter to be cleared and the interval count to be reset. This bit is not sticky.

// Interrupt register bits
#define INT_UBERROR         (1<<4)
#define INT_F1_MATCH        (1<<3)
#define INT_F0_MATCH        (1<<2)
#define INT_ACC_CNT_THRESH  (1<<1)
#define INT_RAGE_CNT_THRESH (1<<0)

// Filter Control register bits
#define ADDR_SENSE_SEL  (1<<15) // Address filter bit sense invert.
#define SPID_SENSE_SEL  (1<<14) // SPED filter bit sense invert.
#define DPID_SENSE_SEL  (1<<13) // DPID filter bit sense invert.
#define HDR_SENSE_SEL   (1<<12) // Header filter bit sense invert.
#define AND_OR_SEL      (1<<11) // Filter elements AND (1) or OR (0) to create a match.
#define HDR_ONLY        (1<<10) // Capture/Count Header Only on Match.
#define CAPT_EN         (1<< 9) // Capture on Match Enable.
#define ACC_EN          (1<< 8) // Accumulate on Match Enable.
#define RATE_EN         (1<< 7) // Rate on Match Enable.
#define SPID_RANGE_SEL  (1<< 6) // Source PID Filter Range mode - base/mask (0) or min/max (1)
#define DPID_RANGE_SEL  (1<< 5) // Dest PID Filter Range mode - base/mask (0) or min/max (1)
#define ADDR_RANGE_SEL  (1<< 4) // Address Filter Range mode - base/mask (0) or min/max (1)
#define HDR_EN          (1<< 3) // Header Filter Enable.
#define DPID_EN         (1<< 2) // Dest PID Filter Enable.
#define SPID_EN         (1<< 1) // Source PID Filter Enable.
#define ADDR_EN         (1<< 0) // Address Filter Enable.


// These are the UBUS header bit definitions from .../ubus3/inc/ubus3_define.vh
// and ubus3_capture_blk.v.
// The Confluence page on UBUSv3 doesn't match the RTL.  [<-- Note from 3385 script.]
// http://confluence.broadcom.com/display/UBUSSpecV30/Chapter+11.++System+Debug+Support
//
// 127:96   Timestamp       timestamp value
//
// 95       Valid           0 = FIFO is empty
// 94       Filter1         Filter 1 hit.
// 93       Filter0         Filter 0 hit.
// 92       Header          Header cycle indicator
//
// 84:80    SecLevel        Security level
// 79:72    HighAddr        High address bits (extended address in UBUS3)
// 71       Error           In-band bus error
// 70:68    Command         Command ID (r/w/wa_ck)
// 67       PktTag          Packet tag(?)
// 66:64    LastAddr        Last address (low bits - with dlen indicates transfer length)
//
// 63:32    Address         Bus address start
//
// 31:24    Dlen            Data length
// 23:16    PktId           Packet ID
// 15:8     SrcPid          Source port ID
// 7:0      DestPid         Destination port ID

// If not a header:
// 63:0     DataLane        Data lane bits


// These are the bits in the header filter register:
// 23       Error
// 22:20    Command
// 19       PktTag
// 18:16    LastAddr
// 15:8     Dlen
// 7:0      PktId


// Bit 31 of buf2 indicates the buffer entry is valid.
#define CAPT_VALID (1<<31)


// Device/port mapping from RTL and Confluence
//
//                 Ports
//  Device      Source  Dest    Description
//  ----        ------  ----    -----------
//  MEMC0_P0    n/a      0      DDR0 Memory Controller Port 0
//  MEMC1_P0    n/a      1      DDR1 Controller Port 0
//  UNIWAN       2       2      WAN Gig EMAC
//  DAVIC        3       3      DAVIC MAC
//  SEGDMA       4      n/a     Segmenter/DMA Manager - Master only
//  UTP          5       5      Upstream Token Processor, Segmenter/DMA Manager and US Packet Pre-Processor
//  CRYPTO       6      n/a     Crypto - Master Only
//  USMAC        7       7      Upstream MAC (2.0 and 3.0/3.1) + RBUS bridge to TC's and USPHY
//  DTP          8       8      Downstream Token Processor
//  DFAP         9       9      DOCSIS Forward Assist Processor
//  D3MAC0      10      10      32ch D3.0 DS MAC
//  D3OFDMA     11      11      D3.1 OFDM DS MAC A
//  D3OFDMB     12      12      D3.1 OFDM DS MAC B
//  TCOFDM      13      13      OFDMA TC module
//  APM         14      14      APM/BMU
//  DECT        15      15      Dect Baseband MAC
//  LEAP        16      16      LEAP Processor
//  LEAP        17      17      LEAP Processor
//  MIPS        18      n/a     CMIPS - Cable Modem CPU
//
//  U2U         26      26      U2U Bridge - P0 to P1
//  EMC         27      27      Energy Management Controller
//  CPUCOM      28      28      CPU Communication module (formerly part of G2U)
//  FPM         n/a     29      Free Pool Manager
//  PERIPH      30      30      Peripheral module & RBUS bridge
//  UBUS_CORE   n/a     31      UBUS3 Core control port -
//				Error Port and Capture engine control for UBUS partition 0
//
//  MEMC0_P1    n/a     64      DDR0 Controller Port 1
//  MEMC1_P1    n/a     65      DDR1 Controller Port 1
//  B15CPU-C0   66      n/a     Dual Core B15 CPU - Core 0
//  B15CPU-C1   67      n/a     Dual Core B15 CPU - Core 1
//
//  SF2         n/a     71      Starfighter II Switch
//  DBR         72      72      Runner DBR port 0
//  DBR1        73      73      Runner DBR port 1
//  RABR        74      74      Runner RABR port
//  RBBR        75      75      Runner RBBR port
//  NATC        76      76      Runner NATC port
//
//  U2U         93      93      U2U Bridge - P1 to P0
//  UGB         94      94      UBUS-GISB Bridge
//  UBUS_CORE   n/a     95      UBUS3 Core control port -
// 				Error Port and Capture engine control for UBUS partition 1

#define  MEMC0_P0	0	// DDR0 Memory Controller Port 0
#define  MEMC1_P0	1	// DDR1 Controller Port 0
#define  UNIWAN		2	// WAN Gig EMAC
#define  DAVIC		3	// DAVIC MAC
#define  SEGDMA		4	// Segmenter/DMA Manager - Master only
#define  UTP		5	// Upstream Token Processor,
				// Segmenter/DMA Manager and US Packet Pre-Processor
#define  CRYPTO		6	// Crypto - Master Only
#define  USMAC		7       // Upstream MAC (2.0 and 3.0/3.1) + RBUS bridge to TC's and USPHY
#define  DTP		8       // Downstream Token Processor
#define  DFAP		9       // DOCSIS Forward Assist Processor
#define  D3MAC0		10      // 32ch D3.0 DS MAC
#define  D3OFDMA	11      // D3.1 OFDM DS MAC A
#define  D3OFDMB	12      // D3.1 OFDM DS MAC B
#define  TCOFDM		13      // OFDMA TC module
#define  APM		14      // APM/BMU
#define  DECT		15      // Dect Baseband MAC
#define  LEAP0		16      // LEAP Processor
#define  LEAP1		17      // LEAP Processor
#define  MIPS		18      // CMIPS - Cable Modem CPU
#define  U2U0		26      // U2U Bridge - P0 to P1
#define  EMC		27      // Energy Management Controller
#define  CPUCOM		28      // CPU Communication module (formerly part of G2U)
#define  FPM		29      // Free Pool Manager
#define  PERIPH		30      // Peripheral module & RBUS bridge
#define  UBUS_CORE0	31      // UBUS3 Core control port -
				// Error Port and Capture engine control for UBUS partition 0
#define  MEMC0_P1	64      // DDR0 Controller Port 1
#define  MEMC1_P1	65      // DDR1 Controller Port 1
#define  B15CPU_C0	66      // Dual Core B15 CPU - Core 0
#define  B15CPU_C1	67      // Dual Core B15 CPU - Core 1
#define  SF2            71      // Starfighter II Switch
#define  DBR            72      // Runner DBR port 0
#define  DBR1           73      // Runner DBR port 1
#define  RABR           74      // Runner RABR port
#define  RBBR           75      // Runner RBBR port
#define  NATC           76      // Runner NATC port
#define  U2U1           93      // U2U Bridge - P1 to P0
#define  UGB            94      // UBUS-GISB Bridge
#define  UBUS_CORE1     95      // UBUS3 Core control port
				// Error Port and Capture engine control for UBUS partition 1

extern u32 ubus_capt_addr_start;
extern u32 ubus_capt_addr_end;
extern u32 ubus_capt_addr_exclude;
extern u32 ubus_capt_pid_start;
extern u32 ubus_capt_pid_end;
extern u32 ubus_capt_pid_exclude;
extern u32 ubus_capt_on;

extern int __init ubus_capt_early_init(void);
extern int __init ubus_capt_alloc_bases(unsigned num_bases);
extern int __init ubus_capt_map_base(int n, u64 base, u32 size, u32 segs);
extern void ubus_capt_start(void);
extern void ubus_capt_stop(void);
extern void ubus_capt_print(void);
void ubus_capt_header_only(int fnum);
void ubus_capt_enable_dest_pid_range(int fnum, int min, int max, int exclude);
void ubus_capt_enable_src_pid_range(int fnum, int min, int max, int exclude);
void ubus_capt_stop_on_full(void);
void ubus_capt_stop_on_error(void);
void ubus_capt_stop_on_filter(int fnum);

extern void ubus_capture_exception_print(void);

extern void *ubus_capt_show_proc_start(loff_t *pos);
extern void *ubus_capt_show_proc_next(void *v);
extern int   ubus_capt_show_proc_show(struct seq_file *seq, void *v);

#endif
