 /****************************************************************************
 *
 * Broadcom Proprietary and Confidential.
 * (c) 2016 Broadcom. All rights reserved.
 * The term "Broadcom" refers to Broadcom Limited and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Tim Ross <tross@broadcom.com>
 *****************************************************************************/
#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/if_vlan.h>
#include <linux/mdio.h>
#include "dqnet.h"
#include "dqnet_priv.h"
#include "dqnet_dbg.h"
#include "dqnet_switch.h"
#include "ethsw.h"
#include "ethsw_core.h"
#include "ethsw_priv.h"

static int moca_enabled = 0;

static int dqnet_get_max_ports_ethsw(void)
{
	return ETHSW_PORT_MAX;
}

/*
 * The FAP exception Q is muxed with packets from various devices that
 * we have to demux to the correct device. Here we demux packets from
 * the switch by querying the switch's ARL table to translate the
 * source MAC address to a port number. The interface ID and port
 * number are used to index into the demux table for the
 * dqnet_netdevice.
 *
 * 	chan	DQM channel on which packet was received
 *      fb	FPM buffer descriptor
 *
 * Returns
 *	0 for success, < 0 is error code
 */
static int dqnet_demux_arl_ethsw(struct dqnet_channel *chan,
				 struct fpm_buff *fb)
{
	int status = 0;
	struct dqnet_netdev *ndev = NULL;
	struct vlan_ethhdr *hdr = NULL;
	u16 port;
	u16 vid = 0;
	unsigned long flags;
	u8 *data;
	u32 len;

	pr_debug("-->\n");

	if (fb->type == BUF_TYPE_FPM) {
		data = fb->data;
		len = fb->len;
	} else {
		data = fb->skb->data;
		len = fb->skb->len;
	}
	if (len < sizeof(struct ethhdr)) {
		pr_debug("Packet too small for Ethernet header.\n");
		err_stats.no_eth_hdr++;
		goto err_ndev_lookup;
	}
	hdr = (struct vlan_ethhdr *)data;
	if (hdr->h_vlan_proto == htons(ETH_P_8021Q))
		vid = htons(hdr->h_vlan_TCI & VLAN_VID_MASK);

	status = ethsw_arl_table_find(hdr->h_source, vid, &port);
	if (status)
		goto err_ndev_lookup;
	if (port >= if_sub_id_max) {
		status = -EIO;
		goto err_ndev_lookup;
	}
	spin_lock_irqsave(&demux_tbl_lock, flags);
	ndev = demux_tbl[fb->if_id][port];
	spin_unlock_irqrestore(&demux_tbl_lock, flags);

err_ndev_lookup:
	if (!ndev) {
		if (hdr && !is_multicast_ether_addr(hdr->h_source)) {
			pr_debug("=========================================");
			pr_debug("===\n");
			pr_debug("DROP: Unable to determine switch port #.\n");
			pr_debug("Chan %s RX packet\n", chan->name);
			show_dbg_pkt_desc(fb);
			show_dbg_pkt(data, len);
			pr_debug("=========================================");
			pr_debug("===\n");
		}
		err_stats.switch_arl_lookup++;
		goto done;
	}

	status = chan->rx_pkt(ndev, fb);
	goto done;

done:
	pr_debug("<--\n");
	return status;
}

/*
 * Query the switch for stats on the particular port
 *
 * Parameters
 *      dev	device to query
 *      stats	statistics struct to fill in
*/
static void dqnet_get_stats_ethsw(struct net_device *dev,
				  struct rtnl_link_stats64 *stats)
{
	struct dqnet_netdev *ndev = netdev_priv(dev);
	u64 tmp;

	pr_debug("%s:-->\n",dev->name);

	ethsw_mib_counter64_get(ndev->if_sub_id,
				MIB_RX_BROADCAST_PKTS,
				&tmp);
	ethsw_mib_counter64_get(ndev->if_sub_id,
				MIB_RX_MULTICAST_PKTS,
				&stats->multicast);
	stats->multicast += tmp;
	stats->rx_packets = stats->multicast;
	ethsw_mib_counter64_get(ndev->if_sub_id,
				MIB_RX_UNICAST_PKTS,
				&tmp);
	stats->rx_packets += tmp;
	ethsw_mib_counter64_get(ndev->if_sub_id,
				MIB_RX_GOOD_OCTETS,
				&stats->rx_bytes);
	ethsw_mib_counter64_get(ndev->if_sub_id,
				MIB_RX_UNDERSIZE_PKTS,
				&tmp);
	stats->rx_length_errors = tmp;
	ethsw_mib_counter64_get(ndev->if_sub_id,
				MIB_RX_OVERSIZE_PKTS,
				&tmp);
	stats->rx_length_errors += tmp;
	ethsw_mib_counter64_get(ndev->if_sub_id,
				MIB_RX_FCS_ERRORS,
				&stats->rx_crc_errors);
	ethsw_mib_counter64_get(ndev->if_sub_id,
				MIB_RX_ALIGNMENT_ERRORS,
				&stats->rx_frame_errors);
	stats->rx_errors += stats->rx_length_errors + stats->rx_crc_errors +
		stats->rx_frame_errors;
	ethsw_mib_counter64_get(ndev->if_sub_id,
				MIB_RX_SYMBOL_ERRORS,
				&tmp);
	stats->rx_errors += tmp;
	ethsw_mib_counter64_get(ndev->if_sub_id,
				MIB_RX_IN_RANGE_ERRORS,
				&tmp);
	stats->rx_errors += tmp;
	ethsw_mib_counter64_get(ndev->if_sub_id,
				MIB_RX_OUT_OF_RANGE_ERRORS,
				&tmp);
	stats->rx_errors += tmp;
	ethsw_mib_counter64_get(ndev->if_sub_id,
				MIB_RX_DROP_PKTS,
				&tmp);
	stats->rx_dropped += tmp;
	ethsw_mib_counter64_get(ndev->if_sub_id,
				MIB_RX_DISCARD,
				&tmp);
	stats->rx_dropped += tmp;
	ethsw_mib_counter64_get(ndev->if_sub_id,
				MIB_TX_COLLISIONS,
				&stats->collisions);

	ethsw_mib_counter64_get(ndev->if_sub_id,
				MIB_TX_BROADCAST_PKTS,
				&tmp);
	stats->tx_packets = tmp;
	ethsw_mib_counter64_get(ndev->if_sub_id,
				MIB_TX_MULTICAST_PKTS,
				&tmp);
	stats->tx_packets += tmp;
	ethsw_mib_counter64_get(ndev->if_sub_id,
				MIB_TX_UNICAST_PKTS,
				&tmp);
	stats->tx_packets += tmp;
	ethsw_mib_counter64_get(ndev->if_sub_id,
				MIB_TX_OCTETS,
				&stats->tx_bytes);
	ethsw_mib_counter64_get(ndev->if_sub_id,
				MIB_TX_DROP_PKTS,
				&tmp);
	stats->tx_dropped += tmp;
	pr_debug("%s:<--\n",dev->name);
}

static void dqnet_clr_stats_ethsw(struct net_device *dev)
{
	struct dqnet_netdev *ndev = netdev_priv(dev);
	pr_debug("%s:-->\n",dev->name);
	ethsw_mib_counters_clear(ndev->if_sub_id);
	pr_debug("%s:<--\n",dev->name);
}

static struct rtnl_link_stats64 *dqnet_link_stats_ethsw(struct net_device *dev,
		struct rtnl_link_stats64 *stats)
{
	if (stats)
		dqnet_get_stats_ethsw(dev, stats);
	else
		dqnet_clr_stats_ethsw(dev);

	return stats;
}

/*
 * Add a VID on a ethernet device
 *
 * Parameters
 *	dev	device to modify
 *	vid	non-default VID
 *
 * Returns
 *	0 for success, < 0 is error code
 */
static int dqnet_vlan_rx_add_vid_ethsw(struct net_device *dev,
				       __be16 proto, u16 vid)
{
	struct dqnet_netdev *ndev = netdev_priv(dev);
	int status = 0;

	pr_debug("%s:-->\n",dev->name);

	if (ndev->link_type != DQNET_LINK_TYPE_SWITCH || vid == 0 || vid == 1)
		goto done;

#if defined(SINGLE_VLAN_SUPPORT)
	status = ethsw_vlan_port_vid_delete(ndev->if_sub_id, 1, 1);
	if (status) {
		netdev_err(dev, "Failure (%d) delete vid 1.\n", status);
		goto done;
	}

	status = ethsw_vlan_port_vid_add(ndev->if_sub_id, vid, 1);
#else
	status = ethsw_vlan_port_vid_add(ndev->if_sub_id, vid, 0);
#endif
	if (status) {
		netdev_err(dev, "Failure (%d) add vid %d.\n", status, vid);
		goto done;
	}

#if defined(SINGLE_VLAN_SUPPORT)
	status = ethsw_vlan_port_tag_set(ndev->if_sub_id, vid, 0);
	if (status) {
		netdev_err(dev, "Failure (%d) tag vid %d.\n", status, vid);
		goto done;
	}
#endif

done:
	pr_debug("%s:<--\n",dev->name);
	return status;
}

/*
 * Kill a VID on a ethernet device
 *
 * Parameters
 *	dev	device to modify
 *	vid	non-default VID
 *
 * Returns
 *	0 for success, < 0 is error code
 */
static int dqnet_vlan_rx_kill_vid_ethsw(struct net_device *dev,
					__be16 proto, u16 vid)
{
	struct dqnet_netdev *ndev = netdev_priv(dev);
	int status = 0;

	pr_debug("%s:-->\n",dev->name);

	if (ndev->link_type != DQNET_LINK_TYPE_SWITCH || vid == 0 || vid == 1)
		goto done;

#if defined(SINGLE_VLAN_SUPPORT)
	status = ethsw_vlan_port_vid_delete(ndev->if_sub_id, vid, 1);
#else
	status = ethsw_vlan_port_vid_delete(ndev->if_sub_id, vid, 0);
#endif
	if (status) {
		netdev_err(dev, "Failure (%d) delete vid %d.\n", status, vid);
		goto done;
	}

#if defined(SINGLE_VLAN_SUPPORT)
	status = ethsw_vlan_port_vid_add(ndev->if_sub_id, 1, 1);
	if (status) {
		netdev_err(dev, "Failure (%d) add vid 1.\n", status);
		goto done;
	}

	status = ethsw_vlan_port_tag_set(ndev->if_sub_id, 1, 0);
	if (status) {
		netdev_err(dev, "Failure (%d) tag vid 1.\n", status);
		goto done;
	}
#endif

done:
	pr_debug("%s:<--\n",dev->name);
	return status;
}

/*
 * Get switch port power configuration & status
 *
 * Parameters
 *      dev	net device
 *      pwr	port power config
 *
 * Returns
 *	0 for success, < 0 is error code
 */
static int dqnet_get_port_pwr_ethsw(struct net_device *dev,
				    struct dqnet_port_pwr *pwr)
{
	int status = 0;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	struct ethsw_power_config port_pwr_cfg;
	struct ethsw_power_status port_pwr_status;

	pr_debug("%s:-->\n",dev->name);

	netdev_dbg(dev, "Getting port %d power config.\n", ndev->if_sub_id);
	status = ethsw_port_power_config_get(ndev->if_sub_id, &port_pwr_cfg);
	if (status) {
		netdev_dbg(dev, "Failed to get %s switch ", ndev->name);
		netdev_dbg(dev, "port power config.");
		goto done;
	}
	status = ethsw_port_power_status_get(ndev->if_sub_id, &port_pwr_status);
	if (status) {
		netdev_dbg(dev, "Failed to get %s switch ", ndev->name);
		netdev_dbg(dev, "port power config.");
		goto done;
	}
	pwr->auto_pwr_down = port_pwr_cfg.auto_power;
	pwr->eee_enabled = port_pwr_cfg.eee_params.enable;
	pwr->eee_active = port_pwr_status.eee_active;
	pwr->eee_supported = mmd_eee_cap_to_ethtool_sup_t(port_pwr_status.eee_supported);
	pwr->eee_advertised = mmd_eee_adv_to_ethtool_adv_t(port_pwr_cfg.eee_params.advertised);
	pwr->eee_lp_advertised = mmd_eee_adv_to_ethtool_adv_t(port_pwr_status.eee_lp_advertised);
	pwr->eee_tx_lpi_enabled = port_pwr_status.eee_assert;
	switch (ethsw_switch_port_speed(ndev->if_sub_id)) {
	case PORT_SPEED_100M:
		pwr->eee_tx_lpi_timer = port_pwr_cfg.eee_params.min_lp_timer_h;
		break;
	case PORT_SPEED_1000M:
		pwr->eee_tx_lpi_timer = port_pwr_cfg.eee_params.min_lp_timer_g;
		break;
	default:
		pwr->eee_tx_lpi_timer = -1;
	}

done:
	pr_debug("%s:<--\n",dev->name);
	return status;
}

/*
 * Set switch port power configuration
 *
 * Parameters
 *      dev	 net device
 *      pwr	port power config
 *
 * Returns
 *	0 for success, < 0 is error code
 */
static int dqnet_set_port_pwr_ethsw(struct net_device *dev,
				    struct dqnet_port_pwr *pwr)
{
	int status = 0;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	struct ethsw_power_config port_pwr_cfg;

	pr_debug("%s:-->\n",dev->name);

	netdev_dbg(dev, "Setting port %d power config.\n", ndev->if_sub_id);
	memset(&port_pwr_cfg, 0,
	       sizeof(struct ethsw_power_config));
	status = ethsw_port_power_config_get(ndev->if_sub_id, &port_pwr_cfg);
	if (status) {
		netdev_dbg(dev, "Failed to get %s switch ", ndev->name);
		netdev_dbg(dev, "port power config.");
		goto done;
	}
	port_pwr_cfg.auto_power = pwr->auto_pwr_down;
	port_pwr_cfg.auto_dll = pwr->auto_pwr_down;
	port_pwr_cfg.eee_params.enable = pwr->eee_enabled;
	port_pwr_cfg.eee_params.advertised = ethtool_adv_to_mmd_eee_adv_t(pwr->eee_advertised);
	switch (ethsw_switch_port_speed(ndev->if_sub_id)) {
	case PORT_SPEED_100M:
		port_pwr_cfg.eee_params.min_lp_timer_h = (pwr->eee_tx_lpi_timer == -1) ?
			CREG_EEE_MIN_LP_TIMER_H_DFLT : pwr->eee_tx_lpi_timer;
		break;
	case PORT_SPEED_1000M:
		port_pwr_cfg.eee_params.min_lp_timer_g = (pwr->eee_tx_lpi_timer == -1) ?
			CREG_EEE_MIN_LP_TIMER_G_DFLT : pwr->eee_tx_lpi_timer;
		break;
	}
	status = ethsw_port_power_config_set(ndev->if_sub_id, &port_pwr_cfg);
	if (status) {
		netdev_dbg(dev, "Failed to set %s switch ", ndev->name);
		netdev_dbg(dev, "port power config.");
	}

done:
	pr_debug("%s:<--\n",dev->name);
	return status;
}

/*
 * Enable switch port
 *
 * Parameters
 *      dev	 net device
 *
 * Returns
 *	0 for success, < 0 is error code
 */
static int dqnet_enable_port_ethsw(struct net_device *dev)
{
	int status = 0;
	struct dqnet_netdev *ndev = netdev_priv(dev);

	pr_debug("%s:-->\n",dev->name);

	netdev_dbg(dev, "Enabling %s switch port.\n", ndev->name);
	if (0 == (strncmp(ndev->name, DQNET_MOCA_NAME_STRING,
			strlen(DQNET_MOCA_NAME_STRING)))) {
		moca_enabled = 1;
	}

	status = ethsw_port_enable(ndev->if_sub_id);
	if (status)
		netdev_err(dev, "Failed to enable %s switch port.", ndev->name);

	pr_debug("%s:<--\n",dev->name);
	return status;
}

/*
 * Disable switch port
 *
 * Parameters
 *      dev	 net device
 *
 * Returns
 *	0 for success, < 0 is error code
 */
static int dqnet_disable_port_ethsw(struct net_device *dev)
{
	int status = 0;
	struct dqnet_netdev *ndev = netdev_priv(dev);

	pr_debug("%s:-->\n",dev->name);

	netdev_dbg(dev, "Disabling %s switch port.\n", ndev->name);
	status = ethsw_port_disable(ndev->if_sub_id);
	if (status)
		netdev_err(dev, "Failed to disable %s switch port.", ndev->name);

	pr_debug("%s:<--\n",dev->name);
	return status;
}

static inline int dqnet_balance_imp_get_action(int action_in)
{
	/* Account for MoCA HW enabled/disabled when determining final action */
	if (!moca_enabled)
		return (action_in + DQNET_IMP_LAG_ACT_DEFAULT_NOMOCA);
	else
		return (action_in);
}

static int dqnet_balance_imp_lag_port_ethsw(struct dqnet_netdev *ndev)
{
	int status = 0;
	int port_num = 0;
	int found = -1;
	int action = 0;
	struct port_stat_info {
		int enabled;
		struct ethsw_port_status status;
	};
	struct port_stat_info port_stat[ETHSW_PORT_MAX];
	struct ethsw_port_status *port_status = NULL;

	pr_debug("IMP rebalance triggered by dev:%s if_id:%d curr_imp:%d\n",
		ndev->name, ndev->if_sub_id, ndev->swport_imp);

	/* Get port status for all interesting switch ports */
	for (port_num = 0; port_num < ETHSW_PORT_MAX; port_num++) {
		if (ethsw_port_is_enabled(port_num)) {
			port_stat[port_num].enabled = 1;
			port_status = &(port_stat[port_num].status);
			status = ethsw_port_status_get(port_num,
						       port_status);
			if (status) {
				continue;
			}
		} else {
			port_stat[port_num].enabled = 0;
		}
	}

	/* If port 7 link speed is 2.5G, then reset to default mapping.
	   Treat link state down the same as slow speed and do not "reserve"
	   the IMP port for port 7 */
	if ((port_stat[7].status.speed > PORT_SPEED_1000M) &&
	    (port_stat[7].status.link_up)) {
		action = dqnet_balance_imp_get_action(DQNET_IMP_LAG_ACT_DEFAULT);
		status = dqnet_balance_imp_lag_ports(action, -1);
		goto done;
	} else if ((port_stat[7].status.speed < PORT_SPEED_1000M) ||
		   (!port_stat[7].status.link_up)) {
		action = dqnet_balance_imp_get_action(DQNET_IMP_LAG_ACT_OPT3);
		status = dqnet_balance_imp_lag_ports(action, -1);
		goto done;
	}

	/* Find if there is a 100M or slower speed port */
	for (port_num = 0; port_num < ETHSW_PHY_PORT_MAX; port_num++) {
		if (!port_stat[port_num].enabled) {
			continue;
		}

		port_status = &(port_stat[port_num].status);
		if ((port_status->speed <= PORT_SPEED_100M) &&
		    (port_status->link_up)) {
			found = port_num;
			break;
		}
	}

	if (found != -1) {
		/* Found a 100M port */
		action = dqnet_balance_imp_get_action(DQNET_IMP_LAG_ACT_OPT2);
		status = dqnet_balance_imp_lag_ports(action, found);
		goto done;
	}

	/* All ports found to be at 1G spped */
	action = dqnet_balance_imp_get_action(DQNET_IMP_LAG_ACT_OPT1);
	status = dqnet_balance_imp_lag_ports(action, -1);

done:
	return status;
}

/*
 * Handle link status changes from ethernet switch port PHY
 *
 * Parameters
 *      context	ptr to our private net device struct
 *      status	struct containing status of specific port
 */
static void dqnet_port_status_cb_ethsw(void *context,
				       struct ethsw_port_status *status)
{
	struct dqnet_netdev *ndev = context;

	netdev_dbg(ndev->dev, "-->\n");

	if (status->link_up) {
		dqnet_switch_eee(ndev->dev, 1);
		netif_err(ndev, link, ndev->dev, "Link up.\n");
		ndev->link_state = DQNET_LINK_STATE_UP;
		netif_carrier_on(ndev->dev);
	} else {
		dqnet_switch_eee(ndev->dev, 0);
		netif_err(ndev, link, ndev->dev, "Link down.\n");
		ndev->link_state = DQNET_LINK_STATE_DOWN;
		netif_carrier_off(ndev->dev);
	}

	/* Balance IMP LAG port mapping after link state change */
	dqnet_balance_imp_lag_port_ethsw(ndev);

	netdev_dbg(ndev->dev, "<--\n");
	return;
}

/*
 * Connect switch port
 *
 * Parameters
 * 	dev	net device
 *
 * Returns
 *	0 for success, < 0 is error code
 */
static int dqnet_connect_port_ethsw(struct net_device *dev)
{
	int status = 0;
	struct dqnet_netdev *ndev = netdev_priv(dev);

	pr_debug("%s:-->\n",dev->name);

	netdev_dbg(dev, "Connecting to %s switch port.\n", ndev->name);
	status = ethsw_port_connect(dev,&ndev->if_sub_id,ndev->name,
				    dqnet_port_status_cb_ethsw, ndev);
	if (status) {
		netdev_err(dev, "Failed to connect to switch.\n");
	}

	pr_debug("%s:<--\n",dev->name);
	return status;
}

/*
 * Disconnect switch port
 *
 * Parameters
 * 	dev	net device
 *
 * Returns
 *	0 for success, < 0 is error code
 */
static int dqnet_disconnect_port_ethsw(struct net_device *dev)
{
	int status = 0;
	struct dqnet_netdev *ndev = netdev_priv(dev);

	pr_debug("%s:-->\n",dev->name);

	status = ethsw_port_disconnect(ndev->if_sub_id);

	pr_debug("%s:<--\n",dev->name);
	return status;
}

/*
 * Get switch port config
 *
 * Parameters
 * 	dev	net device
 *
 * Returns
 *	0 for success, < 0 is error code
 */
static int dqnet_get_port_config_ethsw(struct net_device *dev)
{
	int status = 0;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	struct ethsw_port_config port_config;
	struct ethsw_port_status port_status;
	int speed = (ndev->phy_mode == PHY_INTERFACE_MODE_SGMII) ?
		SPEED_2500 : SPEED_1000;

	pr_debug("%s:-->\n",dev->name);

	status = ethsw_port_config_get(ndev->if_sub_id, &port_config);
	if (status) {
		ndev->link_autoneg = AUTONEG_DISABLE;
		ndev->link_speed = speed;
		ndev->link_duplex = DUPLEX_UNKNOWN;
		ndev->link_pause = -1;
		goto done;
	}

	status = ethsw_port_status_get(ndev->if_sub_id, &port_status);
	if (status) {
		ndev->link_autoneg = AUTONEG_DISABLE;
		ndev->link_speed = speed;
		ndev->link_duplex = DUPLEX_UNKNOWN;
		ndev->link_pause = -1;
		goto done;
	}

	ndev->link_autoneg = port_config.auto_neg ? AUTONEG_ENABLE :
		AUTONEG_DISABLE;
	ndev->link_pause = -1;
	if (port_status.link_up)
		ndev->link_duplex = port_status.full_duplex ? DUPLEX_FULL : DUPLEX_HALF;
	else
		ndev->link_duplex = DUPLEX_UNKNOWN;

	switch (port_status.speed) {
	case PORT_SPEED_10M:
		ndev->link_speed = SPEED_10;
		break;
	case PORT_SPEED_100M:
		ndev->link_speed = SPEED_100;
		break;
	case PORT_SPEED_2500M:
		ndev->link_speed = SPEED_2500;
		break;
	case PORT_SPEED_1000M:
	default:
		ndev->link_speed = SPEED_1000;
		break;
	}

done:
	pr_debug("%s:<--\n",dev->name);
	return status;
}

/*
 * Set switch port config
 *
 * Parameters
 * 	dev	net device
 *
 * Returns
 *	0 for success, < 0 is error code
 */
static int dqnet_set_port_config_ethsw(struct net_device *dev)
{
	int status = 0;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	struct ethsw_port_config config;

	pr_debug("%s:-->\n",dev->name);

	config.auto_neg = ndev->link_autoneg == AUTONEG_ENABLE ? true : false;
	config.full_duplex = ndev->link_duplex == DUPLEX_FULL ? true : false;
	switch (ndev->link_speed) {
	case SPEED_10:
		config.speed = PORT_SPEED_10M;
		break;
	case SPEED_100:
		config.speed = PORT_SPEED_100M;
		break;
	case SPEED_2500:
		config.speed = PORT_SPEED_2500M;
		break;
	case SPEED_1000:
	default:
		config.speed = PORT_SPEED_1000M;
		break;
	}
	status = ethsw_port_config_set(ndev->if_sub_id, &config);

	pr_debug("%s:<--\n",dev->name);
	return status;
}

static bool dqnet_vlan_enabled_ethsw(void)
{
	if (ethsw_vlan_enabled()) {
		return true;
	}
	else {
		return false;
	}
}

static bool dqnet_ipv6_mc_flood_disabled_ethsw(void)
{
	if (ethsw_ipv6_mc_flood_disabled())
		return true;
	else
		return false;
}

static bool dqnet_sw_drop_b_m(int port, int start_port)
{
	return ethsw_drop_packet(port,start_port);
}


/*
 * Get switch port MIB counters
 *
 * Parameters
 * 	dev	net device
 *
 * Returns
 *	0 for success, < 0 is error code
 */
static int dqnet_get_mib_counters_ethsw(struct net_device *dev,
					struct dqnet_switch_stats *stats)
{
	int status = 0;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	u32 mib_values[MIB_COUNTER_MAX - MIB_COUNTER_START];

	pr_debug("%s:-->\n",dev->name);

	status = ethsw_mib_counters_get(ndev->if_sub_id,
					MIB_COUNTER_START,
					MIB_COUNTER_MAX - MIB_COUNTER_START,
					mib_values);
	stats->tx_bytes = *(u64 *)&mib_values[MIB_TX_OCTETS];
	stats->tx_drop_pkts = mib_values[MIB_TX_DROP_PKTS];
	stats->tx_broadcast_pkts = mib_values[MIB_TX_BROADCAST_PKTS];
	stats->tx_multicast_pkts = mib_values[MIB_TX_MULTICAST_PKTS];
	stats->tx_unicast_pkts = mib_values[MIB_TX_UNICAST_PKTS];
	stats->tx_pause_pkts = mib_values[MIB_TX_PAUSE_PKTS];
	stats->tx_collisions = mib_values[MIB_TX_COLLISIONS];
	stats->tx_single_collision = mib_values[MIB_TX_SINGLE_COLLISION];
	stats->tx_multiple_collision = mib_values[MIB_TX_MULTIPLE_COLLISION];
	stats->tx_late_collision = mib_values[MIB_TX_LATE_COLLISION];
	stats->tx_excessive_collision =
		mib_values[MIB_TX_EXCESSIVE_COLLISION];
	stats->tx_deferred_transmit = mib_values[MIB_TX_DEFERRED_TRANSMIT];
	stats->tx_frame_in_disc = mib_values[MIB_TX_FRAME_IN_DISC];
	stats->tx_q0_pkt = mib_values[MIB_TX_Q0_PKT];
	stats->tx_q1_pkt = mib_values[MIB_TX_Q1_PKT];
	stats->tx_q2_pkt = mib_values[MIB_TX_Q2_PKT];
	stats->tx_q3_pkt = mib_values[MIB_TX_Q3_PKT];
	stats->tx_q4_pkt = mib_values[MIB_TX_Q4_PKT];
	stats->tx_q5_pkt = mib_values[MIB_TX_Q5_PKT];
	stats->tx_q6_pkt = mib_values[MIB_TX_Q6_PKT];
	stats->tx_q7_pkt = mib_values[MIB_TX_Q7_PKT];
	stats->tx_pkts_64_bytes =
		mib_values[MIB_TX_PKTS_64_OCTETS];
	stats->tx_pkts_65to127_bytes =
		mib_values[MIB_TX_PKTS_65TO127_OCTETS];
	stats->tx_pkts_128to255_bytes =
		mib_values[MIB_TX_PKTS_128TO255_OCTETS];
	stats->tx_pkts_256to511_bytes =
		mib_values[MIB_TX_PKTS_256TO511_OCTETS];
	stats->tx_pkts_512to1023_bytes =
		mib_values[MIB_TX_PKTS_512TO1023_OCTETS];
	stats->tx_pkts_1024tomax_bytes =
		mib_values[MIB_TX_PKTS_1024TOMAX_OCTETS];
	stats->rx_bytes = *(u64 *)&mib_values[MIB_RX_OCTETS];
	stats->rx_good_bytes = *(u64 *)&mib_values[MIB_RX_GOOD_OCTETS];
	stats->rx_drop_pkts = mib_values[MIB_RX_DROP_PKTS];
	stats->rx_broadcast_pkts = mib_values[MIB_RX_BROADCAST_PKTS];
	stats->rx_multicast_pkts = mib_values[MIB_RX_MULTICAST_PKTS];
	stats->rx_unicast_pkts = mib_values[MIB_RX_UNICAST_PKTS];
	stats->rx_pause_pkts = mib_values[MIB_RX_PAUSE_PKTS];
	stats->rx_sa_changes = mib_values[MIB_RX_SA_CHANGES];
	stats->rx_undersize_pkts = mib_values[MIB_RX_UNDERSIZE_PKTS];
	stats->rx_pkts_64_bytes =
		mib_values[MIB_RX_PKTS_64_OCTETS];
	stats->rx_pkts_65to127_bytes =
		mib_values[MIB_RX_PKTS_65TO127_OCTETS];
	stats->rx_pkts_128to255_bytes =
		mib_values[MIB_RX_PKTS_128TO255_OCTETS];
	stats->rx_pkts_256to511_bytes =
		mib_values[MIB_RX_PKTS_256TO511_OCTETS];
	stats->rx_pkts_512to1023_bytes =
		mib_values[MIB_RX_PKTS_512TO1023_OCTETS];
	stats->rx_pkts_1024tomax_bytes =
		mib_values[MIB_RX_PKTS_1024TOMAX_OCTETS];
	stats->rx_oversize_pkts = mib_values[MIB_RX_OVERSIZE_PKTS];
	stats->rx_jumbo_pkts = mib_values[MIB_RX_JUMBO_PKTS];
	stats->rx_fragments = mib_values[MIB_RX_FRAGMENTS];
	stats->rx_jabbers = mib_values[MIB_RX_JABBERS];
	stats->rx_discard = mib_values[MIB_RX_DISCARD];
	stats->rx_alignment_errors = mib_values[MIB_RX_ALIGNMENT_ERRORS];
	stats->rx_fcs_errors = mib_values[MIB_RX_FCS_ERRORS];
	stats->rx_symbol_errors = mib_values[MIB_RX_SYMBOL_ERRORS];
	stats->rx_in_range_errors = mib_values[MIB_RX_IN_RANGE_ERRORS];
	stats->rx_out_of_range_errors = mib_values[MIB_RX_OUT_OF_RANGE_ERRORS];
	stats->eee_lpi_event = mib_values[MIB_EEE_LPI_EVENT];
	stats->eee_lpi_duration = mib_values[MIB_EEE_LPI_DURATION];

	pr_debug("%s:<--\n",dev->name);
	return status;
}

static int dqnet_set_port_stp_state_ethsw(struct net_device *dev,
					  int port, unsigned int state)
{
	int status = 0;
	struct dqnet_netdev *ndev = netdev_priv(dev);

	pr_debug("%s:-->\n",dev->name);
	status = ethsw_port_set_stp_state(ndev->if_sub_id, state);
	if (status)
		netdev_err(dev, "Failed to set switch port STP state.\n");
	pr_debug("%s:<--\n",dev->name);
	return status;
}

static int dqnet_set_mtu_ethsw(unsigned int mtu)
{
	return ethsw_set_mtu(mtu);
}

struct dqnet_switch_ops ethsw = {
	.get_max_ports		= dqnet_get_max_ports_ethsw,
	.demux_arl		= dqnet_demux_arl_ethsw,
	.link_stats		= dqnet_link_stats_ethsw,
	.vlan_rx_add_vid	= dqnet_vlan_rx_add_vid_ethsw,
	.vlan_rx_kill_vid	= dqnet_vlan_rx_kill_vid_ethsw,
	.get_port_pwr		= dqnet_get_port_pwr_ethsw,
	.set_port_pwr		= dqnet_set_port_pwr_ethsw,
	.enable_port		= dqnet_enable_port_ethsw,
	.disable_port		= dqnet_disable_port_ethsw,
	.connect_port		= dqnet_connect_port_ethsw,
	.disconnect_port	= dqnet_disconnect_port_ethsw,
	.get_port_config	= dqnet_get_port_config_ethsw,
	.set_port_config	= dqnet_set_port_config_ethsw,
	.get_mib_counters	= dqnet_get_mib_counters_ethsw,
	.set_port_stp_state	= dqnet_set_port_stp_state_ethsw,
	.vlan_enabled		= dqnet_vlan_enabled_ethsw,
	.switch_ipv6_mc_flood_disabled = dqnet_ipv6_mc_flood_disabled_ethsw,
	.switch_drop_brcst_mult  = dqnet_sw_drop_b_m,
	.set_mtu		= dqnet_set_mtu_ethsw
};
EXPORT_SYMBOL(ethsw);
