 /****************************************************************************
 *
 * Copyright (c) 2015 Broadcom Corporation
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 *	Authors:Jayesh Patel <jayeshp@broadcom.com>
 *
 *  September, 2013
 *
 ****************************************************************************/
#include <linux/module.h>
#include <linux/netfilter/x_tables.h>
#include <linux/netfilter_bridge/ebtables.h>
#include <linux/netfilter_bridge/ebt_ip.h>
#include <linux/ip.h>
#include <linux/udp.h>
#include "ebt_dhcpsnoop.h"

/* DHCP packet UDP payload */
struct dhcp {
	__u8 op;		/* Message OP Code */
	__u8 htype;		/* Hardware Address Type */
	__u8 hlen;		/* Harware Address Lenght */
	__u8 hops;		/* Hop Count */
	__be32 xid;		/* Transaction ID */
	__be16 secs;		/* Seconds Elapsed */
	__be16 flags;		/* Bootp Flags */
	__be32 ciaddr;		/* Client IP Address */
	__be32 yiaddr;		/* Your Client IP Address */
	__be32 siaddr;		/* Next Server IP Address */
	__be32 giaddr;		/* Relay Agent IP Address */
	__u8 chaddr[16];	/* Client Hardware Address */
	__u8 sname[64];		/* Server Host Name */
	__u8 file[128];		/* Boot File Name*/
	__u32 cookie;		/* First Option - DHCP*/
	__u8 options[308];	/* Size= 312 - cookie size */
};

struct dhcp_packet {
	struct iphdr ip;
	struct udphdr udp;
	struct dhcp data;
};

#define SERVER_PORT	67
#define CLIENT_PORT	68

#define DHCP_MAGIC	0x63825363

#define BOOTREQUEST	1
#define BOOTREPLY	2

#define OPT_TAG		0
#define OPT_LEN		1

struct dhcp_option_ops {
	const char *name;
	int (*print)(const struct dhcp_option_ops *ops,
		     const struct ebt_dhcp_option *option,
		     const char *prefix_str);
};

int print_ip(const struct dhcp_option_ops *ops,
	     const struct ebt_dhcp_option *option,
	     const char *prefix_str)
{
	pr_err("%s%s = %pI4\n", prefix_str, ops->name, option->data);
	return 0;
}

int print_ip_list(const struct dhcp_option_ops *ops,
		  const struct ebt_dhcp_option *option,
		  const char *prefix_str)
{
	int i;
	for (i = 0; i < option->len; i += 4)
		pr_err("%s%s = %pI4\n", prefix_str, ops->name,
		       &option->data[i]);
	return 0;
}

int print_str(const struct dhcp_option_ops *ops,
	      const struct ebt_dhcp_option *option,
	      const char *prefix_str)
{
	/* option->data contains non null terminated string */
	pr_err("%s%s = %.*s\n", prefix_str, ops->name,
	       option->len, option->data);
	return 0;
}

int print_u8(const struct dhcp_option_ops *ops,
	     const struct ebt_dhcp_option *option,
	     const char *prefix_str)
{
	pr_err("%s%s = %d\n", prefix_str, ops->name, option->data[0]);
	return 0;
}

static char *dhcp_mt_str[] = {
	"UNKNOWN" ,
	"DHCPDISCOVER" ,
	"DHCPOFFER" ,
	"DHCPREQUEST" ,
	"DHCPDECLINE" ,
	"DHCPACK" ,
	"DHCPNAK" ,
	"DHCPRELEASE" ,
	"DHCPINFORM"
};

int print_mt(const struct dhcp_option_ops *ops,
	     const struct ebt_dhcp_option *option,
	     const char *prefix_str)
{
	pr_err("%s%s = %s\n", prefix_str, ops->name,
	       dhcp_mt_str[option->data[0]]);
	return 0;
}

int print_time_sec(const struct dhcp_option_ops *ops,
		   const struct ebt_dhcp_option *option,
		   const char *prefix_str)
{
	pr_err("%s%s = %d Seconds\n", prefix_str, ops->name,
	       (option->data[0] << 24) +
	       (option->data[1] << 16) +
	       (option->data[2] << 8)  +
	       option->data[3]);
	return 0;
}

int print_param_list(const struct dhcp_option_ops *ops,
		     const struct ebt_dhcp_option *option,
		     const char *prefix_str);

#define DHCP_PAD		0x00
#define DHCP_HOSTNAME		0x0c
#define DHCP_MESSAGE_TYPE	0x35
#define DHCP_CLIENT_ID		0x3D
#define DHCP_END		0xFF

/* http://www.iana.org/assignments/bootp-dhcp-parameters */
/* http://www.networksorcery.com/enp/protocol/bootp/options.htm */
struct dhcp_option_ops dhcp_ops_table[DHCP_END+1] __read_mostly = {
	[0]		= { "Pad", NULL },
	[1]		= { "Subnet Mask", print_ip },
	[2]		= { "Time Offset", print_time_sec },
	[3]		= { "Router", print_ip_list },
	[4]		= { "Time Server", print_ip_list },
	[5]		= { "Name Server", print_ip_list },
	[6]		= { "Domain Server", print_ip_list },
	[7]		= { "Log Server", print_ip_list },
	[8]		= { "Quotes Server", print_ip_list },
	[9]		= { "LPR Server", print_ip_list },
	[10]		= { "Impress Server", print_ip_list },
	[11]		= { "RLP Server", print_ip_list },
	[12]		= { "Hostname", print_str },
	[13]		= { "Boot File Size", NULL },
	[14]		= { "Merit Dump File", NULL },
	[15]		= { "Domain Name", print_str },
	[16]		= { "Swap Server", print_ip },
	[17]		= { "Root Path", NULL },
	[18]		= { "Extension Path", NULL },
	[19]		= { "IP Forward On/Off", NULL },
	[20]		= { "Src Routing On/Off", NULL },
	[21]		= { "Policy Filter", NULL },
	[22]		= { "Max DG Assembly", NULL },
	[23]		= { "Default IP TTL", print_u8 },
	[24]		= { "Path MTU Aging Timeout", NULL },
	[25]		= { "Path MTU Plateau Timeout", NULL },
	[26]		= { "MTU Interface", NULL },
	[27]		= { "MTU Subnet", NULL },
	[28]		= { "Broadcast Address", print_ip },
	[29]		= { "Mask Discovery", NULL },
	[30]		= { "Mask Supplier", NULL },
	[31]		= { "Router Discovery", print_u8 },
	[32]		= { "Router Request", print_ip },
	[33]		= { "Static Route", NULL },
	[34]		= { "Trailers", NULL },
	[35]		= { "ARP Timeout", NULL },
	[36]		= { "Ethernet", NULL },
	[37]		= { "Default TCP TTL", NULL },
	[38]		= { "TCP Keepalive Time", NULL },
	[39]		= { "TCP Keepalive Data", NULL },
	[40]		= { "NIS Domain", print_str },
	[41]		= { "NIS Servers", print_ip_list },
	[42]		= { "NTP Servers", print_ip_list },
	[43]		= { "Vendor Specific", NULL },
	[44]		= { "NETBIOS Name Srv", print_ip_list },
	[45]		= { "NETBIOS Dist Srv", print_ip_list },
	[46]		= { "NETBIOS Node Typ", print_u8 },
	[47]		= { "NETBIOS Scope", NULL },
	[48]		= { "X Window Font", NULL },
	[49]		= { "X Window Manager", NULL },
	[50]		= { "Address Request", print_ip },
	[51]		= { "Address Lease Time", print_time_sec },
	[52]		= { "Overload", NULL },
	[53]		= { "DHCP Msg Type", print_mt },
	[54]		= { "DHCP Server Id", print_ip },
	[55]		= { "Parameter List", print_param_list },
	[56]		= { "DHCP Message", NULL },
	[57]		= { "DHCP Max Msg Size", NULL },
	[58]		= { "Renewal Time", NULL },
	[59]		= { "Rebinding Time", NULL },
	[60]		= { "Class Id", NULL },
	[61]		= { "Client Id", print_ip },
	[62]		= { "NetWare/IP Domain", NULL },
	[63]		= { "NetWare/IP Option", NULL },
	[64]		= { "NIS-Domain-Name", NULL },
	[65]		= { "NIS-Server-Addr", NULL },
	[66]		= { "Server-Name", print_str },
	[67]		= { "Bootfile-Name", print_str },
	[68]		= { "Home-Agent-Addrs", NULL },
	[69]		= { "SMTP-Server", NULL },
	[70]		= { "POP3-Server", NULL },
	[71]		= { "NNTP-Server", NULL },
	[72]		= { "WWW-Server", NULL },
	[73]		= { "Finger-Server ", NULL },
	[74]		= { "IRC-Server ", NULL },
	[75]		= { "StreetTalk-Server", NULL },
	[76]		= { "STDA-Server ", NULL },
	[77]		= { "User-Class ", NULL },
	[78]		= { "Directory Agent ", NULL },
	[79]		= { "Service Scope ", NULL },
	[80]		= { "Rapid Commit ", NULL },
	[81]		= { "Client FQDN", NULL },
	[82]		= { "Relay Agent Information", NULL },
	[83]		= { "iSNS", NULL },
	[84]		= { "REMOVED/Unassigned", NULL },
	[85]		= { "NDS Servers", NULL },
	[86]		= { "NDS Tree Name", NULL },
	[87]		= { "NDS Context", NULL },
	[88]		= { "BCMCS Controller Domain Name list", NULL },
	[89]		= { "BCMCS Controller IPv4 address option", NULL },
	[90]		= { "Authentication", NULL },
	[91]		= { "client-last-transaction-time", NULL },
	[92]		= { "associated-ip option", NULL },
	[93]		= { "Client System", NULL },
	[94]		= { "Client NDI", NULL },
	[95]		= { "LDAP", NULL },
	[96]		= { "REMOVED/Unassigned", NULL },
	[97]		= { "UUID/GUID", NULL },
	[98]		= { "User-Auth", NULL },
	[99]		= { "GEOCONF_CIVIC", NULL },
	[100]		= { "PCode", NULL },
	[101]		= { "TCode", NULL },
	[102 ... 107]	= { "REMOVED/Unassigned", NULL },
	[108]		= { "REMOVED/Unassigned", NULL },
	[109]		= { "Unassigned", NULL },
	[110]		= { "REMOVED/Unassigned", NULL },
	[111]		= { "Unassigned", NULL },
	[112]		= { "Netinfo Address", NULL },
	[113]		= { "Netinfo Tag", NULL },
	[114]		= { "URL", NULL },
	[115]		= { "REMOVED/Unassigned", NULL },
	[116]		= { "Auto-Config", NULL },
	[117]		= { "Name Service Search", NULL },
	[118]		= { "Subnet Selection Option", NULL },
	[119]		= { "Domain Search", NULL },
	[120]		= { "SIP Servers DHCP Option", NULL },
	[121]		= { "Classless Static Route Option", NULL },
	[122]		= { "CCC", NULL },
	[123]		= { "GeoConf Option", NULL },
	[124]		= { "V-I Vendor Class", NULL },
	[125]		= { "V-I Vendor-Specific Information", NULL },
	[126]		= { "Removed/Unassigned", NULL },
	[127]		= { "Removed/Unassigned", NULL },
	[128]		= { "TFTP Server IP address", NULL },
	[129]		= { "Call Server IP address", NULL },
	[130]		= { "Discrimination string", NULL },
	[131]		= { "Remote statistics server IP address", NULL },
	[132]		= { "IEEE 802.1Q VLAN ID", NULL },
	[133]		= { "IEEE 802.1D/p Layer 2 Priority", NULL },
	[134]		= { "Diffserv Code Point", NULL },
	[135]		= { "HTTP Proxy for phone-specific apps", NULL },
	[136]		= { "OPTION_PANA_AGENT", NULL },
	[137]		= { "OPTION_V4_LOST", NULL },
	[138]		= { "OPTION_CAPWAP_AC_V4", NULL },
	[139]		= { "OPTION-IPv4_Address-MoS", NULL },
	[140]		= { "OPTION-IPv4_FQDN-MoS", NULL },
	[141]		= { "SIP UA Configuration Service Domains", NULL },
	[142]		= { "OPTION-IPv4_Address-ANDSF", NULL },
	[143]		= { "OPTION-IPv6_Address-ANDSF", NULL },
	[144]		= { "GeoLoc", NULL },
	[145]		= { "FORCERENEW_NONCE_CAPABLE", NULL },
	[146]		= { "RDNSS Selection", NULL },
	[147 ... 149]	= { "Unassigned", NULL },
	[150]		= { "TFTP server address", NULL },
	[150]		= { "Etherboot", NULL },
	[150]		= { "GRUB configuration path name", NULL },
	[151]		= { "status-code", NULL },
	[152]		= { "base-time", NULL },
	[153]		= { "start-time-of-state", NULL },
	[154]		= { "query-start-time", NULL },
	[155]		= { "query-end-time", NULL },
	[156]		= { "dhcp-state", NULL },
	[157]		= { "data-source", NULL },
	[158 ... 174]	= { "Unassigned", NULL },
	[175]		= { "Etherboot", NULL },
	[176]		= { "IP Telephone", NULL },
	[177]		= { "Etherboot or CCC", NULL },
	[178 ... 207]	= { "Unassigned", NULL },
	[208]		= { "PXELINUX Magic", NULL },
	[209]		= { "PXELINUX Config File", NULL },
	[210]		= { "PXELINUX Path Prefix", NULL },
	[211]		= { "PXELINUX Reboot Time", NULL },
	[212]		= { "OPTION_6RD", NULL },
	[213]		= { "OPTION_V4_ACCESS_DOMAIN", NULL },
	[214 ... 219]	= { "Unassigned", NULL },
	[220]		= { "Subnet Allocation Option", NULL },
	[221]		= { "Virtual Subnet Selection Option", NULL },
	[222 ... 223]	= { "Unassigned", NULL },
	[224 ... 254]	= { "Private Use", NULL },
	[255]		= { "End", NULL },
};

int print_param_list(const struct dhcp_option_ops *ops,
		     const struct ebt_dhcp_option *option,
		     const char *prefix_str)
{
	int i;
	pr_err("%s%s\n", prefix_str, ops->name);
	for (i = 0; i < option->len; i++)
		pr_err("%s%03d = %s\n", prefix_str, option->data[i],
		       dhcp_ops_table[option->data[i]].name);
	return 0;
}

static const struct ebt_dhcp_option *
__get_dhcp_option(const __u8 *optionptr, int tag, int length)
{
	int i;
	int done = 0;

	i = 0;
	while (!done) {
		if (i >= length) {
			pr_warn("bogus packet, option %d not found\n",
				tag);
			return NULL;
		}
		if (optionptr[i + OPT_TAG] == tag) {
			if (i + 1 + optionptr[i + OPT_LEN] >= length) {
				pr_warn("bogus packet, option %d: len %d > %d\n",
					tag, i + 1 + optionptr[i + OPT_LEN],
					length);
				return NULL;
			}
			return (struct ebt_dhcp_option *) (optionptr + i);
		}
		switch (optionptr[i + OPT_TAG]) {
		case DHCP_PAD:
			i++;
			break;
		case DHCP_END:
			if (tag == DHCP_END)
				return
				(struct ebt_dhcp_option *) (optionptr + i);
			done = 1;
			break;
		default:
			i += optionptr[OPT_LEN + i] + 2;
		}
	}
	return NULL;
}

static const struct ebt_dhcp_option *
get_dhcp_option(const __u8 *optionptr, int tag)
{
	return __get_dhcp_option(optionptr, tag, 308);
}

static const struct ebt_dhcp_option *
next_dhcp_option(const struct ebt_dhcp_option *option)
{
	__u8 *optionptr;
	if (option->tag == DHCP_END)
		return NULL;
	optionptr = (__u8 *) option;
	optionptr += option->len + 2;
	while (*optionptr == DHCP_PAD)
		optionptr++;
	return (struct ebt_dhcp_option *) optionptr;
}

static __u8
ebt_dhcp_pkt_type(const struct dhcp *data)
{
	const struct ebt_dhcp_option *option;
	option = get_dhcp_option(data->options, DHCP_MESSAGE_TYPE);
	if (option == NULL)
		return 0;
	return option->data[0];
}

static const __u8 *
ebt_dhcp_clientid(const struct dhcp *data)
{
	const struct ebt_dhcp_option *option;
	option = get_dhcp_option(data->options, DHCP_CLIENT_ID);
	if (option == NULL)
		return NULL;
	if (option->data[0] == 1) /* Only handle ethernet type */
		return &option->data[1];
	return NULL;
}

static bool
ebt_dhcpsnoop_mt(const struct dhcp *data,
		 const struct ebt_dhcpsnoop_info *info)
{
	uint8_t i, type;
	if (info->bitmask & EBT_DHCPSNOOP_GIADDR &&
	   NF_INVF(info, EBT_DHCPSNOOP_GIADDR,
		(data->giaddr & info->gimask) != info->giaddr))
		return false;
	if (info->bitmask & EBT_DHCPSNOOP_SIADDR &&
	   NF_INVF(info, EBT_DHCPSNOOP_SIADDR,
		(data->siaddr & info->simask) != info->siaddr))
		return false;
	if (info->bitmask & EBT_DHCPSNOOP_CHADDR) {
		uint8_t verdict;

		if (data->htype != 1) /* Ethernet type */
			return false;
		verdict = 0;
		for (i = 0; i < 6; i++) {
			verdict |= (data->chaddr[i] ^ info->chaddr[i]) &
				       info->chmask[i];
		}
		if (NF_INVF(info, EBT_DHCPSNOOP_CHADDR,
			!ether_addr_equal_masked(data->chaddr, info->chaddr,
						 info->chmask)))
			return false;
	}
	type = ebt_dhcp_pkt_type(data);
	if (type && !((1<<(type-1)) & info->pkt_type_mask))
		return false;
	for (i = 0; i < info->noption; i++) {
		if (info->optlist[i].action ==
		    EBT_DHCPSNOOP_OPTION_ACTION_MATCH) {
			__u8 search_tag;
			const struct ebt_dhcp_option *option;
			search_tag = info->optlist[i].parent_tag;

			if (search_tag) {
				/* Sub tag */
				option = get_dhcp_option(data->options,
							 search_tag);
				if (!option)
					return false;
				search_tag = info->optlist[i].option.tag;
				option = __get_dhcp_option(option->data,
							   search_tag,
							   option->len);
				if (!option)
					return false;
			} else {
				search_tag = info->optlist[i].option.tag;
				option = get_dhcp_option(data->options,
							 search_tag);
				if (!option)
					return false;
			}
			if (option->len != info->optlist[i].option.len)
				return false;

			if (memcmp(option->data,
				   (const __u8 *)info->optlist[i].option.data,
				   option->len))
				return false;
		}
	}
	return true;
}

static int
ebt_dhcp_remove_option(struct sk_buff *skb,
		       struct dhcp_packet *packet, int tag)
{
	const struct ebt_dhcp_option *option;
	const struct ebt_dhcp_option *nextoption;
	__u8 *optionptr, *nextoptionptr, *endpacketptr;
	int copy_len, rem_len;
	option = get_dhcp_option(packet->data.options, tag);
	if (option == NULL)
		return -1;
	nextoption = next_dhcp_option(option);
	if (nextoption == NULL)
		return -1;
	optionptr = (__u8 *) option;
	nextoptionptr = (__u8 *) nextoption;
	endpacketptr = (__u8 *) skb_tail_pointer(skb);
	rem_len = nextoptionptr - optionptr;
	copy_len = endpacketptr - nextoptionptr;
	memcpy((void *)optionptr, nextoptionptr, copy_len);
	skb_trim(skb, skb->len-rem_len);
	return 0;
}

static int
ebt_dhcp_insert_option(const struct ebt_dhcpsnoop_info *info,
		       struct sk_buff *skb,
		       struct dhcp_packet *packet,
		       struct ebt_dhcp_option_info *optinfo)
{
	__u8 *optionptr;
	const struct ebt_dhcp_option *autofill;
	struct ethhdr *eh = (struct ethhdr *)skb_mac_header(skb);
	int copy_len;
	optionptr = (__u8 *) get_dhcp_option(packet->data.options, DHCP_END);
	if (!optionptr) {
		pr_warn("ebt_dhcpsnoop: bogus packet, end option fields not found\n");
		return -1;
	}
	copy_len = optinfo->option.len + 2;
	if (skb_tailroom(skb) < copy_len) {
		/* Expand SKB */
		if (pskb_expand_head(skb, 0, copy_len, GFP_ATOMIC)) {
			pr_warn("ebt_dhcpsnoop: Expand SKB - pskb_expand_head failed\n");
			return -1;
		}
		eh = (struct ethhdr *)skb_mac_header(skb);
		packet = (struct dhcp_packet *)skb->data;
		optionptr = (__u8 *) get_dhcp_option(packet->data.options,
						     DHCP_END);
		if (!optionptr) {
			pr_warn("ebt_dhcpsnoop: bogus packet, end option fields not found after expand\n");
			return -1;
		}
	}
	memcpy(optionptr, &optinfo->option, copy_len);
	if (info->bitmask & EBT_DHCPSNOOP_OPTION_AUTOFILL) {
		if (optinfo->option.tag == 82 /* Relay Agent */) {
			autofill = get_dhcp_option(optionptr+2,
			                           2 /* Remote ID */);
			if( autofill != NULL )
			{
				if (autofill->len == 17)
				{
					snprintf((char *)autofill->data, 18, "%pM", eh->h_source);
				}
			}
		}
	}
	optionptr += copy_len;
	*optionptr = DHCP_END;
	skb_put(skb, copy_len);
	return 0;
}

static void
ebt_dhcpsnoop_dump(struct sk_buff *skb)
{
	struct dhcp_packet *dhcp;
	struct ethhdr *eh = (struct ethhdr *)skb_mac_header(skb);
	const struct ebt_dhcp_option *option;
	dhcp = (struct dhcp_packet *)skb->data;
	pr_err("   DEV: %s\n", skb->dev->name);
	pr_err("   MAC: %pM > %pM\n", eh->h_source, eh->h_dest);
	pr_err("    IP: %pI4 > %pI4\n", &dhcp->ip.saddr, &dhcp->ip.daddr);
	pr_err("    OP: %d\n", dhcp->data.op);
	pr_err(" HTYPE: %d\n", dhcp->data.htype);
	pr_err("  HLEN: %d\n", dhcp->data.hlen);
	pr_err("  HOPS: %d\n", dhcp->data.hops);
	pr_err("   XID: 0x%x\n", ntohl(dhcp->data.xid));
	pr_err("  SECS: %d\n", ntohs(dhcp->data.secs));
	pr_err(" FLAGS: 0x%0x\n", ntohl(dhcp->data.flags));
	pr_err("CIADDR: %pI4\n", &dhcp->data.ciaddr);
	pr_err("YIADDR: %pI4\n", &dhcp->data.yiaddr);
	pr_err("SIADDR: %pI4\n", &dhcp->data.siaddr);
	pr_err("GIADDR: %pI4\n", &dhcp->data.giaddr);
	if (dhcp->data.htype == 1) {
		pr_err("CHADDR: %pM\n", &dhcp->data.chaddr);
	} else {
		print_hex_dump(KERN_ERR, "CHADDR: ", DUMP_PREFIX_NONE,
			       16, 1, &dhcp->data.chaddr, 16, false);
	}
	pr_err(" SNAME: %s\n", dhcp->data.sname);
	pr_err("  FILE: %s\n", dhcp->data.file);
	pr_err("COOKIE: 0x%x\n", dhcp->data.cookie);
	option = get_dhcp_option(dhcp->data.options, DHCP_MESSAGE_TYPE);
	if (option) {
		const struct dhcp_option_ops *ops;
		pr_err("OPTION: t=%d l=%d\n", option->tag, option->len);
		if (option->len > EBT_DHCPSNOOP_OPTION_MAX_LEN)
			return;
		print_hex_dump(KERN_ERR, "      : ",
			       DUMP_PREFIX_NONE,
			       16, 1,
			       option->data, option->len,
			       false);
		ops = &dhcp_ops_table[option->tag];
		if (ops->print)
			ops->print(ops, option, "      : ");
		option = next_dhcp_option(option);
		while ((option != NULL) && (option->tag != DHCP_END)) {
			pr_err("OPTION: t=%d l=%d\n",
			       option->tag, option->len);
			if (option->len)
				print_hex_dump(KERN_ERR, "      : ",
					       DUMP_PREFIX_NONE,
					       16, 1,
					       option->data, option->len,
					       true);
			ops = &dhcp_ops_table[option->tag];
			if (ops->print)
				ops->print(ops, option, "      : ");
			option = next_dhcp_option(option);
		}
	}
	pr_err("----------------------------------------------------------\n");
	print_hex_dump(KERN_ERR, "", DUMP_PREFIX_NONE, 16, 1,
		       eh, skb->len+14, true);
	pr_err("**********************************************************\n");
}

extern int ebt_dhcpsnoop_db_init(void);
extern int ebt_dhcpsnoop_db_fini(void);
extern void ebt_dhcpsnoop_update_db(struct sk_buff *skb);
extern int ebt_dhcpsnoop_db_update(const struct net_device *in,
				   const unsigned char *addr,
				   const __u8 pkt_type,
				   const __be32 *ipaddr,
				   const __u8 *hostname,
				   const __u32 *leasetime);

void ebt_dhcpsnoop_update_db(struct sk_buff *skb)
{
	struct dhcp_packet *dhcp;
	uint8_t type;
	const __be32 *pipaddr = NULL;
	const __u8 *phostname = NULL;
	const __u32 *please = NULL;
	__u8 hostname[255];
	const struct ebt_dhcp_option *option;
	dhcp = (struct dhcp_packet *)skb->data;
	type = ebt_dhcp_pkt_type(&dhcp->data);
	option = get_dhcp_option(dhcp->data.options, DHCP_HOSTNAME);
	if (option) {
		snprintf(hostname, 255, "%.*s", option->len, option->data);
		phostname = hostname;
	}
	if (dhcp->data.htype == 1) {
		if (type == DHCPACK) {
			/* Extract IP address */
			if (dhcp->data.yiaddr)
				pipaddr = &dhcp->data.yiaddr;
			else
				pipaddr = &dhcp->data.ciaddr;
			/* Extract lease time */
			option = get_dhcp_option(dhcp->data.options, 51);
			if (option)
				please = (__u32 *) option->data;
		} else if (type == DHCPINFORM) {
			/* Extract IP address */
			pipaddr = &dhcp->data.ciaddr;
		}
		ebt_dhcpsnoop_db_update(skb->dev, dhcp->data.chaddr,
					type, pipaddr, phostname, please);
	}
}

static unsigned int
ebt_dhcpsnoop_tg(struct sk_buff *skb, const struct xt_action_param *par)
{
	const struct ebt_dhcpsnoop_info *info = par->targinfo;
	struct dhcp_packet *dhcp;
	struct ethhdr *eh = (struct ethhdr *)skb_mac_header(skb);
	const __u8 *clientid = NULL;
	int i, pkt_upd = -1;
	unsigned int skb_len;

	dhcp = (struct dhcp_packet *)skb->data;
	if (dhcp == NULL)
		return EBT_CONTINUE;

	if ((dhcp->udp.dest == htons(SERVER_PORT)) &&
	    (dhcp->udp.source == htons(CLIENT_PORT))) {
		if (dhcp->data.op != BOOTREQUEST)
			return EBT_CONTINUE;
	} else if ((dhcp->udp.dest == htons(CLIENT_PORT)) &&
	    (dhcp->udp.source == htons(SERVER_PORT))) {
		if (dhcp->data.op != BOOTREPLY)
			return EBT_CONTINUE;
	} else {
		return EBT_CONTINUE;
	}

	if (dhcp->data.cookie != htonl(0x63825363))
		return EBT_CONTINUE;

	if (dhcp->data.op == BOOTREQUEST) {
		clientid = ebt_dhcp_clientid(&dhcp->data);
		if (clientid && info->clientid_init == 0) {
			memcpy((void *)info->clientid, clientid, ETH_ALEN);
			memset((void *)&info->clientid_init, 1, 1);
		}
	}

	if (!ebt_dhcpsnoop_mt(&dhcp->data, info))
		return EBT_CONTINUE;

	if (info->bitmask & EBT_DHCPSNOOP_DUMP)
		ebt_dhcpsnoop_dump(skb);

	if (info->bitmask & EBT_DHCPSNOOP_CLIENTDB)
		ebt_dhcpsnoop_update_db(skb);

	/* Execute verify rules */
	if ((info->bitmask & EBT_DHCPSNOOP_CHADDR_VERIFY) &&
	    (dhcp->data.op == BOOTREQUEST)) {
		if (memcmp(dhcp->data.chaddr,
			   eh->h_source, ETH_ALEN))
			return EBT_DROP;
	}
	if ((info->bitmask & EBT_DHCPSNOOP_CLIENTID_VERIFY) &&
	    (dhcp->data.op == BOOTREQUEST) &&
	    clientid) {
		if (memcmp(clientid,
			   info->clientid, ETH_ALEN))
			return EBT_DROP;
	}
	if ((info->bitmask & EBT_DHCPSNOOP_B2U) &&
	    (dhcp->data.op == BOOTREPLY) &&
	    is_broadcast_ether_addr(eh->h_dest)) {
		clientid = ebt_dhcp_clientid(&dhcp->data);
		if (clientid)
			memcpy(eh->h_dest, clientid, ETH_ALEN);
		else
			memcpy(eh->h_dest, dhcp->data.chaddr, ETH_ALEN);
	}
	if (info->target == EBT_DROP)
		return EBT_DROP;

	skb_len = skb->len;

	/* Execute option replace/remove/insert rules */
	for (i = 0; i < info->noption; i++) {
		struct ebt_dhcp_option_info *optinfo;
		optinfo = (struct ebt_dhcp_option_info *)&(info->optlist[i]);
		if (optinfo->action ==
		    EBT_DHCPSNOOP_OPTION_ACTION_REMOVE) {
			pkt_upd = ebt_dhcp_remove_option(skb, dhcp,
							 optinfo->option.tag);
		} else if (optinfo->action ==
			   EBT_DHCPSNOOP_OPTION_ACTION_REPLACE) {
			pkt_upd = ebt_dhcp_remove_option(skb, dhcp,
							 optinfo->option.tag);
			pkt_upd = ebt_dhcp_insert_option(info, skb, dhcp,
							 optinfo);
		} else if (optinfo->action ==
			   EBT_DHCPSNOOP_OPTION_ACTION_INSERT) {
			pkt_upd = ebt_dhcp_insert_option(info, skb, dhcp,
							 optinfo);
		}
		/* Insert may reallocated data so reinitialize dhcp pointer */
		dhcp = (struct dhcp_packet *)skb->data;
	}

	if (pkt_upd == 0) {
		__wsum csum;
		__sum16	check;
		__be16 old_len, new_len;

		if (skb_len > skb->len) {
			old_len = ntohs(dhcp->ip.tot_len);
			new_len = old_len - (skb_len - skb->len);
			/* Update IP Checksum */
			csum_replace2(&dhcp->ip.check,
				      htons(old_len),
				      htons(new_len));
			dhcp->ip.tot_len = htons(new_len);
			old_len = ntohs(dhcp->udp.len);
			new_len = old_len - (skb_len - skb->len);
			dhcp->udp.len = htons(new_len);
			/* Recompute UDP Cheksum */
			dhcp->udp.check = 0;
			csum = csum_partial(&dhcp->udp, new_len, 0);
			check = csum_tcpudp_magic(dhcp->ip.saddr,
						  dhcp->ip.daddr,
						  new_len,
						  IPPROTO_UDP, csum);
			dhcp->udp.check = check;
		} else {
			old_len = ntohs(dhcp->ip.tot_len);
			new_len = old_len + (skb->len - skb_len);
			/* Update IP Checksum */
			csum_replace2(&dhcp->ip.check,
				      htons(old_len),
				      htons(new_len));
			dhcp->ip.tot_len = htons(new_len);
			old_len = ntohs(dhcp->udp.len);
			new_len = old_len + (skb->len - skb_len);
			dhcp->udp.len = htons(new_len);
			/* Recompute UDP Cheksum */
			dhcp->udp.check = 0;
			csum = csum_partial(&dhcp->udp, new_len, 0);
			check = csum_tcpudp_magic(dhcp->ip.saddr,
						  dhcp->ip.daddr,
						  new_len,
						  IPPROTO_UDP, csum);
			dhcp->udp.check = check;
		}
	}

	return info->target;
}

static int ebt_dhcpsnoop_tg_check(const struct xt_tgchk_param *par)
{
	const struct ebt_dhcpsnoop_info *info = par->targinfo;
	const struct ebt_entry *e = par->entryinfo;

	if (BASE_CHAIN && info->target == EBT_RETURN)
		return -EINVAL;
	if (e->ethproto != htons(ETH_P_IP))
		return -EINVAL;
	if (info->bitmask & EBT_DHCPSNOOP_CLIENTDB)
		ebt_dhcpsnoop_db_init();
	return 0;
}

void ebt_dhcpsnoop_tg_destroy(const struct xt_tgdtor_param *par)
{
	const struct ebt_dhcpsnoop_info *info = par->targinfo;
	if (info->bitmask & EBT_DHCPSNOOP_CLIENTDB)
		ebt_dhcpsnoop_db_fini();
}

static struct xt_target ebt_dhcpsnoop_tg_reg __read_mostly = {
	.name		= "dhcpsnoop",
	.revision	= 0,
	.family		= NFPROTO_BRIDGE,
	.hooks		= (1 << NF_BR_NUMHOOKS) | (1 << NF_BR_PRE_ROUTING) |
			  (1 << NF_BR_BROUTING) | (1 << NF_BR_POST_ROUTING),
	.target		= ebt_dhcpsnoop_tg,
	.checkentry	= ebt_dhcpsnoop_tg_check,
	.destroy	= ebt_dhcpsnoop_tg_destroy,
	.targetsize	= sizeof(struct ebt_dhcpsnoop_info),
	.me		= THIS_MODULE,
};

static int __init ebt_dhcpsnoop_init(void)
{
	return xt_register_target(&ebt_dhcpsnoop_tg_reg);
}

static void __exit ebt_dhcpsnoop_fini(void)
{
	xt_unregister_target(&ebt_dhcpsnoop_tg_reg);
}

module_init(ebt_dhcpsnoop_init);
module_exit(ebt_dhcpsnoop_fini);
MODULE_DESCRIPTION("Ebtables: DHCP Snoop target");
MODULE_LICENSE("GPL");
