/****************************************************************************
 *
 * Broadcom Proprietary and Confidential.
 * (c) 2015-2019 Broadcom. All rights reserved.
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Paul Tweedale <paul.tweedale@broadcom.com>
 ****************************************************************************/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/times.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <net/netfilter/nf_conntrack.h>
#include <net/ipv6.h>
#include "flowmgr.h"
#include "flowmgr_fap.h"
#include "flowmgr_fap_ops.h"
#include "flowmgr_tunnel.h"
#include "flowmgr_map.h"
#include "dqnet_priv.h"
#include <net/netfilter/nf_conntrack_expect.h>
#include <net/netfilter/nf_conntrack_helper.h>
#include <net/netfilter/nf_conntrack_acct.h>
#include "mso_feat.h"

DEFINE_SPINLOCK(flowmgr_map_lock);

#define FLOWMGR_MAP_ENABLE_IPV6_FRAGMENT	1
#define FLOWMGR_MAP_DEFAULT_DISABLE_LEVEL	3

#define FLOWMGR_SINGLE_DESC	"(single)"
#define FLOWMGR_DUPLEX_12_DESC	"(duplex 1/2)"
#define FLOWMGR_DUPLEX_22_DESC	"(duplex 2/2)"

static struct mso_feat *feat;

static int mso_feat_mapt_init(void)
{
	get_mso_feat_fptr get_mso_feat;

	if (!feat) {
		get_mso_feat = get_mso_feat_symbol();
		if (!get_mso_feat)
			return -1;

		feat = get_mso_feat(mf_mapt);
		if (!feat)
			return -1;
	}
	return 0;
}

static int mso_feat_mapt_deinit(void)
{
	if (feat) {
		put_mso_feat_symbol();
		feat = NULL;
	}
	return 0;
}

int flowmgr_map_ipv6_frag_check(struct ipv6hdr *ip6h,
				__u8 **payload, int *protocol)
{
#ifdef FLOWMGR_MAP_ENABLE_IPV6_FRAGMENT
	struct frag_hdr *fragh;
	if (*protocol == IPPROTO_FRAGMENT) {
		fragh = (struct frag_hdr *)((__u8 *)ip6h + sizeof(struct ipv6hdr));
		*protocol = (int)fragh->nexthdr;
		if (payload != NULL)
			*payload += sizeof(struct frag_hdr);
	}
#endif
	return 0;
}

int flowmgr_mapt_domain_cmp(struct nf_conn *ct, void *match)
{
	struct nf_conn_offload *ct_offload;
	struct flow_params *params;
	int ct_domain, domain;

	if (ct == NULL)
		return 0;

	if (match == NULL)
		return 0;

	domain = *((int *)match);

	ct_offload = nf_conn_offload_find(ct);
	if (!ct_offload)
		return 0;

	if (!ct_offload_orig.map)
		goto check_repl;

	params = (struct flow_params *)ct_offload_orig.map;
	if (params->type == ft_ipv4)
		ct_domain = (int)params->ipv4.mapt.domain_index;
	else
		ct_domain = (int)params->ipv6.mapt.domain_index;

	if (ct_domain == domain)
		return 1;

check_repl:
	if (!ct_offload_repl.map)
		return 0;

	params = (struct flow_params *)ct_offload_repl.map;
	if (params->type == ft_ipv4)
		ct_domain = (int)params->ipv4.mapt.domain_index;
	else
		ct_domain = (int)params->ipv6.mapt.domain_index;

	if (ct_domain == domain)
		return 1;

	// No match found
	return 0;
}


static int map_unmap_help(struct sk_buff *skb, unsigned int protoff,
				struct nf_conn *ct, enum ip_conntrack_info ctinfo)
{
	return NF_ACCEPT;
}

static const struct nf_conntrack_expect_policy map_exp_policy = {
	.max_expected   = 1,
	.timeout        = 10,
};

static struct nf_conntrack_helper nf_conntrack_helper_map_udp __read_mostly = {
	.name			= "MAP UDP",
	.me			= THIS_MODULE,
	.data_len		= 0,
	.tuple.src.l3num	= AF_INET,
	.tuple.dst.protonum	= IPPROTO_UDP,
	.help			= map_unmap_help,
	.expect_policy		= &map_exp_policy,
};

static struct nf_conntrack_helper nf_conntrack_helper_map_tcp __read_mostly = {
	.name			= "MAP TCP",
	.me			= THIS_MODULE,
	.data_len		= 0,
	.tuple.src.l3num	= AF_INET,
	.tuple.dst.protonum	= IPPROTO_TCP,
	.help			= map_unmap_help,
	.expect_policy		= &map_exp_policy,
};

static struct nf_conntrack_helper nf_conntrack_helper_unmap_udp __read_mostly = {
	.name			= "UNMAP UDP",
	.me			= THIS_MODULE,
	.data_len		= 0,
	.tuple.src.l3num	= AF_INET6,
	.tuple.dst.protonum	= IPPROTO_UDP,
	.help			= map_unmap_help,
	.expect_policy		= &map_exp_policy,
};

static struct nf_conntrack_helper nf_conntrack_helper_unmap_tcp __read_mostly = {
	.name			= "UNMAP TCP",
	.me			= THIS_MODULE,
	.data_len		= 0,
	.tuple.src.l3num	= AF_INET6,
	.tuple.dst.protonum	= IPPROTO_TCP,
	.help			= map_unmap_help,
	.expect_policy		= &map_exp_policy,
};

static
struct nf_conntrack_helper *determine_helper(u_int8_t pf, int protocol)
{
	struct nf_conntrack_helper *helper = NULL;

	if (protocol == IPPROTO_TCP) {
		if (pf == PF_INET)
			helper = &nf_conntrack_helper_unmap_tcp;
		else
			helper = &nf_conntrack_helper_map_tcp;
	} else if (protocol == IPPROTO_UDP) {
	if (pf == PF_INET)
		helper = &nf_conntrack_helper_unmap_udp;
	else
		helper = &nf_conntrack_helper_map_udp;
	} else {
		FLOWMGR_MAP_RETURN_NULL
	}

	return helper;
}

static
struct nf_conn *create_conntrack(struct sk_buff *skb,
				struct net_device *dev,
				u_int8_t pf,
				int protocol,
				bool add_helper,
				bool confirm)
{
	int ret;
	enum ip_conntrack_info ctinfo;
	struct nf_conn *ct;
	struct nf_conntrack_helper *helper;
	struct nf_conn_help *help;
	struct nf_hook_state state;

	state.pf = pf;
	state.hook = NF_INET_PRE_ROUTING;
	state.net = dev_net(dev);

	rcu_read_lock();
	ret = nf_conntrack_in(skb, &state);
	rcu_read_unlock();
	if (ret != NF_ACCEPT)
		FLOWMGR_MAP_GOTO(error)

	ct = nf_ct_get(skb, &ctinfo);
	if (!ct)
		FLOWMGR_MAP_GOTO(error)

	if (!nf_ct_is_confirmed(ct) && add_helper) {
		// Add helper, needed to make master association
		helper = determine_helper(pf, protocol);
		if (helper == NULL)
			FLOWMGR_MAP_GOTO(error)

		help = nf_ct_helper_ext_add(ct, GFP_KERNEL);
		if (help == NULL)
			FLOWMGR_MAP_GOTO(error)
		if (help)
			rcu_assign_pointer(help->helper, helper);
	}

	if (confirm) {
		rcu_read_lock();
		ret = nf_conntrack_confirm(skb);
		rcu_read_unlock();
		if (ret != NF_ACCEPT)
			FLOWMGR_MAP_GOTO(error)
	}

	ct = nf_ct_get(skb, &ctinfo);
	if (!ct)
		FLOWMGR_MAP_GOTO(error)

	return ct;

error:
	return NULL;
}

static
int set_skb_headers(struct sk_buff *skb)
{
	int offset;
	struct iphdr *ip4h;
	struct ipv6hdr *ip6h;
	uint8_t nexthdr;

	if (!(skb->protocol == __constant_ntohs(ETH_P_IP)) &&
	    !(skb->protocol == __constant_ntohs(ETH_P_IPV6)))
		return -1;

	// set the network header
	skb_reset_network_header(skb);
	// set the mac header
	if (skb->mac_len)
		skb_set_mac_header(skb, skb->network_header - skb->mac_len);

	// set the transport header
	if (skb->protocol == __constant_ntohs(ETH_P_IP)) {
		ip4h = ip_hdr(skb);
		offset = (ip4h->ihl << 2);
	} else if (skb->protocol == __constant_ntohs(ETH_P_IPV6)) {
		__be16 frag_off;
		ip6h = ipv6_hdr(skb);
		nexthdr = ip6h->nexthdr;
		offset = sizeof(*ip6h);
		if (ipv6_ext_hdr(nexthdr)) {
			offset = ipv6_skip_exthdr(skb, offset, &nexthdr, &frag_off);
			if (offset == -1)
				return -1;
		}
	}
	skb_set_transport_header(skb, offset);
	return 0;
}

static
int set_skb_dev_in_iif_group(struct sk_buff *skb,
			     struct sk_buff *newskb,
			     int group)
{
	if (skb->dev == NULL)
		FLOWMGR_MAP_RETURN(-1)

	// Valid for function to be called without newskb
	if (newskb == NULL)
		return 0;

	if (!newskb->dev_in)
		newskb->dev_in = skb->dev_in;
	if (!newskb->skb_iif)
		newskb->skb_iif = skb->skb_iif;

	if (newskb->dev_in == NULL)
		FLOWMGR_MAP_RETURN(-1)

	if (newskb->dev_in->group != group) {
		// we should only update the group if it is not previously set
		if (newskb->dev_in->group == 0)
			newskb->dev_in->group = group;
		//else
		//	FLOWMGR_MAP_RETURN(-1)
	}

	return 0;
}

static
int flowmgr_expect_init(struct nf_conn *ct,
			u_int8_t family,
			const union nf_inet_addr *saddr,
			const union nf_inet_addr *daddr,
			u_int8_t protocol, __u8 *payload)
{
	struct tcphdr *tcph = NULL;
	struct udphdr *udph = NULL;
	__be16 src, dst;
	struct nf_conntrack_expect *exp;
	struct nf_conntrack_helper *helper;
	int ret = -1;

	if (protocol == IPPROTO_TCP) {
		tcph = (struct tcphdr *)payload;
		src = tcph->source;
		dst = tcph->dest;
	} else if (protocol == IPPROTO_UDP) {
		udph = (struct udphdr *)payload;
		src = udph->source;
		dst = udph->dest;
	} else {
		FLOWMGR_MAP_RETURN(-1)
	}

	exp = nf_ct_expect_alloc(ct);
	if (exp == NULL)
		FLOWMGR_MAP_RETURN(-1)

	nf_ct_expect_init(exp, NF_CT_EXPECT_CLASS_DEFAULT,
			  family,
			  saddr, daddr,
			  protocol, &src, &dst);

	// Helper needs to be set for nf_ct_expect_related,
	// so that connection in reverse direction can setup
	// its master link.
	helper = determine_helper(family, protocol);
	if (helper == NULL)
		FLOWMGR_MAP_RETURN(-1)
	exp->helper = helper;

	nf_ct_dump_tuple(&exp->tuple);

	rcu_read_lock();
	ret = nf_ct_expect_related(exp, 0);
	rcu_read_unlock();
	if (ret != 0)
		FLOWMGR_MAP_RETURN(ret)

	nf_ct_expect_put(exp);

	return 0;
}

static
int flowmgr_mapt_add_4to6(struct sk_buff *v4skb,
			  struct sk_buff *v6skb,
			  int index,
			  __u32 lan_addr, __u16 lan_port)
{
	struct iphdr *ip4h;
	struct ipv6hdr *ip6h;
	__u8 *ip4p, *ip6p;
	int protocol;
	struct tcphdr *tcph = NULL;
	struct udphdr *udph = NULL;
	struct flow_params *params;
	struct flow_ipv4_params *ipv4_params;
	struct nf_conn *ct4;
	struct nf_conn *ct6;
	struct nf_conn_offload *ct_offload;
	struct offload_info *ct_offload_info;
	struct flowmgr_map_offload *map_offload;
	char *directions;
	int ret = -1;

	ip4h = ip_hdr(v4skb);
	ip4p = (__u8 *)(ip4h) + (ip4h->ihl << 2);

	// By pass multicast packet
	if (ipv4_is_multicast(ip4h->daddr) ||
	    ipv4_is_lbcast(ip4h->daddr)    ||
	    ipv4_is_loopback(ip4h->daddr))
		return 0;

	// Leave to protcol stack
	if (ip4h->ttl <= 1)
		return 0;

	// Get protocol
	protocol = (int)ip4h->protocol;

	// Check for connection tracking on original skb
	ct4 = (struct nf_conn *)skb_nfct(v4skb);
	if (!ct4) {
		ct4 = create_conntrack(v4skb, v4skb->dev,
				       PF_INET, protocol,
				       true, true);
		if (!ct4)
			FLOWMGR_MAP_RETURN(-1)
	}

	// Valid for function to be called with no v6skb, just exit here
	if (v6skb == NULL)
		return 0;

	ip6h = ipv6_hdr(v6skb);
	ip6p = (__u8 *)(ip6h) + sizeof(struct ipv6hdr);

	// Check protocol, and no IPv6 extension headers
	if ((ip6h->nexthdr != IPPROTO_UDP) &&
	    (ip6h->nexthdr != IPPROTO_TCP))
		return 0;

	// Check protocols are consistant
	if (protocol != (int)ip6h->nexthdr) {
		pr_info("%s %d ip4 protocol=%d ip6 protocol=%d\n",
			__func__, __LINE__, protocol, ip6h->nexthdr);
		FLOWMGR_MAP_RETURN(-1)
	}

	// Check/Create connection tracking for v6skb
	ct6 = (struct nf_conn *)skb_nfct(v6skb);
	if (!ct6) {
		// ct4 may not have set its ext helper, flowmgr_expect_init()
		// will failed if it does not have one
		if (!nfct_help(ct4)) {
			struct nf_conntrack_helper *helper;
			struct nf_conn_help *help;
			bool tmp_undo_ct_confirm = nf_ct_is_confirmed(ct4);

			// Add helper, needed to make master association
			helper = determine_helper(AF_INET, protocol);
			if (helper == NULL)
				FLOWMGR_MAP_RETURN(-1);

			// Workaround: temporarily unconfirm the CT, as a confirmed CT
			// will spawn a kernel warning when adding a CT helper.
			if (tmp_undo_ct_confirm)
				ct4->status &= ~IPS_CONFIRMED;
			help = nf_ct_helper_ext_add(ct4, GFP_KERNEL);
			if (tmp_undo_ct_confirm)
				ct4->status |= IPS_CONFIRMED;
			if (help == NULL)
				FLOWMGR_MAP_RETURN(-1);
			if (help)
				rcu_assign_pointer(help->helper, helper);
		}

		ret = flowmgr_expect_init(ct4, AF_INET6,
			(union nf_inet_addr *)&ip6h->saddr.s6_addr,
			(union nf_inet_addr *)&ip6h->daddr.s6_addr,
			protocol, ip6p);
		if (ret != 0)
			FLOWMGR_MAP_RETURN(-1)

		ct6 = create_conntrack(v6skb, v4skb->dev,
				       PF_INET6, ip6h->nexthdr,
				       false, true);
		if (!ct6)
			FLOWMGR_MAP_RETURN(-1)
	}

	// Get offload
	ct_offload = nf_conn_offload_find(ct6);
	if (!ct_offload)
		FLOWMGR_MAP_RETURN(-1)

	// Establish offload info direction
	ct_offload_info = &ct_offload->info[CTINFO2DIR(nfctinfo(v6skb))];
	if (ct_offload_info == NULL)
		FLOWMGR_MAP_RETURN(-1)

	spin_lock(&flowmgr_map_lock);
	if (!ct_offload_info->map) {
		ct_offload_info->map = kzalloc(sizeof(struct flowmgr_map_offload),
						GFP_KERNEL);
		if (!ct_offload_info->map) {
			spin_unlock(&flowmgr_map_lock);
			FLOWMGR_MAP_RETURN(-1)
		}

		// Ensure flow id invalid until promotion
		ct_offload_info->flow_id = -1;

		map_offload = (struct flowmgr_map_offload *)ct_offload_info->map;
		map_offload->master_ct = ct4;
	}
	spin_unlock(&flowmgr_map_lock);

	// Set params in offload info map
	map_offload = (struct flowmgr_map_offload *)ct_offload_info->map;
	params = (struct flow_params *)&map_offload->params;
	ipv4_params = &(params->ipv4);

	// Check if flow information already set
	if ((params->type == ft_ipv4) &&
	    ((ipv4_params->type == ft_ipv4_mapt_map) ||
		 (ipv4_params->type == ft_ipv4_mapt_gre2map)))
		FLOWMGR_MAP_RETURN(0)

	// Match fields
	ipv4_params->src_ip = lan_addr ? lan_addr : ip4h->saddr;
	ipv4_params->dst_ip = ip4h->daddr;

	if (protocol == IPPROTO_TCP) {
		tcph = (struct tcphdr *)ip4p;
		ipv4_params->src_port = lan_port ? lan_port : tcph->source;
		ipv4_params->dst_port = tcph->dest;

		// Replacement fields
		tcph = (struct tcphdr *)ip6p;
		ipv4_params->replacement_port = tcph->source;

	} else if (protocol == IPPROTO_UDP) {
		udph = (struct udphdr *)ip4p;
		ipv4_params->src_port = lan_port ? lan_port : udph->source;
		ipv4_params->dst_port = udph->dest;

		// Replacement fields
		udph = (struct udphdr *)ip6p;
		ipv4_params->replacement_port = udph->source;

	} else {
		pr_err("%s %d protocol=%d invalid at this point\n",
			__func__, __LINE__, protocol);
		FLOWMGR_MAP_RETURN(-1)
	}
	// Replacement fields continued
	ipv4_params->replacement_ip = ip4h->saddr;

	ipv4_params->ip_prot = (__u8)protocol;

	// Final mac replacement fields not available yet,
	// zero for now
	memset(&(ipv4_params->replacement_mac_src),
		0, ETH_ALEN);
	memset(&(ipv4_params->replacement_mac_dst),
		0, ETH_ALEN);

	memcpy(&(ipv4_params->mapt.ipv6_src_ip), ip6h->saddr.s6_addr, 16);
	memcpy(&(ipv4_params->mapt.ipv6_dst_ip), ip6h->daddr.s6_addr, 16);

	// Set domain and valid flow type
	ipv4_params->mapt.domain_index = (__be16)index;
	ipv4_params->type = ft_ipv4_mapt_map;

	// Complete parameter setup in offload info
	// Note: tx interface will be set later, once known by transmit
	params->type = ft_ipv4;

	// Dump direction params for debug
	flowmgr_ctdebug_check(ct4, ct_offload, IP_CT_DIR_ORIGINAL);
	flowmgr_ctdebug_check(ct6, ct_offload, IP_CT_DIR_ORIGINAL);
	if (ct_offload_info->debug) {
		pr_alert("FLOWMGR: MAP-T v4v6 saved for domain (%d) " \
			"ct4: %px:%d | ct6: %px:%d master: %px\n",
			index, ct4, nfctinfo(v4skb),
			ct6, nfctinfo(v6skb),
			master_ct(ct6));
		directions = flowmgr_get_directions(2);
		flowmgr_dump_ipv4(directions,
				  ct6->mark, ct_offload_info,
				  ipv4_params);
	}

	return 0;
}

static
int flowmgr_mapt_add_6to4(struct sk_buff *v6skb,
			  struct sk_buff *v4skb,
			  int index,
			  __u32 lan_addr, __u16 lan_port)
{
	struct iphdr *ip4h;
	struct ipv6hdr *ip6h;
	__u8 *ip4p = NULL, *ip6p = NULL;
	int protocol;
	struct tcphdr *tcph = NULL;
	struct udphdr *udph = NULL;
	struct flow_params *params;
	struct flow_ipv6_params *ipv6_params;
	struct nf_conn *ct4;
	struct nf_conn *ct6;
	struct nf_conn_offload *ct_offload;
	struct offload_info *ct_offload_info;
	struct flowmgr_map_offload *map_offload;
	char *directions;

	ip6h = ipv6_hdr(v6skb);
	ip6p = (__u8 *)(ip6h) + sizeof(struct ipv6hdr);

	// By pass ipv6 multicast packet
	if (ip6h->daddr.s6_addr[0] == 0xff)
		return 0;

	// Leave to the protocol stack
	if (ip6h->hop_limit <= 1)
		return 0;

	// Get protocol
	protocol = (int)ip6h->nexthdr;
	flowmgr_map_ipv6_frag_check(ip6h, &ip6p, &protocol);

	// Check for connection tracking on original skb
	ct6 = (struct nf_conn *)skb_nfct(v6skb);
	if (!ct6) {
		ct6 = create_conntrack(v6skb, v6skb->dev,
				       PF_INET6, protocol,
				       true, true);
		if (!ct6)
			FLOWMGR_MAP_RETURN(-1)
	}

	// Valid for function to be called with no v4skb, just exit here
	if (v4skb == NULL)
		return 0;

	ip4h = ip_hdr(v4skb);
	ip4p = (__u8 *)(ip4h) + (ip4h->ihl << 2);

	// Check protocol
	if ((ip4h->protocol != IPPROTO_UDP) &&
	    (ip4h->protocol != IPPROTO_TCP))
		return 0;

	// Check protocols are consistant
	if (protocol != (int)ip4h->protocol) {
		pr_debug("%s %d ip6 protocol=%d ip4 protocol=%d\n",
			__func__, __LINE__, protocol, ip4h->protocol);
		FLOWMGR_MAP_RETURN(-1)
	}

	// Check/Create connection tracking for v4skb
	ct4 = (struct nf_conn *)skb_nfct(v4skb);
	if (!ct4) {
		/*
		// Removing the IPv4 expected CT.
		// There is no need for creating an expected CT for the IPv4 flow in the 6to4 direction,
		// as all mapt flows starts from 4to6, which has created the CT for the IPv4 flow.
		int ret = -1;
		ret = flowmgr_expect_init(ct6, AF_INET,
			(union nf_inet_addr *)&ip4h->saddr,
			(union nf_inet_addr *)&ip4h->daddr,
			protocol, ip4p);
		if (ret != 0)
			FLOWMGR_MAP_RETURN(-1)
		*/

		ct4 = create_conntrack(v4skb, v6skb->dev,
				       PF_INET, protocol,
				       false, true);
		if (!ct4)
			FLOWMGR_MAP_RETURN(-1)
	}

	// Get offload
	ct_offload = nf_conn_offload_find(ct4);
	if (!ct_offload)
		FLOWMGR_MAP_RETURN(-1)

	// Establish offload info direction
	ct_offload_info = &ct_offload->info[CTINFO2DIR(nfctinfo(v4skb))];
	if (ct_offload_info == NULL)
		FLOWMGR_MAP_RETURN(-1)

	spin_lock(&flowmgr_map_lock);
	if (!ct_offload_info->map) {
		ct_offload_info->map = kzalloc(sizeof(struct flowmgr_map_offload),
					       GFP_KERNEL);
		if (!ct_offload_info->map) {
			spin_unlock(&flowmgr_map_lock);
			FLOWMGR_MAP_RETURN(-1)
		}

		// Ensure flow id invalid until promotion
		ct_offload_info->flow_id = -1;

		map_offload = (struct flowmgr_map_offload *)ct_offload_info->map;
		map_offload->master_ct = ct6;
	}
	spin_unlock(&flowmgr_map_lock);

	// Set params in offload info map
	map_offload = (struct flowmgr_map_offload *)ct_offload_info->map;
	params = (struct flow_params *)&map_offload->params;
	ipv6_params = &(params->ipv6);

	// Check if flow information already set
	if ((params->type == ft_ipv6) &&
	    ((ipv6_params->type == ft_ipv6_mapt_unmap) ||
		 (ipv6_params->type == ft_ipv6_mapt_unmap2gre)))
		FLOWMGR_MAP_RETURN(0)

	// Match fields
	memcpy(&(ipv6_params->src_ip),
		ip6h->saddr.s6_addr, 16);
	memcpy(&(ipv6_params->dst_ip),
		ip6h->daddr.s6_addr, 16);

	if (protocol == IPPROTO_TCP) {
		tcph = (struct tcphdr *)ip6p;
		ipv6_params->dst_port = tcph->dest;
		ipv6_params->src_port = tcph->source;

		// Replacement field
		tcph = (struct tcphdr *)ip4p;
		ipv6_params->mapt.dst_port = lan_port ? lan_port : tcph->dest;

	} else if (protocol == IPPROTO_UDP) {
		udph = (struct udphdr *)ip6p;
		ipv6_params->src_port = udph->source;
		ipv6_params->dst_port = udph->dest;

		// Replacement field
		udph = (struct udphdr *)ip4p;
		ipv6_params->mapt.dst_port = lan_port ? lan_port : udph->dest;
	} else {
		pr_err("%s %d protocol=%d invalid at this point\n",
			 __func__, __LINE__, protocol);
		FLOWMGR_MAP_RETURN(-1)
	}
	ipv6_params->ip_prot = (__u8)protocol;

	// Final mac replacement fields not available yet,
	// zero for now
	memset(&(ipv6_params->replacement_mac_src),
	       0, ETH_ALEN);
	memset(&(ipv6_params->replacement_mac_dst),
	       0, ETH_ALEN);

	ipv6_params->mapt.src_ip = ip4h->saddr;
	ipv6_params->mapt.dst_ip = lan_addr ? lan_addr : ip4h->daddr;

	// Set domain and valid flow type
	ipv6_params->mapt.domain_index = (__be16)index;
	ipv6_params->type = ft_ipv6_mapt_unmap;

	// Complete parameter setup in offload info
	// Note: tx interface will be set later, once known by transmit
	params->type = ft_ipv6;

	// Dump REPL direction params for debug
	flowmgr_ctdebug_check(ct6, ct_offload, IP_CT_DIR_REPLY);
	flowmgr_ctdebug_check(ct4, ct_offload, IP_CT_DIR_REPLY);
	if (ct_offload_info->debug) {
		pr_alert("FLOWMGR: MAP-T v6v4 saved for domain (%d) " \
			"ct6: %px:%d | ct4: %px:%d master: %px\n",
			index, ct6, nfctinfo(v6skb),
			ct4, nfctinfo(v4skb),
			master_ct(ct4));
		directions = flowmgr_get_directions(1);
		flowmgr_dump_ipv6(directions,
				  ct4->mark, ct_offload_info,
				  ipv6_params);
	}

	return 0;
}

static
int inner_map_flush(int domain)
{
	struct flowmgr_tdb_entry *db = NULL;

	// Domain range
	if ((domain < 0) || (domain >= FLOWMGR_MAPT_MAX_DOMAINS))
		return -1;

	// Flush flows
	flowmgr_conntrack_clean_by_mapt_domain(domain);

	// Unconfigure FAP setting
	if (flowmgr_is_feat_map_enable() && flowmgr.disable_mapt_accel < 3) {
		int ret;
		struct flow_mapt_params params;
		memset(&params, 0, sizeof(params));
		params.domain_index = domain;

		ret = flowmgr_config_mapt(&params, 0);
		if (ret != 0)
			FLOWMGR_MAP_RETURN(-1)

		pr_info("FLOWMGR: MAP-T domain (%d) config unset\n", domain);
	}

	// Invalidate stored config for domain
	db = flowmgr_tunnel_find_mapt_domain(domain);
	if (db && (db->state == 1)) {
		// Set state to unconfigure
		db->state = 0;
		db->id = -1;
	}

	return 0;
}

int flowmgr_map_flush(int domain)
{
	int ret = 0;

	if (flowmgr.disable_mapt_accel < 4)
		ret = inner_map_flush(domain);

	return ret;
}
EXPORT_SYMBOL(flowmgr_map_flush);

static
void cleanup_prefix(__u8 *prefix, __be16 prefix_length)
{
	int prefixlen;
	u8 remainder;
	__be32 prefix_copy[2];
	__u8 mask = 0;

	// Clean up br_prefix based off actual length
	prefixlen = prefix_length >> 3;
	remainder = prefix_length - (prefixlen << 3);

	// Save and clean
	memcpy(&prefix_copy, prefix, 8);
	memset(prefix, 0, 8);

	// If prefix length isn't on a BYTE boundary, we have to copy (prefixlen + 1) bytes
	if (remainder) {
		// Need to handle fact that mask effects most significant bits first
		mask = ((1 << remainder) - 1) << (8-remainder);
		prefixlen += 1;
	}

	memcpy(prefix, &prefix_copy, prefixlen);

	// Mask last byte if needed
	if (mask) {
		prefix += (prefixlen-1);
		*prefix = (*prefix & mask);
	}

}

static
int inner_map_config(struct net_device *dev,
		     struct flow_map_params *params)
{
	int index;
	int ret = -1;
	struct flowmgr_tdb_entry *db = NULL;

	// Parameters
	if (params == NULL)
		FLOWMGR_MAP_RETURN(-1)

	// Transport, currently only MAPT supported
	if (params->type != BCMFLOW_MAPT)
		FLOWMGR_MAP_RETURN(-1)

	// Domain range
	index = (int)params->mapt.domain_index;
	if ((index < 0) || (index >= FLOWMGR_MAPT_MAX_DOMAINS))
		FLOWMGR_MAP_RETURN(-1)

	// Clean up prefixes based off actual length
	cleanup_prefix((__u8 *)&params->mapt.br_prefix[0],
		       params->mapt.br_prefix_length);
	cleanup_prefix((__u8 *)&params->mapt.ipv6_prefix[0],
		       params->mapt.ipv6_prefix_length);

	// Check if config parameter has changed
	db = flowmgr_tunnel_find_mapt_domain(index);
	if (db && memcmp(&db->mapt, &(params->mapt), sizeof(struct flow_mapt_params)) == 0) {
		pr_debug("FLOWMGR: MAP-T domain (%d) re-config with same parameters\n", index);
		return 0;
	}

	// Check if domain already configured and flush
	flowmgr_map_flush(index);
	flowmgr_tunnel_mapt_delete(index);

	// map_ipv6_address will be overwritten on first v4tov6,
	// zero out for now.
	memset(&params->mapt.map_ipv6_address, 0, 16);

	// Store
	ret = flowmgr_tunnel_mapt_insert(dev, &(params->mapt));
	if (ret != 0)
		FLOWMGR_MAP_RETURN(-1)

	flowmgr_dump_mapt(&params->mapt);

	pr_info("FLOWMGR: MAP-T domain (%d) config stored\n", index);

	return 0;
}
int flowmgr_map_config(struct net_device *dev, struct flow_map_params *params)
{
	int ret = 0;

	if (dev && (flowmgr.disable_mapt_accel < 4))
		ret = inner_map_config(dev, params);

	return ret;
}
EXPORT_SYMBOL(flowmgr_map_config);

static
int inner_map_add(struct sk_buff *skb,
		struct sk_buff *newskb,
		int domain,
		__u32 lan_addr, __u16 lan_port)
{
	bool pkt4, pkt6;
	__be16 prot;
	bool v4_v6;
	int ret;

	// Domain range
	if (((domain < 0) && (newskb != NULL)) ||
	    (domain >= FLOWMGR_MAPT_MAX_DOMAINS))
		FLOWMGR_MAP_RETURN(-1)

	// Check buffer
	if (skb == NULL)
		FLOWMGR_MAP_RETURN(-1)

	// Determine MAP direction
	pkt4 = false;
	pkt6 = false;
	v4_v6 = false;

	prot = skb->protocol;
	if (prot == __constant_ntohs(ETH_P_IP)) {
		v4_v6 = true;
		pkt4 = true;
	} else if (prot == __constant_ntohs(ETH_P_IPV6)) {
		v4_v6 = false;
		pkt6 = true;
	} else {
		// Just accept, this is not for us
		return 0;
	}

	// Clean up newskb if we have one
	if (newskb != NULL) {
		prot = newskb->protocol;
		if (prot == __constant_ntohs(ETH_P_IP)) {
			pkt4 = true;
		} else if (prot == __constant_ntohs(ETH_P_IPV6)) {
			pkt6 = true;
		} else {
			FLOWMGR_MAP_RETURN(-1)
		}

		// There is an issue with cernet IPv6 packet, the skb header pointers
		// are not set.  Update the skb headers.
		// Example:
		//    newskb->protocol = dd86 (86dd)
		//    newskb->mac_header = 66
		//    newskb->network_header = 0
		//    newskb->transport_header = 65535
		if (newskb->network_header == 0 ||
		    !skb_transport_header_was_set(newskb))
			set_skb_headers(newskb);
	}

	// Add flow
	if (v4_v6) {
		ret = set_skb_dev_in_iif_group(skb, newskb,
						BCM_NETDEVICE_GROUP_LAN);
		if (ret != 0)
			FLOWMGR_MAP_RETURN(-1)

		ret = flowmgr_mapt_add_4to6(skb, newskb, domain, lan_addr, lan_port);
	} else {
		ret = set_skb_dev_in_iif_group(skb, newskb,
						BCM_NETDEVICE_GROUP_WAN);
		if (ret != 0)
			FLOWMGR_MAP_RETURN(-1)

		ret = flowmgr_mapt_add_6to4(skb, newskb, domain, lan_addr, lan_port);
	}

	return ret;
}

int flowmgr_map_add(struct sk_buff *skb,
		    struct sk_buff *newskb,
		    int domain,
		    __u32 lan_addr, __u16 lan_port)
{
	int ret = 0;

	// only add flow from non-host packet, host packet will not have skb->dev_in set
	if (flowmgr.disable_mapt_accel < 4 && skb->dev_in)
		ret = inner_map_add(skb, newskb, domain, lan_addr, lan_port);
	else {
		// If MAPT accel is disable, we don't want the flowmgr to promote this connection as a flowmgr nf flow.
		if (newskb)
			newskb->dev_in = NULL;
	}

	return ret;
}
EXPORT_SYMBOL(flowmgr_map_add);

struct net_device *flowmgr_get_mapt_dev(void)
{
	struct net_device *dev;

	// Use wandev as key
	dev = flowmgr_get_wandev();

	// cm0 is fallback if wandev not available
	if (dev == NULL)
		dev = __dev_get_by_name(&init_net, "cm0");

	return dev;
}

static
int hw_config_map(struct sk_buff *skb,
		  struct nf_conn *ct,
		  struct nf_conn_offload *ct_offload,
		  const struct net_device *dev)
{
	struct offload_info *ct_offload_info;
	struct flowmgr_map_offload *map_offload;
	struct flow_params *params;
	int domain;
	struct flowmgr_tdb_entry *db = NULL;
	int ret = -1;

	// Get offload
	ct_offload_info = &ct_offload->info[CTINFO2DIR(nfctinfo(skb))];
	if (ct_offload_info == NULL)
		FLOWMGR_MAP_RETURN(-1)

	// Check for map information
	if (ct_offload_info->map == NULL)
		return 0;

	map_offload = (struct flowmgr_map_offload *)ct_offload_info->map;
	params = (struct flow_params *)&map_offload->params;;

	if ((params->type != ft_ipv4) ||
	    (params->ipv4.type != ft_ipv4_mapt_map))
		return 0;

	domain = params->ipv4.mapt.domain_index;
	db = flowmgr_tunnel_find_mapt_domain(domain);

	if (db && (db->state == 0)) {
		// Update ipv6 address from skb
		memcpy(&(db->mapt.map_ipv6_address),
		       params->ipv4.mapt.ipv6_src_ip, 16);

		// Configure
		if (flowmgr_is_feat_map_enable() && flowmgr.disable_mapt_accel < 3)
			ret = flowmgr_config_mapt(&(db->mapt), dev->ifindex);

		if (ret != 0)
			FLOWMGR_MAP_RETURN(-1)

		// Set state to configured
		db->state = 1;
		db->oif = dev->ifindex;
		db->id = domain;

		flowmgr_dump_mapt(&(db->mapt));

		pr_info("FLOWMGR: MAP-T domain (%d) config applied\n",
			db->mapt.domain_index);
	}

	return 0;
}
int flowmgr_mapt_process(struct sk_buff *skb,
			 struct nf_conn *ct,
			 struct nf_conn_offload *ct_offload,
			 const struct net_device *dev)
{
	int ret = 0;

	if (flowmgr.disable_mapt_accel < 4)
		ret = hw_config_map(skb, ct, ct_offload, dev);

	return ret;
}

static inline
void dump_interface(u32 interface, char *direction)
{
	struct net_device *dev = NULL;
	struct dqnet_netdev *ndev = NULL;

	pr_alert(" %s interface %d\n",
		 direction, interface);
	dev = __dev_get_by_index(&init_net, interface);
	pr_alert(" derived from %s interface:\n",
		 direction);
	pr_alert("   dev        %s\n",
	dev ? dev->name:"null");
	if (dev == NULL)
		return;
	ndev = netdev_priv(dev);
	if (ndev == NULL)
		return;
	pr_alert(" derived from %s dev priv (dqnet_netdev):\n",
		 direction);
	pr_alert("   if_id      %d\n",
		 ndev->if_id);
	pr_alert("   if_sub_id  %d\n",
		 ndev->if_sub_id);
}

static inline
void dump_tx_interface(u32 interface)
{
	dump_interface(interface, "tx");
}

static inline
void dump_rx_interface(u32 interface)
{
	dump_interface(interface, "rx");
}

static inline
void dump_subif(u32 subif, char *direction)
{
	pr_alert(" %s subif     %d\n",
		 direction, subif);
}

static inline
void dump_tx_subif(u32 subif)
{
	dump_subif(subif, "tx");
}

static inline
void dump_rx_subif(u32 subif)
{
	dump_subif(subif, "rx");
}

static inline
void dump_tx_params_control(struct flow_tx_control *control)
{
	pr_alert(" control:\n");
	pr_alert("   vlan_tci   %d\n",
	control->vlan_tci);
	pr_alert("   vlan_idx   %d\n",
	control->vlan_idx);
	pr_alert("   priority   %d\n",
	control->priority);
	pr_alert("   dscp_mark  %d\n",
	control->dscp_mark);
	pr_alert("   dscp_val   %d\n",
	control->dscp_val);
	pr_alert("   lag        %d\n",
	control->lag);
}

static inline
void dump_rx_params_control(struct flow_rx_control *control)
{
	pr_alert(" control:\n");
	pr_alert("   vlan_untag %d\n",
		 control->vlan_untag);
}

static inline
void dump_tx_params(struct flow_tx_params *tx)
{
	dump_tx_interface(tx->interface);
	dump_tx_subif(tx->interface);
	dump_tx_params_control(&tx->control);
}

static inline
void dump_rx_params(struct flow_rx_params *rx)
{
	dump_rx_interface(rx->interface);
	dump_rx_subif(rx->interface);
	dump_rx_params_control(&rx->control);
}

static
int inner_promote_mapt(int prot,
		       struct sk_buff *skb,
		       struct nf_conn *ct,
		       struct offload_info *ct_offload_info,
		       char *desc)
{
	struct flow_params *params;
	enum flow_type type;
	int combined_type;
	int domain;
	const char *flow_desc = "unknown";
	struct flowmgr_tdb_entry *db = NULL;
	int flow_id = -1, ret = -1;
	int dscp;
	struct net_device *dev_in, *dev_out;
	struct nf_conn_offload *ct_offload = NULL;
	struct nf_conn_offload *ct_offload_master = NULL;
	bool tuple_replacement_use_dst;
	struct flowmgr_map_offload *map_offload;
	int in_dev_cat;
	int out_dev_cat;
	struct flow_tunnel_params tunnel;
	struct flowmgr_tdb_entry *tdb;

	if (!flowmgr_is_feat_map_enable()) {
		if (ct_offload_info->debug)
			pr_alert("FLOWMGR: Promotion error - MAPT mso feature not enabled\n");
		goto err_out;
	}

	dscp = ip_get_dscp(skb);
	if (dscp != -1)
		ct_offload_info->dscp_new = dscp;

	ct_offload = nf_conn_offload_find(ct);

	map_offload = (struct flowmgr_map_offload *)ct_offload_info->map;
	params = (struct flow_params *)&map_offload->params;
	if (!params) {
		if (ct_offload_info->debug)
			pr_alert("FLOWMGR: Promotion error - MAPT offload not created\n");
		goto err_out;
	}

	// set up the flow paramters for promotion
	dev_in = __dev_get_by_index(&init_net, ct_offload_info->iif);
	if (dev_in == NULL) {
		if (ct_offload_info->debug)
			pr_alert("FLOWMGR: Promotion error - MAPT input device not set\n");
		goto err_out;
	}

	// Now that we know output device, set in parameter structure
	dev_out = __dev_get_by_index(&init_net, ct_offload_info->oif);
	if (dev_out == NULL) {
		if (ct_offload_info->debug)
			pr_alert("FLOWMGR: Promotion error - MAPT output device not set\n");
		goto err_out;
	}

	if (!feat->mapt || !feat->mapt->ops2) {
		if (ct_offload_info->debug)
			pr_alert("FLOWMGR: Promotion error - MAPT mso feature options not set\n");
		goto err_out;
	}

	in_dev_cat = BCM_NETDEVICE_GROUP_CAT(dev_in->group);
	out_dev_cat = BCM_NETDEVICE_GROUP_CAT(dev_out->group);

	if (in_dev_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP ||
		in_dev_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP6) {
		tdb = flowmgr_tdb_find(dev_in->ifindex);
		if (!tdb) {
			if (ct_offload_info->debug)
				pr_alert("FLOWMGR: Promotion error - MAPT input gre tunnel not found\n");
			goto err_out;
		}

		if (!flowmgr_is_feature_enabled((FLOW_F_GRETAP4_MAPT |
										FLOW_F_GRETAP6_MAPT))) {
			if (ct_offload_info->debug)
				pr_alert("FLOWMGR: Promotion error - MAPT flowmgr feature not enabled\n");
			goto err_out;
		}

		params->rx.interface = tdb->oif;
		ct_offload_info->decap_tid = tdb->id;
		if (out_dev_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP)
			params->decap_tun.gre.type = ft_ipv4;
		else
			params->decap_tun.gre.type = ft_ipv6;
	} else
		params->rx.interface = dev_in->ifindex;

	if (out_dev_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP ||
		out_dev_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP6) {
		tdb = flowmgr_tdb_find(dev_out->ifindex);
		if (!tdb) {
			if (ct_offload_info->debug)
				pr_alert("FLOWMGR: Promotion error - MAPT output gre tunnel not found\n");
			goto err_out;
		}

		if (!flowmgr_is_feature_enabled((FLOW_F_MAPT_GRETAP4 |
										FLOW_F_MAPT_GRETAP6))) {
			if (ct_offload_info->debug)
				pr_alert("FLOWMGR: Promotion error - MAPT flowmgr feature not enabled\n");
			goto err_out;
		}

		params->tx.interface =
				flowmgr_tunnel_out_ifindex(dev_out,
				&tunnel, &params->wifi);
		ct_offload_info->encap_tid = tdb->id;
		if (out_dev_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP)
			params->encap_tun.gre.type = ft_ipv4;
		else
			params->encap_tun.gre.type = ft_ipv6;
	} else
		params->tx.interface = dev_out->ifindex;

	ret = feat->mapt->ops2(skb, ct_offload_info, dev_in->ifindex,
			dev_out->ifindex, params, &flow_desc,
			&domain, (int *)&type, &combined_type);
	if (ret) {
		if (ct_offload_info->debug)
			pr_alert("FLOWMGR: Promotion error - MAPT mso feature ops error %d\n", ret);
		goto err_out;
	}

	// Check configuration state
	db = flowmgr_tunnel_find_mapt_domain(domain);
	if (!db) {
		if (ct_offload_info->debug)
			pr_alert("FLOWMGR: Promotion error - MAPT domain not found %d\n", domain);
		ret = -1;
		goto err_out;
	}

	if (db->state == 0) {
		if (ct_offload_info->debug)
			pr_alert("FLOWMGR: Promotion error - MAPT tunnel not initialized in hw\n");
		return 0;
	}

	if (type == ft_ipv4) {
		tuple_replacement_use_dst = 1;
		if (flowmgr.disable_mapt_accel & 1) {
			ct_offload_info->flow_type = ft_ignore<<16;
			atomic_inc(&flowmgr.flow_ignore_cnt);
			return -1;
		}
	} else {
		tuple_replacement_use_dst = 0;
		if (flowmgr.disable_mapt_accel & 2) {
			ct_offload_info->flow_type = ft_ignore<<16;
			atomic_inc(&flowmgr.flow_ignore_cnt);
			return -1;
		}
	}

	// Set wifi parameters
	params->wifi.flowring = ct_offload_info->wifi_flowring;
	params->wifi.pri = ct_offload_info->wifi_pri;

	if (ct_offload_info->debug) {
		pr_alert("FLOWMGR: Tx & Rx params:\n");
		dump_tx_params(&params->tx);
		dump_rx_params(&params->rx);
	}

	ret = -1;
	if (flowmgr.disable_mapt_accel < 3) {
		// Demote the expected flow that was previously added,
		// allowing the current flow to be added.
		if (type == ft_ipv6 && ct) {
			struct nf_conn_offload *ct4_offload;
			struct nf_conn_offload *ct6_offload;
			struct nf_conn *ct6;
			struct offload_info *ct4_offload_info;
			struct flowmgr_map_offload *map_offload;

			ct4_offload = nf_conn_offload_find(ct);
			if (ct4_offload) {
				ct4_offload_info = &ct4_offload->info[IP_CT_DIR_REPLY];
				map_offload = (struct flowmgr_map_offload *) ct4_offload_info->map;

				if (map_offload && map_offload->master_ct) {
					ct6 = map_offload->master_ct;
					ct6_offload = nf_conn_offload_find(ct6);
					if (ct6_offload) {
						struct offload_info *ct6_offload_info;
						ct6_offload_info = &ct6_offload->info[IP_CT_DIR_REPLY];
						flowmgr_expected_demote(ct6_offload_info);
					}
				}
			}
		}
		ret = flowmgr_flow_add(params, &flow_id, ct_offload_info);
	}

	if (ret != 0)
		return ret;

	if (ct_offload_info->debug)
		pr_alert("FLOWMGR: Flow MAP-T (%s) added: %d %s\n", flow_desc, flow_id, desc);

	if (ct->master)
		ct_offload_master = nf_conn_offload_find(ct->master);

	flowmgr_update_ct_offload(flow_id,
				  combined_type,
				  params->tx.control.lag,
				  NULL,
				  ct_offload_info,
				  ct_offload,
				  ct_offload_master);

	ct->mark &= ~(0xFFFF << (tuple_replacement_use_dst*16));
	ct->mark |= flow_id << (tuple_replacement_use_dst*16);

	if (type == ft_ipv4) {
		flowmgr_dump_ipv4(flowmgr_get_directions(flow_lanwan),
				  ct->mark, ct_offload_info,
				  &(params->ipv4));
	} else {
		flowmgr_dump_ipv6(flowmgr_get_directions(flow_wanlan),
				  ct->mark, ct_offload_info,
				  &(params->ipv6));
	}

	return 0;
err_out:
	ct_offload_info->create_err = 1;
	return ret;
}


int flowmgr_promote_mapt(int prot,
			 struct sk_buff *skb,
			 struct nf_conn *ct,
			 struct offload_info *ct_offload_info)
{
	int ret = 0;

	if (flowmgr.disable_mapt_accel < 4) {
		ret = inner_promote_mapt(prot, skb,
					 ct, ct_offload_info,
					 FLOWMGR_SINGLE_DESC);
	}

	return ret;
}

static
int inner_promote_mapt_duplex(int prot,
			      struct sk_buff *skb,
			      struct nf_conn *ct,
			      struct offload_info *ct_offload_info)
{
	struct nf_conn *master;
	enum ip_conntrack_info ctinfo = nfctinfo(skb);
	struct nf_conn_offload *ct_offload = NULL;
	int ret = 0;

	// Need a master
	master = master_ct(ct);

	if (master == NULL)
		FLOWMGR_MAP_RETURN(0);

	// Need ct_offload for later, but check we can obtain it now
	ct_offload = nf_conn_offload_find(ct);
	if (!ct_offload)
		FLOWMGR_MAP_RETURN(-1)

	// Attempt promote this direction, a return code of 1
	// will indicate that an connection has not been established
	// yet.
	ret = inner_promote_mapt(prot, skb,
				 ct, ct_offload_info,
				 FLOWMGR_DUPLEX_12_DESC);
	if (ret != 0)
		FLOWMGR_MAP_RETURN(ret)

	// Get offload info from master and reverse
	ct_offload = nf_conn_offload_find(master);
	if (!ct_offload)
		FLOWMGR_MAP_RETURN(-1)

	if (CTINFO2DIR(ctinfo) == IP_CT_DIR_ORIGINAL)
		ct_offload_info = &ct_offload_repl;
	else
		ct_offload_info = &ct_offload_orig;

	// Promote reverse direction, don't have corresponding skb
	// but we don't need it at this point - all params should be
	// already set in offload info.
	ret = inner_promote_mapt(prot, NULL,
				 master, ct_offload_info,
				 FLOWMGR_DUPLEX_22_DESC);
	if (ret != 0)
		FLOWMGR_MAP_RETURN(ret)

	return 0;
}

int flowmgr_promote_mapt_duplex(int prot,
				struct sk_buff *skb,
				struct nf_conn *ct,
				struct offload_info *ct_offload_info)
{
	int ret = 0;

	if (flowmgr.disable_mapt_accel < 4) {
		ret = inner_promote_mapt_duplex(prot, skb,
						ct, ct_offload_info);
	}

	return ret;
}

void flowmgr_dump_mapt(struct flow_mapt_params *params)
{
	__be16 *p;
	__be32 v4prefix;

	if (params == NULL)
		return;

	pr_info("MAPT domain (%d) configuration\n",
		params->domain_index);
	p = (__be16 *)&params->br_prefix;
	pr_info("   br_prefix        %x:%x:%x:%x\n",
		__be16_to_cpu(p[0]), __be16_to_cpu(p[1]),
		__be16_to_cpu(p[2]), __be16_to_cpu(p[3]));
	pr_info("   br_prefix_length %d\n",
		params->br_prefix_length);
	p = (__be16 *)&params->ipv6_prefix;
	pr_info("   ipv6_prefix        %x:%x:%x:%x\n",
		__be16_to_cpu(p[0]), __be16_to_cpu(p[1]),
		__be16_to_cpu(p[2]), __be16_to_cpu(p[3]));
	pr_info("   ipv6_prefix_length %d\n",
		params->ipv6_prefix_length);
	// The ipv4_prefix is stored as host representation
	v4prefix = htonl(params->ipv4_prefix);
	pr_info("   ipv4_prefix        %pI4\n",
		&v4prefix);
	pr_info("   ipv4_prefix_length %d\n",
		params->ipv4_prefix_length);
	pr_info("   map_ipv6_address %pI6c\n",
		&params->map_ipv6_address);
	pr_info("   psid_offset      %d\n",
		params->psid_offset);
	pr_info("   psid_length      %d\n",
		params->psid_length);
	pr_info("   psid             %d\n",
		params->psid);
}

int flowmgr_map_init(void)
{
	int ret;

	// For now disable everything but unit test by default
	flowmgr.disable_mapt_accel = FLOWMGR_MAP_DEFAULT_DISABLE_LEVEL;

	ret = nf_conntrack_helper_register(&nf_conntrack_helper_map_udp);
	if (ret != 0)
		FLOWMGR_MAP_RETURN(-1)

	ret = nf_conntrack_helper_register(&nf_conntrack_helper_map_tcp);
	if (ret != 0)
		FLOWMGR_MAP_RETURN(-1)

	ret = nf_conntrack_helper_register(&nf_conntrack_helper_unmap_udp);
	if (ret != 0)
		FLOWMGR_MAP_RETURN(-1)

	ret = nf_conntrack_helper_register(&nf_conntrack_helper_unmap_tcp);
	if (ret != 0)
		FLOWMGR_MAP_RETURN(-1)

	return 0;
}

void flowmgr_map_fini(void)
{
	int i;

	for (i = 0; i < FLOWMGR_MAPT_MAX_DOMAINS; i++)
		flowmgr_map_flush(i);

	nf_conntrack_helper_unregister(&nf_conntrack_helper_map_udp);
	nf_conntrack_helper_unregister(&nf_conntrack_helper_map_tcp);
	nf_conntrack_helper_unregister(&nf_conntrack_helper_unmap_udp);
	nf_conntrack_helper_unregister(&nf_conntrack_helper_unmap_tcp);
}

int flowmgr_map_get_src(struct sk_buff *skb, __u32 *saddr, __u16 *port)
{
	struct iphdr *ip4h;
	__u8 *payload;
	struct tcphdr *tcph;
	struct udphdr *udph;

	if ((skb == NULL) || (saddr == NULL) || (port == NULL))
		FLOWMGR_MAP_RETURN(-1)

	ip4h = ip_hdr(skb);

	payload = (__u8 *)(ip4h) + (ip4h->ihl << 2);
	if (ip4h->protocol == IPPROTO_TCP) {
		tcph = (struct tcphdr *)payload;
		*port = tcph->source;
	} else if (ip4h->protocol == IPPROTO_UDP) {
		udph = (struct udphdr *)payload;
		*port = udph->source;
	} else
		return -1;

	*saddr = ip4h->saddr;

	return 0;
}
EXPORT_SYMBOL(flowmgr_map_get_src);

int flowmgr_map_set_src(struct sk_buff *skb, __u32 saddr, __u16 port)
{
	struct iphdr *ip4h;
	__u8 *payload;
	struct tcphdr *tcph;
	struct udphdr *udph;

	if (skb == NULL)
		FLOWMGR_MAP_RETURN(-1)

	ip4h = ip_hdr(skb);

	payload = (__u8 *)(ip4h) + (ip4h->ihl << 2);
	if (ip4h->protocol == IPPROTO_TCP) {
		tcph = (struct tcphdr *)payload;
		tcph->source = htons(port);
	} else if (ip4h->protocol == IPPROTO_UDP) {
		udph = (struct udphdr *)payload;
		udph->source = htons(port);
	} else
		return -1;

	ip4h->saddr = saddr;

	return 0;
}
EXPORT_SYMBOL(flowmgr_map_set_src);

int flowmgr_map_disable(int disable)
{
	flowmgr.disable_mapt_accel = disable;
	if (flowmgr.disable_mapt_accel < 3) {
		if (mso_feat_mapt_init())
			pr_alert("FLOWMGR: MAPT mso feature not supported\n");
		else if (!flowmgr_is_feat_map_enable())
			pr_alert("FLOWMGR: MAPT mso feature not enabled\n");
	} else {
		mso_feat_mapt_deinit();
	}
	return 0;
}

bool flowmgr_is_feat_map_enable(void)
{
	bool enable = false;
	if (feat && feat->mapt && feat->mapt->ops0) {
		if (!feat->mapt->ops0())
			enable = true;
	}
	return enable;
}

int flowmgr_update_stats_mapt(const struct nf_conn *ct,
							  int direction,
							  unsigned int packets_diff,
							  unsigned int bytes_diff)
{
	struct nf_conn_offload *ct_offload = nf_conn_offload_find(ct);
	struct nf_conn *master_ct = NULL;
	struct flowmgr_map_offload *map_offload = NULL;
	struct nf_conn_acct *acct;
	int domain;
	struct flowmgr_tdb_entry *tdb;
	struct net_device *map_dev;

	if (ct_offload_orig.map)
		map_offload = (struct flowmgr_map_offload *) ct_offload_orig.map;
	else if (ct_offload_repl.map)
		map_offload = (struct flowmgr_map_offload *) ct_offload_repl.map;
	else
		return 0;

	master_ct = map_offload->master_ct;
	if (!master_ct || nf_ct_is_dying(master_ct))
		return 0;

	if (map_offload->params.type == ft_ipv6) {
		domain = map_offload->params.ipv6.mapt.domain_index;
		tdb = flowmgr_tunnel_find_mapt_domain(domain);
		if (tdb) {
			map_dev = __dev_get_by_index(&init_net, tdb->ifi);
			if (map_dev) {
				/* nat46 driver stats */
				map_dev->stats.rx_packets += packets_diff;
				map_dev->stats.rx_bytes += bytes_diff;
			}
		}
	} else {
		domain = map_offload->params.ipv4.mapt.domain_index;
		tdb = flowmgr_tunnel_find_mapt_domain(domain);
		if (tdb) {
			map_dev = __dev_get_by_index(&init_net, tdb->ifi);
			if (map_dev) {
				/* nat46 driver stats */
				map_dev->stats.tx_packets += packets_diff;
				map_dev->stats.tx_bytes += bytes_diff;
			}
		}
	}

	/* Update nf connection acc stats for master ct*/
	acct = nf_conn_acct_find(master_ct);
	if (acct) {
		struct nf_conn_counter *nf_counter;
		nf_counter = &(acct->counter[direction]);
		atomic64_add(packets_diff,
			     &nf_counter->packets);
		atomic64_add(bytes_diff,
			     &nf_counter->bytes);
	}
	flowmgr_update_offload_retry_timeout(master_ct);
	return 0;
}
