 /****************************************************************************
 *
 * Copyright (c) 2015-2018 Broadcom. All rights reserved
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************/
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/phy.h>
#include <linux/of_mdio.h>

#include "ethsw.h"
#include "ethsw_priv.h"
#include "ethsw_core.h"
#include "ethsw_3390.h"
#include <brcm_mbox.h>

#include <linux/brcmstb/brcmstb.h>
#include <uapi/linux/if_bridge.h>

#ifdef __BIG_ENDIAN
#error "This file is only used on little-endian platforms!"
#endif
/*
 * Log Utilities
 */

/* define log module */
#define LOG_MODULE "ethsw_3390"

static void ethsw_mii_init(struct ethsw_device *swdev, int port_id)
{
	struct ethsw_port *port = &swdev->port[port_id];
	u32 reg, val, speed;

	reg = BCHP_SWITCH_CORE_STS_OVERRIDE_GMII_P0 + (port_id * 4);
	speed = (port->mii.speed == 10) ? 0 : (port->mii.speed == 100 ? 1 : 2);
	val = ethsw_reg_read(swdev, reg);
	val &= ~BCHP_SWITCH_CORE_STS_OVERRIDE_GMII_P0_SPEED_MASK;
	val |= speed << BCHP_SWITCH_CORE_STS_OVERRIDE_GMII_P0_SPEED_SHIFT;
	val |= 1 << BCHP_SWITCH_CORE_STS_OVERRIDE_GMII_P0_DUPLX_MODE_SHIFT;
	val |= 1 << BCHP_SWITCH_CORE_STS_OVERRIDE_GMII_P0_LINK_STS_SHIFT;
	val |= 1 << BCHP_SWITCH_CORE_STS_OVERRIDE_GMII_P0_SW_OVERRIDE_SHIFT;
	ethsw_reg_write(swdev, reg, val);

	if (port->mii.rgmii) {
		reg = (port_id == 4) ? BCHP_SWITCH_REG_RGMII_9_CNTRL :
			BCHP_SWITCH_REG_RGMII_10_CNTRL;
		val = ethsw_reg_read(swdev, reg);
		val |= 1 << BCHP_SWITCH_REG_RGMII_9_CNTRL_rgmii_mode_en_SHIFT;
		ethsw_reg_write(swdev, reg, val);
	}
}

/*
 * Ethsw Device Functions
 */
int ethsw_swdev_board_init_sf2(struct ethsw_device *swdev)
{
	int ret = 0;
	int timeout = 0;
	int i;
	u8 val8;

	FUNC_ENTER();

	/* we first need to reset the switch  chip to a known state. */
	val8 = ethsw_reg_read(swdev, BCHP_SWITCH_CORE_WATCH_DOG_CTRL);
	val8 |= (BCHP_SWITCH_CORE_WATCH_DOG_CTRL_SOFTWARE_RESET_MASK |
			BCHP_SWITCH_CORE_WATCH_DOG_CTRL_EN_SW_RESET_MASK);
	ethsw_reg_write(swdev, BCHP_SWITCH_CORE_WATCH_DOG_CTRL, val8);
	do {
		udelay(100);
		val8 = ethsw_reg_read(swdev, BCHP_SWITCH_CORE_WATCH_DOG_CTRL);
		timeout++;
	}
	while ((val8 & BCHP_SWITCH_CORE_WATCH_DOG_CTRL_SOFTWARE_RESET_MASK) &&
			(timeout < 1000));

	if (timeout == 1000) {
		LOG_ERR("Ethsw chip reset failed\n");
		return -ENOENT;
	}

	swdev->ethsw_pwr_status = true;
	/* set up rgmii mode and speed for port 4 and port 7 */
	if (swdev->port[4].type == MII_PORT)
		ethsw_mii_init(swdev, 4);

	if (swdev->port[7].type == MII_PORT)
		ethsw_mii_init(swdev, 7);

	/* For each internal phy port save off the Qphy reg for power operations */
	for (i = 0; i < ETHSW_PHY_PORT_MAX; i++) {
		swdev->port[i].qphy_ctrl_reg = (unsigned int)&SWITCH_REG->QphyCntrl;
	}

	FUNC_LEAVE();
	return ret;
}

void ethsw_swdev_board_exit_sf2(struct ethsw_device *swdev)
{
	u32 val32;
	FUNC_ENTER();

	/*Set the eth switch power status */
	swdev->ethsw_pwr_status = false;

	/* Update switch core to sleep mode and disable the switch timer */
	val32 = ethsw_reg_read(swdev, BCHP_SWITCH_CORE_LOW_POWER_CTRL);
	val32 |= (BCHP_SWITCH_CORE_LOW_POWER_CTRL_SLEEP_SYS_MASK |
			BCHP_SWITCH_CORE_LOW_POWER_CTRL_TIMER_DISABLE_MASK |
			BCHP_SWITCH_CORE_LOW_POWER_CTRL_EN_LOW_POWER_MASK);
	ethsw_reg_write(swdev, BCHP_SWITCH_CORE_LOW_POWER_CTRL, val32);

	FUNC_LEAVE();
}

static int ethsw_phy_set_iddq_global_pwr(struct ethsw_device *swdev, bool global_pwrdn)
{
	int ret = 0;
	u32 reg_val32;
	struct ethsw_port *port;
	int port_id = 0;

	FUNC_ENTER();
	if (global_pwrdn) {
		for (port_id = 0; port_id < ETHSW_PHY_PORT_MAX; port_id++) {
			port = &swdev->port[port_id];
			if ((port->type == PHY_PORT || port->type == MII_PORT)) {
				if (port->phy.dev) {
					ethsw_set_phy_state(port, ETHSW_PHY_STOP);

					if(!port->phy.dev->link)
						phy_suspend(port->phy.dev);
				}
			}
		}
		reg_val32 = __raw_readl((u32 *)&(SWITCH_REG->QphyCntrl));
		((QphyCntrlRegister *)&reg_val32)->Bits.IddqGlobalPwr = 0x1;
		__raw_writel(reg_val32, &(SWITCH_REG->QphyCntrl));
	} else {
		reg_val32 = __raw_readl((u32 *)&(SWITCH_REG->QphyCntrl));
		((QphyCntrlRegister *)&reg_val32)->Bits.IddqGlobalPwr = 0;
		__raw_writel(reg_val32, &(SWITCH_REG->QphyCntrl));
		/* special PHY power up sequence */
		((QphyCntrlRegister *)&reg_val32)->Bits.PhyReset = 0;
		__raw_writel(reg_val32, &(SWITCH_REG->QphyCntrl));

		udelay(10);

		((QphyCntrlRegister *)&reg_val32)->Bits.PhyReset = 1;
		__raw_writel(reg_val32, &(SWITCH_REG->QphyCntrl));

		udelay(100);

		((QphyCntrlRegister *)&reg_val32)->Bits.PhyReset = 0;
		__raw_writel(reg_val32, &(SWITCH_REG->QphyCntrl));

		udelay(100);
		for (port_id = 0; port_id < ETHSW_PHY_PORT_MAX; port_id++) {
			port = &swdev->port[port_id];
			if ((port->type == PHY_PORT || port->type == MII_PORT)) {
				ethsw_set_phy_state(port, ETHSW_PHY_START);
			}
		}
	}
	FUNC_LEAVE();

	return ret;
}

int ethsw_core_low_power_mode_sf2(struct ethsw_device *swdev)
{
	u32 val32;
	FUNC_ENTER();

	/*Set the eth switch power status */
	swdev->ethsw_pwr_status = false;

	/* Read the CHIP type for SMIC */
	val32 = jtag_otp_reg_read(swdev, BCHP_JTAG_OTP_UB_GENERAL_STATUS_5);
	if (((val32 >> BCHP_JTAG_OTP_UB_GENERAL_STATUS_5_FAB_ID_SHIFT) &
			BCHP_JTAG_OTP_UB_GENERAL_STATUS_5_FAB_ID_MASK) == FAB_ID_SMIC) {
		ethsw_phy_set_iddq_global_pwr(swdev, 1);
	}
	/* Update switch core to sleep mode and disable the switch timer */
	val32 = ethsw_reg_read(swdev, BCHP_SWITCH_CORE_LOW_POWER_CTRL);

	val32 |= (BCHP_SWITCH_CORE_LOW_POWER_CTRL_SLEEP_SYS_MASK |
			BCHP_SWITCH_CORE_LOW_POWER_CTRL_TIMER_DISABLE_MASK |
			BCHP_SWITCH_CORE_LOW_POWER_CTRL_SLEEP_P8_MASK |
			BCHP_SWITCH_CORE_LOW_POWER_CTRL_SLEEP_P5_MASK |
			BCHP_SWITCH_CORE_LOW_POWER_CTRL_SLEEP_P4_MASK
			);

	ethsw_reg_write(swdev, BCHP_SWITCH_CORE_LOW_POWER_CTRL, val32);

	FUNC_LEAVE();

	return 0;
}

int ethsw_core_normal_power_mode_sf2(struct ethsw_device *swdev)
{
	u32 val32;
	FUNC_ENTER();

	/* Update switch core to sleep mode and disable the switch timer */
	val32 = ethsw_reg_read(swdev, BCHP_SWITCH_CORE_LOW_POWER_CTRL);
	val32 &= (~BCHP_SWITCH_CORE_LOW_POWER_CTRL_SLEEP_SYS_MASK &
			~BCHP_SWITCH_CORE_LOW_POWER_CTRL_TIMER_DISABLE_MASK &
			~BCHP_SWITCH_CORE_LOW_POWER_CTRL_SLEEP_P8_MASK &
			~BCHP_SWITCH_CORE_LOW_POWER_CTRL_SLEEP_P5_MASK &
			~BCHP_SWITCH_CORE_LOW_POWER_CTRL_SLEEP_P4_MASK);
	ethsw_reg_write(swdev, BCHP_SWITCH_CORE_LOW_POWER_CTRL, val32);

	/* Read the CHIP type for SMIC */
	val32 = jtag_otp_reg_read(swdev, BCHP_JTAG_OTP_UB_GENERAL_STATUS_5);
	if (((val32 >> BCHP_JTAG_OTP_UB_GENERAL_STATUS_5_FAB_ID_SHIFT) &
			BCHP_JTAG_OTP_UB_GENERAL_STATUS_5_FAB_ID_MASK) == FAB_ID_SMIC) {
		ethsw_phy_set_iddq_global_pwr(swdev, 0);
	}
	/*Set the eth switch power status */
	swdev->ethsw_pwr_status = true;

	FUNC_LEAVE();

	return 0;
}

#define ETHSW_3390_PORT_OFFSET 0
int ethsw_port_idx_to_phy_id_sf2(struct ethsw_device *swdev, int port_idx)
{
	return (port_idx + swdev->phy_min_addr);
}

int ethsw_port_idx_to_port_id_sf2(struct ethsw_device *swdev, int port_idx)
{
	return (port_idx + ETHSW_3390_PORT_OFFSET);
}

int ethsw_port_id_to_port_idx_sf2(struct ethsw_device *swdev, int port_id)
{
	return (port_id - ETHSW_3390_PORT_OFFSET);
}

/*
 * Switch PHY Functions
 */
int ethsw_phy_board_init_sf2(struct ethsw_device *swdev)
{
	int ret = 0;
	u32 reg_val32;

	FUNC_ENTER();

	reg_val32 = 0;
	((QphyCntrlRegister *)&reg_val32)->Bits.PhyAd = swdev->phy_min_addr;
	dev_dbg(&swdev->pdev->dev, "QGPhy base addr %d\n",
			swdev->phy_min_addr);

	/* special PHY power up sequence */
	((QphyCntrlRegister *)&reg_val32)->Bits.PhyReset = 0;
	__raw_writel(reg_val32, &(SWITCH_REG->QphyCntrl));

	udelay(10);

	((QphyCntrlRegister *)&reg_val32)->Bits.PhyReset = 1;
	__raw_writel(reg_val32, &(SWITCH_REG->QphyCntrl));

	udelay(100);

	((QphyCntrlRegister *)&reg_val32)->Bits.PhyReset = 0;
	__raw_writel(reg_val32, &(SWITCH_REG->QphyCntrl));

	udelay(100);

	if (swdev->high_speed_imp) {
		if (swdev->port[8].type == IMP_PORT)
			ethsw_reg_set_field(swdev, BCHP_SWITCH_REG_SWITCH_CNTRL,
					BCHP_SWITCH_REG_SWITCH_CNTRL_p8_2_5g_en_MASK,
					BCHP_SWITCH_REG_SWITCH_CNTRL_p8_2_5g_en_SHIFT,
					0x1);

		if (swdev->port[7].type == IMP_PORT)
			ethsw_reg_set_field(swdev, BCHP_SWITCH_REG_SWITCH_CNTRL,
					BCHP_SWITCH_REG_SWITCH_CNTRL_p7_2_5g_en_MASK,
					BCHP_SWITCH_REG_SWITCH_CNTRL_p7_2_5g_en_SHIFT,
					0x1);

		if (swdev->port[5].type == IMP_PORT)
			ethsw_reg_set_field(swdev, BCHP_SWITCH_REG_SWITCH_CNTRL,
					BCHP_SWITCH_REG_SWITCH_CNTRL_p5_2_5g_en_MASK,
					BCHP_SWITCH_REG_SWITCH_CNTRL_p5_2_5g_en_SHIFT,
					0x1);
	}

	/* Make sure all ports are powered up */
	((QphyCntrlRegister *)&reg_val32)->Bits.ExtPwrDown =
		QPHYCTRL_AFE_ON_MASK(swdev->phy_port_mask);
	__raw_writel(reg_val32, &(SWITCH_REG->QphyCntrl));

	udelay(100);

	swdev->chip_id = 0x3390;
	LOG_INFO("Device ID %x\n", swdev->chip_id);

	FUNC_LEAVE();
	return ret;
}

int ethsw_phy_set_afe_power_state(struct ethsw_device *swdev,
		int port_id, bool afe_pwrdn)
{
	int ret = 0;
	u32 reg_val32;
	struct ethsw_port *port = &swdev->port[port_id];

	if ((port->type == PHY_PORT || port->type == MII_PORT)) {

		if (afe_pwrdn) {
			if (port->phy.dev) {
				ethsw_set_phy_state(port, ETHSW_PHY_STOP);

				if(!port->phy.dev->link)
					phy_suspend(port->phy.dev);
			}

			if (port->qphy_ctrl_reg) {
				reg_val32 = __raw_readl((volatile void __iomem *)port->qphy_ctrl_reg);
			((QphyCntrlRegister *)&reg_val32)->Bits.ExtPwrDown |= (0x1 << port_id);
				__raw_writel(reg_val32, (volatile void __iomem *)port->qphy_ctrl_reg);
			}

		} else {
			if (port->qphy_ctrl_reg) {
				reg_val32 = __raw_readl((volatile void __iomem *)port->qphy_ctrl_reg);
			((QphyCntrlRegister *)&reg_val32)->Bits.ExtPwrDown &= ~(0x1 << port_id);
				__raw_writel(reg_val32, (volatile void __iomem *)port->qphy_ctrl_reg);
			}
			ethsw_set_phy_state(port, ETHSW_PHY_START);
		}
	}

	return ret;
}

#if defined(CONFIG_BCM_MBOX)
/*
 * NOTE: This function is designed to be called in ISR context. Make sure
 * any future additions are ISR safe.
 */
int ethsw_phy_mbox_cb_sf2(struct notifier_block *this,
		unsigned long event, void *ptr)
{
	struct ethsw_device *swdev = container_of(this,struct ethsw_device,nb);
	struct brcm_mbox_info *states = ptr;
	struct ethsw_core *swcore;
	struct ethsw_port *port;
	unsigned int xcvr_reset_duration;
	int ret = NOTIFY_OK, qphy_eth_altwan = 0, port_cnt;
	u32 port_afe_powerdown_mask = QPHYCTRL_AFE_OFF_BITS, reg_val32;

	if (swdev == NULL || states == NULL)
		return notifier_from_errno(EPERM);

	swcore = swdev->swcore;
	if (swcore == NULL)
		return notifier_from_errno(EPERM);

	switch (event)	{
	case MBOX_CHANGE_EVENT:
		if (BATT_STATE(states->mbox[MBOX_BMU]) == BS_BATTERY  &&
				states->delay_battery_mode) {
			pr_debug("Power operated through battery, delay_battery_mode "
					 "enabled, so no action on Ethernet power control\n");
			return ret;
		}
		/*
		 * Depending on certain configurations we might want to skip
		 * shutting down the AFE power on certain ports.
		 * For ALT-WAN applications - make sure we do not power off
		 * the LAN port power.
		 */
		if (swcore->altwan_enble && (swcore->altwan_lanport != 7))
			qphy_eth_altwan = 1;

		if (qphy_eth_altwan)
			port_afe_powerdown_mask &= ~(1 << swcore->altwan_lanport);

		/*
		 * Currently, we only handle notifications for AC to battery
		 * transitions. Transitions to AC from battery result in a
		 * system reboot - no need to handle that case.
		 */
		if (BATT_STATE(states->mbox[MBOX_BMU]) == BS_BATTERY) {
			reg_val32 = __raw_readl(&(SWITCH_REG->QphyCntrl));
			((QphyCntrlRegister *)&reg_val32)->
				Bits.ExtPwrDown |= port_afe_powerdown_mask;
			__raw_writel(reg_val32,&(SWITCH_REG->QphyCntrl));

			for (port_cnt = 0; port_cnt < ETHSW_PHY_PORT_MAX; port_cnt++) {
				if (!(swdev->phy_port_mask & (1 << port_cnt)))
					continue;
				if ((swdev->port[port_cnt].type == PHY_PORT) && (swdev->ethsw_led_powercntrl)) {
					if ((swcore->altwan_enble) && (port_cnt == swcore->altwan_lanport))
						continue;
					swdev->ethsw_led_powercntrl(swdev, port_cnt, LED_OFF);
				}
			}

			port = &swdev->port[7];
			if (qphy_eth_altwan && port->type &&
					port->ethsw_phy_set_afe_power_state) {
				xcvr_reset_duration = port->xcvr_reset_duration;
				port->xcvr_reset_duration = 0;
				port->ethsw_phy_set_afe_power_state(swdev, 7,
						true);
				port->xcvr_reset_duration = xcvr_reset_duration;
			}
		}
		break;
	default:
		ret = NOTIFY_DONE;
		break;
	}

	return ret;
}
#endif

bool ethsw_phy_get_afe_power_state(struct ethsw_device *swdev, int port_id)
{
	u32 reg_val32;
	struct ethsw_port *port = &swdev->port[port_id];

	if (port->qphy_ctrl_reg) {
		reg_val32 = __raw_readl((u32 *)&(SWITCH_REG->QphyCntrl));
		if (((QphyCntrlRegister *)&reg_val32)->Bits.ExtPwrDown & (0x1 << port_id))
			return true;
	} else if (port->phy.dev && port->phy.dev->suspended == true) {
		return true;
	}

	return false;
}

void __exit ethsw_phy_board_exit_sf2(struct ethsw_device *swdev)
{
	u32 reg_val32;
	int i;

	FUNC_ENTER();

	for (i = 0; i < ETHSW_PORT_MAX; i++)
		ethsw_set_phy_state(&swdev->port[i], ETHSW_PHY_STOP);


	reg_val32 = 0;

	/* PHY power down configuration */
	((QphyCntrlRegister *)&reg_val32)->Bits.ExtPwrDown =
		QPHYCTRL_AFE_ON_MASK(0);
	__raw_writel(reg_val32, &(SWITCH_REG->QphyCntrl));

	FUNC_LEAVE();
}

int ethsw_led_board_init_sf2(struct ethsw_device *swdev)
{
	int ret = 0;
	int i;
	u32 led_mask;
	u32 reg_val32;

	FUNC_ENTER();

	/* build LED mask */
	led_mask = 0;
	for (i = 0; i < ETHSW_PORT_MAX; i++) {
		if ((swdev->port[i].type == PHY_PORT ||
					swdev->port[i].type == MII_PORT) &&
				(swdev->led_map & (1 << i))) {
			led_mask |= (1 << i);
		}
	}

	/* program LED serial control */
	/* Smode         = 1, 2-LED per port */
	/* RefreshPeriod = 4, serial LED interface refresh every 20ms */
	reg_val32 = 0;
	((LedSerialCntrlRegister *)&reg_val32)->Bits.PortEn        = led_mask;
	((LedSerialCntrlRegister *)&reg_val32)->Bits.Smode         = 1;
	((LedSerialCntrlRegister *)&reg_val32)->Bits.RefreshPeriod = 4;

	__raw_writel(reg_val32,(u32 *)&(SWITCH_REG->LedSerialCntrl));

	/* program LED control for all PHY ports */
	/* SpdlnkLed0ActSel = 0x0, link dn = off, link up = on, Rx/Tx activity */
	/* TxEnEn/RxDvEn    = 0x1, enable Tx/Rx activity */
	/* SelxxxEncode     = 0x1, LED1 sel speed encode, LED0 sel activity */
	/* SpdxxxEncode     = 0xx, LED1 on for 1G speed only */
	reg_val32 = 0;
	((LedCntrlRegister *)&reg_val32)->Bits.SpdlnkLed0ActSel = 0x0;
	((LedCntrlRegister *)&reg_val32)->Bits.TxEnEn           = 0x1;
	((LedCntrlRegister *)&reg_val32)->Bits.RxDvEn           = 0x1;
	((LedCntrlRegister *)&reg_val32)->Bits.Sel1000mEncode   = 0x1;
	((LedCntrlRegister *)&reg_val32)->Bits.Sel100mEncode    = 0x1;
	((LedCntrlRegister *)&reg_val32)->Bits.Sel10mEncode     = 0x1;
	((LedCntrlRegister *)&reg_val32)->Bits.SelNoLinkEncode  = 0x1;
	((LedCntrlRegister *)&reg_val32)->Bits.Spd1000mEncode   = 0x0;
	((LedCntrlRegister *)&reg_val32)->Bits.Spd100mEncode    = 0x2;
	((LedCntrlRegister *)&reg_val32)->Bits.Spd10mEncode     = 0x2;
	((LedCntrlRegister *)&reg_val32)->Bits.SpdNoLinkEncode  = 0x3;

	for (i = 0; i < MAX_LED_PORT_CNTRL_REG; i++) {
		if (swdev->port[i].type == PHY_PORT ||
				swdev->port[i].type == MII_PORT) {
			__raw_writel(reg_val32,(u32 *)&(SWITCH_REG->LedCntrl[i]));
		}
	}
	if (swdev->port[7].type == PHY_PORT ||
			swdev->port[7].type == MII_PORT)
		__raw_writel(reg_val32,(u32 *)&(SWITCH_REG->Led7Cntrl));

	FUNC_LEAVE();
	return ret;
}

void __exit ethsw_led_board_exit_sf2(struct ethsw_device *swdev)
{
	FUNC_ENTER();

	FUNC_LEAVE();
}

int ethsw_led_powercntrl_sf2(struct ethsw_device *swdev, int port_id, bool state)
{
	int ret = 0;
	u32 reg_val32;

	FUNC_ENTER();

	/* program LED control for all PHY ports */
	reg_val32 = 0;

	switch ((int)state) {
	case LED_ON:
		((LedCntrlRegister *)&reg_val32)->Bits.SpdlnkLed0ActSel = 0x0;
		((LedCntrlRegister *)&reg_val32)->Bits.TxEnEn           = 0x1;
		((LedCntrlRegister *)&reg_val32)->Bits.RxDvEn           = 0x1;
		((LedCntrlRegister *)&reg_val32)->Bits.Sel1000mEncode   = 0x1;
		((LedCntrlRegister *)&reg_val32)->Bits.Sel100mEncode    = 0x1;
		((LedCntrlRegister *)&reg_val32)->Bits.Sel10mEncode     = 0x1;
		((LedCntrlRegister *)&reg_val32)->Bits.SelNoLinkEncode  = 0x1;
		((LedCntrlRegister *)&reg_val32)->Bits.Spd1000mEncode   = 0x0;
		((LedCntrlRegister *)&reg_val32)->Bits.Spd100mEncode    = 0x2;
		((LedCntrlRegister *)&reg_val32)->Bits.Spd10mEncode     = 0x2;
		((LedCntrlRegister *)&reg_val32)->Bits.SpdNoLinkEncode  = 0x3;

		if (swdev->port[port_id].type == PHY_PORT ||
				swdev->port[port_id].type == MII_PORT) {
			if (port_id == 7)
				__raw_writel(reg_val32,(u32 *)&(SWITCH_REG->Led7Cntrl));
			else
				__raw_writel(reg_val32,(u32 *)&(SWITCH_REG->LedCntrl[port_id]));
		}

		break;
	case LED_OFF:
		((LedCntrlRegister *)&reg_val32)->Bits.SpdlnkLed0ActSel = 0x0;
		((LedCntrlRegister *)&reg_val32)->Bits.TxEnEn           = 0x1;
		((LedCntrlRegister *)&reg_val32)->Bits.RxDvEn           = 0x1;
		((LedCntrlRegister *)&reg_val32)->Bits.Sel1000mEncode   = 0x0;
		((LedCntrlRegister *)&reg_val32)->Bits.Sel100mEncode    = 0x0;
		((LedCntrlRegister *)&reg_val32)->Bits.Sel10mEncode     = 0x0;
		((LedCntrlRegister *)&reg_val32)->Bits.SelNoLinkEncode  = 0x0;
		((LedCntrlRegister *)&reg_val32)->Bits.Spd1000mEncode   = 0x3;
		((LedCntrlRegister *)&reg_val32)->Bits.Spd100mEncode    = 0x3;
		((LedCntrlRegister *)&reg_val32)->Bits.Spd10mEncode     = 0x3;
		((LedCntrlRegister *)&reg_val32)->Bits.SpdNoLinkEncode  = 0x3;

		if (swdev->port[port_id].type == PHY_PORT ||
				swdev->port[port_id].type == MII_PORT) {
			if (port_id == 7)
				__raw_writel(reg_val32,(u32 *)&(SWITCH_REG->Led7Cntrl));
			else
				__raw_writel(reg_val32,(u32 *)&(SWITCH_REG->LedCntrl[port_id]));
		}

		break;
	default:
		break;
	}

	FUNC_LEAVE();

	return ret;
}

/*
 * Switch Interrupt Functions
 */

/* port L2 interrupt mapping */
struct port_int_map {
	int cntlr;
	int shift;
};

static struct port_int_map port_int_tbl[ETHSW_PORT_MAX] =
{
	{ 0,  0},
	{ 1, 25},
	{ 1, 20},
	{ 1, 15},
	{ 1, 10},
	{ 1,  5},
	{-1,  0},
	{ 1,  0},
	{-1,  0}
};

#define PORT_INT_SHIFT_LINK_UP          0
#define PORT_INT_SHIFT_LINK_DN          1
#define PORT_INT_SHIFT_ENERGY_ON        2
#define PORT_INT_SHIFT_ENERGY_OFF       3
#define PORT_INT_SHIFT_GPHY             4

#define PORT_INT_BIT_LINK_UP            (1 << PORT_INT_SHIFT_LINK_UP)
#define PORT_INT_BIT_LINK_DN            (1 << PORT_INT_SHIFT_LINK_DN)
#define PORT_INT_BIT_ENERGY_ON          (1 << PORT_INT_SHIFT_ENERGY_ON)
#define PORT_INT_BIT_ENERGY_OFF         (1 << PORT_INT_SHIFT_ENERGY_OFF)
#define PORT_INT_BIT_GPHY               (1 << PORT_INT_SHIFT_GPHY)

int ethsw_int_board_init_sf2(struct ethsw_device *swdev)
{
	int ret = 0;
	/*int i;*/

	FUNC_ENTER();
	FUNC_LEAVE();
	return ret;
}

void ethsw_int_board_exit_sf2(struct ethsw_device *swdev)
{
	FUNC_ENTER();

	FUNC_LEAVE();
}

int ethsw_int_board_isr_sf2(struct ethsw_device *swdev, int irq)
{
	int ret = 0;
	int i;
	volatile IntrL2Registers *pIntrL2;
	u32 reg_val32, reg1_val32;

	/* find IRQ index */
	for (i = 0; i < ETHSW_IRQ_MAX; i++) {
		if (irq == swdev->irq[i])
			break;
	}

	if (i == ETHSW_IRQ_MAX) {
		return -ENOENT;
	}

	/* get L2 controller */
	pIntrL2 = (i == 0) ?
		&SWITCH->IntrL20 :
		&SWITCH->IntrL21;

	reg_val32 = __raw_readl((u32 *)&(pIntrL2->CpuStatus));
	reg1_val32 = __raw_readl((u32 *)&(pIntrL2->CpuMaskStatus));
	/* save L2 status for bottom half */
	swdev->irq_status[i] =  reg_val32 & ~reg1_val32;

	/* clear L2 interrupts */
	__raw_writel(swdev->irq_status[i],(u32 *)&(pIntrL2->CpuClear));

	return ret;
}

int ethsw_int_board_proc_sf2(struct ethsw_device *swdev)
{
	int ret = 0;
	int i;

	/* check IRQ status for each port */
	for (i = 0; i < ETHSW_PHY_PORT_MAX; i++) {
		u32 irq_status;
		u32 port_status;
		struct ethsw_port *port;

		if (!(swdev->phy_port_mask & (1 << i)))
			continue;

		if (port_int_tbl[i].cntlr < 0)
			continue;

		irq_status = swdev->irq_status[port_int_tbl[i].cntlr];
		port_status = (irq_status >> port_int_tbl[i].shift) & 0x1F;

		if (!port_status)
			continue;

		port = &swdev->port[i];

		if (port->cb) {
			struct ethsw_port_status new_status;
			ethsw_port_status_get_sf2(swdev, i, &new_status);

			/* anything interesting changed? */
			if (new_status.link_up     == port->status.link_up &&
					new_status.full_duplex == port->status.full_duplex &&
					new_status.speed       == port->status.speed) {
				continue;
			}

			/* update port status */
			port->status = new_status;

			/* callback with port status */
			(*port->cb)(port->cb_priv, &port->status);
		}

		/* clear port status bits */
		swdev->irq_status[port_int_tbl[i].cntlr] =
			irq_status ^ (port_status << port_int_tbl[i].shift);
	}

	return ret;
}

int ethsw_int_port_enable_sf2(
		struct ethsw_device *swdev,
		int port_id, u32 int_map)
{
	int ret = 0;
	u32 irq_mask_clear;
	u32 port_mask_clear;
	volatile IntrL2Registers *pIntrL2;

	FUNC_ENTER();

	if (port_int_tbl[port_id].cntlr < 0) {
		ret = -ENOENT;
		goto EXIT;
	}

	/* get L2 controller */
	pIntrL2 = (port_int_tbl[port_id].cntlr == 0) ?
		&SWITCH->IntrL20 :
		&SWITCH->IntrL21;

	/* prepare L2 masks to be cleared */
	port_mask_clear =
		((int_map & PORT_INT_LINK_UP)    ? PORT_INT_BIT_LINK_UP    : 0) |
		((int_map & PORT_INT_LINK_DN)    ? PORT_INT_BIT_LINK_DN    : 0) |
		((int_map & PORT_INT_ENERGY_ON)  ? PORT_INT_BIT_ENERGY_ON  : 0) |
		((int_map & PORT_INT_ENERGY_OFF) ? PORT_INT_BIT_ENERGY_OFF : 0) |
		((int_map & PORT_INT_GPHY)       ? PORT_INT_BIT_GPHY       : 0);

	irq_mask_clear =
		port_mask_clear << port_int_tbl[port_id].shift;

	/* clear L2 interrupts for masks to be cleared */
	__raw_writel(irq_mask_clear, &(pIntrL2->CpuClear));

	/* clear L2 masks */
	__raw_writel(irq_mask_clear, &(pIntrL2->CpuMaskClear));

EXIT:
	FUNC_LEAVE();
	return ret;
}

int ethsw_port_tx_enable(struct ethsw_device *swdev, int port_id, int enable)
{
	u32 val;

	val = ethsw_reg_read(swdev, (BCHP_SWITCH_CORE_START + (port_id * 4)));

	if(enable == 0)
		val |= (1 << CREG_G_PCTL_TX_DIS_SHIFT);
	else
		val &= ~(1 << CREG_G_PCTL_TX_DIS_SHIFT);

	ethsw_reg_write(swdev, (BCHP_SWITCH_CORE_START + (port_id * 4)),val);

	return 0;

}

int ethsw_port_rx_enable(struct ethsw_device *swdev, int port_id, int enable)
{
	u32 val;

	val = ethsw_reg_read(swdev, (BCHP_SWITCH_CORE_START + (port_id * 4)));

	if(enable == 0)
		val |= (1 << CREG_G_PCTL_RX_DIS_SHIFT);
	else
		val &= ~(1 << CREG_G_PCTL_RX_DIS_SHIFT);

	ethsw_reg_write(swdev, (BCHP_SWITCH_CORE_START + (port_id * 4)),val);

	return 0;
}

int ethsw_int_port_disable_sf2(
		struct ethsw_device *swdev,
		int port_id, u32 int_map)
{
	int ret = 0;
	u32 irq_mask_set;
	u32 port_mask_set;
	volatile IntrL2Registers *pIntrL2;

	FUNC_ENTER();

	if (port_int_tbl[port_id].cntlr < 0) {
		ret = -ENOENT;
		goto EXIT;
	}

	/* get L2 controller */
	pIntrL2 = (port_int_tbl[port_id].cntlr) ?
		&SWITCH->IntrL20 :
		&SWITCH->IntrL21;

	/* prepare L2 masks to be set */
	port_mask_set =
		((int_map & PORT_INT_LINK_UP)    ? PORT_INT_BIT_LINK_UP    : 0) |
		((int_map & PORT_INT_LINK_DN)    ? PORT_INT_BIT_LINK_DN    : 0) |
		((int_map & PORT_INT_ENERGY_ON)  ? PORT_INT_BIT_ENERGY_ON  : 0) |
		((int_map & PORT_INT_ENERGY_OFF) ? PORT_INT_BIT_ENERGY_OFF : 0) |
		((int_map & PORT_INT_GPHY)       ? PORT_INT_BIT_GPHY       : 0);

	irq_mask_set =
		port_mask_set << port_int_tbl[port_id].shift;

	/* set L2 masks */
	__raw_writel(irq_mask_set, &(pIntrL2->CpuMaskSet));

EXIT:
	FUNC_LEAVE();
	return ret;
}

int ethsw_switch_set_port_stp_state_specific_sf2(struct ethsw_device *swdev,
		int port_id, unsigned int state)
{
	int ret = 0;
	u8 val;

	FUNC_ENTER();

	ret = ethsw_creg_read(swdev, PAGE_CTLREG, CREG_G_PCTL_P0 + port_id,
			&val, CREG_G_PCTL_SIZE);
	if (ret)
		goto EXIT;

	switch (state) {
	case BR_STATE_DISABLED:
		pr_debug("Switch port %d STP state set to disabled.\n",
				port_id);
		val = CREG_FLD_SET(CREG_G_PCTL, G_MISTP_STATE, val,
				CREG_G_PCTL_G_MISTP_STATE_NO_ST);
		break;
	case BR_STATE_BLOCKING:
		pr_debug("Switch port %d STP state set to blocking.\n",
				port_id);
		val = CREG_FLD_SET(CREG_G_PCTL, G_MISTP_STATE, val,
				CREG_G_PCTL_G_MISTP_STATE_BLOCKING);
		break;
	case BR_STATE_LISTENING:
		pr_debug("Switch port %d STP state set to listening.\n",
				port_id);
		val = CREG_FLD_SET(CREG_G_PCTL, G_MISTP_STATE, val,
				CREG_G_PCTL_G_MISTP_STATE_LISTENING);
		break;
	case BR_STATE_LEARNING:
		pr_debug("Switch port %d STP state set to learning.\n",
				port_id);
		val = CREG_FLD_SET(CREG_G_PCTL, G_MISTP_STATE, val,
				CREG_G_PCTL_G_MISTP_STATE_LEARNING);
		break;
	case BR_STATE_FORWARDING:
		pr_debug("Switch port %d STP state set to forwarding.\n",
				port_id);
		val = CREG_FLD_SET(CREG_G_PCTL, G_MISTP_STATE, val,
				CREG_G_PCTL_G_MISTP_STATE_FORWARDING);
		break;
	}
	ret = ethsw_creg_write(swdev, PAGE_CTLREG, CREG_G_PCTL_P0 + port_id,
			&val, CREG_G_PCTL_SIZE);
	if (ret)
		goto EXIT;

EXIT:
	FUNC_LEAVE();
	return ret;
}

/*
 * Switch Core Access Functions
 */

int ethsw_creg_read_sf2(
		struct ethsw_device *swdev,
		int page_num, int reg_addr, u8 *data, int size)
{
	int ret = 0;
	int rbus_woffset;
	u32 data32;
	unsigned long flags;
	struct ethsw_core *swcore = swdev->swcore;

	if (!swdev->ethsw_pwr_status)
		return -EIO;

	if (!swcore) {
		LOG_ERR("Ethsw switch core not initialized\n");
		return -EINVAL;
	}

	LOG_DEBUG("Switch core read: page_num = 0x%x, reg_addr = 0x%x, size = %d\n",
			page_num, reg_addr, size);

	if (page_num > 0xFF ||
			reg_addr > 0xFF ||
			size > 8) {
		LOG_ERR("Invalid parameters\n");
		return -EINVAL;
	}

	if (!data) {
		LOG_ERR("Data pointer is NULL\n");
		return -EINVAL;
	}

	/* lock core register access */
	spin_lock_irqsave(&swcore->creg_lock, flags);

	/* read lo 32-bit through RBUS address */
	rbus_woffset = (page_num << 8) | (reg_addr);

	data32 = __raw_readl((u32 *)SWITCH_CORE + rbus_woffset);

	LOG_DEBUG("Switch core read: value_lo = 0x%x\n", data32);

	switch (size) {
	case 1:
		*data = *(u8 *)&data32;
		break;
	case 2:
		*(u16 *)data = *(u16 *)&data32;
		break;
	case 4:
	case 6:
	case 8:
		*(u32 *)data = *(u32 *)&data32;
		break;
	default:
		ret = -EINVAL;
		goto EXIT;
	}

	if (size > 4) {
		/* read hi 32-bit through direct data */
		data32 = __raw_readl(&(SWITCH_REG->DirDataRead));

		//data32 = SWITCH_REG->DirDataRead;
		LOG_DEBUG("Switch core read: value_hi = 0x%x\n", data32);

		switch (size) {
		case 6:
			*(u16 *)(data + 4) = *(u16 *)&data32;
			break;
		case 8:
			*(u32 *)(data + 4) = *(u32 *)&data32;
			break;
		}
	}

EXIT:
	spin_unlock_irqrestore(&swcore->creg_lock, flags);
	return ret;
}

int ethsw_creg_write_sf2(
		struct ethsw_device *swdev,
		int page_num, int reg_addr, u8 *data, int size)
{
	int ret = 0;
	int rbus_woffset;
	u32 data32;
	unsigned long flags;
	struct ethsw_core *swcore = swdev->swcore;

	if (!swdev->ethsw_pwr_status)
		return -EIO;

	if (!swcore) {
		LOG_ERR("Ethsw switch core not initialized\n");
		return -EINVAL;
	}

	LOG_DEBUG("Switch core write: page_num = 0x%x, reg_addr = 0x%x, size = %d\n",
			page_num, reg_addr, size);

	if (page_num > 0xFF ||
			reg_addr > 0xFF ||
			size > 8) {
		LOG_ERR("Invalid parameters\n");
		return -EINVAL;
	}

	if (!data) {
		LOG_ERR("Data pointer is NULL\n");
		return -EINVAL;
	}

	/* lock core register access */
	spin_lock_irqsave(&swcore->creg_lock, flags);

	if (size > 4) {
		switch (size) {
		case 6:
			data32 = *(u16 *)(data + 4);
			break;
		case 8:
			data32 = *(u32 *)(data + 4);
			break;
		default:
			ret = -EINVAL;
			goto EXIT;
		}

		/* write hi 32-bit into direct data */
		__raw_writel(data32, &(SWITCH_REG->DirDataWrite));

		LOG_DEBUG("Switch core write: value_hi = 0x%x\n", data32);
	}

	switch (size) {
	case 1:
		data32 = *(u8 *)data;
		break;
	case 2:
		data32 = *(u16 *)data;
		break;
	case 4:
	case 6:
	case 8:
		data32 = *(u32 *)data;
		break;
	default:
		ret = -EINVAL;
		goto EXIT;
	}

	/* write lo 32-bit into RBUS address */
	rbus_woffset = (page_num << 8) | (reg_addr);
	__raw_writel(data32,(u32 *)SWITCH_CORE + rbus_woffset);

	LOG_DEBUG("Switch core write: value_lo = 0x%x\n", data32);

EXIT:
	spin_unlock_irqrestore(&swcore->creg_lock, flags);
	return ret;
}
