 /****************************************************************************
 *
 * Copyright (c) 2015-2018 Broadcom. All rights reserved
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Jayesh Patel <jayeshp@broadcom.com>
 ****************************************************************************/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/inet.h>
#include <linux/netdevice.h>
#include <linux/if_arp.h>
#include <linux/uaccess.h>
#include <linux/buffer_head.h>
#include "flowmgr.h"
#include "flowmgr_fap.h"
#include "flowmgr_fap_ops.h"
#include "flowmgr_tunnel.h"
#include <proc_cmd.h>
#include "bcmflowmap.h"

#define FLOWMGR_PROC_DIR_NAME	"driver/flowmgr"
#define BOLDBLACK   "\033[1m\033[30m"      /* Bold Black */
#define RESET   	"\033[0m"

extern u64 gdscp_notrack;

static int dev_index_by_name(const char *name)
{
	struct net_device *dev;
	dev = __dev_get_by_name(&init_net,
				name);
	if (dev)
		return dev->ifindex;
	else
		return -1;
}

static u8 flowtype_from_string(const char *str)
{
	u8 flowtype = 0;
	if (kstrtou8(str, 0, &flowtype) != 0) {
		if (strcasecmp(str, "ipv4") == 0)
			return ft_ipv4;
		else if (strcasecmp(str, "ipv6") == 0)
			return ft_ipv6;
		else if (strcasecmp(str, "bridge") == 0)
			return ft_mac_bridge;
		else if (strcasecmp(str, "mcast") == 0)
			return ft_multicast;
	}
	if ((flowtype > ft_none) && (flowtype < ft_max))
		return flowtype;
	else
		return 0;
}

static u8 ip_prot_from_string(const char *str)
{
	u8 prot = 0;
	if (kstrtou8(str, 0, &prot) != 0) {
		if (strcasecmp(str, "udp") == 0)
			return IPPROTO_UDP;
		else if (strcasecmp(str, "tcp") == 0)
			return IPPROTO_TCP;
	}
	if (prot == IPPROTO_UDP)
		return prot;
	else if (prot == IPPROTO_TCP)
		return prot;
	else
		return 0;
}

static void vlan_by_interface(int *rxinf, int *txinf, u16 *vlan_tci, u8 *vlan_untag)
{
	struct net_device *rxdev = __dev_get_by_index(&init_net, *rxinf);
	struct net_device *txdev = __dev_get_by_index(&init_net, *txinf);

	if (!rxdev || !txdev) {
		pr_info("rx net device (or) tx net device invalid\n");
		return;
    }
	if (is_vlan_dev(rxdev) && !is_vlan_dev(txdev)) {
		*vlan_tci = 0;
		*vlan_untag = 1;
		*rxinf = vlan_dev_priv(rxdev)->real_dev->ifindex;
	} else if (!is_vlan_dev(rxdev) && is_vlan_dev(txdev)) {
		*vlan_tci = vlan_dev_vlan_id(txdev);
		*vlan_untag = 0;
		*txinf = vlan_dev_priv(txdev)->real_dev->ifindex;
	}
	return;
}

static int flowmgr_cmd_add_flow_ipv4(int argc, char *argv[])
{
	int flow_id;
	struct flow_params params;
	struct flow_ipv4_params *fparams;
	int ret = -1;
	int rxinf;
	int txinf;
	u8 dscp = 0;
	memset(&params, 0, sizeof(params));
	fparams = &params.ipv4;
	params.type = ft_ipv4;
	fparams->ip_prot = ip_prot_from_string(argv[3]);
	rxinf = dev_index_by_name(argv[12]);
	txinf = dev_index_by_name(argv[13]);
	if (rxinf  < 0) {
		pr_info("Adding Flow: wrong value - rx interface name\n");
	} else if (txinf  < 0) {
		pr_info("Adding Flow: wrong value - tx interface name\n");
	} else if (kstrtoint(argv[2], 0, (int *)&fparams->type) != 0) {
		pr_info("Adding Flow: wrong value - IPv4 flow type\n");
	} else if (fparams->ip_prot == 0) {
		pr_info("Adding Flow: wrong value - ip_prot\n");
	} else if (in4_pton(argv[4], strlen(argv[4]),
			    (u8 *)&fparams->src_ip,
			     -1, NULL) == 0) {
		pr_info("Adding Flow: wrong value - src_ip\n");
	} else if (in4_pton(argv[5], strlen(argv[5]),
			    (u8 *)&fparams->dst_ip,
			     -1, NULL) == 0) {
		pr_info("Adding Flow: wrong value - dst_ip\n");
	} else if (kstrtou16(argv[6], 0, &fparams->src_port) != 0) {
		pr_info("Adding Flow: wrong value - src_port\n");
	} else if (kstrtou16(argv[7], 0, &fparams->dst_port) != 0) {
		pr_info("Adding Flow: wrong value - dst_port\n");
	} else if (in4_pton(argv[8], strlen(argv[8]),
			    (u8 *)&fparams->replacement_ip,
			     -1, NULL) == 0) {
		pr_info("Adding Flow: wrong value - replacement_ip\n");
	} else if (kstrtou16(argv[9], 0, &fparams->replacement_port) != 0) {
		pr_info("Adding Flow: wrong value - replacement_port\n");
	} else if (mac_pton(argv[10], fparams->replacement_mac_src) == 0) {
		pr_info("Adding Flow: wrong value - replacement_mac_src\n");
	} else if (mac_pton(argv[11], fparams->replacement_mac_dst) == 0) {
		pr_info("Adding Flow: wrong value - replacement_mac_dst\n");
	} else if (kstrtou8(argv[14], 0, &dscp) != 0) {
		pr_info("Adding Flow: wrong value - dscp\n");
	} else {
		vlan_by_interface(&rxinf, &txinf, &params.tx.control.vlan_tci, &params.rx.control.vlan_untag);
		params.rx.interface = rxinf;
		params.tx.interface = txinf;
		if (dscp) {
			params.tx.control.dscp_mark = 1;
			params.tx.control.dscp_val = dscp;
		}
		fparams->src_port = htons(fparams->src_port);
		fparams->dst_port = htons(fparams->dst_port);
		fparams->replacement_port = htons(fparams->replacement_port);

		ret = fap()->flow_add(&params, &flow_id);
		if (ret == 0)
			pr_info("Flow (ipv4) added: %d\n", flow_id);
	}
	return ret;
}

static int flowmgr_cmd_add_flow_ipv4_tunnel(int argc, char *argv[])
{
	int flow_id;
	struct flow_params params;
	struct flow_ipv4_params *fparams;
	struct flow_tunnel_params *tparams;
	int ret = -1;
	int rxinf;
	int txinf;
	struct flowmgr_tdb_entry *db;
	__be16 vlan_tag;
	memset(&params, 0, sizeof(params));
	fparams = &params.ipv4;
	tparams = &params.encap_tun;
	params.type = ft_ipv4;
	fparams->ip_prot = ip_prot_from_string(argv[4]);
	rxinf = dev_index_by_name(argv[13]);
	txinf = dev_index_by_name(argv[14]);
	if (rxinf  < 0) {
		pr_info("Adding Flow: wrong value - rx interface name\n");
	} else if (txinf  < 0) {
		pr_info("Adding Flow: wrong value - tx interface name\n");
	} else if (kstrtoint(argv[3], 0, (int *)&fparams->type) != 0) {
		pr_info("Adding Flow: wrong value - IPv4 flow type\n");
	} else if (fparams->ip_prot == 0) {
		pr_info("Adding Flow: wrong value - ip_prot\n");
	} else if (in4_pton(argv[5], strlen(argv[5]),
			    (u8 *)&fparams->src_ip,
			     -1, NULL) == 0) {
		pr_info("Adding Flow: wrong value - src_ip\n");
	} else if (in4_pton(argv[6], strlen(argv[6]),
			    (u8 *)&fparams->dst_ip,
			     -1, NULL) == 0) {
		pr_info("Adding Flow: wrong value - dst_ip\n");
	} else if (kstrtou16(argv[7], 0, &fparams->src_port) != 0) {
		pr_info("Adding Flow: wrong value - src_port\n");
	} else if (kstrtou16(argv[8], 0, &fparams->dst_port) != 0) {
		pr_info("Adding Flow: wrong value - dst_port\n");
	} else if (in4_pton(argv[9], strlen(argv[9]),
			    (u8 *)&fparams->replacement_ip,
			     -1, NULL) == 0) {
		pr_info("Adding Flow: wrong value - replacement_ip\n");
	} else if (kstrtou16(argv[10], 0, &fparams->replacement_port) != 0) {
		pr_info("Adding Flow: wrong value - replacement_port\n");
	} else if (mac_pton(argv[11], fparams->replacement_mac_src) == 0) {
		pr_info("Adding Flow: wrong value - replacement_mac_src\n");
	} else if (mac_pton(argv[12], fparams->replacement_mac_dst) == 0) {
		pr_info("Adding Flow: wrong value - replacement_mac_dst\n");
	} else {
		params.rx.interface = rxinf;
		params.tx.interface = txinf;
		fparams->src_port = htons(fparams->src_port);
		fparams->dst_port = htons(fparams->dst_port);
		fparams->replacement_port = htons(fparams->replacement_port);

		/* configure tunnel */
		if (fparams->type == ft_ipv4_gre_decap || fparams->type == ft_ipv4_gre_encap) {
			__be32 remote_ip[4];
			if (in4_pton(argv[15], strlen(argv[15]), (u8 *)remote_ip, -1, NULL) == 0) {
				pr_info("Adding Flow: wrong value - remote_ip\n");
				return ret;
			}
			if (kstrtou16(argv[16], 0, &vlan_tag) != 0) {
				pr_info("Adding Flow: wrong value - vlan\n");
				return ret;
			}
			db = flowmgr_tdb_find_by_raddr(remote_ip);
			if (!db) {
				pr_info("Adding Flow: no matching db for input remote ip\n");
				return ret;
			}
			tparams->gre.tunnel_id = db->id;
			if (fparams->type == ft_ipv4_gre_encap) {
				tparams->type = ARPHRD_ETHER;
				memcpy(tparams->gre.remote_mac, db->parms.eh.h_dest,
				       ETH_ALEN);
				memcpy(tparams->gre.local_mac, db->parms.eh.h_source,
				       ETH_ALEN);
				tparams->gre.remote_ip[0] = db->parms.raddr.in.s_addr;
				tparams->gre.local_ip[0] = db->parms.laddr.in.s_addr;
				params.tx.control.vlan_tci = vlan_tag;
				tparams->gre.vlan_tag = vlan_tag;
			} else if (vlan_tag) {
				params.rx.control.vlan_untag = 1;
				tparams->gre.vlan_tag = 1;
			}
		} else if (fparams->type == ft_ipv4_dslite_encap) {
			__be32 remote_ip6[4];
			if (in6_pton(argv[15], strlen(argv[15]), (u8 *)&remote_ip6, -1, NULL) == 0) {
				pr_info("Adding Flow: wrong value - remote_ip\n");
				return ret;
			}
			db = flowmgr_tdb_find_by_raddr(remote_ip6);
			if (!db) {
				pr_info("Adding Flow: no matching db for input remote ip\n");
				return ret;
			}
			tparams->type = ARPHRD_TUNNEL6;
			memcpy(tparams->dslite.remote_mac, db->parms.eh.h_dest,
			       ETH_ALEN);
			memcpy(tparams->dslite.local_mac, db->parms.eh.h_source,
			       ETH_ALEN);
			memcpy((char *)tparams->dslite.remote_ip,
			       db->parms.raddr.in6.s6_addr,
			       sizeof(struct in6_addr));
			memcpy((char *)tparams->dslite.local_ip,
			       db->parms.laddr.in6.s6_addr,
			       sizeof(struct in6_addr));
		}

		ret = fap()->flow_add(&params, &flow_id);
		if (ret == 0)
			pr_info("Flow (ipv4 tunnel) added: %d\n", flow_id);
	}
	return ret;
}

static int flowmgr_cmd_add_flow_mac_bridge(int argc, char *argv[])
{
	int flow_id;
	struct flow_params params;
	struct flow_mac_bridge_params *fparams;
	int ret = -1;
	int rxinf;
	int txinf;
	fparams = &params.mac_bridge;
	params.type = ft_mac_bridge;
	rxinf = dev_index_by_name(argv[4]);
	txinf = dev_index_by_name(argv[5]);
	if (rxinf  < 0) {
		pr_info("Adding Flow: wrong value - rx interface name\n");
	} else if (txinf  < 0) {
		pr_info("Adding Flow: wrong value - tx interface name\n");
	} else if (mac_pton(argv[2], fparams->mac_src) == 0) {
		pr_info("Adding Flow: wrong value - src_mac_address\n");
	} else if (mac_pton(argv[3], fparams->mac_dst) == 0) {
		pr_info("Adding Flow: wrong value - dst_mac_address\n");
	} else {
		fparams->rx_interface = rxinf;
		params.tx.interface = txinf;
		ret = fap()->flow_add(&params, &flow_id);
		if (ret == 0)
			pr_info("Flow (mac_bridge) added: %d\n", flow_id);
	}
	return ret;
}

static int flowmgr_cmd_add_flow_ipv6(int argc, char *argv[])
{
	int flow_id;
	struct flow_params params;
	struct flow_ipv6_params *fparams;
	int ret = -1;
	int rxinf;
	int txinf;
	u8 dscp = 0;
	memset(&params, 0, sizeof(params));
	fparams = &params.ipv6;
	params.type = ft_ipv6;
	fparams->ip_prot = ip_prot_from_string(argv[3]);
	rxinf = dev_index_by_name(argv[10]);
	txinf = dev_index_by_name(argv[11]);
	if (rxinf  < 0) {
		pr_info("Adding Flow: wrong value - rx interface name\n");
	} else if (txinf  < 0) {
		pr_info("Adding Flow: wrong value - tx interface name\n");
	} else if (kstrtoint(argv[2], 0, (int *)&fparams->type) != 0) {
		pr_info("Adding Flow: wrong value - IPv6 flow type\n");
	} else if (fparams->ip_prot == 0) {
		pr_info("Adding Flow: wrong value - ip_prot\n");
	} else if (in6_pton(argv[4], strlen(argv[4]),
			    (u8 *)&fparams->src_ip,
			     -1, NULL) == 0) {
		pr_info("Adding Flow: wrong value - src_ip\n");
	} else if (in6_pton(argv[5], strlen(argv[5]),
			    (u8 *)&fparams->dst_ip,
			     -1, NULL) == 0) {
		pr_info("Adding Flow: wrong value - dst_ip\n");
	} else if (kstrtou16(argv[6], 0, &fparams->src_port) != 0) {
		pr_info("Adding Flow: wrong value - src_port\n");
	} else if (kstrtou16(argv[7], 0, &fparams->dst_port) != 0) {
		pr_info("Adding Flow: wrong value - dst_port\n");
	} else if (mac_pton(argv[8], fparams->replacement_mac_src) == 0) {
		pr_info("Adding Flow: wrong value - replacement_mac_src\n");
	} else if (mac_pton(argv[9], fparams->replacement_mac_dst) == 0) {
		pr_info("Adding Flow: wrong value - replacement_mac_dst\n");
	} else if (kstrtou8(argv[12], 0, &dscp) != 0) {
		pr_info("Adding Flow: wrong value - dscp\n");
	} else {
		vlan_by_interface(&rxinf, &txinf, &params.tx.control.vlan_tci, &params.rx.control.vlan_untag);
		params.rx.interface = rxinf;
		params.tx.interface = txinf;
		if (dscp) {
			params.tx.control.dscp_mark = 1;
			params.tx.control.dscp_val = dscp;
		}
		fparams->src_port = htons(fparams->src_port);
		fparams->dst_port = htons(fparams->dst_port);
		ret = fap()->flow_add(&params, &flow_id);
		if (ret == 0)
			pr_info("Flow (ipv6) added: %d\n", flow_id);
	}
	return ret;
}

static void cmd_add_help(char *str)
{
	pr_alert("%s flow_add: Manual addition/promotion of various flows\n", str);
	pr_alert("%s  For adding ipv4 flow:\n", str);
	pr_alert("%s   flow_add ipv4 <type:1|2|3|8|9|10>\n", str);
	pr_alert("%s    <ip_prot:udp|tcp> <src_ip> <dst_ip> <src_port> <dst_port>\n", str);
	pr_alert("%s    <rep_ip> <rep_port> <rep_mac_src> <rep_mac_dst>\n", str);
	pr_alert("%s    <rx_inf_name> <tx_inf_name> <dscp>\n", str);
	pr_alert("%s    optional: <tunnel_remote_ip> <gre_tunnel_vlan>\n", str);
	pr_alert("%s    type 1  Ipv4 Bridged Flow            \n", str);
	pr_alert("%s    type 2  Nat Mod Dst (Wan->Lan)       \n", str);
	pr_alert("%s    type 3  Nat Mod Src (Lan->Wan)       \n", str);
	pr_alert("%s    type 8  IPv4 Routed Wan->Lan         \n", str);
	pr_alert("%s    type 9  IPv4 Routed Lan->WAN         \n", str);
	pr_alert("%s    type 10 Ipv4 Downstream Bridged Flow \n", str);

	pr_alert("%s  For adding ipv4 tunnel flow:\n", str);
	pr_alert("%s   flow_add ipv4 tunnel <type:4|5|6|7>\n", str);
	pr_alert("%s    <ip_prot:udp|tcp> <src_ip> <dst_ip> <src_port> <dst_port>\n", str);
	pr_alert("%s    <rep_ip> <rep_port> <rep_mac_src> <rep_mac_dst>\n", str);
	pr_alert("%s    <rx_inf_name> <tx_inf_name>\n", str);
	pr_alert("%s    <tunnel_remote_ip> <tunnel_vlan>\n", str);
	pr_alert("%s    type 4  Ds-Lite Decapsulate Ipv6 Header              \n", str);
	pr_alert("%s    type 5  Ds-Lite Encapsulate with Ipv6 Header         \n", str);
	pr_alert("%s    type 6  GRE Decapsulate Ipv4 Header no NAT           \n", str);
	pr_alert("%s    type 7  GRE Encapsulate with Ipv4 Header no NAT      \n", str);
	pr_alert("%s    type 13 GRE Decapsulate Ipv4 Header WANGRE->LAN      \n", str);
	pr_alert("%s    type 14 GRE Encapsulate with Ipv4 Header LAN->WANGRE \n", str);
	pr_alert("%s    type 15 GRE Decapsulate Ipv4 Header LANGRE->WAN      \n", str);
	pr_alert("%s    type 16 GRE Encapsulate with Ipv4 Header WAN->LANGRE \n", str);

	pr_alert("%s  For adding bridge flow:\n", str);
	pr_alert("%s   flow_add bridge <mac_src> <mac_dst>\n", str);
	pr_alert("%s    <rx_inf_name> <tx_inf_name>\n", str);

	pr_alert("%s  For adding ipv6:\n", str);
	pr_alert("%s   flow_add ipv6 <type:1|2|3>\n", str);
	pr_alert("%s    <ip_prot:udp|tcp> <src_ip> <dst_ip> <src_port> <dst_port>\n", str);
	pr_alert("%s    <rep_mac_src> <rep_mac_dst>\n", str);
	pr_alert("%s    <rx_inf_name> <tx_inf_name> <dscp>\n", str);
	pr_alert("%s    type 1  Ipv6 Bridged Flow            \n", str);
	pr_alert("%s    type 2  Ipv6 Routed Flow             \n", str);
	pr_alert("%s    type 3  Ipv6 Downstream Bridged Flow \n", str);
}

static int cmd_add(int argc, char *argv[])
{
	if (argc > 2) {
		int type, ret = -1;
		type = flowtype_from_string(argv[1]);
		pr_info("Adding Flow Type %d\n", type);
		if (type == ft_ipv4) {
			if ((strcasecmp(argv[2], "tunnel") != 0) && argc == 15)
				ret = flowmgr_cmd_add_flow_ipv4(argc,
								argv);
			else if (argc == 17)
				ret = flowmgr_cmd_add_flow_ipv4_tunnel(argc,
								argv);
		} else if ((type == ft_ipv6) && argc == 13) {
			ret = flowmgr_cmd_add_flow_ipv6(argc,
							argv);
		} else if ((type == ft_mac_bridge) && argc == 6) {
			ret = flowmgr_cmd_add_flow_mac_bridge(argc,
							      argv);
		}
		if (ret == 0)
			goto done;
		pr_info("Error Adding Flow: ret=%d\n", ret);
	}
/* help */
	cmd_add_help("");
done:
	return 0;
}

static int flowmgr_cmd_add_flow_ipv4_expected(int argc, char *argv[])
{
	int flow_id;
	struct flow_params params;
	struct flow_ipv4_params *fparams;
	int ret = -1;
	int rxinf = dev_index_by_name(argv[7]);
	int txinf = dev_index_by_name(argv[8]);
	fparams = &params.ipv4;
	params.type = ft_ipv4;
	fparams->type = ft_ipv4_expected;
	fparams->ip_prot = ip_prot_from_string(argv[2]);
	if (fparams->ip_prot == 0) {
		pr_info("Adding Flow: wrong value - ip_prot\n");
	} else if (in4_pton(argv[3], strlen(argv[3]),
			    (u8 *)&fparams->src_ip,
			     -1, NULL) == 0) {
		pr_info("Adding Flow: wrong value - src_ip\n");
	} else if (in4_pton(argv[4], strlen(argv[4]),
			    (u8 *)&fparams->dst_ip,
			     -1, NULL) == 0) {
		pr_info("Adding Flow: wrong value - dst_ip\n");
	} else if (kstrtou16(argv[5], 0, &fparams->src_port) != 0) {
		pr_info("Adding Flow: wrong value - src_port\n");
	} else if (kstrtou16(argv[6], 0, &fparams->dst_port) != 0) {
		pr_info("Adding Flow: wrong value - dst_port\n");
	} else {
		params.rx.interface = rxinf;
		params.tx.interface = txinf;
		fparams->src_port = htons(fparams->src_port);
		fparams->dst_port = htons(fparams->dst_port);
		ret = fap()->flow_add_expected(&params, &flow_id);
		if (ret == 0)
			pr_info("Flow (ipv4) expected added: %d\n", flow_id);
	}
	return ret;
}

static int flowmgr_cmd_add_flow_ipv6_expected(int argc, char *argv[])
{
	int flow_id;
	struct flow_params params;
	struct flow_ipv6_params *fparams;
	int ret = -1;
	int rxinf = dev_index_by_name(argv[7]);
	int txinf = dev_index_by_name(argv[8]);
	fparams = &params.ipv6;
	params.type = ft_ipv6;
	fparams->type = ft_ipv6_expected;
	fparams->ip_prot = ip_prot_from_string(argv[2]);
	if (fparams->ip_prot == 0) {
		pr_info("Adding Flow: wrong value - ip_prot\n");
	} else if (in6_pton(argv[3], strlen(argv[3]),
			    (u8 *)&fparams->src_ip,
			     -1, NULL) == 0) {
		pr_info("Adding Flow: wrong value - src_ip\n");
	} else if (in6_pton(argv[4], strlen(argv[4]),
			    (u8 *)&fparams->dst_ip,
			     -1, NULL) == 0) {
		pr_info("Adding Flow: wrong value - dst_ip\n");
	} else if (kstrtou16(argv[5], 0, &fparams->src_port) != 0) {
		pr_info("Adding Flow: wrong value - src_port\n");
	} else if (kstrtou16(argv[6], 0, &fparams->dst_port) != 0) {
		pr_info("Adding Flow: wrong value - dst_port\n");
	} else {
		params.rx.interface = rxinf;
		params.tx.interface = txinf;
		fparams->src_port = htons(fparams->src_port);
		fparams->dst_port = htons(fparams->dst_port);
		ret = fap()->flow_add_expected(&params, &flow_id);
		if (ret == 0)
			pr_info("Flow (ipv6) expected added: %d\n", flow_id);
	}
	return ret;
}

static void cmd_add_expected_help(char *str)
{
	pr_alert("%s flow_add_expected: Manual addition/promotion of expected flow\n", str);
	pr_alert("%s  flow_add_expected ipv4|ipv6 <ip_prot:udp|tcp>\n", str);
	pr_alert("%s   <src_ip> <dst_ip> <src_port> <dst_port>\n", str);
	pr_alert("%s   <rx_inf_name> <tx_inf_name>\n", str);
}

static int cmd_add_expected(int argc, char *argv[])
{
	if (!fap()->flow_add_expected)
		pr_err("Expected flow not supported\n");

	if (argc > 2) {
		int type, ret = -1;
		type = flowtype_from_string(argv[1]);
		pr_info("Adding Flow Type %d\n", type);
		if ((type == ft_ipv4) && argc == 9) {
			ret = flowmgr_cmd_add_flow_ipv4_expected(argc,
								 argv);
		} else if ((type == ft_ipv6) && argc == 9) {
			ret = flowmgr_cmd_add_flow_ipv6_expected(argc,
								 argv);
		}
		if (ret == 0)
			goto done;
		pr_info("Error Adding Flow: ret=%d\n", ret);
	}
/* help */
	cmd_add_expected_help("");
done:
	return 0;
}
static void cmd_del_help(char *str)
{
	pr_alert("%s flow_del: Manual deletion of flow\n", str);
	pr_alert("%s  flow_del <flow_id>\n", str);
}
static int cmd_del(int argc, char *argv[])
{
	if (argc > 1) {
		int index;
		if (kstrtoint(argv[1], 0, &index) == 0) {
			fap()->flow_remove(index);
			pr_info("Flow %d Deleted\n", index);
			goto done;
		}
	}
/* help */
	cmd_del_help("");
done:
	return 0;
}
static void cmd_flushmac_help(char *str)
{
	pr_alert("%s flushmac: Manual deletion of all flows with sr/dst mac associated with specific interface\n", str);
	pr_alert("%s  flushmac <mac_address> <dev name> <mode: 1, 2(default), 3>\n", str);
	pr_alert("%s    mode  1: Delete from FAP: Fast->Slow\n", str);
	pr_alert("%s    mode  2: Delete from FAP and Unlearn flow: Fast->Slow->Fast for both US and DS flows\n", str);
	pr_alert("%s    mode  3: Remove conntrack: Fast->Slow->Fast for upstream flows maybe with different NAT\n", str);
}
static int cmd_flushmac(int argc, char *argv[])
{
	if (argc > 2) {
		unsigned char mac[6];
		int mode = -1;
		if (argc < 4)
			mode = 2;
		else if (kstrtoint(argv[3], 0, &mode) != 0)
			mode = 2;

		if ((mode > 0) && (mode < 4) && mac_pton(argv[1], mac)) {
			int inf = dev_index_by_name(argv[2]);
			if (inf > 0) {
				flowmgr_flow_flush(&init_net, mac,
						   inf, mode);
				goto done;
			}
		}
	}
/* help */
	cmd_flushmac_help("");
done:
	return 0;
}
static void cmd_flushinf_help(char *str)
{
	pr_alert("%s flushinf: Manual deletion of all flows with sr/dst mac associated with specific interface\n", str);
	pr_alert("%s  flushinf <dev name> <mode: 1, 2(default), 3>\n", str);
	pr_alert("%s    mode  1: Delete from FAP: Fast->Slow\n", str);
	pr_alert("%s    mode  2: Delete from FAP and Unlearn flow: Fast->Slow->Fast for both US and DS flows\n", str);
	pr_alert("%s    mode  3: Remove conntrack: Fast->Slow->Fast for upstream flows maybe with different NAT\n", str);
}
static int cmd_flushinf(int argc, char *argv[])
{
	if (argc > 1) {
		int mode = -1;
		if (argc < 3)
			mode = 2;
		else if (kstrtoint(argv[2], 0, &mode) != 0)
			mode = 2;

		if ((mode > 0) && (mode < 4)) {
			int inf = dev_index_by_name(argv[1]);
			if (inf > 0) {
				flowmgr_flow_flush(&init_net, NULL,
						   inf, mode);
				goto done;
			}
		}
	}
/* help */
	cmd_flushinf_help("");
done:
	return 0;
}
static void cmd_flushwifi_help(char *str)
{
	pr_alert("%s flushwifi: Delete all flows with sr/dst mac associated with specific wifi flring\n", str);
	pr_alert("%s flushwifi <mac_address> <wifi flring> <mode: 1, 2(default), 3>\n", str);
	pr_alert("%s   mode  1: Delete from FAP: Fast->Slow\n", str);
	pr_alert("%s   mode  2: Delete from FAP and Unlearn flow: Fast->Slow->Fast for both US and DS flows\n", str);
	pr_alert("%s   mode  3: Remove conntrack: Fast->Slow->Fast for upstream flows maybe with different NAT\n", str);
}
static int cmd_flushwifi(int argc, char *argv[])
{
	if (argc > 2) {
		unsigned char mac[6];
		int mode = -1;
		if (argc < 4)
			mode = 2;
		else if (kstrtoint(argv[3], 0, &mode) != 0)
			mode = 2;

		if ((mode > 0) && (mode < 4) && mac_pton(argv[1], mac)) {
			int flring;
			if (kstrtoint(argv[2], 0, &flring) == 0) {
				flowmgr_flow_flush_wifi(&init_net, mac,
							flring, mode);
				goto done;
			}
		}
	}
/* help */
	cmd_flushwifi_help("");
done:
	return 0;
}
static void cmd_counter_help(char *str)
{
	pr_alert("%s flow_counter: Get flow counter with flow id\n", str);
	pr_alert("%s  flow_counter <flow_id>\n", str);
}
static int cmd_counter(int argc, char *argv[])
{
	if (argc > 1) {
		int index;
		if (kstrtoint(argv[1], 0, &index) == 0) {
			u32 packets = 0;
			u32 bytes = 0;
			fap()->flow_get_counter(index, &packets,
						&bytes, false);
			pr_info("Flow %d:  %d %d\n", index, packets, bytes);
			goto done;
		}
	}
/* help */
	cmd_counter_help("");
done:
	return 0;
}
static void cmd_enable_help(char *str)
{
	pr_alert("%s fap_enable: Enable/Disable FAP\n", str);
	pr_alert("%s  fap_enable <1|0>\n", str);
}
static int cmd_enable(int argc, char *argv[])
{
	if (argc > 1) {
		int index;
		if (kstrtoint(argv[1], 0, &index) == 0) {
			fap()->enable(index);
			pr_info("FAP Pkt Replacement %d\n",
				index);
			goto done;
		}
	}
/* help */
	cmd_enable_help("");
done:
	return 0;
}
static void cmd_config_help(char *str)
{
	pr_alert("%s fap_config: Configure FAP\n", str);
	pr_alert("%s  fap_config <pri_lan_dev_name> <wan_dev_name> <sec_lan_dev_name>\n", str);
}
static int cmd_config(int argc, char *argv[])
{
	if (argc == 3) {
		flowmgr_config(argv[1], argv[2], NULL);
		goto done;
	} else if (argc == 4) {
		flowmgr_config(argv[1], argv[2], argv[3]);
		goto done;
	}
/* help */
	cmd_config_help("");
done:
	return 0;
}
static void cmd_gre_add_help(char *str)
{
	pr_alert("%s fap_gre_add: Add GRE V4/V6 Tunnel\n", str);
	pr_alert("%s  fap_gre_add <type:gre4|gre6> <localIp> <remoteIp> <tos> <ttl> <localMac> <remoteMac> <vlan> <mtu> <tx_inf>\n", str);
}
static int cmd_gre_add(int argc, char *argv[])
{
	struct flow_gre_tunnel_params gre_header;
	int ret = -1;
	int txinf = -1;

	if (argc > 9) {
		memset(&gre_header, 0,
		       sizeof(struct flow_gre_tunnel_params));
		gre_header.tunnel_id = 0;

		if (kstrtou8(argv[4], 0, &gre_header.tos) != 0) {
			pr_info("Adding GRE Tunnel: wrong value - tos\n");
		} else if (kstrtou8(argv[5], 0, &gre_header.ttl) != 0) {
			pr_info("Adding GRE Tunnel: wrong value - ttl\n");
		} else if (mac_pton(argv[6], gre_header.local_mac) == 0) {
			pr_info("Adding GRE Tunnel: wrong value - local_mac\n");
		} else if (mac_pton(argv[7], gre_header.remote_mac) == 0) {
			pr_info("Adding GRE Tunnel: wrong value - remote_mac\n");
		} else if (kstrtou16(argv[8], 0, &gre_header.vlan_tag) != 0) {
			pr_info("Adding GRE Tunnel: wrong value - vlan\n");
		} else if (kstrtou16(argv[9], 0, &gre_header.encap_mtu) != 0) {
			pr_info("Adding GRE Tunnel: wrong value - mtu\n");
		} else {
			if (strcasecmp("gre4", argv[1]) == 0) {
				gre_header.type = ft_ipv4;
				if (in4_pton(argv[2], strlen(argv[2]),
					    (u8 *)gre_header.local_ip,
					     -1, NULL) == 0) {
					pr_info("Adding GRE4 Tunnel: wrong value - local_ip\n");
					goto done;
				} else if (in4_pton(argv[3], strlen(argv[3]),
					    (u8 *)&gre_header.remote_ip,
					     -1, NULL) == 0) {
					pr_info("Adding GRE4 Tunnel: wrong value - remote_ip\n");
					goto done;
				}
			} else if (strcasecmp("gre6", argv[1]) == 0) {
				gre_header.type = ft_ipv6;
				if (in6_pton(argv[2], strlen(argv[2]),
					    (u8 *)gre_header.local_ip,
					     -1, NULL) == 0) {
					pr_info("Adding GRE6 Tunnel: wrong value - local_ip\n");
					goto done;
				} else if (in6_pton(argv[3], strlen(argv[3]),
					    (u8 *)&gre_header.remote_ip,
					     -1, NULL) == 0) {
					pr_info("Adding GRE6 Tunnel: wrong value - remote_ip\n");
					goto done;
				}
			}

			if (argc > 10)
				txinf = dev_index_by_name(argv[10]);

			ret = fap()->config_gre(&gre_header, txinf);
			if (ret == 0)
				pr_info("Tunnel GREv4 added: %d\n", gre_header.tunnel_id);
			goto done;
		}
	}
/* help */
	cmd_gre_add_help("");
done:
	return 0;
}
static void cmd_gre_del_help(char *str)
{
	pr_alert("%s fap_gre_del: Delete GRE V4/V6 Tunnel with tunnel id\n", str);
	pr_alert("%s  fap_gre_del <tunnel_id>\n", str);
}
static int cmd_gre_del(int argc, char *argv[])
{
	if (argc > 1) {
		int index;
		if (kstrtoint(argv[1], 0, &index) == 0) {
			fap()->unconfig_gre(index);
			pr_info("Tunnel GRE %d Deleted\n", index);
			goto done;
		}
	}
/* help */
	cmd_gre_del_help("");
done:
	return 0;
}
static void cmd_show_help(char *str)
{
	pr_alert("%s show: Show status information\n", str);
}
static int cmd_show(int argc, char *argv[])
{
	flowmgr_fap_dbg_show(NULL);
	flowmgr_debug_show(NULL);
	flowmgr_nfct_dbg_show(NULL);
	flowmgr_ignore_port_show(NULL);
	flowmgr_netdevice_show(NULL);
	flowmgr_trace_show(NULL);
	return 0;
}
static void cmd_ignore_port_help(char *str)
{
	pr_alert("%s ignore_port: Ignore TCP/UDP port for flow promotion\n", str);
	pr_alert("%s  ignore_port <add> <port_num1> .. <port_numN>\n", str);
	pr_alert("%s  ignore_port <del> <port_num1> .. <port_numN>\n", str);
	pr_alert("%s  ignore_port <clear>\n", str);
}
static int cmd_ignore_port(int argc, char *argv[])
{
	if (argc > 2) {
		if (strcasecmp("add", argv[1]) == 0) {
			int i;
			u16 port;
			for (i = 2; i < argc; i++) {
				if (kstrtou16(argv[i], 0, &port) == 0)
					flowmgr_ignore_port_add(port);
			}
			goto done;
		} else if (strcasecmp("del", argv[1]) == 0) {
			int i;
			u16 port;
			for (i = 2; i < argc; i++) {
				if (kstrtou16(argv[i], 0, &port) == 0)
					flowmgr_ignore_port_del(port);
			}
			goto done;
		} else if (strcasecmp("clear", argv[1]) == 0) {
			flowmgr_ignore_port_clear();
			goto done;
		}
	}
/* help */
	cmd_ignore_port_help("");
done:
	return 0;
}
static void cmd_inf_help(char *str)
{
	pr_alert("%s inf: Add/Delete interface for flow promotion learning\n", str);
	pr_alert("%s  inf <add|del> <dev_name> <optional:wan|lan>\n", str);
}
static int cmd_inf(int argc, char *argv[])
{
	if (argc > 2) {
		if (strcasecmp("add", argv[1]) == 0) {
			if (flowmgr_netdevice_add(argv[2], argv[3]) == 0)
				goto done;
		} else if (strcasecmp("del", argv[1]) == 0) {
			if (flowmgr_netdevice_del(argv[2]) == 0)
				goto done;
		}
	}
/* help */
	cmd_inf_help("");
done:
	return 0;
}
static void cmd_addif_help(char *str)
{
	pr_alert("%s addif: Add interface for flow promotion learning\n", str);
	pr_alert("%s  addif <dev_name> <optional:wan|lan>\n", str);
}
static int cmd_addif(int argc, char *argv[])
{
	if (argc > 1) {
		if (flowmgr_netdevice_add(argv[1], argv[2]) == 0)
			goto done;
		pr_alert("Flowmgr: addif %s %s - failed\n", argv[1], argv[2]);
		goto done;
	}
/* help */
	cmd_addif_help("");
done:
	return 0;
}
static void cmd_delif_help(char *str)
{
	pr_alert("%s delif: Delete interface for flow promotion learning\n", str);
	pr_alert("%s  delif <dev_name>\n", str);
}
static int cmd_delif(int argc, char *argv[])
{
	if (argc > 1) {
		if (flowmgr_netdevice_del(argv[1]) == 0)
			goto done;
		pr_alert("Flowmgr: delif %s - failed\n", argv[1]);
		goto done;
	}
/* help */
	cmd_delif_help("");
done:
	return 0;
}
static void cmd_ifdb_enable_help(char *str)
{
	pr_alert("%s ifdb_enable: Enable/Disable client (mac address) tracking per interface\n", str);
	pr_alert("%s  ifdb_enable <1|0> <dev_name>\n", str);
}
static int cmd_ifdb_enable(int argc, char *argv[])
{
	int enable;

	if (argc > 2) {
		if (kstrtoint(argv[1], 0, &enable) == 0 &&
		    (enable == 0 || enable == 1)) {
			struct net_device *dev;
			dev = __dev_get_by_name(&init_net,
						argv[2]);
			if (dev) {
				flowmgr_db_ifenable(dev, enable);
				goto done;
			}
		}
	}
/* help */
	cmd_ifdb_enable_help("");
done:
	return 0;
}
static void cmd_flushdb_help(char *str)
{
	pr_alert("%s flushdb: Flush client database\n", str);
	pr_alert("%s  Flowmgr must be disable to flush the database\n", str);
}
static int cmd_flushdb(int argc, char *argv[])
{
	flowmgr_db_flush();
	return 0;
}
static void cmd_delall_help(char *str)
{
	pr_alert("%s flow_delall: Manual deletion of all Flows\n", str);
}
static int cmd_delall(int argc, char *argv[])
{
	flowmgr_conntrack_clean_all();
	return 0;
}
#ifdef CONFIG_BCM_FLOWMGR_MCAST
static void cmd_mdb_add_help(char *str)
{
	pr_alert("%s mdb_add: Manual addition of multicast flow\n", str);
	pr_alert("%s  mdb_add <indev> <outdev> <group>\n", str);
}
static int cmd_mdb_add(int argc, char *argv[])
{
	if (argc > 3) {
		struct net_device *in;
		struct net_device *out;
		__be32 group[4];
		__u8 *mac = NULL;
		__u8 mc2uc[ETH_ALEN];
		in = __dev_get_by_name(&init_net, argv[1]);
		out = __dev_get_by_name(&init_net, argv[2]);

		memset(group, 0, sizeof(group));
		if (!in || !in->group) {
			pr_info("Invalid/Unsupported input device\n");
		} else if (!out || !out->group) {
			pr_info("Invalid/Unsupported output device\n");
		} else if (strstr(argv[3], ".") && in4_pton(argv[3], strlen(argv[3]),
			    (u8 *)&group[3],
			     -1, NULL)) {
			if (argc > 4) {
				if (mac_pton(argv[4], mc2uc))
					mac = mc2uc;
			}
			flowmgr_mdb_insert(in, out, group, mac);
			goto done;
		} else if (strstr(argv[3], ":") && in6_pton(argv[3], strlen(argv[3]),
			    (u8 *)&group[0],
			     -1, NULL)) {
			if (argc > 4) {
				if (mac_pton(argv[4], mc2uc))
					mac = mc2uc;
			}

			flowmgr_mdb_insert(in, out, group, mac);
			goto done;
		} else {
			pr_info("Adding Flow: wrong value - ip\n");
		}
	}
/* help */
	cmd_mdb_add_help("");
done:
	return 0;
}
static void cmd_mdb_del_help(char *str)
{
	pr_alert("%s mdb_del: Manual deletion of multicast flow\n", str);
	pr_alert("%s  mdb_del <indev> <outdev> <group>\n", str);
}
static int cmd_mdb_del(int argc, char *argv[])
{
	if (argc > 3) {
		struct net_device *in;
		struct net_device *out;
		__be32 group[4];
		__u8 *mac = NULL;
		__u8 mc2uc[ETH_ALEN];
		in = __dev_get_by_name(&init_net, argv[1]);
		out = __dev_get_by_name(&init_net, argv[2]);

		memset(group, 0, sizeof(group));
		if (!in || !in->group) {
			pr_info("Invalid/Unsupported input device\n");
		} else if (!out || !out->group) {
			pr_info("Invalid/Unsupported output device\n");
		} else if (strstr(argv[3], ".") &&
			   in4_pton(argv[3], strlen(argv[3]),
				    (u8 *)&group[3], -1, NULL)) {
			if (argc > 4) {
				if (mac_pton(argv[4], mc2uc))
					mac = mc2uc;
			}

			flowmgr_mdb_delete(in, out, group, mac);
			goto done;
		} else if (strstr(argv[3], ":") &&
			   in6_pton(argv[3], strlen(argv[3]),
				    (u8 *)&group[0], -1, NULL)) {
			if (argc > 4) {
				if (mac_pton(argv[4], mc2uc))
					mac = mc2uc;
			}

			flowmgr_mdb_delete(in, out, group, mac);
			goto done;
		} else {
			pr_info("Deleting Flow: wrong value - ip\n");
		}
	}
/* help */
	cmd_mdb_del_help("");
done:
	return 0;
}
#endif
static void cmd_wandev_help(char *str)
{
	pr_alert("%s wandev: Set WAN device\n", str);
	pr_alert("%s  wandev <dev_name>\n", str);
}
static int cmd_wandev(int argc, char *argv[])
{
	if (argc > 1) {
		if (flowmgr_set_wandev_by_name(argv[1]) == 0)
			goto done;
	}
/* help */
	cmd_wandev_help("");
done:
	return 0;
}

static struct proc_cmd_ops command_entries[] = {
	PROC_CMD_INIT("flow_add", cmd_add),
	PROC_CMD_INIT("flow_add_expected", cmd_add_expected),
	PROC_CMD_INIT("flow_del", cmd_del),
	PROC_CMD_INIT("flow_delall", cmd_delall),
	PROC_CMD_INIT("flow_flushmac", cmd_flushmac),
	PROC_CMD_INIT("flow_flushinf", cmd_flushinf),
	PROC_CMD_INIT("flow_flushwifi", cmd_flushwifi),
	PROC_CMD_INIT("flow_counter", cmd_counter),
	PROC_CMD_INIT("fap_enable", cmd_enable),
	PROC_CMD_INIT("fap_config", cmd_config),
	PROC_CMD_INIT("fap_gre_add", cmd_gre_add),
	PROC_CMD_INIT("fap_gre_del", cmd_gre_del),
	PROC_CMD_INIT("ignore_port", cmd_ignore_port),
	PROC_CMD_INIT("show", cmd_show),
	PROC_CMD_INIT("inf", cmd_inf),
	PROC_CMD_INIT("addif", cmd_addif),
	PROC_CMD_INIT("delif", cmd_delif),
	PROC_CMD_INIT("ifdb_enable", cmd_ifdb_enable),
	PROC_CMD_INIT("flushdb", cmd_flushdb),
#ifdef CONFIG_BCM_FLOWMGR_MCAST
	PROC_CMD_INIT("mdb_add", cmd_mdb_add),
	PROC_CMD_INIT("mdb_del", cmd_mdb_del),
#endif
	PROC_CMD_INIT("wandev", cmd_wandev),
};

static struct proc_cmd_table command_table = {
	.module_name = "FLOWMGR",
	.size = ARRAY_SIZE(command_entries),
	.ops = command_entries
};

static int show_db_format;

static void *flowmgr_db_seq_start(struct seq_file *seq, loff_t *pos)
	__acquires(RCU)
{
	int path, track, inf, detail, dscp;
	path =  (show_db_format&0xFF00)>>8;
	track = show_db_format&0x0F;
	inf = (show_db_format&0xF0)>>4;
	detail = (show_db_format&0xFF000000)>>24;
	dscp = (show_db_format&0xFF0000)>>16;
	rcu_read_lock();
	seq->private = 0;
	seq_printf(seq, "Format: detail:%d path:%d track:%d inf:%d dscp:%d\n",
		   detail, path, track, inf, dscp);
	if (detail == 0 || detail == 1)
		seq_printf(seq, " Mac-Address       @ Interface  |%*s|%*s|%*s|%*s| %*s %*s rxflow-act/tot txflow-act/tot\n",
			   12, "Rx packets", 12, "Rx bytes",
			   12, "Tx packets", 12, "Tx Bytes",
			   8, "Created", 8, "Updated");
	else
		seq_printf(seq, " Mac-Address       @ Interface  |%*s|%*s|%*s|%*s|\n",
			   12, "Rx packets", 12, "Rx bytes",
			   12, "Tx packets", 12, "Tx Bytes");
	return flowmgr_db_get_idx((int *)&seq->private, *pos);
}

static void *flowmgr_db_seq_next(struct seq_file *seq, void *v,
				      loff_t *pos)
{
	(*pos)++;
	return flowmgr_db_get_next((int *)&seq->private, v);
}

static void flowmgr_db_seq_stop(struct seq_file *seq, void *v)
	__releases(RCU)
{
	rcu_read_unlock();
}

static int flowmgr_db_seq_show(struct seq_file *seq, void *v)
{
	struct flowmgr_db_entry *db = v;
	struct timeval tvc;
	struct timeval tvu;
	struct link_stats stats;
	struct link_stats *dscp_stats;
	struct net_device *dev;
	int path, track, inf, detail, dscp;
	int cnt = 0;
	u64 dscp_cgz = 0;

	path =  (show_db_format&0xFF00)>>8;
	track = show_db_format&0x0F;
	inf = (show_db_format&0xF0)>>4;
	detail = (show_db_format&0xFF000000)>>24;
	dscp = (show_db_format&0xFF0000)>>16;

	if (!v)
		return -1;

	if (track) {
		if ((track == 1) && db->notrack)
			return 0;
		if ((track == 2) && !db->notrack)
			return 0;
	}
	dscp_stats = kzalloc(DSCP_MAX_LIMIT * sizeof(struct link_stats),
					GFP_KERNEL);
	if (!dscp_stats) {
		pr_err("DSCP Link stat memory allocate failure\n");
		return -ENOMEM;
	}
	memset (&stats, 0, sizeof(stats));
	if (path == 0) {
		if (detail == 0) {
			flowmgr_db_get_stats64(db, &stats);
			flowmgr_db_get_dscp_stats64(db, &dscp_stats[0], &dscp_cgz);
		} else if (detail == 3)
			flowmgr_db_get_dscp_stats64(db, &dscp_stats[0], &dscp_cgz);
		else
			flowmgr_db_get_stats64(db, &stats);
	} else if (path == 1) {
		if (detail == 0) {
			flowmgr_db_get_stats64_slow(db, &stats, 1);
			flowmgr_db_get_dscp_stats64_slow(db, &dscp_stats[0], 1, &dscp_cgz);
		} else if (detail == 3)
			flowmgr_db_get_dscp_stats64_slow(db, &dscp_stats[0], 1, &dscp_cgz);
		else
			flowmgr_db_get_stats64_slow(db, &stats, 1);
	} else if (path == 2) {
		if (detail == 0) {
			flowmgr_db_get_stats64_fast(db, &stats);
			flowmgr_db_get_dscp_stats64_fast(db, &dscp_stats[0], &dscp_cgz);
		} else if (detail == 3)
			flowmgr_db_get_dscp_stats64_fast(db, &dscp_stats[0], &dscp_cgz);
		else
			flowmgr_db_get_stats64_fast(db, &stats);
	}
	jiffies_to_timeval(jiffies - db->created, &tvc);
	jiffies_to_timeval(jiffies - db->updated, &tvu);
	dev = __dev_get_by_index(&init_net, db->iif);

	if (!detail) {
		seq_printf(seq, " "BOLDBLACK"%pM"RESET" @ %*s |%12lld|%12lld|%12lld|%12lld| %8ld %8ld %5u/%-8u %5u/%-8u\n",
			   db->addr, 10, !dev ? "*" : dev->name,
			   stats.rx_packets, stats.rx_bytes,
			   stats.tx_packets, stats.tx_bytes,
			   tvc.tv_sec, tvu.tv_sec,
			   atomic_read(&db->rxflows),
			   atomic_read(&db->totrxflows),
			   atomic_read(&db->txflows),
			   atomic_read(&db->tottxflows));

		for (cnt = 0; cnt < DSCP_MAX_LIMIT; cnt++) {
			if ((track == 1) && ((gdscp_notrack>>cnt) & 0x1ULL))
				continue;
			if ((track == 2) && !((gdscp_notrack>>cnt) & 0x1ULL))
				continue;
			if (((dscp_cgz>>cnt) & 0x1ULL) && (dscp == 0 || (dscp == cnt+1))) {
				seq_printf(seq, " DSCP-%2d @ %*s |%12lld|%12lld|%12lld|%12lld|\n",
						   cnt, 20, !dev ? "*" : dev->name,
						   dscp_stats[cnt].rx_packets, dscp_stats[cnt].rx_bytes,
						   dscp_stats[cnt].tx_packets, dscp_stats[cnt].tx_bytes);
			}
		}
		seq_printf(seq, " ----------------------------------------------------------------------"
			   "-----------------------------------------------------------------------\n");
	} else if (detail == 1) {
		seq_printf(seq, " "BOLDBLACK"%pM"RESET" @ %*s |%12lld|%12lld|%12lld|%12lld| %8ld %8ld %5u/%-8u %5u/%-8u\n",
			   db->addr, 10, !dev ? "*" : dev->name,
			   stats.rx_packets, stats.rx_bytes,
			   stats.tx_packets, stats.tx_bytes,
			   tvc.tv_sec, tvu.tv_sec,
			   atomic_read(&db->rxflows),
			   atomic_read(&db->totrxflows),
			   atomic_read(&db->txflows),
			   atomic_read(&db->tottxflows));
	} else if (detail == 2) {
		seq_printf(seq, " "BOLDBLACK"%pM"RESET" @ %*s |%12lld|%12lld|%12lld|%12lld|\n",
			   db->addr, 10, !dev ? "*" : dev->name,
			   stats.rx_packets, stats.rx_bytes,
			   stats.tx_packets, stats.tx_bytes);
	} else if (detail == 3) {
		seq_printf(seq, " "BOLDBLACK"%pM"RESET" @ %*s \n",
			   db->addr, 10, !dev ? "*" : dev->name);
		for (cnt = 0; cnt < DSCP_MAX_LIMIT; cnt++) {
			if ((track == 1) && ((gdscp_notrack>>cnt) & 0x1ULL))
				continue;
			if ((track == 2) && !((gdscp_notrack>>cnt) & 0x1ULL))
				continue;
			if (((dscp_cgz>>cnt) & 0x1ULL) && (dscp == 0 || (dscp == cnt+1))) {
				seq_printf(seq, " DSCP-%2d @ %*s |%12lld|%12lld|%12lld|%12lld|\n",
						   cnt, 20, !dev ? "*" : dev->name,
						   dscp_stats[cnt].rx_packets, dscp_stats[cnt].rx_bytes,
						   dscp_stats[cnt].tx_packets, dscp_stats[cnt].tx_bytes);
			}
		}
		seq_printf(seq, " ------------------------------------------------------------------------------------\n");
	}

	if (dscp_stats)
		kfree(dscp_stats);
	return 0;
}

static const struct seq_operations flowmgr_db_seq_ops = {
	.start	= flowmgr_db_seq_start,
	.next	= flowmgr_db_seq_next,
	.stop	= flowmgr_db_seq_stop,
	.show	= flowmgr_db_seq_show,
};

static void cmd_cleardb_help(char *str)
{
	pr_alert("%s cleardb: Clear client database stats\n", str);
	pr_alert("%s  cleardb <all|mac_address|netdevice>\n", str);
}
static int cmd_cleardb(int argc, char *argv[])
{
	unsigned char mac[6];
	if (argc == 2) {
		if (strcasecmp("all", argv[1]) == 0) {
			flowmgr_db_clear();
			goto done;
		} else if (mac_pton(argv[1], mac)) {
			flowmgr_db_clear_stats_by_addr(mac);
			goto done;
		} else {
			struct net_device *dev;
			dev = __dev_get_by_name(&init_net,
						argv[1]);
			if (dev) {
				flowmgr_db_clear_stats_by_port(dev->ifindex);
				goto done;
			}
		}
	}
/* help */
	cmd_cleardb_help("");
done:
	return 0;
}

static void cmd_formatdb_help(char *str)
{
	pr_alert("%s formatdb: Control format of output\n", str);
	pr_alert("%s  formatdb <path:0-both|1-slow|2-fast> "
			"\n<track:0-all|1-tracked|2-notracked>"
			"\n<inf:0-WAN only|1-all>"
			"\n<detail:0-all|1-Detailed without DSCP|2-Limited Without DSCP|3 DSCP only>"
			"\n<dscp:0-all|1-64 for specific dscp number+1 to show\n", str);
}
static int cmd_formatdb(int argc, char *argv[])
{
	int i, path, track, inf, detail, dscp;
	u8 val;
	path =  show_db_format&0xFF00;
	track = show_db_format&0x0F;
	inf = show_db_format&0xF0;
	detail = show_db_format&0xFF000000;
	dscp = show_db_format&0xFF0000;

	pr_info("Old format: detail:%d path:%d track:%d inf:%d dscp:%d\n",
		detail>>24, path>>8, track, inf>>4, dscp>>16);
	if (argc > 1) {
		for (i = 0; i < argc; i++) {
			if (strstr(argv[i], "path:0"))
				path = 0 << 8;
			else if (strstr(argv[i], "path:1"))
				path = 1 << 8;
			else if (strstr(argv[i], "path:2"))
				path = 2 << 8;
			if (strstr(argv[i], "track:0"))
				track = 0;
			else if (strstr(argv[i], "track:1"))
				track = 1;
			else if (strstr(argv[i], "track:2"))
				track = 2;
			if (strstr(argv[i], "inf:0"))
				inf = 0 << 4;
			else if (strstr(argv[i], "inf:1"))
				inf = 1 << 4;
			if (strstr(argv[i], "detail:0"))
				detail = 0 << 24;
			else if (strstr(argv[i], "detail:1"))
				detail = 1 << 24;
			else if (strstr(argv[i], "detail:2"))
				detail = 2 << 24;
			else if (strstr(argv[i], "detail:3"))
				detail = 3 << 24;
			else if (strstr(argv[i], "dscp:")) {
				if (kstrtou8(&argv[i][strlen("dscp:")], 0, &val) != 0) {
					goto help;
				} else {
					if (val > DSCP_MAX_LIMIT)
						goto help;
					else
						dscp = val << 16;
				}
			}
		}
		show_db_format = detail | path | track | dscp | inf;
		goto done;
	}
/* help */
help:
	cmd_formatdb_help("");
done:
	pr_info("New format: detail:%d path:%d track:%d inf:%d dscp:%d\n",
		detail>>24, path>>8, track, inf>>4, dscp>>16);
	flowmgr.inf_db_info = (inf>>4);

	return 0;
}

static void cmd_notrackdb_help(char *str)
{
	pr_alert("%s notrackdb: Disable updating counters for specific client\n", str);
	pr_alert("%s  notrackdb <mac_address> <notrack>\n", str);
	pr_alert("%s   	notrack - %d - enable track\n", str, DISABLE_NOTRACK);
	pr_alert("%s   	notrack - %d - disable track\n", str, ENABLE_NOTRACK);
	pr_alert("%s   	notrack - %d - disable all track except for this address\n",
		 str, ENABLE_ALL_NOTRACK);
	pr_alert("%s   	notrack - %d - enable all track except for this address\n",
		 str, DISABLE_ALL_NOTRACK);
	pr_alert("%s   	notrack - %d - reset to default\n", str, RESTORE_DEFAULT_NOTRACK);
}
static int cmd_notrackdb(int argc, char *argv[])
{
	unsigned char mac[6];
	u8 notrack;
	int ret = 0;

	if (argc == 3) {
		if (mac_pton(argv[1], mac)) {
			if (kstrtou8(argv[2], 0, &notrack) != 0) {
				pr_info("Invaild notrack value\n");
			} else {
				ret = flowmgr_db_set_notrack(mac,
						(enum flow_notrack_type)notrack);
				if (!ret)
					goto done;
				else if (ret == -EINVAL)
					pr_info("Invaild Address\n");
				else if (ret == -ENOMEM) {
					pr_info("Address not available\n");
					goto done;
				}
			}
		}
	} else if (argc == 2) {
		if (kstrtou8(argv[1], 0, &notrack) == 0) {
			if (RESTORE_DEFAULT_NOTRACK ==
				(enum flow_notrack_type)notrack) {
				flowmgr_db_set_notrack(0,
					(enum flow_notrack_type)notrack);
				goto done;
			}
		}
	}
/* help */
	cmd_notrackdb_help("");
done:
	return 0;
}

static void cmd_dscp_notrackdb_help(char *str)
{
	pr_alert("%s dscp_notrackdb: Disable updating counters for specific client\n", str);
	pr_alert("%s  dscp_notrackdb <mac_address> <dscp_notrack>\n", str);
	pr_alert("%s   	dscp_notrack - %d - enable track\n", str, DISABLE_NOTRACK);
	pr_alert("%s   	dscp_notrack - %d - disable track\n", str, ENABLE_NOTRACK);
	pr_alert("%s   	dscp_notrack - %d - disable all track except for this address\n",
		 str, ENABLE_ALL_NOTRACK);
	pr_alert("%s   	dscp_notrack - %d - enable all track except for this address\n",
		 str, DISABLE_ALL_NOTRACK);
	pr_alert("%s   	dscp_notrack - %d - reset to default\n", str, RESTORE_DEFAULT_NOTRACK);
}
static int cmd_dscp_notrackdb(int argc, char *argv[])
{
	u8 dscp;
	u8 notrack;
	int ret = 0;

	if (argc == 3) {
		if (kstrtou8(argv[1], 0, &dscp) == 0) {
			if (kstrtou8(argv[2], 0, &notrack) != 0) {
				pr_info("Invaild dscp notrack value\n");
			} else {
				ret = flowmgr_db_set_dscp_notrack(dscp,
						(enum flow_notrack_type)notrack);
				if (!ret)
					goto done;
				else if (ret == -EINVAL)
					pr_info("Invaild DSCP value\n");
			}
		} else
			pr_info("Invaild dscp value\n");
	} else if (argc == 2) {
		if (kstrtou8(argv[1], 0, &notrack) == 0) {
			if (RESTORE_DEFAULT_NOTRACK ==
				(enum flow_notrack_type)notrack) {
				flowmgr_db_set_dscp_notrack(0,
				(enum flow_notrack_type)notrack);
				goto done;
			}
		}
	}
/* help */
	cmd_dscp_notrackdb_help("");
done:
	return 0;
}

static struct proc_cmd_ops db_command_entries[] = {
	PROC_CMD_INIT("flushdb", cmd_flushdb),
	PROC_CMD_INIT("clear",   cmd_cleardb),
	PROC_CMD_INIT("format",  cmd_formatdb),
	PROC_CMD_INIT("notrack", cmd_notrackdb),
	PROC_CMD_INIT("dscp_notrack", cmd_dscp_notrackdb),
};

struct proc_cmd_table db_command_table = {
	.module_name = "FLOWMGR",
	.size = ARRAY_SIZE(db_command_entries),
	.data_seq_read = (void *) &flowmgr_db_seq_ops,
	.ops = db_command_entries
};

static void *flowmgr_dbflow_seq_start(struct seq_file *seq, loff_t *pos)
	__acquires(RCU)
{
	rcu_read_lock();
	seq->private = 0;
	seq_printf(seq, " Mac-Address       @ Interface  | vlan  | ifdb | accel | steer | #steers  | rxflow-act/tot txflow-act/tot\n");
	return flowmgr_db_get_idx((int *)&seq->private, *pos);
}

static void *flowmgr_dbflow_seq_next(struct seq_file *seq, void *v,
				     loff_t *pos)
{
	(*pos)++;
	return flowmgr_db_get_next((int *)&seq->private, v);
}

static void flowmgr_dbflow_seq_stop(struct seq_file *seq, void *v)
	__releases(RCU)
{
	rcu_read_unlock();
}

static int flowmgr_dbflow_seq_show(struct seq_file *seq, void *v)
{
	struct flowmgr_db_entry *db = v;
	struct net_device *dev;
	dev = __dev_get_by_index(&init_net, db->iif);

	seq_printf(seq, " "BOLDBLACK"%pM"RESET" @ %*s | %-5u | %-4s | %-5s | %-5s | %-8u | %5u/%-8u %5u/%-8u\n",
		   db->addr, 10, !dev ? "*" : dev->name,
		   db->vlan_id,
		   is_flowmgr_db_ifenable(dev) ? "Y" : "N",
		   db->accel ? "Y" : "N",
		   db->steer ? "Y" : "N",
		   atomic_read(&db->nsteers),
		   atomic_read(&db->rxflows),
		   atomic_read(&db->totrxflows),
		   atomic_read(&db->txflows),
		   atomic_read(&db->tottxflows));
	return 0;
}

static const struct seq_operations flowmgr_dbflow_seq_ops = {
	.start	= flowmgr_dbflow_seq_start,
	.next	= flowmgr_dbflow_seq_next,
	.stop	= flowmgr_dbflow_seq_stop,
	.show	= flowmgr_dbflow_seq_show,
};

static void cmd_db_clearsteer_help(char *str)
{
	pr_alert("%s clear_steer: Clear client steering stats\n", str);
	pr_alert("%s  clear_steer <all|mac_address|netdevice>\n", str);
}
static int cmd_db_clearsteer(int argc, char *argv[])
{
	unsigned char mac[6];
	if (argc == 2) {
		if (strcasecmp("all", argv[1]) == 0) {
			flowmgr_dbflow_clear_steer();
			goto done;
		} else if (mac_pton(argv[1], mac)) {
			flowmgr_dbflow_clear_steer_stats_by_addr(mac);
			goto done;
		} else {
			struct net_device *dev;
			dev = __dev_get_by_name(&init_net,
						argv[1]);
			if (dev) {
				flowmgr_dbflow_clear_steer_stats_by_port(dev->ifindex);
				goto done;
			}
		}
	}
/* help */
	cmd_db_clearsteer_help("");
done:
	return 0;
}

static void cmd_db_clearflow_help(char *str)
{
	pr_alert("%s clear_flow: Clear client flow` stats\n", str);
	pr_alert("%s  clear_flow <all|mac_address|netdevice>\n", str);
}
static int cmd_db_clearflow(int argc, char *argv[])
{
	unsigned char mac[6];
	if (argc == 2) {
		if (strcasecmp("all", argv[1]) == 0) {
			flowmgr_dbflow_clear();
			goto done;
		} else if (mac_pton(argv[1], mac)) {
			flowmgr_dbflow_clear_stats_by_addr(mac);
			goto done;
		} else {
			struct net_device *dev;
			dev = __dev_get_by_name(&init_net,
						argv[1]);
			if (dev) {
				flowmgr_dbflow_clear_stats_by_port(dev->ifindex);
				goto done;
			}
		}
	}
/* help */
	cmd_db_clearflow_help("");
done:
	return 0;
}

static void cmd_db_enablesteer_help(char *str)
{
	pr_alert("%s enable_steer: Enable client steering\n", str);
	pr_alert("%s  enable_steer <all|mac_address|netdevice> <1|0>\n", str);
}
static int cmd_db_enablesteer(int argc, char *argv[])
{
	unsigned char mac[6];
	if (argc == 3) {
		u8 steer;
		if (kstrtou8(argv[2], 0, &steer) != 0) {
			pr_info("%s: Invalid accel value\n", __func__);
			goto done;
		}
		if (strcasecmp("all", argv[1]) == 0) {
			flowmgr_dbflow_set_steer_all(steer);
			goto done;
		} else if (mac_pton(argv[1], mac)) {
			flowmgr_dbflow_set_steer_by_addr(mac, steer);
			goto done;
		} else {
			struct net_device *dev;
			dev = __dev_get_by_name(&init_net,
						argv[1]);
			if (dev) {
				flowmgr_dbflow_set_steer_by_port(dev->ifindex, steer);
				goto done;
			}
		}
	}
/* help */
	cmd_db_enablesteer_help("");
done:
	return 0;
}

static void cmd_db_enableaccel_help(char *str)
{
	pr_alert("%s enable_accel: Enable flow acceleration for client\n", str);
	pr_alert("%s  enable_accel <all|mac_address|netdevice> <1|0>\n", str);
}
static int cmd_db_enableaccel(int argc, char *argv[])
{
	unsigned char mac[6];
	if (argc == 3) {
		u8 accel;
		if (kstrtou8(argv[2], 0, &accel) != 0) {
			pr_info("%s: Invalid accel value\n", __func__);
			goto done;
		}
		if (strcasecmp("all", argv[1]) == 0) {
			flowmgr_dbflow_set_accel_all(accel);
			goto done;
		} else if (mac_pton(argv[1], mac)) {
			flowmgr_dbflow_set_accel_by_addr(mac, accel);
			goto done;
		} else {
			struct net_device *dev;
			dev = __dev_get_by_name(&init_net,
						argv[1]);
			if (dev) {
				flowmgr_dbflow_set_accel_by_port(dev->ifindex, accel);
				goto done;
			}
		}
	}
/* help */
	cmd_db_enableaccel_help("");
done:
	return 0;
}

static struct proc_cmd_ops dbflow_command_entries[] = {
	PROC_CMD_INIT("clear_steer",  cmd_db_clearsteer),
	PROC_CMD_INIT("clear_flow",   cmd_db_clearflow),
	PROC_CMD_INIT("enable_steer", cmd_db_enablesteer),
	PROC_CMD_INIT("enable_accel", cmd_db_enableaccel),
};

struct proc_cmd_table dbflow_command_table = {
	.module_name = "FLOWMGR",
	.size = ARRAY_SIZE(dbflow_command_entries),
	.data_seq_read = (void *) &flowmgr_dbflow_seq_ops,
	.ops = dbflow_command_entries
};

#ifdef CONFIG_BCM_FLOWMGR_MCAST
static void *flowmgr_mdb_seq_start(struct seq_file *seq, loff_t *pos)
	__acquires(RCU)
{
	rcu_read_lock();
	seq->private = 0;
	return flowmgr_mdb_get_idx((int *)&seq->private, *pos);
}

static void *flowmgr_mdb_seq_next(struct seq_file *seq, void *v,
				      loff_t *pos)
{
	(*pos)++;
	return flowmgr_mdb_get_next((int *)&seq->private, v);
}

static void flowmgr_mdb_seq_stop(struct seq_file *seq, void *v)
	__releases(RCU)
{
	rcu_read_unlock();
}

static int flowmgr_mdb_seq_show(struct seq_file *seq, void *v)
{
	struct flowmgr_mdb_entry *db = v;
	struct timeval tv;
	int i, h = 0;

	if (!v)
		return -1;
	jiffies_to_timeval(jiffies - db->created, &tv);

	if (db->group[0])
		seq_printf(seq, " %pI6c iif=%d ", &db->group, db->iif);
	else
		seq_printf(seq, " %pI4 iif=%d ", &db->group[3], db->iif);
	for (i = 0;  i < NUM_MCTX; i++) {
		if (db->oif[i]) {
			if (h == 0)
				seq_puts(seq, "oif=");
			seq_printf(seq, "%d(%d) ", i, db->oif[i]);
			h = 1;
		}
	}

	for (i = 0;  i < db->num_mc2uc; i++) {
		if (i == 0)
			seq_puts(seq, "mc2uc=");
		seq_printf(seq, "[%d %pM]",
			   db->mc2uc[i].oif,
			   db->mc2uc[i].addr);
	}
	seq_printf(seq, " flow=%u update %ld use=%u packets=%lu\n",
		   db->flow_id, tv.tv_sec, atomic_read(&db->use), db->packets);
	return 0;

}

static const struct seq_operations flowmgr_mdb_seq_ops = {
	.start	= flowmgr_mdb_seq_start,
	.next	= flowmgr_mdb_seq_next,
	.stop	= flowmgr_mdb_seq_stop,
	.show	= flowmgr_mdb_seq_show,
};

static int flowmgr_mdb_seq_open(struct inode *inode, struct file *file)
{
	int ret = seq_open(file, &flowmgr_mdb_seq_ops);
	return ret;
};

static const struct proc_ops flowmgr_proc_mdb_fops = {
	.proc_open     = flowmgr_mdb_seq_open,
	.proc_read     = seq_read,
	.proc_lseek    = seq_lseek,
	.proc_release  = seq_release,
};
#endif
#ifdef CONFIG_BCM_FLOWMGR_TUNNEL
static void *flowmgr_tdb_seq_start(struct seq_file *seq, loff_t *pos)
	__acquires(RCU)
{
	rcu_read_lock();
	seq->private = 0;
	return flowmgr_tdb_get_idx((int *)&seq->private, *pos);
}

static void *flowmgr_tdb_seq_next(struct seq_file *seq, void *v,
				      loff_t *pos)
{
	(*pos)++;
	return flowmgr_tdb_get_next((int *)&seq->private, v);
}

static void flowmgr_tdb_seq_stop(struct seq_file *seq, void *v)
	__releases(RCU)
{
	rcu_read_unlock();
}

static int flowmgr_tdb_seq_show(struct seq_file *seq, void *v)
{
	struct net_device *tun = NULL;
	struct net_device *out = NULL;
	struct flowmgr_tdb_entry *db = v;
	struct timeval tv;

	if (!v)
		return -1;
	jiffies_to_timeval(jiffies - db->created, &tv);
	tun = __dev_get_by_index(&init_net, db->ifi);
	if (!tun){
		seq_printf(seq, "unable to find dev db->ifi=%d\n", db->ifi);
		return 0;
	}
	out = __dev_get_by_index(&init_net, db->oif);

	seq_printf(seq, "%s: oif=%s id=%d ", tun->name,
		   out ? out->name : "NULL", db->id);
	seq_printf(seq, "type=%u ", db->type);
	if (db->type != ARPHRD_VOID) {
		seq_printf(seq, "dmac=%pM ", db->parms.eh.h_dest);
		seq_printf(seq, "smac=%pM ", db->parms.eh.h_source);
		seq_printf(seq, "vlan=0x%x ", db->parms.vlan_tag);
		seq_printf(seq, "mtu=%d ", db->parms.mtu);
		seq_printf(seq, "tos=%X ", db->parms.tos);
		if (db->wifi_mask) {
			int i;

			seq_printf(seq, "wifi=");
			for (i = 0; i < 8; i++) {
				if (db->wifi_flowring[i] != 0xFFFF) {
					seq_printf(seq, "%d-%d| ",
						   db->wifi_flowring[i],
						   db->wifi_pri[i]);
				} else
					seq_printf(seq, "-| ");
			}
		}
	} else {
		struct flow_mapt_params *params;
		__be16 *p;
		__be32 v4prefix;

		params = &db->mapt;
		seq_printf(seq, "domain=%d ", params->domain_index);
		p = (__be16 *)&params->br_prefix;
		seq_printf(seq, "br_pf=%x:%x:%x:%x ",
			__be16_to_cpu(p[0]), __be16_to_cpu(p[1]),
			__be16_to_cpu(p[2]),__be16_to_cpu(p[3]));
		seq_printf(seq, "br_pfl=%d ", params->br_prefix_length);
		p = (__be16 *)&params->ipv6_prefix;
		seq_printf(seq, "ipv6_pf=%x:%x:%x:%x ",
			__be16_to_cpu(p[0]), __be16_to_cpu(p[1]),
			__be16_to_cpu(p[2]),__be16_to_cpu(p[3]));
		seq_printf(seq, "ipv6_pfl=%d ", params->ipv6_prefix_length);
		// The ipv4_prefix is stored as host representation
		v4prefix = htonl(params->ipv4_prefix);
		seq_printf(seq, "ipv4_pf=%pI4 ", &v4prefix);
		seq_printf(seq, "ipv4_pfl=%d ", params->ipv4_prefix_length);
		seq_printf(seq, "map_ipv6=%pI6c ", &params->map_ipv6_address);
		seq_printf(seq, "psid_off=%d ", params->psid_offset);
		seq_printf(seq, "psid_len=%d ", params->psid_length);
		seq_printf(seq, "psid=%d ",  params->psid);
	}

	if ((db->type == ARPHRD_TUNNEL6)
	    || (db->type == ARPHRD_IP6GRE)
	    ) {
		seq_printf(seq, "raddr=%pI6c ", db->parms.raddr.ip6);
		seq_printf(seq, "laddr=%pI6c ", db->parms.laddr.ip6);
	} else if ((db->type == ARPHRD_TUNNEL) ||
		   (db->type == ARPHRD_IPGRE)) {
		seq_printf(seq, "raddr=%pI4 ", &db->parms.raddr.ip);
		seq_printf(seq, "laddr=%pI4 ", &db->parms.laddr.ip);
	} else if (flowmgr_is_dev_gretap_tunnel(tun)) {
		seq_printf(seq, "raddr=%pI4 ", &db->parms.raddr.ip);
		seq_printf(seq, "laddr=%pI4 ", &db->parms.laddr.ip);
	} else if (flowmgr_is_dev_ip6gretap_tunnel(tun)) {
		seq_printf(seq, "raddr=%pI6c ", db->parms.raddr.ip6);
		seq_printf(seq, "laddr=%pI6c ", db->parms.laddr.ip6);
	}
	seq_printf(seq, " state=%d update=%ld use=%u\n",
		   db->state, tv.tv_sec, atomic_read(&db->use));
	return 0;

}

static const struct seq_operations flowmgr_tdb_seq_ops = {
	.start	= flowmgr_tdb_seq_start,
	.next	= flowmgr_tdb_seq_next,
	.stop	= flowmgr_tdb_seq_stop,
	.show	= flowmgr_tdb_seq_show,
};

static void cmd_hwflushtdb_help(char *str)
{
	pr_alert("%s hwflush: Flush tunnel from hw\n", str);
	pr_alert("%s  hwflush <dev name>\n", str);
}
static int cmd_hwflushtdb(int argc, char *argv[])
{
	if (argc == 2) {
		struct net_device *dev;
		struct flowmgr_tdb_entry *db;

		dev = __dev_get_by_name(&init_net, argv[1]);
		if (!dev)
			goto help;

		db = flowmgr_tdb_find(dev->ifindex);
		if (!db)
			goto help;

		if (!db->state)
			goto done;

		if (db->type == ARPHRD_VOID) {
			/* mapt */
			flowmgr_map_flush(db->mapt.domain_index);
			goto done;
		} else if (db->type == ARPHRD_TUNNEL6) {
			/* Dslite */
			flowmgr_unconfig_dslite(db->id);
			db->state = 0;
			db->id = -1;
			goto done;
		} else if (flowmgr_is_dev_gretap_tunnel(dev)) {
			/* gretap */
			flowmgr_unconfig_gre(db->id);
			db->state = 0;
			db->id = -1;
			goto done;
		} else if (flowmgr_is_dev_ip6gretap_tunnel(dev)) {
			/* gretap6 */
			flowmgr_unconfig_gre(db->id);
			db->state = 0;
			db->id = -1;
			goto done;
		}
	}
help:
	cmd_hwflushtdb_help("");
done:
	return 0;
}

static struct proc_cmd_ops tdb_command_entries[] = {
	PROC_CMD_INIT("hwflush", cmd_hwflushtdb),
};

struct proc_cmd_table tdb_command_table = {
	.module_name = "FLOWMGR",
	.size = ARRAY_SIZE(tdb_command_entries),
	.data_seq_read = (void *) &flowmgr_tdb_seq_ops,
	.ops = tdb_command_entries
};
#endif

static void *flowmgr_usage_seq_start(struct seq_file *seq, loff_t *pos)
{
	if (*pos < flowmgr.max_fap_flows)
		return pos;
	return NULL;
}

static void *flowmgr_usage_seq_next(struct seq_file *seq, void *v,
				      loff_t *pos)
{
	++*pos;
	if (*pos < flowmgr.max_fap_flows)
		return pos;
	return NULL;
}

static void flowmgr_usage_seq_stop(struct seq_file *seq, void *v)
{
}

static int flowmgr_usage_seq_show(struct seq_file *seq, void *v)
{
	struct flowmgr_flow_usage *usage;
	loff_t *pos = v;

	if (!flowmgr.flow_usage)
		return 0;

	if (*pos == 0) {
		seq_printf(seq, "%*s %*s %*s %*s %*s\n",
			   5, "Index", 10, "Add",
			   10, "Del", 12, "LastDuration",
			   12, "MaxDuration ");
	}

	usage = &flowmgr.flow_usage[*pos];
	if (usage->add)
		seq_printf(seq, "%-5lld %-10x %-10x %-12ld %-12ld\n",
			   *pos, usage->add, usage->del,
			   usage->ltv.tv_sec, usage->mtv.tv_sec);

	return 0;
}

static const struct seq_operations flowmgr_usage_seq_ops = {
	.start	= flowmgr_usage_seq_start,
	.next	= flowmgr_usage_seq_next,
	.stop	= flowmgr_usage_seq_stop,
	.show	= flowmgr_usage_seq_show,
};

static void cmd_clearusage_help(char *str)
{
	pr_alert("%s clear: Clear Flow Usage Counters\n", str);
	pr_alert("%s  clear <all|flow_id>\n", str);
}

static int cmd_clearusage(int argc, char *argv[])
{
	int i;
	if (!flowmgr.flow_usage)
		return 0;

	if (argc == 2) {
		if (strcasecmp("all", argv[1]) == 0) {
			for (i = 0; i < flowmgr.max_fap_flows; i++) {
				memset(&flowmgr.flow_usage[i], 0,
				       sizeof(struct flowmgr_flow_usage));
			}
			goto done;
		} else if (kstrtoint(argv[1], 0, &i) == 0) {
			memset(&flowmgr.flow_usage[i], 0,
			       sizeof(struct flowmgr_flow_usage));
			goto done;
		}
	}
/* help */
	cmd_clearusage_help("");
done:
	return 0;
}

static void cmd_enableusage_help(char *str)
{
	pr_alert("%s enable: Enable Counting for flow usage\n", str);
	pr_alert("%s  enable <1|0>\n", str);
}

static int cmd_enableusage(int argc, char *argv[])
{
	u8 usagetype = 0;

	if (argc == 2) {
		if (kstrtou8(argv[1], 0, &usagetype) == 0) {
			if (usagetype == 0) {
				if (flowmgr.flow_usage && !flowmgr.enable) {
					kfree(flowmgr.flow_usage);
					flowmgr.flow_usage = NULL;
					pr_info("FLOWMGR Usage disabled\n");
				}
				goto done;
			} else if (usagetype == 1) {
				if (flowmgr.max_fap_flows && !flowmgr.flow_usage) {
					flowmgr.flow_usage = kzalloc(flowmgr.max_fap_flows *
										 sizeof(struct flowmgr_flow_usage),
										 GFP_KERNEL);
					pr_info("FLOWMGR Usage: %d\n", flowmgr.max_fap_flows);
				}
				goto done;
			}
		}
	}
/* help */
	cmd_enableusage_help("");
done:
	return 0;
}

static struct proc_cmd_ops usage_command_entries[] = {
	PROC_CMD_INIT("clear", cmd_clearusage),
	PROC_CMD_INIT("enable", cmd_enableusage),
};

struct proc_cmd_table usage_command_table = {
	.module_name = "FLOWMGR",
	.size = ARRAY_SIZE(usage_command_entries),
	.data_seq_read = (void *) &flowmgr_usage_seq_ops,
	.ops = usage_command_entries
};

static void *flowmgr_status_seq_start(struct seq_file *seq, loff_t *pos)
{
	if (!*pos)
		return SEQ_START_TOKEN;
	return 0;
}

static void *flowmgr_status_seq_next(struct seq_file *seq, void *v,
				      loff_t *pos)
{
	(*pos)++;
	return 0;
}

static void flowmgr_status_seq_stop(struct seq_file *seq, void *v)
{
}

static int flowmgr_status_seq_show(struct seq_file *seq, void *v)
{
	if (!v)
		return -1;
	flowmgr_fap_dbg_show(seq);
	flowmgr_debug_show(seq);
	flowmgr_nfct_dbg_show(seq);
	flowmgr_ignore_port_show(seq);
	flowmgr_netdevice_show(seq);
	flowmgr_trace_show(seq);
	return 0;
}

static const struct seq_operations flowmgr_status_seq_ops = {
	.start	= flowmgr_status_seq_start,
	.next	= flowmgr_status_seq_next,
	.stop	= flowmgr_status_seq_stop,
	.show	= flowmgr_status_seq_show,
};

static int flowmgr_status_seq_open(struct inode *inode, struct file *file)
{
	int ret = seq_open(file, &flowmgr_status_seq_ops);
	return ret;
};

static const struct proc_ops flowmgr_proc_status_fops = {
	.proc_open     = flowmgr_status_seq_open,
	.proc_read     = seq_read,
	.proc_lseek    = seq_lseek,
	.proc_release  = seq_release,
};

static int flowmgr_device_seq_show(struct seq_file *seq, void *v)
{
	if (!v)
		return -1;
	flowmgr_netdevice_show(seq);
	return 0;
}

static const struct seq_operations flowmgr_device_seq_ops = {
	.start	= flowmgr_status_seq_start,
	.next	= flowmgr_status_seq_next,
	.stop	= flowmgr_status_seq_stop,
	.show	= flowmgr_device_seq_show,
};

static struct proc_cmd_ops device_command_entries[] = {
	PROC_CMD_INIT("inf", cmd_inf),
	PROC_CMD_INIT("addif", cmd_addif),
	PROC_CMD_INIT("delif", cmd_delif),
};

struct proc_cmd_table device_command_table = {
	.module_name = "FLOWMGR",
	.size = ARRAY_SIZE(device_command_entries),
	.data_seq_read = (void *) &flowmgr_device_seq_ops,
	.ops = device_command_entries
};

static int flowmgr_counter_seq_show(struct seq_file *seq, void *v)
{
	if (!v)
		return -1;
	flowmgr_debug_counter_show(seq);
	return 0;
}

static const struct seq_operations flowmgr_counter_seq_ops = {
	.start	= flowmgr_status_seq_start,
	.next	= flowmgr_status_seq_next,
	.stop	= flowmgr_status_seq_stop,
	.show	= flowmgr_counter_seq_show,
};

static void cmd_clearcounter_help(char *str)
{
	pr_alert("%s clear: Clear debug stats\n", str);
}
static int cmd_clearcounter(int argc, char *argv[])
{
	flowmgr_debug_counter_clear();
	return 0;
}

static struct proc_cmd_ops counter_command_entries[] = {
	PROC_CMD_INIT("clear", cmd_clearcounter),
};

struct proc_cmd_table counter_command_table = {
	.module_name = "FLOWMGR",
	.size = ARRAY_SIZE(counter_command_entries),
	.data_seq_read = (void *) &flowmgr_counter_seq_ops,
	.ops = counter_command_entries
};

static int flowmgr_fap_seq_show(struct seq_file *seq, void *v)
{
	if (!v)
		return -1;
	flowmgr_fap_dbg_show(seq);
	return 0;
}

static const struct seq_operations flowmgr_fap_seq_ops = {
	.start	= flowmgr_status_seq_start,
	.next	= flowmgr_status_seq_next,
	.stop	= flowmgr_status_seq_stop,
	.show	= flowmgr_fap_seq_show,
};

static struct proc_cmd_ops fap_command_entries[] = {
	PROC_CMD_INIT("enable", cmd_enable),
	PROC_CMD_INIT("config", cmd_config),
};

struct proc_cmd_table fap_command_table = {
	.module_name = "FLOWMGR",
	.size = ARRAY_SIZE(fap_command_entries),
	.data_seq_read = (void *) &flowmgr_fap_seq_ops,
	.ops = fap_command_entries
};

static void *flowmgr_features_seq_start(struct seq_file *seq, loff_t *pos)
{
	if (!*pos)
		return SEQ_START_TOKEN;
	return 0;
}

static void *flowmgr_features_seq_next(struct seq_file *seq, void *v,
				       loff_t *pos)
{
	(*pos)++;
	return 0;
}

static void flowmgr_features_seq_stop(struct seq_file *seq, void *v)
{
}

static int flowmgr_features_seq_show(struct seq_file *seq, void *v)
{
	if (!v)
		return -1;
	flowmgr_features_show(seq, "* sw features", flowmgr.sw_features);
	flowmgr_features_show(seq, "* hw features", flowmgr.hw_features);
	flowmgr_features_show(seq, "* active features", flowmgr.features);
	return 0;

}

static const struct seq_operations flowmgr_features_seq_ops = {
	.start	= flowmgr_features_seq_start,
	.next	= flowmgr_features_seq_next,
	.stop	= flowmgr_features_seq_stop,
	.show	= flowmgr_features_seq_show,
};

static int active_wandev_info_proc_read(struct seq_file *m, void *v)
{
	struct net_device *dev = flowmgr_get_wandev();

	seq_printf(m, "Active WAN Dev:%s\n",
		dev ? dev->name:"Uninitialized");
	seq_printf(m, "WAN type:%d\n", flowmgr.wantype);
	return 0;
}

static int active_wandev_info_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, active_wandev_info_proc_read, NULL);
}

static const struct proc_ops flowmgr_proc_active_wandev_info_fops = {
	.proc_open		= active_wandev_info_proc_open,
	.proc_read		= seq_read,
	.proc_lseek		= seq_lseek,
	.proc_release	= single_release,
};

static int cmd_f_enable(int argc, char *argv[], int mask)
{
	u8 enable = 0;

	if (argc == 2) {
		if (kstrtou8(argv[1], 0, &enable) == 0) {
			if (enable) {
				flowmgr.features |= mask;
				goto done;
			} else {
				flowmgr.features &= ~mask;
				goto done;
			}
		}
	}
/* help */
	pr_info("%s <1|0>\n", argv[0]);
done:
	return 0;
}

static void cmd_f_tcp_help(char *str)
{
	pr_alert("%s tcp: Enable/disable tcp flow promotion", str);
	pr_alert("%s  tcp <1|0>\n", str);
}
static int cmd_f_tcp(int argc, char *argv[])
{
	int mask = (FLOW_F_TCP4 | FLOW_F_TCP6);
	return cmd_f_enable(argc, argv, mask);
}

static void cmd_f_udp_help(char *str)
{
	pr_alert("%s udp: Enable/disable udp flow promotion", str);
	pr_alert("%s  udp <1|0>\n", str);
}
static int cmd_f_udp(int argc, char *argv[])
{
	int mask = (FLOW_F_UDP4 | FLOW_F_UDP6);
	return cmd_f_enable(argc, argv, mask);
}

static void cmd_f_esp_help(char *str)
{
	pr_alert("%s esp: Enable/disable edp flow promotion", str);
	pr_alert("%s  esp <1|0>\n", str);
}
static int cmd_f_esp(int argc, char *argv[])
{
	int mask = (FLOW_F_ESP4 | FLOW_F_ESP6);
	return cmd_f_enable(argc, argv, mask);
}

static void cmd_f_ah_help(char *str)
{
	pr_alert("%s tcp: Enable/disable ah flow promotion", str);
	pr_alert("%s  ah <1|0>\n", str);
}
static int cmd_f_ah(int argc, char *argv[])
{
	int mask = (FLOW_F_AH4 | FLOW_F_AH6);
	return cmd_f_enable(argc, argv, mask);
}

static void cmd_f_ipv4_help(char *str)
{
	pr_alert("%s ipv4: Enable/disable ipv4 flow promotion", str);
	pr_alert("%s  ipv4 <1|0>\n", str);
}
static int cmd_f_ipv4(int argc, char *argv[])
{
	int mask = (FLOW_F_TCP4 |
		    FLOW_F_UDP4 |
		    FLOW_F_ESP4 |
		    FLOW_F_AH4);
	return cmd_f_enable(argc, argv, mask);
}

static void cmd_f_ipv6_help(char *str)
{
	pr_alert("%s ipv6: Enable/disable ipv6 flow promotion", str);
	pr_alert("%s  ipv6 <1|0>\n", str);
}
static int cmd_f_ipv6(int argc, char *argv[])
{
	int mask = (FLOW_F_TCP6 |
		    FLOW_F_UDP6 |
		    FLOW_F_ESP6 |
		    FLOW_F_AH6);
	return cmd_f_enable(argc, argv, mask);
}

static void cmd_f_gretap4_help(char *str)
{
	pr_alert("%s gretap4: Enable/disable gretapv4 flow promotion", str);
	pr_alert("%s  gretap4 <1|0>\n", str);
}
static int cmd_f_gretap4(int argc, char *argv[])
{
	int mask = (FLOW_F_GRETAP4WAN | FLOW_F_GRETAP4ETHWAN |
		    FLOW_F_GRETAP4LAN);
	return cmd_f_enable(argc, argv, mask);
}

static void cmd_f_gretap6_help(char *str)
{
	pr_alert("%s gretap6: Enable/disable gretapv6 flow promotion", str);
	pr_alert("%s  gretap6 <1|0>\n", str);
}
static int cmd_f_gretap6(int argc, char *argv[])
{
	int mask = (FLOW_F_GRETAP6WAN | FLOW_F_GRETAP6ETHWAN |
		    FLOW_F_GRETAP6LAN);
	return cmd_f_enable(argc, argv, mask);
}

static void cmd_f_gretaplan_help(char *str)
{
	pr_alert("%s gretaplan: Enable/disable gretap lan flow promotion", str);
	pr_alert("%s  gretaplan <1|0>\n", str);
}
static int cmd_f_gretaplan(int argc, char *argv[])
{
	int mask = (FLOW_F_GRETAP4LAN |
		    FLOW_F_GRETAP6LAN);
	return cmd_f_enable(argc, argv, mask);
}

static void cmd_f_gretaplan4_help(char *str)
{
	pr_alert("%s gretaplan4: Enable/disable gretap lan ipv4 flow promotion", str);
	pr_alert("%s  gretaplan4 <1|0>\n", str);
}
static int cmd_f_gretaplan4(int argc, char *argv[])
{
	int mask = (FLOW_F_GRETAP4LAN);
	return cmd_f_enable(argc, argv, mask);
}

static void cmd_f_gretaplan6_help(char *str)
{
	pr_alert("%s gretaplan6: Enable/disable gretap lan ipv6 flow promotion", str);
	pr_alert("%s  gretaplan6 <1|0>\n", str);
}
static int cmd_f_gretaplan6(int argc, char *argv[])
{
	int mask = (FLOW_F_GRETAP6LAN);
	return cmd_f_enable(argc, argv, mask);
}

static void cmd_f_gretapwan_help(char *str)
{
	pr_alert("%s gretapwan: Enable/disable gretap wan flow promotion", str);
	pr_alert("%s  gretapwan <1|0>\n", str);
}
static int cmd_f_gretapwan(int argc, char *argv[])
{
	int mask = (FLOW_F_GRETAP4WAN |
		    FLOW_F_GRETAP6WAN);
	return cmd_f_enable(argc, argv, mask);
}

static void cmd_f_gretapwan4_help(char *str)
{
	pr_alert("%s gretapwan4: Enable/disable gretap wan ipv4 flow promotion", str);
	pr_alert("%s  gretapwan4 <1|0>\n", str);
}
static int cmd_f_gretapwan4(int argc, char *argv[])
{
	int mask = (FLOW_F_GRETAP4WAN);
	return cmd_f_enable(argc, argv, mask);
}

static void cmd_f_gretapwan6_help(char *str)
{
	pr_alert("%s gretapwan6: Enable/disable gretap wan ipv6 flow promotion", str);
	pr_alert("%s  gretapwan6 <1|0>\n", str);
}
static int cmd_f_gretapwan6(int argc, char *argv[])
{
	int mask = (FLOW_F_GRETAP6WAN);
	return cmd_f_enable(argc, argv, mask);
}

static void cmd_f_gretapethwan_help(char *str)
{
	pr_alert("%s gretapethwan: Enable/disable gretap ethwan flow promotion", str);
	pr_alert("%s  gretapethwan <1|0>\n", str);
}
static int cmd_f_gretapethwan(int argc, char *argv[])
{
	int mask = (FLOW_F_GRETAP4ETHWAN |
		    FLOW_F_GRETAP6ETHWAN);
	return cmd_f_enable(argc, argv, mask);
}

static void cmd_f_gretapethwan4_help(char *str)
{
	pr_alert("%s gretapethwan4: Enable/disable gretap ethwan ipv4 flow promotion", str);
	pr_alert("%s  gretapethwan4 <1|0>\n", str);
}
static int cmd_f_gretapethwan4(int argc, char *argv[])
{
	int mask = (FLOW_F_GRETAP4ETHWAN);
	return cmd_f_enable(argc, argv, mask);
}

static void cmd_f_gretapethwan6_help(char *str)
{
	pr_alert("%s gretapethwan6: Enable/disable gretap ethwan ipv6 flow promotion", str);
	pr_alert("%s  gretapethwan6 <1|0>\n", str);
}
static int cmd_f_gretapethwan6(int argc, char *argv[])
{
	int mask = (FLOW_F_GRETAP6ETHWAN);
	return cmd_f_enable(argc, argv, mask);
}

static void cmd_f_dslite_help(char *str)
{
	pr_alert("%s dslite: Enable/disable dslite flow promotion", str);
	pr_alert("%s  dslite <1|0>\n", str);
}
static int cmd_f_dslite(int argc, char *argv[])
{
	int mask = (FLOW_F_DSLITE);
	return cmd_f_enable(argc, argv, mask);
}

static void cmd_f_mapt_help(char *str)
{
	pr_alert("%s mapt: Enable/disable mapt flow promotion", str);
	pr_alert("%s  mapt <1|0>\n", str);
}
static int cmd_f_mapt(int argc, char *argv[])
{
	int mask = (FLOW_F_MAPT);
	return cmd_f_enable(argc, argv, mask);
}

static void cmd_f_macbr_help(char *str)
{
	pr_alert("%s macbr: Enable/disable mac bridge flow promotion", str);
	pr_alert("%s  macbr <1|0>\n", str);
}
static int cmd_f_macbr(int argc, char *argv[])
{
	int mask = (FLOW_F_MACBR);
	return cmd_f_enable(argc, argv, mask);
}

static void cmd_f_mcast_help(char *str)
{
	pr_alert("%s mcast: Enable/disable multicast flow promotion", str);
	pr_alert("%s  mcast <1|0>\n", str);
}
static int cmd_f_mcast(int argc, char *argv[])
{
	int mask = (FLOW_F_MCAST);
	return cmd_f_enable(argc, argv, mask);
}

static void cmd_f_mask_help(char *str)
{
	pr_alert("%s mask: Set flow promotion mask", str);
	pr_alert("%s  mask <hex mask>\n", str);
}
static int cmd_f_mask(int argc, char *argv[])
{
	int mask;

	if (argc == 2) {
		if (kstrtou32(argv[1], 0, &mask) == 0) {
			flowmgr.features = mask;
			goto done;
		}
	}
/* help */
	cmd_f_mask_help("");
done:
	return 0;
}

static struct proc_cmd_ops features_command_entries[] = {
	PROC_CMD_INIT("tcp",           cmd_f_tcp),
	PROC_CMD_INIT("udp",           cmd_f_udp),
	PROC_CMD_INIT("esp",           cmd_f_esp),
	PROC_CMD_INIT("ah",            cmd_f_ah),
	PROC_CMD_INIT("ipv4",          cmd_f_ipv4),
	PROC_CMD_INIT("ipv6",          cmd_f_ipv6),
	PROC_CMD_INIT("gretap4",       cmd_f_gretap4),
	PROC_CMD_INIT("gretap6",       cmd_f_gretap6),
	PROC_CMD_INIT("gretaplan",     cmd_f_gretaplan),
	PROC_CMD_INIT("gretaplan4",    cmd_f_gretaplan4),
	PROC_CMD_INIT("gretaplan6",    cmd_f_gretaplan6),
	PROC_CMD_INIT("gretapwan",     cmd_f_gretapwan),
	PROC_CMD_INIT("gretapwan4",    cmd_f_gretapwan4),
	PROC_CMD_INIT("gretapwan6",    cmd_f_gretapwan6),
	PROC_CMD_INIT("gretapethwan",  cmd_f_gretapethwan),
	PROC_CMD_INIT("gretapethwan4", cmd_f_gretapethwan4),
	PROC_CMD_INIT("gretapethwan6", cmd_f_gretapethwan6),
	PROC_CMD_INIT("dslite",        cmd_f_dslite),
	PROC_CMD_INIT("mapt",          cmd_f_mapt),
	PROC_CMD_INIT("macbr",         cmd_f_macbr),
	PROC_CMD_INIT("mcast",         cmd_f_mcast),
	PROC_CMD_INIT("mask",          cmd_f_mask),
};

struct proc_cmd_table features_command_table = {
	.module_name = "FLOWMGR",
	.size = ARRAY_SIZE(features_command_entries),
	.data_seq_read = (void *) &flowmgr_features_seq_ops,
	.ops = features_command_entries,
};

static void cmd_tu_port_track_timeout_help(char *str)
{
	pr_alert("%s Configure Port Clear Timeout\n", str);
	pr_alert("%s timeout <timeout>\n", str);
}
static int cmd_tu_port_track_timeout(int argc, char *argv[])
{
	u8 timeout = 90;

	if (argc == 2) {
		if (kstrtou8(argv[1], 0, &timeout) != 0) {
			pr_info("Port tracking: invalid timeout value\n");
		} else {
			flowmgr.tu_port_track_timeout = timeout;
			goto done;
		}
	}
/* help */
	cmd_tu_port_track_timeout_help("");
done:
	return 0;
}

static void cmd_tu_port_track_clear_help(char *str)
{
	pr_alert("%s Clear Timeout for specific port\n", str);
	pr_alert("%s clear <port>\n", str);
}
static int cmd_tu_port_track_clear(int argc, char *argv[])
{
	u16 port;

	if (argc == 2) {
		if (kstrtou16(argv[1], 0, &port) != 0) {
			pr_info("Port tracking: invalid port value\n");
		} else {
			flowmgr_tu_port_clear(port);
			goto done;
		}
	}
/* help */
	cmd_tu_port_track_clear_help("");
done:
	return 0;
}

static void *flowmgr_tu_port_track_seq_start(struct seq_file *seq, loff_t *pos)
{
	if (!*pos)
		return SEQ_START_TOKEN;
	return 0;
}

static void *flowmgr_tu_port_track_seq_next(struct seq_file *seq, void *v,
					 loff_t *pos)
{
	(*pos)++;
	return 0;
}

static void flowmgr_tu_port_track_seq_stop(struct seq_file *seq, void *v)
{
}

static int flowmgr_tu_port_track_seq_show(struct seq_file *seq, void *v)
{
	if (!v)
		return -1;
	flowmgr_tu_port_show(seq);
	return 0;
}

static const struct seq_operations flowmgr_tu_port_track_seq_ops = {
	.start	= flowmgr_tu_port_track_seq_start,
	.next	= flowmgr_tu_port_track_seq_next,
	.stop	= flowmgr_tu_port_track_seq_stop,
	.show	= flowmgr_tu_port_track_seq_show,
};

static struct proc_cmd_ops tu_port_track_command_entries[] = {
	PROC_CMD_INIT("timeout", cmd_tu_port_track_timeout),
	PROC_CMD_INIT("clear",   cmd_tu_port_track_clear),
};

struct proc_cmd_table tu_port_track_command_table = {
	.module_name = "FLOWMGR",
	.size = ARRAY_SIZE(tu_port_track_command_entries),
	.data_seq_read = (void *) &flowmgr_tu_port_track_seq_ops,
	.ops = tu_port_track_command_entries,
};

void flowmgr_procfs_init(void)
{
	flowmgr.proc_dir = proc_mkdir(FLOWMGR_PROC_DIR_NAME, NULL);
	if (flowmgr.proc_dir == NULL) {
		pr_warn("FLOWMGR Warning: cannot create /proc/%s\n",
			FLOWMGR_PROC_DIR_NAME);
		return;
	}
	proc_create_cmd("cmd", flowmgr.proc_dir, &command_table);
	proc_create_cmd("db", flowmgr.proc_dir, &db_command_table);
	proc_create_cmd("dbflow", flowmgr.proc_dir, &dbflow_command_table);
	proc_create_cmd("usage", flowmgr.proc_dir, &usage_command_table);

#ifdef CONFIG_BCM_FLOWMGR_MCAST
	proc_create_data("mdb", S_IRUGO, flowmgr.proc_dir,
			 &flowmgr_proc_mdb_fops, NULL);
#endif
#ifdef CONFIG_BCM_FLOWMGR_TUNNEL
	proc_create_cmd("tdb", flowmgr.proc_dir, &tdb_command_table);
#endif
	proc_create_data("status", S_IRUGO, flowmgr.proc_dir,
			 &flowmgr_proc_status_fops, NULL);
	proc_create_data("wandev_info", S_IRUGO, flowmgr.proc_dir,
			 &flowmgr_proc_active_wandev_info_fops, NULL);
	proc_create_cmd("device", flowmgr.proc_dir, &device_command_table);
	proc_create_cmd("counter", flowmgr.proc_dir, &counter_command_table);
	proc_create_cmd("fap", flowmgr.proc_dir, &fap_command_table);
	proc_create_cmd("features", flowmgr.proc_dir,
			&features_command_table);
	proc_create_cmd("tu_port_track", flowmgr.proc_dir,
			&tu_port_track_command_table);
}

void flowmgr_procfs_exit(void)
{
	if (flowmgr.proc_dir) {
		remove_proc_entry("db", flowmgr.proc_dir);
		remove_proc_entry("dbflow", flowmgr.proc_dir);
		remove_proc_entry("usage", flowmgr.proc_dir);
#ifdef CONFIG_BCM_FLOWMGR_MCAST
		remove_proc_entry("mdb", flowmgr.proc_dir);
#endif
#ifdef CONFIG_BCM_FLOWMGR_TUNNEL
		remove_proc_entry("tdb", flowmgr.proc_dir);
#endif
		remove_proc_entry("status", flowmgr.proc_dir);
		remove_proc_entry("wandev_info", flowmgr.proc_dir);
		remove_proc_entry("device", flowmgr.proc_dir);
		remove_proc_entry("counter", flowmgr.proc_dir);
		remove_proc_entry("fap", flowmgr.proc_dir);
		remove_proc_entry("cmd", flowmgr.proc_dir);
		remove_proc_entry("features", flowmgr.proc_dir);
		remove_proc_entry("tu_port_track", flowmgr.proc_dir);
		remove_proc_entry(FLOWMGR_PROC_DIR_NAME, NULL);
		flowmgr.proc_dir = NULL;
	}
}
