/*
 * nf_conntrack handling for offloading flows to acceleration engine
 *
 * Copyright (c) 2018 Broadcom Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation (or any later at your option).
 *
 * Author: Jayesh Patel <jayeshp@broadcom.com>
 */

#ifdef CONFIG_BCM_KF_CM

#ifndef _NF_CONNTRACK_OFFLOAD_H
#define _NF_CONNTRACK_OFFLOAD_H

#include <linux/netfilter/nf_conntrack_common.h>
#include <linux/netfilter/nf_conntrack_tuple_common.h>
#include <net/netfilter/nf_conntrack.h>
#include <net/netfilter/nf_conntrack_extend.h>
#include <net/netfilter/nf_conntrack_ecache.h>
#include <net/dsfield.h>
#include <linux/if_vlan.h>

#define NF_CT_OFFLOAD_SKIP_ACTIVE 0x1
#define NF_CT_OFFLOAD_SKIP_ENABLE 0x2

#define IPCT_OFFLOAD (IPCT_LABEL+1)

struct offload_info {
	struct ethhdr           eh;        /* Ethernet Header     */
	int                     flow_id;   /* Flow ID             */
	int                     flow_type; /* Flow Type           */
	int                     oif;       /* Out Interface Index */
	int                     iif;       /* In Interface Index  */
	int                     encap_tid; /* Encap Tunnel ID */
	int                     decap_tid; /* Decap Tunnel ID */
	int                     vlan_id;   /* VLAN ID */
	u32                     packets_slow;
	u32                     packets[2];/* Packet Count        */
	u32                     bytes;     /* Byte Count          */
	unsigned long           tstamp;    /* Timestamp           */
	enum ip_conntrack_info  ctinfo;    /* Last Conntrack Info */
	u8                      create_err;/* Flow Create error   */
	u8                      lag;
	u8                      vlan_untag;
	u8                      dscp_old;
	u8                      dscp_new;
	u8                      dscp_ovr;
	u8                      force_del;
	u8                      expected;
	u8                      skip;      /* Skip offload processing */
	union {
		u32             spi;       /* ESP SPI value       */
		u16             tu_port;   /* TCP/UDP port */
	};
	void                    *idb;
	void                    *odb;
	struct ethhdr           ehb;       /* Ethernet Header backup */
	int                     wifi_flowring;
	u8                      wifi_pri;
	u8                      start_promote;
	u32                     packets_promote;
	u32                     exp_deferred;
	int                     debug;
	/* Below this is not cleared */
	struct nf_bridge_info   *nf_bridge;
	void			*map;	   /* MAP information        */
	struct nf_conn          *ct;       /* For reverse lookup */
	u16                     err_dport;
};

struct nf_conn_offload {
	struct offload_info info[IP_CT_DIR_MAX];
	atomic_t slavecnt;
	u32 check_status_timeout;
	unsigned long crtstamp;  /* Create Timestamp    */
	void (*destructor)(struct nf_conn *ct,
			   struct nf_conn_offload *ct_offload);
	void (*update_stats)(const struct nf_conn *ct);
};

#define ct_offload_orig (ct_offload->info[IP_CT_DIR_ORIGINAL])
#define ct_offload_repl (ct_offload->info[IP_CT_DIR_REPLY])

#define nf_ct_offload_dump_full(pr_func, f, info, label) \
	do {\
		pr_func(f, "%s: offload_info eh(dmac=%pM smac=%pM) ebh(dmac=%pM smac=%pM)\n", label, \
		   (info)->eh.h_dest, (info)->eh.h_source, \
		   (info)->ehb.h_dest, (info)->ehb.h_source); \
		pr_func(f, "    ct=%px ctinfo=%d nf_bridge=%px oif=%d iif=%d odb=%px idb=%px map=%px\n", \
		   (info)->ct, (info)->ctinfo, (info)->nf_bridge, \
		   (info)->oif, (info)->iif, (info)->odb, (info)->idb, (info)->map); \
		pr_func(f, "    vlan_id=%d vlan_untag=%d dscp_old=%x dscp_new=%x\n", \
		   (info)->vlan_id, (info)->vlan_untag, \
		   (info)->dscp_old, (info)->dscp_new); \
		pr_func(f, "    encap_tid=%d decap_tid=%d spi=%x wifi_flowring=%d wifi_pri=%d start_promote=%d force_del=%d\n", \
		   (info)->encap_tid, (info)->decap_tid, (info)->spi, (info)->wifi_flowring, (info)->wifi_pri, \
		   (info)->start_promote, (info)->force_del); \
		pr_func(f, "    packets_slow=%d exp_deferred=%d packets_promote=%d packets[0]=%d packets[1]=%d bytes=%d\n", \
		   (info)->packets_slow, (info)->exp_deferred, (info)->packets_promote, (info)->packets[0], \
		   (info)->packets[1], (info)->bytes); \
		pr_func(f, "    flow_type=%d flow_id=%d lag=%d expected=%d create_err=%d\n", \
		   (info)->flow_type, (info)->flow_id, (info)->lag, \
		   (info)->expected, (info)->create_err); \
		pr_func(f, "    force_del=%d skip=%d tu_port=%d err_dport=%d debug=%d\n", \
		   (info)->force_del, (info)->skip, (info)->tu_port, \
		   (info)->err_dport, (info)->debug); \
	} while (0)

#define nf_ct_offload_dump(pr_func, info, label) \
	do {\
		pr_func("%s offload_info eh(dmac=%pM smac=%pM) ebh(dmac=%pM smac=%pM)\n", label, \
		   (info)->eh.h_dest, (info)->eh.h_source, \
		   (info)->ehb.h_dest, (info)->ehb.h_source); \
		pr_func("    ctinfo=%d nf_bridge=%p map=%p oif=%d iif=%d odb=%p idb=%p\n", \
		   (info)->ctinfo, (info)->nf_bridge, (info)->map, \
		   (info)->oif, (info)->iif, (info)->odb, (info)->idb); \
		pr_func("    vlan_id=%d vlan_untag=%d dscp_old=%x dscp_new=%x debug=%d\n", \
		   (info)->vlan_id, (info)->vlan_untag, \
		   (info)->dscp_old, (info)->dscp_new, (info)->debug); \
	} while (0)

#define nf_ct_tuple_dump(pr_func, tuple, label) \
	do {\
		if (tuple->src.l3num == PF_INET) { \
			pr_func("%s tuple src=%pI4 dst=%pI4 prot=%d sport=%d dport=%d\n", label, \
			   &tuple->src.u3.ip, &tuple->dst.u3.ip, tuple->dst.protonum, \
			   ntohs(tuple->src.u.tcp.port), ntohs(tuple->dst.u.tcp.port)); \
		} else { \
			pr_func("%s tuple src=%pI6 dst=%pI6 prot=%d sport %d dport %d\n", label, \
			   tuple->src.u3.ip6, tuple->dst.u3.ip6, tuple->dst.protonum, \
			   ntohs(tuple->src.u.tcp.port), ntohs(tuple->dst.u.tcp.port)); \
		} \
	} while (0)

static inline
struct nf_conn_offload *nf_conn_offload_find(const struct nf_conn *ct)
{
#ifdef CONFIG_NF_CONNTRACK_OFFLOAD
	return nf_ct_ext_find(ct, NF_CT_EXT_OFFLOAD);
#else
	return NULL;
#endif
}

extern struct nf_ct_event_notifier __rcu *nf_conntrack_offload_event_cb;

static inline
struct nf_conn_offload *nf_ct_offload_ext_add(struct nf_conn *ct, gfp_t gfp)
{
#ifdef CONFIG_NF_CONNTRACK_OFFLOAD
	struct net *net = nf_ct_net(ct);
	struct nf_conn_offload *ct_offload;
	u_int8_t protonum = nf_ct_protonum(ct);
	struct nf_ct_event_notifier *notify;
	struct nf_ct_event item = {
		.ct	= ct,
	};
	unsigned int eventmask = 1<<IPCT_NEW;

	if (!net->ct.sysctl_offload)
		return NULL;

	if ((protonum != IPPROTO_TCP) && (protonum != IPPROTO_UDP)
		&& (protonum != IPPROTO_ICMP) && (protonum != IPPROTO_ICMPV6)
		&& (protonum != IPPROTO_ESP) && (protonum != IPPROTO_AH))
		return NULL;

	ct_offload = nf_ct_ext_add(ct, NF_CT_EXT_OFFLOAD, gfp);
	if (ct_offload) {
		memset(ct_offload, 0, sizeof(struct nf_conn_offload));
		ct_offload_orig.ctinfo = -1;
		ct_offload_repl.ctinfo = -1;
		ct_offload_orig.flow_id = -1;
		ct_offload_repl.flow_id = -1;
		ct_offload_orig.encap_tid = -1;
		ct_offload_orig.decap_tid = -1;
		ct_offload_repl.encap_tid = -1;
		ct_offload_repl.decap_tid = -1;
		ct_offload_orig.ct = ct;
		ct_offload_repl.ct = ct;
		ct_offload_orig.wifi_flowring = 0xFFFF;
		ct_offload_repl.wifi_flowring = 0xFFFF;
		ct_offload->crtstamp = jiffies;
	}

	rcu_read_lock();
	notify = rcu_dereference(nf_conntrack_offload_event_cb);
	rcu_read_unlock();
	if (notify)
		notify->ct_event(eventmask, &item);

	return ct_offload;
#else
	return NULL;
#endif
};

static inline
int ip_get_dscp(struct sk_buff *skb)
{
	if (skb->protocol == htons(ETH_P_IP)) {
		if ((ip_hdr(skb)->protocol != IPPROTO_TCP) &&
		    (ip_hdr(skb)->protocol != IPPROTO_UDP) &&
		    (ip_hdr(skb)->protocol != IPPROTO_ESP) &&
		    (ip_hdr(skb)->protocol != IPPROTO_AH) &&
			(ip_hdr(skb)->protocol != IPPROTO_ICMP))
			return -1;
		return ipv4_get_dsfield(ip_hdr(skb)) >> 2;
	} else if (skb->protocol == htons(ETH_P_IPV6)) {
		if ((ipv6_hdr(skb)->nexthdr != IPPROTO_TCP) &&
		    (ipv6_hdr(skb)->nexthdr != IPPROTO_UDP) &&
		    (ipv6_hdr(skb)->nexthdr != IPPROTO_ESP) &&
		    (ipv6_hdr(skb)->nexthdr != IPPROTO_AH) &&
			(ipv6_hdr(skb)->nexthdr != IPPROTO_ICMPV6))
			return -1;
		return ipv6_get_dsfield(ipv6_hdr(skb)) >> 2;
	}

	return -1;
}

static inline
__u8 ip_get_slow_dscp_bp(struct sk_buff *skb)
{
	struct ethhdr *eh = (struct ethhdr *) eth_hdr(skb);
	struct ipv6hdr *ipv6h = NULL;
	struct iphdr *iph = NULL;
	struct vlan_hdr *vhdr = NULL;
	int eth_proto = eh->h_proto;
	int offset = ETH_HLEN;

repeat:
	if (eth_proto == htons(ETH_P_IP)) {
		iph =  (struct iphdr *) ((void *) eh + offset);
		return ipv4_get_dsfield(iph) >> 2;
	} else if (eth_proto == htons(ETH_P_IPV6)) {
		ipv6h =  (struct ipv6hdr *) ((void *) eh + offset);
		return ipv6_get_dsfield(ipv6h) >> 2;
	} else if (eth_proto == ntohs(ETH_P_8021Q)) {
		vhdr =  (struct vlan_hdr *) ((void *) eh + offset);
		eth_proto = vhdr->h_vlan_encapsulated_proto;
		offset += sizeof(struct vlan_hdr);
		goto repeat;
	}
	return U8_MAX;
}

static inline
void nf_ct_offload_update(struct sk_buff *skb)
{
#ifdef CONFIG_NF_CONNTRACK_OFFLOAD
	enum ip_conntrack_info ctinfo;
	struct nf_conn *ct = nf_ct_get(skb, &ctinfo);
	struct nf_conn_offload *ct_offload;
	int dscp;

	if (!skb_mac_header_was_set(skb))
		return;
	if (!ct)
		return;
	ct_offload = nf_conn_offload_find(ct);
	if (!ct_offload)
		return;
	dscp = ip_get_dscp(skb);
	if (dscp == -1)
		return;
	if (CTINFO2DIR(ctinfo) == IP_CT_DIR_ORIGINAL) {
		ct_offload_orig.dscp_old = dscp;
		ct_offload_orig.skip  &= ~NF_CT_OFFLOAD_SKIP_ACTIVE;
		if (!ct_offload_orig.eh.h_proto) {
			memcpy(&(ct_offload_orig.eh), eth_hdr(skb), ETH_HLEN);
			memcpy(&(ct_offload_orig.ehb), eth_hdr(skb), ETH_HLEN);
		}
	} else {
		ct_offload_repl.dscp_old = dscp;
		ct_offload_repl.skip  &= ~NF_CT_OFFLOAD_SKIP_ACTIVE;
		if (!ct_offload_repl.eh.h_proto) {
			memcpy(&(ct_offload_repl.eh), eth_hdr(skb), ETH_HLEN);
			memcpy(&(ct_offload_repl.ehb), eth_hdr(skb), ETH_HLEN);
		}
	}
#else
	return;
#endif
};

static inline
void nf_ct_offload_skip(struct sk_buff *skb)
{
#ifdef CONFIG_NF_CONNTRACK_OFFLOAD
	enum ip_conntrack_info ctinfo;
	struct nf_conn *ct = nf_ct_get(skb, &ctinfo);
	struct nf_conn_offload *ct_offload;

	if (!ct)
		return;
	ct_offload = nf_conn_offload_find(ct);
	if (!ct_offload)
		return;
	if (CTINFO2DIR(ctinfo) == IP_CT_DIR_ORIGINAL) {
		ct_offload_orig.skip  = NF_CT_OFFLOAD_SKIP_ENABLE;
		ct_offload_orig.skip |= NF_CT_OFFLOAD_SKIP_ACTIVE;
	} else {
		ct_offload_repl.skip  = NF_CT_OFFLOAD_SKIP_ENABLE;
		ct_offload_repl.skip |= NF_CT_OFFLOAD_SKIP_ACTIVE;
	}
#else
	return;
#endif
};

static inline
int is_nf_ct_offload_skip(const struct nf_conn *ct,
			  struct nf_conn_offload *ct_offload,
			  const enum ip_conntrack_info ctinfo)
{
#ifdef CONFIG_NF_CONNTRACK_OFFLOAD
	struct offload_info *ct_offload_info = &ct_offload->info[CTINFO2DIR(ctinfo)];

	return ct_offload_info->skip;
#else
	return 0;
#endif
}

static inline
void nf_ct_offload_update_wifi(struct sk_buff *skb, int flowring, u8 pri)
{
#ifdef CONFIG_NF_CONNTRACK_OFFLOAD
	enum ip_conntrack_info ctinfo;
	struct nf_conn *ct = nf_ct_get(skb, &ctinfo);
	struct nf_conn_offload *ct_offload;
	struct offload_info *ct_offload_info;

	if (!ct)
		return;
	ct_offload = nf_conn_offload_find(ct);
	if (!ct_offload)
		return;
	ct_offload_info = &ct_offload->info[CTINFO2DIR(ctinfo)];
	ct_offload_info->wifi_flowring = flowring;
	ct_offload_info->wifi_pri = pri;
	return;
#else
	return;
#endif
};

#ifdef CONFIG_NF_CONNTRACK_OFFLOAD
int nf_conntrack_offload_init(void);
void nf_conntrack_offload_fini(void);
int nf_conntrack_offload_pernet_init(struct net *net);
void nf_conntrack_offload_pernet_fini(struct net *net);
int nf_conntrack_offload_register_notifier(struct net *net,
					   struct nf_ct_event_notifier *new);
void nf_conntrack_offload_unregister_notifier(struct net *net,
					      struct nf_ct_event_notifier *new);
#else
static inline int nf_conntrack_offload_init(void)
{
	return 0;
}
static inline void nf_conntrack_offload_fini(void)
{
	return;
}
int nf_conntrack_offload_pernet_init(struct net *net)
{
	return 0;
}
void nf_conntrack_offload_pernet_fini(struct net *net)
{
	return;
}
int nf_conntrack_offload_register_notifier(struct net *net,
					   struct nf_ct_event_notifier *new)
{
	return 0;
}
void nf_conntrack_offload_unregister_notifier(struct net *net,
					      struct nf_ct_event_notifier *new)
{
	return;
}
#endif /* CONFIG_NF_CONNTRACK_OFFLOAD */

#endif /* _NF_CONNTRACK_OFFLOAD_H */

#endif /* CONFIG_BCM_KF_CM */
