 /****************************************************************************
 *
 * Broadcom Proprietary and Confidential.
 * (c) 2016 Broadcom. All rights reserved.
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Tim Ross <tross@broadcom.com>
 *****************************************************************************/
#include <linux/etherdevice.h>
#include "dqnet.h"
#include "dqnet_priv.h"
#include "dqnet_dbg.h"
#include "dqnet_brcmtag.h"
#include "fpm.h"

/*
 * Tag is constructed and desconstructed using byte by byte access
 * because the tag is placed after the MAC Source Address, which does
 * not make it 4-bytes aligned, so this might cause unaligned accesses
 * on most systems where this is used.
 */

/* Ingress and egress opcodes */
#define BRCM_OPCODE_SHIFT	5
#define BRCM_OPCODE_MASK	0x7

/* Ingress fields */
/* 1st byte in the tag */
#define BRCM_IG_TC_SHIFT	2
#define BRCM_IG_TC_MASK		0x7
/* 2nd byte in the tag */
#define BRCM_IG_TE_MASK		0x3
#define BRCM_IG_TS_SHIFT	7
/* 3rd byte in the tag */
#define BRCM_IG_DSTMAP2_MASK	1
#define BRCM_IG_DSTMAP1_MASK	0xff

/* Egress fields */
/* 2nd byte in the tag */
#define BRCM_EG_CID_MASK	0xff
/* 3rd byte in the tag */
#define BRCM_EG_RC_MASK		0xff
#define BRCM_EG_RC_RSVD		(3 << 6)
#define BRCM_EG_RC_EXCEPTION	(1 << 5)
#define BRCM_EG_RC_PROT_SNOOP	(1 << 4)
#define BRCM_EG_RC_PROT_TERM	(1 << 3)
#define BRCM_EG_RC_SWITCH	(1 << 2)
#define BRCM_EG_RC_MAC_LEARN	(1 << 1)
#define BRCM_EG_RC_MIRROR	(1 << 0)
#define BRCM_EG_TC_SHIFT	5
#define BRCM_EG_TC_MASK		0x7
#define BRCM_EG_PID_MASK	0x1f

int dqnet_decode_brcm_tag(u8 *brcm_tag, u8 *port)
{
	int status = 0;
	u8 code;

	code = (brcm_tag[0] >> BRCM_OPCODE_SHIFT) & BRCM_OPCODE_MASK;
	if (unlikely(code)) {
		pr_err("%s: Bad BRCM tag opcode %02x\n", __func__, code);
		status = -EINVAL;
		goto done;
	}
	code = brcm_tag[2] & BRCM_EG_RC_RSVD;
	if (unlikely(code)) {
		pr_err("%s: Bad BRCM tag reason code %02x\n", __func__, code);
		status = -EINVAL;
		goto done;
	}
	*port = brcm_tag[3] & BRCM_EG_PID_MASK;

done:
	if ((unlikely(status))) {
		pr_err("%s: tag: %02x %02x %02x %02x\n", __func__,
		       brcm_tag[0], brcm_tag[1], brcm_tag[2], brcm_tag[3]);
	}
	return status;
}

int dqnet_encode_brcm_tag(u8 port, u32 opcode, u32 priority, u8 *brcm_tag)
{
	int status = 0;

	/*
	 * Build the BRCM Tag.
	 * Set the ingress opcode, traffic class, tag enforcment is
	 * deprecated.
	 */
	brcm_tag[0] = ((opcode & BRCM_OPCODE_MASK) << BRCM_OPCODE_SHIFT) |
		((priority & BRCM_IG_TC_MASK) << BRCM_IG_TC_SHIFT);
	brcm_tag[1] = 0;
	brcm_tag[2] = 0;
	brcm_tag[3] = 0;
	if (port == 8)
		brcm_tag[2] = BRCM_IG_DSTMAP2_MASK;
	else
		brcm_tag[3] = (1 << port) & BRCM_IG_DSTMAP1_MASK;

	return status;
}
EXPORT_SYMBOL(dqnet_encode_brcm_tag);

static inline void memmoveb(void *dst, const void *src, int len)
{
	char *d = (char *) (dst + len - 1);
	char *s = (char *) (src + len - 1);
	while (len--)
		*d-- = *s--;
}

static inline void memmovef(void *dst, const void *src, int len)
{
	char *d = (char *) (dst);
	char *s = (char *) (src);
	while (len--)
		*d++ = *s++;
}

int dqnet_rm_brcm_tag(struct dqnet_netdev *ndev, struct fpm_buff *fb)
{
	int status = 0;
	u8 *brcm_tag;
	u8 port;
	u32 if_id, if_sub_id;

	if (!fb->brcm_tag_len)
		goto done;

	if (fb->type == BUF_TYPE_FPM)
		brcm_tag = fb->data + 2*ETH_ALEN;
	else
		brcm_tag = fb->skb->data + 2*ETH_ALEN;
	status = dqnet_decode_brcm_tag(brcm_tag, &port);
	if (status) {
		netif_err(ndev, rx_err, ndev->dev,
			  "=======================================\n");
		netif_err(ndev, rx_err, ndev->dev,
			  "RX DROP: unable to decode BRCM tag. buf type %d\n",
			  fb->type);
		show_rx_err_pkt_desc(ndev, fb);
		if (fb->type == BUF_TYPE_FPM)
			show_rx_err_pkt(ndev, fb->data, fb->len);
		else
			show_rx_err_pkt(ndev, fb->skb->data, fb->skb->len);
		netif_err(ndev, rx_err, ndev->dev,
			  "=======================================\n");
		goto done;
	}
	if_id = fb->if_id;
	if_sub_id = fb->if_sub_id;
	ndev->chan->port_to_if_id_subid(port, &if_id, &if_sub_id);
	if (if_id != fb->if_id || if_sub_id != fb->if_sub_id) {
		netif_err(ndev, rx_err, ndev->dev,
			  "=======================================\n");
		netif_err(ndev, rx_err, ndev->dev,
			  "RX DROP: BRCM tag, IF ID/sub-ID mismatch.\n");
		netif_err(ndev, rx_err, ndev->dev,
			  "port:          %d\n"
			  "if_id:         %d\n"
			  "fb->if_id:     %d\n"
			  "if_sub_id:     %d\n"
			  "fb->if_sub_id: %d\n",
			  port, if_id, fb->if_id, if_sub_id, fb->if_sub_id);
		if (fb->type == BUF_TYPE_FPM)
			netif_err(ndev, rx_err, ndev->dev,
				  "BRCM Tag:      %02x%02x%02x%02x\n",
				  (int)fb->data[12], (int)fb->data[13],
				  (int)fb->data[14], (int)fb->data[15]);
		else
			netif_err(ndev, rx_err, ndev->dev,
				  "BRCM Tag:      %02x%02x%02x%02x\n",
				  (int)fb->skb->data[12],
				  (int)fb->skb->data[13],
				  (int)fb->skb->data[14],
				  (int)fb->skb->data[15]);
		show_rx_err_pkt_desc(ndev, fb);
		if (fb->type == BUF_TYPE_FPM)
			show_rx_err_pkt(ndev, fb->data, fb->len);
		else
			show_rx_err_pkt(ndev, fb->skb->data, fb->skb->len);
		netif_err(ndev, rx_err, ndev->dev,
			  "=======================================\n");
		status = -EINVAL;
		goto done;
	}
	if (fb->type == BUF_TYPE_FPM) {
		memmoveb(fb->data + BRCM_TAG_LEN, fb->data, 2*ETH_ALEN);
		fb->data += BRCM_TAG_LEN;
		fb->offset += BRCM_TAG_LEN;
		fb->len -= BRCM_TAG_LEN;
	} else {
		memmoveb(fb->skb->data + BRCM_TAG_LEN, fb->skb->data, 2*ETH_ALEN);
		skb_pull(fb->skb, BRCM_TAG_LEN);
	}
	fb->brcm_tag_len = 0;

done:
	return status;
}

int dqnet_add_brcm_tag(struct dqnet_netdev *ndev, struct fpm_buff *fb)
{
	int status = 0;
	struct ethhdr *hdr;
	u8 port;
	u8 *tag;

	if (!fb->brcm_tag_len)
		goto done;

	if (fb->type == BUF_TYPE_FPM) {
		fb->data -= fb->brcm_tag_len;
		fb->offset -= fb->brcm_tag_len;
		fb->len += fb->brcm_tag_len;
		memmovef(fb->data, fb->data + fb->brcm_tag_len, 2*ETH_ALEN);
		tag = fb->data + 2*ETH_ALEN;
		hdr = (struct ethhdr *) fb->data;
	} else {
		if (skb_headroom(fb->skb) >= fb->brcm_tag_len) {
			skb_push(fb->skb, fb->brcm_tag_len);
			memmovef(fb->skb->data,
				fb->skb->data + fb->brcm_tag_len,
				2*ETH_ALEN);
			tag = fb->skb->data + 2*ETH_ALEN;
		} else {
			struct sk_buff *new_skb;
			new_skb = skb_realloc_headroom(fb->skb,
						       fb->brcm_tag_len);
			dev_kfree_skb(fb->skb);
			if (!new_skb)
				return -1;
			fb->skb = new_skb;
			skb_push(fb->skb, fb->brcm_tag_len);
			memmovef(fb->skb->data,
				fb->skb->data + fb->brcm_tag_len,
				2*ETH_ALEN);
			tag = fb->skb->data + 2*ETH_ALEN;
		}
		hdr = eth_hdr(fb->skb);
	}

	status = ndev->chan->if_id_subid_to_port(ndev->if_id,
		ndev->if_sub_id, &port);
	if (status) {
		netif_err(ndev, tx_err, ndev->dev,
			  "=======================================\n");
		netif_err(ndev, tx_err, ndev->dev,
			  "TX DROP: IF ID/sub-ID conversion to port #.\n");
		netif_err(ndev, tx_err, ndev->dev,
			  "if_id: %d\n", ndev->if_id);
		netif_err(ndev, tx_err, ndev->dev,
			  "if_sub_id: %d\n", ndev->if_sub_id);
		if (fb->type == BUF_TYPE_FPM)
			show_tx_err_pkt(ndev, fb->data, fb->len);
		else
			show_tx_err_pkt(ndev, fb->skb->data, fb->skb->len);
		netif_err(ndev, tx_err, ndev->dev,
			  "=======================================\n");
		goto done;

	}
	/* For broadcast/multicast packet use opcode=1 to send packet to specific port */
	/* For unicast packet use opcode=0 so that ARL table is updated with SRC mac address */
	/* Default brcmtag_opcode_mc is set to 1 and brcmtag_opcode_uc is set to 0 */
	if (is_multicast_ether_addr(hdr->h_dest))
		status = dqnet_encode_brcm_tag(port, ndev->brcmtag_opc_mc,
					       fb->priority, tag);
	else
		status = dqnet_encode_brcm_tag(port, ndev->brcmtag_opc_uc,
					       fb->priority, tag);
	if (status) {
		netif_err(ndev, tx_err, ndev->dev,
			  "=======================================\n");
		netif_err(ndev, tx_err, ndev->dev,
			  "TX DROP: Failed to create BRCM tag.\n");
		netif_err(ndev, tx_err, ndev->dev,
			  "if_id: %d\n", ndev->if_id);
		netif_err(ndev, tx_err, ndev->dev,
			  "if_sub_id: %d\n", ndev->if_sub_id);
		if (fb->type == BUF_TYPE_FPM)
			show_tx_err_pkt(ndev, fb->data, fb->len);
		else
			show_tx_err_pkt(ndev, fb->skb->data, fb->skb->len);
		netif_err(ndev, tx_err, ndev->dev,
			  "=======================================\n");
		goto done;
	}

done:
	return status;
}
