 /****************************************************************************
 *
 * Copyright (c) 2015-2018 Broadcom. All rights reserved
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Jayesh Patel <jayeshp@broadcom.com>
 ****************************************************************************/

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/rculist.h>
#include <linux/spinlock.h>
#include <linux/times.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/jhash.h>
#include <linux/random.h>
#include <linux/slab.h>
#include <linux/atomic.h>
#include <net/ip.h>
#include <net/ipv6.h>
#include <net/dst_metadata.h>
#include <net/ip6_tunnel.h>
#include "flowmgr.h"
#include <net/ip_tunnels.h>
#include <linux/if_arp.h>
#include <asm/unaligned.h>
#include "flowmgr.h"
#include "flowmgr_fap.h"
#include "flowmgr_map.h"
#include "flowmgr_fap_ops.h"
#include "flowmgr_tunnel.h"
#include "bcmflow.h"
#include "bcmnethooks.h"

#define FLOWMGR_HASH_BITS 8
#define FLOWMGR_HASH_SIZE (1 << FLOWMGR_HASH_BITS)

#define NETIF_F_DIS_OFF  (NETIF_F_TSO | NETIF_F_TSO6 | NETIF_F_GSO | NETIF_F_GRO)

static spinlock_t			hash_lock;
static struct hlist_head		hash[FLOWMGR_HASH_SIZE];

static struct kmem_cache *flowmgr_tdb_cache __read_mostly;

#ifdef CONFIG_BCM_FLOWMGR_FAP_WQ
struct flowmgr_tdb_work {
	struct work_struct work;
	union {
		struct flow_tunnel_params params;
		int id;
	};
	struct flowmgr_tdb_entry *db;
};

static void flowmgr_tdb_config_dslite_deferred(struct work_struct *w)
{
	struct flowmgr_tdb_work *work;
	struct flowmgr_tdb_entry *db;
	work = container_of(w, struct flowmgr_tdb_work, work);
	db = work->db;
	if (flowmgr_config_dslite(&work->params.dslite, db->oif) == 0) {
		db->id = work->params.dslite.tunnel_id;
	} else {
		db->oif = 0;
		db->state = 0;
		db->id = 0;
	}
	kfree(work);
}

int flowmgr_tdb_config_dslite(struct flow_dslite_tunnel_params *dslite,
			      struct flowmgr_tdb_entry *db,
			      int oindex)
{
	int ret = 0;
	struct flowmgr_tdb_work *work;
	work = kzalloc(sizeof(*work), GFP_ATOMIC);
	if (!work)
		return -ENOMEM;

	INIT_WORK(&work->work, flowmgr_tdb_config_dslite_deferred);
	work->params.dslite = *dslite;
	db->oif = oindex;
	db->state = 1;
	db->id = 0xDEFE22ED;
	work->db = db;
	queue_work_on(WORK_CPU_UNBOUND, kflowmgr_wq, &work->work);
	return ret;
}

static void flowmgr_tdb_config_gre_deferred(struct work_struct *w)
{
	struct flowmgr_tdb_work *work;
	struct flowmgr_tdb_entry *db;
	work = container_of(w, struct flowmgr_tdb_work, work);
	db = work->db;
	if (flowmgr_config_gre(&work->params.gre, db->oif) == 0) {
		db->id = work->params.gre.tunnel_id;
	} else {
		db->oif = 0;
		db->state = 0;
		db->id = 0;
	}
	kfree(work);
}

int flowmgr_tdb_config_gre(struct flow_gre_tunnel_params *gre,
			   struct flowmgr_tdb_entry *db,
			   int oindex)
{
	int ret = 0;
	struct flowmgr_tdb_work *work;
	work = kzalloc(sizeof(*work), GFP_ATOMIC);
	if (!work)
		return -ENOMEM;

	INIT_WORK(&work->work, flowmgr_tdb_config_gre_deferred);
	work->params.gre = *gre;
	db->oif = oindex;
	db->state = 1;
	db->id = 0xDEFE22ED;
	work->db = db;
	queue_work_on(WORK_CPU_UNBOUND, kflowmgr_wq, &work->work);
	return ret;
}
#else
int flowmgr_tdb_config_dslite(struct flow_dslite_tunnel_params *dslite,
			      struct flowmgr_tdb_entry *db,
			      int oindex)
{
	if (flowmgr_config_dslite(dslite, oindex) == 0) {
		db->oif = oindex;
		db->state = 1;
		db->id = dslite->tunnel_id;
	}
	return 0;
}

int flowmgr_tdb_config_gre(struct flow_gre_tunnel_params *gre,
			   struct flowmgr_tdb_entry *db,
			   int oindex)
{
	if (flowmgr_config_gre(gre, oindex) == 0) {
		db->oif = oindex;
		db->state = 1;
		db->id = gre->tunnel_id;
	}
	return 0;
}
#endif

int flowmgr_tdb_init(void)
{
	flowmgr_tdb_cache = kmem_cache_create("flowmgr_tdb_cache",
					      sizeof(struct flowmgr_tdb_entry),
					      0,
					      SLAB_HWCACHE_ALIGN, NULL);
	if (!flowmgr_tdb_cache)
		return -ENOMEM;

	return 0;
}

void flowmgr_tdb_fini(void)
{
	kmem_cache_destroy(flowmgr_tdb_cache);
}

static inline int flowmgr_tunnel_hash(int index)
{
	return index & (FLOWMGR_HASH_SIZE - 1);
}

static void tdb_rcu_free(struct rcu_head *head)
{
	struct flowmgr_tdb_entry *ent
		= container_of(head, struct flowmgr_tdb_entry, rcu);
	kmem_cache_free(flowmgr_tdb_cache, ent);
}

static void tdb_delete(struct flowmgr_tdb_entry *f)
{
	if (f && atomic_dec_and_test(&f->use)) {
		hlist_del_rcu(&f->hlist);
		call_rcu(&f->rcu, tdb_rcu_free);
	}
}

void flowmgr_tdb_flush(void)
{
	int i;

	spin_lock_bh(&hash_lock);
	for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
		struct flowmgr_tdb_entry *f;
		struct hlist_node *n;
		hlist_iterate_safe(f, n, &hash[i], hlist) {
			tdb_delete(f);
		}
	}
	spin_unlock_bh(&hash_lock);
}

/* Flush all entries referring to a specific type.
 * if do_all is set also flush static entries
 */
void flowmgr_db_delete_by_type(unsigned short type)
{
	int i;

	spin_lock_bh(&hash_lock);
	for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
		struct hlist_node *h, *g;

		hlist_for_each_safe(h, g, &hash[i]) {
			struct flowmgr_tdb_entry *f;
			f = hlist_entry(h, struct flowmgr_tdb_entry, hlist);
			if (f->type != type)
				continue;

			tdb_delete(f);
		}
	}
	spin_unlock_bh(&hash_lock);
}

/* Flush all entries referring to a specific type.
 * if do_all is set also flush static entries
 */
void flowmgr_db_delete_by_index(int index)
{
	int i;

	spin_lock_bh(&hash_lock);
	for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
		struct hlist_node *h, *g;

		hlist_for_each_safe(h, g, &hash[i]) {
			struct flowmgr_tdb_entry *f;
			f = hlist_entry(h, struct flowmgr_tdb_entry, hlist);
			if (f->ifi != index)
				continue;

			tdb_delete(f);
		}
	}
	spin_unlock_bh(&hash_lock);
}

struct flowmgr_tdb_entry *flowmgr_tdb_find_by_raddr(__be32 *addr)
{
	int i;

	spin_lock_bh(&hash_lock);
	for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
		struct hlist_node *h, *g;

		hlist_for_each_safe(h, g, &hash[i]) {
			struct flowmgr_tdb_entry *f;
			f = hlist_entry(h, struct flowmgr_tdb_entry, hlist);
			switch (f->type) {
			case ARPHRD_TUNNEL6:
			case ARPHRD_IP6GRE:
				if (f->parms.raddr.ip6[3] == addr[3]) {
					spin_unlock_bh(&hash_lock);
					return f;
				}
				break;
			case ARPHRD_TUNNEL:
			case ARPHRD_IPGRE:
				if (f->parms.raddr.ip == *addr) {
					spin_unlock_bh(&hash_lock);
					return f;
				}
				break;
			case ARPHRD_ETHER:
				if (f->parms.raddr.ip == *addr) {
					spin_unlock_bh(&hash_lock);
					return f;
				}
				break;
			default:
				break;
			}
		}
	}
	spin_unlock_bh(&hash_lock);
	return NULL;
}

struct flowmgr_tdb_entry *flowmgr_tdb_find_by_cat(__u8 cat)
{
	int i;

	spin_lock_bh(&hash_lock);
	for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
		struct hlist_node *h, *g;

		hlist_for_each_safe(h, g, &hash[i]) {
			struct flowmgr_tdb_entry *f;
			f = hlist_entry(h, struct flowmgr_tdb_entry, hlist);
			if (BCM_NETDEVICE_GROUP_CAT(f->group) == cat) {
				spin_unlock_bh(&hash_lock);
				return f;
			}
		}
	}
	spin_unlock_bh(&hash_lock);
	return NULL;
}

/* Internal Function: Find entry based on interface index in specific list */
static struct flowmgr_tdb_entry *tdb_find(struct hlist_head *head,
					  int index)
{
	struct flowmgr_tdb_entry *db;

	hlist_iterate(db, head, hlist) {
		if (db->ifi == index)
			return db;
	}
	return NULL;
}

/* Find entry based on interface index */
struct flowmgr_tdb_entry *flowmgr_tdb_find(int index)
{
	struct hlist_head *head = &hash[flowmgr_tunnel_hash(index)];
	struct flowmgr_tdb_entry *db = NULL;

	spin_lock_bh(&hash_lock);
	db = tdb_find(head, index);
	spin_unlock_bh(&hash_lock);
	return db;
}

/* Find entry based on type and domain */
struct flowmgr_tdb_entry *flowmgr_tdb_find_by_type_domain(unsigned short type,
							  unsigned short domain)
{
	int i;

	spin_lock_bh(&hash_lock);
	for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
		struct hlist_node *h, *g;

		hlist_for_each_safe(h, g, &hash[i]) {
			struct flowmgr_tdb_entry *f;
			f = hlist_entry(h, struct flowmgr_tdb_entry, hlist);
			if ((f->type == type) && (f->domain == domain)) {
				spin_unlock_bh(&hash_lock);
				return f;
			}
		}
	}
	spin_unlock_bh(&hash_lock);
	return NULL;
}

/* Internal Function: Create entry based on interface index */
static struct flowmgr_tdb_entry *tdb_create(struct hlist_head *head,
					    int ifindex)
{
	struct flowmgr_tdb_entry *db;
	int i;

	db = kmem_cache_alloc(flowmgr_tdb_cache, GFP_ATOMIC);
	if (db) {
		memset(db, 0, sizeof(struct flowmgr_tdb_entry));
		db->ifi = ifindex;
		atomic_set(&db->use, 1);
		db->created = jiffies;
		db->id = -1;
		for (i = 0; i < 8; i++)
			db->wifi_flowring[i] = 0xFFFF;
		hlist_add_head_rcu(&db->hlist, head);
	}
	return db;
}

/* Internal Function: Delete entry based on interface index */
static int db_delete_by_index(int index)
{
	struct hlist_head *head = &hash[flowmgr_tunnel_hash(index)];
	struct flowmgr_tdb_entry *db;

	while ((db = tdb_find(head, index)) != NULL) {
		if (db->id != -1) {
			/* For GRETAP tunnel device */
			if (db->type == ARPHRD_ETHER)
				flowmgr_unconfig_gre(db->id);
			else if (db->type == ARPHRD_TUNNEL6)
				flowmgr_unconfig_dslite(db->id);
		}
		tdb_delete(db);
	}

	return 0;
}

/* Delete entry based on interface index */
int flowmgr_tunnel_delete(struct net_device *dev)
{
	int ret;

	pr_info("FLOWMGR: Tunnel Delete %s\n", dev->name);
	spin_lock_bh(&hash_lock);
	ret = db_delete_by_index(dev->ifindex);
	if (ret == 0)
		dev->group = 0;
	spin_unlock_bh(&hash_lock);
	return ret;
}

/* Internal Function: Insert entry based on interface index */
static struct flowmgr_tdb_entry *tdb_insert(int index)
{
	struct hlist_head *head = &hash[flowmgr_tunnel_hash(index)];
	struct flowmgr_tdb_entry *db;

	db = tdb_find(head, index);
	if (!db)
		db = tdb_create(head, index);
	return db;
}

/* Insert entry based on input interface and mac address */
int flowmgr_tunnel_insert(struct net_device *dev)
{
	struct flowmgr_tdb_entry *db;

	spin_lock_bh(&hash_lock);
	db = tdb_insert(dev->ifindex);
	if (!db) {
		spin_unlock_bh(&hash_lock);
		return 0;
	}
	db->type = dev->type;
	db->parms.mtu = dev->mtu;
	switch (db->type) {
	case ARPHRD_TUNNEL6:
	case ARPHRD_IP6GRE:
		{
			struct ip6_tnl *nt;
			nt = netdev_priv(dev);
			memcpy(db->parms.laddr.in6.s6_addr,
			       nt->parms.laddr.s6_addr,
			       sizeof(struct in6_addr));
			memcpy(db->parms.raddr.in6.s6_addr,
			       nt->parms.raddr.s6_addr,
			       sizeof(struct in6_addr));
			dev->group = BCM_NETDEVICE_GROUP_CAT_VIR << 8 |
				BCM_NETDEVICE_GROUP_WAN;
			db->group = dev->group;
		}
		break;
	case ARPHRD_TUNNEL:
	case ARPHRD_IPGRE:
		{
			struct ip_tunnel *nt;
			nt = netdev_priv(dev);
			db->parms.laddr.ip = nt->parms.iph.saddr;
			db->parms.raddr.ip = nt->parms.iph.daddr;
			dev->group = BCM_NETDEVICE_GROUP_CAT_VIR << 8 |
				BCM_NETDEVICE_GROUP_WAN;
			db->group = dev->group;
		}
		break;

	case  ARPHRD_ETHER:
		{
			if (flowmgr_is_dev_gretap_tunnel(dev)) {
				struct ip_tunnel *nt;
				nt = netdev_priv(dev);
				db->parms.laddr.ip = nt->parms.iph.saddr;
				db->parms.raddr.ip = nt->parms.iph.daddr;
				dev->group =
				BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP << 8 |
				BCM_NETDEVICE_GROUP_WAN;
				db->group = dev->group;
				db->parms.tos = nt->parms.iph.tos;
				db->parms.ttl = nt->parms.iph.ttl;
				db->parms.iphid = nt->parms.iph.id;
				db->parms.frag_off = nt->parms.iph.frag_off;
			} else if (flowmgr_is_dev_ip6gretap_tunnel(dev)) {
				struct ip6_tnl *nt;
				nt = netdev_priv(dev);
				memcpy(db->parms.laddr.in6.s6_addr,
				   nt->parms.laddr.s6_addr,
				   sizeof(struct in6_addr));
				memcpy(db->parms.raddr.in6.s6_addr,
				   nt->parms.raddr.s6_addr,
				   sizeof(struct in6_addr));
				dev->group =
				BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP6 << 8 |
				BCM_NETDEVICE_GROUP_WAN;
				db->group = dev->group;
				db->parms.tos = ip6_tclass(nt->parms.flowinfo);
				db->parms.ttl = nt->parms.hop_limit;
				dev->features &= ~NETIF_F_DIS_OFF;
				dev->wanted_features &= ~NETIF_F_DIS_OFF;
				rtnl_lock();
				netdev_update_features(dev);
				rtnl_unlock();
			}
		}
		break;
	}
	spin_unlock_bh(&hash_lock);
	pr_info("FLOWMGR: Tunnel Insert %s\n", dev->name);
	return 0;
}

static int flowmgr_tunnel_gre_reconfig(struct flowmgr_tdb_entry *db)
{
	struct flow_gre_tunnel_params gre_header;

	memset(&gre_header, 0,
	sizeof(struct flow_gre_tunnel_params));
	if (db->id == -1) {
		pr_err("FLOWMGR: Reconfig error - GRE Tunnel not configured!!");
		return -1;
	}
	gre_header.tunnel_id = db->id;
	if (BCM_NETDEVICE_GROUP_CAT(db->group) ==
	    BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP) {
		gre_header.type = ft_ipv4;
		gre_header.remote_ip[0] = db->parms.raddr.in.s_addr;
		gre_header.local_ip[0] = db->parms.laddr.in.s_addr;
		/* GFAP/Runner needds to know the physical link MTU, so
		   add back the GRE overhead = Inner ethernet header (14)
		   + GRE header (4)
		   + Outer IPv4 header (20)
		   + Outer ethernet header (14) */
		gre_header.encap_mtu = db->parms.mtu+ 38 + 14;
		/* Add VLAN header for Inner Packet if present */
		if (db->parms.vlan_tag)
			gre_header.encap_mtu += 4;
	} else if (BCM_NETDEVICE_GROUP_CAT(db->group) ==
		 BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP6) {
		gre_header.type = ft_ipv6;
		memcpy((char *)gre_header.remote_ip,
			   db->parms.raddr.in6.s6_addr,
			   sizeof(struct in6_addr));
		memcpy((char *)gre_header.local_ip,
			   db->parms.laddr.in6.s6_addr,
			   sizeof(struct in6_addr));
		/* GFAP/Runner needds to know the physical link MTU, so
		   add back the GRE overhead = Inner ethernet header (14)
		   + GRE header (4)
		   + Outer IPv6 header (40)
		   + Outer ethernet header (14) */
		gre_header.encap_mtu = db->parms.mtu+ 58 + 14;
		/* Add VLAN header for Inner Packet if present */
		if (db->parms.vlan_tag)
			gre_header.encap_mtu += 4;
	}
	ether_addr_copy(gre_header.remote_mac, db->parms.eh.h_dest);
	ether_addr_copy(gre_header.local_mac, db->parms.eh.h_source);
	gre_header.vlan_tag = db->parms.vlan_tag;
	flowmgr_tdb_config_gre(&gre_header, db, db->oif);
	db->created = jiffies;
	return 0;
}

/* Insert entry based on input interface and mac address */
int flowmgr_tunnel_insert_v4link(struct net_device *dev, struct sk_buff *skb,
				 struct nf_conn_offload *ct_offload)
{
	struct hlist_head *head;
	struct flowmgr_tdb_entry *db;
	struct ip_tunnel_info *tun_info;
	union nf_inet_addr laddr;
	union nf_inet_addr raddr;
	const struct net_device_ops *netdev_ops = NULL;

	if (!flowmgr_is_dev_gretap_tunnel(dev))
		return 0;
	head = &hash[flowmgr_tunnel_hash(dev->ifindex)];
	/* tun_info is defined for Flow based Tunnel */
	tun_info = skb_tunnel_info(skb);
	if (unlikely(!tun_info || !(tun_info->mode & IP_TUNNEL_INFO_TX) ||
		ip_tunnel_info_af(tun_info) != AF_INET))
		return 0;
	laddr.ip = tun_info->key.u.ipv4.src;
	raddr.ip = tun_info->key.u.ipv4.dst;
	/*The following in NOT a coerity error because the array aritmatic in the called function is limited to the size of ip */
	/* coverity [ARRAY_VS_SINGLETON] */
	db = flowmgr_tdb_find_by_raddr(&raddr.ip);
	if (db) {
		if (db->parms.mtu != dev->mtu) {
			pr_alert("FLOWMGR: Tunnel v4link %s MTU changed %d -> %d\n",
				 dev->name, db->parms.mtu, dev->mtu);
			db->parms.mtu = dev->mtu;
			flowmgr_tunnel_gre_reconfig(db);
		}
		return 0;
	}

	spin_lock_bh(&hash_lock);
	db = tdb_find(head, dev->ifindex);
	if (!db) {
		pr_err("FLOWMGR: Add %s to flowmgr first!\n", dev->name);
		spin_unlock_bh(&hash_lock);
		return 0;
	}

	/* If the current tdb entry is taken, create another entry for the new tunnel */
	if (db->parms.raddr.ip != 0) {
		netdev_ops = db->netdev_ops;
		db = tdb_create(head, dev->ifindex);
		if (!db) {
			spin_unlock_bh(&hash_lock);
			return 0;
		}
	}

	db->type = dev->type;
	db->parms.laddr.ip = laddr.ip;
	db->parms.raddr.ip = raddr.ip;
	dev->group =
		BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP << 8 |
		BCM_NETDEVICE_GROUP_WAN;
	db->group = dev->group;
	if (!db->netdev_ops)
		db->netdev_ops = netdev_ops;
	db->parms.mtu = dev->mtu;
	spin_unlock_bh(&hash_lock);
	pr_info("FLOWMGR: Tunnel link Insert %s\n", dev->name);
	return 0;
}


int flowmgr_tunnel_insert_v6link(struct net_device *dev, struct sk_buff *skb,
				 struct nf_conn_offload *ct_offload)
{
	struct hlist_head *head;
	struct flowmgr_tdb_entry *db;
	struct ip_tunnel_info *tun_info;
	union nf_inet_addr laddr;
	union nf_inet_addr raddr;
	const struct net_device_ops *netdev_ops = NULL;

	if (!flowmgr_is_dev_ip6gretap_tunnel(dev))
		return 0;

	head = &hash[flowmgr_tunnel_hash(dev->ifindex)];
	/* tun_info is defined for Flow based Tunnel */
	tun_info = skb_tunnel_info(skb);

	if (unlikely(!tun_info ||
		!(tun_info->mode & IP_TUNNEL_INFO_TX) ||
		ip_tunnel_info_af(tun_info) != AF_INET6))
		return 0;

	memcpy(laddr.ip6, &tun_info->key.u.ipv6.src,
		   sizeof(struct in6_addr));
	memcpy(raddr.ip6, &tun_info->key.u.ipv6.dst,
		   sizeof(struct in6_addr));
	db = flowmgr_tdb_find_by_raddr(raddr.ip6);
	if (db) {
		if (db->parms.mtu != dev->mtu) {
			pr_alert("FLOWMGR: Tunnel v6link %s MTU changed %d -> %d\n",
				 dev->name, db->parms.mtu, dev->mtu);
			db->parms.mtu = dev->mtu;
			flowmgr_tunnel_gre_reconfig(db);
		}
		return 0;
	}
	spin_lock_bh(&hash_lock);
	db = tdb_find(head, dev->ifindex);
	if (!db) {
		pr_err("FLOWMGR: Add %s to flowmgr first!\n", dev->name);
		spin_unlock_bh(&hash_lock);
		return 0;
	}

	/* If the current tdb entry is taken, create another entry for the new tunnel */
	if (db->parms.raddr.in6.s6_addr32[0] != 0) {
		netdev_ops = db->netdev_ops;
		db = tdb_create(head, dev->ifindex);
		if (!db) {
			spin_unlock_bh(&hash_lock);
			return 0;
		}
	}
	db->type = dev->type;
	memcpy(db->parms.laddr.in6.s6_addr,
			   laddr.ip6, sizeof(struct in6_addr));
	memcpy(db->parms.raddr.in6.s6_addr,
			   raddr.ip6, sizeof(struct in6_addr));
	dev->group =
		BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP6 << 8 |
		BCM_NETDEVICE_GROUP_WAN;
	db->group = dev->group;

	if (!db->netdev_ops)
		db->netdev_ops = netdev_ops;

	db->parms.mtu = dev->mtu;
	spin_unlock_bh(&hash_lock);
	pr_info("FLOWMGR: Tunnel link Insert %s\n", dev->name);
	return 0;
}

struct flowmgr_tdb_entry *flowmgr_tunnel_find_mapt_domain(int domain)
{
	struct flowmgr_tdb_entry *db;

	db = flowmgr_tdb_find_by_type_domain(ARPHRD_VOID,
					     (unsigned short)domain);

	return db;
}

int flowmgr_tunnel_mapt_insert(struct net_device *dev,
			       struct flow_mapt_params *params)
{
	struct flowmgr_tdb_entry *db;

	spin_lock_bh(&hash_lock);
	db = tdb_insert(dev->ifindex);
	if (!db) {
		spin_unlock_bh(&hash_lock);
		return -1;
	}

	db->type = ARPHRD_VOID;
	db->state = 0;
	db->domain = (unsigned short)params->domain_index;
	memcpy(&db->mapt, params, sizeof(struct flow_mapt_params));

	spin_unlock_bh(&hash_lock);

	pr_info("FLOWMGR: Tunnel (MAPT) Insert %s\n", dev->name);
	return 0;
}

void flowmgr_tunnel_mapt_delete(int domain)
{
	struct flowmgr_tdb_entry *db = NULL;
	struct net_device *dev = NULL;
	int i;
	spin_lock_bh(&hash_lock);
	for (i = 0; i < FLOWMGR_HASH_SIZE && !db; i++) {
		struct hlist_node *h, *g;

		hlist_for_each_safe(h, g, &hash[i]) {
			struct flowmgr_tdb_entry *f;
			f = hlist_entry(h, struct flowmgr_tdb_entry, hlist);
			if ((f->type == ARPHRD_VOID) && (f->domain == domain)) {
				db = f;
				dev = __dev_get_by_index(&init_net, db->ifi);
				break;
			}
		}
	}
	if (db)
		tdb_delete(db);
	spin_unlock_bh(&hash_lock);

	if (dev)
		pr_info("FLOWMGR: Tunnel (MAPT) Delete %s\n", dev->name);
	else
		pr_warn("FLOWMGR Warning: Tunnel (MAPT) Delete - cannot find domain=%d\n", domain);
}

/* Internal Function: Update entry based on input interface and mac address */
static int tdb_update(int index,
		      int oif,
		      const unsigned char *h_dest,
		      const unsigned char *h_source,
		      __be16 h_vlan_TCI)
{
	struct hlist_head *head = &hash[flowmgr_tunnel_hash(index)];
	struct flowmgr_tdb_entry *db;

	db = tdb_find(head, index);
	if (db) {
		db->oif = oif;
		memcpy(db->parms.eh.h_dest, h_dest, ETH_ALEN);
		memcpy(db->parms.eh.h_source, h_source, ETH_ALEN);
		db->parms.vlan_tag = h_vlan_TCI;
		db->created = jiffies;
	}

	return 0;
}

/* Update entry: mac address */
int flowmgr_tunnel_update(int ifindex, int oif,
			  const unsigned char *h_dest,
			  const unsigned char *h_source,
			  __be16 h_vlan_TCI)
{
	int ret;

	spin_lock_bh(&hash_lock);
	ret = tdb_update(ifindex, oif, h_dest, h_source, h_vlan_TCI);
	spin_unlock_bh(&hash_lock);
	return ret;
}

/* Get first entry in specific list */
struct hlist_node  *flowmgr_tdb_get_first(int *hashid)
{
	struct hlist_node *h;
	int i;

	for (i = 0; i < FLOWMGR_HASH_SIZE; i++) {
		h = rcu_dereference(hlist_first_rcu(&hash[i]));
		if (h) {
			*hashid = i;
			return h;
		}
	}
	return NULL;
}

/* Get next entry in specific list */
struct hlist_node *flowmgr_tdb_get_next(int *hashid,
					struct hlist_node *head)
{
	head = rcu_dereference(hlist_next_rcu(head));
	while (!head) {
		if (++*hashid >= FLOWMGR_HASH_SIZE)
			return NULL;
		head = rcu_dereference(
				hlist_first_rcu(
				   &hash[*hashid]));
	}
	return head;
}

/* Get index in specific list */
struct hlist_node *flowmgr_tdb_get_idx(int *hashid, loff_t pos)
{
	struct hlist_node *head = flowmgr_tdb_get_first(hashid);

	if (head)
		while (pos && (head = flowmgr_tdb_get_next(hashid, head)))
			pos--;
	return pos ? NULL : head;
}

int  flowmgr_tunnel_process(struct sk_buff *skb,
			    const struct net_device *in,
			    const struct net_device *out,
			    int protonum)
{
	struct ethhdr *eh;
	__be16 eth_proto;
	__be16 vlan_tag = 0;
	struct flowmgr_tdb_entry *db = NULL;
	struct ipv6hdr *ipv6h;
	struct iphdr *iph;
	struct net_device *tunneldev;
	int wifi_flowring;
	u8  wifi_pri;
	int skb_pri = skb->priority & 7;

	int odev_cat = BCM_NETDEVICE_GROUP_CAT(out->group);

	if ((odev_cat != BCM_NETDEVICE_GROUP_CAT_PHY) &&
	    (odev_cat != BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP) &&
	    (odev_cat != BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP6))
		return -1;

	eh = eth_hdr(skb);
	eth_proto = eh->h_proto;
	if (eth_proto == htons(ETH_P_8021Q)) {
		struct vlan_ethhdr *veh = vlan_eth_hdr(skb);
		eth_proto = veh->h_vlan_encapsulated_proto;
		vlan_tag = ntohs(veh->h_vlan_TCI);
		skb_set_network_header(skb, VLAN_ETH_HLEN);
	}

	//if (in) {
	//	db = flowmgr_tdb_find(in->ifindex);
	if (eth_proto == htons(ETH_P_IPV6)) {
		ipv6h =  (struct ipv6hdr *) skb_network_header(skb);
		db = flowmgr_tdb_find_by_raddr(ipv6h->daddr.s6_addr32);
	} else if (eth_proto == htons(ETH_P_IP)) {
		iph =  (struct iphdr *) skb_network_header(skb);
		db = flowmgr_tdb_find_by_raddr(&(iph->daddr));
	} else {
		return -1;
	}

	if (db && (db->state == 0) && (db->type == ARPHRD_TUNNEL6)) {
		struct flow_dslite_tunnel_params dslite_header = {};
		memcpy(db->parms.eh.h_dest, eh->h_dest, ETH_ALEN);
		memcpy(db->parms.eh.h_source, eh->h_source, ETH_ALEN);
		memcpy(dslite_header.remote_mac, db->parms.eh.h_dest, ETH_ALEN);
		memcpy(dslite_header.local_mac, db->parms.eh.h_source, ETH_ALEN);
		memcpy((char *)dslite_header.remote_ip,
		       db->parms.raddr.in6.s6_addr,
		       sizeof(struct in6_addr));
		memcpy((char *)dslite_header.local_ip,
		       db->parms.laddr.in6.s6_addr,
		       sizeof(struct in6_addr));
		flowmgr_tdb_config_dslite(&dslite_header, db, out->ifindex);
		db->created = jiffies;
		return 0;
	}

	/* GRE shows up as type ethernet... */
	if (!db)
		return -1;

	if (db->type != ARPHRD_ETHER)
		return -1;

	if (db->state) {
		tunneldev = __dev_get_by_index(&init_net, db->ifi);
		if (!tunneldev)
			return -1;

		if ((BCM_NETDEVICE_GROUP_TYPE(out->group) ==
		    BCM_NETDEVICE_GROUP_LAN) &&
			fap()->flow_get_dhdol) {

			fap()->flow_get_dhdol(skb, out,
					      &wifi_flowring,
					      &wifi_pri);
			if ((db->wifi_flowring[skb_pri] != wifi_flowring) ||
			    (db->wifi_pri[skb_pri] != wifi_pri)) {
				pr_debug("FLOWMGR: Tunnel %s skb pri %d WIFI Flowring changed %d -> %d WIFI Pri changed %d -> %d\n",
					 tunneldev->name, skb->priority,
					 db->wifi_flowring[skb_pri], wifi_flowring,
					 db->wifi_pri[skb_pri], wifi_pri);
				db->wifi_mask |= (skb_pri<<1);
				db->wifi_flowring[skb_pri] = wifi_flowring;
				db->wifi_pri[skb_pri] = wifi_pri;
			}
		}

		if (db->parms.mtu != tunneldev->mtu) {
			pr_alert("FLOWMGR: Tunnel %s MTU changed %d -> %d\n",
				 tunneldev->name, db->parms.mtu, tunneldev->mtu);
			db->parms.mtu = tunneldev->mtu;
			flowmgr_tunnel_gre_reconfig(db);
			return 0;
		}
		return -1;
	}


	if (!in)
		return -1;

	if (BCM_NETDEVICE_GROUP_CAT(in->group) ==
		BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP) {
		/* IPv4 GRE */
		struct flow_gre_tunnel_params gre_header;
		struct iphdr *iph = ip_hdr(skb);

		tunneldev = __dev_get_by_index(&init_net, db->ifi);
		if (!tunneldev)
			return -1;

		memset(&gre_header, 0,
		sizeof(struct flow_gre_tunnel_params));
		gre_header.tunnel_id = 0;
		gre_header.type = ft_ipv4;
		ether_addr_copy(db->parms.eh.h_dest, eh->h_dest);
		ether_addr_copy(db->parms.eh.h_source, eh->h_source);
		db->parms.vlan_tag = vlan_tag;
		ether_addr_copy(gre_header.remote_mac, db->parms.eh.h_dest);
		ether_addr_copy(gre_header.local_mac, db->parms.eh.h_source);
		if (!db->parms.laddr.ip)
			db->parms.laddr.ip = iph->saddr;
		gre_header.remote_ip[0] = db->parms.raddr.in.s_addr;
		gre_header.local_ip[0] = db->parms.laddr.in.s_addr;
		gre_header.tos = db->parms.tos;
		gre_header.ttl = db->parms.ttl;
		gre_header.id = ntohs(iph->id);
		gre_header.frag_off = ntohs(iph->frag_off);
		gre_header.check = 0;
		gre_header.tot_len = 0;
		gre_header.vlan_tag = db->parms.vlan_tag;
		/* GFAP/Runner needds to know the physical link MTU, so
		   add back the GRE overhead = Inner ethernet header (14)
		   + GRE header (4)
		   + Outer IPv4 header (20)
		   + Outer ethernet header (14) */
		gre_header.encap_mtu = tunneldev->mtu+ 38 + 14;
		/* Add VLAN header for Inner Packet if present */
		if (db->parms.vlan_tag)
			gre_header.encap_mtu += 4;
		if ((BCM_NETDEVICE_GROUP_TYPE(out->group) ==
		    BCM_NETDEVICE_GROUP_LAN) &&
			fap()->flow_get_dhdol) {
			fap()->flow_get_dhdol(skb, out,
					      &wifi_flowring,
					      &wifi_pri);
			db->wifi_mask |= (skb_pri<<1);
			db->wifi_flowring[skb_pri] = wifi_flowring;
			db->wifi_pri[skb_pri] = wifi_pri;
		}
		flowmgr_tdb_config_gre(&gre_header, db, out->ifindex);
		db->created = jiffies;
		return 0;
	} else if (BCM_NETDEVICE_GROUP_CAT(in->group) ==
				BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP6) {
		/* IPv6 GRE */
		struct flow_gre_tunnel_params gre_header;
		struct ipv6hdr *ipv6h = ipv6_hdr(skb);
		//struct ip6_tnl *nt;
		tunneldev = __dev_get_by_index(&init_net, db->ifi);
		if (!tunneldev)
			return -1;

		memset(&gre_header, 0,
		sizeof(struct flow_gre_tunnel_params));
		gre_header.tunnel_id = 0;
		gre_header.type = ft_ipv6;
		ether_addr_copy(db->parms.eh.h_dest, eh->h_dest);
		ether_addr_copy(db->parms.eh.h_source, eh->h_source);
		db->parms.vlan_tag = vlan_tag;
		ether_addr_copy(gre_header.remote_mac, db->parms.eh.h_dest);
		ether_addr_copy(gre_header.local_mac, db->parms.eh.h_source);
		if (!db->parms.laddr.in6.s6_addr32[0])
			memcpy(db->parms.laddr.in6.s6_addr,
				   ipv6h->saddr.s6_addr32,
				   sizeof(struct in6_addr));
		memcpy((char *)gre_header.remote_ip,
			   db->parms.raddr.in6.s6_addr,
			   sizeof(struct in6_addr));
		memcpy((char *)gre_header.local_ip,
			   db->parms.laddr.in6.s6_addr,
			   sizeof(struct in6_addr));
		gre_header.tos = db->parms.tos;
		gre_header.ttl = db->parms.ttl;
		gre_header.check = 0;
		gre_header.tot_len = 0;
		gre_header.vlan_tag = db->parms.vlan_tag;
		/* GFAP/Runner needds to know the physical link MTU, so
		   add back the GRE overhead = Inner ethernet header (14)
		   + GRE header (4)
		   + Outer IPv6 header (40)
		   + Outer ethernet header (14) */
		gre_header.encap_mtu = tunneldev->mtu+ 58 + 14;
		/* Add VLAN header for Inner Packet if present */
		if (db->parms.vlan_tag)
			gre_header.encap_mtu += 4;
		if ((BCM_NETDEVICE_GROUP_TYPE(out->group) ==
		    BCM_NETDEVICE_GROUP_LAN) &&
			fap()->flow_get_dhdol) {
			fap()->flow_get_dhdol(skb, out,
					      &wifi_flowring,
					      &wifi_pri);
			db->wifi_mask |= (skb_pri<<1);
			db->wifi_flowring[skb_pri] = wifi_flowring;
			db->wifi_pri[skb_pri] = wifi_pri;
		}
		flowmgr_tdb_config_gre(&gre_header, db, out->ifindex);
		db->created = jiffies;
		return 0;
	}
	return -1;
}

int  flowmgr_tunnel_out_ifindex(const struct net_device *dev,
				struct flow_tunnel_params *params,
				struct flow_wifi_params *wifi)
{
	struct flowmgr_tdb_entry *db;
	db = flowmgr_tdb_find(dev->ifindex);
	if (!db)
		return 0;

	if (db->type == ARPHRD_TUNNEL6) {
		params->type = ARPHRD_TUNNEL6;
		memcpy(params->dslite.remote_mac, db->parms.eh.h_dest,
		       ETH_ALEN);
		memcpy(params->dslite.local_mac, db->parms.eh.h_source,
		       ETH_ALEN);
		memcpy((char *)params->dslite.remote_ip,
		       db->parms.raddr.in6.s6_addr,
		       sizeof(struct in6_addr));
		memcpy((char *)params->dslite.local_ip,
		       db->parms.laddr.in6.s6_addr,
		       sizeof(struct in6_addr));
		params->dslite.tunnel_id = db->id;
		if (wifi) {
			wifi->flowring = db->wifi_flowring[wifi->skb_pri];
			wifi->pri = db->wifi_pri[wifi->skb_pri];
			pr_debug("wifi: flowring %d pri %d\n", wifi->flowring,
				 wifi->pri );

		}
		return db->oif;
	} else if (db->type == ARPHRD_ETHER) {
		if (BCM_NETDEVICE_GROUP_CAT(dev->group) ==
		    BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP) {
			params->type = ARPHRD_ETHER;
			memcpy(params->gre.remote_mac, db->parms.eh.h_dest,
			       ETH_ALEN);
			memcpy(params->gre.local_mac, db->parms.eh.h_source,
			       ETH_ALEN);
			params->gre.remote_ip[0] = db->parms.raddr.in.s_addr;
			params->gre.local_ip[0] = db->parms.laddr.in.s_addr;
			params->gre.vlan_tag = db->parms.vlan_tag;
		} else if (BCM_NETDEVICE_GROUP_CAT(dev->group) ==
			  BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP6) {
			params->type = ARPHRD_ETHER;
			memcpy(params->gre.remote_mac, db->parms.eh.h_dest,
			       ETH_ALEN);
			memcpy(params->gre.local_mac, db->parms.eh.h_source,
			       ETH_ALEN);
			memcpy((char *)params->gre.remote_ip,
			       db->parms.raddr.in6.s6_addr,
			       sizeof(struct in6_addr));
			memcpy((char *)params->gre.local_ip,
			       db->parms.laddr.in6.s6_addr,
			       sizeof(struct in6_addr));
			params->gre.vlan_tag = db->parms.vlan_tag;
		}
		if (wifi) {
			wifi->flowring = db->wifi_flowring[wifi->skb_pri];
			wifi->pri = db->wifi_pri[wifi->skb_pri];
			pr_debug("wifi: flowring %d pri %d\n", wifi->flowring,
				 wifi->pri );

		}
		return db->oif;
	}
	return 0;
}

static struct flowmgr_tdb_entry *tdb_find_id(struct hlist_head *head,
					     int id)
{
	struct flowmgr_tdb_entry *db;

	hlist_iterate(db, head, hlist) {
		if (db->id == id)
			return db;
	}
	return NULL;
}

int  flowmgr_tunnelid_out_ifindex(const struct net_device *dev, int id,
				  struct flow_tunnel_params *params,
				  struct flow_wifi_params *wifi)
{
	struct hlist_head *head = &hash[flowmgr_tunnel_hash(dev->ifindex)];
	struct flowmgr_tdb_entry *db;

	if (id == -1)
		return flowmgr_tunnel_out_ifindex(dev, params, wifi);

	db = tdb_find_id(head, id);
	if (!db)
		return 0;

	if (db->type == ARPHRD_ETHER) {
		if (BCM_NETDEVICE_GROUP_CAT(dev->group) ==
		    BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP) {
			params->type = ARPHRD_ETHER;
			memcpy(params->gre.remote_mac, db->parms.eh.h_dest,
			       ETH_ALEN);
			memcpy(params->gre.local_mac, db->parms.eh.h_source,
			       ETH_ALEN);
			params->gre.remote_ip[0] = db->parms.raddr.in.s_addr;
			params->gre.local_ip[0] = db->parms.laddr.in.s_addr;
		} else if (BCM_NETDEVICE_GROUP_CAT(dev->group) ==
			  BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP6) {
			params->type = ARPHRD_ETHER;
			memcpy(params->gre.remote_mac, db->parms.eh.h_dest,
			       ETH_ALEN);
			memcpy(params->gre.local_mac, db->parms.eh.h_source,
			       ETH_ALEN);
			memcpy((char *)params->gre.remote_ip,
			       db->parms.raddr.in6.s6_addr,
			       sizeof(struct in6_addr));
			memcpy((char *)params->gre.local_ip,
			       db->parms.laddr.in6.s6_addr,
			       sizeof(struct in6_addr));
		}
	}
	if (wifi) {
		wifi->flowring = db->wifi_flowring[wifi->skb_pri];
		wifi->pri = db->wifi_pri[wifi->skb_pri];
	}
	return db->oif;
}

int flowmgr_tunnel_init(void)
{
	return flowmgr_tdb_init();
}

void flowmgr_tunnel_fini(void)
{
	flowmgr_tdb_fini();
}
