 /****************************************************************************
 *
 * Copyright (c) 2016 Broadcom Limited
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Jayesh Patel <jayesh.patel@broadcom.com>
 *****************************************************************************/
#include <linux/types.h>
#include <linux/platform_device.h>
#include <linux/of.h>
#include <linux/types.h>
#include "mdqm_dev.h"

static int mdqm_read_prop_u32_array(struct device_node *of_node,
				    const char *propname, int cols, int *rows,
				    u32 *dst)
{
	int status = 0;
	struct property *prop;
	int num_entries;
	int i, j;
	const __be32 *ptr;

	prop = of_find_property(of_node, propname, NULL);
	if (!prop) {
		pr_debug("Unable to find %s property!\n", propname);
		status = -EINVAL;
		goto done;
	}

	if ((prop->length % cols*sizeof(u32)) != 0) {
		pr_err("%s property is malformed!\n", propname);
		status = -EINVAL;
		goto done;
	}

	num_entries = prop->length / sizeof(u32) / cols;
	if (num_entries > *rows) {
		pr_err("Array provided for %s not large enough!\n", propname);
		status = -EINVAL;
		goto done;
	}
	*rows = num_entries;

	ptr = prop->value;
	for (i = 0; i < *rows; i++)
		for (j = 0; j < cols; j++)
			dst[i*cols + j] = be32_to_cpup(ptr++);

done:
	return status;
}

int mdqm_parse_dt_prop_u32_array(struct device_node *of_node,
				 const char *propname, u32 *dst,
				 int *count)
{
	int status = 0;
	int q_count = MDQM_MAX_Q;

	status = mdqm_read_prop_u32_array(of_node, propname, 1,
					  &q_count, dst);
	if (q_count > MDQM_MAX_Q) {
		pr_err("Too many Q's specified for channel %s.\n",
		       propname);
		status = -EINVAL;
	} else if (status) {
		pr_err("Missing %s property!\n", propname);
		status = -EFAULT;
	} else
		*count = q_count;
	return status;
}

int mdqm_parse_dt_prop_string(struct device_node *of_node,
				  const char *propname, char **dst)
{
	int status = 0;

	if (!of_property_read_string(of_node, propname, (const char **)dst))
		pr_debug("%s = %s\n", propname, *dst);
	else {
		pr_err("Missing %s property!\n", propname);
		status = -EFAULT;
	}

	return status;
}

int mdqm_parse_dt_node(struct platform_device *pdev)
{
	int i, status = 0;
	struct mdqm_dev *mdev = pdev->dev.platform_data;
	struct device_node *of_node = pdev->dev.of_node;
	char *str;
	u32 qs[MDQM_MAX_Q];

	if (!mdev)
		goto done;

	status = mdqm_parse_dt_prop_string(of_node, "dev-name",
					    &str);
	if (status)
		goto done;
	strncpy(mdev->name, str, sizeof(mdev->name));
	mdev->name[sizeof(mdev->name)-1] = '\0';

	status = mdqm_parse_dt_prop_string(of_node, "dqm",
					    &str);
	if (status)
		goto done;
	strncpy(mdev->dqm_dev, str, sizeof(mdev->dqm_dev));
	mdev->dqm_dev[sizeof(mdev->dqm_dev)-1] = '\0';

	status = mdqm_parse_dt_prop_string(of_node, "q-type",
					    &str);
	if (status)
		goto done;

	if (!strncmp(str, "rx", sizeof("rx")))
		mdev->type = MDEV_TYPE_RX_KTHREAD;
	else if (!strncmp(str, "rx-napi", sizeof("rx-napi")))
		mdev->type = MDEV_TYPE_RX_NAPI;
	else if (!strncmp(str, "tx", sizeof("tx")))
		mdev->type = MDEV_TYPE_TX;

	status = mdqm_parse_dt_prop_u32_array(of_node, "q-num",
					      qs, &mdev->q_count);
	if (status)
		goto done;

	for (i = 0; i < mdev->q_count; i++) {
		mdev->q[i].priority = i;
		mdev->q[i].num = qs[i];
		mdev->q[i].mdev = mdev;
	}

done:
	return status;
}
