/****************************************************************************
 *
 * Broadcom Proprietary and Confidential.
 * (c) 2024 Broadcom. All rights reserved.
 * The term "Broadcom" refers to Broadcom Limited and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 *****************************************************************************/


#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/err.h>
#include <linux/of.h>
#include <linux/clk-provider.h>
#include <linux/clkdev.h>
#include <linux/bcm_media_gw/itc_rpc/itc_rpc.h>

#define MODULE_NAME "brcm-clk-rpc"

#ifdef pr_fmt
#undef pr_fmt
#endif
#define pr_fmt(fmt) MODULE_NAME": " fmt

#define RPC_TIMEOUT     1 /* sec */
#define CLK_DOMAINS_MAX 255
enum clk_rpc_domain_func {
	CLK_RPC_DOMAIN_ID,
	CLK_RPC_DOMAIN_NAME,
	CLK_RPC_GET_DOMAIN_STATE,
	CLK_RPC_SET_DOMAIN_STATE
};

struct clk_rpc {
	struct clk_hw hw;
	char *name;
	int id;
};
struct clk_rpc clk_rpcs[CLK_DOMAINS_MAX];
#define to_clk_rpc(p) container_of(p, struct clk_rpc, hw)

int tunnel;

static inline u8 clk_rpc_msg_rc(rpc_msg *msg)
{
	return (msg->data[0] >> 24);
}
static inline void clk_rpc_msg_clear_rc(rpc_msg *msg)
{
	msg->data[0] &= 0x00ffffff;
}
static inline u8 clk_rpc_msg_get_id(rpc_msg *msg)
{
	return (msg->data[0] & 0xff);
}
static inline void clk_rpc_msg_set_id(rpc_msg *msg, u8 id)
{
	msg->data[0] &= ~0xff;
	msg->data[0] |= id & 0xff;
}
static inline bool clk_rpc_msg_get_enable(rpc_msg *msg)
{
	return (msg->data[0] & 0x100);
}
static inline void clk_rpc_msg_set_enable(rpc_msg *msg, bool enable)
{
	if (enable)
		msg->data[0] |= 0x100;
	else
		msg->data[0] &= ~0x100;
}
static inline bool clk_rpc_msg_get_ssc(rpc_msg *msg)
{
	return (msg->data[0] & 0x200);
}
static inline void clk_rpc_msg_set_ssc(rpc_msg *msg, bool enable)
{
	if (enable)
		msg->data[0] |= 0x200;
	else
		msg->data[0] &= ~0x200;
}
static inline bool clk_rpc_msg_get_rate(rpc_msg *msg)
{
	return msg->data[2];
}
static inline void clk_rpc_msg_set_rate(rpc_msg *msg, u32 rate)
{
	msg->data[2] = rate;
}

static int clk_rpc_get_id(char *name)
{
	rpc_msg msg;
	int rc;

	rpc_msg_init(&msg, RPC_SERVICE_CLK, CLK_RPC_DOMAIN_ID, 0, 0, 0, 0);
	strncpy((char *)&msg.data[1], name, 8);
	rc = rpc_send_request_timeout(tunnel, &msg, RPC_TIMEOUT);
	if (rc || clk_rpc_msg_rc(&msg)) {
		pr_err("failed getting clock domain ID\n");
		return rc ? rc : -EIO;
	}
	return clk_rpc_msg_get_id(&msg);
}

static int clk_rpc_get_state(struct clk_hw *hw, bool *enabled, bool *ssc, u32 *rate)
{
	struct clk_rpc *clk = to_clk_rpc(hw);
	rpc_msg msg;
	int rc;

	rpc_msg_init(&msg, RPC_SERVICE_CLK, CLK_RPC_GET_DOMAIN_STATE, 0, 0, 0, 0);
	clk_rpc_msg_set_id(&msg, clk->id);
	rc = rpc_send_request_timeout(tunnel, &msg, RPC_TIMEOUT);
	if (rc || clk_rpc_msg_rc(&msg)) {
		pr_err("failed getting clock domain state\n");
		return rc ? rc : -EIO;
	}
	*enabled = clk_rpc_msg_get_enable(&msg);
	*ssc = clk_rpc_msg_get_ssc(&msg);
	*rate = clk_rpc_msg_get_rate(&msg);

	pr_debug("clock domain %s currently: %sabled, %sSSC, %u kHz\n",
		 *enabled ? "en" : "dis", clk->name, *ssc ? "" : "!", *rate);
	return 0;
}

static int clk_rpc_set_state(struct clk_hw *hw, bool enable, bool ssc, u32 rate)
{
	struct clk_rpc *clk = to_clk_rpc(hw);
	rpc_msg msg;
	int rc;

	rpc_msg_init(&msg, RPC_SERVICE_CLK, CLK_RPC_SET_DOMAIN_STATE, 0, 0, 0, 0);
	clk_rpc_msg_set_id(&msg, clk->id);
	clk_rpc_msg_set_enable(&msg, enable);
	clk_rpc_msg_set_ssc(&msg, ssc);
	clk_rpc_msg_set_rate(&msg, rate);
	rc = rpc_send_request_timeout(tunnel, &msg, RPC_TIMEOUT);
	if (rc || clk_rpc_msg_rc(&msg)) {
		pr_err("failed setting clock domain state\n");
		return rc ? rc : -EIO;
	}

	pr_debug("set clock domain %s to: %sabled, %sSSC, %u kHz\n",
		 enable ? "en" : "dis", clk->name, ssc ? "" : "!", rate);
	return 0;
}

static int clk_rpc_enable_disable(struct clk_hw *hw, bool enable)
{
	bool ssc, tmp;
	u32 rate;
	int rc;

	rc = clk_rpc_get_state(hw, &tmp, &ssc, &rate);
	if (rc)
		return rc;
	return clk_rpc_set_state(hw, enable, ssc, rate);
}

static int clk_rpc_prepare(struct clk_hw *hw)
{
	return clk_rpc_enable_disable(hw, true);
}

static void clk_rpc_unprepare(struct clk_hw *hw)
{
	clk_rpc_enable_disable(hw, false);
}

static int clk_rpc_is_prepared(struct clk_hw *hw)
{
	bool enabled, ssc;
	u32 rate;
	int rc;

	rc = clk_rpc_get_state(hw, &enabled, &ssc, &rate);
	if (rc)
		return rc;
	return enabled;
}

static const struct clk_ops clk_rpc_ops = {
	/*
	 * Need to use (un)prepare instead of (en/dis)able because we need
	 * to use synchronous RPC so we are sure clock is configured prior to
	 * returning and synchronous RPC allows sleeping.
	 */
	.prepare = clk_rpc_prepare,
	.unprepare = clk_rpc_unprepare,
	.is_prepared = clk_rpc_is_prepared,
};

static int clk_rpc_probe(struct platform_device *pdev)
{
	struct device_node *node = pdev->dev.of_node;
	char *name;
	int rc;
	int id;
	struct clk_rpc *clk;
	struct clk_init_data init;
	bool enabled, ssc;
	u32 rate;

	rc = of_property_read_string(node, "clk-domain-name", (const char **)&name);
	if (rc) {
		pr_err("failed to get clock domain name\n");
		goto done;
	}

	tunnel = rpc_get_fifo_tunnel_id("rg-smc");
	if (tunnel < 0) {
		pr_err("failed getting RPC tunnel ID\n");
		goto done;
	}

	id = rc = clk_rpc_get_id(name);
	if (rc < 0)
		goto done;

	clk = &clk_rpcs[id];
	clk->id = id;
	clk->name = name;
	platform_set_drvdata(pdev, (void *)clk);

	init.name = name;
	init.ops = &clk_rpc_ops;
	init.parent_names = NULL;
	init.num_parents = 0;
	init.flags = 0;

	clk->hw.init = &init;

	rc = of_clk_hw_register(node, &clk->hw);
	if (rc) {
		pr_err("failed to register hw clk for clock domain %s (%d)\n",
		       name, rc);
		goto done;
	}

	rc = of_clk_add_hw_provider(node, of_clk_hw_simple_get, clk);
	if (rc) {
		pr_err("failed to add clk provider for clock domain %s (%d)\n",
		       name, rc);
		goto err_unreg_clk_hw;
	}

	rc = clk_rpc_get_state(&clk->hw, &enabled, &ssc, &rate);
	if (rc)
		goto err_del_prov;
	ssc = of_property_read_bool(node, "brcm,enable-ssc");
	clk_rpc_set_state(&clk->hw, enabled, ssc, rate);

	pr_info("created clock device for clock domain %s: %sabled, %sSSC, %u kHz\n", name,
		enabled ? "en" : "dis", ssc ? "" : "!", rate);
	goto done;

err_del_prov:
	of_clk_del_provider(node);
err_unreg_clk_hw:
	clk_hw_unregister(&clk->hw);
done:
	return rc;
}

static int clk_rpc_remove(struct platform_device *pdev)
{
	struct device_node *node = pdev->dev.of_node;
	struct clk_rpc *clk = platform_get_drvdata(pdev);

	of_clk_del_provider(node);
	clk_hw_unregister(&clk->hw);
	return 0;
}

static const struct of_device_id clk_rpc_of_match[] = {
	{.compatible = "brcm,brcmcm-clk-rpc"},
	{}
};

MODULE_DEVICE_TABLE(of, clk_rpc_of_match);

static struct platform_driver clk_rpc_driver = {
	.probe  = clk_rpc_probe,
	.remove = clk_rpc_remove,
	.driver = {
		.name = MODULE_NAME,
		.owner = THIS_MODULE,
		.of_match_table = clk_rpc_of_match
	}
};

int clk_rpc_init(void)
{
	return platform_driver_register(&clk_rpc_driver);
}
EXPORT_SYMBOL(clk_rpc_init);

void clk_rpc_cleanup(void)
{
	platform_driver_unregister(&clk_rpc_driver);
}
EXPORT_SYMBOL(clk_rpc_cleanup);

subsys_initcall(clk_rpc_init);
module_exit(clk_rpc_cleanup);

MODULE_LICENSE("GPL");
