 /****************************************************************************
 *
 * Copyright (c) 2019 Broadcom. All rights reserved
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Jayesh Patel <jayeshp@broadcom.com>
 ****************************************************************************/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/if_arp.h>
#include <net/netfilter/nf_conntrack_core.h>
#include <net/netfilter/nf_conntrack_offload.h>
#include "proc_cmd.h"
#include "flowmgr.h"

#define FLOWMGR_TRACE_ENTRIES 128
#define FLOWMGR_TRACE_MAX_TYPE 4
char *trace_type[FLOWMGR_TRACE_MAX_TYPE] = {
	"    add",
	"    del",
	"exp-add",
	"exp-del"
};

struct flowmgr_trace {
	int idx;
	int type;
	struct nf_conntrack_tuple tuple;
	struct offload_info info;
};

static char type_mask;
static u32 buf_idx;
static spinlock_t lock;

static struct flowmgr_trace tracebuf[FLOWMGR_TRACE_ENTRIES];

int flowmgr_trace_add_entry(char type, void *tuple, void *info)
{
	if (!(type_mask & (1<<type)))
		return 0;
	if (!flowmgr_cttrace_check(tuple, info))
		return 0;
	spin_lock_bh(&lock);
	memcpy(&tracebuf[buf_idx].tuple, tuple, sizeof(struct nf_conntrack_tuple));
	memcpy(&tracebuf[buf_idx].info, info, sizeof(struct offload_info));
	tracebuf[buf_idx].type = type;
	tracebuf[buf_idx].idx = buf_idx+1;
	tracebuf[buf_idx].info.tstamp = jiffies;
	buf_idx++;
	if (buf_idx >= FLOWMGR_TRACE_ENTRIES)
		buf_idx = 0;
	spin_unlock_bh(&lock);
	return 0;
}
static const char* l3proto_name(u16 proto)
{
	switch (proto) {
	case AF_INET: return "ipv4";
	case AF_INET6: return "ipv6";
	}

	return "unknown";
}

static const char* l4proto_name(u16 proto)
{
	switch (proto) {
	case IPPROTO_ICMP: return "icmp";
	case IPPROTO_TCP: return "tcp";
	case IPPROTO_UDP: return "udp";
	case IPPROTO_DCCP: return "dccp";
	case IPPROTO_GRE: return "gre";
	case IPPROTO_SCTP: return "sctp";
	case IPPROTO_UDPLITE: return "udplite";
	}

	return "unknown";
}

int flowmgr_trace_show_entry(struct seq_file *s, struct flowmgr_trace *tracebuf)
{
	const struct nf_conntrack_l4proto *l4proto;
	u_int16_t l3num;
	u_int8_t protonum;
	struct net_device *in = NULL;
	struct net_device *out = NULL;
	struct timeval tv;
	l3num = tracebuf->tuple.src.l3num;
	protonum = tracebuf->tuple.dst.protonum;
	l4proto = nf_ct_l4proto_find(protonum);

	pr_seq(s, "%-3d: %s %-4s %u %-3s %2u ",
	       tracebuf->idx,
	       tracebuf->type < FLOWMGR_TRACE_MAX_TYPE ? trace_type[tracebuf->type] : "   none",
	       l3proto_name(l3num), l3num,
	       l4proto_name(l4proto->l4proto), protonum);
	in = __dev_get_by_index(&init_net, tracebuf->info.iif);
	out = __dev_get_by_index(&init_net, tracebuf->info.oif);
	if (tracebuf->type < 2) {
		pr_seq_cont(s,
		       "flow=%-5u type=%-2u|%-2u exp=%d lag=%u src=%pM dst=%pM in=%-6s out=%-6s ",
		       tracebuf->info.flow_id,
		       tracebuf->info.flow_type>>16,
		       tracebuf->info.flow_type&0xFFFF,
		       tracebuf->info.expected,
		       tracebuf->info.lag,
		       tracebuf->info.eh.h_source,
		       tracebuf->info.eh.h_dest,
		       in ? in->name:"null",
		       out ? out->name:"null");
		pr_seq_cont(s, "dscp=%-2d->%-2d vlan_untag=%-2d vlan_id=%-4x ",
		       tracebuf->info.dscp_old,
		       tracebuf->info.dscp_new,
		       tracebuf->info.vlan_untag,
		       tracebuf->info.vlan_id);
	} else {
		pr_seq_cont(s,
		       "flow=%-5u type=%-2u|%-2u exp=%d in=%-6s out=%-6s ",
		       tracebuf->info.flow_id,
		       tracebuf->info.flow_type>>16,
		       tracebuf->info.flow_type&0xFFFF,
		       tracebuf->info.expected,
		       in ? in->name:"null",
		       out ? out->name:"null");
	}
	pr_seq_cont(s, "src=%-15pI4 dst=%-15pI4 ",
	       &tracebuf->tuple.src.u3.ip,
	       &tracebuf->tuple.dst.u3.ip);
	pr_seq_cont(s, "sport=%-5hu dport=%-5hu ",
	       ntohs(tracebuf->tuple.src.u.tcp.port),
	       ntohs(tracebuf->tuple.dst.u.tcp.port));
	jiffies_to_timeval(jiffies - tracebuf->info.tstamp, &tv);
	pr_seq_cont(s, "dur=%ld ", tv.tv_sec);
	pr_seq_cont(s, "\n");
	return 0;
}

int flowmgr_trace_show(struct seq_file *s)
{
	int i, idx;
	for (i = 0; i < FLOWMGR_TRACE_ENTRIES; i++) {
		idx = (i + buf_idx) % FLOWMGR_TRACE_ENTRIES;
		if (tracebuf[idx].idx)
			flowmgr_trace_show_entry(s, &tracebuf[idx]);
	}
	return 0;
}

static void *flowmgr_trace_seq_start(struct seq_file *seq, loff_t *pos)
{
	int idx;
	if (*pos == 0)
		pr_seq(seq, "FLOWMGR: Dumping Trace buffer\n");
	if (*pos >= FLOWMGR_TRACE_ENTRIES)
		return 0;
#if defined (CONFIG_ARM64)
	seq->private = (void *) (u64) buf_idx;
#else
	seq->private = (void *) buf_idx;
#endif
	idx = *pos + buf_idx;
	idx = idx % FLOWMGR_TRACE_ENTRIES;
	return (void *)&tracebuf[idx];
}

static void *flowmgr_trace_seq_next(struct seq_file *seq, void *v,
				    loff_t *pos)
{
	int idx;
	(*pos)++;
	if (*pos >= FLOWMGR_TRACE_ENTRIES)
		return 0;
#if defined (CONFIG_ARM64)
	idx = *pos + (int)(u64)seq->private;
#else
	idx = *pos + (int)seq->private;
#endif
	idx = idx % FLOWMGR_TRACE_ENTRIES;
	return (void *) &tracebuf[idx];
}

static void flowmgr_trace_seq_stop(struct seq_file *seq, void *v)
{
}

static int flowmgr_trace_seq_show(struct seq_file *seq, void *v)
{
	struct flowmgr_trace *tracebuf;
	if (!v)
		return -1;
	tracebuf = (struct flowmgr_trace *) v;

	if (!tracebuf->idx)
		return 0;

	flowmgr_trace_show_entry(seq, tracebuf);

	return 0;
}

static const struct seq_operations flowmgr_trace_seq_ops = {
	.start	= flowmgr_trace_seq_start,
	.next	= flowmgr_trace_seq_next,
	.stop	= flowmgr_trace_seq_stop,
	.show	= flowmgr_trace_seq_show,
};

static int flowmgr_trace_seq_open(struct inode *inode, struct file *file)
{
	int ret = seq_open(file, &flowmgr_trace_seq_ops);
	return ret;
};

static const struct file_operations flowmgr_proc_trace_fops = {
	.owner    = THIS_MODULE,
	.open     = flowmgr_trace_seq_open,
	.read     = seq_read,
	.llseek   = seq_lseek,
	.release  = seq_release,
};

static int cmd_clear(int argc, char *argv[])
{
	int i;
	if (argc == 2) {
		spin_lock_bh(&lock);
		for (i = 0; i < FLOWMGR_TRACE_ENTRIES; i++) {
			memset(&tracebuf[i], 0,
			       sizeof(struct flowmgr_trace));
		}
		spin_unlock_bh(&lock);
		goto done;
	}
/* help */
	pr_info("%s <1>\n", argv[0]);
done:
	return 0;
}

static int cmd_enable(int argc, char *argv[])
{
	u32 type = 0;

	if (argc == 2) {
		if (kstrtou32(argv[1], 0, &type) == 0) {
			type_mask = type;
			if (type == 0) {
				pr_info("FLOWMGR Trace disabled\n");
				goto done;
			} else {
				pr_info("FLOWMGR Trace mask: 0x%x\n", type_mask);
				goto done;
			}
		}
	}
/* help */
	pr_info("%s <mask>\n", argv[0]);
	pr_info("   bit 0: Flow Add, bit 1: Flow Del\n");
	pr_info("   bit 2: Expected Flow Add, bit 3: Expeced Flow Del\n");
done:
	return 0;
}

static struct proc_cmd_ops trace_command_entries[] = {
	{ .name = "clear",   .do_command = cmd_clear},
	{ .name = "enable",  .do_command = cmd_enable},
};

struct proc_cmd_table trace_command_table = {
	.module_name = "FLOWMGR",
	.size = ARRAY_SIZE(trace_command_entries),
	.data_seq_read = (void *) &flowmgr_trace_seq_ops,
	.ops = trace_command_entries
};


int flowmgr_trace_init(void)
{
	proc_create_cmd("trace", flowmgr.proc_dir, &trace_command_table);
	return 0;
}

void flowmgr_trace_exit(void)
{
	if (flowmgr.proc_dir) {
		remove_proc_entry("trace", flowmgr.proc_dir);
	}
}
