 /****************************************************************************
 *
 * Copyright (c) 2015 Broadcom Corporation
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************/
/*
 ***************************************************************************
 * File Name  : vflash.c
 *
 * Description: The virtual flash driver which does flash io via
 * IPC calls to eCos.
 *
 * Created on :  4/4/2012  Peter Sulc
 *
 * rewritten 10/23/12 to use itc-rpc services and support nand and nor
 *
 ***************************************************************************/
#include <linux/kernel.h>
#include <linux/log2.h>
#include <linux/param.h>
#include <linux/sched.h>
#include <linux/timer.h>
#include <linux/platform_device.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/partitions.h>
#include <linux/spinlock.h>
#include <linux/types.h>
#include <linux/of.h>
#include <asm/io.h>
#include <linux/version.h>
#include <linux/vmalloc.h>
#if LINUX_VERSION_CODE > KERNEL_VERSION(5, 9, 1)
#include <linux/dma-map-ops.h>
#else
#include <linux/dma-mapping.h>
#include <linux/dma-contiguous.h>
#endif
#if LINUX_VERSION_CODE > KERNEL_VERSION(2, 6, 30)
#include <linux/semaphore.h>
#include <linux/skbuff.h>
#include <linux/module.h>
#endif
#include <linux/proc_fs.h>

#include <linux/bcm_media_gw/itc_rpc/itc_rpc.h>
#include "vfmtd.h"
#include <asm/cacheflush.h>
#include <bcmcache.h>

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("Virtual Flash Client");
MODULE_AUTHOR("Peter Sulc");

#define PROCFS_NAME "vfc"

void dump_buffer(u8 *buf, int len);

/***************************************************************
 * block cache
 **************************************************************/
typedef struct {
	loff_t		start_addr;
	loff_t		end_addr;
	u_char		*buf;
	struct semaphore	lock;
	struct mtd_info		*mtd;
} BlockCache;

static void init_block_cache(u32 cachesize);
static BlockCache *find_block_cache(struct mtd_info *mtd, loff_t addr);
static BlockCache *new_block_cache(struct mtd_info  *mtd, loff_t addr, u32 size, bool nand);
static void release_block_cache(BlockCache *bc);
static void invalidate_block_cache(BlockCache *bc);

/***************************************************************
 * other stuff
 **************************************************************/
static char bcmmtd_name[] = "bcmmtd_vflash_device";
static u8 *gp_bcache;
static u8 *gp_bcache_phys;
static u16 largest_page_size;

static const struct of_device_id of_platform_brcmvflash_table[] = {
	{ .compatible = "brcm,brcm-vflashclient", },
	{},
};

/***************************************************************
 * mtd partition stuff
 **************************************************************/

typedef struct {
	loff_t	offset;
	loff_t	end;
	int	region;  /*region mapping */
} flash_partition;

typedef struct {
	struct mtd_info		mtd;
	struct mtd_partition	*mtd_parts;
	flash_partition		*partitions;
	int			part_cnt;
	int			mru; /* most recently used */
} mtd_device;

static int get_flash_region(mtd_device *mtd_dev, loff_t address);

static mtd_device	*mtd_device_ptr;
static int		mtd_device_cnt;
static int		g_rpc_tunnel;

#define DRIVER_NAME	"brcm-vflashclient"

#define VFLASH_RPC_TIMEOUT	5

void vflash_flush_buffer(void *va, u32 size)
{
	FUNC_TRACE(1);
	cache_flush_invalidate_buffer((u8 *)va, size);
	FUNC_TRACE(0);
}
EXPORT_SYMBOL(vflash_flush_buffer);

void vflash_invalidate_buffer(void *va, u32 size)
{
	FUNC_TRACE(1);
	cache_invalidate_buffer((u8 *)va, size);
	FUNC_TRACE(0);
}
EXPORT_SYMBOL(vflash_invalidate_buffer);

#ifdef VFLASH_PROC_DEBUG
static dma_addr_t debug_dma_addr;
#define PATTERN1 0xdeadbeef
#define PATTERN2 0xcafebabe

#ifdef USE_SDMA_MAP
int
vfc_proc_read(char *buffer,
		char **buffer_location,
		off_t offset, int buffer_length, int *eof, void *data)
{
	int i;
	static int trial;
	if (offset)
		return 0;

	MSG_TRACEI("vfs_proc_read (%s) called\n", PROCFS_NAME);
	MSG_TRACEI("Unmap: 0x%lx\n", debug_dma_addr);
	dma_unmap_single(NULL, debug_dma_addr, (1024*1024), DMA_FROM_DEVICE);

	MSG_TRACEN("First value: 0x%x\n", ((u32 *)gp_bcache)[0]);
	for (i = 0; i < (1024*1024)/4; i++) {
		if (trial == 0) {
			if (((u32 *)gp_bcache)[i] != PATTERN1) {
				MSG_TRACEI("FAILS at [%d]: x%x\n", i, ((u32 *)gp_bcache)[i]);
				/*break;*/
			}
		} else {
			if (((u32 *)gp_bcache)[i] != PATTERN2) {
				MSG_TRACEI("FAILS at [%d]: x%x\n", i, ((u32 *)gp_bcache)[i]);
				/*break;*/
			}
		}

		if (i % (64*1024) == 0)
			MSG_TRACEN("[%d]: x%x\n", i, ((u32 *)gp_bcache)[i]);
	}
	if (trial == 0)
		trial = 1;
	else
		trial = 0;

	*eof = 1;
	return 1;
}

/**
 * This function is called with the /proc file is written
 *
 */
int vfc_proc_write(struct file *file, const char *buffer, unsigned long count,
			void *data)
{

	MSG_TRACEI("vfs_proc_write (%s) called\n", PROCFS_NAME);

	debug_dma_addr = dma_map_single(NULL, (void *)gp_bcache, (1024*1024),
						DMA_FROM_DEVICE);

	if (dma_mapping_error(NULL, debug_dma_addr))
		MSG_TRACEE("\n\n\ndma mapping fails\n\n\n");
	else
		MSG_TRACEI("mapped to 0x%lx\n", debug_dma_addr);

	return count;
}
#else
int
vfc_proc_read(char *buffer,
		char **buffer_location,
		off_t offset, int buffer_length, int *eof, void *data)
{
	int i;
	static int trial;
	if (offset)
		return 0;

	MSG_TRACEI("vfs_proc_read (%s) called\n", PROCFS_NAME);
	MSG_TRACEI("Unmap: 0x%lx\n", debug_dma_addr);

	vflash_invalidate_buffer(gp_bcache, (1024*1024));

	MSG_TRACEN("First value: 0x%x\n", ((u32 *)gp_bcache)[0]);
	for (i = 0; i < (1024*1024)/4; i++) {
		if (trial == 0) {
			if (((u32 *)gp_bcache)[i] != PATTERN1) {
				MSG_TRACEI("FAILS at [%d]: x%x\n", i, ((u32 *)gp_bcache)[i]);
				/*break;*/
			}
		} else {
			if (((u32 *)gp_bcache)[i] != PATTERN2) {
				MSG_TRACEI("FAILS at [%d]: x%x\n", i, ((u32 *)gp_bcache)[i]);
				/*break;*/
			}
		}

		if (i % (64*1024) == 0)
			MSG_TRACEN("[%d]: x%x\n", i, ((u32 *)gp_bcache)[i]);
	}
	if (trial == 0)
		trial = 1;
	else
		trial = 0;

	*eof = 1;
	return 1;
}

/**
 * This function is called with the /proc file is written
 *
 */
int vfc_proc_write(struct file *file, const char *buffer, unsigned long count,
			void *data)
{

	MSG_TRACEI("vfs_proc_write (%s) called\n", PROCFS_NAME);

	return count;
}

#endif
#endif

/*******************************************************************
 * vflash_read_nor_buf
 ******************************************************************/
static int vflash_read_nor_buf(int region,
				int offsetxpages,
				u_char *buffer,
				int log2pages,
				int log2pagesize)
{
	struct vflash_msg   msg;
	int bytes = (1 << log2pages) * (1 << log2pagesize);
	int ret;

	FUNC_TRACE(1);

#ifdef USE_SDMA_MAP
	dma_addr_t dma_addr = dma_map_single(NULL, (void *)buffer, bytes,
					DMA_FROM_DEVICE);
#endif

	rpc_msg_init_header((rpc_msg *)&msg, RPC_SERVICE_FLASH, VF_NOR_READ, 0);

	vflash_msg_set_pagesize(&msg, log2pagesize);
	msg.region = region;
	msg.offsetxpages = offsetxpages;
	msg.pages = log2pages;
	MSG_TRACEI("log2pages 0x%x log2pagesize: %d\n", log2pages, log2pagesize);
	MSG_TRACEI("Sending offset 0x%x\n", (u8 *)buffer - (u8 *)gp_bcache);
	msg.buffer = (u8 *)((u8 *) buffer - (u8 *)gp_bcache);/*offset into memory region*/

	while ((ret = rpc_send_request_timeout(g_rpc_tunnel, (rpc_msg *)&msg, VFLASH_RPC_TIMEOUT))) {
		pr_err("%s: rpc_send_request_timeout %d\n", __func__, ret);
		if (ret != -EAGAIN)
			break;
	}
#ifdef USE_SDMA_MAP
	dma_unmap_single(NULL, dma_addr, bytes,
				DMA_FROM_DEVICE);
#else
	vflash_invalidate_buffer((void *)buffer, bytes);
#endif
	FUNC_TRACE(0);

	return vflash_msg_error(&msg) ? -1 : 0;
}

/*******************************************************************
 * vflash_read_nand_buf
 ******************************************************************/
static int vflash_read_nand_buf(int region,
				int offsetxpages,
				u_char *buffer,
				int pages,
				int log2pagesize)
{
	struct vflash_msg   msg;
	int bytes = pages * (1 << log2pagesize);
	int ret;

	FUNC_TRACE(1);

#ifdef USE_SDMA_MAP
	dma_addr_t dma_addr = dma_map_single(NULL, (void *)buffer, bytes,
				DMA_FROM_DEVICE);
	if (dma_mapping_error(NULL, dma_addr))
		MSG_TRACEE("\n\n\ndma mapping fails\n\n\n");
#endif
	rpc_msg_init_header((rpc_msg *)&msg, RPC_SERVICE_FLASH, VF_NAND_READ, 0);

	vflash_msg_set_pagesize(&msg, log2pagesize);
	msg.region = region;
	msg.offsetxpages = offsetxpages;
	msg.pages = pages;
	MSG_TRACEI("Sending offset 0x%x for va %px\n", (u8 *)buffer - (u8 *)gp_bcache, (u8 *)buffer);
	msg.buffer = (u8 *)((u8 *) buffer - (u8 *)gp_bcache); /*offset into memory region*/

	while ((ret = rpc_send_request_timeout(g_rpc_tunnel, (rpc_msg *)&msg, VFLASH_RPC_TIMEOUT))) {
		pr_err("%s: rpc_send_request_timeout %d\n", __func__, ret);
		if (ret != -EAGAIN)
			break;
	}
#ifdef USE_SDMA_MAP
	dma_unmap_single(NULL, dma_addr, bytes,
				DMA_FROM_DEVICE);
#else
	vflash_invalidate_buffer((void *)buffer, bytes);
#endif
	FUNC_TRACE(0);

	return vflash_msg_error(&msg) ? -1 : 0;
}

/*******************************************************************
 * vflash_write_buf
 ******************************************************************/
static int vflash_write_buf(int region,
			int offsetxpages,
			const u_char *buffer,
			u32 pages,
			int log2pagesize)
{
	struct vflash_msg   msg;
	int i, ret;
	int bytes = pages * (1 << log2pagesize);

	FUNC_TRACE(1);

	for (i = 0; i < 64; i += 4)
		MSG_TRACEN("%d: 0x%x 0x%x 0x%x 0x%x\n", i, buffer[i], buffer[i+1], buffer[i+2], buffer[i+3]);

#ifdef USE_SDMA_MAP
	dma_addr_t dma_addr = dma_map_single(NULL, (void *)buffer, bytes,
					DMA_TO_DEVICE);
#else
	vflash_flush_buffer((void *)buffer, bytes);
#endif

	rpc_msg_init_header((rpc_msg *)&msg, RPC_SERVICE_FLASH, VF_WRITE, 0);

	msg.region = region;
	msg.offsetxpages = offsetxpages;
	vflash_msg_set_pagesize(&msg, log2pagesize);
	if ((pages > (0xffff)) && (pages <= (1<<16))) {
		MSG_TRACEI("Setting xtrabit\n");
		vflash_msg_set_xtrabit(&msg);
	}
	msg.pages = (u16)pages;
	MSG_TRACEI("Sending offset 0x%x for va %px\n", (u8 *)buffer - (u8 *)gp_bcache, (u8 *)buffer);
	MSG_TRACEI("gp_bcache: %px\n", gp_bcache);
	msg.buffer = (u8 *)((u8 *) buffer - (u8 *)gp_bcache); /*offset into memory region*/

	while ((ret = rpc_send_request_timeout(g_rpc_tunnel, (rpc_msg *)&msg, VFLASH_RPC_TIMEOUT))) {
		pr_err("%s: rpc_send_request_timeout %d\n", __func__, ret);
		if (ret != -EAGAIN)
			break;
	}
#ifdef USE_SDMA_MAP
	dma_unmap_single(NULL, dma_addr, bytes,
			DMA_TO_DEVICE);
#endif
	FUNC_TRACE(0);

	return vflash_msg_error(&msg) ? -1 : 0;
}

/*******************************************************************
 * vflash_erase_block
 ******************************************************************/
static int vflash_erase_block(int region, u32 block)
{
	struct vflash_msg   msg;
	int ret;

	FUNC_TRACE(1);

	rpc_msg_init_header((rpc_msg *)&msg, RPC_SERVICE_FLASH, VF_ERASE, 0);
	msg.region = region;
	vflash_msg_set_block(&msg, block);

	while ((ret = rpc_send_request_timeout(g_rpc_tunnel, (rpc_msg *)&msg, VFLASH_RPC_TIMEOUT))) {
		pr_err("%s: rpc_send_request_timeout %d\n", __func__, ret);
		if (ret != -EAGAIN)
			break;
	}
	FUNC_TRACE(0);

	return vflash_msg_error(&msg) ? -1 : 0;
}

static int vflash_block_isbad(int region, int offsetxpages, int log2pagesize)
{
	struct vflash_msg   msg;
	int ret;

	FUNC_TRACE(1);

	rpc_msg_init_header((rpc_msg *)&msg, RPC_SERVICE_FLASH, VF_NAND_BLOCK_ISBAD, 0);
	msg.region = region;
	msg.offsetxpages = offsetxpages;
	vflash_msg_set_pagesize(&msg, log2pagesize);

	while ((ret = rpc_send_request_timeout(g_rpc_tunnel, (rpc_msg *)&msg, VFLASH_RPC_TIMEOUT))) {
		pr_err("%s: rpc_send_request_timeout %d\n", __func__, ret);
		if (ret != -EAGAIN)
			break;
	}
	FUNC_TRACE(0);

	return vflash_msg_true(&msg);
}

static int vflash_block_markbad(int region, int offsetxpages, int log2pagesize)
{
	struct vflash_msg   msg;
	int ret;

	FUNC_TRACE(1);

	rpc_msg_init_header((rpc_msg *)&msg, RPC_SERVICE_FLASH, VF_NAND_BLOCK_MARKBAD, 0);
	msg.region = region;
	msg.offsetxpages = offsetxpages;
	vflash_msg_set_pagesize(&msg, log2pagesize);
	while ((ret = rpc_send_request_timeout(g_rpc_tunnel, (rpc_msg *)&msg, VFLASH_RPC_TIMEOUT))) {
		pr_err("%s: rpc_send_request_timeout %d\n", __func__, ret);
		if (ret != -EAGAIN)
			break;
	}
	FUNC_TRACE(0);

	return vflash_msg_error(&msg) ? -1 : 0;
}

/*******************************************************************
 * mtd stuff
 ******************************************************************/
static int bcmmtd_erase(struct mtd_info *mtd, struct erase_info *instr)
{
	u32 	len;
	int 	region;
	loff_t 	addr;
	int 	ret;
	BlockCache *bc;
	int i;
	u32 bcachesize = (mtd->erasesize < largest_page_size) ? mtd->erasesize : largest_page_size;

	FUNC_TRACE(1);

	MSG_TRACEI("%s: offset=%0llx, len=%0llx bcachesize: %d\n", __func__, instr->addr, instr->len, bcachesize);

	/* sanity czech */
	if (instr->addr + instr->len > mtd->size)
		return -EINVAL;

	addr = instr->addr;
	len = instr->len;

	/* whole-chip erase is not supported */
	if (len == mtd->size) {
		FUNC_TRACE(0);
		return -EIO;
	} else {
		while (len) {
			i = 0;
			/*potentially need to invalidate/release several entries*/
			while (i*bcachesize < mtd->erasesize) {
				bc = find_block_cache(mtd, addr + i*bcachesize);
				if (bc) {
					invalidate_block_cache(bc);
					release_block_cache(bc);
				}
				i++;
			}
			region = get_flash_region((mtd_device *)mtd->priv, addr);
			ret = vflash_erase_block(region, (u32)addr / mtd->erasesize);
			if (ret < 0) {
				FUNC_TRACE(0);
				return ret;
			}
			addr += mtd->erasesize;
			len -= mtd->erasesize;
		}
	}

	FUNC_TRACE(0);

	return 0;
}

static int bcmmtd_read(struct mtd_info *mtd,
				loff_t addr,
				size_t len,
				size_t *retlen,
				u_char *buf)
{
	BlockCache *bc;
	int 	region, ret, i;
	u32 	pagesize, log2pagesize, left, bytes;
	u_char 	*target = buf;
	u32 bcachesize = (mtd->erasesize < largest_page_size) ? mtd->erasesize : largest_page_size;

	FUNC_TRACE(1);

	pagesize = mtd->writesize;
	log2pagesize = ilog2(pagesize);

	if (retlen)
		*retlen = 0;

	if (addr + len > mtd->size) {
		MSG_TRACEE("%s ERROR: trying to read beyond flash size %u\n", __func__, (u32)(addr+len));
		FUNC_TRACE(0);
		return -EINVAL;
	}
	region = get_flash_region((mtd_device *)mtd->priv, addr);
	if (region < 0) {
		MSG_TRACEE("ERROR: %s invalid address %08x\n", __func__, (u32)addr);
		FUNC_TRACE(0);
		return -ENODEV;
	}

	MSG_TRACEI("%s: offset=%0llx, len=%08x, region %d bcachesize: %d\n", __func__, addr, len, region, bcachesize);

	ret = 0;
	left = len;
	while (left) {
		bc = find_block_cache(mtd, addr);
		if (!bc) {
			u32 blockaddr = addr & ~(bcachesize - 1);
			bc = new_block_cache(mtd, blockaddr, bcachesize , mtd->type == MTD_NANDFLASH);
			if (mtd->type == MTD_NANDFLASH) {
				ret = vflash_read_nand_buf(region,
							blockaddr/pagesize,
							bc->buf,
							bcachesize/pagesize,
							log2pagesize);
			} else {
				MSG_TRACEI("New transaction: blockaddr/pagesize: %d ilog2(mtd->erasesize/pagesize) %d\n", blockaddr/pagesize, (int)ilog2(mtd->erasesize/pagesize));
				ret = vflash_read_nor_buf(region,
							blockaddr/pagesize,
							bc->buf,
							(int)ilog2(bcachesize/pagesize),
							log2pagesize);
			}
			if (ret < 0) {
				release_block_cache(bc);
				FUNC_TRACE(0);
				return ret;
			}
		}
		i = addr - bc->start_addr;
		bytes = bc->end_addr - addr;
		if (bytes > left)
			bytes = left;
		memcpy(target, bc->buf + i, bytes);
		release_block_cache(bc);
		target += bytes;
		left -= bytes;
		addr += bytes;
		MSG_TRACEI("bytes: %d left: %d addr: 0x%llx target: 0x%lx i: %d\n", bytes, left, addr, target, i);
		if (retlen)
			*retlen += bytes;
	}
	for (i = 0; i < 64; i += 4)
		MSG_TRACEN("%d: 0x%x 0x%x 0x%x 0x%x\n", i, buf[i], buf[i+1], buf[i+2], buf[i+3]);

	FUNC_TRACE(0);

	return ret;
}

static int bcmmtd_write(struct mtd_info *mtd,
				loff_t addr,
				size_t len,
				size_t *retlen,
				const u_char *buf)
{
	BlockCache *bc;
	int region, ret, bytes;
	int left = len;
	int pagesize = mtd->writesize;
	int log2pagesize = ilog2(pagesize);
	const u_char *bufptr = buf;
	u32 bcachesize = (mtd->erasesize < largest_page_size) ? mtd->erasesize : largest_page_size;

	FUNC_TRACE(1);

	MSG_TRACEI("%s: offset=%0llx, len=%08x bcachesize: %d\n", __func__, addr, len, bcachesize);

	if (retlen)
		*retlen = 0;

	/* sanity czech */
	if (addr + len > mtd->size) {
		MSG_TRACEE("%s ERROR: trying to write a beyond flash size %u\n",
				__func__, (u32)addr+len);
		FUNC_TRACE(0);
		return -EINVAL;
	}

	ret = 0;
	while (left) {
		region = get_flash_region((mtd_device *)mtd->priv, addr);
		bytes = left;

		if (bytes > bcachesize)
			bytes = bcachesize;

		bc = find_block_cache(mtd, addr);
		if (!bc)
			bc = new_block_cache(mtd, addr, bcachesize, mtd->type == MTD_NANDFLASH);

		memcpy(bc->buf, bufptr, bytes);

		ret = vflash_write_buf(region,
					(u32)addr/pagesize,
					bc->buf,
					bytes/pagesize,
					log2pagesize);
		if (ret < 0) {
			invalidate_block_cache(bc);
			release_block_cache(bc);
			break;
		}
		addr += bytes;
		left -= bytes;
		bufptr += bytes;
		if (retlen)
			*retlen += bytes;

		/*too many variables involved in keeping these written pieces in cache*/
		invalidate_block_cache(bc);
		release_block_cache(bc);
	}

	FUNC_TRACE(0);

	return ret;
}

static int get_flash_region(mtd_device *dev, loff_t addr)
{
	flash_partition	*partitions = dev->partitions;
	int part_cnt = dev->part_cnt;
	int i = dev->mru;

	FUNC_TRACE(1);

	if ((i >= 0) && (i < part_cnt) &&
	(addr < partitions[i].end) &&
	(addr >= partitions[i].offset))
		return partitions[i].region;

	for (i = 0; i < part_cnt; i++) {
		if ((addr < partitions[i].end) && (addr >= partitions[i].offset)) {
			dev->mru = i;
			FUNC_TRACE(0);
			return partitions[i].region;
		}
	}
	FUNC_TRACE(0);
	return -1;
}

static int bcmmtd_block_isbad(struct mtd_info *mtd, loff_t addr)
{
	int pagesize = mtd->writesize;
	int log2pagesize = ilog2(pagesize);

	int region = get_flash_region((mtd_device *)mtd->priv, addr);
	return vflash_block_isbad(region, (u32)addr/pagesize, log2pagesize);
}

static int bcmmtd_block_markbad(struct mtd_info *mtd, loff_t addr)
{
	int pagesize = mtd->writesize;
	int log2pagesize = ilog2(pagesize);

	int region = get_flash_region((mtd_device *)mtd->priv, addr);
	return vflash_block_markbad(region, (u32)addr/pagesize, log2pagesize);
}


static int bcmmtd_write_oob(struct mtd_info *mtd, loff_t to, struct mtd_oob_ops *ops)
{
	FUNC_TRACE(1);

	MSG_TRACEI("-->%s, offset=%0llx, len=%08x\n", __func__,  to, (int) ops->len);
	if (ops->datbuf && ((to + ops->len) > mtd->size)) {
		MSG_TRACEE("Attempt to write beyond end of device\n");
		FUNC_TRACE(0);
		return -EINVAL;
	}


	if (!ops->datbuf)
		MSG_TRACEI("oob only write, mode = %d\n", ops->mode);
	else
		MSG_TRACEE("oob + page write not supported yet!\n");

	FUNC_TRACE(0);

	return 0;
}

static int bcmmtd_read_oob(struct mtd_info *mtd, loff_t from,
			 struct mtd_oob_ops *ops)
{
	FUNC_TRACE(1);

	MSG_TRACEI("-->%s, offset=%lx len=%x, databuf=%px\n", __func__,
		(unsigned long)from, (unsigned)ops->len, ops->datbuf);

	if (ops->datbuf)
		MSG_TRACEI("oob only read!\n");
	else
		MSG_TRACEE("oob + page read not supported yet!\n");

	FUNC_TRACE(0);

	return 0;
}

static int init_mtd(void)
{
	int dev, part, part_cnt, ret;
	u32 erasesize;
	struct vflash_msg_device_cnt		devcnt_msg;
	struct vflash_msg_device_info		devinfo_msg;
	struct vflash_msg_partition_info	partinfo_msg;
	struct vflash_msg_partition_name	partname_msg;
	flash_partition				*partitions;

	FUNC_TRACE(1);

	rpc_msg_init_header((rpc_msg *)&devcnt_msg, RPC_SERVICE_FLASH, VF_INFO_DEVICE_CNT, 0);
	ret = rpc_send_request_timeout(g_rpc_tunnel, (rpc_msg *)&devcnt_msg, VFLASH_RPC_TIMEOUT);

	if (ret < 0) {
		MSG_TRACEE("%s: Fatal Error getting device info\n", __func__);
		return -1;
	}
	mtd_device_cnt = devcnt_msg.devicecnt;
	MSG_TRACEI("mtd_device_cnt: %d\n", mtd_device_cnt);
	if (mtd_device_cnt > 5)
		MSG_TRACEE("bogus device count %d\n", mtd_device_cnt);

	if (mtd_device_cnt <= 0)
		return 0;

	mtd_device_ptr = kmalloc(sizeof(mtd_device) * mtd_device_cnt, GFP_KERNEL);
	memset(mtd_device_ptr, 0, sizeof(mtd_device) * mtd_device_cnt);
	for (dev = 0; dev < mtd_device_cnt; dev++) {
		rpc_msg_init_header((rpc_msg *)&devinfo_msg, RPC_SERVICE_FLASH, VF_INFO_DEVICE_INFO, 0);
		devinfo_msg.device = dev;
		ret = rpc_send_request_timeout(g_rpc_tunnel, (rpc_msg *)&devinfo_msg, VFLASH_RPC_TIMEOUT);

	if (ret < 0 || vflash_msg_device_info_error(&devinfo_msg)) {
		MSG_TRACEE("%s: Fatal Error getting device info dev %d\n", __func__, dev);
		FUNC_TRACE(0);
		return -1;
	}
	if (devinfo_msg.type == kFlashTypeNand) {
		mtd_device_ptr[dev].mtd.name = bcmmtd_name;
		mtd_device_ptr[dev].mtd.type = MTD_NANDFLASH;
		mtd_device_ptr[dev].mtd.flags = MTD_CAP_NANDFLASH;

		mtd_device_ptr[dev].mtd._read_oob = bcmmtd_read_oob;
		mtd_device_ptr[dev].mtd._write_oob = bcmmtd_write_oob;
		mtd_device_ptr[dev].mtd._block_isbad = bcmmtd_block_isbad;
		mtd_device_ptr[dev].mtd._block_markbad = bcmmtd_block_markbad;

		mtd_device_ptr[dev].mtd.oobsize = 0;
		mtd_device_ptr[dev].mtd._point = NULL;
		mtd_device_ptr[dev].mtd._unpoint = NULL;
		mtd_device_ptr[dev].mtd._writev = NULL;
		mtd_device_ptr[dev].mtd._sync = NULL;
		mtd_device_ptr[dev].mtd._lock = NULL;
		mtd_device_ptr[dev].mtd._unlock = NULL;
		mtd_device_ptr[dev].mtd._suspend = NULL;
		mtd_device_ptr[dev].mtd._resume = NULL;
	} else {
		mtd_device_ptr[dev].mtd.name = bcmmtd_name;
		mtd_device_ptr[dev].mtd.type = MTD_NORFLASH;
		mtd_device_ptr[dev].mtd.flags = MTD_CAP_NORFLASH;
	}
	erasesize = (1 << devinfo_msg.block_size_in_pages) * devinfo_msg.page_size;
	if(devinfo_msg.page_size > largest_page_size)
		largest_page_size = devinfo_msg.page_size;
	mtd_device_ptr[dev].mtd.size = devinfo_msg.size_in_blocks * erasesize;
	mtd_device_ptr[dev].mtd.erasesize = erasesize;
	mtd_device_ptr[dev].mtd.writesize = devinfo_msg.page_size;
#if LINUX_VERSION_CODE > KERNEL_VERSION(2, 6, 37)
	mtd_device_ptr[dev].mtd.writebufsize = mtd_device_ptr[dev].mtd.writesize;
#endif
	mtd_device_ptr[dev].mtd._read = bcmmtd_read;
	mtd_device_ptr[dev].mtd._write = bcmmtd_write;
	mtd_device_ptr[dev].mtd._erase = bcmmtd_erase;

	part_cnt = devinfo_msg.partition_cnt;
	mtd_device_ptr[dev].mtd_parts =
		kmalloc(sizeof(struct mtd_partition) * part_cnt, GFP_KERNEL);
	memset(mtd_device_ptr[dev].mtd_parts, 0, sizeof(struct mtd_partition) * part_cnt);
	mtd_device_ptr[dev].part_cnt = part_cnt;
	MSG_TRACEI("Flash device %d: %4s %d partitions bs: %08x ps: %08x\n",
		dev, devinfo_msg.type == kFlashTypeNand ? "Nand" : "Nor",
		part_cnt, erasesize, devinfo_msg.page_size);

	mtd_device_ptr[dev].partitions = partitions =
	kmalloc(sizeof(flash_partition) * part_cnt, GFP_KERNEL);
	mtd_device_ptr[dev].part_cnt = part_cnt;
	mtd_device_ptr[dev].mtd.priv = &mtd_device_ptr[dev];
	MSG_TRACEI("%d partitions\n", part_cnt);
	MSG_TRACEI("mtd   offset     size name\n");

	for (part = 0; part < mtd_device_ptr[dev].part_cnt; part++) {
		rpc_msg_init_header((rpc_msg *)&partinfo_msg,
					RPC_SERVICE_FLASH,
					VF_INFO_PARTITION_INFO, 0);
		partinfo_msg.device = dev;
		partinfo_msg.partition = devinfo_msg.first_partition + part;
		ret = rpc_send_request_timeout(g_rpc_tunnel, (rpc_msg *)&partinfo_msg, VFLASH_RPC_TIMEOUT);

		if (ret < 0 || vflash_msg_partition_info_error(&partinfo_msg)) {
			MSG_TRACEE("Fatal Error getting partition info partition %d\n",
				partinfo_msg.partition);
			return -1;
		}
		mtd_device_ptr[dev].mtd_parts[part].offset =
			partinfo_msg.start_block * mtd_device_ptr[dev].mtd.erasesize;
		mtd_device_ptr[dev].mtd_parts[part].size =
			(partinfo_msg.end_block - partinfo_msg.start_block) *
				mtd_device_ptr[dev].mtd.erasesize;
		mtd_device_ptr[dev].mtd_parts[part].mask_flags =
				partinfo_msg.read_only ? MTD_WRITEABLE : 0;

		rpc_msg_init_header((rpc_msg *)&partname_msg,
					RPC_SERVICE_FLASH,
					VF_INFO_PARTITION_NAME, 0);
		partname_msg.region = partinfo_msg.region;
		ret = rpc_send_request_timeout(g_rpc_tunnel, (rpc_msg *)&partname_msg, VFLASH_RPC_TIMEOUT);

		if (ret < 0 || vflash_msg_partition_name_error(&partname_msg)) {
			MSG_TRACEE("Fatal Error getting region %d name\n",
				partname_msg.region);
			return -1;
		}
		mtd_device_ptr[dev].mtd_parts[part].name = kzalloc(32, GFP_KERNEL);
		memcpy((void *) mtd_device_ptr[dev].mtd_parts[part].name,
		       (void *) partname_msg.name, sizeof(partname_msg.name));

		partitions[part].offset = mtd_device_ptr[dev].mtd_parts[part].offset;
		partitions[part].end =
			mtd_device_ptr[dev].mtd_parts[part].offset +
			mtd_device_ptr[dev].mtd_parts[part].size;
		partitions[part].region = partinfo_msg.region;
		MSG_TRACEI("  %d %08x %08x %s\n", part,
			(u32)mtd_device_ptr[dev].mtd_parts[part].offset,
			(u32)mtd_device_ptr[dev].mtd_parts[part].size,
			mtd_device_ptr[dev].mtd_parts[part].name);
		}
	}
#if LINUX_VERSION_CODE > KERNEL_VERSION(2, 6, 30)
	for (dev = 0; dev < mtd_device_cnt; dev++)
		mtd_device_parse_register(&mtd_device_ptr[dev].mtd, NULL, 0,
			mtd_device_ptr[dev].mtd_parts, mtd_device_ptr[dev].part_cnt);
#else
	for (dev = 0; dev < mtd_device_cnt; dev++)
		add_mtd_partitions(&mtd_device_ptr[dev].mtd,
				mtd_device_ptr[dev].mtd_parts,
				mtd_device_ptr[dev].part_cnt);
#endif
	FUNC_TRACE(0);
	return 0;
}


/***************************************************************
 * block cache
 **************************************************************/
static BlockCache *blockcache;
static unsigned block_cache_lru_index;	/* last recently used */
static u16 num_block_caches;

static DEFINE_SPINLOCK(block_cache_lock);

static void	init_block_cache(u32 cachesize)
{
	int i;

	FUNC_TRACE(1);

	num_block_caches = cachesize/largest_page_size;
	blockcache = kmalloc(sizeof(BlockCache) * num_block_caches, GFP_KERNEL);
	memset(blockcache, 0, sizeof(BlockCache) * num_block_caches);
	for (i = 0; i < num_block_caches; i++)
		sema_init(&blockcache[i].lock, 1);

	FUNC_TRACE(0);
}

static BlockCache *find_block_cache(struct mtd_info	*mtd, loff_t addr)
{
	int i, locked;
	unsigned long flags;
	BlockCache *bc;

	FUNC_TRACE(1);

	spin_lock_irqsave(&block_cache_lock, flags);

	bc = &blockcache[block_cache_lru_index];
	if (likely((mtd == bc->mtd && bc->start_addr <= addr && addr < bc->end_addr))) {
		locked = down_trylock(&bc->lock) == 0;
		spin_unlock_irqrestore(&block_cache_lock, flags);
		if (likely(locked)) {
			FUNC_TRACE(0);
			return bc;
		} else { /* if its in flight wait for it */
			while (down_interruptible(&bc->lock));	/* stupid watchdog */
			/* reverify */
			if (likely((mtd == bc->mtd && bc->start_addr <= addr && addr < bc->end_addr))) {
				FUNC_TRACE(0);
				return bc;
			}
		}
		/* failure due to contention so try again */
		spin_lock_irqsave(&block_cache_lock, flags);
	}
	for (i = 0; i < num_block_caches; i++) {
		bc = &blockcache[i];
		if (mtd == bc->mtd && bc->start_addr <= addr && addr < bc->end_addr) {
			locked = down_trylock(&bc->lock) == 0;
			spin_unlock_irqrestore(&block_cache_lock, flags);
			if (likely(locked)) {
				FUNC_TRACE(0);
				return bc;
			} else { /* if its in flight wait for it */
				while (down_interruptible(&bc->lock));	/* stupid watchdog */
				/* reverify */
				if (likely((mtd == bc->mtd && bc->start_addr <= addr && addr < bc->end_addr))) {
					block_cache_lru_index = i;
					FUNC_TRACE(0);
					return bc;
				} else {
					FUNC_TRACE(0);
					return NULL;
				}
			}
		}
	}
	spin_unlock_irqrestore(&block_cache_lock, flags);
	FUNC_TRACE(0);
	return NULL;
}

static BlockCache *new_block_cache(struct mtd_info	*mtd, loff_t addr, u32 size, bool nand)
{
	int i, locked;
	unsigned long flags;
	BlockCache *bc;

	FUNC_TRACE(1);

	spin_lock_irqsave(&block_cache_lock, flags);

	i = block_cache_lru_index + 1;
	if (i == num_block_caches)
		i = 0;
	bc = &blockcache[i];
	locked = down_trylock(&bc->lock) == 0;
	if (unlikely(!locked)) {
		for (i = 0; i < num_block_caches; i++) {
			bc = &blockcache[i];
			locked = down_trylock(&bc->lock) == 0;
			if (locked)
				break;
		}
		if (i == num_block_caches) { /* everything is in use */
			i = block_cache_lru_index + 1;
			if (i == num_block_caches)
				i = 0;
			bc = &blockcache[i];
			spin_unlock_irqrestore(&block_cache_lock, flags);
			down(&bc->lock);
			spin_lock_irqsave(&block_cache_lock, flags);
		}
	}
	/*assign from pool of buffers*/
	bc->buf = gp_bcache + i*largest_page_size ;
	bc->start_addr = addr & ~(size - 1);
	bc->end_addr = bc->start_addr + size;
	bc->mtd = mtd;
	block_cache_lru_index = i;
	spin_unlock_irqrestore(&block_cache_lock, flags);
	MSG_TRACEI("New Block Cache Start: 0x%llx End: 0x%llx buffer: %px\n", bc->start_addr, bc->end_addr, bc->buf);

	FUNC_TRACE(0);

	return bc;
}

static void release_block_cache(BlockCache *bc)
{
	FUNC_TRACE(1);
	up(&bc->lock);
	FUNC_TRACE(0);
}

static void invalidate_block_cache(BlockCache *bc)
{
	FUNC_TRACE(1);
	bc->buf = NULL;
	bc->start_addr = bc->end_addr = 0;
	FUNC_TRACE(0);
}

/***************************************************************
 * debug
 **************************************************************/
void dump_buffer(u8 *buf, int len)
{
	int i = 0;

	while (i < len) {
		if (i % 16) {
			MSG_TRACEN(" %02x", buf[i]);
		} else {
			MSG_TRACEN("\n%08x:  %02x", i, buf[i]);
		}
		i++;
	}
}
#if 1
#if defined(CONFIG_OF)
/**
* brcm_vflashclient_of_map() - Get memory mapping information from OF if present
*
* Check the device tree for shared memory information
*
*/
static int brcm_vflashclient_of_map(struct device_node *node, u32* client_id)
{
	int ret = 0;
	const __be32 *prop_reg;
	const char *dev_name;
	struct device_node *phan_node;

	FUNC_TRACE(1);

	if (!node) {
		MSG_TRACEE("No devtree node found!!!\n");
		return -1;
	}

	prop_reg = of_get_property(node, "client_id", NULL);
	if (prop_reg == NULL) {
		MSG_TRACEI("missing client_id prop\n");
		ret = -EFAULT;
	} else {
		*client_id = be32_to_cpup(prop_reg);
		MSG_TRACEI("client_id: %ld\n", *client_id);
	}

	phan_node = of_parse_phandle(node, "rpc_channel", 0);
	if (!phan_node) {
		MSG_TRACEI("Unable to retrieve rpc-channel phandle ");
		ret = -EFAULT;
	}

	if (!of_property_read_string(phan_node, "dev-name", &dev_name)) {
		MSG_TRACEI("dev-name: %s\n", dev_name);
	} else {
		MSG_TRACEI("Missing dev-name property!\n");
		ret = -EINVAL;
	}
	of_node_put(phan_node);

	g_rpc_tunnel = rpc_get_fifo_tunnel_id((char *)dev_name);
	MSG_TRACEI("retrieved tunnel id: %d\n", g_rpc_tunnel);

	FUNC_TRACE(0);

	return ret;
}
#endif
#endif
/***************************************************************
Init and shutdown
***************************************************************/
static int vflash_init_dev(struct platform_device *pdev)
{
	int ret;
	struct device *dev = &pdev->dev;
	struct vflash_msg_client_register client_msg;
	u32 membase = 0, memsize = 0, client_id = -1;
    struct resource *res;
#if !defined(CONFIG_OF)
	unsigned long size;
	u32 shmem_offset;
#endif
	FUNC_TRACE(1);

	/*get static region set aside for us and block caching*/
#if defined(CONFIG_OF)

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	membase = (u32)res->start; /*TBD CAST*/
	memsize = (u32)resource_size(res); /*TBD CAST*/
	MSG_TRACEI("vfclient's memory is 0x%x at 0x%x\n", *memsize, *membase);

	/*process devtree information for this driver*/
	if(brcm_vflashclient_of_map(dev->of_node, &client_id) ) {
		MSG_TRACEE("vflash_init_dev failed to retrieve memory node from DT\n");
		FUNC_TRACE(0);
		return -1;
	}
	gp_bcache_phys = (u8 *) membase;

	gp_bcache = ioremap(membase, memsize);
	if (!gp_bcache) {
		MSG_TRACEE("ioremap failed\n");
		FUNC_TRACE(0);
		return -1;
	}
#else
	/*malloc the whole region into a global pointer*/
	shmem_offset = 0;
	memsize = g_largestEraseBlock * (num_block_caches + 1);
	size = (num_block_caches + 1) * g_largestEraseBlock;
	g_max_blockcache = (size/g_largestEraseBlock) - 1;
	MSG_TRACEI("block cache area: %ld bytes g_max_blockcache: %d\n", size, g_max_blockcache);
	gp_bcache = kmalloc(size, GFP_KERNEL);
#endif

	rpc_msg_init_header((rpc_msg *)&client_msg, RPC_SERVICE_FLASH, VF_INFO_CLIENT_REGISTER, 0);
	client_msg.sharedmem_addr = (u32)membase;
	client_msg.sharedmem_size = memsize;
	client_msg.client_id = client_id;

	while ((ret = rpc_send_request_timeout(g_rpc_tunnel, (rpc_msg *)&client_msg, VFLASH_RPC_TIMEOUT))) {
		pr_err("%s: rpc_send_request_timeout %d\n", __func__, ret);
		if (ret != -EAGAIN)
			break;
	}
	ret = init_mtd();
	if (ret < 0) {
		MSG_TRACEE("vflash_init_dev catastrophic ERROR in init_mtd\n");
		FUNC_TRACE(0);
		return ret;
	}

	if ((largest_page_size > 0) && (mtd_device_cnt > 0)) {
		init_block_cache(memsize);
	} else {
		pr_err("No shared partitions, flash will not be functional\n");
	}

	FUNC_TRACE(0);

	return 0;
}

static int vflash_probe(struct platform_device *pdev)
{
	int err;
#ifdef VFLASH_PROC_DEBUG
	struct proc_dir_entry *entry;
#endif
	FUNC_TRACE(1);

	err = vflash_init_dev(pdev);
	if (err) {
		FUNC_TRACE(0);
		return err;
	}

	platform_set_drvdata(pdev, mtd_device_ptr);
#ifdef VFLASH_PROC_DEBUG
	entry = create_proc_entry(PROCFS_NAME, 0644, NULL);
	if (entry) {
		entry->read_proc = vfc_proc_read;
		entry->write_proc = vfc_proc_write;
	}
	MSG_TRACEI("vfc_procfile: %px\n", entry);
#endif
	FUNC_TRACE(0);

	return 0;
}

static int vflash_remove(struct platform_device *pdev)
{
	int dev, part;

	FUNC_TRACE(1);

	for (dev = 0; dev < mtd_device_cnt; dev++)
		mtd_device_unregister(&mtd_device_ptr[dev].mtd);

	for (dev = 0; dev < mtd_device_cnt; dev++) {
		for (part = 0; part < mtd_device_ptr[dev].part_cnt; part++) {
			kfree(mtd_device_ptr[dev].mtd_parts[part].name);
		}
		kfree(mtd_device_ptr[dev].mtd_parts);
		kfree(mtd_device_ptr[dev].partitions);
	}
	kfree(blockcache);
	kfree(mtd_device_ptr);
	platform_set_drvdata(pdev, NULL);

#if !(defined(CONFIG_OF))
	kfree(gp_bcache);
#endif

#ifdef VFLASH_PROC_DEBUG
	remove_proc_entry(PROCFS_NAME, NULL);
#endif
	FUNC_TRACE(0);

	return 0;
}


static struct platform_driver vflash_driver = {
	.driver = {
		.name		= DRIVER_NAME,
		.owner		= THIS_MODULE,
		.of_match_table = of_platform_brcmvflash_table,
	},
	.probe		= vflash_probe,
	.remove		= vflash_remove,
};

module_platform_driver(vflash_driver);
