/****************************************************************************
*
* Broadcom Proprietary and Confidential. (c) 2018 Broadcom. All rights reserved.
* The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
*
* This program is the proprietary software of Broadcom and/or
* its licensors, and may only be used, duplicated, modified or distributed
* pursuant to the terms and conditions of a separate, written license
* agreement executed between you and Broadcom (an "Authorized License").
* Except as set forth in an Authorized License, Broadcom grants no license
* (express or implied), right to use, or waiver of any kind with respect to
* the Software, and Broadcom expressly reserves all rights in and to the
* Software and all intellectual property rights therein. IF YOU HAVE NO
* AUTHORIZED LICENSE, THEN YOU HAVE NO RIGHT TO USE THIS SOFTWARE IN ANY WAY,
* AND SHOULD IMMEDIATELY NOTIFY BROADCOM AND DISCONTINUE ALL USE OF THE
* SOFTWARE.
*
* Except as expressly set forth in the Authorized License,
*
* 1. This program, including its structure, sequence and organization,
* constitutes the valuable trade secrets of Broadcom, and you shall use all
* reasonable efforts to protect the confidentiality thereof, and to use this
* information only in connection with your use of Broadcom integrated circuit
* products.
*
* 2. TO THE MAXIMUM EXTENT PERMITTED BY LAW, THE SOFTWARE IS PROVIDED
* "AS IS" AND WITH ALL FAULTS AND BROADCOM MAKES NO PROMISES, REPRESENTATIONS
* OR WARRANTIES, EITHER EXPRESS, IMPLIED, STATUTORY, OR OTHERWISE, WITH
* RESPECT TO THE SOFTWARE. BROADCOM SPECIFICALLY DISCLAIMS ANY AND ALL
* IMPLIED WARRANTIES OF TITLE, MERCHANTABILITY, NONINFRINGEMENT, FITNESS FOR
* A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY OR COMPLETENESS, QUIET
* ENJOYMENT, QUIET POSSESSION OR CORRESPONDENCE TO DESCRIPTION. YOU ASSUME
* THE ENTIRE RISK ARISING OUT OF USE OR PERFORMANCE OF THE SOFTWARE.
*
* 3. TO THE MAXIMUM EXTENT PERMITTED BY LAW, IN NO EVENT SHALL BROADCOM
* OR ITS LICENSORS BE LIABLE FOR (i) CONSEQUENTIAL, INCIDENTAL, SPECIAL,
* INDIRECT, OR EXEMPLARY DAMAGES WHATSOEVER ARISING OUT OF OR IN ANY WAY
* RELATING TO YOUR USE OF OR INABILITY TO USE THE SOFTWARE EVEN IF BROADCOM
* HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES; OR (ii) ANY AMOUNT IN
* EXCESS OF THE AMOUNT ACTUALLY PAID FOR THE SOFTWARE ITSELF OR U.S. $1,
* WHICHEVER IS GREATER. THESE LIMITATIONS SHALL APPLY NOTWITHSTANDING ANY
* FAILURE OF ESSENTIAL PURPOSE OF ANY LIMITED REMEDY.
*
****************************************************************************
*
* Filename:    ramlogview.c
*
****************************************************************************
*
* Description: ramlog view kernel driver
*
****************************************************************************/

#include <linux/init.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/device.h>
#include <linux/io.h>
#include <linux/fs.h>
#include <linux/slab.h>
#include <linux/proc_fs.h>
#include <linux/string.h>
#include <linux/ctype.h>
#include <linux/wait.h>
#include <linux/poll.h>
#include <linux/delay.h>
#include <linux/fsnotify.h>
#include <asm/errno.h>

#include <linux/of_platform.h>

#include "ramlogrpc.h"

#define	MAX_FORMATTED_LENGTH		1024
#define	MAX_TOKEN_LENGTH		63
#define	MAX_RAMLOG_LINE_LENGTH		256
#define	RAMLOG_PAGE_SIZE		4096
#define	DEF_RAMLOG_PARAMETER_LENGTH	1024
#define	RAMLOG_DEVICE_NAME		"smclog"
#define	RAMLOG_PROCFS_DIR_NAME		"driver/smc"
#define	RAMLOG_PROCFS_NAME		"log"

#define	BYTES_PER_LINE			32

#define	GET_TIMESTAMP_DIFF(left, right)	\
	(((int32_t) (left)) - ((int32_t) (right)))

struct ramlog_file_buffer_t {
	struct file *filep;
	struct device *dev;
	struct ramlog_rpc_data *ramlog;
	uint32_t buf_size;
	uint8_t *buf_ptr;
	dma_addr_t dma_addr;
	uint32_t fifo_index, fifo_count;
	uint32_t text_length;
	char text_buffer[MAX_FORMATTED_LENGTH + 1];
	struct semaphore sem;
};

static int ramlog_open(struct inode *inodep, struct file *filep);
static int ramlog_release(struct inode *inodep, struct file *filep);
static ssize_t ramlog_read(struct file *filep, char *buffer, size_t count,
	loff_t *offp);
static ssize_t ramlog_write(struct file *filep, const char *buffer,
	size_t count, loff_t *offp);
static unsigned int ramlog_poll(struct file *filep,
	struct poll_table_struct *wait);

static int ramlog_probe(struct platform_device *pdev);
static int ramlog_remove(struct platform_device *pdev);

static uint8_t *alloc_ramlog_buffer (struct ramlog_file_buffer_t *ramlog_view,
	unsigned int buf_size);
static void execute_ramlog_command(struct ramlog_file_buffer_t *ramlog_view,
	const char *command);
static const char *get_ramlog_token(char *token, unsigned int max_token_size,
	const char *str_to_parse);

static const struct of_device_id ramlog_of_match[] = {
	{ .compatible = "brcm,smcramlog-rpc", },
	{ /* end of the list */ }
};

static struct platform_driver ramlog_platform_driver = {
	.driver = {
		.name           = RAMLOG_DEVICE_NAME,
		.owner          = THIS_MODULE,
		.of_match_table = ramlog_of_match,
	},
	.probe  = ramlog_probe,
	.remove = ramlog_remove,
};

static const struct proc_ops ramlog_file_fops = {
	.proc_open    = ramlog_open,
	.proc_release = ramlog_release,
	.proc_read    = ramlog_read,
	.proc_write   = ramlog_write,
	.proc_poll    = ramlog_poll,
};

struct ramlog_rpc_data *smc_ramlog;
EXPORT_SYMBOL(smc_ramlog);

static struct proc_dir_entry *ramlog_proc_entry;
static struct proc_dir_entry *ramlog_proc_parent;
static DECLARE_WAIT_QUEUE_HEAD(ramlog_wait);

static const char *const help_msgs[] = {
	"\r\nTo dump encrypted SMC ramlog:\r\n",
	"    cat /proc/driver/smc/log\r\n",
	"To clear ramlog:\r\n",
	"    echo \"clear\" >/proc/driver/smc/log\r\n",
	"To add a message to ramlog:\r\n",
	"    echo \"<message>\" >/proc/driver/smc/log\r\n",
	"To add a message to ramlog at a specific level:\r\n",
	"    echo \"(error|warning|notice|message|debug|deep):\r\n",
	"        <message>\" >/proc/driver/smc/log\r\n",
	"To change level:\r\n",
	"    echo \"level=(error|warning|notice|msg|debug|deep)\"\r\n",
	"        >/proc/driver/smc/log\r\n",
	"To change level for a specific source:\r\n",
	"    echo \"level=(error|warning|notice|msg|debug|deep)\r\n",
	"        source=<source name>\" >/proc/driver/smc/log\r\n",
	"To clear all custom levels:\r\n",
	"    echo \"level=(error|warning|notice|msg|debug|deep)\r\n",
	"        source=*\" >/proc/driver/smc/log\r\n\r\n"
};

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Hadula, Tomasz");
MODULE_DESCRIPTION("SMC ramlog viewer");
MODULE_VERSION("1.0");

static int ramlog_open(struct inode *inodep, struct file *filep)
{
	struct ramlog_file_buffer_t *ramlog_view;

	(void) inodep;

	if (!filep)
		return -EBADF;

	ramlog_view = (struct ramlog_file_buffer_t *)
		kmalloc(sizeof(struct ramlog_file_buffer_t), GFP_KERNEL);
	if (ramlog_view) {
		memset(ramlog_view, 0, sizeof(struct ramlog_file_buffer_t));
		ramlog_view->fifo_index = RAMLOG_HEAD_START_INDEX;
		sema_init(&ramlog_view->sem, 1);
		ramlog_view->filep = filep;
		filep->private_data = (void *) ramlog_view;
		ramlog_view->ramlog = smc_ramlog;
		ramlog_view->dev = smc_ramlog->dev;
	} else
		return -EINVAL;
	return 0;
}

static int ramlog_release(struct inode *inodep, struct file *filep)
{
	struct ramlog_file_buffer_t *ramlog_view;

	(void) inodep;

	if (!filep)
		return -EBADF;

	if (filep->private_data) {
		ramlog_view = (struct ramlog_file_buffer_t *)
			filep->private_data;
		if (ramlog_view->buf_size) {
			dma_free_coherent(ramlog_view->dev, ramlog_view->buf_size,
				ramlog_view->buf_ptr, ramlog_view->dma_addr);
		}
		ramlog_view->buf_size = 0;
		ramlog_view->buf_ptr = 0;
		ramlog_view->dma_addr = 0;
		ramlog_view->filep = 0;
		kfree(filep->private_data);
		filep->private_data = NULL;
	}
	return 0;
}

static ssize_t ramlog_read(struct file *filep, char *buffer, size_t count,
	loff_t *offp)
{
	struct ramlog_file_buffer_t *ramlog_view;
	struct smc_ramlog_hdr *header;
	size_t max_chunk_size, max_chunk_text_size, total_count;
	size_t chunk_size, chunk_text_size;
	const char *buffer_ptr;
	char *chunk_buffer, *chunk_ptr;
	uint32_t next_start;
	uint8_t digit;
	uint32_t i, j;
	int result, eof;

	if (!filep)
		return -EBADF;

	ramlog_view = (struct ramlog_file_buffer_t *) filep->private_data;
	if (!ramlog_view) {
		pr_err("ramlog read buffer not allocated\n");
		return -EBADF;
	}
	result = down_interruptible(&ramlog_view->sem);
	if (result < 0) {
	    dev_err(ramlog_view->dev, "Cannot enter mutex, result=%d\n", result);
		return result;
	}

#ifdef	SMC_RAMLOG_VERBOSE
	dev_info(ramlog_view->dev, "count=%lu\n",
			(unsigned long) (offp ? *offp : 0));
#endif	/* SMC_RAMLOG_VERBOSE */
	eof = 0;
	max_chunk_size = (((RAMLOG_MSG_BUFFER_SIZE -
			sizeof (struct smc_ramlog_hdr)) & ~0xf) +
			sizeof (struct smc_ramlog_hdr));
	max_chunk_text_size = (max_chunk_size * (2 * BYTES_PER_LINE + 2) +
			BYTES_PER_LINE - 1) / BYTES_PER_LINE;
	if (count < max_chunk_text_size) {
		max_chunk_size = (count * BYTES_PER_LINE) / (2 * BYTES_PER_LINE + 2);
		max_chunk_size = (((max_chunk_size - sizeof (struct smc_ramlog_hdr)) &
				~0xf) + sizeof (struct smc_ramlog_hdr));
		max_chunk_text_size = (max_chunk_size * (2 * BYTES_PER_LINE + 2) +
				BYTES_PER_LINE - 1) / BYTES_PER_LINE;;
	}
#ifdef	SMC_RAMLOG_VERBOSE
	dev_info(ramlog_view->dev,
			"count=%lu max_chunk_size=%lu max_chunk_text_size=%lu\n",
			(unsigned long) count, (unsigned long) max_chunk_size,
			(unsigned long) max_chunk_text_size);
#endif	/* SMC_RAMLOG_VERBOSE */
	/* assert(count >= max_chunk_text_size); */
	header = (struct smc_ramlog_hdr *)
		alloc_ramlog_buffer(ramlog_view, max_chunk_size);
	chunk_buffer = (char*) kmalloc(max_chunk_text_size + 1, GFP_KERNEL);
	total_count = 0;
	for (;;) {
		memset(header, 0, sizeof(struct smc_ramlog_hdr));
		next_start = get_encrypted_ramlog_entries(
			ramlog_view->dma_addr, max_chunk_size,
			ramlog_view->ramlog, &ramlog_view->fifo_count, NULL,
			ramlog_view->fifo_index);
#ifdef	SMC_RAMLOG_VERBOSE
		dev_info(ramlog_view->dev,
				"header: length=%lu max chunk=%lu start=%lu next=%lu\n",
				(unsigned long) header->length,
				(unsigned long) max_chunk_size,
				(unsigned long) ramlog_view->fifo_index,
				(unsigned long) next_start);
#endif	/* SMC_RAMLOG_VERBOSE */
		if (!header->length) {
			eof = 1;
			break;
		}
		chunk_size = header->length + sizeof(struct smc_ramlog_hdr);
		chunk_text_size = (chunk_size * (2 * BYTES_PER_LINE + 2) +
				BYTES_PER_LINE - 1) / BYTES_PER_LINE;
#ifdef	SMC_RAMLOG_VERBOSE
		dev_info(ramlog_view->dev,
				"chunk_size=%lu chunk text size=%lu total=%lu\n",
				(unsigned long) chunk_size, (unsigned long) chunk_text_size,
				(unsigned long) total_count);
#endif	/* SMC_RAMLOG_VERBOSE */
		if (count <= total_count + chunk_text_size)
			break;
		buffer_ptr = (const uint8_t*) header;
		chunk_ptr = chunk_buffer;
		for (i = 0; i < chunk_size;) {
			for (j = 0; j < BYTES_PER_LINE && i < chunk_size; j++, i++) {
				digit = (uint8_t) ((*buffer_ptr & 0xf0) >> 4);
				if (digit < 10)
					*chunk_ptr++ = (char) (digit + '0');
				else
					*chunk_ptr++ = (char) (digit - 10 + 'a');
				digit = (uint8_t) (*buffer_ptr++ & 0x0f);
				if (digit < 10)
					*chunk_ptr++ = (char) (digit + '0');
				else
					*chunk_ptr++ = (char) (digit - 10 + 'a');
			}
			*chunk_ptr++ = '\r';
			*chunk_ptr++ = '\n';
		}
		*chunk_ptr = '\0';
#ifdef	SMC_RAMLOG_VERBOSE
		dev_info(ramlog_view->dev, "text_size=%lu\n",
				(unsigned long) (chunk_ptr - chunk_buffer));
#endif	/* SMC_RAMLOG_VERBOSE */
		result = copy_to_user(&buffer[total_count],
				chunk_buffer, (chunk_ptr - chunk_buffer));
		if (result < 0) {
		    dev_err(ramlog_view->dev, "Ramlog not copied\n");
			if (chunk_buffer)
				kfree(chunk_buffer);
			up(&ramlog_view->sem);
			return result;
		}
		total_count += (chunk_ptr - chunk_buffer);
		ramlog_view->fifo_index = next_start;
	}
	if (!eof && count > total_count) {
		chunk_ptr = chunk_buffer;
		for (i = 0; i < count - total_count - 1; i++)
			*chunk_ptr++ = '\r';
		*chunk_ptr++ = '\n';
		*chunk_ptr = '\0';
		result = copy_to_user(&buffer[total_count],
				chunk_buffer, (chunk_ptr - chunk_buffer));
		if (result < 0) {
		    dev_err(ramlog_view->dev, "Ramlog not copied\n");
			if (chunk_buffer)
				kfree(chunk_buffer);
			up(&ramlog_view->sem);
			return result;
		}
		total_count = count;
	}
	if (offp)
		*offp += total_count;
	if (chunk_buffer)
		kfree(chunk_buffer);
	up(&ramlog_view->sem);
	return (ssize_t) total_count;
}

static ssize_t ramlog_write(struct file *filep, const char *buffer,
	size_t count, loff_t *offp)
{
	struct ramlog_file_buffer_t *ramlog_view;
	char *ptr;
	const char *source;
	unsigned int length;
	int result;

	if (!filep)
		return -EBADF;

	ramlog_view = (struct ramlog_file_buffer_t *) filep->private_data;
	if (!ramlog_view) {
		pr_err("ramlog write buffer not allocated\n");
		return -EBADF;
	}

	result = down_interruptible(&ramlog_view->sem);
	if (result < 0) {
	    dev_err(ramlog_view->dev, "Cannot enter mutex, result=%d\n", result);
		return result;
	}

	ramlog_view->text_buffer[0] = '\0';
	ramlog_view->text_buffer[sizeof(ramlog_view->text_buffer) - 1] = '\0';
	length = (unsigned int) count;
	if (length > sizeof(ramlog_view->text_buffer) - 1)
		length = sizeof(ramlog_view->text_buffer) - 1;
	if (buffer && length) {
		result = copy_from_user(ramlog_view->text_buffer,
				buffer, length);
		barrier();
		if (result < 0) {
		    dev_err(ramlog_view->dev, "Cannot copy from user, result=%d\n",
				result);
			return result;
		}
	}
	ramlog_view->text_buffer[length] = '\0';
	if (!length)
		return count;

	ptr = ramlog_view->text_buffer + length - 1;
	while (ptr >= ramlog_view->text_buffer)
		if (*ptr == '\r' || *ptr == '\n') {
			*ptr-- = '\0';
			--length;
		} else
			break;
	ramlog_view->text_buffer[length] = '\0';
	source = ramlog_view->text_buffer;
	while (*source && isspace(*source)) {
		++source;
		--length;
	}
	if (!length)
		return count;
	execute_ramlog_command(ramlog_view, source);
	up(&ramlog_view->sem);
	return (ssize_t) count;
}

static void execute_ramlog_command(struct ramlog_file_buffer_t *ramlog_view,
	const char *command)
{
	const char *parsing, *module;
	unsigned int severity;
	char token[MAX_TOKEN_LENGTH + 1];
	unsigned int i, length;

	parsing = get_ramlog_token(token, sizeof(token), command);
	severity = RAMLOGFLAG_SEVERITY_MESSAGE;
	if (!strcmp(token, "help") || !strcmp(token, "HELP")) {
		if (!parsing) {
			for (i = 0; i < sizeof(help_msgs) / sizeof(*help_msgs);
				i++)
				pr_info("%s", help_msgs[i]);
			return;
		}
	} else if (!strcmp(token, "clear") || !strcmp(token, "CLEAR") ||
		!strcmp(token, "erase") || !strcmp(token, "ERASE")) {
		if (!parsing) {
			clear_ramlog(ramlog_view->ramlog);
			dev_info(ramlog_view->dev, "ramlog cleared\n");
			return;
		}
	} else if (!strcmp(token, "level") || !strcmp(token, "LEVEL")) {
		parsing = get_ramlog_token(token, sizeof(token), parsing);
		if (strcmp(token, "=")) {
			if (!command || !*command)
				return;
			if (!alloc_ramlog_buffer(ramlog_view,
				DEF_RAMLOG_PARAMETER_LENGTH))
				return;
			length = strlen(command);
			if (length >= DEF_RAMLOG_PARAMETER_LENGTH)
				length = DEF_RAMLOG_PARAMETER_LENGTH - 1;
			memcpy(ramlog_view->buf_ptr, command, length);
			ramlog_view->buf_ptr[length] = (uint8_t) 0;
			add_to_ramlog(ramlog_view->ramlog, severity,
				length + 1, ramlog_view->dma_addr);
			return;
		}
		severity = RAMLOGFLAG_SEVERITY_NONE;
		parsing = get_ramlog_token(token,
			sizeof(token), parsing);
		if (!strcmp(token, "error") ||
			!strcmp(token, "ERROR") ||
			!strcmp(token, "err") || !strcmp(token, "ERR")) {
			severity = RAMLOGFLAG_SEVERITY_ERROR;
		} else if (!strcmp(token, "warning") ||
			!strcmp(token, "WARNING") ||
			!strcmp(token, "warn") ||
			!strcmp(token, "WARN")) {
			severity = RAMLOGFLAG_SEVERITY_WARNING;
		} else if (!strcmp(token, "notice") ||
			!strcmp(token, "NOTICE")) {
			severity = RAMLOGFLAG_SEVERITY_NOTICE;
		} else if (!strcmp(token, "message") ||
			!strcmp(token, "MESSAGE") ||
			!strcmp(token, "msg") ||
			!strcmp(token, "MSG")) {
			severity = RAMLOGFLAG_SEVERITY_MESSAGE;
		} else if (!strcmp(token, "debug") ||
			!strcmp(token, "DEBUG") ||
			!strcmp(token, "dbg") ||
			!strcmp(token, "DBG")) {
			severity = RAMLOGFLAG_SEVERITY_DEBUG;
		} else if (!strcmp(token, "debug2") ||
			!strcmp(token, "DEBUG2") ||
			!strcmp(token, "dbg2") ||
			!strcmp(token, "DBG2") ||
			!strcmp(token, "deep") ||
			!strcmp(token, "DEEP") ||
			!strcmp(token, "deepdbg") ||
			!strcmp(token, "DEEPDBG")) {
			severity = RAMLOGFLAG_SEVERITY_DEEP_DEBUG;
		}
		if (severity == RAMLOGFLAG_SEVERITY_NONE) {
			severity = RAMLOGFLAG_SEVERITY_MESSAGE;
			if (!command || !*command)
				return;
			if (!alloc_ramlog_buffer(ramlog_view,
				DEF_RAMLOG_PARAMETER_LENGTH))
				return;
			length = strlen(command);
			if (length >= DEF_RAMLOG_PARAMETER_LENGTH)
				length = DEF_RAMLOG_PARAMETER_LENGTH - 1;
			memcpy(ramlog_view->buf_ptr, command, length);
			ramlog_view->buf_ptr[length] = (uint8_t) 0;
			add_to_ramlog(ramlog_view->ramlog, severity,
				length + 1, ramlog_view->dma_addr);
			return;
		}
		if (!parsing) {
			set_ramlog_level(ramlog_view->ramlog, severity, 0,
				(dma_addr_t) NULL);
			dev_info(ramlog_view->dev, "ramlog level=%u\n", severity);
			return;
		}
		parsing = get_ramlog_token(token, sizeof(token), parsing);
		module = NULL;
		if (!strcmp(token, "module") ||
			!strcmp(token, "MODULE") ||
			!strcmp(token, "source") ||
			!strcmp(token, "SOURCE")) {
			if (!alloc_ramlog_buffer(ramlog_view,
				DEF_RAMLOG_PARAMETER_LENGTH))
				return;
			parsing = get_ramlog_token(token,
				sizeof(token), parsing);
			if (strcmp(token, "=")) {
				severity = RAMLOGFLAG_SEVERITY_MESSAGE;
				if (!command || !*command)
					return;
				length = strlen(command);
				if (length >= DEF_RAMLOG_PARAMETER_LENGTH)
					length = DEF_RAMLOG_PARAMETER_LENGTH
						- 1;
				memcpy(ramlog_view->buf_ptr, command, length);
				ramlog_view->buf_ptr[length] = (uint8_t) 0;
				add_to_ramlog(ramlog_view->ramlog, severity,
					length + 1, ramlog_view->dma_addr);
				return;
			}
			token[0] = '\0';
			parsing = get_ramlog_token(token, sizeof(token),
				parsing);
			if (!strcmp(token, "!") ||
				!strcmp(token, "*"))
				module = "*";
			else if (token[0])
				module = token;
			if (module) {
				length = strlen(module);
				if (length >= DEF_RAMLOG_PARAMETER_LENGTH)
					length = DEF_RAMLOG_PARAMETER_LENGTH
						- 1;
				memcpy(ramlog_view->buf_ptr, module, length);
				ramlog_view->buf_ptr[length] = (uint8_t) 0;
				set_ramlog_level(ramlog_view->ramlog,
					severity, length + 1,
					ramlog_view->dma_addr);
				dev_info(ramlog_view->dev, "ramlog level=%u source=%s\n",
					severity, module);
				return;
			}
		}
		severity = RAMLOGFLAG_SEVERITY_MESSAGE;
	} else if (!strcmp(token, "error") || !strcmp(token, "ERROR") ||
			!strcmp(token, "err") || !strcmp(token, "ERR")) {
		parsing = get_ramlog_token(token, sizeof(token), parsing);
		if (!strcmp(token, ":")) {
			severity = RAMLOGFLAG_SEVERITY_ERROR;
			command = parsing;
		}
	} else if (!strcmp(token, "warning") ||
		!strcmp(token, "WARNING") ||
		!strcmp(token, "warn") || !strcmp(token, "WARN")) {
		parsing = get_ramlog_token(token, sizeof(token), parsing);
		if (!strcmp(token, ":")) {
			severity = RAMLOGFLAG_SEVERITY_WARNING;
			command = parsing;
		}
	} else if (!strcmp(token, "notice") || !strcmp(token, "NOTICE")) {
		parsing = get_ramlog_token(token, sizeof(token), parsing);
		if (!strcmp(token, ":")) {
			severity = RAMLOGFLAG_SEVERITY_NOTICE;
			command = parsing;
		}
	} else if (!strcmp(token, "message") ||
		!strcmp(token, "MESSAGE") ||
		!strcmp(token, "msg") || !strcmp(token, "MSG")) {
		parsing = get_ramlog_token(token, sizeof(token), parsing);
		if (!strcmp(token, ":")) {
			severity = RAMLOGFLAG_SEVERITY_MESSAGE;
			command = parsing;
		}
	} else if (!strcmp(token, "debug") || !strcmp(token, "DEBUG") ||
		!strcmp(token, "dbg") || !strcmp(token, "DBG")) {
		parsing = get_ramlog_token(token, sizeof(token), parsing);
		if (!strcmp(token, ":")) {
			severity = RAMLOGFLAG_SEVERITY_DEBUG;
			command = parsing;
		}
	} else if (!strcmp(token, "debug2") || !strcmp(token, "DEBUG2") ||
		!strcmp(token, "dbg2") || !strcmp(token, "DBG2") ||
		!strcmp(token, "deep") || !strcmp(token, "DEEP") ||
		!strcmp(token, "deepdbg") || !strcmp(token, "DEEPDBG")) {
		parsing = get_ramlog_token(token, sizeof(token), parsing);
		if (!strcmp(token, ":")) {
			severity = RAMLOGFLAG_SEVERITY_DEEP_DEBUG;
			command = parsing;
		}
	}

	if (!command || !*command)
		return;
	if (!alloc_ramlog_buffer(ramlog_view, DEF_RAMLOG_PARAMETER_LENGTH))
		return;
	length = strlen(command);
	if (length >= DEF_RAMLOG_PARAMETER_LENGTH)
		length = DEF_RAMLOG_PARAMETER_LENGTH - 1;
	memcpy(ramlog_view->buf_ptr, command, length);
	ramlog_view->buf_ptr[length] = (uint8_t) 0;
	add_to_ramlog(ramlog_view->ramlog, severity,
			length + 1, ramlog_view->dma_addr);
}

static uint8_t *alloc_ramlog_buffer (struct ramlog_file_buffer_t *ramlog_view,
	unsigned int buf_size)
{
	dma_set_mask(ramlog_view->dev, 0xffffffff);
	if (ramlog_view->buf_size < buf_size) {
		buf_size = ALIGN(buf_size, RAMLOG_PAGE_SIZE);
		if (ramlog_view->buf_size) {
			dma_free_coherent(ramlog_view->dev, ramlog_view->buf_size,
				ramlog_view->buf_ptr, ramlog_view->dma_addr);
		}
		ramlog_view->buf_ptr = (uint8_t*) dma_alloc_coherent(ramlog_view->dev,
		        buf_size, &ramlog_view->dma_addr, GFP_KERNEL);
		if (!ramlog_view->buf_ptr)
			return 0;
		if (dma_mapping_error(ramlog_view->dev,
			ramlog_view->dma_addr)) {
		    dev_err(ramlog_view->dev, "DMA address not mapped to %#lx\n",
				(unsigned long) ramlog_view->buf_ptr);
			return 0;
		}
		ramlog_view->buf_size = buf_size;
	}
	return ramlog_view->buf_ptr;
}

static const char *get_ramlog_token(char *token, unsigned int max_token_size,
	const char *str_to_parse)
{
	const char *processing;
	char *target;

	if (token && max_token_size) {
		token[0] = '\0';
		if (max_token_size > 1)
			token[max_token_size - 1] = '\0';
	}
	if (!str_to_parse || !*str_to_parse)
		return 0;
	processing = str_to_parse;
	while (*processing && isspace(*processing))
		++processing;
	if (!*processing)
		return 0;
	switch (*processing) {
	case ':':
	case '=':
		if (max_token_size > 0) {
			token[0] = *processing;
			if (max_token_size > 1)
				token[1] = '\0';
		}
		++processing;
		while (*processing && isspace(*processing))
			++processing;
		if (!*processing)
			processing = 0;
		return processing;
	default:
		break;
	}
	target = token;
	while (*processing) {
		if (isspace(*processing) || *processing == ':' ||
			    *processing == '=')
			break;
		if (target && max_token_size > 1) {
			*target++ = *processing;
			--max_token_size;
		}
		++processing;
	}
	if (target && max_token_size > 0)
		*target++ = '\0';
	while (*processing && isspace(*processing))
		++processing;
	if (!*processing)
		processing = 0;
	return processing;
}


static unsigned int ramlog_poll(struct file *filep,
				struct poll_table_struct *wait)
{
	poll_wait(filep, &ramlog_wait, wait);
	return 0;
}

static int ramlog_probe(struct platform_device *pdev)
{
	int status;

	/* allocate resources */
	smc_ramlog = (struct ramlog_rpc_data *)
		kzalloc(sizeof(struct ramlog_rpc_data), GFP_KERNEL);
	if (!smc_ramlog)
		return -ENOMEM;
	status = register_ramlog_rpc_from_platform_device(smc_ramlog, pdev);
	if (status)
		return status;

	/* create /proc pseudo file */
	ramlog_proc_parent = proc_mkdir(RAMLOG_PROCFS_DIR_NAME, NULL);
	if (!ramlog_proc_parent)
	    dev_err(&pdev->dev, "/proc/%s cannot be created.\n",
			RAMLOG_PROCFS_DIR_NAME);
	else {
		ramlog_proc_entry = proc_create(RAMLOG_PROCFS_NAME, 0644,
			ramlog_proc_parent, &ramlog_file_fops);
		if (!ramlog_proc_entry)
		    dev_err(&pdev->dev, "/proc/%s/%s cannot be created.\n",
				RAMLOG_PROCFS_DIR_NAME, RAMLOG_PROCFS_NAME);
		else
			dev_info(&pdev->dev, "/proc/%s/%s created.\n",
				RAMLOG_PROCFS_DIR_NAME, RAMLOG_PROCFS_NAME);
	}

	dev_info(&pdev->dev,"ramlog viewer probe succeeded\n");
	return 0;
}

static int ramlog_remove(struct platform_device *pdev)
{
	/* get rid of /proc entry */
	if (ramlog_proc_parent) {
		if (ramlog_proc_entry) {
			remove_proc_entry(RAMLOG_PROCFS_NAME,
					ramlog_proc_parent);
			ramlog_proc_entry = NULL;
			dev_info(&pdev->dev, "/proc/%s/%s deleted\n",
				RAMLOG_PROCFS_DIR_NAME, RAMLOG_PROCFS_NAME);
		}
		remove_proc_entry(RAMLOG_PROCFS_DIR_NAME, NULL);
		ramlog_proc_parent = NULL;
		dev_info(&pdev->dev, "/proc/%s deleted\n", RAMLOG_PROCFS_DIR_NAME);
	}

	/* release resources */
	if (smc_ramlog) {
		release_ramlog_rpc(smc_ramlog);
		kfree(smc_ramlog);
		smc_ramlog = NULL;
	}

	dev_info(&pdev->dev, "ramlog viewer removed\n");
	return 0;
}

static int __init ramlogview_init(void)
{
	return platform_driver_register(&ramlog_platform_driver);
}

/**
 * This routine unloads the ramlog viewer from kernel
 */
static void __exit ramlogview_exit(void)
{
	platform_driver_unregister(&ramlog_platform_driver);
}

module_init(ramlogview_init);
module_exit(ramlogview_exit);
