 /****************************************************************************
 *
 * Copyright (c) 2015 Broadcom Corporation
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************/
#include <linux/kernel.h>
#include <linux/interrupt.h>
#include <linux/workqueue.h>

#include "ethsw.h"
#include "ethsw_priv.h"

/*
 * Log Utilities
 */

/* define log module */
#define LOG_MODULE "ethsw_int"

/* ethsw interrupt control block */
struct ethsw_int {
	struct ethsw_device *swdev;

	/* work bottom half */
	struct work_struct irq_work;
};

static irqreturn_t ethsw_int_isr(int irq, void *data)
{
	struct ethsw_device *swdev;
	struct ethsw_int *swint;

	swint = (struct ethsw_int *)data;
	swdev = swint->swdev;

	/* call ISR, platform dependent */
	if (swdev->ethsw_int_board_isr) {
		if (!swdev->ethsw_int_board_isr(swdev, irq)) {
			/* schedule work for bottom half */
			schedule_work(&swint->irq_work);
		}
	}

	return(IRQ_HANDLED);
}

static void ethsw_int_proc(struct work_struct *work)
{
	struct ethsw_device *swdev;
	struct ethsw_int *swint;

	swint = container_of(work, struct ethsw_int, irq_work);
	swdev = swint->swdev;

	/* call proc func for bottom half, platform dependent */
	if (swdev->ethsw_int_board_proc) {
		if (swdev->ethsw_int_board_proc(swdev)) {
			LOG_ERR("Interrupt processing failed.\n");
			return;
		}
	}
}

int ethsw_int_init(struct ethsw_device *swdev)
{
	int ret = 0;
	int i;
	struct ethsw_int *swint;

	FUNC_ENTER();

	/* alloc ethsw interrupt control block */
	swint = kzalloc(sizeof(struct ethsw_int), GFP_KERNEL);
	if (!swint) {
		LOG_CRIT("Ethsw interrupt control block alloc failed\n");
		ret = -ENOMEM;
		goto ERROR;
	}

	swdev->swint = swint;
	swint->swdev = swdev;

	/* init interrupt, platform dependent */
	if (swdev->ethsw_int_board_init) {
		ret = swdev->ethsw_int_board_init(swdev);
		if (ret) {
			LOG_CRIT("Interrupt board init failed\n");
			goto ERROR;
		}
	}

	/* init work for bottom half */
	INIT_WORK(&swint->irq_work, ethsw_int_proc);

	/* request IRQ */
	for (i = 0; i < ETHSW_IRQ_MAX; i++) {
		if (request_irq(swdev->irq[i], ethsw_int_isr, IRQF_SHARED,
						swdev->name, swint) < 0) {
			LOG_CRIT("Request IRQ%d = %d failed\n", i, swdev->irq[i]);
			goto ERROR;
		}
	}

	LOG_INFO("Ethsw interrupt initialized\n");

	FUNC_LEAVE();
	return(ret);

 ERROR:
	/* exit interrupt properly */
	ethsw_int_exit(swdev);

	FUNC_LEAVE();
	return(ret);
}

void ethsw_int_exit(struct ethsw_device *swdev)
{
	int i;
	struct ethsw_int *swint;

	FUNC_ENTER();

	swint = swdev->swint;
	if (!swint) {
		goto EXIT;
	}

	/* free IRQ */
	for (i = 0; i < ETHSW_IRQ_MAX; i++) {
		free_irq(swdev->irq[i], swint);
	}

	/* cancel scheduled work */
	cancel_work_sync(&swint->irq_work);

	/* exit interrupt, platform dependent */
	swdev->ethsw_int_board_exit(swdev);

	/* free ethsw interrupt control block */
	kfree(swint);
	swdev->swint = NULL;

	LOG_INFO("Ethsw interrupt exited\n");

 EXIT:
	FUNC_LEAVE();
}
